
/*
 * The "PIPE" mode interface to the newsclip program.
 *
 * This interface, defined in our news filter program specification, is for
 * use by newsreaders that wish to talk directly to a newsclip program.
 *
 * In essence, when they have an article they might wish to present to the
 * user, they can first pass it to the newsclip program, to see if the
 * user really wants to read it or not.
 *
 * This 'passing' can range from just providing the filename to engaging in
 * a dialogue to send the article down a pipe if it doesn't exist in a
 * conventional file.
 *
 * At all times we must take care that we never try to read more from a pipe
 * than is in it, or we will block.
 *
 * In pipe mode, the newsclip program is called with stdin (desc 0) as a
 * command pipe from the caller to the newsclip program, and stdout as an
 * answer pipe from the newclip program to the caller.
 */

 /*
  * Newsclip(TM) Library Source Code.
  * Copyright 1989 Looking Glass Software Limited.  All Rights Reserved.
  * Unless otherwise licenced, the only authorized use of this source
  * code is compilation into a binary of the newsclip library for the
  * use of licenced Newsclip customers.  Minor source code modifications
  * are allowed.
  * Use of this code for a short term evaluation of the product, as defined
  * in the associated file, 'Licence', is permitted.
  */


#include "nl.h"
#include "pipemode.h"
#include <signal.h>

/* function that reads a command from the pipe.
 * This function returns the argument count.
 * The arguments are stored in a static buffer.  Pointers into that
 * buffer are stored into argv, which should be passed by the user.
 * The first argument, argv[0], will always be the command code.  This
 * is normally the letter 'C' followed by a command letter.
 *
 * The actual arguments will follow in argv[1]..argv[argc-1].  The first
 * argument will normally be a sequence number which must be returned in
 * answers.
 *
 * If there is a problem reading from the pipe, the arg count will be 0
 * If there is a problem in the argus, the arg count will be -1
 */

struct command command_buf;		/* command input buffer */
char argument_buf[MAX_ARGLEN];		/* buffer for arguments */
char last_sequence[SEQ_SIZE+1] = "-1";		/* last sequence num */
char cur_newsgroup[MAX_NGLEN];		/* last newsgroup */

FILE *pipelog = 0;
int inpipe = -1;			/* log of all input down pipe */

int
read_command( argv, avsize, storeseq )
char **argv;		/* pointer to array to store arg pointers in */
int avsize;		/* number of elements in argv */
bool storeseq;		/* store the sequence number? */
{
	int argsize;		/* size of argument buffer */
	int argc;		/* argument count */
	int scanloc;		/* where in the arg list we are */
	int howmanbytes;

	argc = 0;

	howmanbytes = read( F_COMPIPE, &command_buf, sizeof(struct command) );
	if( inpipe >= 0 )
		write( inpipe, &command_buf, howmanbytes );
	/* fprintf( stderr, "Filter: Read %d bytes\n", howmanbytes ); */
	if( pipelog )
		fprintf(pipelog,"F:Got: %s\n", &command_buf.comtype );
	if( howmanbytes == sizeof(struct command) ) {
		argv[argc++] = &command_buf.comtype;
		command_buf.space = 0;
		argsize = atoi( command_buf.arg_size );
		/* save away the sequence for replies */
		if( storeseq ) {
			strncpy(last_sequence, command_buf.seq_num, SEQ_SIZE);
			last_sequence[SEQ_SIZE] = 0;
			}
		if( argsize > 0 ) {
			if( argsize > MAX_ARGLEN ) {
				int i;
				char c;
				/* read away the too long arguments */
				for( i = 0; i < argsize; i++ ) {
					read( F_COMPIPE, &c, 1 );
					if( inpipe >= 0 )
						write( inpipe, &c, 1 );
					}
				return ERRCODE;
				}
			 else {
				int size;
				size=read(F_COMPIPE,argument_buf,argsize);
				if( inpipe > 0 )
					write(inpipe,argument_buf,size);
				if( size != argsize ) {
					return 0;
					}
				/* if not null terminated, give an error */
				if( argument_buf[argsize-1] != 0 )
					return ERRCODE;
				}
			/* now scan the arguments into argv */
			for( scanloc = 0; scanloc < argsize;
				     scanloc += strlen(argument_buf+scanloc)+1 )
				if( argc < avsize )
					argv[argc++] = argument_buf+scanloc;
			if( pipelog ) {
				int i;
				for( i = 1; i < argc; i++ )
					fprintf(pipelog, "F:Got Arg %d: %s\n",
							i, argv[i] );
				}
			}
		}
	 else {
		warning( 2, "Error reading from command pipe\n" );
		return 0;		/* end of file */
		}

	return argc;
}


char argcount[] = "Invalid argument count";

/* The master loop for pipe mode.  Read commands from the pipe and act
   upon them */

pipe_loop()
{
	int argc;			/* count of arguments to command */
	char *argv[MAX_ARGS];		/* argument pointer vector */
	bool running;			/* control for main loop */
	extern int accept_all;		/* accept all articles in this group */
	extern int reject_all;		/* reject all articles in this group */
	char *debenv;			/* debug environment variable */
	extern char *getenv();
	extern bool do_debug;


	/* debug log pipes */
#ifdef DEBUG
	debenv = getenv("NCLIPDEBUG");
	debenv = "truepipe";		/* temp for now */
	if( do_debug || (debenv && lowerlet(debenv[0]) == 't') ) {
		char pipename[MAX_FNAME];
		extern char *dotdir;
		sprintf( pipename, "%s/pipelog", dotdir );
		pipelog = fopen( pipename, "w" );
		sprintf( pipename, "%s/inpipe", dotdir );
		inpipe = creat( pipename, 0666 );
		if( pipelog )
			setbuf( pipelog, NULL );
		}
#endif

	/* First tell the newsreader that we're alive and kicking */
	reply_ok();		/* not really a 'reply' */

	/* do simple init.  Perhaps we wish to read nglas file? */

	initngs(FALSE);

	Uinit();

	/* ignore possible keyboard signals */

	signal( SIGINT, SIG_IGN );
	signal( SIGQUIT, SIG_IGN );

	running = TRUE;
	cur_newsgroup[0] = 0;

	while( running ) {
		argc = read_command( argv, MAX_ARGS, TRUE );
		if( argc == 0 ) {
			fprintf( stderr, "End of File from command process\n" );
			if( pipelog )
				fprintf( pipelog, "End of file from command process\n" );
			break;		/* terminate and close */
			}
		if( argc < 0 ) {
			/* send an error reply */
			reply_err( "Invalid Command" );
			continue;
			}
		reset_tempalloc();
		/* switch on the various commands */
		switch( argv[0][1] ) {
			case 'V':	/* version number */
				if( argc != ABASE+1 ) {
					reply_err( argcount );
					break;
					}
				reply_arg( 'V', "V100", "ABHNPQV", "ABEHORV",
						"NULL", "100", NULL );
				break;
			case 'Q':	/* quit */
				if( argc != ABASE ) {
					reply_err( argcount );
					/* terminate anyway */
					}
				reply_ok();
				running = FALSE;
				break;
			case 'P': 	/* program command */
				/* how many args? */
				if( argc < ABASE+1 ) {
					reply_err( argcount );
					break;
					}
				/* handle kill commands */
				handle_command( argv[ABASE] );
				break;
			case 'N': 
				/* query a newsgroup */
				if( argc != ABASE+1 ) {
					reply_err( argcount );
					break;
					};
				try_newgroup( argv[ABASE] );
				/* check flags? */
				if( accept_all )
					reply_arg( 'A', "1000", NULL );
				else if( reject_all )
					reply_arg( 'R', "-1000", NULL );
				 else
					reply_ok();
				break;
			case 'A': 	/* article dialogue */
				/* newsgroup artnum [filestyle filename] */
				do_art_dialogue( argc, argv );
				break;
			default:
				reply_err( "No such command" );
				break;
			}
		}
	/* terminate this newsgroup if there was one */
	if( cur_newsgroup[0] )
		finish_group();
	Uterminate();
}

extern int reading_mode;		/* style of reading articles */

/* do an article dialogue.  To understand this, you really have to read
   the spec, so a lot of comments here won't do a lot of good. */

do_art_dialogue( argc, argv )
int argc;
char **argv;
{
	newsgroup n;			/* newsgroup of dialogue */
	char artname[MAX_FNAME];	/* filename for article file */
	char scorebuf[10];		/* buffer to make ascii score string */
	extern int score;
	extern int article_number;
	int stat;			/* status of article */
	FILE *artfile;


	if( argc != ABASE+2 && argc != ABASE+4 ) {
		reply_err( argcount );
		return;
		}
	/* fprintf( stderr, "Article group %s %s mode %s file %s\n",
		argv[ABASE], argv[ABASE+1], argv[ABASE+2], argv[ABASE+3] ); */
	try_newgroup( argv[ABASE] );
	article_number = atoi(argv[ABASE+1]);

	if( argc > ABASE+2 ) {		/* there is a file name */
		
		strcpy( artname, argv[ABASE+3] );

		if( argv[ABASE+2][0] == 'R' )
			reading_mode = FILE_REQUEST;
		 else 
			reading_mode = FILE_FULL;

		}
	 else {
		/* pipe read currently unimplemented */
		reading_mode = PIPE_READ;
		artname[0] = 0;
		}
	stat = accept_article( artname );

	if( stat == ERRCODE )
		reply_err( "Bad Article Dialogue" );

	sprintf( scorebuf, "%d", score );
	reply_arg( stat ? 'A' : 'R', scorebuf, NULL );
}

/* Send a reply with a given command code and various string args.  The
 * list of string args (up to 4) is terminated by a null string
 */

reply_arg( code, a,b,c,d,e )
char code;
char *a,*b,*c,*d,*e;
{
	g_reply_arg( 'R', code, a,b,c,d,e );
}

 /* low level reply routine */

g_reply_arg( rtype, code, a,b,c,d,e )
char rtype;			/* reply type */
char code;			/* reply code */
char *a,*b,*c,*d,*e;		/* args */
{
	struct command reply_buf;
	char repargs[MAX_ARGLEN];
	char *vector[6];
	int i, pos;		/* loop counter and position in reparts */

	reply_buf.comtype = rtype;
	reply_buf.comcode = code;
	reply_buf.space = ' ';

	/* make a vector out of the various args, up to 5 of them */

	vector[0] = a;
	vector[1] = b;
	vector[2] = c;
	vector[3] = d;
	vector[4] = e;
	vector[5] = NULL;

	pos = 0;
	for( i = 0; i < 5 && vector[i]; i++ ) {
		strcpy( repargs+pos, vector[i] );
		/* check overflow? */
		pos += strlen(vector[i]) + 1;
		}
	sprintf( reply_buf.arg_size, "%03d", pos );
	reply_buf.zerob =0;
	/* copy back in the sequence number */
	sprintf( reply_buf.seq_num, "%5.5s", last_sequence );
	reply_buf.space2 = ' ';

	if( pipelog ) {
		int ic;
		fprintf( pipelog, "F:Send %s\n", &reply_buf.comtype );
		for( ic = 0; ic < 5 && vector[ic]; ic++ )
			fprintf( pipelog, ":%s:", vector[ic] );
		fprintf( pipelog, "\n" );
		}


	if( write(F_ANSPIPE, &reply_buf, sizeof(reply_buf))==sizeof(reply_buf)){
		if( pos > 0 && write( F_ANSPIPE, repargs, pos ) != pos )
			pipe_abort();
		}
	 else
		pipe_abort();
}

/* A write failed to the pipe, we have to terminate */

pipe_abort()
{
	if( pipelog )
		fprintf( pipelog, "Pipe Abort\n" );
	warning( 2, "Pipe mode abort\n" );
	if( cur_newsgroup[0] )
		finish_group();
	Uterminate();
	wrapup();
	exit(1);
}

/* check group name and if the group changed, call user routines if it did */
/* The main group must stay around until it changes.  To do this, we
   effectively reserve slot -1 for it using the ex_group_base variable. */

try_newgroup( gname )
char *gname;
{
	extern newsgroup main_newsgroup;
	extern int extra_groups, ex_group_base;	/* count of un-named groups */
	newsgroup new_newsgroup;

	if( strcmp( gname, cur_newsgroup ) != 0 ) {
		if( cur_newsgroup[0] ) {
			finish_group();
			}
		strcpy( cur_newsgroup, gname );
		/* allocate an extra group that stays until the next change */
		ex_group_base = extra_groups = 0;
		main_newsgroup = ng_number( cur_newsgroup );
		ex_group_base = extra_groups;
		Ustartgroup( 1 );
		}
}

reply_ok()
{
	reply_arg( 'O', NULL );
}

reply_err( ermsg )
char *ermsg;
{
	reply_arg( 'E', ermsg, NULL );
}

handle_command(com)
char *com;
{
	extern int score;
	score = 0;
	Ucommand(com);
	if( score > 0 )
		reply_ok();
	 else
		reply_err( "Invalid Kill Command" );
}

/*
 * Send a query, and await a response to that query
 * Returns 0 for OK, -1 for error
 */

query( wquery, qarg )
char wquery;		/* what sort of query */
char *qarg;		/* argument, if any */
{
	char *argv[MAX_ARGS];		/* argument pointer vector */
	int argc;
	g_reply_arg( 'Q', wquery, qarg, NULL );
	/* Read response, do not set sequence number */
	argc = read_command( argv, MAX_ARGS, FALSE );
	/* we don't care about the argument */
	if( argv[0][1] != wquery || argc > ABASE+2 )
		return ERRCODE;
	 else
		return 0;
}

/* ask that the header be written to the file */
/* This routine is not to be called in FILE_FULL mode */

int
ask_for_header()
{
	return query( 'H', NULL );
}

/* ask that the body be written to the article file */
/* Not much we can do with error status at this point, as we are too
   deep in code, but we will return it regardless */
/* This routine is not to be called in FILE_FULL mode */

int
ask_for_body()
{
	return query( 'B', NULL );
}
