

#include "nc.h"

/*
 * Typechecking routines for parse trees
 */

 /*
  * Newsclip(TM) Compiler Source Code.
  * Copyright 1989 Looking Glass Software Limited.  All Rights Reserved.
  * Unless otherwise licenced, the only authorized use of this source
  * code is compilation into a binary of the newsclip compiler for the
  * use of licenced Newsclip customers.  Minor source code modifications
  * are allowed before compiling.
  * A short time evaluation of this product is also permitted.  See the file
  * 'Licence' in the library source directory for details.
  */

#define numtype(x)	((x)==T_INTEGER||(x)==T_DATE||(x)==T_NEWSGROUP)
#define stringtype(x)	((x)==T_STRING||(x)==T_NEWSGROUP||(x)==T_USERNAME)

dtype
check( tp )
nodep tp;			/* the tree pointer */
{
	extern struct node_info node_table[];
	int nt;			/* node type */
	int i;
	dtype rtype;		/* our return type */
	dtype ktypes[MAX_KIDS];

	if( !tp )
		return 0;

	nt = tp->ntype;
	for( i = 0; i < node_table[nt].kids; i++ )
		ktypes[i] = check( tp->kids[i] );

	rtype = T_INTEGER;

	/* deal with the special cases */
	switch( nt ) {
		case N_LIST:
			{
			listp ol;
			/* check the parent for proc/func */
			/* we must return to avoid setting type */
			for( ol = (listp)tp; ol; ol = ol->next )
				check( ol->kid );
			return 0;
			}
		case N_FOREACH:

			if( insist_variable( kid0(tp) ) )
				break;

			if( ktypes[1] & T_ARRAY ) {
				if( (ktypes[1] & T_BASETYPE) != ktypes[0] ) {
					terror( tp, "Loop variable and range are not of matching type" );
					break;
					}
				}
			 else if( ktypes[1] == T_DATABASE ) {
				if( ktypes[0] != T_STRING ) {
					terror( tp, "Database loop requires a string variable" );
					break;
					}
				}
			 else {
				terror( tp, "for( xx in yy ) -- invalid 'yy' to search through" );
				}
			break;
		case N_INDEX:
			if( ktypes[0] == T_DATABASE ) {
				if( stringtype(ktypes[1]) )
					make_string( kid1(tp), ktypes[1] );
				 else
					terror( tp, "Database index must be a single string" );
				rtype = T_INTEGER;
				}
			else {
				if( !( ktypes[0] & T_ARRAY ) )
				
					terror( tp, "Indexing requires an array or database" );
				else if( ktypes[1] != T_INTEGER )
					terror( tp, "Array index must be an integer");
				rtype = ktypes[0] & T_BASETYPE;
				}
			break;
		case N_STRING:
			rtype = T_STRING;
			break;
		case N_NGROUP:
			rtype = T_NEWSGROUP;
			break;
		case N_PAREN:
			rtype = ktypes[0];
			break;
			
		case N_EQ:
		case N_NE: 
			if( ktypes[0] != ktypes[1] ) {
				if( numtype(ktypes[0]) && numtype(ktypes[1]) ) {
					rtype = T_INTEGER;		/* no cast */
					break;
					}
				if( stringtype(ktypes[0]) && stringtype(ktypes[1]) ) {
					make_string(kid0(tp), ktypes[0]);
					make_string(kid1(tp), ktypes[1]);
					rtype = T_INTEGER;
					break;
					}
				if( (ktypes[0] == T_ARRAY || ktypes[1] ==
						T_ARRAY) && ktypes[0] &
						ktypes[1] & T_ARRAY ) {
					rtype = T_INTEGER;
					break;
					}
				/* check for comparison to NIL */
				terror( tp, "Comparison of incompatible types");
				}
			 else if( ktypes[0] > T_STRING ) {
				/* if one is a predeclared symbol, that's ok */
				if( !predsym(kid0(tp)) && !predsym(kid1(tp)) )
					terror( tp, "Comparison on uncomparable types");
				}
			rtype = T_INTEGER;
			break;
		case N_IN:
		case N_NOT_IN:
			if( ktypes[1] == T_DATABASE ) {
				if( stringtype(ktypes[0]) )
					make_string( kid0(tp), ktypes[0] );
				else if( ktypes[0] != arrayof(T_STRING) )
					terror( tp, "Can only check for strings and strings arrays in databases" );
				}
			else if( !( ktypes[1] & T_ARRAY ) )
				terror( tp, "Can only check IN array or database" );
			else if((ktypes[0]&T_BASETYPE)!=(ktypes[1]&T_BASETYPE))
				terror( tp, "Types don't match on IN" );
			rtype = T_INTEGER;
			break;
		case N_HAS:
		case N_NOT_HAS:
			if( stringtype(ktypes[1]) ) {
				if( kid1(tp)->ntype == N_STRING ) {
					/* turn the hard string into a
						precompiled pat*/
					nodep skid;
					int patnum;
					skid = kid1(tp);
					patnum = pat_number((char *)kid0(skid));
					free( (char *)kid0(skid) );
					skid->kids[0] = (nodep) patnum;
					skid->ntype = N_PATTERN;
					}
				 else
					make_string( kid1(tp), ktypes[1] );
				}
			 else if( !( ktypes[1] & T_ARRAY && stringtype(ktypes[1]
						& T_BASETYPE) ) && ktypes[1] !=
						T_DATABASE ) 
				terror( tp, "HAS pattern must be string or database" );
			if( stringtype(ktypes[0]) ) 
				make_string( kid0(tp), ktypes[0] );
			else if( !stringtype(ktypes[0]&T_BASETYPE) &&
					ktypes[0] != T_DATABASE &&
					ktypes[0] != T_TEXT ) 
				terror( tp, "HAS search area must be string, database or text region" );
			rtype = T_INTEGER;
			break;

		case N_POSTINC:
		case N_POSTDEC:
		case N_PREINC:
		case N_PREDEC:
			insist_variable( kid0(tp) );
			if( ktypes[0] != T_INTEGER ) 
				terror(tp,"Increment and decrement allowed on numbers only" );
			rtype = T_INTEGER;
			break;
		case N_QUERY:
			if( ktypes[0] != T_INTEGER )
				terror(tp, "Query condition must be numeric" );
			 else if( ktypes[1] != ktypes[2] )
				terror(tp, "Types of query clauses don't match" );
			rtype = ktypes[1];
			break;
		case N_ID:
			rtype = ((symptr)kid0(tp))->type;
			break;
		case N_CALL:
			{
			symptr prsym;
			if( kid0(tp) && (prsym = (symptr)kid0(kid0(tp)) ) ) {
				check_args( prsym, (listp)kid1(tp) );
				if( prsym->decl_type != ST_PROC )
					terror( tp, "'%s' is not a procedure",
						prsym->name );
				}
			break;
			}
		case N_FUNCALL:
			{
			symptr funsym;
			if( kid0(tp) && (funsym = (symptr)kid0(kid0(tp)) ) ) {
				check_args( funsym, (listp)kid1(tp) );
				rtype = funsym->type;
				if( funsym->decl_type != ST_FUNC )
					terror(tp, "'%s' is not a function",
						funsym->name );
				}
			break;
			}
		case N_ASSIGN:
			if( !insist_variable( kid0(tp) ) ) {
				if( assign_check( ktypes[0], kid1(tp),
								ktypes[1] ) )
					terror( tp, "Incompatible types on assignment" );
				}
			rtype = 0;
			break;
		case N_PARSE:		/* assign array */
			make_string( kid1(tp), ktypes[1] );
			insist_variable( kid0(tp) );
			if( ktypes[0] & T_ARRAY ) {
				if( kid2(tp) )
					make_string( kid2(tp), ktypes[2] );
				 else
					terror(tp,"Array parse requires delimiters" );
				}
			 else {
				if( kid2(tp) )
					terror(tp,"Delimiters are only meaningful on an array parse");
				}
			break;
		case N_ARINIT:		/* init empty array */
			insist_variable( kid0(tp) );
			if( ktypes[0] & T_ARRAY ) {
				if( ktypes[1] != T_INTEGER )
					terror(tp,"Array size must be integer");
				}
			 else
				terror(tp,"Array assign requires array variable");
			break;
		case N_FOR:
			if( kid1(tp) != NIL && ktypes[1] != T_INTEGER )
				terror( tp, "For loop condition requires bool/int" );
			rtype = 0;
			break;
		case N_GOTO: {
			symptr sym;
			if( kid0(tp) && (sym = (symptr)kid0(kid0(tp))) &&
					sym->decl_type != ST_LABEL )
				terror( tp, "'%s' is not a label", sym->name );
			break;
			}
		case N_RETURN: {
			extern int in_routine;
			if( in_routine == ST_FUNC ) {
				extern dtype routine_type;
				if( kid0(tp) == NIL )
					terror( tp, "Function returns must return a value" );
				else if( assign_check( routine_type, kid0(tp),
							ktypes[0] ) )
					terror( tp, "Invalid type for function return value" );
				}
			 else if( kid0(tp) != NIL )
				terror( tp, "Only functions may return values");
			
			break;
			}
		default: {
			byte nfl;		/* flags for node type */
			extern int in_routine;

			nfl = node_table[nt].flags;
			if( nfl & TF_RET && in_routine == ST_FUNC )  {
				terror( tp, "Accept and Reject are not allowed inside functions" );
				break;
				}
			if( !(nfl & TF_RETINT) ) {
				if( nfl & TF_ONEINT ) 
					rtype = procint( tp, rtype, ktypes[0] );
			 	else if( nfl & TF_2INT ) {
					rtype = procint( tp, rtype, ktypes[0] );
					rtype = procint( tp, rtype, ktypes[1] );
					}
				}
			break;
			}

		}
	tp->ndtype = rtype;
	return rtype;

}

/* make sure a node returns a string */

make_string( tp, tpt )
nodep tp;		/* tree pointer */
dtype tpt;		/* type of this tree */
{
	switch(tpt) {
		case T_NEWSGROUP:
			tp->nflags |= CAST_NGNAME;
			tp->ndtype = T_STRING;
			break;
		case T_USERNAME:
			tp->nflags |= CAST_MAILNAME;
			tp->ndtype = T_STRING;
			break;
		case T_STRING:	
			break;
		default:
			terror( tp, "String required" );
			break;
		}
}

/* Expect an integer or numeric argument */

dtype
procint( tp, otype, newtype )
nodep tp;		/* field */
dtype otype;		/* old type */
dtype newtype;		/* type of argument */
{
	if( newtype == T_DATE || newtype == T_INTEGER )
		return newtype;
	 else if( newtype == T_NEWSGROUP )
		return T_INTEGER;
	 else
		terror( tp, "Number type required" );
	return T_INTEGER;
}



/* insist that the tree is a variable, return give error and return
   TRUE if it is not */

bool
insist_variable( tr )
nodep tr;		/* tree that must be a variable */
{
	if( is_variable(tr) ) {
		tr->nflags |= NF_LVALUE;
		return FALSE;
		}
	else {
		terror( tr, "Assignment to non-variable" );
		return TRUE;
		}
}

bool
is_variable( tr )
nodep tr;		/* variable tree */
{
	int ttype;

	ttype = tr->ntype;
	if( ttype == N_ID ) {
		symptr tid;
		tid = (symptr)kid0(tr);
		return tid->decl_type == ST_VAR && !(tid->sflags & OSF_CONST);
		}
	else if( ttype == N_INDEX && is_variable(kid0(tr)) )
		return TRUE;
	return FALSE;
}

/*
 * Test if two types are assignment compatible and set cast flags
 * returns true if there is an error.
 */


bool
assign_check( destype, src, srctype )
dtype destype;			/* type of destination var */
nodep src;			/* source tree */
dtype srctype;			/* source type */
{
	if( destype != srctype ) {
		if( destype & T_ARRAY && srctype == T_ARRAY )
			return FALSE;
		switch( destype ) {
			case T_DATE:
				if( srctype == T_INTEGER ) 
					src->nflags |= CAST_DATE;
				 else {
					terror(src,"Integer or date required");
					return TRUE;
					}
				break;
			case T_INTEGER:
				if( srctype == T_DATE )
					src->nflags |= CAST_INT;
				 else if( srctype != T_NEWSGROUP ) {
					terror(src,"Numeric value required");
					return TRUE;
					}
				break;
			case T_STRING:
				if( srctype == T_NEWSGROUP )
					src->nflags |= CAST_NGNAME;
				 else if( srctype == T_USERNAME )
					src->nflags |= CAST_MAILNAME;
				 else {
					terror(src,"String value required");
					return TRUE;
					}
				break;
			case T_GENARRAY:
				if( !(srctype & T_ARRAY) ) {
					terror( src,"Array value required" );
					return TRUE;
					}
				break;
			case T_GENPTR:
				if( !( srctype & T_ARRAY || srctype == T_STRING || srctype == T_DATABASE || srctype == T_USERNAME ) ){
					terror( src,"Structured data value required" ); 
					return TRUE;
					}
				break;
			default:
				return TRUE;
			}
		}
	return FALSE;		/* types match */
}

/* Check the arguments on a call to a procedure or function */

check_args( funsym, funargs )
symptr funsym;		/* the symbol for the subroutine */
listp funargs;		/* the argument list */
{
	struct typelist *dtlist;	/* declared type list */
	listp curarg;
	int i;


	dtlist = funsym->argtypes;

	/* a null type list means an arbitrary argument list */
	if( !dtlist )
		return;

	curarg = funargs;
	for( i = 0; i < dtlist->argcount; i++ ) {
		if( curarg ) {
			if( assign_check( dtlist->args[i], curarg->kid,
						curarg->kid->ndtype ) )
				terror( curarg, "Type mismatch on argument %d of '%s'", i+1, funsym->name );
			}
		 else {
			/* end of list */
			if( i >= dtlist->argmin ) {
				terror( funargs, "%s: Too few arguments",
						funsym->name );
				return;
				}
			break;		/* loop done */
			}
		curarg = curarg->next;
		}
	/* if there were still arguments left, count them */
	while( curarg ) {
		if( ++i >= dtlist->argmax ) {
			terror( funargs, "%s: Too many arguments",funsym->name);
			return;
			}
		curarg = curarg->next;
		}

}

/* is this tree a reference to a predefined symbol? */
int
predsym( tp )
nodep tp;
{
	return tp->ntype == N_ID && ((symptr)kid0(tp))->sflags & OSF_PREDEF;
}
