#include "date.h"
#include "timecls.h"
#include "oopsconfig.h"

#define THIS    Time
#define BASE    Object
DEFINE_CLASS(Time,Object);
////////////////////////////////////////////////////////////
//  class Time
//  Provides an object that represents
//  a Time, stored as the number of
//  seconds since January 1, 1901, GMT.
////////////////////////////////////////////////////////////

#include <time.h>
#define TIME_ZONE timezone
#define DST_OBSERVED daylight



//-gmv remove static for debugging
/*static*/ const unsigned long seconds_in_day = 24L*60*60; //-gmv made long
/*static*/ const Date refDate(0);
/*static*/ const Date maxDate(49709);   // ((2**32)-1)/seconds_in_day -1 
    
//////////////////////////////////////////////////
// Return a local Time for the specified
// Standard Time date, hour, minute, and second.
//////////////////////////////////////////////////
static Time localTime(const Date& date, hourTy h=0, minuteTy m=0, secondTy s=0)
{
    if (!date.between(refDate,maxDate))
        DTerror("Date out  of range: ", "");
    return Time(seconds_in_day*(date-refDate) + 60*60L*h + 60*m + s);
}

//////////////////////////////////////////////////
//  Construct a Time for this instant
//////////////////////////////////////////////////
Time::Time()                      
{
    sec += 2177452800L;         // seconds from 1/1/01 to 1/1/70 
}

//////////////////////////////////////////////////
//  update Time for this instant
//////////////////////////////////////////////////
void Time::update()             
{
    sec += 2177452800L;         // seconds from 1/1/01 to 1/1/70
}

//////////////////////////////////////////////////
//  Construct a Time for today at
//  the specified (local) hour,
//  minute, and second.
//////////////////////////////////////////////////
Time::Time(hourTy h, minuteTy m, secondTy s, bool dst)
{
    sec = Time(Date(),h,m,s,dst).sec;
}


//////////////////////////////////////////////////
//  Construct a Time for
//  the specified (local) Date, hour,
//  minute, and second.
//////////////////////////////////////////////////
Time::Time(const Date& date, hourTy h, minuteTy m, secondTy s, bool dst)
{
    sec = ::localTime(date,h,m,s).sec;
    if (isDST()) {
        if (isDST() || dst) sec -= 3600;
    }
    else {
        if (isDST())
            DTerror("Invalid date/time: ", "");
    }
    sec += TIME_ZONE;               // adjust to GMT
}

//////////////////////////////////////////////////
//  Convert a Time to a local Date
//////////////////////////////////////////////////
#ifdef DATE_CAST
Time::operator Date() const
#else
Date Time::getDate() const
#endif

{
    clockTy daycount = localTime().sec/seconds_in_day;
    return Date(daycount);
}

//////////////////////////////////////////////////
//  Return the hour of this Time
//  in local time; i.e., adjust for
//  time zone and Daylight Savings Time.
//////////////////////////////////////////////////
hourTy Time::hour() const
{
    return localTime().hourGMT();
}

//////////////////////////////////////////////////
//  Return the hour of this Time in GMT.
//////////////////////////////////////////////////
hourTy Time::hourGMT() const
{
    return (hourTy)((sec % 86400L) / 3600L);
}

//////////////////////////////////////////////////
//  Return the local Standard Time
//  at which Daylight Savings Time
//  begins in the specified year.
//////////////////////////////////////////////////
static Time beginDST(unsigned year)
{
    Time DSTtime(localTime(Date(31,"Mar",year).previous("Sun")+7,2));
    if (year<=1986) {
        DSTtime = localTime(Date(30,"Apr",year).previous("Sun"),2);
        if (year==1974) DSTtime = localTime(Date(6,"Jan",1974),2);
        if (year==1975) DSTtime = localTime(Date(23,"Feb",1975),2);
    }
    return DSTtime;
}

//////////////////////////////////////////////////
//  Return the local Standard Time
//  at which Daylight Savings Time
//  ends in the specified year.
//////////////////////////////////////////////////
static Time endDST(unsigned year)
{
    Time STDtime(localTime(Date(31,"Oct",year).previous("Sun"),2-1));
    return STDtime;
}

//////////////////////////////////////////////////
//  Return YES if this local Standard Time
//  should be adjusted for Daylight Savings Time.
//////////////////////////////////////////////////
bool Time::isDST() const
{
    int daycount = this->sec/seconds_in_day;
    unsigned year = Date(daycount).year();
    if (DST_OBSERVED && *this >= beginDST(year) && *this < endDST(year)) 
        return YES;
    return NO;
}

//////////////////////////////////////////////////
//  Adjusts this GM Time for local
//  time zone and Daylight Savings Time.
//////////////////////////////////////////////////
Time Time::localTime() const
{
    Time local_time(sec-TIME_ZONE);
    if (local_time.isDST()) local_time.sec += 3600;
    return local_time;
}

//////////////////////////////////////////////////
//  Return the minute of this Time        
//  in local time; i.e., adjust for       
//  time zone and Daylight Savings Time.  
//////////////////////////////////////////////////
minuteTy Time::minute() const
{
    return localTime().minuteGMT();
}

//////////////////////////////////////////////////
//  Return the minute of this Time in GMT.  
//////////////////////////////////////////////////
minuteTy Time::minuteGMT() const
{
    return ((sec % 86400L) % 3600L) / 60L;
}

//////////////////////////////////////////////////
//  Return the second of this Time.
//////////////////////////////////////////////////
secondTy Time::second() const
{
    return ((sec % 86400L) % 3600L) % 60L;
}

Time Time::max(Time t) const
{
    if (t.sec < sec)
        return *this;
    else
        return t;
}

Time Time::min(Time t) const
{
    if (t.sec > sec)
        return *this;
    else
        return t;
}

int Time::compare(const Object& ob) const
{
    assertArgSpecies(ob,class_Time,"compare");
    register clockTy t = ((Time*)&ob)->sec;
    if (sec < t) return -1;
    if (sec > t) return 1;
    return 0;
}

Object*         Time::copy() const          { return shallowCopy(); }
void            Time::deepenShallowCopy()   {}
unsigned        Time::hash() const          { return sec; }
bool            Time::isEqual(const Object& ob) const
                {
                    return ob.isSpecies(class_Time) && *this==*(Time*)&ob;
                }
const Class*    Time::species() const       { return &class_Time; }
void            Time::printOn(ostream& strm) const
                {
                    unsigned hh = hour();
#ifdef DATE_CAST
                    ((Date)*this).printOn(strm);
#else
                    getDate().printOn(strm);
                    cout << " ";
#endif
                    char saveFillChar = strm.fill();
                    long saveFlags = strm.flags();

                    strm.width(2);
                    strm.fill(' ');
                    strm << ((hh <= 12) ? hh : hh-12) << ":";
                    strm.width(2);
                    strm.fill('0');
                        strm << minute() << ":";
                    strm.width(2);
                    strm.fill('0');
                    strm << second();

                    strm.fill(saveFillChar);
                    strm.flags(saveFlags);
                    if (hh < 12) strm << " am";
                    else strm << " pm";
}

