/*
 * Copyright (c) 1998-2006 TeamDev Ltd. All Rights Reserved.
 * Use is subject to license terms.
 */

// ================================== PUBLIC API METHODS

/**
 * Returns true if the tree table has any node(s) selected. This method works with any selection mode.
 */
function q_isTreeTableSelectionEmpty(treeTableId) {
  if (!treeTableId)
    throw "q_isTreeTableSelectionEmpty: treeTable's clientId must be passed as a parameter";
  var table = document.getElementById(treeTableId);
  if (!table)
    throw "q_isTreeTableSelectionEmpty: Invalid clientId passed - no such component was found: " + treeTableId;
  if (!table._q_treeTableComponentMarker)
    throw "q_isTreeTableSelectionEmpty: The clientId passed refers to a component other than <q:treeTable> : " + treeTableId;
  return table.__isSelectionEmpty();
}

/**
 * Returns the number of selected nodes, or 0 if no nodes are selected. This method works with any selection mode.
 */
function q_getSelectedNodeCount(treeTableId) {
  if (!treeTableId)
    throw "q_getSelectedNodeCount: treeTable's clientId must be passed as a parameter";
  var table = document.getElementById(treeTableId);
  if (!table)
    throw "q_getSelectedNodeCount: Invalid clientId passed - no such component was found: " + treeTableId;
  if (!table._q_treeTableComponentMarker)
    throw "q_getSelectedNodeCount: The clientId passed refers to a component other than <q:treeTable> : " + treeTableId;
  return table.__getSelectedNodeCount();
}

/**
 * Reloads the tree-table using Ajax without reloading the whole page. The tree-table parameters such as sorting and
 * expansion state, as well as parameters of tree-table's sub-components and facets are submitted to the server just
 * like during the standard form submission. This method can only be invoked on a tree-table which has its "useAjax"
 * attribute set to true (this is default value).
 * 
 * Note that this method reloads the tree-table asynchronously, that is it initiates ajax request for reloading a
 * tree-table and returns without waiting for this request to complete.
 *
 * Parameters:
 *   treeTableId - client ID of a tree-table which should be refreshed. Data for all sub-components and facets of this
 *                 tree-table will be submitted like during an ordinary form submission.
 *   submittedComponentIds - Optional - can be null. An array of client IDs for components those should be submitted in
 *                           addition to the tree-table and its inner components.
 *   serverAction - Optional - can be null. An action in the form of "backingBeanName.methodName", which should be
 *                  executed during this Ajax request. The method to which this action refers should be a public method
 *                  without parameters and having a "void" return type.
 */
function q_refreshTreeTable(treeTableId, submittedComponentIds, serverAction) {
  if (!treeTableId)
    throw "q_reloadTreeTable: TreeTable's clientId must be passed as a parameter";
  var treeTable = document.getElementById(treeTableId);
  if (!treeTable)
    throw "q_reloadTreeTable: Invalid clientId passed - no such component was found: " + treeTableId;
  if (!treeTable._q_treeTableComponentMarker)
    throw "q_reloadTreeTable: The clientId passed refers to a component other than <q:treeTable> : " + treeTableId;
  if (!treeTable._useAjax)
    throw "q_reloadTreeTable: This method can't be invoked if Ajax is turned off for this TreeTable. Please set the useAjax attribute to true on the corresponding <q:treeTable> tag. TreeTable's clientId: " + treeTableId;
  q__reloadComponent(treeTableId, null, submittedComponentIds, serverAction);
}


// ================================== END OF PUBLIC API METHODS


// -------------------------- INITIALIZATION

function q__initTreeTableAPI(table) {
  table.__getSelectedNodeCount = function () {
    var selectedItems = this._getSelectedItems();
    if (!selectedItems || (selectedItems.length == 1 && selectedItems[0] == -1))
      return 0;
    var result = selectedItems.length;
    return result;
  }
  table.isSelectionEmpty = function() { return this.__isSelectionEmpty(); }
  table.getSelectedNodeCount = function() { return this.__getSelectedNodeCount(); }
  table._q_treeTableComponentMarker = true;
}

// -------------------------- FOLDING SUPPORT

function q__initTreeTableFolding(tableId, toggleClassName, clientFoldingParams) {
  var table = document.getElementById(tableId);

  var rowIndexToChildCount     = clientFoldingParams ? clientFoldingParams[0] : null;
  var expandedRowIndexes       = clientFoldingParams ? clientFoldingParams[1] : null;
  var treeColumnExpansionDatas = clientFoldingParams ? clientFoldingParams[2] : null;
  var hiddenRowClassName       = clientFoldingParams ? clientFoldingParams[3] : null;

  table._rowIndexToChildCount = rowIndexToChildCount;
  table._foldingMode = clientFoldingParams ? "client" : "server";
  table._treeColumnExpansionDatas = treeColumnExpansionDatas;
  table._toggleClassName = toggleClassName;

  table._updateRowVisibility = function() {
    var rootNodeCount = this._rowIndexToChildCount["root"];
    var rows = table._getBodyRows();
    var rowIndexToChildCount = this._rowIndexToChildCount;
    var result = q__treeTable_processRowVisibility (rows, rowIndexToChildCount, 0, rootNodeCount, true);
    return result;
  }

  table._updateExpandedNodesField = function() {
    var fieldId = this.id + "::expandedNodes";
    var field = document.getElementById(fieldId);
    var expandedRowIndexes = "";
    var rows = this._getBodyRows();
    for (var rowIndex = 0, rowCount = rows.length; rowIndex < rowCount; rowIndex++) {
      var row = rows[rowIndex];
      var expanded = row._expanded;
      if (expanded) {
        if (expandedRowIndexes.length > 0)
          expandedRowIndexes += ",";
        expandedRowIndexes += rowIndex;
      }
    }
    field.value = expandedRowIndexes;
  }

  var rowIndexToExpandedMap = new Object();
  if (expandedRowIndexes)
    for (var i = 0, count = expandedRowIndexes.length; i < count; i++) {
      var rowIndex = expandedRowIndexes[i];
      rowIndexToExpandedMap[rowIndex] = true;
    }

  var rows = table._getBodyRows();
  for (var rowIndex = 0, rowCount = rows.length; rowIndex < rowCount; rowIndex++) {
    var row = rows[rowIndex];
    q__initTreeTableRow(row);
  }
  if (table._foldingMode == "client") {
    table._updateRowVisibility();
    table._updateExpandedNodesField();

    for (var rowIndex = 0, rowCount = rows.length; rowIndex < rowCount; rowIndex++) {
      var row = rows[rowIndex];
      q__excludeClassNames(row, [hiddenRowClassName]);
    }

  }

  table._onKeyboardNavigation = function(e) {
    var selectedItems = this._getSelectedItems();
    if (selectedItems.length != 1)
      return true;
    var selectedRowIndex = selectedItems[0];
    if (selectedRowIndex == -1)
      return true;
    var bodyRows = this._getBodyRows();
    var row = bodyRows[selectedRowIndex];
    if (!row._hasChildren)
      return true;
    if ((e.rightPressed || e.plusPressed) && !row._expanded) {
      row._setExpanded(true);
      return false;
    }
    if ((e.leftPressed || e.minusPressed) && row._expanded) {
      row._setExpanded(false);
      return false;
    }

    return true;
  }
}

function q__treeTable_processRowVisibility(rows, rowIndexToChildCount, rowIndex, rowCount, currentLevelVisible) {
  for (var i = 0; i < rowCount; i++) {
    var row = rows[rowIndex];
    q__assert(row, "processRowVisibility: rowIndex == " + rowIndex);
    row._setVisible(currentLevelVisible);
    if (currentLevelVisible && q__isExplorer()) { // workaround for IE issue: when custom RowStyle is applied to parent rows and a node is expanded, spacing between cells becomes white after node expansion
      row._updayStyle();
    }
    var childCount = rowIndexToChildCount[rowIndex];
    if (childCount) {
      row._hasChildren = true;
      row._childrenLoaded = (childCount != "?");
      var nextLevelVisible = currentLevelVisible && row._expanded;
      if (row._childrenLoaded) {
        rowIndex = q__treeTable_processRowVisibility(rows, rowIndexToChildCount, rowIndex + 1, childCount, nextLevelVisible);
      } else
        rowIndex++;
    } else {
      row._hasChildren = false;
      rowIndex++;
    }
  }
  return rowIndex;
}

function q__initTreeTableRow(row) {
  var table = row._table;
  var rowIndex = row._index;

  row._setExpanded = function(expanded) {
    if (row._childrenEmpty)
      return;
    if (this._expanded == expanded)
      return;
    var rowTable = this._table;
    var foldingMode = rowTable._foldingMode;
    this._expanded = expanded;
    var toggles = this._toggles;
    for (var toggleIndex = 0, toggleCount = toggles.length; toggleIndex < toggleCount; toggleIndex++) {
      var toggle = toggles[toggleIndex];
      var treeColExpansionData = rowTable._treeColumnExpansionDatas[toggleIndex];
      toggle.src = expanded ? treeColExpansionData[0] : treeColExpansionData[1];
    }
    rowTable._updateExpandedNodesField();
    if (expanded && !this._childrenLoaded) {
      if (rowTable._useAjax)
        q__requestComponentPortions(rowTable.id, ["subRows:" + this._index], q__subRowsLoaded);
      else
        q__submitFormWithAdditionalParam(rowTable, rowTable.id + "::toggleExpansion", this._index);
    } else {
      rowTable._updateRowVisibility();
    }
  }
  var expandedToggles = q__findChildNodesByClass(row, "tc_toggle_e");
  var collapsedToggles = q__findChildNodesByClass(row, "tc_toggle_c");
  var expanded = expandedToggles && expandedToggles.length > 0;
  q__assert(!collapsedToggles || collapsedToggles.length == 0, "A row can't contain both expanded and collapsed nodes. rowIndex = " + rowIndex + "; table id = " + this.id);
  var toggles = expanded ? expandedToggles : collapsedToggles;
  row._toggles = toggles;
  row._hasChildren = toggles && toggles.length > 0;
  if (false) //rowIndexToExpandedMap[rowIndex])
    row._expanded = true;
  else {
    if (row._hasChildren)
      row._expanded = expanded;
  }
  for (var toggleIndex = 0, toggleCount = toggles.length; toggleIndex < toggleCount; toggleIndex++) {
    var toggle = toggles[toggleIndex];
    toggle.className = table._toggleClassName;
    toggle._rowIndex = rowIndex;
    toggle._row = row;
    toggle.onclick = function(evt) {
      if (evt)
        event = evt;
      var foldingMode = table._foldingMode;
      var clickedRow = this._row;
      clickedRow._setExpanded(!clickedRow._expanded);
      event.cancelBubble = true;
      return false;
    }
    toggle.ondblclick = function(evt) {
      this.onclick(evt);
    }
  }
}

function q__subRowsLoaded(treeTable, portionName, portionHTML, portionScripts) {
  var sepIdx = portionName.indexOf(":");
  var indexStr = portionName.substring(sepIdx + 1);
  var parentRowIndex = eval(indexStr);

  var tempDiv = document.createElement("div");
  tempDiv.style.display = 'none';
  tempDiv.innerHTML = "<table><tbody>" + portionHTML + "</tbody></table>";
  var tableBody = tempDiv.getElementsByTagName("tbody")[0];
  var children = tableBody.childNodes;
  var newNodes = new Array();
  for (var i = 0, count = children.length; i < count; i++) {
    var child = children[i];
    newNodes.push(child);
  }

  var newRows = new Array();
  for (var i = 0, count = newNodes.length; i < count; i++) {
    var newNode = newNodes[i];
    var prnt = newNode.parentNode;
    prnt.removeChild(newNode);
    if (!newNode || !newNode.tagName)
      continue;
    var tagName = newNode.tagName.toUpperCase();
    if (tagName == "TR") {
      newRows.push(newNode);
    }
  }

  treeTable.__parentRowIndex = parentRowIndex;
  treeTable.__newRows = newRows;
  q__executeScripts(portionScripts);
}

function q__insertSubrows(treeTableId, subRowsData) {
  var treeTable = document.getElementById(treeTableId);
  var newRows = treeTable.__newRows;
  var parentRowIndex = treeTable.__parentRowIndex;
  var newRowIndexToChildCount = subRowsData[0];
  var newRowsToStylesMap = subRowsData[1];
  var newRowCellsToStylesMap = subRowsData[2];

  treeTable._insertRowsAfter(parentRowIndex, newRows, newRowsToStylesMap, newRowCellsToStylesMap);

  var rows = treeTable._getBodyRows();
  var parentRow = rows[parentRowIndex];

  if (newRows == null || newRows.length == 0) {
    parentRow._childrenEmpty = true;
    var toggles = parentRow._toggles;
    for (var toggleIndex = 0, toggleCount = toggles.length; toggleIndex < toggleCount; toggleIndex++) {
      var toggle = toggles[toggleIndex];
      toggle.style.visibility = "hidden";
      toggle.className = "";
    }
  }
  parentRow._childrenLoaded = true;
  var addedRowCount = newRows.length;
  for (var i = 0; i < addedRowCount; i++) {
    var newRow = newRows[i];
    q__initTreeTableRow(newRow);
  }

  var rowIndexToChildCount = treeTable._rowIndexToChildCount;
  for (var rowIndex = rows.length - 1 - addedRowCount; rowIndex > parentRowIndex; rowIndex--) {
    if (rowIndexToChildCount[rowIndex] != undefined) {
      var newRowIndex = rowIndex + addedRowCount;
      rowIndexToChildCount[newRowIndex] = rowIndexToChildCount[rowIndex];
      rowIndexToChildCount[rowIndex] = undefined;
    }
  }
  rowIndexToChildCount[parentRowIndex] = newRowIndexToChildCount[0];
  for (var i = 0; i < addedRowCount; i++) {
    var newRowIndex = parentRowIndex + 1 + i;
    rowIndexToChildCount[newRowIndex] = newRowIndexToChildCount[i + 1];
  }

//  }

}
