/*
 *  Copyright 2004 the mime4j project
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
using System;
using System.IO;
using System.Text;

using NI.Email.Mime;
using NUnit.Framework;

namespace NI.Tests.Email.Mime {

	/**
	* 
	*
	* @author Niklas Therning
	* @version $Id: MimeBoundaryInputStreamTest.java,v 1.2 2004/10/02 12:41:11 ntherning Exp $
	*/
	public class MimeBoundaryInputStreamTest  {
		
		string Read(Stream stream) {
			MemoryStream resultStream = new MemoryStream();
			int b;
			while ( (b = stream.ReadByte() ) != -1 ) {
				resultStream.WriteByte( (byte)b);
			}
			return Encoding.ASCII.GetString( resultStream.ToArray() );
		}		
		
		
		/**
		* Tests that a CRLF immediately preceding a boundary isn't included in
		* the stream.
		*/
		[Test]
		public void test_CRLFPrecedingBoundary() {
			String text = "Line 1\r\nLine 2\r\n--boundary\r\n" +
					"Line 3\r\nLine 4\r\n\r\n--boundary\r\n";
	        
			MemoryStream mem = new MemoryStream(Encoding.ASCII.GetBytes(text));
	        
			Assert.AreEqual( 
				Read(new MimeBoundaryInputStream(mem, "boundary")),
				"Line 1\r\nLine 2" );

			Assert.AreEqual( 
				Read(new MimeBoundaryInputStream(mem, "boundary")),
				"Line 3\r\nLine 4\r\n" );
	        
		}

	    [Test]
		public void test_BigEnoughPushbackBuffer() {
			String text = "Line 1\r\nLine 2\r\n--boundar\r\n" +
					"Line 3\r\nLine 4\r\n\r\n--boundary\r\n";
	        
			MemoryStream mem = new MemoryStream(Encoding.ASCII.GetBytes(text));
	        
			Assert.AreEqual(
				Read(new MimeBoundaryInputStream(mem, "boundary")),
				"Line 1\r\nLine 2\r\n--boundar\r\n" +
				"Line 3\r\nLine 4\r\n");
		}
	    
	    
		/**
		* Tests that a stream containing only a boundary is empty.
		*/
		[Test]
		public void test_ImmediateBoundary() {
			String text = "--boundary\r\n";
	        MemoryStream mem = new MemoryStream(Encoding.ASCII.GetBytes(text));

			MimeBoundaryInputStream stream = new MimeBoundaryInputStream(mem, "boundary");
			Assert.AreEqual( stream.ReadByte(), -1);
	        
			text = "\r\n--boundary\r\n";
			mem = new MemoryStream(Encoding.ASCII.GetBytes(text));	        
			stream = new MimeBoundaryInputStream(mem, "boundary");
			
			Assert.AreEqual( stream.ReadByte(), -1);
		}
	    
		/**
		* Tests that hasMoreParts behave as expected.
		*/
		[Test]
		public void test_HasMoreParts() {
			String text = "--boundary--\r\n";

			MemoryStream mem = new MemoryStream(Encoding.ASCII.GetBytes(text));
			MimeBoundaryInputStream stream = new MimeBoundaryInputStream(mem, "boundary");

			Assert.AreEqual(stream.HasMoreParts, false);
			Assert.AreEqual( stream.ReadByte(), -1);
		}
	    
		/**
		* Tests that a stream containing only a boundary is empty.
		*/
		[Test]
		public void test_PrefixIsBoundary() {
			String text = "Line 1\r\n\r\n--boundaryyada\r\n";

			MemoryStream mem = new MemoryStream(Encoding.ASCII.GetBytes(text));
			MimeBoundaryInputStream stream = new MimeBoundaryInputStream(mem, "boundary");
	        
			Assert.AreEqual( Read(stream), "Line 1\r\n" );
	        
			text = "--boundaryyada\r\n";
			mem = new MemoryStream(Encoding.ASCII.GetBytes(text));	        
			stream = new MimeBoundaryInputStream(mem, "boundary");

			Assert.AreEqual( stream.ReadByte(), -1);
		}    
	}


}