/*******************************************************}
{                                                       }
{       RichView                                        }
{       Demo: URL detection                             }
{                                                       }
{       Copyright (c) Sergey Tkachenko                  }
{       svt@trichview.com                               }
{       http://www.trichview.com                        }
{                                                       }
{*******************************************************/

/*
  This demo shows:
  - how to detect all URLs in text;
  - how to detect URLs in pasted text;
  - how to detect URLs on typing;
  - how to close hyperlink when user presses Space, Enter or punctuation character;
  - how to remove hyperlink using popup menu;
  - how to implement hypertext in editor without using Ctrl key;
  - how to modify hyperlink's tag when its visible text is changed;
  - how to remove hyperlink when its visible text was changed to non-URL;
  - how to display hyperlinks targets in hints.
*/

//---------------------------------------------------------------------------
#include <vcl\vcl.h>
#include <ShellApi.h>
#pragma hdrstop

#include "Unit1.h"
#include "URLScan.hpp"
#include "ForceHypertext.hpp"
#include "RVLinear.hpp"
//---------------------------------------------------------------------------
#pragma link "RVEdit"
#pragma link "RichView"
#pragma link "RVScroll"
#pragma link "RVStyle"
#pragma link "RVUni"
#pragma resource "*.dfm"
TForm1 *Form1;
//---------------------------------------------------------------------------
__fastcall TForm1::TForm1(TComponent* Owner)
    : TForm(Owner)
{
}
//---------------------------------------------------------------------------
void __fastcall TForm1::FormCreate(TObject *Sender)
{
  // Uncomment this line to test working with Unicode
  // RVStyle1->TextStyles->Items[0]->Unicode = true;
  rve->Clear();
  rve->AddNLWTag("This demo shows how to implement URL scanning",0,0,0);
  rve->AddNLWTag("Click the button, and URLs (like this - \"www.richedit.com/\") will be highlighted.",0,0,0);
  rve->AddNLWTag("Ctrl+click URL to launch browser or e-mail client",0,0,0);
  rve->Format();
  Application->OnHint = DisplayHint;
}
//---------------------------------------------------------------------------
// Opening text or RTF file
void __fastcall TForm1::btnOpenClick(TObject *Sender)
{
  if (! od->Execute())
    return;
  Screen->Cursor = crHourGlass;
  rve->Clear();
  rve->DeleteUnusedStyles(true,true,true);
  bool r = false;
  switch (od->FilterIndex)
  {
   case 1:
     r = rve->LoadText(od->FileName,0,0,false);
     break;
   case 2:
     r = rve->LoadRTF(od->FileName);
     break;
  }
  rve->Format();
  Screen->Cursor = crDefault;
  if (!r)
    Application->MessageBox(L"Error loading file", L"Error", MB_OK | MB_ICONSTOP);
}
//---------------------------------------------------------------------------
// Removing all hyperlinks, then rescanning the whole document for URLs.
//  Clearing undo buffers.
void __fastcall TForm1::btnScanClick(TObject *Sender)
{
  Screen->Cursor = crHourGlass;
  // Moving caret to the beginning
  rve->SetSelectionBounds(0, rve->GetOffsBeforeItem(0), 0, rve->GetOffsBeforeItem(0));
  // Clearing undo/redo buffers
  rve->ClearUndo();
  // removing old hypertext links
  bool r = ClearHypertext(rve->RVData, URLScanEvent, true);
  // scanning for URLs
  r = ScanURLs(rve->RVData, URLScanEvent, true) || r;
  if (r)
    rve->Format();
  Screen->Cursor = crDefault;
  rve->SetFocus();
}
//---------------------------------------------------------------------------
// Callback procedure, called from ClearHypertext and ScanURLs. Also used in
//  rve.OnItemTextEdit and OnPaste events.
//  Returns index of style (NewStyleNo) for converting the original style
//  (OldStyleNo) to.
//  If ToHypertext=True, this procedure converts style to hyperlink.
//  If ToHypertext=False, this procedure converts style to normal text. }
void __fastcall TForm1::URLScanEvent(int OldStyleNo, int& NewStyleNo,
                                     bool ToHypertext)
{
  // Constructing the desired style
  TFontInfo* Style = new TFontInfo(NULL);
  Style->Assign(RVStyle1->TextStyles->Items[OldStyleNo]);
  Style->Jump = ToHypertext;
  if (ToHypertext)
  {
    // Hypertext links will be blue and underlined
    Style->Style << fsUnderline;
    Style->Color = clBlue;
    Style->JumpCursor = (TCursor)crJump;
  }
  else
  {
    // Plain text will be black and not underlined
    Style->Style >> fsUnderline;
    Style->Color = clWindowText;
  }
  // May be such style already exists?
  NewStyleNo = RVStyle1->TextStyles->FindSuchStyle(OldStyleNo,Style,RVAllFontInfoProperties);
  if (NewStyleNo==-1)
  {
    // Not exists, adding...
    RVStyle1->TextStyles->Add()->Assign(Style);
    NewStyleNo = RVStyle1->TextStyles->Count-1;
    RVStyle1->TextStyles->Items[NewStyleNo]->Standard = false;
  }
  delete Style;
}
//---------------------------------------------------------------------------
// OnJump event. Called when the user clicks hyperlink in standard hypertext mode
//  (holding Ctrl key)
void __fastcall TForm1::rveJump(TObject *Sender, int id)
{
  TCustomRVFormattedData* RVData;
  int ItemNo;
  rve->GetJumpPointLocation(id, RVData, ItemNo);
  UnicodeString url = (wchar_t*)(RVData->GetItemTag(ItemNo));
  ShellExecute(0, L"open", url.w_str(), NULL, NULL, SW_SHOW);
}
//---------------------------------------------------------------------------
// OnRVMouseUp event. We use it to process click on hyperlink in the simple click
//  mode (without using Ctrl key)
void __fastcall TForm1::rveRVMouseUp(TCustomRichView *Sender,
      TMouseButton Button, TShiftState Shift, int ItemNo, int X, int Y)
{
  if (!SimpleClickHypertext || Button!=mbLeft || Sender->SelectionExists())
    return;
  TPoint pt = Sender->ClientToDocument(Point(X,Y));
  TCustomRVFormattedData* LRVData;
  int LItemNo, LOffs;
  if (Sender->GetItemAt(pt.x, pt.y, LRVData, LItemNo, LOffs, true) &&
     LRVData->GetItem(LItemNo)->GetBoolValueEx(rvbpJump, Sender->Style))
  {
    UnicodeString url = (wchar_t*)(LRVData->GetItemTag(LItemNo));
    ShellExecute(0, L"open", url.w_str(), NULL, NULL, SW_SHOW);
  }
}
//---------------------------------------------------------------------------
void __fastcall TForm1::rveKeyDown(TObject *Sender, WORD &Key,
	TShiftState Shift)
{
  if (Key==VK_SPACE || Key==VK_RETURN || Key==VK_TAB || Key==';' || Key == ',' )
  {
    // url detection
    if (cbAutodetect->Checked)
      DetectURL(rve, URLScanEvent, true);
    // closing url if necessary
    TerminateHyperlink(rve, URLScanEvent, Key!=VK_RETURN);
  }
}
//---------------------------------------------------------------------------
// OnItemTextEdit. Updating tag when text of hyperlink is edited.
//  Removing hyperlink when the text is not an URL any more (if cbAutoremove is
//  checked
void __fastcall TForm1::rveItemTextEdit(TCustomRichViewEdit *Sender,
      const TRVRawByteString OldText, TCustomRVData *RVData, int ItemNo,
      int &NewTag, int &NewStyleNo)
{
  if (!NewTag)
    return;
  int StyleNo = RVData->GetItemStyle(ItemNo);
  UnicodeString OldText2;
  if (Sender->Style->TextStyles->Items[StyleNo]->Unicode)
    OldText2 = RVU_RawUnicodeToWideString(OldText);
  else
    OldText2 = RVU_RawUnicodeToWideString(
      RVU_AnsiToUnicode(RVData->GetStyleCodePage(StyleNo), OldText));
  UnicodeString NewText = RVData->GetItemTextW(ItemNo);
  if (cbAutoremove->Checked && NewText=="")
  {
    // If new text is empty, removing hyperlink
    NewTag = 0;
    URLScanEvent(NewStyleNo, NewStyleNo, false);
    return;
  }
  if (wcscmp(OldText2.w_str(), (wchar_t*)NewTag)==0)
  {
    // If text before editing was equal to tag ...
    if (cbAutoremove->Checked &&
        (RVIsURL(OldText2) || RVIsEmail(OldText2)) &&
        !(RVIsURL(NewText) || RVIsEmail(NewText)))
    {
      // ... if text is not URL any more, removing hyperlink
      NewTag = 0;
      URLScanEvent(NewStyleNo, NewStyleNo, false);
      return;
    }
    // ... update tag to new text
    NewTag = (int)StrNew(NewText.w_str());
  }
}
//---------------------------------------------------------------------------
// OnItemHint. Displaying tag strings in hints
void __fastcall TForm1::rveItemHint(TCustomRichView *Sender,
      TCustomRVData *RVData, int ItemNo, UnicodeString &HintText)
{
  HintText = (wchar_t*)(RVData->GetItemTag(ItemNo));
}
//---------------------------------------------------------------------------
// Switching the standard and the simple-click hypertext modes.
//  SimpleClickHypertext is defined in ForceHyperText.pas
void __fastcall TForm1::cbUseCtrlClick(TObject *Sender)
{
  SimpleClickHypertext = !cbUseCtrl->Checked;
}
//---------------------------------------------------------------------------
// Disabling/enabling the popup menu items on popup
void __fastcall TForm1::PopupMenu1Popup(TObject *Sender)
{
  mitRemoveHyperlink->Enabled = rve->TopLevelEditor->GetItem
    (rve->TopLevelEditor->CurItemNo)->GetBoolValueEx(rvbpJump, RVStyle1);
}
//---------------------------------------------------------------------------
// Removing hyperlink at the caret position
void __fastcall TForm1::mitRemoveHyperlinkClick(TObject *Sender)
{
  TCustomRichViewEdit* toprve = rve->TopLevelEditor;
  if (!toprve->GetItem(toprve->CurItemNo)->GetBoolValueEx(rvbpJump, RVStyle1))
    return;
  int LPos = RVGetLinearCaretPos(toprve);
  toprve->SetSelectionBounds(
    toprve->CurItemNo, toprve->GetOffsBeforeItem(toprve->CurItemNo),
    toprve->CurItemNo, toprve->GetOffsAfterItem(toprve->CurItemNo));
  toprve->BeginUndoGroup(rvutTag);
  toprve->SetUndoGroupMode(True);
  try
  {
    toprve->SetCurrentTag(0);
    int LStyleNo;
    URLScanEvent(toprve->CurItemStyle, LStyleNo, false);
    toprve->ApplyTextStyle(LStyleNo);
  }
  __finally
  {
    toprve->SetUndoGroupMode(False);
  }
  RVSetLinearCaretPos(toprve, LPos);
}
//---------------------------------------------------------------------------
void __fastcall TForm1::DisplayHint(TObject* Sender)
{
  StatusBar1->SimpleText = GetLongHint(Application->Hint);
}
//---------------------------------------------------------------------------
// Detecting URLs on pasting plain text (only if RVF or RTF is not available)
void __fastcall TForm1::rvePaste(TCustomRichViewEdit *Sender,
      bool &DoDefault)
{
  if (cbPasteDetect->Checked)
    DoDefault = !PasteTextWithURLs(Sender, URLScanEvent);
}
//---------------------------------------------------------------------------
