Program TapeSch;
{
  Tape Scheduler

  Borland Pascal 7.0
  (c) Reinoud van Leeuwen 1993
  E-mail: Leeuwen@rullf2.Leidenuniv.nl

  Usage:  TapeSch /n=<n> [/t=hh:mm] [/f=<filename>] [/p]

  Output: (to screen) 'put tape <n> in tapestreamer and press a key'

  This simple tape scheduler program calculates a tape number, roulating
  every n weeks. It returns this number, multiplied by 10, and adds 1 on
  mondays. This number is returned in the errorlevel for usage in batch
  files.

  Example:
  tapesch /n=3      returns
  11 on mondays weeks 1,4,7,...
  10 other days weeks 1,4,7,...
  21 on mondays weeks 2,5,8,...
  20 other days weeks 2,5,8,...
  31 on mondays weeks 3,6,9,...
  30 other days weeks 3,6,9,...

  Errorlevel 0 indicates some kind of error (mostly in the options of tapesch)

  The '/n=' option is required and indicates the number of tapes to roulate.

  Other options

  /t=hh:mm      wait until time hh:mm to continue (to enable batch files to make
                backups in the night)

  /f=<filename> filename containing descriptions for inserting tape in another
                language. Each line a different message. Format
                Line 1: 'Tape roulating system'
                Line 2: 'Press a key to continue'
                Line 3: 'Insert tape 1 (The BLUE one)'
                Line 4: 'Insert tape 2 (The RED one)'
                etc.

  /p            Enables the user to cancel the wait by pressing a key

  -----------------------------------------------------------------------------
  The usual disclaimers apply; anyone uses this program on his own risk. Anyone
  may distribue this program or alter it, but I would like it if your document
  changes, and leave my name in it.
  -----------------------------------------------------------------------------
 }

Uses Crt, Dos;

Const
  Allowed_Options : set of Char = ['N', 'T', 'F', 'P', 'H', '?']; { The allowed command-line options }
  Switch_Chars    : set of Char = ['/','-'];                      { Usable switch chars              }
  No_Options                    = 1;                              { Help numbersu                    }
  Wrong_Option_Format           = 2;
  Non_Allowed_Option            = 3;
  Wrong_Time_Format             = 4;
  General_help                  = 5;
  N_Required                    = 6;



Procedure ErrorMsg (Number : Integer;
                    Op_Str : String);
{ Procedure     ErrorMsg

  Description   Write a help text to the screen and quit program
                with errorlevel 0

  Inputs        Number : Integer; number of helpscreen (defined in global
                                  Const section)
                Op_str : String;  String that is displayed in some helpscreens
                                  (mostly wrong options)

  Output        (to screen) selected help screen

  Side effects  PROGRAM TERMINATES WITH ERRORLEVEL 0
}
Begin
  Writeln;
  Writeln ('TapeSch 1.0');
  Writeln ('(c) 1993 R. van Leeuwen');
  Writeln ('tapesch /h for help');
  Writeln;
  Case Number Of
    No_Options,
    General_help        : Begin
                           Writeln (' Usage:');
                           Writeln ('TapeSch /n=<n> [/t=hh:mm [/p]] [/f=<filename>]');
                           Writeln ;
                           Writeln ('Where');
                           Writeln ('(Required)   <n>        is the number of roulating tapes');
                           Writeln ('(Optional)   hh:mm      is the time to wait until');
                           Writeln ('(Optional)   <filename> is a file containing desriptions of the tapes,');
                           Writeln ('                        Each line containing a desscription like:');
                           Writeln ('                        "Put the blue tape in the tapestreamer"');
                           Writeln ('with /p the user can cancel the wait by pressing a key');
                           Writeln;
                           Writeln ('The program will return a errorlevel for use in batch files:');
                           Writeln ('The erorlevel will be the calculated tape number, multiplied by 10');
                           Writeln ('plus 1 when on a monday.');
                           Writeln ('e.g. 11 = week 1, monday');
                           Writeln ('     10 = week 1, other day');
                           Writeln ('     21 = week 2, monday');
                           Writeln;
                           Writeln (' NO OPTIONS SPECIFIED: ERRORLEVEL 0 RETURNED');
                          End;
    Wrong_Option_Format : Begin
                           Writeln ('ERROR: Wrong option format: ',Op_str);
                           Writeln ('Format the options like /n=3 /t=04:30 /f=myfile.des');
                           Writeln;
                           Writeln ('OPTIONS WRONG SPECIFIED: ERRORLEVEL 0 RETURNED');
                          End;
    Non_Allowed_Option  : Begin
                           Writeln ('ERROR: Non allowed option: ',Op_str);
                           Writeln ('Allowed options are: /n /t /f /p /h /?');
                           Writeln;
                           Writeln ('OPTIONS WRONG SPECIFIED: ERRORLEVEL 0 RETURNED');
                          End;
    Wrong_Time_Format   : Begin
                           Writeln ('ERROR: Wrong time format: ', Op_Str);
                           Writeln ('Correct usage: /t=hh:mm');
                           Writeln ('0 <= hh <= 24');
                           Writeln ('0 <= mm <= 59');
                           Writeln ('If hh is 24, then mm must be 00');
                           Writeln;
                           Writeln ('OPTIONS WRONG SPECIFIED: ERRORLEVEL 0 RETURNED');
                          End;
    N_Required          : Begin
                           Writeln ('ERROR: required /n option missing');
                           Writeln ('OPTIONS WRONG SPECIFIED: ERRORLEVEL 0 RETURNED');
                          End;
  End; {Case}
  Halt (0);   { END OF PROGRAM !!!!! }
End;




Procedure Parse_options (Var NumberOfTapes : Integer;
                         Var Wait          : Boolean;
                         Var Time          : Integer;
                         Var FileOption    : Boolean;
                         Var FileName      : PathStr;
                         Var PressKey      : Boolean);
{ Procedure      Parse_options

  Description    Parses comman-line options and returns them in VAR parameters.
                 Allowed options and switchchars are defined in sets in global
                 Const section. Command line options may be in any order.

  Inputs/Outputs NumberOfTapes : Integer; number of tapes that are used to
                                          roulate
                 Wait          : Boolean; True if time option is used
                 Time          : Integer; Time to wait until
                 FileOption    : Boolean; True if File option is used
                 FileName      : PathStr; Expaneded filename for descriptions
                 PressKey      : Boolean; True when presskey option is used

  Side effects   none
}

Var
  n_option  :   Boolean;
  Count,
  Hours,
  Minutes,
  ErrCode1,
  ErrCode2  :   Integer;
  Tempstr   :    String;
  DirInfo   : SearchRec;

Begin
  n_option   := False;  { /n option not yet set }
  Wait       := False;  { /t option not yet set }
  FileOption := False;  { /f option not yet set }
  PressKey   := False;  { /p option not yet set }

  If ParamCount = 0 Then
    ErrorMsg (No_Options, '');

  For Count := 1 To ParamCount Do
  Begin
    Tempstr := ParamStr (Count);
    If Not (Tempstr [1] in Switch_Chars)  Then
      ErrorMsg (Wrong_Option_Format, Tempstr);

    If Not (UpCase (Tempstr [2]) In Allowed_Options) Then
      ErrorMsg (Non_Allowed_Option, Tempstr);

    If (Tempstr [3] <> '=') And
       (Length (Tempstr) > 2) Then
      ErrorMsg (Wrong_Option_Format, Tempstr);

    Case UpCase (Tempstr [2]) Of
      'N': Begin
             n_option := True;
             Val (Copy (Tempstr, 4, Length (Tempstr) -3), NumberOfTapes, ErrCode1);
             If ErrCode1 <> 0 Then
               ErrorMsg (Wrong_Option_Format, TempStr);
           End;

      'T': Begin
             Wait := True;
             Val (Copy (Tempstr, 4, 2), Hours,   ErrCode1);
             Val (Copy (Tempstr, 7, 2), Minutes, ErrCode2);
             If (ErrCode1    <>   0) Or
                (ErrCode2    <>   0) Or
                (Tempstr [6] <> ':') Or
                (Hours       <    0) Or
                (Hours       >   24) Or
                (Minutes     <    0) Or
                (Minutes     >   59) Or
               ((Hours = 24) And (Minutes > 0)) Then
               ErrorMsg (Wrong_Time_Format, TempStr);

             Time := hours * 100 + minutes
           End;

      'F': Begin
             FileOption := True;
             FileName := FExpand (Copy (Tempstr, 4, Length (Tempstr) -3));
             FindFirst(FileName, AnyFile, DirInfo);
             If DosError <> 0 Then
             Begin
               Writeln ('ERROR: File not found: ', TempStr);
               Writeln;
               Writeln ('Continuing without descriptions');
               FileOption := False;
             End;
           End;

      'P' : PressKey := True;

      'H',
      '?': ErrorMsg (General_help, '');
    End; { Case }
  End; { For Count = 1 To ParamCount Do }

  If Not n_Option Then
    ErrorMsg (N_Required, '');
End; { Procedure Parse_options }






Procedure Calc_Weeknumber (Var WeekNumber : Integer;
                           Var SpecialDay : Boolean);
{ Procedure      Calc_Weeknumber

  Description    Calculate the weeknumber (1st week of 1992 is 1)

  Inputs/Outputs WeekNumber : Integer; The calculated week number
                 SpecialDay : Boolean; True on mondays, false on other days

  Side effects   none
}

Const
  MonthDays : Array [1..12] of Shortint = (31,28,31,30,31,30,31,31,30,31,30,31);

Var
  Year,
  Month,
  Day,
  DayOfWeek : Word;
  Count,
  Leaps,
  This_Leap,
  Before_this_month,
  CalcWeek,
  CalcDOW,
  CalcDays  : Integer;

Begin
  GetDate (Year, Month, Day, DayOfWeek);
  Year  := Year - 1992;
  Leaps := (Year - 1) Div 4;

  If ((Year Mod 4) = 0) And (Month > 2) Then
    This_Leap := 1
  Else
    This_Leap := 0;

  Before_this_month := 0;

  For Count := 1 to Month - 1 Do
    Before_this_month := Before_this_month + MonthDays [Count];

  { - 4 because of the first monday in 1992  }
  CalcDays := (Year * 365) + Leaps + This_Leap + Before_this_month + Day - 4;
  WeekNumber := CalcDays div 7;
  CalcDOW  := CalcDays Mod 7;
  If CalcDOW = 1 Then
    SpecialDay := True
  Else
    SpecialDay := False;
End;




Procedure DispMsg (TapeNumber : Integer;
                   FileOption : Boolean;
                   Filename   : PathStr);
{ Procedure    DispMsg

  Description  Displays message (to screen) prompting user to insert the right
               tape. The user has to press a key to continue. If FileOption is
               True, the displayed text is read from the Filname file. In this
               file, on the first line should be a message like 'tape roulating
               system'; on the second line a message like 'press a key to
               continue'; and on the other lines messages like 'put tape 1 in
               the tapestreamer'.
               NOTE: when you translate messages please use something like
               'press a key' instead of 'press any key'. Some users have spent
               hours searching for the 'any-key'!

  Inputs       TapeNumber : Integer; The number of the tape to insert.
               FileOption : Boolean; Switches between internal or external
                                     messages
               Filename   : PathStr; file with messages

  Output       message to screen

  Side effects Program waits until user presses a key
}

Var
  InFile   : Text;
  Line1,
  Line2,
  TapeLine : String;
  Count    : Integer;
Begin
  ClrScr;
  If Not FileOption Then
  Begin
    Writeln;
    Writeln;
    Writeln;
    Writeln;
    Writeln;
    Writeln;
    Writeln ('         Ŀ');
    Writeln ('                      Tape roulation systeem 1.0                    ');
    Writeln ('                                                                    ');
    Writeln ('                                                                    ');
    Writeln ('                      Put Tape number ',Tapenumber:2,' in the tapestreamer        ');
    Writeln ('                                                                    ');
    Writeln ('                      and press a key to proceed with backup        ');
    Writeln ('                                                                    ');
    Writeln ('         ');
  End
  Else
  Begin
    Assign (InFile, FileName);
    Reset  (InFile);
    ReadLn (Infile, Line1);
    ReadLn (Infile, Line2);

    If Tapenumber = 1 Then
      ReadLn (Infile, TapeLine)
    Else
    For Count := 1 to Tapenumber Do
      ReadLn (Infile, TapeLine);

    Close (Infile);
    Writeln;
    Writeln (Line1);
    Writeln (TapeLine);
    Writeln (Line2);
  End;
  Repeat
  Until KeyPressed;
End;



Procedure WaitUntil (Time     : Integer;
                     PressKey : Boolean);
{ Procedure    WaitUntil

  Description  Waits until the givven time. When presskey is True, the user
               is able to skip waitng by pressing a key. The Time parameter
               is not validated!

  Inputs       Time     : Integer;  The time to wait to.
                                    Format: hours * 100 + minutes; 23:52 becomes
                                    2352
               PressKey : Boolean;  When true, the user may skip waiting by
                                    pressing a key

  Output       (to screen) message "awaiting backup at <time>"

  Side effects program can wait for several hours
}

Var
  Hour,
  Minute,
  Second,
  Sec100   :    Word;
  Its_Time : Boolean;
  Dummy    :    Char;
Begin
  If KeyPressed Then            { Empty keyboard buffer}
    Repeat
      Dummy := ReadKey;
    Until Not KeyPressed;

  Its_Time := False;
  ClrScr;
  Repeat
    GetTime ( Hour, Minute, Second, Sec100);
    If (Hour * 100 + Minute) <> Time Then
    Begin
      GotoXY (1, Random (24));
      Writeln ('Awaiting backup at ',Time Div 100,':',Time Mod 100, ' (Time now is ',Hour,':',Minute,')');

      If PressKey Then
        Writeln ('Press a key to start NOW');

      Delay (5000); { 5 secs }
      ClrScr;

      If KeyPressed And PressKey Then
        Its_Time := True;
    End
    Else
      Its_Time := True;
  Until Its_Time
End;



Var                  { Variables for main program }
  WeekNumber,
  Time,
  TapeNumber,
  ReturnErrorLevel,
  NumberOfTapes    :   Integer;
  PressKey,
  SpecialDay,
  Wait,
  FileOption       :   Boolean;
  FileName         :   PathStr;

Begin { Main Program }
  Parse_options (NumberOfTapes, Wait, Time, FileOption, Filename, PressKey);

  Calc_Weeknumber (WeekNumber, SpecialDay);

  TapeNumber := (Weeknumber Mod NumberOfTapes) + 1;   { + 1 because range is 1..n instead of 0..n-1 }

  DispMsg (TapeNumber, FileOption, Filename);

  If Wait Then
    WaitUntil (Time, PressKey);

  ReturnErrorLevel := TapeNumber * 10;
  If SpecialDay Then                    { Mondays + 1 }
    Inc (ReturnErrorLevel);

  Halt (ReturnErrorLevel);
End.
