/*++
/* NAME
/*      logs 3
/* SUMMARY
/*      error logging, status reports, debugging
/* PROJECT
/*      pc-mail
/* PACKAGE
/*      cico
/* SYNOPSIS
/*      void dbg(fmt[,args]);
/*      char *fmt;
/*
/*      int open_log();
/*
/*      void log(fmt[,args]);
/*      char *fmt;
/*
/*      void trap(code,fmt[,args]);
/*      char *fmt;
/* DESCRIPTION
/*      All functions in this module do some form of logging, and accept
/*      printf-like format strings with %s, %c, and %S, %C. The latter
/*	two cause output mapping of arbitrary byte values to printable codes.
/*
/*      dbg() formats its arguments and writes the result to the standard
/*      output.
/*
/*      open_log() tries to open the logfile for writing. It returns
/*      a status E_WRITERR if the file could not be opened or created.
/*
/*      log() writes status info to the log file. If debugging is enabled,
/*	the message is also written to the standard output.
/*
/*      trap() writes a message to the log file and performs a longjmp call
/*      (systrap) with the status as given in the code parameter. If 
/*	debugging is enabled, the message is also written to the standard 
/*	output.
/* FUNCTIONS AND MACROS
/*      longjmp()
/* FILES
/*      LOGFILE         status reports
/* BUGS
/*      Logfile info may be lost if the program terminates abnormally.
/*      We do not open/close the with each log() call since that would
/*      slow down performance on floppy-based systems dramatically.
/* AUTHOR(S)
/*      W.Z. Venema
/*      Eindhoven University of Technology
/*      Department of Mathematics and Computer Science
/*      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*      Thu Mar 26 17:45:19 GMT+1:00 1987
/* LAST MODIFICATION
/*	Mon Apr  4 23:44:00 MET 1988
/* VERSION/RELEASE
/*	1.3
/*--*/

#include <setjmp.h>
#include <ctype.h>
#include <varargs.h>
#include "defs.h"
#include "logs.h"
#include "path.h"
#include "status.h"

#define	dbgout	stdout		/* where debugging output should go */

hidden FILE *logfp = NULL;	/* log file file pointer */
hidden char *visible();		/* map characters to readable codes */
hidden void dprintf();		/* special-purpose formatting function */

/* dbg - write debugging info to the debugging output */

/* VARARGS1 */

public void dbg(fmt,va_alist)
register char *fmt;
va_dcl
{
    va_list s;

    va_start(s);
    dprintf(dbgout,fmt,s);
    va_end(s);
}

/* open_log - check the logfile can be written */

public int open_log()
{
    if (logfp == NULL && (logfp = fopen(logfile(),"a")) == NULL)
	return(E_WRITERR);
    else
	return(0);
}

/* log - write status info to the log file */

/* VARARGS1 */

public void log(fmt,va_alist)
register char *fmt;
va_dcl
{
    va_list s;

    /* log file should be open! */

    if (logfp == NULL)
	exit(E_CONFUSED);

    /* write status record to log file */

    va_start(s);
    dprintf(logfp,fmt,s);
    putc('\n',logfp);
    va_end(s);

    /* if debugging on, write also to debugging output */

    if (dflag) {
	va_start(s);
	dprintf(dbgout,fmt,s);
	putc('\n',dbgout);
	va_end(s);
   }
}

/* trap - exception handler */

/* VARARGS2 */

public void trap(code,fmt,va_alist)
int code;
char *fmt;
va_dcl
{
    va_list s;

    /* write exception record to log file */

    va_start(s);
    dprintf(logfp,fmt,s);
    putc('\n',logfp);
    va_end(s);

    /* if debugging on, write also to debugging output */

    if (dflag) {
	va_start(s);
	dprintf(dbgout,fmt,s);
	putc('\n',logfp);
	va_end(s);
    }
    longjmp(systrap,code);
}

/* visible - turn arbitrary character into something visible */

static char *visible(c)
register int c;
{
    static char buf[5];

    switch(c&=0377) {
    default:
	sprintf(buf,isascii(c) && isprint(c) ? "%c" : "\\%03o",c);
	return(buf);
    case ' ':
	return("\\s");
    case '\b':
	return("\\b");
    case '\t':
	return("\\t");
    case '\r':
	return("\\r");
    case '\n':
	return("\\n");
    case '\f':
	return("\\f");
    case '\\':
	return("\\\\");
    }
}

/* dprintf - handle %s, %c, %S and %C format requests */

static void dprintf(fp,fmt,s)
register FILE *fp;
register char *fmt;
va_list s;
{
    register int c;

    for (/* void */; c = *fmt; fmt++) {
	if (c != '%') {
	    putc(c,fp);
	} else if ((c = *++fmt) == 'S') {		/* %S: translated */
	    register char *cp = va_arg(s,char *);
	    while(*cp)
		fputs(visible(*cp++&0377),fp);
	} else if (c == 'C') {				/* %C: translated */
	    fputs(visible(va_arg(s,int)),fp);
	} else if (c == 's') {				/* %s: string, as is */
	    fputs(va_arg(s,char *),fp);
	} else if (c == 'c') {				/* %c: char, as is */
	    putc(va_arg(s,int),fp);
	} else if (c == '%') {				/* real % character */
	    putc(c,fp);
	}
    }
}
