/*++
/* NAME
/*	kbdinp 3
/* SUMMARY
/*	keyboard interpreter
/* PROJECT
/*	pc-mail
/* PACKAGE
/*	mailsh
/* SYNOPSIS
/*	void kbdinp(screen)
/*	Screen *screen;
/*
/*	void kbdinit()
/*	void kbdrest()
/* DESCRIPTION
/*	The keyboard interpreter is the machine that executes the program
/*	that is recorded in the form of Screen data structures.
/*	Its task is to interpret keyboard input and to
/*	invoke the appropriate action functions.
/*
/*	Depending on the return value of an action function
/*	the keyboard interpreter i) returns (S_BREAK), ii) repaints the 
/*	screen (S_REDRAW), or iii) just waits for more keyboard 
/*	input. Error handling is entirely up to the action functions.
/*
/*	The routines in this module are responsible for what appears in the
/*	top (function-key labels) and bottom sections (command dialogue)
/*	of the tty screen.
/*
/*	The middle screen section is handled by the pager (except when 
/*	help info is displayed).
/*
/*	kbdinit() sets the tty driver and keypad modes (no echo,
/*	punctual input).
/*	kbrest() restores the modes to what they were before the
/*	program was entered.
/*
/*	Terminal-specific codes for function keys and keypad are borrowed 
/*	from window.c.
/* FUNCTIONS AND MACROS
/*	printcl(), printat(), putw(), printw(), beep(), winout()
/* SEE ALSO
/*	window(3)       window management routines, function key codes
/*	window(5)       window definitions
/*	screen(3)       command key tables for each screen
/*	screen(5)       structure of command key tables
/* DIAGNOSTICS
/*	It beeps when an illegal key is pressed. Otherwise, no error
/*	handling at all.
/* AUTHOR(S)
/*	W.Z. Venema
/*	Eindhoven University of Technology
/*	Department of Mathematics and Computer Science
/*	Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*	Thu Apr  2 18:43:12 GMT+1:00 1987
/* LAST MODIFICATION
/*	Mon Apr  4 23:42:29 MET 1988
/* VERSION/RELEASE
/*	1.3
/*--*/

#include <signal.h>
#include <ctype.h>
#include "defs.h"
#include "mailsh.h"
#include "screen.h"
#include "window.h"

#ifdef  unix
#   if (SIII||SYSV)				/* AT&T */
#	include <termio.h>
	struct termio oldmode;
#   else                                    	/* V7 or Berkeley */
#	include <sgtty.h>
	struct sgttyb oldmode;
#   endif
#endif

#define QUEST   "? "				/* prompt for input */
#define STRLEN  ((wsize[BOT]-1)*CO-sizeof(QUEST)-2) /* max. input length */

hidden void kb_help(),kb_pause();		/* forward declarations */
hidden int kb_paint();
hidden int kb_str(),kb_key(),kb_cr();
hidden int input(),isempty();			/* forward declarations */

hidden char sect[] = "===============================================================================";

/* kbdinp - recursively interpret screen descriptions */

public int kbdinp(screen)
Screen *screen;
{
    kb_paint(screen);

    if (iskey(screen->key))
	return(kb_key(screen));			/* single-key commands */
    else if (screen->key == STRING)
	return(kb_str(screen));			/* string input screen */
    else if (screen->key == ESCCR)
	return(kb_cr(screen));			/* confirm/cancel screen */
    else
	fatal("kbdinp");			/* unexpected screen type */
    /* NOTREACHED */
}

/* kb_paint - paint the screen (clean up this function) */

hidden int kb_paint(p)
register Screen *p;
{
    char topline[BUFSIZ];			/* key label line */
    register int k;				/* loop control variable */
    int stat;					/* status from mid window */
    int promptloc;				/* where prompt "?" goes */

    /*
    * The top section of the screen consists of one line with
    * key labels (in case of single-key input screen) and a
    * bar that separates this section from the middle screen section.
    *
    * We always add a Help and ? label. The keyboard interpreter
    * preempts the H and ? characters.
    */

    for (topline[0] = 0; p->key; p++) {		/* start top window */
	if (iskey(p->key)) {			/* keystroke input ? */
	    strcat(topline,p->name);		/* append key label */
	    strcat(topline,"  ");		/* and some blanks */
	} else if (topline[0]) {		/* leak if first entry */
	    fatal("mixed single-key and string input");
	}
    }
    printcl(TOP,0,topline);			/* display key labels */
    if (topline[0])				/* if there are labels */
	printw("Help  ?");			/* display help key label too */
    printcl(TOP,1,sect);			/* finish top window with bar */

    /*
    * The bottom section of the screen consists of a bar that separates
    * us from the middle section, followed by the "help" string in
    * the last entry of the current screen definition, followed by
    * (if not a single-key input screen) a prompting question mark.
    *
    * We display the middle window after doing most of the bottom 
    * section, so that the cursor can stay in the middle window in
    * case of single-key input screens.
    *
    * We display the prompt (no single-key input screens) in the
    * bottom section after doing the middle screen section, so that 
    * the cursor can stay together with the prompt.
    */

    printcl(BOT,0,sect);			/* start lower window */
    promptloc = printcl(BOT,1,p->help ? p->help : "")+1; /* general info */
    for (k = promptloc; k < wsize[BOT]; k++) 	/* clear rest of lower window */
	printcl(BOT,k,"");			/* lower window done */

    if (p->action)				/* fill middle window */
	stat = CALL(p->action)();		/* middle window done */

    if (topline[0] == '\0')			/* single-key screen? */
	printat(BOT,promptloc,QUEST);		/* output "?" prompt */

    return(stat);				/* from middle window filler */
}

/* kb_str - handle string input */

hidden int kb_str(p)
register Screen *p;
{
    char string[BUFSIZ];			/* a character buffer */
    register char *cp = string;			/* a character pointer */
    register int c;				/* a character */
    register int stat;

    for (;;) {
	if (!isascii(c = input())) {		/* ignore non-ascii codes */
	    beep();				/* complain */
	} else if (c == ESC) {			/* ESC means don't do it */
	    printw(" (ESC)");			/* confirm input */
	    return(0);				/* nothing left here to do */
	} else if ((c == ' ' || isprint(c)) && cp-string < STRLEN) {
	    putw(*cp++ = c);			/* accept/echo the character */
	} else if (cp > string && (c == BS || c == DEL)) {/* delete character */
	    cp--;				/* remove char from buffer */
	    printw("\b \b");			/* remove char from screen */
	} else if (c != ENTER || (*cp = 0,isempty(string))) {
	    beep();				/* complain */
	} else if (putw(c),((stat = CALL(p->action)(string)) & S_BREAK)) {
	    return(stat);			/* we're done here */
	} else if (stat & S_REDRAW) {		/* screen was changed */
	    kb_paint(p);			/* restore display */
	}
    }
}

/* kb_key - handle single-key input */

hidden int kb_key(p)
Screen *p;
{
    register int c;				/* a character */
    register Screen *q;				/* a screen (eh?) */
    register int stat;				/* a status */

    for (;;) {
	if ((c = getkey()) == '?' || c == 'H') {/* is it a help request */
	    kb_help(p);				/* yes, display key info */
	    continue;				/* skip rest of loop */
	}
	for (q = p; q->key && q->key != c; q++)	/* look key up in table */
	    /* void */;
	if (q->key == 0) {			/* unrecognized key */
	    beep();				/* complain */
	    continue;				/* skip rest of loop */
	} else if (q->action == 0) {		/* action-less key */
	    return(0);				/* we are done here */
	} else if ((stat = CALL(q->action)()) & S_BREAK) {/* action key */
	    return(stat);			/* we are done here */
	} else if (stat & S_REDRAW) {		/* screen was changed */
	    kb_paint(p);			/* restore screen */
	}
    }
}

/* kb_cr - handle escape/enter input */

hidden int kb_cr(p)
Screen *p;
{
    register int c;

    for (;;) {
	if ((c = input()) == ESC) {		/* don't do it */
	    printw(" (ESC)");			/* confirm input */
	    return(0);				/* we are done */
	} else if (c == ENTER) {		/* do the action */
	    register int stat = CALL(p->action)();
	    if (stat & S_BREAK) {		/* child kills parent */
		return(stat);			/* we are done */
	    } else if (stat & S_REDRAW) {	/* screen was changed */
		kb_paint(p);			/* restore screen */
	    }
	} else {				/* unacceptable input */
	    beep();				/* complain */
	}
    }
}

/* kb_help - display per-key help info; redraw screen when done */

hidden void kb_help(p)
register Screen *p;
{
    static char any[] = "Press any key to continue";
    register int k;

    for (k = 0; k < wsize[MID]; k++)		/* erase middle window */
	printcl(MID,k,"");
    for (k = 0; p[k].key; k++)			/* display key info */
	printcl(MID,k+1,strcons("   %-10s %s",p[k].name,p[k].help));
    for (k = 1; k < wsize[BOT]-1; k++)		/* erase bottom window */
	printcl(BOT,k,"");
    printcl(BOT,1,any);				/* press any key to continue */
    getkey();
    kb_paint(p);				/* redraw screen */
}

/* structure that associates token value with function-key strings */

typedef struct {
    int token;					/* key value */
    char **seq;					/* key string */
} Key;

hidden Key kv[] = {
    UP,     &KU,				/* key strings are set */
    DOWN,   &KD,				/* in window.c */
    LEFT,   &KL,
    RIGHT,  &KR,
    PGUP,   &PU,
    PGDN,   &PD,
    0,      0,
};

/* getkey - get key stroke, detect function keys, ignore case otherwise */

hidden int getkey()
{
    register int c;
    register Key *kp;
    char kstr[BUFSIZ];

    /* 
    * We assume that all function keys produce strings that start with 
    * the same character, and that those strings all have the same 
    * length. This is a reasonable assumption for cursor-control keys.
    */

    if ((c = input()) == *(kv[0].seq)[0]) {	/* lead-in char */
	register int lvl;
	for (lvl = 1; lvl < strlen(*(kv[0].seq)); lvl++)
	    kstr[lvl] = c = input();		/* read characters first */
	kstr[lvl] = '\0';
	for (kp = kv; kp->token; kp++)		/* then compare with strings */
	    if (strcmp(*(kp->seq)+1,kstr+1) == 0)
		return(kp->token);		/* return token value */
    }
    return(islower(c) ? toupper(c) : c);	/* return last character */
}

/* input - read one character without echoing or waiting for carriage return */

hidden int input()
{
    /*
    * On unix systems, the terminal driver has been instructed to
    * not echo and to return one character as soon as it comes available. 
    * Also the stdio routines have been instructed to work in an unbuffered 
    * fashion. See kbdinit().
    */

#ifdef	unix
    return(getchar());
#endif

    /*
    * On IBM-PC machines a function key produces a null character
    * followed by a scan code. We translate the null prefix to
    * an escape character since that is more like normal terminals do.
    * The trick is to find out when we read a null character whether it 
    * was produced by pressing a real function-key or by pressing ctrl-@.
    */

#ifdef	MSDOS
    register int c;
    return((c = getch()) ? c : kbhit() ? ESC : 0);
#endif
}

/* kbdinit - set input mode, turn keypad on */

public void kbdinit()
{
    /*
    * On unix systems, instruct the terminal driver to not echo
    * terminal input, and to return from a read as soon as one
    * character comes available.
    */

#ifdef  unix
# if (SIII||SYSV)
    struct termio newmode;			/* AT&T */

    ioctl(0,TCGETA,&oldmode);			/* save terminal mode */
    ioctl(0,TCGETA,&newmode);			/* get terminal mode */
    newmode.c_iflag &= ~(INLCR|ICRNL|IUCLC|BRKINT);
    newmode.c_oflag &= ~OPOST;
    newmode.c_lflag &= ~(ICANON|ISIG|ECHO);
    newmode.c_cc[4] = 1;			/* do single-character reads */
    ioctl(0,TCSETAF,&newmode);			/* set terminal mode */
# else
    struct sgttyb newmode;			/* V7 or Berkeley */

    gtty(0,&oldmode);				/* save terminal mode */
    gtty(0,&newmode);				/* get terminal mode */
    newmode.sg_flags |= RAW;			/* don't wait for newline */
    newmode.sg_flags &= ~(ECHO|CRMOD);		/* no echo, crlf mapping */
    stty(0,&newmode);				/* set terminal mode */
# endif
#endif

    signal(SIGINT,SIG_IGN);			/* ignore control-c */

#ifdef	unix
    setbuf(stdin,(char *) 0);			/* select unbuffered input */

    if (KS && *KS)				/* if there is a keypad */
	tputs(KS,1,fputchar);			/* enable it */
#endif
}

/* kbdrest - reset terminal driver to previous state, turn keypad off */

public void kbdrest()
{
#ifdef  unix
# if (SIII||SYSV)				/* AT&T */
    ioctl(0,TCSETAF,&oldmode);
# else						/* V7 or Berkeley */
    stty(0,&oldmode);				/* restore terminal mode */
# endif

    if (KE && *KE)				/* if there is a keypad */
	tputs(KE,1,fputchar);			/* disable it */
#endif
}

/* isempty - check a string is all blanks or empty */

hidden int isempty(s)
register char *s;
{
    return(*s == 0 || (isspace(*s) && isempty(s+1)));
}
