/*++
/* NAME
/*      invoke 3
/* SUMMARY
/*      system-dependent process control stuff
/* PROJECT
/*      pc-mail
/* PACKAGE
/*      mailsh
/* SYNOPSIS
/*      #include "status.h"
/*
/*	int invokelp(arg0,arg1,...)
/*      char *arg0,*arg1,...
/*
/*	int invokevp(argv)
/*	char **argv;
/*
/*	int onexit(command)
/*	char *command;
/* DESCRIPTION
/*      invokelp() creates a child process to execute a command.
/*      arg0, arg1,... is a null-terminated list of string pointers,
/*	the first being the name of the program. Use is made
/*	of the search path to locate the program in arg0.
/*
/*	invokevp() is similar to invokelp; the difference is that
/*	argv is an array of pointers to arguments.
/*
/*	onexit() executes a command, thereby terminating the current process.
/* DIAGNOSTICS
/*	invokelp(), invokevp() return the exit status of the child process, 
/*	E_SYSFAIL if there were insufficient resources, and
/*	E_NOPROG if the program in arg0 or argv[0] could not be found.
/*
/*	onexit() return -1 if there were problems.
/* BUGS
/*	The invokexx() functions should not be used if the command needs to 
/*	be handled by a command-language processor (e.g. shell built-ins,
/*	or i/o redirection).
/* AUTHOR(S)
/*      W.Z. Venema
/*      Eindhoven University of Technology
/*      Department of Mathematics and Computer Science
/*      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*      Sun Apr  5 15:27:37 GMT+1:00 1987
/* LAST MODIFICATION
/*	Wed Apr  6 00:19:56 MET 1988
/* VERSION/RELEASE
/*	1.4
/*--*/

#include <errno.h>
#include "defs.h"
#include "status.h"

#ifdef  MSDOS
#include <process.h>
#endif

/* invokelp - create child process to execute command */

/* VARARGS2 */

public int invokelp(arg0,arg1,arg2,arg3,arg4)
char *arg0,*arg1,*arg2,*arg3,*arg4;
{
    /*
    * On unix systems we fork a process and overlay the child with
    * the desired program.
    * This means we get -1 if the fork did not succeed, otherwise
    * the exit status if the child process. The code is a bit elaborate
    * since we want to handle various error conditions.
    */
#ifdef unix
    register int pid;

    if ((pid = fork()) < 0) {		/* fork off a process */
	return(E_SYSFAIL);		/* resources exhausted */
    } else if (pid == 0) {		/* this is the child process */
	execlp(arg0,arg0,arg1,arg2,arg3,arg4);/* try to replace it */
	_exit(errno == ENOENT ? E_NOPROG : E_SYSFAIL); /* sorry, failed */
	/* NOTREACHED */
    } else {
	int xstat,wstat;		/* wait till above child terminates */
	while ((wstat = wait(&xstat)) != -1 && wstat != pid)
	    /* void */ ;
	if (wstat == -1) {
	    return(E_SYSFAIL);		/* oops: no child! */
	} else if (xstat&0377) {
	    return(E_UNKNOWN);		/* child was killed */
	} else {
	    return(xstat>>8);		/* child died naturally */
	}
	/* NOTREACHED */
    }
#endif

    /*
    * On MS-DOS systems we try to avoid the command.com shell because
    * it always returns a zero status code. 
    */
#ifdef MSDOS
    int stat;
    char *p;

    if ((stat = spawnlp(P_WAIT,arg0,arg0,arg1,arg2,arg3,arg4)) < 0
    && errno == ENOENT 
    && (strcmp(p = arg0+strlen(arg0)-4,".bat") == 0 || strcmp(p,".BAT") == 0))
	stat = spawnlp(P_WAIT,"command","command","/c",arg0,arg1,arg2,arg3,arg4);
    return(stat >= 0 ? stat : (errno == ENOENT ? E_NOPROG : E_SYSFAIL));
#endif
}

/* invokelp - create child process to execute command */

public int invokevp(argv)
char **argv;
{
    /*
    * On unix systems we fork a process and overlay the child with
    * the desired program.
    * This means we get -1 if the fork did not succeed, otherwise
    * the exit status if the child process. The code is a bit elaborate
    * since we want to handle various error conditions.
    */
#ifdef unix
    register int pid;

    if ((pid = fork()) < 0) {		/* fork off a process */
	return(E_SYSFAIL);		/* resources exhausted */
    } else if (pid == 0) {		/* this is the child process */
	execvp(*argv,argv);		/* try to replace it */
	_exit(errno == ENOENT ? E_NOPROG : E_SYSFAIL); /* sorry, failed */
	/* NOTREACHED */
    } else {
	int xstat,wstat;		/* wait till above child terminates */
	while ((wstat = wait(&xstat)) != -1 && wstat != pid)
	    /* void */ ;
	if (wstat == -1) {
	    return(E_SYSFAIL);		/* oops: no child! */
	} else if (xstat&0377) {
	    return(E_UNKNOWN);		/* child was killed */
	} else {
	    return(xstat>>8);		/* child died naturally */
	}
	/* NOTREACHED */
    }
#endif

    /*
    * On MS-DOS systems we try to avoid the command.com shell because
    * it always returns a zero status code. 
    */
#ifdef MSDOS
    int stat;

    return((stat = spawnvp(P_WAIT,*argv,argv)) >= 0 ? 
	stat : (errno == ENOENT ? E_NOPROG : E_SYSFAIL));
#endif
}

/* onexit - exec another command */

int onexit(command)
char *command;
{
    if (command && *command) {
#ifdef unix
	return(execlp("/bin/sh","sh","-c",command));
#endif

#ifdef MSDOS
	return(system(command));
#endif
    }
}
