/*++
/* NAME
/*      file 3
/* SUMMARY
/*      manipulate ordinary files
/* PROJECT
/*      pc-mail
/* PACKAGE
/*      mailsh
/* SYNOPSIS
/*      int file()
/*
/*	int junk_file()
/* DESCRIPTION
/*      file() is called when the user selected the "file" option
/*	in the main menu. The pager displays a directory listing on the
/*      screen. The user may then walk through the file display and
/*	select a file or directory. 
/*
/*	In case a regular file is selected the user may specify
/*	an action (send file as mail, delete file, display
/*	file, print file, copy file to workfile for subsequent
/*	editing etc.).
/*
/*	junk_file() should be invoked when the file display is no longer
/*	valid.
/* FILES
/*      By selecting directory files the user can walk through the whole
/*      file system.
/* SEE ALSO
/*      kbdinp(3), pager(3)
/* BUGS
/*	On some MS-DOS systems, we never get the ".." entry.
/*	This is an unsolved problem.
/*
/*	Changing the current directory may cause problems if
/*	the MAILDIR and EDITOR environment variables hold
/*	relative path names.
/*
/*	No provision to change default drive on MS-DOS systems.
/* AUTHOR(S)
/*      W.Z. Venema
/*      Eindhoven University of Technology
/*      Department of Mathematics and Computer Science
/*      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*      Sun Apr  5 17:18:06 GMT+1:00 1987
/* LAST MODIFICATION
/*	Mon Apr  4 23:40:13 MET 1988
/* VERSION/RELEASE
/*	1.3
/*--*/

#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include "defs.h"
#include "screen.h"
#include "pager.h"
#include "mailsh.h"
#include "dir.h"
#include "path.h"
#include "status.h"
#include "window.h"

extern struct tm *localtime();          /* std C library */

hidden File *bldlist();			/* forward declaration */
hidden int show_list();
hidden int pick_file();
hidden int delfile();
hidden int show_file();
hidden int rmfile();

hidden File *dfile = 0;			/* file listing */
hidden File *dirfile = 0;               /* directory listing */

/* file - called from keyboard interpreter, call keyboard driver */

public int file()
{
    static Screen screen[] = {
	'C',	"Close",        0,      initscreen,
	'P',	"Print",        print,	"Print file listing",
	'S',	"Save",         save,	"Save file listing to ordinary file",
	PGUP,	PgUp,		pu_pager,pageup,
	PGDN,	PgDn,		pd_pager,pagedn,
	UP,	"Up",           up_pager,csrup,
	DOWN,	"Down",         dn_pager,csrdn,
	ENTER,	"Enter",        pick_file,"Display file at cursor",
	0,	0,              show_list,
	"To display a file, select it with cursor keys, then press ENTER",
    };

    kbdinp(screen);			/* set up dialogue */
    close_pager(dirfile);		/* deallocate pager file */
    dirfile = 0;			/* say it's gone */
    return(S_REDRAW);			/* force screen update */
}

/* show_list - create or refresh file listing */

hidden int show_list()
{
    if (dirfile == 0) {			/* no pager file? */
	patience();
	dirfile = bldlist();		/* create one */
    } else {
	set_pager(dirfile);		/* select existing pager file */
    }
    ds_pager();				/* display the thing */
    return(0);				/* say screen is up-to-date */
}

/* bldlist - build file listing display */

hidden File *bldlist()
{
    register int dd = opendir(THISDIR);	/* start directory search */
    register File *pp = open_pager();	/* allocate pager file */
    register char *f;			/* file name pointer */

    while (f = readdir(dd)) {		/* read next file name */
	struct stat s;
	if (stat(f,&s) == 0) {		/* get file info (size, date) */
	    app_pager(pp,strcons("%-20s %9ld %s",f,
		s.st_size,tstamp(&s.st_mtime)));
	} else {
	    app_pager(pp,f);
	}
    }
    sort_pager(pp,FORW_SORT);		/* sort by file name */
    closedir(dd);			/* directory search done */
    return(pp);				/* current pager file */
}

/* pick_file - display selected file or change directory */

hidden int pick_file()
{
    static char botline[80];
    static Screen screen[] = {
	'C',	"Close",        0,      "Return to file listing",
	'D',	"Delete",	delfile,"Delete this file",
	'M',	"Mail",		mailfile,"Mail a copy of this file",
	'P',	"Print",        print,	"Print this file",
	'S',	"Save",         save,	"Save a copy to an ordinary file",
	'W',	"Work",		makework,"Save a copy to a work file",
	PGUP,	PgUp,		pu_pager,pageup,
	PGDN,	PgDn,		pd_pager,pagedn,
	UP,	"Up",           up_pager,csrup,
	DOWN,	"Down",         dn_pager,csrdn,
	0,	0,              show_file,
	botline,
    };
    struct stat s;

    if (scan_pager(dirfile,"%s",message) == 0) {	/* cannot read name */
	beep();						/* "notify" user */
	return(0);					/* nothing happened */
    } else if (access(message,04) || stat(message,&s)) {/* cannot read file */
	beep();						/* "notify" user */
	return(0);					/* nothing happened */
    } else if ((s.st_mode&S_IFMT) == S_IFDIR) {         /* directory file */
	chdir(message);					/* change directory */
	close_pager(dirfile);				/* purge display */
	dirfile = bldlist();				/* new file display */
	return(S_REDRAW);				/* update display */
    } else {                                            /* ordinary file */
 	sprintf(botline,"(Reading file \"%s\")",message);
	kbdinp(screen);					/* look at screen */
	close_pager(dfile);				/* and forget it */
	dfile = 0;					/* say it's gone */
	return(S_REDRAW);                               /* force redrawing */
    }
}

/* show_file - create or refresh display of selected file */

hidden int show_file()
{
    if (dfile) {
	set_pager(dfile);				/* select dispay */
    } else {
	if (rd_pager(dfile = open_pager(),message))	/* try to read file */
	    mesg_pager(dfile,m_msgread);		/* read error */
    }
    ds_pager();						/* rewrite screen */
    return(0);						/* screen up-to-date */
}

/* delfile - ask confirmation before removing a file */

hidden int delfile()
{
    static Screen screen[] = {
	ESCCR,	0,		rmfile,int_error,
	0,	0,		0,
	"Press ESC to cancel. Confirm with ENTER",
    };

    return(kbdinp(screen)|S_REDRAW);		/* "are you sure?" */
}

/* rmfile - actually remove file */

hidden int rmfile()
{
    if (unlink(message)) {			/* try to remove file */
	errdisp(E_UNLINK);			/* notify the user */
	return(S_BREAK|S_REDRAW);		/* redisplay, stop caller */
    } else {
	junk_file();				/* say file display outdated */
	return(S_BREAK);			/* terminate caller */
    }
}

/* junk_file - directory listing is no longer up to date */

public int junk_file()
{
    if (dirfile) {
	close_pager(dirfile);			/* close pager file */
	dirfile = 0;				/* say it is gone */
    }
    return(0);					/* in case we need that */
}
