/*++
/* NAME
/*	email
/* SUMMARY
/*	manipulate work files (mail in preparation)
/* PROJECT
/*	pc-mail
/* PACKAGE
/*	mailsh
/* SYNOPSIS
/*	#include "email.h"
/*
/*	int mail()
/* DESCRIPTION
/*      The functions in this module are responsible for manipulations 
/*	on work files and other mail messages in preparation.
/*
/*      mail() can be called either when the user has selected a work file in
/*	the mail box display or when the user wants to create a new letter.
/*
/*	The message file is displayed on the screen and the can choose
/*	to print, mail, edit or delete etc. the message.
/*
/*	The code in this module is a little tricky, to avoid that a work
/*	file exists without a metafile (for the mail box display). 
/* COMMANDS
/*	the program specified in the EDITOR environment variable,
/*	or a system-dependent default.
/* FILES
/*      temporary edit file in current directory
/*	work file and meta file in spool directory
/* SEE ALSO
/*      pager(3), pager(5), kbdinp(3), edit(3)
/* AUTHOR(S)
/*      W.Z. Venema
/*      Eindhoven University of Technology
/*      Department of Mathematics and Computer Science
/*      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*	Tue May 12 15:35:20 GMT+1:00 1987
/* LAST MODIFICATION
/*	Mon Apr  4 23:39:13 MET 1988
/* VERSION/RELEASE
/*	1.3
/*--*/

#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include "defs.h"
#include "path.h"
#include "dir.h"
#include "pager.h"
#include "screen.h"
#include "mailsh.h"
#include "status.h"

/* forward declarations */

hidden void junk_work();
hidden int queue_work();
hidden int edit_work();
hidden int show_work();
hidden int send_work();
hidden int hold_work();
hidden int label_work();

hidden File *workfile = 0;			/* pager file */

/* mail - user has made a choice. show message at cursor */

public int mail()
{
    static Screen screen[] = {
	'C',	"Close",	hold_work,"Send message later, return to main menu",
	'D',	"Delete",       delete,	delcurr,
	'E',	"Edit",		edit_work,"Edit this message",
	'M',	"Mail",		send_work,"Send this message to destination",
	'P',	"Print",	print,	printcurr,
	PGUP,	PgUp,		pu_pager,pageup,
	PGDN,	PgDn,		pd_pager,pagedn,
	UP,	"Up",           up_pager,csrup,
	DOWN,	"Down",         dn_pager,csrdn,
	0,	0,              show_work,
	"(Reading a message in preparation)",
    };
    struct stat s;

    if (stat(message,&s) && errno == ENOENT)	/* if new message file */
	edit_work();				/* invoke editor first */
    kbdinp(screen);				/* ask disposition */
    junk_work();				/* destroy mail pager file */
    return(S_REDRAW);				/* say screen was changed */
}

/* show_work - show work file or error message in middle window */

hidden int show_work()
{
    if (workfile) {				/* check pager file exists */
	set_pager(workfile);			/* select existent display */
    } else if (rd_pager(workfile = open_pager(),message)) {/* create display */
	mesg_pager(workfile,m_msgread);		/* cannot display message */
    }
    ds_pager();					/* (re)draw display */
    return(0);					/* screen is up-to-date */
}

/* junk_work - destroy work file display */

hidden void junk_work()
{
    if (workfile) {				/* no-op if no display */
	close_pager(workfile);			/* release memory */
	workfile = 0;				/* say it is gone */
    }
}

/* edit_work - edit or create a work file */

hidden int edit_work()
{
    register int stat;

    if (stat = edit(message,MAILFILE)) 
	errdisp(stat);				/* edit() had a problem */
    junk_work();				/* force new message display */
    return(S_REDRAW);				/* say screen has changed */
}

/* hold_work - stop editing but do not yet mail a work file */

hidden int hold_work()
{
    static Screen screen[] = {
	STRING,	0,              label_work,int_error,
	0,	0,              0,
	getsummary,
    };
    struct stat s;

    /*
    * The user has terminated an editor session, but does not yet want
    * to send the message off. The purpose of the following code is to
    * ask for a one-line summary, but only if such a comment does not yet
    * exist. The summary is used to identify the work file in the main
    * mail box display.
    */

    if (stat(message,&s) || !stat(comment,&s)) {/* no msg, or comment exists */
	return(S_BREAK);			/* we are done here */
    } else {
	return(kbdinp(screen)|S_REDRAW);	/* ask for a summary */
    }
}

/* label_work - save summary line to meta file */

hidden label_work(string)
char *string;
{
    register int stat;

    if (stat = metafile(string,comment)) {	/* try to create meta file */
	errdisp(stat);				/* oops, notify the user */
	return(S_REDRAW);			/* say screen has changed */
    } else {
	chmod(comment,0444);			/* make comments read-only */
	junk_desk();				/* say mail box has changed */
	return(S_BREAK);			/* say no more work */
    }
}

/* send_work - user wants to send work file, ask for destination */

hidden int send_work()
{
    static Screen screen[] = {
	STRING,	0,              queue_work,int_error,
	0,	0,              when,
	"Press ESC to cancel. Send message to:",
    };
    return(kbdinp(screen)|S_REDRAW);
}

/* queue_work - spool mail, delete work file and meta file */

hidden int queue_work(to)
char *to;
{
    register int stat;

    if (stat = submit(message,to)) {
	errdisp(stat);				/* cannot queue message */
	return(S_REDRAW);			/* say screen has changed */
    } else {
	return(unspool()|S_BREAK);		/* remove work and meta file */
    }
}
