/*++
/* NAME
/*	deskutil 3
/* SUMMARY
/*	utility functions
/* PROJECT
/*	pc-mail
/* PACKAGE
/*	mailsh
/* SYNOPSIS
/*	#include "mailsh.h"
/*
/*	void patience()
/*
/*	int when()
/*
/*	int delete()
/*
/*	int unspool()
/*
/*	int print()
/*
/*	int save()
/*
/*	char *tstamp(ltime)
/*	long *ltime()
/* DESCRIPTION
/*	tstamp() converts absolute time to a string. If called with
/*	recent time argument (less than 100 days ago) the string will
/*	be of the form "Sun Apr 17 12:50", otherwise "Sun Apr 17  1988".
/*
/*      delete() gives the user another chance before a message is deleted.
/*
/*      unspool() actually deletes a message. As a side effect it destroys
/*      the current mail box display so that the next display will
/*	reflect the actual status of the spool directory.
/*	The affected message and meta file names are taken from the
/*	global "message" and "commant" string variables.
/*
/*	print() copies a pager file to the printer.
/*
/*	save() asks where the pager file should be saved.
/*
/*	when() should be called after the user has entered a mail destination
/*	address. It informs the user that messages are not sent right away, 
/*	but after selection of the Network option in the main menu.
/*
/*	patience() prints a 'one moment please' message in the middle
/*	screen window. As a side effect, the current pager file is set
/*	to none.
/* FILES
/*      mail header files in the spool directory
/* SEE ALSO
/*      pager(3), pager(5), kbdinp(3)
/* AUTHOR(S)
/*      W.Z. Venema
/*      Eindhoven University of Technology
/*      Department of Mathematics and Computer Science
/*      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*	Tue May 12 15:35:20 GMT+1:00 1987
/* LAST MODIFICATION
/*	Mon Apr  4 23:38:26 MET 1988
/* VERSION/RELEASE
/*	1.3
/*--*/

#include <errno.h>

#include "defs.h"
#include "pager.h"
#include "mailsh.h"
#include "screen.h"
#include "status.h"

hidden int save_desk();

/* patience - say this will take some time */

public void patience()
{
    static char *m_wait[] = {
	"",
	"One moment please...",
	0,
    };

    register File *pp = open_pager();		/* create pager file */

    mesg_pager(pp,m_wait);			/* write pager file */
    ds_pager();					/* show om middle window */
    close_pager(pp);				/* forget pager file */
}

/* delete - user wants to delete a message; ask for confirmation */

public int delete()
{
    static Screen screen[] = {
	ESCCR,	"Enter",	unspool,int_error,
	0,	0,		0,
	"Press ESC to cancel. Confirm with ENTER",
    };

    return(kbdinp(screen)|S_REDRAW);
}

/* unspool - actually delete a message; force mail box display rebuild */

public int unspool()
{
    if (((chmod(message,0666) || unlink(message)) && errno != ENOENT)
    || ((chmod(comment,0666) || unlink(comment)) && errno != ENOENT)) {
	errdisp(E_UNLINK);			/* notify user of problem */
	return(S_REDRAW);			/* say screen has changed */
    } else {
	junk_desk();				/* say mail box has changed */
	return(S_BREAK);			/* no more work to do */
    }
}

/* print - print pager display on default printer */

public int print()
{
    return(pr_pager() ? (errdisp(E_PRINTERR),S_REDRAW) : 0);
}

/* save - ask where pager display should be copied to */

public int save()
{
    static Screen screen[] = {
	STRING,	0,              save_desk,int_error,
	0,	0,              0,
	"Press ESC to cancel. Save to file:",
    };

    kbdinp(screen);			/* prompt for file name, then copy */
    return(S_REDRAW);			/* force screen repaint */
}

/* save_desk - copy pager file to ordinary file */

hidden int save_desk(to)
char *to;
{
    if (cp_pager(to)) {			/* if file copy failed */
	unlink(to);			/* remove that file */
	errdisp(E_WRITERR);		/* notify the user */
	return(S_BREAK|S_REDRAW);	/* redisplay, terminate caller */
    } else {
	junk_file();			/* say file display maybe outdated */
	return(S_BREAK);		/* terminate caller */
    }
}

/* when - say when mail will actually be sent */

public int when()
{
    static char *msg[] = {
	"",
	"To send messages through the network, use the Network",
	"option in the main menu.",
	0,
    };
    File *pp = open_pager();			/* open new pager file */

    mesg_pager(pp,msg);				/* fill pager file */
    ds_pager();					/* display the file */
    close_pager(pp);				/* forget pager file */
    return(0);					/* don't care value */
}

/* tstamp - time format as produced by the ls(1) command */

public char *tstamp(ltime)
long *ltime;
{
    static char buf[25];

    /*
    * Output from asctime() is of the form
    *	"Sun Apr 17 13:34:35 1988"
    * Depending on how recent the time in question is, we
    * supress the time or year field.
    */

    (void) strcpy(buf,asctime(localtime(ltime)));
    if (time((long *)0)-*ltime > 60L*60L*24L*100L) {
	buf[24] = '\0';				/* remove the \n */
	(void) strcpy(buf+11,buf+19);		/* old file, show year */
    } else
	buf[16] = '\0';				/* recent, show time */
    return(buf);
}
