/*++
/* NAME
/*      connect 3
/* SUMMARY
/*      pre- and post protocol host access
/* PROJECT
/*      pc-mail
/* PACKAGE
/*      cico
/* SYNOPSIS
/*      int connect()
/*
/*      int disconnect()
/* DESCRIPTION
/*      connect() tries to make a connection to the remote host
/*      and to log on, using the dial-up script and login-name 
/*	entries in the communications parameter file, and the password
/*	provided as command-line parameter to the cico program. 
/*	A simple send-expect script facility is used (no retries 
/*	or delays). After the dial-up script has completed the program 
/*	proceeds by expecting the login: prompt and does the final login.
/*
/*      disconnect() tries to break a connection, using the disconnect
/*      entry in the communications parameter file. Unlike connect() 
/*	this function is not driven by a send-expect script.
/*
/*	The following escape sequences are recognized in send or expect
/*	strings:
/*
/* .nf
/*	\\b	backspace
/*	\\r	carriage return
/*	\\n	newline
/*	\\t	tab
/*	\\s	space
/*	\\f	form feed
/*	\\nnn	octal character value
/*	\\\\	a real backslash
/* FUNCTIONS AND MACROS
/*      xwrite(), xgetc(), trap(), debug(4)(), log()
/* FILES
/*      $MAILDIR/s00000		communications parameter file
/*	$MAILDIR/LOGFILE	system logfile
/* SEE ALSO
/*      params(5)       communications parameter file entries
/* DIAGNOSTICS
/*      connect() returns a status E_BADSETUP if the systems parameter
/*	file contains bad data, and E_NOLINE if the login script fails.
/* AUTHOR(S)
/*      W.Z. Venema
/*      Eindhoven University of Technology
/*      Department of Mathematics and Computer Science
/*      Den Dolech 2, P.O. Box 513, 5600 MB Eindhoven, The Netherlands
/* CREATION DATE
/*      Fri Mar 27 17:11:12 GMT+1:00 1987
/* LAST MODIFICATION
/*	Wed Apr  6 00:19:26 MET 1988
/* VERSION/RELEASE
/*	1.4
/*--*/

#include <setjmp.h>
#include <ctype.h>
#include "defs.h"
#include "params.h"
#include "status.h"
#include "comm.h"
#include "logs.h"
#include "sysdep.h"

hidden char *blnk = " \t";			/* send/expect separators */

/* forward declarations */

hidden void conn_send();
hidden void conn_xpct();
hidden char *escape();

/* connect - connect to remote system; simple script processing (no retries) */

public int connect()
{
    int *savetrap = systrap;                    /* save exception handler */
    jmp_buf mytrap;				/* our exception handler */
    int retval;					/* completion code */
    register char *cp;

    /* set up exception handler */

    if (retval = setjmp(systrap = mytrap)) {	/* get here if expect fails */
	systrap = savetrap;			/* it just happened */
	return(retval);
    }

    /* optional dial-up sequence */

    for (cp = strtok(DIAL_SEQUENCE,blnk); cp; cp = strtok((char *)0,blnk)) {
	conn_send(escape(cp));
	if (cp = strtok((char *)0,blnk))
	    conn_xpct(escape(cp));
    }

    /* mandatory login sequence */

    conn_xpct("ogin:");
    conn_send(strcons("%s\r",LOGIN_NAME));
    conn_xpct("ssword:");
    conn_send(strcons("%s\r",password));

    /* restore exception handler */

    systrap = savetrap;                         /* get here if expect wins */
    return(0);					/* say no problems... */
}

/* disconnect - disconnect line */

public int disconnect()
{
    conn_send(escape(DISC_SEQUENCE));		/* send disconnect sequence */
    return(0);					/* always succeeds... */
}

/* conn_send - quick-and-dirty output function */

hidden void conn_send(s)
register char *s;
{
    sleep(1);

    if (*s) {
	debug(4)("Sending: %S\n",s);
	while (*s) {
	    delay();
	    xwrite(ttfd,s++,1);
	}
    }
}

/* conn_xpct - primitive pattern matching without meta characters */

hidden void conn_xpct(s)
char *s;
{
    register int c,i;
    register int n = strlen(s);

    /*
    * Keep listening until we time out or until we receive the 
    * expected string. Make sure that we do not overrun our buffer.
    * Parity bits are ignored.
    */

    if (n) {
	debug(4)("Expecting: %S\nReceiving: ",s);

	if (n > MSGBUF) n = MSGBUF;

	for (i = 0; (c = xgetc()) != EOF; ) {
	    msgin[i++] = (c &= 0177);
	    debug(4)("%C",c);
	    if (i >= n && strncmp(s,&msgin[i-n],n) == 0) {
		debug(4)(" ok!\n");
		return;
	    } else if (i >= MSGBUF) {
		strncpy(msgin,&msgin[i-(n-1)],n-1);
		i = n-1;
	    }
	}
	debug(4)(" failed!\n");
	trap(E_NOLINE,"LOGIN FAILED (at \"%S\")",s);
    }
}

/* escape - interpret backslash sequences (this function is too big) */

hidden char *escape(s)
register char *s;
{
    static char buf[BUFSIZ];
    register char *cp = buf;

    while (*s && cp < buf+BUFSIZ-1) {           /* don't overflow the buffer */
	register char ch;
	if (*s != '\\') {                       /* ordinary character */
	    *cp++ = *s++;
	} else if (isdigit(*++s) && *s < '8') { /* \nnn octal code */
	    int c, i;
	    sscanf(s,"%3o",&c);
	    *cp++ = c;
	    i = 1;
            s++;
	    while (i++ < 3 && isdigit(*s) && *s <'8')
		s++;
	} else if ((ch = *s++) == 0) {          /* at string terminator */
	    break;
	} else if (ch == 'b') {                 /* \b becomes backspace */
	    *cp++ = '\b';
	} else if (ch == 'f') {                 /* \f becomes formfeed */
	    *cp++ = '\f';
	} else if (ch == 'n') {                 /* \n becomes newline */
	    *cp++ = '\n';
	} else if (ch == 'r') {                 /* \r becomes carriage ret */
	    *cp++ = '\r';
	} else if (ch == 's') {                 /* \s becomes blank */
	    *cp++ = ' ';
	} else if (ch == 't') {                 /* \t becomes tab */
	    *cp++ = '\t';
	} else {                                /* \any becomes any */
	    *cp++ = ch;
	}
    }
    *cp = '\0';                                 /* terminate the result */
    return(buf);
}
