/*
 
 *  mountd.c --
 
 *      Mount daemon program for PC NFS.  Runs on top of the net daemon
 
 *      (netd.c).  This is a partial implementation of the UNIX mount
 
 *      daemon mountd(8c).
 
 *
 
 *  Author:
 
 *      See-Mong Tan, 6/12/88]
 
 */
 

 
#include "common.h"
 

 
/*
 
 *  bool_t mountd_init() --
 
 *      Mount daemon initialization.  Creates and registers transport
 
 *      handle, and sets exports in the file EXPORTS.
 
 */
 
bool_t mountd_init()
 
{
 
	int sock;
 
	struct sockaddr_in addr;
 
	SVCXPRT *transp;
 

 
	addr.sin_family = AF_INET;
 
	addr.sin_port = htons(MOUNTPORT);
 
	addr.sin_addr.s_addr = INADDR_ANY;
 
	if ((sock = sock_create(SOCK_DGRAM, IPPROTO_UDP, &addr)) < 0) {
 
		(void) fprintf(stderr, "mountd: cannot get mount socket\n");
 
		sock_close(sock);
 
		return FALSE;
 
	}
 
	if ((transp = svcudp_create(sock, 0)) == (SVCXPRT *) NULL) {
 
		(void) fprintf(stderr, "mountd: cannot create udp handle\n");
 
		sock_close(sock);
 
		return FALSE;
 
	}
 
	if (! svc_register(transp, MOUNTPROG, MOUNTVERS, mountd_dispatch, 
 
	    IPPROTO_UDP)) {
 
		(void) fprintf(stderr, "mountd: cannot register transport\n");
 
		sock_close(sock);
 
		return FALSE;
 
	}
 
	if (! exps_parse()) {
 
		(void) fprintf(stderr, "mountd: cannot parse exports file\n");
 
		sock_close(sock);
 
		return FALSE;
 
	}
 
	
 
	return TRUE;
 
}
 

 
/*
 
 *  void mountd_dispatch(struct svc_req *req, SVCXPRT *xprt) --
 
 *      Dispatch routine for the mount daemon.
 
 */
 
void mountd_dispatch(req, xprt)
 
	struct svc_req *req;
 
	SVCXPRT *xprt;
 
{
 
	switch((int) req->rq_proc) {
 

 
	case NULLPROC:			/* "ping" the mount daemon */
 
		(void) printf("\n>>> MOUNT NULLPROC\n");
 
		if (! svc_sendreply(xprt, xdr_void, 0)) {
 
			(void) fprintf(stderr, "mountd: cannot send reply\n");
 
			return;
 
		}
 
		break;
 
	
 
	case MOUNTPROC_MNT:		/* a mount request */
 
		(void) printf("\n>>> MOUNTPROC_MNT\n");
 
		mount(req, xprt);
 
		break;
 

 
	case MOUNTPROC_UMNT:		/* unmount request from client */
 
		(void) printf("\n>>> MOUNTPROC_UMNT\n");
 
		unmount(req, xprt);
 
		break;
 

 
	case MOUNTPROC_DUMP:		/* dump mountings */
 
		(void) printf("\n>>> MOUNTPROC_DUMP - not supported yet\n");
 
		/* FALLTHROUGH */
 

 
	case MOUNTPROC_UMNTALL:		/* unmounts everything */
 
		(void) printf("\n>>> MOUNTPROC_UMNTALL\n");
 
/*		unmountall(req, xprt); */
 
		/* FALLTHROUGH */
 
	
 
	case MOUNTPROC_EXPORT:
 
	case MOUNTPROC_EXPORTALL:	/* not supported */
 
		(void) printf("\n>>> EXPORT/EXPORTALL not supported\n");
 
		/* FALLTHROUGH */
 

 
	default:			/* error */
 
		svcerr_noproc(xprt);
 
	}
 
}		
 

 
/*
 
 *  void mount(struct svc_req *req, SVCXPRT *xprt) --
 
 *      Services a mount request
 
 */
 
void mount(req, xprt)
 
	struct svc_req *req;
 
	SVCXPRT *xprt;
 
{
 
	char *path = NULL, *hostname;
 
	struct fhstatus fhs;		/* file handle status */
 
	struct sockaddr_in addr;
 

 
	if (! svc_getargs(xprt, xdr_path, &path)) {
 
		(void) fprintf(stderr, "mount: cannot decode mount path\n");
 
		svcerr_decode(xprt);
 
		return;
 
	}
 
	(void) printf("> Mount path: %s\n", path);
 
	addr = *(svc_getcaller(xprt));
 
	if (! exps_isclnt(path, addr.sin_addr.s_addr)) {
 
		(void) fprintf(stderr, "mount: access denied\n");
 
		fhs.fhs_status = NFSERR_ACCES;
 
	}
 
	else if (! mntpntofh(path, &(fhs.fhs_fh))) {
 
		(void) fprintf(stderr, "mount: illegal path\n");
 
		fhs.fhs_status = (int) NFSERR_NOENT;
 
	}
 
	else 
 
		fhs.fhs_status = (int) NFS_OK;
 

 
	/* reply to caller */
 
	if (! svc_sendreply(xprt, xdr_fhstatus, &fhs))
 
		(void) fprintf(stderr, "mount: cannot reply to mount req\n");
 

 
	svc_freeargs(xprt, xdr_path, &path);
 
}
 

 
/*
 
 *  void unmount(struct svc_req *req, SVCXPRT *xprt) --
 
 *      Unmount a filesystem.
 
 */
 
void unmount(req, xprt)
 
	struct svc_req *req;
 
	SVCXPRT *xprt;
 
{
 
	char *path = NULL;
 
	fhandle_t fh;
 
	
 
	if (! svc_getargs(xprt, xdr_path, &path)) {
 
		(void) fprintf(stderr, "unmount: cannot decode\n");
 
		svcerr_decode(xprt);
 
		return;
 
	}
 
	(void) printf(">>> MOUNTPROC_UMNT:  unmounting %s\n", path);
 
	if (! mntpntofh(path, &fh))
 
		(void) fprintf(stderr, "unmount: not in mount list\n");
 

 
	if (! svc_sendreply(xprt, xdr_void, NULL)) {
 
		(void) fprintf(stderr, "unmount: cannot send reply\n");
 
	}
 
	svc_freeargs(xprt, xdr_path, &path);
 
}
 
		
 
/*
 
 *  bool_t mntpntofh(char *path, fhandle_t *fhp) --
 
 *      Checks that path matches something in the export list and
 
 *      converts the path name to file status handle.
 
 */
 
bool_t mntpntofh(path, fhp)
 
	char *path;
 
	fhandle_t *fhp;
 
{
 
	Exports *ptr;
 

 
	if ((ptr = exps_isexport(path)) == NULL) {	/* not in list */
 
		(void) fprintf(stderr, "> \"%s\" not in export list\n", path);
 
		return FALSE;
 
	}
 
	/* anything in export list should already be in directory tree cache */
 
	bzero(fhp, sizeof(fhandle_t));
 
	fhp->p.fh_fsid = fhp->f.fh_fsid = ptr->drive;	/* file system id */
 
	fhp->f.fh_fno = pntoin(path);		/* get inode from path name */
 
	fhp->p.fh_fno = parentinode(fhp->f.fh_fno);
 
	(void) strcpy(fhp->fh_pn, intoname(fhp->f.fh_fno));
 

 
	return TRUE;
 
}
 

 
