/* ========================================================================
 * Copyright 1988-2007 University of Washington
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * 
 * ========================================================================
 */

/*
 * Program:	mix index conversion utility
 *
 * Author:	Mark Crispin
 *		Networks and Distributed Computing
 *		Computing & Communications
 *		University of Washington
 *		Administration Building, AG-44
 *		Seattle, WA  98195
 *		Internet: MRC@CAC.Washington.EDU
 *
 * Date:	9 April 2007
 * Last Edited:	9 November 2007
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>
#include <time.h>

#define NIL 0
#define T 1
#define TMPLEN 1024

/* Globals */

char *version = "2006h.1";	/* program version */
int debugp = NIL;		/* flag saying debug */
int verbosep = NIL;		/* flag saying verbose */
int critical = NIL;		/* flag saying in critical code */
FILE *f = NIL;


/* MIX message metadata */

#define MESSAGE struct mix_message

MESSAGE {
  unsigned long uid;		/* unique identifier */
  char date[20];		/* yyymmddhhmmss+zzzz */
  unsigned long size;		/* full message size */
  unsigned long fileno;		/* file number */
  unsigned long pos;		/* position of message in file */
  unsigned long offset;		/* offset of message text */
  unsigned long hdrsize;	/* header size */
  MESSAGE *next;		/* next in line */
};


/* Function prototypes */

int main (int argc,char *argv[]);
int mix_select (struct dirent *name);
int mix_msgfsort (const void *d1,const void *d2);

/* Definitions from mix.c */

/* MIX files */

#define MIXNAME ".mix"		/* prefix for all MIX file names */
#define MIXMETA "meta"		/* suffix for metadata */
#define MIXINDEX "index"	/* suffix for index */
#define MIXSTATUS "status"	/* suffix for status */


/* MIX file formats */

				/* sequence format (all but msg files) */
#define SEQFMT "S%08lx\015\012"
				/* index file record format */
#define IXRFMT ":%08lx:%19s:%08lx:%08lx:%08lx:%08lx:%08lx:\015\012"

/* Main program */

int main (int argc,char *argv[])
{
  int c,cs;
  long i,nfiles;
  unsigned long j,fileno,pos;
  char *s,file[TMPLEN],tmp[TMPLEN];
  FILE *f;
  struct dirent **names = NIL;
  MESSAGE *msg = NIL;
  MESSAGE *cur,*tail;
  if ((argc != 2) || ((strlen (argv[1]) + 100) >= TMPLEN)) {
    puts ("usage: mixrbld path");
    return 1;
  }
  if ((nfiles = scandir (argv[1],&names,mix_select,mix_msgfsort)) <= 0) {
    puts ("no mix data files there");
    return 1;
  }
  for (i = 0; i < nfiles; ++i) {
    sprintf (file,"%s/%s",argv[1],names[i]->d_name);
    if (!(f = fopen (file,"rb"))) {
      perror (file);
      return 1;
    }
    fileno = strtoul (names[i]->d_name + 4,NIL,16);
    pos = ftell (f);
    while (fgets (tmp,TMPLEN,f)) {
      if (!(!strncmp (tmp,":msg:",5) && (s = tmp + 5) && isxdigit(*s++) &&
	    isxdigit(*s++) && isxdigit(*s++) && isxdigit(*s++) &&
	    isxdigit(*s++) && isxdigit(*s++) && isxdigit(*s++) &&
	    isxdigit(*s++) && (*s++ == ':') &&
	    isdigit(*s++) && isdigit(*s++) && isdigit(*s++) && isdigit(*s++) &&
	    isdigit(*s++) && isdigit(*s++) && isdigit(*s++) && isdigit(*s++) &&
	    isdigit(*s++) && isdigit(*s++) && isdigit(*s++) && isdigit(*s++) &&
	    isdigit(*s++) && isdigit(*s++) &&
	    (((c = *s++) == '+') || (c == '-')) &&
	    isdigit(*s++) && isdigit(*s++) && isdigit(*s++) && isdigit(*s++) &&
	    (*s ++ == ':') && isxdigit(*s++) && isxdigit(*s++) &&
	    isxdigit(*s++) && isxdigit(*s++) && isxdigit(*s++) &&
	    isxdigit(*s++) && isxdigit(*s++) && isxdigit(*s++) &&
	    (*s++ == ':') && (*s++ == '\r') && (*s++ == '\n') && !*s)) {
	printf ("Data file %s header error @ %ld: %s",file,pos,tmp);
	return 1;
      }

      cur = (MESSAGE *) memset (malloc (sizeof (MESSAGE)),NIL,sizeof(MESSAGE));
      if (!(cur->uid = strtoul (tmp+5,NIL,16))) {
	printf ("Data file %s zero UID @ %ld: %s",file,pos,tmp);
	return 1;
      }
      memcpy (cur->date,tmp+14,19);
      if (!(cur->size = strtoul (tmp+34,NIL,16))) {
	printf ("Data file %s zero length size @ %ld: %s",file,pos,tmp);
	return 1;
      }
      cur->fileno = fileno;
      cur->offset = ftell (f) - (cur->pos = pos);
      for (cs = 0, j = 1; j < cur->size; ++j) {
	c = getc (f);		/* read character from message */
	switch (cs) {		/* decide what to do based on state */
	case 0:			/* previous char ordinary, advance if CR */
	  if (c == '\015') cs = 1;
	  break;
	case 1:			/* previous CR, advance if LF */
	  cs = (c == '\012') ? 2 : 0;
	  break;
	case 2:			/* previous CRLF, advance if CR */
	  cs = (c == '\015') ? 3 : 0;
	  break;
	case 3:			/* previous CRLFCR, done if LF */
	  if (c == '\012') {
	    cur->hdrsize = j;
	    j = cur->size;	/* exit the loop */
	  }
	  cs = 0;		/* reset mechanism */
	  break;
	}
      }
				/* header is entire message */
      if (!cur->hdrsize) cur->hdrsize = cur->size;

      if (msg) {		/* have a list? */
	if (cur->uid > msg->uid) {
	  tail->next = cur;
	  tail = cur;
	}
	else if (cur->uid == msg->uid) {
	  printf ("Data file %s duplicate UID @ %ld: %ld\n",file,pos,cur->uid);
	  return 1;
	}
	else {
	  printf ("Data file %s UID ran backwards (%lx < %lx)\n",
		  file,cur->uid,msg->uid);
	  if (cur->uid < msg->uid) {
	    cur->next = msg;	/* make this the new head */
	    msg = cur;
	  }
	  else {		/* do it the hard way */
	    for (tail = msg; tail->next->uid < cur->uid; tail = tail->next);
	    if (cur->uid == tail->next->uid) {
	      printf ("Data file %s duplicate UID @ %ld: %ld\n",
		      file,pos,cur->uid);
	      return 1;
	    }
	    cur->next = tail->next;
	    tail->next = cur;
	  }
	}
      }
      else msg = tail = cur;	/* no, make new list */
      if (fseek (f,pos += cur->offset + cur->size,SEEK_SET)) perror (file);
    }
    if (ferror (f)) {
      perror (file);
      return 1;
    }
    fclose (f);
  }
  sprintf (file,"%s/%s",argv[1],".mixindex-rebuild");
  if (!(f = fopen (file,"wb"))) {
    perror (file);
    return 1;
  }
  fprintf (f,SEQFMT,(unsigned long) time (0));
  for (cur = msg; cur; cur = cur->next)
    fprintf (f,IXRFMT,cur->uid,cur->date,cur->size,cur->fileno,cur->pos,
	     cur->offset,cur->hdrsize);
  fclose (f);
  printf ("New index written to %s\n",file);
  return 0;
}

/* MIX test for message file name
 * Accepts: candidate directory name
 * Returns: T if message file name, NIL otherwise
 *
 * ".mix" with no suffix was used by experimental versions
 */

int mix_select (struct dirent *name)
{
  char c;
  char *s = name->d_name;
				/* make sure name has prefix */
  if ((*s++ == '.') && (*s++ == 'm') && (*s++ == 'i') && (*s++ == 'x')) {
    for (c = *s; c && isxdigit (c); c = *s++);
				/* all-hex or no suffix */
    if (((s - name->d_name) < 30) && !c) return T;
  }
  return NIL;			/* not suffix or non-hex */
}


/* MIX msg file name comparision
 * Accepts: first candidate directory entry
 *	    second candidate directory entry
 * Returns: -1 if d1 < d2, 0 if d1 == d2, 1 d1 > d2
 */

int mix_msgfsort (const void *d1,const void *d2)
{
  char *n1 = (*(struct dirent **) d1)->d_name + 4;
  unsigned long i1 = strtoul (n1,NIL,16);
  char *n2 = (*(struct dirent **) d2)->d_name + 4;
  unsigned long i2 = strtoul (n2,NIL,16);
  if (i1 < i2) return -1;
  if (i1 > i2) return 1;
  return 0;
}
