////////////////////////////////////////////////////////////////
// NetworkMonitor.java
//
// Copyright (C) 2002-2002 by ObjectPlanet, Inc.
// All rights reserved. 
// Confidential, unpublished property of ObjectPlanet, Inc.
////////////////////////////////////////////////////////////////

package com.objectplanet.chart.examples;

import com.objectplanet.gui.*;
import com.objectplanet.chart.*;
import java.applet.*;
import java.awt.*;
import java.awt.event.*;


/**
 * This network monitor applet shows the use of the ObjectPlanet Developer
 * Suite components and how they can be combined to make applications in
 * very short time.
 * <p>
 * This Network Monitor is a simplification of the Network Monitor in the
 * Network Probe product.
 * <p>
 * See the index.html file for applet parameters.
 *  
 * @author Bjorn J. Kvande.
 */
public class NetworkMonitor extends Applet implements TableListener, ItemListener, TabbedListener {


	/**
	 * The tabs displaying the traffic tables and charts.
	 */
	private TabbedPanel trafficTabs;
	
	
	/**
	 * The tables displaying the network traffic statistics.
	 */
	private Table[] trafficTables;
	
	
	/**
	 * The bar chart appplets showing the traffic distribution graphically.
	 */
	private BarChartApplet[] chartApplets;
	
	
	/**
	 * The table applet prefix.DO NOT CHANGE THE SEQUENCE OF 
	 * THESE LABELS IF YOU DON'T KNOW WHAT YOU ARE DOING.
	 */
	private String[] tablePrefix = new String[] {
		"protocol_",
		"nlHost_",
		"nlMatrix_",
		"alHost_",
		"alMatrix_",
		"macHost_",
		"macMatrix_",
	};
	
	
	/**
	 * The tab labels. DO NOT CHANGE THE SEQUENCE OF THESE 
	 * LABELS IF YOU DON'T KNOW WHAT YOU ARE DOING.
	 */
	private String[] tabLabels = new String[] {
		"protocols",
		"hosts",
		"conversations",
		"protocols per host",
		"protocols per conversation",
		"network cards",
		"network card conversations",
	};
	
	
	/**
	 * This index is used when setting chart data after a table event.
	 * Only numerical data is set in the bar chart, so we set only new
	 * data if a new numerical column has been selected. If a textual
	 * column has been selected, the column in this array is used.
	 * DO NOT CHANGE THE SEQUENCE OR VALUES OF THESE NUMBERS IF YOU 
	 * DON'T KNOW WHAT YOU ARE DOING.
	 */
	private int[] selectedNumberColumn = new int[] {
		4,	// protocols, bytes
		3, // hosts, bytes in
		5, // conversations, bytes
		4, // protocols per host, bytes in
		6, // protocols per conversaion, bytes
		2, // network cards, bytes in
		5	// network card conversations, bytes
	};
	
	
	/**
	 * The columns to get the sample labels from. DO NOT CHANGE THE SEQUENCE 
	 * OR VALUES OF THESE NUMBERS IF YOU DON'T KNOW WHAT YOU ARE DOING.
	 */
	private int[][] tableSampleLabelColumns = new int[][] {
		{0},		// protocols, protocol name
		{0},		// hosts, host name
		{0,2},	// conversations, source host name + destination host name
		{0,2},	// protocols per host, host name + protocol name
		{0,2,4},	// protocols per conversiaon, src name + dst name + protocol name
		{0},		// network cards, MAC address
		{0,2}		// network card conversations, src mac address + dst mac address
	};
	
	
// construction
	
	
	/**
	 * Creates the Network Monitor instance and its subcomponents.
	 */
	public NetworkMonitor() {
		// create the tables and charts
		trafficTables = new Table[7];
		chartApplets = new BarChartApplet[trafficTables.length];
		for (int i = 0; i < trafficTables.length; i++) {
			// create table
			trafficTables[i] = new Table();
			trafficTables[i].setLabelCacheOn(true);
			trafficTables[i].setAutomaticRepaintOn(false);
			trafficTables[i].setParentApplet(this);
			trafficTables[i].addTableListener(this);
			trafficTables[i].setParameterPrefix(tablePrefix[i]);
			trafficTables[i].setBackground(new Color(0xcfcfce));
			// create bar chart
			chartApplets[i] = new BarChartApplet();
			chartApplets[i].setParentApplet(this);
			chartApplets[i].setParameterPrefix("bar_");
			chartApplets[i].theChart.setAutomaticRepaintOn(false);
			chartApplets[i].theChart.addItemListener(this);
		}
		
		// create the tabbed panel and add the tables
		trafficTabs = new TabbedPanel();
		trafficTabs.setBackground(Color.white);
		trafficTabs.setTabColor(new Color(0xcfcfce));
		trafficTabs.addTabbedListener(this);
		for (int i = 0; i < trafficTables.length; i++) {
			Panel p = new Panel(new GridLayout(2,1));
			p.add(trafficTables[i]);
			p.add(chartApplets[i]);
			trafficTabs.add(tabLabels[i], p);
		}
		
		// place the applet components
		setLayout(new BorderLayout());
		add("Center", trafficTabs);
	}
	
	
	/**
	 * Read the applet parameters and data.
	 */
	public void init() {
		// read the table data
		for (int i = 0; i < trafficTables.length; i++) {
			chartApplets[i].init();
			chartApplets[i].theChart.setTitle(tabLabels[i]);
			chartApplets[i].theChart.setTitleOn(true);
			trafficTables[i].init();
		}
		trafficTables[0].requestFocus();
	}


// handles table events and sets the chart data
	
	
	/**
	 * Handles the events from the table.
	 */
	public void tableSelection(TableEvent event) {
		// get the event details
		Table table = (Table) event.getSource();
		int column_index = event.getColumn();
		int row_position = event.getRowPos();
		int column_type = Table.COLUMN_TEXT;
		if (column_index >= 0) {
			column_type = table.getColumnType(column_index);
		}

		// find the correct table
		int table_index = -1;
		for (int i = 0; i < trafficTables.length; i++) {
			if (table == trafficTables[i]) {
				table_index = i;
				break;
			}
		}
		
		// set the data on sorting, filtering, and data loading
		if (table_index >= 0) {
			// get the chart
			BarChartApplet chart = chartApplets[table_index];
			chart.theChart.setAutomaticRepaintOn(false);
			table.setAutomaticRepaintOn(false);
			
			// set the chart data or chart selection
			switch (event.getType()) {
				// update the data
				case TableEvent.SORT:
				case TableEvent.FILTER:
				case TableEvent.DATA_LOADED:
					// set the chart data
					setChartData(table_index, column_index);
					break;
					
				// scroll the chart
				case TableEvent.SCROLL:
					chart.theChart.setVisibleSamples(table.getFirstRowPosition(), 10);
					break;

				// set the chart selection
				case TableEvent.SELECTED:
				case TableEvent.DOUBLE_CLICK:
					// if a new column was selected, change the chart data
					if (column_type == Table.COLUMN_NUMBER) {
						setChartData(table_index, column_index);
					}
					// select the chart sample
					chart.theChart.setSelection(0, -1, false, false);
					chart.theChart.setSelection(0, row_position, true, false);
					break;
					
				// set the chart selection
				case TableEvent.SEARCH:
					chart.theChart.setSelection(0, -1, false, false);
					chart.theChart.setSelection(0, row_position, true, false);
					break;
			}
			
			// repaint the table and chart
			table.setAutomaticRepaintOn(true);
			chart.theChart.setAutomaticRepaintOn(true);
			table.repaint();
			chart.theChart.repaint();
		}
	}
	
	
	/**
	 * Sets the chart data from the specified table.
	 * @param table_index The index of the table and chart.
	 * @param column The index of the column.
	 */
	private void setChartData(int table_index, int column) {
		// get the table and chart
		Table table = trafficTables[table_index];
		BarChartApplet chart = chartApplets[table_index];

		// set the data in the bar chart
		if (column < table.getColumnCount()) {
			// use a numerical column
			if (column < 0 || table.getColumnType(column) != Table.COLUMN_NUMBER) {
				column = selectedNumberColumn[table_index];
			}
			
			// keep track of the current adjusted range, reset the adjusted
			// range if it's the same as the non-adjusted upper range
			double upper_range = chart.theChart.getRange(0);
			double adjusted_range = chart.theChart.getCurrentRange(0);

			// set the column values as the bar values in the bar chart
			double[] values = table.getColumnValues(column, 0, -1);
			chart.theChart.setSampleCount(Math.max(10, values.length));
			chart.theChart.setSampleValues(0, values);
			chart.theChart.setVisibleSamples(table.getFirstRowPosition(), 10);
			
			// set the bar labels of the chart
			String[] sampleLabels = createSampleLabels(table, tableSampleLabelColumns[table_index], " - ");
			chart.theChart.setSampleLabels(sampleLabels);
			chart.theChart.setLabel("rangeAxisLabel", table.getHeader(column));
			
			// set the chart range
			chart.theChart.setRelativeRange(1.0);
			if (upper_range == adjusted_range) {
				chart.theChart.setCurrentRange(0, chart.theChart.getRange());
			}
		}
	}
	
	
	/**
	 * Creates a comma separated string combining multiple table columns.
	 * @param table The table to get the column data from.
	 * @param columns The index of the columns to get.
	 * @param connector The label used between the labels.
	 * @return A comma seperated string of labels.
	 */
	private String[] createSampleLabels(Table table, int[] columns, String connector) {
		// make sure we have valid input data
		if (table == null || columns == null || columns.length == 0) {
			return new String[0];
		}
		
		// get the columns
		String[][] table_data = new String[columns.length][];
		for (int i = 0; i < table_data.length; i++) {
			table_data[i] = (String[]) table.getColumn(columns[i], 0, -1);
		}
		
		// combine the labels
		String[] labels = new String[table_data[0].length];
		for (int row = 0; row < labels.length; row++) {
			StringBuffer buffer = new StringBuffer(100);
			for (int col = 0; col < table_data.length; col++) {
				// add the label
				if (table_data[col][row] != null) {
					buffer.append(table_data[col][row]);
					// add the connector only if we have multiple columns
					if (col < table_data.length-1 && connector != null) {
						buffer.append(connector);
					}
				}
			}
			labels[row] = buffer.toString();
		}
		return labels;
	}
	
	
// handles chart events and sets the table selection
	
	
	/**
	 * Selects the row in the table based on the chart selection.
	 */
	public void itemStateChanged(ItemEvent event) {
		// select the corresponding row
		if (event.getStateChange() == ItemEvent.SELECTED) {
			// find the table corresponding to the chart the event came from
			Object source = event.getItemSelectable();
			Table table = null;
			for (int i = 0; i < chartApplets.length; i++) {
				if (source == chartApplets[i].theChart) {
					table = trafficTables[i];
					break;
				}
			}
			
			// set the table selection
			if (table != null) {
				// deselect the old row
				table.selectRow(table.getLastSelectedRow(), false);
				// select the new row
				ChartSample item = (ChartSample) event.getItem();
				int position = item.getIndex();
				int row_index = table.getRowIndex(position);
				table.selectRow(row_index, true);
				if (row_index >= 0) {
					table.gotoRow(row_index);
				}
				// let the table maintain the keyboard focus so we can scroll up
				// and down using the arrow keys, page up, and page down
				table.requestFocus();
			}
		}
	}
	
	
// handles tabbed panel events and sets the table keyboard focus
	
	
	/**
	 * Sets the keyboard focus to the selected table.
	 */
	public void tabEvent(TabbedEvent event) {
		switch (event.getType()) {
			// a tab was displayed, set the keyboard focus for the table
			case TabbedEvent.TAB_SHOWN:
				// get the index of the traffic table and chart
				String tab = event.getTab();
				for (int i = 0; i < tabLabels.length; i++) {
					if (tab.equalsIgnoreCase(tabLabels[i])) {
						trafficTables[i].requestFocus();
					}
				}
		}
	}
}