/*
    SimpleAIM
    A miniature console AIM client
    http://simpleaim.sourceforge.net
    Copyright (C) 2002
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  */

import java.io.IOException;
import java.util.StringTokenizer;

/**
 * Handles input messages from TOC -> client
 *
 * @author  simpleaim.sourceforge.net
 */
public class InHandler extends Thread {

    /**
     * Constructor for the InHandler object.
     */
    public InHandler() {
        // here for further add-ons
        // otherwise useless
    }


    /**
     * Returns the remainder of the string after the first occurence of the ':'
     * character
     *
     * @param str The string to parse
     * @return    The remainder of the string
     */
    public String parseColon(String str) {
        return str.substring(str.indexOf(':') + 1);
    }


    /**
     * Main processing method for the InHandler object.
     */
    public void run() {
        try {
            while (true) {

                // handle TOC->client messages
                String str = Operations.readFLAP();
                if (str == null) {
                    continue;
                } else if (str.toUpperCase().startsWith("CONFIG:")) {

                    // process buddy list setup configuration
                    processBuddyListSetup(parseColon(str));
                } else if (str.toUpperCase().startsWith("NICK:")) {

                    // welcome message with correct format of nickname
                    if (!Operations.greeted) {
                        System.out.println("\nWelcome, " + parseColon(str) + "!\n");
                        System.out.print(Operations.username + " ~ > ");
                        Operations.greeted = true;
                    }
                } else if (str.toUpperCase().startsWith("IM_IN:")) {
                    // process incoming IMs
                    str = parseColon(str);
                    String user = str.substring(0, str.indexOf(':'));
                    System.out.print("\n" + user + ": ");
                    str = parseColon(str);
                    if (str.substring(0, str.indexOf(':')).equals("T")) {
                        System.out.print("(Auto Response) ");
                    }
                    str = parseColon(str);
                    System.out.println(Operations.decodeIM(str));

                    // if away, auto respond
                    if (Operations.away) {
                        Operations.writeFLAP(Operations.DATA, "toc_send_im " + Operations.normalize(user) +
                                " " + Operations.encodeIM(Operations.awayMessage) + " auto");
                    }

                    // store user to reply to
                    Operations.replyToUser = Operations.normalize(user);

                    // print console prompt
                    System.out.print(Operations.username + " ~ > ");

                } else if (str.toUpperCase().startsWith("UPDATE_BUDDY:")) {

                    // update buddy status
                    updateBuddy(parseColon(str));

                } else if (str.toUpperCase().startsWith("ERROR:")) {

                    // handle incoming error
                    System.out.print("\n");
                    Errors.expand(str);

                    // print console prompt
                    System.out.print(Operations.username + " ~ > ");

                } else {
                    if (Operations.DEBUG) {
                        System.out.println("\nUnsupported server message: " + str);
                    }
                }
            }
        } catch (Exception e) {
            if (Operations.DEBUG) {
		e.printStackTrace();
            }
        }

    }


    /**
     * Processes the user's configuration by sending a FLAP command to add
     * buddies.
     *
     * @param config                The buddy list setup configuration.
     * @throws IOException
     * @throws InterruptedException
     */
    private void processBuddyListSetup(String config) throws IOException, InterruptedException {

        int numOfBuddies = 0;
        String line;
        String result = "";
        String msg = "toc_add_buddy";
        StringTokenizer st = new StringTokenizer(config, "\n");
        while (st.hasMoreElements()) {
            line = st.nextToken();
            if (line.charAt(0) == 'b') {
                numOfBuddies++;
                msg = msg + " " + Operations.normalize(line.substring(2, line.length()));
                result = result + "\t\t" + Operations.normalize(line.substring(2, line.length())) + "\n";
            }
        }
        if (numOfBuddies > 0) {
            Operations.writeFLAP(Operations.DATA, msg);
        }
        result = "\t" + Operations.username + "'s Buddy List Setup:\n\n" + result.substring(0, result.length() - 1) + "\n";
        if (Operations.DEBUG) {
            System.out.println("\n" + result);
        }
        Operations.buddyListSetup = result;
    }


    /**
     * process the update of buddy status, adding to active buddy list hashtable
     * if necessary
     *
     * @param buddy The buddy to update the status.
     */
    private void updateBuddy(String buddy) {

        String name = Operations.normalize(buddy.substring(0, buddy.indexOf(':')));
        buddy = parseColon(buddy);
        boolean isOnline = buddy.toUpperCase().startsWith("T");
        if (isOnline) {
            buddy = parseColon(buddy);
            int warning = Integer.parseInt(buddy.substring(0, buddy.indexOf(':')));
            buddy = parseColon(buddy);
            buddy = parseColon(buddy);
            int idle = Integer.parseInt(buddy.substring(0, buddy.indexOf(':')));
            if (Operations.buddyList.containsKey(name)) {// update buddy status
                Buddy bud = (Buddy) Operations.buddyList.get(name);
                bud.warning = warning;
                bud.setIdle(idle);
            } else {// add buddy to active buddy list
                Operations.buddyList.put(name, new Buddy(name, warning, idle));
            }
        } else {// if the buddy signed off, then remove them from the active buddy list
            Operations.buddyList.remove(name);
        }
    }
}
