/*********************************************************************

SYBREX SYSTEMS INCORPORATED
Copyright (C) 2005 Sybrex Systems Incorporated
All rights reserved.

NOTICE: Sybrex permits you to use, modify, and distribute this file
in accordance with the terms of the Sybrex license agreement
accompanying it. If you have received this file from a source other
than Sybrex, then your use, modification, or distribution of it
requires the prior written permission of Sybrex.

----------------------------------------------------------------------
VSOutlineA.h
*********************************************************************/

#ifndef VSOutlineA_H
#define VSOutlineA_H
#ifndef __GNUC__ /* suppress gcc warning */
#pragma once
#endif /* __GNUC__ */

#include "VSTypes.h"
#include "VSActionA.h"

#ifdef __cplusplus
extern "C"{
#endif


	/********************************************************************
    Description:
        Returns root of outlines.
    Parameters:
           Doc - [ in ] Current PDF document.
    Returns:
        Handle to document outlines root.
    *********************************************************************/
    PDFOutlineHandle PDFDocGetOutlineRoot(PDFDocHandle Doc);


	/* Description
	   Creates new outline which will be parallel for current
	   \outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   Handle to created outline.                             */
    PDFOutlineHandle PDFOutlineAddNewSibling(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   Adds existing outline which will be parallel for current
	   \outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Sibling :  [ in ] Added outline.
	   Returns
	   None.                                                    */
	void PDFOutlineAddSibling(PDFDocHandle Doc, PDFOutlineHandle Outline, PDFOutlineHandle Sibling);

	/* Description
	   Adds new outline which will be next for current outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   Handle to created outline.                               */
    PDFOutlineHandle PDFOutlineAddNewNext(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   Adds new outline which will be previous for current outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   Handle to created outline.                                   */
    PDFOutlineHandle PDFOutlineAddNewPrev(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   Adds existing outline followed by current outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Next :     [ in ] Handle outline to be added.
	   Returns
	   None.                                              */
    void PDFOutlineAddNext(PDFDocHandle Doc, PDFOutlineHandle Outline, PDFOutlineHandle Next);

	/* Description
	   Adds existing outline before the current outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Prev :     [ in ] Handle outline to be added.
	   Returns
	   None.                                             */
	void PDFOutlineAddPrev(PDFDocHandle Doc, PDFOutlineHandle Outline, PDFOutlineHandle Prev);

	
	/********************************************************************
    Description:
        Creates outline as child for current outline.
    Parameters:
           Doc        - [ in ] Current PDF document.
		   Outline  - [ in ] Current outline.
    Returns:
	    Handle to created outline.
    *********************************************************************/
	PDFOutlineHandle PDFOutlineAddNewChild(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   Sets outline as child for current outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Child :    [ in ] Child handle.
	   Returns
	   None.                                      */
    void PDFOutlineAddChild(PDFDocHandle Doc, PDFOutlineHandle Outline, PDFOutlineHandle Child);

	/* Description
	   \Returns the first outline child.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   Handle to the first outline child.      */
	PDFOutlineHandle PDFOutlineGetFirstChild(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   \Returns the last outline child.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   Handle to the last outline child.       */
    PDFOutlineHandle PDFOutlineGetLastChild(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   \Returns following outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   Handle to following outline.            */
    PDFOutlineHandle PDFOutlineGetNext(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/********************************************************************
    Description:
        Returns previous outline.
    Parameters:
           Doc     - [ in ] Current PDF document.
		   Outline - [ in ] Current outline.
    Returns:
        Handle to previous outline.
    *********************************************************************/
    PDFOutlineHandle PDFOutlineGetPrev(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   \Returns parent outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   Handle to parent outline.               */
	PDFOutlineHandle PDFOutlineGetParent(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   Deletes outline links.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   None.                                   */
	void PDFOutlineUnLink(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   Deletes current outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   None.                                   */
	void PDFOutlineDelete(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   Inspects current outline on children presence.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   If current outline has children - true, else - false. */
    ppBool PDFOutlineHasChildren(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   Calculates count of the children.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   Count of children.                      */
    ppInt32 PDFOutlineGetCount(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   Inspects current outline on expanding.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   true - if outline expanded, false - if no. */
    ppBool PDFOutlineGetExpanded(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   Sets current outline on expanding.
	   Parameters
	   Doc :       [ in ] Current PDF document.
	   Outline :   [ in ] Current outline.
	   Expanded :  [ in ] Outline expand state.
	   Returns
	   None.                                    */
    void PDFOutlineSetExpanded(PDFDocHandle Doc, PDFOutlineHandle Outline, ppBool Expanded);

	/* Description
	   Inspects color of outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Color :    [ in ] Outline color.
	   Returns
	   If color correspond - true, else - no.  */
    ppBool PDFOutlineGetColor(PDFDocHandle Doc, PDFOutlineHandle Outline, PPDFColor Color);

	/********************************************************************
    Description:
        Sets color of outline.
    Parameters:
           Doc      - [ in ] Current PDF document.
		   Outline  - [ in ] Current outline.
		   Color    - [ in ] Outline color.
    Returns:
        None.
    *********************************************************************/
    void PDFOutlineSetColor(PDFDocHandle Doc, PDFOutlineHandle Outline, PPDFColor Color);

	/* Description
	   Calculates outline flags.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   Outline flags.                          */
    ppInt32 PDFOutlineGetFlags(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/********************************************************************
    Description:
        Sets the outline flags.
    Parameters:
           Doc      - [ in ] Current PDF document.
		   Outline  - [ in ] Current outline.
		   Flags    - [ in ] Outline flags.
    Returns:
        None.
    *********************************************************************/
    void PDFOutlineSetFlags(PDFDocHandle Doc, PDFOutlineHandle Outline, ppInt32 Flags);

	/********************************************************************
    Description:
        Returns title of outline.
    Parameters:
           Doc      - [ in ] Current PDF document.
		   Outline  - [ in ] Current outline.
		   str      - [ out ] Outline title.  
		   len      - [ in ] Title length.
    Returns:
        None.
    *********************************************************************/
    ppInt32 PDFOutlineGetTitle(PDFDocHandle Doc, PDFOutlineHandle Outline, char *str, ppInt32 len);

	/********************************************************************
    Description:
        Sets title of outline.
    Parameters:
           Doc      - [ in ] Current PDF document.
		   Outline  - [ in ] Current outline.
		   str      - [ in ] Outline title.  
		   len      - [ in ] Title length.
    Returns:
        None.
    *********************************************************************/
    void PDFOutlineSetTitle(PDFDocHandle Doc, PDFOutlineHandle Outline, char *str, ppInt32 len); 

	/* Description
	   \Returns object to linked outline .
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   Linked object handle.                        */
    PDFDestinationHandle PDFOutlineGetDestination(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/* Description
	   \Returns action linked to outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Returns
	   Outline action handle.                  */
    PDFActionHandle PDFOutlineGetAction(PDFDocHandle Doc, PDFOutlineHandle Outline);

	/********************************************************************
    Description:
        Links outline to object.
    Parameters:
           Doc         - [ in ] Current PDF document.
		   Outline     - [ in ] Current outline.
           Destination - [ in ] Linked object handle. 
    Returns:
        None.
    *********************************************************************/
    void PDFOutlineSetDestination(PDFDocHandle Doc, PDFOutlineHandle Outline,
                                  PDFDestinationHandle Destination);
	/* Description
	   \Links action to outline.
	   Parameters
	   Doc :      [ in ] Current PDF document.
	   Outline :  [ in ] Current outline.
	   Action :   [ in ] Linked action.
	   Returns
	   None.                                   */
    void PDFOutlineSetAction(PDFDocHandle Doc, PDFOutlineHandle Outline, PDFActionHandle Action);


#ifdef __cplusplus
}
#endif

#endif /* #ifndef VSOutlineA_H */
