/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Aug27-GiuseppeG: code write
*/

#include "XSP_Core.h"
#include "XSP_File.h"
#include "XSP_GUI.h"

namespace XSP
{

FontName::FontName(const FontName& t)
: faceName(t.faceName)
, fontSize(t.fontSize)
, boldFont(t.boldFont)
, italicFont(t.italicFont)
, underlineFont(t.underlineFont)
, charset(t.charset)
{
}

FontName::~FontName()
{
}


TextFont::~TextFont()
{
}

TextFont::TextFont(const TextFont& i) 
: data(i.data) 
{
}

TextFont& TextFont::operator = (const TextFont& i) 
{ 
	data = i.data; 
	return *this;
}

bool TextFont::operator == (const TextFont& f) const 
{ 
	return data == f.data; 
}

bool TextFont::operator != (const TextFont& f) const 
{ 
	return data != f.data; 
}

#if 0
#pragma mark -
#endif

TextLine::TextLine()
: refcount(0)
, y(0), h(0)
{
}
TextLine::~TextLine()
{
}

#if 0
#pragma mark -
#endif

TextId::TextId()
: refcount(0)
{
}

TextId::~TextId()
{
}

#if 0
#pragma mark -
#endif

TextPiece::TextPiece()
: refcount(0)
, wdth(0), flgs(0)
{
	// 0 would be a valid value, we need to show this has not been rendered yet
	pos.x = -1;  
}
TextPiece::~TextPiece()
{
}

TextPiece::TextPiece(const TextPiece& t)
: refcount(0)
{
	this-> operator= (t);
}

TextPiece& TextPiece::operator= (const TextPiece& t)
{
	flgs = t.flgs;
	wdth = t.wdth;
	font = t.font;
 	colr = t.colr;
	text = t.text;
	pos	 = t.pos;
	textline = t.textline;
	textId = t.textId;
	image = t.image;
	return *this;
}

bool TextPiece::isEmpty() const 
{ 
//	if (0 != (flgs & TextPiece::EOL))
//		return false; // CR is not empty
	return text.isEmpty(); 
}

TextDocument::Piece TextPiece::SplitAtChar(String::iterator cp, uint32 rdx)
{
	String::iterator e = text.end();
	
	ASSERT(rdx <= wdth)
	if (cp == e)
	{
//		ASSERT((rdx == wdth) || (rdx == 0)); 
		return 0;
	}

	TextDocument::Piece newpiece(new TextPiece(*this));
	newpiece->wdth = wdth - rdx;
	wdth = rdx;
	newpiece->pos  = Point2D(pos.x+rdx, pos.y);
	newpiece->text.assign(cp,e);
	text.resize(cp-text.begin());

//	newpiece->flgs = flgs;
//	newpiece->font = font;
//	newpiece->colr = colr;
//	newpiece->textline= textline;
//	newpiece->textId  = textId; 

	return newpiece;
}

TextDocument::Piece TextPiece::SplitEmpty()
{
	TextDocument::Piece newpiece(new TextPiece(*this));
	newpiece->clear();
	return newpiece;
}


TextDocument::Piece TextPiece::SplitAt(Graphics& g, sint32 x)
{
	x-=pos.x; // turn it to relative coordinate

	if (0 != (flgs & TextPiece::Atomic))
	{
		if (x == 0) 
		{  // the fragment will contain the CR
			TextDocument::Piece np(new TextPiece(*this));
			clear();
			return np;
		}
		else // the fragment will be empty
			return 0;
//		{
//			TextDocument::Piece np(new TextPiece(*this));
//			np->clear();
//			np->pos.x = pos.x+wdth;
//			return np;
//		}
	}

    std::vector<uint32> charPlacements;
	MeasureEachCharWidth(g, charPlacements );
//	g.MeasureEachCharWidth(text.begin(), text.end(), font, charPlacements );
	std::vector<uint32>::iterator cp = 
			std::lower_bound(charPlacements.begin(), charPlacements.end()-1, x);
	// if x is not precisely the start pos of the char, lower_bound
	// will return the next char as the result of the search so we
	// must adjust back
	if (*cp > x) --cp;
    uint32 len = cp - charPlacements.begin();
    
    return SplitAtChar(text.begin()+len, *cp);
}


TextDocument::Piece TextPiece::SplitAtLineEnd(Graphics& g, sint32 x)
{
	if (x <= pos.x)
		return 0;
	x-=pos.x;  // turn it to relative coordinate
	
	// locate the space chars (but not \t\n\r  which cannot be part of the piece)
	String::iterator cb = text.begin();
	String::iterator ce = text.end();
    std::vector<uint32> charPlacements;
//	g.MeasureEachCharWidth(cb, ce, font, charPlacements );
	MeasureEachCharWidth(g, charPlacements );

	std::vector<uint32>::iterator cx = charPlacements.begin();
	String::iterator bestc = cb; // ce;
	std::vector<uint32>::iterator bestx = cx; 
	char separators[] = " ,;:/\\()[]{}+-*|&";
	for (String::iterator cs=cb; ; ++cs)
	{
		cs = std::find_first_of(cs,ce, separators,separators+sizeof(separators)-1);
		if (cs == ce)
			break;
		cx = charPlacements.begin() + (cs-cb);
		if (*cx > x)
			break;
		bestc = cs+1;
		bestx = cx+1;
		if (*bestx >= x)
		{
			if (*bestx > x)
				bestc = cs, bestx = cx;
			break;
		}
	}

//	UnitTest::Log("-Wrapping piece '%s'", text.c_str());		
//	UnitTest::Log("      >Split at '%s'", bestc);		

//	if (*bestx <= 0)
//		return 0;
    return SplitAtChar(bestc, *bestx);
}

bool TextPiece::MergeWith(const TextDocument::Piece& t)
{
//	if (t->text.isEmpty()) // will eliminate the empty t
//		return true; 
//	if (text.isEmpty()) // we eliminate this 
//	{
//		wdth = t->wdth;
//		pos = t->pos;
//		text = t->text;
//		flgs = t->flgs;
//		font = t->font;
//		colr = t->colr;
//		textline = t->textline;
//		textId = t->textId;
//		return true;
//	}

	if (flgs != t->flgs)
		return false;
	if (0 != ((flgs | t->flgs) &  TextPiece::Atomic))
		return false;

	// check if we are compatible
	if (textline != t->textline)
		return false;
	if (textId != t->textId)
		return false;
	if (pos.x+wdth != t->pos.x)
		return false;
	if (colr != t->colr)
		return false;
	if (font != t->font)
		return false;

	// append t to this
    wdth += t->wdth;
    text.append(t->text);
	return true;
}


void TextPiece::SetText(const String& t)
{
	if (text != t)
		wdth = 0;
	text = t;
}
void TextPiece::SetFont(const TextFont& t)
{
	if (font != t)
		wdth = 0;
	font = t;
}

void TextPiece::clear()
{
	flgs = 0;
	wdth = 0;
	text.clear();
	image = 0;
}

void TextPiece::TurnIntoNewLine()
{
	text.clear(); // = String::From_c_str(" ");
	wdth = 0; // must be measured again
	flgs = TextPiece::NewLine | TextPiece::Atomic;
}

void TextPiece::SetImage(OffscreenImage& img)
{
	flgs |= TextPiece::Atomic;
	wdth  = img.GetSize().w;
	image = img;
	text.clear();
}

void TextPiece::Measure(Graphics& g)
{
   	if (0 == (flgs & TextPiece::Atomic))
		wdth = g.MeasureTextWidth(text.begin(), text.end(), font);
   	// else leave wdth unchanged
}
void TextPiece::MeasureEachCharWidth(Graphics& g, std::vector<uint32> & results)
{
	if (0 != (flgs & TextPiece::Atomic))
	{
		// the atomic piece looks just like one wide char
		results.resize(2);
		results[0] =0;
		results[1] =wdth;
	}
	else
		g.MeasureEachCharWidth(text.begin(), text.end(), font, results );
}
sint32 TextPiece::GetAscent() const 	
{ 
	if (0 != (flgs & TextPiece::Atomic))
	{
		if (image != 0)
		{
			sint32 h = image.GetSize().h;
			return h - h/3;
		}
	}	
	return font.GetAscent(); 
}
sint32 TextPiece::GetDescent() const	
{ 
	if (0 != (flgs & TextPiece::Atomic))
	{
		if (image != 0)
		{
			return image.GetSize().h/3;
		}
	}	
	return font.GetDescent(); 
}
Rect2D TextPiece::GetBounds()
{
	if (0 != (flgs & TextPiece::Atomic))
	{
		if (image != 0)
		{
			return Rect2D(pos, image.GetSize());
		}
	}	
	return Rect2D(pos, Size2D(wdth, font.GetHeight()));
}
Rect2D TextPiece::GetBoundsInLine()
{
	return Rect2D(Point2D(pos.x,textline->y), Size2D(wdth, textline->h));
}
void TextPiece::Draw(Graphics& g)
{
	if (0 != (flgs & TextPiece::Atomic))
	{
		if (image != 0)
		{
			Rect2D dst(pos, image.GetSize());
			Rect2D src(dst.GetSize());
			g.DrawImage(image, src,dst);
			return;
		}
	}	
	g.DrawText(text, colr, font, pos);
}

#if 0
#pragma mark -
#endif

// ToDo BUG, select all, CTRL click within selection, SHIFT click within selection


TextDocument::TextDocument()
: selColor(255,255,255), selBgColor(0,0,80)
, selectionFirst(-1U),selectionSize(0)
, acceptKeyENTER(true)
, acceptKeyTAB(true)
, forwardCaret(true)
, withinDND(false)
{
	// let's claim the document is 640 pixels wide and let's render it right now
	{
		Graphics g;
	    Populate();
		ASSERT(!pieces.empty());
	    _MeasurePieces(g);
	}
	SetDocumentWidth(640);
//	docSize.w = 640;   
//	docSize.h = 0;
//	_RenderPieces(pieces.begin());

	caretRect.x = -1;
	caretRect.y = -1; // (*lb)->textline->y;
	caretRect.w = 2;
	caretRect.h = 1; // (*lb)->textline->h;
	caretVPos = caretRect.GetNW();
	MoveCaretTo(0,0,0,0);
	
}
TextDocument::~TextDocument()
{
}

void TextDocument::SetSingleLineMode()
{
	SetDocumentWidth(0x4000000); // wide enough to not need new lines
	// SetIndent(0);
	acceptKeyENTER = false;
	acceptKeyTAB = false;
	pieces.front()->wdth = 1;
}

void TextDocument::Populate()
{
	ColorRGB colr(0,0,0); // black

//	FontName fn;
//	fn.faceName = String::From_c_str("Verdana");
//	fn.fontSize = 9;	
//	TextFont tfont(fn);
	TextFont tfont;	 // use the system default font
	
	pieces.reserve(1);
	// make sure the original new line is there
	Piece tr( new TextPiece() );
	tr->TurnIntoNewLine();
	tr->font = tfont; 
	tr->wdth = 20; // the original indent
	pieces.push_back(tr);
}

void TextDocument::_MeasurePieces(Graphics& g)
{
	Pieces::iterator b = pieces.begin();
	Pieces::iterator e = pieces.end();
	for(; b!=e; ++b)
		(*b)->Measure(g);
}

/*void TextDocument::RenderPieces()
{
	_RenderPieces(pieces.begin());
}*/
void TextDocument::DiscardRendering()
{
	Pieces::iterator b = pieces.begin();
	Pieces::iterator e = pieces.end();
	for(; b!=e; ++b)
		(*b)->textline = 0; // no more textline assigned to this piece
}

void TextDocument::_RenderPieces(Pieces::iterator renderB)
{
	Pieces::iterator renderE = pieces.end();
	
	if (renderB == renderE)
		return;

	// (g) needed for char measurements when a line needs to be split
	Graphics g;	 
	
	// allocate the pool of pieces to be the result of this render
    Pieces newPieces;
    newPieces.reserve(pieces.size()*9/8); // reserve sligtly more to allow for splits

    // we need to start from the start of a textline so go back a bit
    // actually we go back one more line because a change in the current line
    // might allow some pieces to fit on the previous line
	Pieces::iterator piecesB = pieces.begin();
	int linesBack=1;
	for(int lBn=linesBack; ; --lBn)
	{   // go to the start of the line
		refc<TextLine> textline1( (*renderB)->textline );
		while ((renderB != piecesB) && ( (*(renderB-1))->textline == textline1 )) 
			--renderB;
		// go to the previous line
		if (lBn==0)
			break;
		if (renderB != piecesB)
			--renderB;
	}	
	// look at the previous line and set the current (y) at the bottom of that
	sint32 y = 0;
	if ((renderB != piecesB) && ((*(renderB-1))->textline != 0))
		y = (*(renderB-1))->textline->y 
		  + (*(renderB-1))->textline->h;
	
    // copy the pieces before (renderB) into (newPieces) without change
    newPieces.insert(newPieces.end(), piecesB, renderB);

	// this vector will hold the pieces for the line currently beeing processed
	// we need this so that insertions into the line do not reallocate the full
	// document and also to be able to traverse the line more than once
    Pieces linePieces;
    linePieces.reserve(64); // may grow beyond that as we use this buffer

    // remember what is the piece containing the caret so we can preserve the caretPos
	Piece caretPiece = pieces.front();
	if (caretPiece->textline != 0)
	{   // get the piece that the caret is on top of
		TextPos pos;
		pos.LocatePosition(pieces,g, caretRect.x, caretRect.y, 0,0, forwardCaret);
		caretPiece = *pos.pieceP;
	}
	uint32 caretPieceX = caretRect.x - caretPiece->pos.x;

	// now process the (renderB, renderE) range
    while((renderB != renderE) || !linePieces.empty())
    {
		sint32 x = 0;  // 
		sint32 a = 0;  // max ascent within the line
		sint32 d = 0;  // max descent within the line
        
    	if (!linePieces.empty())
    	{   // we already have one piece from the previous line in the buffer
    		linePieces.back()->pos.x = x;
    		x +=  linePieces.back()->wdth;
			a = linePieces.back()->GetAscent();
			d = linePieces.back()->GetDescent();
    		linePieces.back()->pos.y = y-a;
			// textline = linePieces.back()->textline; // this textline may be used on the prev line
    	}
		// collect pieces into (linePieces) for the new line
		for(;;) 
		{
			if (! linePieces.empty())
			{
				Piece& backPiece = linePieces.back();
				// now determine the new right edge of the line    	
		    	x = backPiece->pos.x + backPiece->wdth;
		    	// now see if we exceed the max width a line is allowed to have
		    	if (x > docSize.w)
		    	{
		    		// if the piece can be broken, break it
					if (0 == (backPiece->flgs & TextPiece::CannotLineBreak)) 
					{ 
						Piece fragment = backPiece->SplitAtLineEnd(g, docSize.w);
						if (fragment != 0)
						{
							if (backPiece->isEmpty())
							{
								if (caretPiece == backPiece)
									caretPiece = fragment;
								backPiece = fragment;
								// if this is the only piece, force a split
								if (linePieces.size() == 1) 
								{
									fragment = backPiece->SplitAt(g, docSize.w);
									if (fragment != 0 && backPiece->isEmpty())
									{
										if (caretPiece == backPiece)
											caretPiece = fragment;
										backPiece = fragment;
										fragment = 0;
									}
								}
								else
									fragment = 0;
							}
							if (fragment != 0) // need to push into the line
							{
								if ((caretPiece == backPiece) && (backPiece->wdth < caretPieceX))
								{
									caretPiece  = fragment;
									caretPieceX -= backPiece->wdth;
							    }
								if (selectionSize)
								{
									if (0 != (fragment->flgs & TextPiece::Selected))
										++selectionSize;
									if (selectionFirst >= newPieces.size()+linePieces.size())
										++selectionFirst;
								}
								linePieces.push_back(fragment);
							}
						}
					}
		    	}
		    	// this line is full , time to flush the results
		    	if (x >= docSize.w)
		    		break;
		    }

		    if (renderB == renderE) 
		    	break;

	    	// if we reached a NewLine this line is over !
	    	if (0 != ((*renderB)->flgs & TextPiece::NewLine))
	    	{
	    		if (!linePieces.empty())
	    			break;
	    		else
	    		{	// the new line goes outside to the left of the document
		    		ASSERT(x == 0);
		    		x -= 0; // (*renderB)->wdth
		    	}
	    	}

		    {   // there is more room, try to add one more piece to the line
		   		(*renderB)->pos.x = x; // fix the location also helps the merge
		   		(*renderB)->textline = 0; // helps the merge
		    	if (!linePieces.empty() && linePieces.back()->MergeWith(*renderB))
		    	{
					if (caretPiece == *renderB)
					{
						caretPiece  =  linePieces.back();
						caretPieceX += caretPiece->wdth - (*renderB)->wdth;
					}
		    		if (selectionSize)
		    		{
						if (0 != ((*renderB)->flgs & TextPiece::Selected))
							--selectionSize;
						if (selectionFirst >= newPieces.size()+linePieces.size())
							--selectionFirst;
					}
		    	}
		    	else
		    	{
		    		linePieces.push_back(*renderB); // add one more item to the line

					// determine the baseline for the new piece and remember 
					sint32 fa = (*renderB)->GetAscent();
					(*renderB)->pos.y = y-fa;   // the topleft corner of the character
					a = std::max(a, fa);
					d = std::max(d, (*renderB)->GetDescent());

				}
				++renderB; // piece consumed
		    } // added one more piece to the line
	    } // the line pieces have been acumulated 
	    
    	if (!linePieces.empty())
    	{
			refc<TextLine> textline2(new TextLine());
			textline2->y = y;
			y += (textline2->h = a+d);

			// adjust the pieces in the line 
			Pieces::iterator lineB = linePieces.begin();
			Pieces::iterator lineE = linePieces.end();
			if (x > docSize.w)
				--lineE; // preserve the last piece out of line
			for(Pieces::iterator lineT=lineB; lineT!=lineE; ++lineT)
			{
				(*lineT)->pos.y += a;  // lower the baselines 
				(*lineT)->textline = textline2;
			}
		    newPieces.insert(newPieces.end(), lineB, lineE); // flush the line
		    linePieces.erase(lineB, lineE); // remove the used pieces
		}    
		
		// see if we still need to render pieces individually or we can simply reuse
		// the previous rendering till the end of the document
		if (linePieces.empty() && (renderB != renderE) &&
			((*renderB)->pos.x == 0) && ((*renderB)->textline != 0) && 
			(linesBack-- == 0) )
		{
				break;
//UnitTest::Log(">>>>>>>> render complete at '%s'",(*renderB)->text.c_str());
		}
    } //  while(renderB != renderE)

	// the rest of the pieces can be copied unchanged
    newPieces.insert(newPieces.end(), renderB, renderE);

    // the lines below only need to be adjusted on their vertical position
    while(renderB != renderE)
    {
    	refc<TextLine> textline3((*renderB)->textline);
		sint32 deltaY = y - textline3->y;
		if (deltaY == 0)
			break;
		textline3->y += deltaY;
		y += textline3->h;
		for(; (renderB!=renderE) && (textline3==(*renderB)->textline); ++renderB) 
			(*renderB)->pos.y += deltaY;
    }
    // if adjustments reached the end of the document we need to adjust 
    // the document size
    if (renderB == renderE)
		docSize.h = y;
    	
	// snap the new pieces in place
    pieces.swap(newPieces);
        
	// adjust the caret to the new pos
	if ((caretRect.x != caretPiece->pos.x+caretPieceX) || 
		(caretRect.y != caretPiece->textline->y))
	{
		// change caret virtual pos to force MoveCaretTo to act
	    MoveCaretTo(caretPiece->pos.x+caretPieceX, caretPiece->textline->y,0,0);
		Selection_MarkTail(); 
	}
}

void TextDocument::Draw(Graphics& g, const Rect2D& area)
{
	//g.FillRect(ColorRGB(0,0xB0,0), area.Intersect(Rect2D(docSize)));
	Pieces::iterator e = pieces.end();
	Pieces::iterator b = pieces.begin();
	std::pair<Pieces::iterator, Pieces::iterator> theLine = 
		std::equal_range(b,e , area.y, PageOrder(area.h));
	b = theLine.first;
	e = theLine.second;
	if (b == e)
		return;
	// located the page, now draw it
	for(Pieces::iterator p=b; p!=e; ++p)
	{
//		sint32 fa = p->font.GetAscent();
//		sint32 fd = p->font.GetDescent();
//		Rect2D b(p->pos, Size2D(p->wdth, fa+fd));
		if (0 == ((*p)->flgs & TextPiece::Selected))
		{
			Rect2D b((*p)->GetBounds());
			if (!b.Intersects(area))
				continue;
			(*p)->Draw(g);
		}
		else
		{
			Rect2D b((*p)->GetBoundsInLine());
			if (!b.Intersects(area))
				continue;
			g.FillRect(selBgColor, b);
			//(*p)->Draw(g);
			g.DrawText((*p)->text, selColor, (*p)->font, (*p)->pos);
		}
	}
}

// allows binary search of a line that contains a specific coordinate y
// defines the operator less than   p < y
bool TextDocument::LineOrder::operator () (Piece& p, sint32 y)
{
	return  p->textline->y+p->textline->h <= y;
}
// defines the operator less than   y < p
bool TextDocument::LineOrder::operator () (sint32 y, Piece& p)
{
	return  y < p->textline->y;
}
// defines the operator less than   p < x
bool TextDocument::PieceInLineOrder::operator () (Piece& p, sint32 x)
{
	return  p->pos.x+p->wdth <= x;
}
// defines the operator less than   x < p
bool TextDocument::PieceInLineOrder::operator () (sint32 x, Piece& p)
{
	return  x < p->pos.x;
}
//  for drawing the visible page only
TextDocument::PageOrder::PageOrder(sint32 ph)
: h(ph)
{}
// defines the operator less than   p < y
bool TextDocument::PageOrder::operator () (Piece& p, sint32 y)
{
	return  p->textline->y+p->textline->h <= y;
}
// defines the operator less than   y+h < p
bool TextDocument::PageOrder::operator () (sint32 y, Piece& p)
{
	return  y+h < p->textline->y;
}

bool TextDocument::PieceOrder::operator () (Piece& piece, const Point2D& point)
{
	if (piece->textline->y + piece->textline->h <= point.y)
		return true;
	if (point.y < piece->textline->y)
		return false;
	// same line, look at x
	return (piece->pos.x + piece->wdth <= point.x);
}
bool TextDocument::PieceOrder::operator () (const Point2D& point, Piece& piece)
{
	if (point.y < piece->textline->y)
		return true;
	if (piece->textline->y + piece->textline->h <= point.y)
		return false;
	// same line, look at x
	return (point.x < piece->pos.x);
}

/*
// y is the pos where we want to go
// nl are the number of lines we want to go from there
// bl,el are the iterators to the pieces that make the resulting line
bool TextDocument::FindLine(sint32 y, sint32 nl,
							Pieces::iterator& bl, Pieces::iterator& el)
{
	Pieces::iterator e = pieces.end();
	Pieces::iterator b = pieces.begin();
	ASSERT(b != e);
	
//	std::pair<Pieces::iterator, Pieces::iterator> theLine = 
//		std::equal_range(b,e , y, LineOrder());
//	bl = theLine.first;
//	el = theLine.second;

	el = std::upper_bound(b, e, y, LineOrder());
	if (el != b)
		bl = std::lower_bound(b, el, (*(el-1))->textline->y, LineOrder());
	else
		el = std::upper_bound(bl=el, e, (*el)->textline->y, LineOrder());

	VERIFY (bl != el)
//		return false;

	// now skip the specified number of lines
	for ( ; nl<0; ++nl)
	{
		el = bl;
		if (bl == b)
			return false;
		--bl;
		bl = std::lower_bound(b, bl, (*bl)->textline->y, LineOrder());
	}	
	for ( ; nl>0; --nl)
	{
		bl = el;
		if (el == e)
			return false;
		el = std::upper_bound(bl+1, e, (*bl)->textline->y, LineOrder());
	}	
	return bl != el;
}

bool TextDocument::FindPiece(sint32 x, Pieces::iterator& b, Pieces::iterator& e)
{
	if (b == e)
		return false;
	b = std::lower_bound(b, e, x, PieceInLineOrder());
	return (b != e);
}
*/

void TextDocument::TextPos::AdjustAfterResize(Pieces::iterator p, Pieces& pieces)
{
	// p old iterator from where things are changed
	// pieces, new vector after insertion/erasure
	Pieces::iterator b = pieces.begin();
	Pieces::iterator e = pieces.end();
	sint32 deltaZ = (e-b)-(piecesE-piecesB);

	lineB = b + (lineB-piecesB) + ((lineB >= p) ? deltaZ : 0);
	lineE = b + (lineE-piecesB) + ((lineE >= p) ? deltaZ : 0);
	pieceP = b + (pieceP-piecesB) + ((pieceP >= p) ? deltaZ : 0);
    ASSERT(e == (b + (piecesE-piecesB) + ((piecesE >= p) ? deltaZ : 0)));
    piecesE = e;
    piecesB = b;							     
}

void TextDocument::TextPos::LocatePosition(
			Pieces& pieces, Graphics& g,
			sint32 x, sint32 y, sint32 nc, sint32 nl, 
			bool forwardCaret )
{
	piecesB = pieces.begin();
	piecesE = pieces.end();
	ASSERT(piecesB != piecesE);

	lineE = std::upper_bound(piecesB, piecesE, y, LineOrder());
	if (lineE != piecesB)
		lineB = std::lower_bound(piecesB, lineE, (*(lineE-1))->textline->y, LineOrder());
	else
		lineE = std::upper_bound(lineB=lineE, piecesE, (*lineE)->textline->y, LineOrder());
	ASSERT (lineB != lineE)
	// now skip the specified number of lines
	for ( ; nl<0; ++nl)
	{
		if (lineB == piecesB)
			break;
		lineE = lineB;
		--lineB;
		lineB = std::lower_bound(piecesB, lineB, (*lineB)->textline->y, LineOrder());
	}	
	for ( ; nl>0; --nl)
	{
		if (lineE == piecesE)
			break;
		lineB = lineE;
		lineE = std::upper_bound(lineB+1, piecesE, (*lineB)->textline->y, LineOrder());
	}	
	VERIFY (lineB != lineE)
	// we are at the proper line , now find the proper piece within the line

	pieceP = std::lower_bound(lineB, lineE, x, PieceInLineOrder());
	// if we fail to find the piece it must be because it's the last
	if (pieceP == lineE) --pieceP; 

	// maybe we want to go to the end of the 
	// prev piece if this is the )[ pos in between pieces
	if ( forwardCaret && (pieceP != lineB) && 
		 (x == (*(pieceP-1))->pos.x+(*(pieceP-1))->wdth) ) 
		--pieceP;
	
	(*pieceP)->MeasureEachCharWidth(g,charPlacements);
	std::vector<uint32>::iterator cp = charPlacements.begin();
	if (0 != ((*pieceP)->flgs & TextPiece::NewLine))
	{   // we won't go to the left of a NewLine
		if (nc<0) nc = 0;  
		cp = charPlacements.end()-1;
	}
	else
	{
		cp = std::lower_bound(charPlacements.begin(), charPlacements.end()-1, x-(*pieceP)->pos.x);
		// align x to the left edge of the char
		if (*cp > x-(*pieceP)->pos.x) --cp;
    }
    
	// we found the proper character, now let's move around
	for(; nc<0; ++nc)
	{
		while (cp == charPlacements.begin())
		{
			if (pieceP == piecesB)
			{
				nc = 0;
				break;
			}
			if (pieceP == lineB)
			{
				lineE = lineB;
				lineB = std::lower_bound(piecesB, lineE, (*(lineE-1))->textline->y, LineOrder());
			}
			--pieceP; // piece change within the same line
			(*pieceP)->MeasureEachCharWidth(g,charPlacements);
			cp = charPlacements.end()-1;
			if (0 != ((*pieceP)->flgs & TextPiece::NewLine))
			{
				nc = 0;
				break;	 // we won't go to the right of an EOL
			}
		}
		--cp;
    }
	for(; nc>0; --nc)
	{
		// we may already be at the end of the current piece
		// so maybe we cannot increment
		if (++cp == charPlacements.end())
			--cp, ++nc; // we also take back the counted nc
		// if we reached the EOP (end of piece) we automatically 
		// jump to the start of the next piece. On screen the 2 places
		// look identicall but the insertion cares ....
		while (cp == charPlacements.end()-1)
		{
			if (pieceP+1 == lineE) // Wrap EOL
			{
				if (pieceP+1 == piecesE) // Wrap EOL
				{
					nc = 0;
					break;	 
				}
				if (0 != ((*(pieceP+1))->flgs & TextPiece::NewLine))
				{
					nc = 0;
					break;	 // we won't go to the right of an EOL
				}
				lineB = lineE;
				lineE = std::upper_bound(lineB+1, piecesE, (*lineB)->textline->y, LineOrder());
			}
			++pieceP;
			(*pieceP)->MeasureEachCharWidth(g,charPlacements);
			cp = charPlacements.begin();
		}
	}
	// job done
	charNum = cp - charPlacements.begin();
	this->y = (*pieceP)->textline->y;
	this->x = (*pieceP)->pos.x + *cp;
}


bool TextDocument::MoveCaretTo(sint32 x, sint32 y, sint32 nc, sint32 nl)
{
	//x = std::max(0L,std::min(document->docSize.w, x));
	y = std::max(0L,std::min(docSize.h, y));

	bool preserveX = (caretVPos.x == x) && (nc == 0);
	bool preserveY = (caretVPos.y == y) && (nl == 0);

	if (preserveX && preserveY)
		return false;

	Graphics g;
	TextPos pos;
	pos.LocatePosition(pieces,g, x,y, nc,nl, forwardCaret);

	caretRect.x = pos.x;
	caretRect.y = pos.y;
	caretRect.h = (*pos.pieceP)->textline->h;

	if (!preserveX)
		caretVPos.x = caretRect.x;
//	if (!preserveY)
	caretVPos.y = caretRect.y; // always because backspace can skip lines
	return true;
}
bool TextDocument::IsCaretAtEOL()
{
	Graphics g;
	TextPos pos;
	pos.LocatePosition(pieces,g, caretRect.x,caretRect.y, 0,0, false);
	if (pos.pieceP + 1 == pos.lineE)
	{
		if (pos.x >= (*pos.pieceP)->pos.x + (*pos.pieceP)->wdth) 
			return true;
	}
	return false;
}
bool TextDocument::IsCaretAtBOL()
{
	Graphics g;
	TextPos pos;
	pos.LocatePosition(pieces,g, caretRect.x,caretRect.y, 0,0, true);
	if (pos.pieceP == pos.lineB)
	{
		if (0 != ((*pos.pieceP)->flgs & TextPiece::NewLine))
			return true;
	 	if (pos.x == (*pos.pieceP)->pos.x)
	 		return true;
	}
	return false;
}
bool TextDocument::IsCaretAtEOP()
{
	Graphics g;
	TextPos pos;
	pos.LocatePosition(pieces,g, caretRect.x,caretRect.y, 0,0, false);
	if (pos.pieceP + 1 == pos.lineE)
	{
		if (pos.x >= (*pos.pieceP)->pos.x + (*pos.pieceP)->wdth) 
		{
			if (pos.pieceP + 1 == pos.piecesE)
				return true;
			if (0 != ((*(pos.pieceP+1))->flgs & TextPiece::NewLine))
				return true;
		}
	}
	return false;
}
bool TextDocument::IsCaretAtBOP()
{
	Graphics g;
	TextPos pos;
	pos.LocatePosition(pieces,g, caretRect.x,caretRect.y, 0,0, true);
	if (pos.pieceP == pos.lineB)
	{
		if (0 != ((*pos.pieceP)->flgs & TextPiece::NewLine))
			return true;
	}
	return false;
}
bool TextDocument::IsCaretAtEOF()
{
	Graphics g;
	TextPos pos;
	pos.LocatePosition(pieces,g, caretRect.x,caretRect.y, 0,0, false);
	if (pos.pieceP + 1 == pos.piecesE) 
	{
		if (pos.x >= (*pos.pieceP)->pos.x + (*pos.pieceP)->wdth)
			return true;
	}
	return false;
}
bool TextDocument::IsCaretAtBOF()
{
	Graphics g;
	TextPos pos;
	pos.LocatePosition(pieces,g, caretRect.x,caretRect.y, 0,0, true);
	if (pos.pieceP == pos.piecesB)
	{
		if (0 != ((*pos.pieceP)->flgs & TextPiece::NewLine))
			return true;
		ASSERT(false);
//		if (pos.x == (*pos.pieceP)->pos.x)
//	 		return true;
	}
	return false;
}

/*	Determines the range of pieces between the 2 points provided
	returns true if the range is not empty */
bool TextDocument::_SplitRange( Point2D pointA, Point2D pointB,
							   Pieces::iterator& rb, Pieces::iterator& re )
{
	if (pointA == pointB)
		return false;
	// we place proper order to the 2 points 
	if ((pointA.y > pointB.y) || ((pointA.y == pointB.y) && (pointA.x > pointB.x)))
	{
		Point2D tmp(pointA);
		pointA = pointB;
		pointB = tmp;
	}

	Graphics g;
    TextPos  posA; posA.LocatePosition(pieces,g, pointA.x,pointA.y, 0,0, false);
    TextPos  posB; posB.LocatePosition(pieces,g, pointB.x,pointB.y, 0,0, true);
    
    if ((posA.x == posB.x) && (posA.y == posB.y))
    	return false;
    	
    if (posB.charNum != 0)
    {
	    if (posB.charNum < (*posB.pieceP)->text.length())
	    {
			Piece fragmentB = (*posB.pieceP)->SplitAtChar(
					(*posB.pieceP)->text.begin() + posB.charNum,
					posB.charPlacements[posB.charNum] );
		    _InsertPiece(posB.pieceP+1, fragmentB);
		    posA.AdjustAfterResize(posB.pieceP+1, pieces);
		    posB.AdjustAfterResize(posB.pieceP+1, pieces);
	    }
	    // the current piece remains the last in the selection
    }
    else
    {  // at the beginning of a wrapped line
		if (posA.pieceP == posB.pieceP)
	    	return false;
	    --posB.pieceP;
    }

    // check if we need to split posA
    if (posA.charNum != 0)
    {
	    if (posA.charNum < (*posA.pieceP)->text.length())
	    {
			Piece fragment1 = (*posA.pieceP)->SplitAtChar(
					(*posA.pieceP)->text.begin() + posA.charNum,
					posA.charPlacements[posA.charNum] );
		    _InsertPiece(posA.pieceP+1, fragment1);
		    posB.AdjustAfterResize(posA.pieceP+1, pieces);
		    posA.AdjustAfterResize(posA.pieceP+1, pieces);
	    }
		// the next piece is the first of the selection
		if (posA.pieceP == posB.pieceP)
		    ++posB.pieceP; 
	    ++posA.pieceP; 
    }
    else
    {
    	// that's fine,  we'll take the full piece, thank you
    }
    	
	rb = posA.pieceP;
	re = posB.pieceP+1;
	return true;
}							

/*	Pieces::iterator b = pieces.begin();
	Pieces::iterator e = pieces.end();
	// we will insert/remove from the vector, iterators will be lost
	uint32 sb = std::lower_bound(b, e, pointA, PieceOrder())-b;
	uint32 se = std::upper_bound(b+sb, e, pointB, PieceOrder())-b;

//    if ((sb != se) && ((*sb)->pos.x + (*sb)->wdth == pointA.x) && ((*sb)->wdth != 0))
//    	++sb;

    if ((sb != se) && ((*(b+se-1))->pos.x == pointB.x) && ((*(b+se-1))->wdth != 0))
    {
    	--se; // we move the end selection to the right edge of the prev piece
	    if (sb != se)
    		pointB = Point2D( (*(b+se-1))->pos.x+(*(b+se-1))->wdth, 
    						  (*(b+se-1))->textline->y  );
    }
	if (sb == se) // empty range
		return false;
    

	{   // deal witht partial pieces at start and end 
		uint32 sf = se-1; // the last piece within the range
	
		Piece secondHalf2 = (*(b+sf))->SplitAt(g, pointB.x);
		if (secondHalf2 != 0)
		{
			ASSERT (!(*(b+sf))->isEmpty());
		    _InsertPiece(b+se, secondHalf2);
			b = pieces.begin();
			e = pieces.end();
		}
	 
		Piece secondHalf1 = (*(b+sb))->SplitAt(g, pointA.x);
		if (secondHalf1 != 0)
		{
			if ((*(b+sb))->isEmpty())
	    	    *(b+sb) = secondHalf1;  
	    	else
	    	{
			    ++se; // unconditionally shift the range end since the range start shifted
			    _InsertPiece(b + ++sb, secondHalf1);
				b = pieces.begin();
				e = pieces.end();
			}
		}
		else if (++sb == se)  
		{   // we must skip the first half that was split out
			// but the second half is empty so we may have an empty selection now
			// if the range became empty , try to merge it back
			_MergeNeighbours(b+sb);
			b = pieces.begin();
			e = pieces.end();
			return false;
		}
	}

	rb = b+sb;
	re = b+se;
	return true;
*/    

void TextDocument::_InsertPiece(Pieces::iterator where, Piece& whatPiece)
{
    pieces.insert(where, 1, whatPiece);
    ASSERT(whatPiece != 0);

	if (0 != (whatPiece->flgs & TextPiece::Selected))
	{
		++selectionSize;
		if (selectionFirst >= where-pieces.begin())
			selectionFirst = where-pieces.begin();
	}
	else if (selectionSize && (selectionFirst >= where-pieces.begin()))
		++selectionFirst;
}
void TextDocument::_InsertPieces(Pieces::iterator where, 
								 Pieces::iterator b, Pieces::iterator e)
{
    pieces.insert(where, b,e);
	for (Pieces::iterator p=b; p!=e; ++p)
	{
		if (0 != ((*p)->flgs & TextPiece::Selected))
		{
			++selectionSize;
			if (selectionFirst >= (where-pieces.begin()) + (p-b))
				selectionFirst = (where-pieces.begin()) + (p-b);
		}
		else if (selectionSize && (selectionFirst >= (where-pieces.begin()) + (p-b)))
			++selectionFirst;
		
	}
}

void TextDocument::_MergeNeighbours(Pieces::iterator sb)
{
	Pieces::iterator b = pieces.begin();
	Pieces::iterator e = pieces.end();
	if (sb != e)
	{
		uint32 k = sb-b;  // we are erasing from the  vector,  iterators may be lost
		if ((b+k+1 != e) && (*(b+k))->MergeWith(*(b+k+1)))
		{
			if (0 != ((*(b+k))->flgs & TextPiece::Selected))
				--selectionSize;
			if (selectionSize && selectionFirst >= k+1)
				--selectionFirst;
//			UnitTest::Log("SelFirst=%d size=%d",selectionFirst,selectionSize);
			pieces.erase(b+k+1, b+k+2);
			b = pieces.begin();
			e = pieces.end();
		}
		if ((b+k != b) && (*(b+k-1))->MergeWith(*(b+k)))
		{
			if (0 != ((*(b+k))->flgs & TextPiece::Selected))
				--selectionSize;
			if (selectionSize && selectionFirst >= k)
				--selectionFirst;
//			UnitTest::Log("SelFirst=%d size=%d",selectionFirst,selectionSize);
			pieces.erase(b+k, b+k+1);
		}
	}
}

void TextDocument::InsertPieces(Pieces& newPieces)
{
	if (newPieces.empty())
		return;
    Pieces::iterator toRender;
    {
		Graphics g;
		TextPos pos;
		pos.LocatePosition(pieces,g, caretRect.x,caretRect.y, 0,0, forwardCaret);	
		uint32 insertionX = caretRect.x;
		Piece fragment = *pos.pieceP; // an empty piece will be split out of this

		{   // make the pieces fit into the new context
			Pieces::iterator pb=newPieces.begin();
			Pieces::iterator pe=newPieces.end();
			for(; pb!=pe; ++pb)
			{
				(*pb)->pos.x = insertionX;
				insertionX += (*pb)->wdth;
				(*pb)->textline = fragment->textline;
			}
		}
		// compute the number of pixels inserted
		insertionX -= caretRect.x; 
		{  // we must also adjust all pieces to the right of this piece by the same amount
			Pieces::iterator op = pos.pieceP+1;
			for (; op != pos.lineE; ++op)
				(*op)->pos.x += insertionX;
		}
		
		// now we neet to split the pieces so that the newpieces can be inserted
		fragment = (*pos.pieceP)->SplitAt(g,caretRect.x);
		Pieces::iterator insertionAt;
		if (fragment == 0)
		{
			// insert after (pieceP)
			insertionAt = pos.pieceP+1;
		}
		else if ((*pos.pieceP)->isEmpty())
		{
			*pos.pieceP = fragment;
			fragment->pos.x += insertionX;
			// insert before (pieceP)
			insertionAt = pos.pieceP;
		}
		else
		{
			// ToDo BUG:if this is selected it will be double counted by this insertion
			// but during insertion the selection should be off already
			newPieces.push_back(fragment);	 
			fragment->pos.x += insertionX;
			// insert after (pieceP)
			insertionAt = pos.pieceP+1;
		}
		
		// insert the new fragments into the vector
		_InsertPieces(insertionAt, newPieces.begin(), newPieces.end());
		pos.AdjustAfterResize(insertionAt, pieces); // recompute based on the new begin()

		// adjust the caret by the inserted amount
		caretVPos.x = (caretRect.x += insertionX);

		// all the text pieces including CR have been inserted in a straight line
		// it is the job of the renderer to layout the pieces and possibly merge them
		toRender = pos.pieceP;
		
		// [...] here the Graphics and the TextPos objects go out of scope
	}
	_RenderPieces(toRender);
}


// insert the bloody text right after the caret
void TextDocument::InsertText(const char* sb, const char* se)
{
	if (sb >= se)
		return;
	Pieces newPieces;
	newPieces.reserve(128);
    {
		Graphics g;
		TextPos pos;
		pos.LocatePosition(pieces,g, caretRect.x,caretRect.y, 0,0, forwardCaret);	
		Piece fragment = *pos.pieceP; // an empty piece will be split out of this

	   	char specials[] = { '\r','\n','\t'};
	    for(;;)
	    {
	    	const char* spos = std::find_first_of(
	    			sb,se,	specials,specials+sizeof(specials));
	  		if (sb != spos)
	  		{	// now do the insertion of plain chars (common case)
	  			fragment= fragment->SplitEmpty(); // split a new piece
				newPieces.push_back(fragment);
				fragment->text.insert(fragment->text.begin(), sb, spos);
				fragment->Measure(g);
		    }
	    	if (spos == se)
	    		break;
	    	sb = spos;
	    	if (*sb == '\r' || *sb == '\n')
	    	{ 	// Insert CR
	  			fragment= fragment->SplitEmpty(); // split a new piece
				newPieces.push_back(fragment);
				fragment->TurnIntoNewLine();
				// fragment->Measure(g);
				fragment->wdth = pieces.front()->wdth; // just duplicate the indent of first paragraph

	 		   	if (*sb == '\r')
		    		if (++sb == se)	break;
	 		   	if (*sb == '\n')  // also covers \r\n
		    		if (++sb == se)	break;
	    	}
	    	else if (*sb == '\t')
	    	{ 
	    		// Insert TAB
	    		// ToDo [....]
	    		if (++sb == se)	break;
	    	}
	    }  // end of insertion for loop

		// [...] here the Graphics and the TextPos objects go out of scope
	}
	InsertPieces(newPieces);
}

void TextDocument::SetDocumentWidth(uint32 w)
{
	if (w <= 128)
		w = 128;  // a document cannot be thinner than that
	if (docSize.w == w)
		return;
	docSize.w = w;
	DiscardRendering();
	_RenderPieces(pieces.begin());
}

#if 0
#pragma mark -
#endif

bool TextDocument::Selection_Contains(const Point2D& point)
{
	Pieces::iterator b = pieces.begin();
	Pieces::iterator e = pieces.end();
	Pieces::iterator p = std::lower_bound(b, e, point, PieceOrder());
    if (p == e)
    	return false;
    if (!(*p)->GetBoundsInLine().Contains(point))
    	return false;
	if (0 == ((*p)->flgs & TextPiece::Selected))
		return false;
	return true;
}
void TextDocument::Selection_SelectAll()
{
	Pieces::iterator b = pieces.begin();
	Pieces::iterator e = pieces.end();
	for (; b!=e; ++b)
		(*b)->flgs |= TextPiece::Selected;
	selectionFirst = 0;
	selectionSize  = pieces.size();
	selectionTail  = Point2D(0,0); // any shift select will select from start to caret
}

void TextDocument::Selection_MarkTail()
{
	selectionHead = selectionTail = caretRect.GetNW();
}

/* 	Creates a selection starting at tail and up to the caret
	Unselects from the caret to the old head
	Then sets the head to the current caret */
bool TextDocument::Selection_CreateNew()
{
	Point2D pointA(caretRect.GetNW());
	if (pointA == selectionHead)
		return false; // optimization while the mouse is dragged over the same char
	bool r = Selection_UnselectPieceRange(selectionTail, selectionHead);
	selectionHead = pointA;
	r |= Selection_SelectPieceRange(selectionTail, selectionHead);
//	UnitTest::Log(">SelFirst=%d size=%d",selectionFirst,selectionSize);
	return r;
}

bool TextDocument::Selection_UnselectPieceRange(Point2D pointA, Point2D pointB)
{
	Pieces::iterator sb, se;
	if (!_SplitRange(pointA, pointB, sb,se))
		return false;

	for (Pieces::iterator u = sb; u!=se; ++u)
	{
		if (0 == ((*u)->flgs & TextPiece::Selected))
			continue; // already unselected ?!
		(*u)->flgs &= ~TextPiece::Selected;
		--selectionSize;
	}

	Pieces::iterator b = pieces.begin();
	Pieces::iterator e = pieces.end();
    if (selectionSize == 0)
    	selectionFirst = -1U;
	else if (selectionFirst >= sb-b && selectionFirst < se-b)
	{   // we must search for the new start of selection range
    	selectionFirst = -1U;
		for (Pieces::iterator t = se; t!=e; ++t)
		{
			if (0 != ((*t)->flgs & TextPiece::Selected))
			{
				selectionFirst = t-b;
				break;
			}
		}
		ASSERT(selectionFirst != -1)
	}
	
	// maybe we can merge some adjiacent text
	if (se != b && sb != se-1)
		_MergeNeighbours(se-1);
	_MergeNeighbours(sb);
	
	return true;
}

bool TextDocument::Selection_SelectPieceRange(Point2D pointA, Point2D pointB)
{
	Pieces::iterator sb, se;
	if (!_SplitRange(pointA, pointB, sb,se))
		return false;

	Pieces::iterator b = pieces.begin();
	if (selectionFirst > sb-b)
		selectionFirst = sb-b;
		
//	UnitTest::Log("SELECTION ===================================== start(%d)",selectionSize);
	// all the other pieces are fully becoming selected
	for (; sb!=se; ++sb)
	{
		if (0 != ((*sb)->flgs & TextPiece::Selected))
		{
//			UnitTest::Log("SELECTION [%s] already selected", (*sb)->text.c_str());
			continue; // already selected
		}
		else
		{
			(*sb)->flgs |= TextPiece::Selected;
			++selectionSize;
//			UnitTest::Log("SELECTION [%s] added", (*sb)->text.c_str());
		}
	}
//	UnitTest::Log("SELECTION ===================================== end(%d)",selectionSize);
	
	// maybe we can merge some adjiacent selections
	if (se != b && sb != se-1)
		_MergeNeighbours(se-1);
	_MergeNeighbours(sb);

	return true;
}


bool TextDocument::Selection_Unselect()
{
    if (selectionSize == 0)
    	return false;

	Pieces::iterator b = pieces.begin()+selectionFirst;
	Pieces::iterator e = pieces.end();
	
	ASSERT( b!=e );
	ASSERT (0 != ((*b)->flgs & TextPiece::Selected));
	
	while( b!=e )
	{
		if (0 != ((*b)->flgs & TextPiece::Selected))
		{
			// found one => find the whole range
			Pieces::iterator se = b;
			for(; se!=e; ++se)
			{
				if (0 == ((*se)->flgs & TextPiece::Selected))
					break;
				(*se)->flgs &= ~TextPiece::Selected;
				--selectionSize;
			}
			// maybe we can merge some adjiacent selections
			if (se-1 != b)
				_MergeNeighbours(se-1);
			_MergeNeighbours(b);
			// because of the merges b can actually be smaller at 
			// this point than it was at entry we must retest it...
			e = pieces.end(); // end also may have changed
			
			if (selectionSize == 0)
				break;
		}
		else
			++b;
	}

	ASSERT(selectionSize == 0);
	
	selectionHead = selectionTail;
	selectionFirst = -1U;
	selectionSize = 0;    
	return true;
}
void TextDocument::Selection_DeleteText( bool preserveCaret )
{
	if (selectionSize == 0)
		return;

	// now remove all pieces with empty text
	Pieces::iterator piecesB = pieces.begin();
	Pieces::iterator piecesE = pieces.end();

	Pieces::iterator selB = piecesB + selectionFirst;
	if (selB == piecesB)
	{   // do not allow the original NL to be deleted, rather just unselect it
		(*selB)->flgs &= ~TextPiece::Selected;
		++selB;
		--selectionSize;
	}
	else
	{
		ASSERT (0 != ((*selB)->flgs & TextPiece::Selected));
	}
	
    if (selectionSize)
    {
		ASSERT( selB != piecesE );
		if (!preserveCaret)
		{   // we go to the start of the first selected piece just before deletion
			Point2D pos = (*(selB-1))->pos;
			pos.x += (*(selB-1))->wdth;
			// this will keep the caret out of the deletion area
	    	forwardCaret = true; 
			MoveCaretTo( pos.x,pos.y, 0,0);
	    }

		{   // move all unselected pieces to the front while discarding the selected ones 
			Pieces::iterator b = selB;
			Pieces::iterator a = b;
			while( b != piecesE )
			{   // we need to delete this
				if (0 != ((*b)->flgs & TextPiece::Selected))
					++b, --selectionSize;
				else
					*a++ = *b++;
			}
			pieces.erase(a,b);
		}
		// the original NL cannot be deleted
		ASSERT(!pieces.empty())
		// now we need to ask the renderer to redo the distribution
		// of pieces on screen
		_RenderPieces(selB);
		ASSERT(selectionSize == 0);
	}
	selectionHead = selectionTail = caretRect.GetNW();
	selectionFirst = -1U;    
	selectionSize = 0;    
}

void TextDocument::CopyFromDocument(DragDropType::TypeSet& availableTypes, 
								   	DragDropType::FatData& fatData )
{
	availableTypes.clear();
	if (selectionSize == 0)
		return;

	Pieces::iterator piecesB = pieces.begin();
	Pieces::iterator piecesE = pieces.end();

	Pieces::iterator selB = piecesB + selectionFirst;
	uint32 selZ = selectionSize;
	if (selB == piecesB)
	{   // don't care about the original NL
		++selB; 
		--selZ;
	}

    // we need to join all the rich text into one big plain text string
    String theText;
    theText.reserve(64000);
    bool continuum = true;
	for ( ;(selB != piecesE) && (selZ>0); ++selB )
	{
		if (0 == ((*selB)->flgs & TextPiece::Selected))
		{
			continuum = false;
			continue;
		}
		--selZ;
		if  (!continuum && !theText.isEmpty())
		{
			theText.compact();
			fatData.text_ASCII.push_back(theText);
			theText.clear();
		    theText.reserve(64000);
			continuum =true;
		}
		if (0 != ((*selB)->flgs & TextPiece::Atomic))
		{
			if (0 != ((*selB)->flgs & TextPiece::NewLine))
			{
				char nl[] = { '\n' };
				theText.append(nl, nl+1);
			}
			else if ((*selB)->image != 0)
			{
				fatData.image = (*selB)->image;
			}
		}
		else if (!(*selB)->text.isEmpty())
		{
			theText.append((*selB)->text);
		}
	}

	if (!theText.isEmpty())
	{
		theText.compact();
		fatData.text_ASCII.push_back(theText);
	}
	theText.clear();
	if (!fatData.text_ASCII.empty())
		availableTypes.insert(&DragDropType::Text_Plain);
	if (fatData.image != 0)
		availableTypes.insert(&DragDropType::Image_Native);
}


void TextDocument::Selection_PrepareForPaste(DragDropType::Effect&  effect,
											 bool onlySimulate)
{
	if (withinDND) // this is a local DND, special handling required
	{
		if (!Selection_Empty() && Selection_Contains(caretRect.GetNW()))
		{
			effect = 0; // can do nothing here
			return;
		}
		if (onlySimulate)
			return;
		// local DND drop outside selection
		if (effect == DragDropType::Copy)
			Selection_Unselect(); 
		else if (effect == DragDropType::Move)
			Selection_DeleteText(true); // delete text but preserve caret
	}
	else
	{
		if (onlySimulate)
			return;
		if (!Selection_Empty() && Selection_Contains(caretRect.GetNW()))
			Selection_DeleteText(); // drop on top of selection removes the selection
		else
			Selection_Unselect(); // drop outside of selection cancels the selection first
	}
}

void TextDocument::PasteToDocument(DragDropType::TypeSet& availableTypes, 
					        DragDropType::Effect&  effect,
							bool 				   onlySimulate,
							DragDropType::FatData& fatData )
{
	// we only support copy and move
	effect &= (DragDropType::Copy | DragDropType::Move);

	DragDropType::TypeSet::iterator e = availableTypes.end(); 
	if (availableTypes.find(&DragDropType::Image_Native) != e)
	{
		if (0 != (effect & DragDropType::Move))
			effect = DragDropType::Move; // priority
		Selection_PrepareForPaste(effect, onlySimulate);
		if (onlySimulate)
			return;
		if (fatData.image == 0)
			effect = 0;
	    if (effect == 0)
	    	return;

		TextDocument::Piece  newp = pieces.front()->SplitEmpty(); // to preserve the font,color
		newp->SetImage(fatData.image);
		
		TextDocument::Pieces newps;
		newps.push_back(newp);
		InsertPieces(newps); // this wil move the caret too
		return;
	}  // image
	
	if (availableTypes.find(&DragDropType::Text_Plain) != e)
	{
		if (0 != (effect & DragDropType::Move))
			effect = DragDropType::Move; // priority
		Selection_PrepareForPaste(effect, onlySimulate);
		if (onlySimulate)
			return;
	    if (fatData.text_ASCII.empty())
	    	effect = 0;
	    if (effect == 0)
	    	return;

		std::vector<String>::iterator sb = fatData.text_ASCII.begin();
		std::vector<String>::iterator se = fatData.text_ASCII.end();
		char nl[1] = { '\n' };
		for(; sb!=se; ++sb)
		{
			String& s = *sb;
			InsertText(s.begin(), s.end()); // this wil move the caret too
			if (sb+1 != se) // if we have several pieces we drop them on separate lines
				InsertText(nl,nl+1); 
		}
		return;
	} //text,filename

	if (availableTypes.find(&DragDropType::FileName) != e)
	{
		effect &= DragDropType::Copy; // only tolerate copy for filenames
		Selection_PrepareForPaste(effect, onlySimulate);
		if (onlySimulate)
			return;
	    if (fatData.fileNames.empty())
	    	effect = 0;
	    if (effect == 0)
	    	return;

		std::vector<FileName>::iterator sb = fatData.fileNames.begin();
		std::vector<FileName>::iterator se = fatData.fileNames.end();
		char nl[1] = { '\n' };
		for(; sb!=se; ++sb)
		{
			String s = sb->GetFullPathNameOS();
			InsertText(s.begin(), s.end()); // this wil move the caret too
			if (sb+1 != se) // if we have several pieces we drop them on separate lines
				InsertText(nl,nl+1); 
		}
		return;
	} //text,filename
	
	effect = 0;
	return;
} // _DoPaste


String TextDocument::GetText()
{
    String rText;

	Pieces::iterator piecesB = pieces.begin();
	Pieces::iterator piecesE = pieces.end();
	if (piecesB != piecesE)
		++piecesB;   // don't care about the original NL

    String theText;
    theText.reserve(64000);
	for ( ;piecesB != piecesE; ++piecesB )
	{
		if (0 != ((*piecesB)->flgs & TextPiece::Atomic))
		{
			if (0 != ((*piecesB)->flgs & TextPiece::NewLine))
			{
				char nl[] = { '\n' };
				theText.append(nl, nl+1);
			}
		}
		else if (!(*piecesB)->text.isEmpty())
		{
			theText.append((*piecesB)->text);
			if (theText.length()+1024 >= theText.capacity())
			{
				rText.append(theText);
				theText.setsize(0);
			}
		}
	}

	rText.append(theText);
	return rText;
}


#if 0
#pragma mark -
#pragma mark class TextView::ScrollListener
#endif

class TextView::ScrollListener : public View::ViewListener
{
	TextView* textview;
public:
	ScrollListener(TextView* v)  : textview(v) {}
	~ScrollListener()	      	 { CancelListener(); }
	void CancelListener() 		 { textview = 0; super::CancelListener(); }
protected:
	virtual void ViewScroll(const Size2D& scrollSize, 
							const Rect2D& bounds, 
							View* /*originView*/)
	{
		if (textview == 0)
			return;
		if (textview->IsWithinNotify())
			return; // no recursive loops

		if ((scrollSize.w > 0) || (scrollSize.h > 0))
		{
			Size2D docSize(textview->document->docSize);
			Rect2D targB(textview->GetLocalBounds());
			sint32 x = textview->docPos.x; // unchanged
			sint32 y = textview->docPos.y; // unchanged
			if (scrollSize.w > 0)
			{
				sint32 w = std::max(0L, scrollSize.w - bounds.w);
				x = std::min(w, bounds.x);
				if (x > 0) // implies w>0 too
					x = (x * (docSize.w-targB.w)) / w; // scale to our space
			}
			if (scrollSize.h > 0)
			{
				sint32 h = std::max(0L, scrollSize.h - bounds.h);
				y = std::min(h, bounds.y);
				if (y > 0) // implies h>0 too
					y = (y * (docSize.h-targB.h)) / h; // scale to our space
			}
			// do we need to scroll to a specific position ? , if yes do so
			if ((x != textview->docPos.x) || (y != textview->docPos.y))
				textview->ScrollTo(x,y,true);
		}
		else 
		{
			switch (bounds.x)
			{
			case  1: textview->Action_ScrollRight();	break;	
			case -1: textview->Action_ScrollLeft();		break;	
			case  2: textview->Action_ScrollPageRight();break;	
			case -2: textview->Action_ScrollPageLeft();	break;	
			}
			switch (bounds.y)
			{
			case  1: textview->Action_ScrollDown();		break;	
			case -1: textview->Action_ScrollUp();		break;	
			case  2: textview->Action_ScrollPageDown();	break;	
			case -2: textview->Action_ScrollPageUp();	break;	
			}
		}	
	} // function ViewScroll
}; // ScrollListener


void TextView::ListenToScrollNotificationsFrom(const View::owner& view)
{
	if (scrollListener == 0)
		scrollListener = new ScrollListener(this);
	view->AddListener(scrollListener);
}
void TextView::CancelListenToScrollNotifications()
{
	if (scrollListener != 0)
	{
		scrollListener->CancelListener(); // needed 
		scrollListener=0;
	}
}


TextView::TextView()
: mouseDragState(None)
, anchorDocumentWidth(true)
{
	document = new TextDocument();
}

TextView::~TextView()
{
	CancelListenToScrollNotifications();
}
void TextView::SetSingleLineMode() 
{ 
	anchorDocumentWidth = false;
	document->SetSingleLineMode(); 
}

void TextView::SetDocumentWidth(uint32 w)
{
	anchorDocumentWidth = false;
    document->SetDocumentWidth(w);
}
void TextView::AnchorDocumentWidth()
{
	Rect2D r(GetSkinContentBounds());
	anchorDocumentWidth = true;
    document->SetDocumentWidth(r.w);
}

void TextView::SetBounds(const Rect2D& bounds)
{
	View::SetBounds(bounds);
	if (anchorDocumentWidth)
		AnchorDocumentWidth();
	else if (IsFocused())
		SetCaret(document->caretRect);
    NotifyChange_ViewScroll(document->docSize, 
    						Rect2D(docPos,GetBounds().GetSize()) );
}

void TextView::DemoPopulate()
{
	TextDocument::Pieces pieces;
	{
		pieces.reserve(1000);
	    // other fonts
	    FontName fn;
		fn.faceName = String::From_c_str("Courier");
		fn.fontSize = 9;	
		TextFont courier9(fn);		
		fn.faceName = String::From_c_str("Comic Sans MS");
		fn.boldFont = true;	
		fn.fontSize = 9;	
		TextFont comic9(fn);		
		fn.fontSize = 15;	
		fn.boldFont = false;	
		fn.italicFont = true;	
		fn.underlineFont = true;	
		TextFont comic15BU(fn);		
		
	    Graphics g;
		for (unsigned y=1; y<pieces.capacity(); ++y)
		{
			refc<TextPiece> tr = new TextPiece();

			if (y%20 == 19)
			{
				tr->TurnIntoNewLine();
				tr->wdth = 20;
			}
			else switch (y%3)
			{
			case 0: tr->text = String::From_c_str("Here is some text."); break;
			case 1: tr->text = String::From_c_str("And some more(")
							 +String::From_uint32(y)
							 +String::From_c_str(")."); break;
			case 2: tr->text = String::From_c_str("And the final of 3 pieces."); break;
			};

			tr->colr = ColorRGB(y&1 ? 0 :200, y&2 ? 0:100, y&4 ? 0:100);
			switch (y&3)
			{
			case 0:
			case 1:	tr->font = courier9; break;
			case 2:	tr->font = comic9; break;
			case 3:	tr->font = comic15BU; break;
			}	
			tr->Measure(g);
			pieces.push_back(tr);
		}
	}
	document->InsertPieces(pieces);
}


Rect2D TextView::View2Document(const Rect2D& vr)
{
	Rect2D sb(GetSkinContentBounds());
	Rect2D r(vr);
	r.OffsetBy(docPos.x-sb.x, docPos.y-sb.y);
	return r;
}
Rect2D TextView::Document2View(const Rect2D& dr)
{
	Rect2D sb(GetSkinContentBounds());
	Rect2D r(dr);
	r.OffsetBy(sb.x-docPos.x, sb.y-docPos.y);
	return r;
}

void TextView::Draw(Graphics& g, const Rect2D& area)
{
	Rect2D oldClip = g.GetClipRect();
	Rect2D r(GetLocalBounds());

	ViewSkin::owner skin = GetSkin();
	if (skin != 0)
	{
		StateSkin::owner stateSkin = skin->GetStateSkin(GetState());
		stateSkin->Draw(g, r,area);
		r = stateSkin->GetContent(r);
	}

	Rect2D clip(area.Intersect(r));
	if (!clip.isEmpty())
	{
		g.SetClipRect(clip);

		Point2D origin(g.GetOrigin());
		g.SetOrigin(origin + (r.GetNW() - docPos));
		clip.OffsetBy(docPos.x-r.x, docPos.y-r.y);
		document->Draw(g, clip);
		g.SetOrigin(origin);
		g.SetClipRect(oldClip);
	}
}

bool TextView::ScrollTo(sint32 x, sint32 y, bool rfsh)
{ // x,y in document coordinates
	x = std::max(0L,std::min(document->docSize.w, x));
	y = std::max(0L,std::min(document->docSize.h, y));
	if ((docPos.x == x) && (docPos.y == y))
		return false;
	docPos.x = x;
	docPos.y = y;
	if (rfsh)
	{
		if (IsFocused())
			SetCaret(document->caretRect);

	    NotifyChange_ViewScroll(document->docSize, 
	    						Rect2D(docPos,GetBounds().GetSize()) );

		Refresh();
	}
	return true;
}


void TextView::GoToVisibleCaret(bool rfsh)
{
	// r the visible area of the document in document coordinates
	Rect2D r(GetSkinContentBounds());
	r.OffsetBy(docPos.x-r.x, docPos.y-r.y);
	Rect2D caretRect(document->caretRect);
	sint32 x = (caretRect.x+caretRect.w > r.x+r.w) 
			? (caretRect.x+caretRect.w-r.w) 
			: ((caretRect.x < r.x) ? caretRect.x : docPos.x);
	sint32 y = (caretRect.y+caretRect.h > r.y+r.h) 
			? (caretRect.y+caretRect.h-r.h) 
			: ((caretRect.y < r.y) ? caretRect.y : docPos.y);
	ScrollTo(x,y,rfsh);
}

void TextView::SetCaret(const Rect2D& r)
{   // r in document coordinates
	Rect2D cr(r);
    cr.x -= (r.w>>1); //(r.w-(r.w>>1)); // make the cursor be in front of the char not on top of it
    if (cr.x < 0)
		cr.x = 0;
		
	Rect2D sb(GetSkinContentBounds());
	// adjust for skin content offset (but do not clip)
    // switch from doc coordinates to view coordinates
	cr.OffsetBy(sb.x-docPos.x, sb.y-docPos.y);  

	View::SetCaret(cr);
}

bool TextView::MoveCaretTo(sint32 x, sint32 y, 
						   sint32 nc, sint32 nl)
{
	bool r = document->MoveCaretTo(x,y,nc,nl);
	if (r)
	{
		GoToVisibleCaret(true);
		if (IsFocused())
			SetCaret(document->caretRect);
	}
	return r;
}


void TextView::MouseCursorSetup()
{
	Window wnd = GetParentWindow();
	if (wnd == 0)
		return;
	WinMgr& mgr = wnd.GetWinMgr();
	if (mgr == 0)
		return;
	wnd.SetMouseCursor(mgr.data->textCursor);
}

void TextView::MouseCaptureBroken()
{
	// MouseCursorSetup(); // no need, somebody else will receive an ENTER
   	mouseDragState = None;
	View::MouseCaptureBroken();
}

/*  click out select	unselect, move caret here, move select tail here
	click in select		nothing
	drag in select		initiate drag selection
	release in select   unselect
	
	drag				select head here
	SHIFT click			select head here
*/    
bool TextView::HandleMouseEvent( const UIEvent& ev )
{
	if (IsDisabled())
	{
		if (IsMouseCapturing())
		{
			EndMouseCapture(this);
			MouseCursorSetup();
			Refresh();
		}
		return false;
    }

	Point2D docMouse(docPos.x+ev.mousePos.x, docPos.y+ev.mousePos.y);
	switch (ev.origin)
	{
	case UIEvent::MouseDClick:
	case UIEvent::MouseClick:
		if (!IsFocused()) 
		{
			if (IsFocusable()) 
			{	// use the click to gain focus
				SetFocused();
				Refresh();
			}
		}

		if (UIEvent::MouseLeftState == (ev.modifiers & UIEvent::MModifMask))
		{ // plain click
		    if (document->Selection_Contains(docMouse))
		    { // plain click inside selection
		    	document->forwardCaret = false;
				MoveCaretTo(docMouse.x, docMouse.y,	0,0);
		    	mouseDragState = DragSelection;
				StartMouseCapture(this, ev.mousePos);
		    }
		    else
		    { // plain click outside selection
				if (document->Selection_Unselect())
					Refresh();
		    	document->forwardCaret = false;
				MoveCaretTo(docMouse.x, docMouse.y,	0,0);
				document->Selection_MarkTail(); 
		    	mouseDragState = DragToSelect;
				StartMouseCapture(this, docMouse);
		    }
			return true;
		}
		else if ((UIEvent::MouseLeftState|UIEvent::CtrlState) == (ev.modifiers & UIEvent::MModifMask))
		{ // ctrl click  => move here without removing selection
		    	document->forwardCaret = false;
				MoveCaretTo(docMouse.x, docMouse.y,	0,0); 
				document->Selection_MarkTail(); 
		    	mouseDragState = DragToSelect;
				StartMouseCapture(this, docMouse);
				return true;
		}
		else if ((UIEvent::MouseLeftState|UIEvent::ShiftState) == (ev.modifiers & UIEvent::MModifMask))
		{ // shift click  => select to here
		    	document->forwardCaret = false;
				MoveCaretTo(docMouse.x, docMouse.y,	0,0); 
			    document->Selection_CreateNew();
			    Refresh();
		    	mouseDragState = DragToSelect;
				StartMouseCapture(this, document->caretRect.GetNW());
				return true;
		}
		else if ((UIEvent::MouseLeftState|UIEvent::AltOptionState) == (ev.modifiers & UIEvent::MModifMask))
		{ // alt click  => drag the paper to scroll 
				Window wnd = GetParentWindow();
				if (wnd == 0)
					return true;
				WinMgr& mgr = wnd.GetWinMgr();
				if (mgr == 0)
					return true;
				wnd.SetMouseCursor(mgr.data->grabCursor);
		    	mouseDragState = DragToScroll;
				StartMouseCapture(this, -docPos);
				return true;
		}
		break;
		
	case UIEvent::MouseMove:
		switch (mouseDragState)
		{
		case None:
				// if tehre  is a selection then the mouse shape depends on where it's hovering
				if (!document->Selection_Empty())
				{
					Window wnd = GetParentWindow();
					VERIFY(wnd != 0);
					WinMgr& mgr = wnd.GetWinMgr();

			    	if (document->Selection_Contains(docMouse))
						wnd.SetMouseCursor(mgr.data->arrowCursor);
		    		else
						wnd.SetMouseCursor(mgr.data->textCursor);
		    	}
				break;
		case DragToScroll: // the mousePos is Captured
				ScrollTo(-ev.mousePos.x, -ev.mousePos.y);
				break;
		case DragToSelect: // the mousePos is Captured
		    	document->forwardCaret = false;
				MoveCaretTo(ev.mousePos.x, ev.mousePos.y, 0,0); 
			    document->Selection_CreateNew();
			    Refresh();
				break;
		case DragSelection: // the mousePos is Captured
			{   // release the capture and drag the selection
		    	EndMouseCapture(this);
				mouseDragState = None;
				
				DragDropType::TypeSet availableTypes;
				DragDropType::FatData fatData;
			    document->CopyFromDocument(availableTypes, fatData);
			    if (!availableTypes.empty())
			    {
					DragSource ds;
					ds.Set(fatData, availableTypes);
					DragDropType::Effect effect = DragDropType::Copy 
												| DragDropType::Move;
					document->withinDND = true;
			        ds.DoDND(effect);
					document->withinDND = false;

			        if (effect == DragDropType::Move)
			        {
			        	document->Selection_DeleteText();
						Refresh();
			        }
		        }	
		    	break;
		    }
		}
		return true;

	case UIEvent::MouseUp:
		switch (mouseDragState)
		{
		case None: // absolutely no use to us
				return true;
		case DragToScroll: // the mousePos is Captured
				Refresh();
				break;
		case DragToSelect: // the mousePos is Captured
				break;
		case DragSelection: // the mousePos is Captured
				if (document->Selection_Unselect())
					Refresh();
				break;
		}
		EndMouseCapture(this);
		MouseCursorSetup();
		mouseDragState = None;
		return true;

	case UIEvent::MouseWheel:
		if (0 == (ev.modifiers & UIEvent::ShiftState))
			ScrollTo(docPos.x,docPos.y - ev.count/10);
		else
			ScrollTo(docPos.x - ev.count/10,docPos.y);
		return true;
	};
	return false;
}

void TextView::Action_Up()
{
	if (document->Selection_Unselect())
		Refresh();
	MoveCaretTo(document->caretVPos.x,document->caretVPos.y,0,-1); 
	document->Selection_MarkTail();
	document->forwardCaret = false;
}
void TextView::Action_Down()
{
	if (document->Selection_Unselect())
		Refresh();
	MoveCaretTo(document->caretVPos.x,document->caretVPos.y,0,1); 
	document->Selection_MarkTail();
	document->forwardCaret = true;
}
void TextView::Action_Left()
{
	if (document->Selection_Unselect())
		Refresh();
	MoveCaretTo(document->caretVPos.x,document->caretVPos.y,-1,0); 
	document->Selection_MarkTail();
	document->forwardCaret = false;
}
void TextView::Action_Right()
{
	if (document->Selection_Unselect())
		Refresh();
	MoveCaretTo(document->caretVPos.x,document->caretVPos.y,1,0); 
	document->Selection_MarkTail();
	document->forwardCaret = true;
}
void TextView::Action_PageUp()
{
	if (document->Selection_Unselect())
		Refresh();
	Rect2D bounds = GetSkinContentBounds();
	ScrollTo(docPos.x, docPos.y-bounds.h,true);
	MoveCaretTo(document->caretVPos.x,document->caretVPos.y-bounds.h,0,0); 
	document->Selection_MarkTail();
	document->forwardCaret = false;
}
void TextView::Action_PageDown()
{
	if (document->Selection_Unselect())
		Refresh();
	Rect2D bounds = GetSkinContentBounds();
	sint32 y = std::max(0L, 
			   std::min(docPos.y+bounds.h,
						document->docSize.h-bounds.h + bounds.h/8L));
	ScrollTo(docPos.x, y, true);
	y = document->caretVPos.y+bounds.h;
	if (y < document->docSize.h)
		MoveCaretTo(document->caretVPos.x, y, 0,0);
	else // end of document
		MoveCaretTo(document->docSize.w,document->docSize.h-1,0,0); 
	document->Selection_MarkTail();
	document->forwardCaret = true;
}

void TextView::Action_LineBegin()
{
	if (document->Selection_Unselect())
		Refresh();
	ScrollTo(0,docPos.y, true);
	MoveCaretTo(0,document->caretVPos.y,0,0); 
	document->Selection_MarkTail();
	document->forwardCaret = false;
}

void TextView::Action_LineEnd()
{
	if (document->Selection_Unselect())
		Refresh();
//	Rect2D bounds = GetSkinContentBounds();
//	sint32 x = std::max(0L, document->docSize.w-bounds.w + bounds.w/8);
//	ScrollTo(x,docPos.y, true);
	MoveCaretTo(document->docSize.w,document->caretVPos.y,0,0); 
	document->Selection_MarkTail();
	document->forwardCaret = true;
}

void TextView::Action_SelectUp()
{
	MoveCaretTo(document->caretVPos.x,document->caretVPos.y,0,-1); 
    document->Selection_CreateNew();
	document->forwardCaret = false;
    Refresh();
}
void TextView::Action_SelectDown()
{
	MoveCaretTo(document->caretVPos.x,document->caretVPos.y,0,1); 
    document->Selection_CreateNew();
	document->forwardCaret = true;
    Refresh();
}
void TextView::Action_SelectLeft()
{
	MoveCaretTo(document->caretVPos.x,document->caretVPos.y,-1,0); 
    document->Selection_CreateNew();
	document->forwardCaret = false;
    Refresh();
}
void TextView::Action_SelectRight()
{
	MoveCaretTo(document->caretVPos.x,document->caretVPos.y,1,0); 
    document->Selection_CreateNew();
	document->forwardCaret = true;
    Refresh();
}
void TextView::Action_SelectPageUp()
{
	Rect2D bounds = GetSkinContentBounds();
	ScrollTo(docPos.x, docPos.y-bounds.h,false);
	MoveCaretTo(document->caretVPos.x,document->caretVPos.y-bounds.h,0,0); 
    document->Selection_CreateNew();
	document->forwardCaret = false;
    Refresh();
}
void TextView::Action_SelectPageDown()
{
	Rect2D bounds = GetSkinContentBounds();
	sint32 y = std::max(0L, 
			   std::min(docPos.y+bounds.h,
						document->docSize.h-bounds.h + bounds.h/8L));
	ScrollTo(docPos.x, y, false);
	y = document->caretVPos.y+bounds.h;
	if (y < document->docSize.h)
		MoveCaretTo(document->caretVPos.x, y, 0,0);
	else // end of document
		MoveCaretTo(document->docSize.w,document->docSize.h-1,0,0); 
    document->Selection_CreateNew();
	document->forwardCaret = true;
    Refresh();
}

bool TextView::Action_ScrollUp()
{
	return ScrollTo(docPos.x, docPos.y-12);
}
bool TextView::Action_ScrollDown()
{
	Rect2D bounds = GetSkinContentBounds();
	sint32 y = std::min(docPos.y+12L, document->docSize.h-bounds.h);
	return ScrollTo(docPos.x, y);
}
bool TextView::Action_ScrollLeft()
{
	return ScrollTo(docPos.x-12, docPos.y);
}
bool TextView::Action_ScrollRight()
{
	Rect2D bounds = GetSkinContentBounds();
	sint32 x = std::min(docPos.x+12L, document->docSize.w-bounds.w);
	return ScrollTo(x, docPos.y);
}
bool TextView::Action_ScrollPageUp()
{
	Rect2D bounds = GetSkinContentBounds();
	return ScrollTo(docPos.x, docPos.y-bounds.h);
}
bool TextView::Action_ScrollPageDown()
{
	Rect2D bounds = GetSkinContentBounds();
	return ScrollTo(docPos.x, docPos.y+bounds.h);
}
bool TextView::Action_ScrollPageLeft()
{
	Rect2D bounds = GetSkinContentBounds();
	return ScrollTo(docPos.x-bounds.w, docPos.y);
}
bool TextView::Action_ScrollPageRight()
{
	Rect2D bounds = GetSkinContentBounds();
	return ScrollTo(docPos.x+bounds.w, docPos.y);
}


bool TextView::HandleKeyEvent( const UIEvent& ev )
{
// move char up, down, left, right
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_UP))
	{
		Action_Up();
		return true;
	}
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_DOWN))
	{					 
		Action_Down();
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_LEFT))
	{
		Action_Left();
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_RIGHT))
	{
		Action_Right();
		return true;
    }
// Page Up, Down
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_PRIOR))
	{					 
		Action_PageUp();
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_NEXT))
	{				
		Action_PageDown();
		return true;
    }
// Line begin, end
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_HOME))
	{					 
		Action_LineBegin();
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_END))
	{
		Action_LineEnd();
		return true;
    }
// CTRL HOME/END paragraph begin, end 
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_HOME))
	{					 
		if (document->Selection_Unselect())
			Refresh();
		ScrollTo(0,docPos.y, true);
		MoveCaretTo(0,document->caretVPos.y,-0x70000000,0);   // go back many chars
		document->Selection_MarkTail();
    	document->forwardCaret = false;
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_END))
	{					 
		if (document->Selection_Unselect())
			Refresh();
		MoveCaretTo(document->docSize.w,document->caretVPos.y,0x70000000,0); // go forward many chars
		document->Selection_MarkTail();
    	document->forwardCaret = true;
		return true;
    }
// Document begin, end
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::CtrlState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_PRIOR))
	{					 
		if (document->Selection_Unselect())
			Refresh();
		MoveCaretTo(0,0,0,0); 
		document->Selection_MarkTail();
    	document->forwardCaret = false;
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::CtrlState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_NEXT))
	{					 
		if (document->Selection_Unselect())
			Refresh();
//		Rect2D bounds = GetSkinContentBounds();
//		sint32 x = std::max(0L, document->docSize.w-bounds.w + bounds.w/8);
//		sint32 y = std::max(0L, document->docSize.h-bounds.h + bounds.h/8);
//		ScrollTo(x,y, true);
		MoveCaretTo(document->docSize.w,document->docSize.h-1,0,0); 
		document->Selection_MarkTail();
    	document->forwardCaret = true;
		return true;
    }
// SHIFT move char up, down, left, right 
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::ShiftState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_UP))
	{
		Action_SelectUp();
		return true;
	}
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::ShiftState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_DOWN))
	{					 
		Action_SelectDown();
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::ShiftState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_LEFT))
	{
		Action_SelectLeft();
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::ShiftState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_RIGHT))
	{
		Action_SelectRight();
		return true;
    }
// SHIFT pgup,pgdn
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::ShiftState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_PRIOR))
	{					 
		Action_SelectPageUp();
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::ShiftState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_NEXT))
	{					 
		Action_SelectPageDown();
		return true;
    }
// SHIFT line beg, line end
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::ShiftState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_HOME))
	{	
		ScrollTo(0,docPos.y, false); // to make paragraph indent visible
		MoveCaretTo(0,document->caretVPos.y,0,0); 
	    document->Selection_CreateNew();
    	document->forwardCaret = false;
	    Refresh();
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::ShiftState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_END))
	{					 
//		Rect2D bounds = GetSkinContentBounds();
//		sint32 x = std::max(0L, document->docSize.w-bounds.w + bounds.w/8);
//		ScrollTo(x,docPos.y, false);
		MoveCaretTo(document->docSize.w,document->caretVPos.y,0,0); 
	    document->Selection_CreateNew();
    	document->forwardCaret = true;
	    Refresh();
		return true;
    }
// CTRL+SHIFT doc beg, doc end
	if ((ev.origin == UIEvent::KeyDown) && 
		((UIEvent::CtrlState|UIEvent::ShiftState) == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_PRIOR))
	{					 
		MoveCaretTo(0,0,0,0); 
	    document->Selection_CreateNew();
    	document->forwardCaret = false;
	    Refresh();
		return true;
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		((UIEvent::CtrlState|UIEvent::ShiftState) == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_NEXT))
	{					 
//		Rect2D bounds = GetSkinContentBounds();
//		sint32 x = std::max(0L, document->docSize.w-bounds.w + bounds.w/8);
//		sint32 y = std::max(0L, document->docSize.h-bounds.h + bounds.h/8);
//		ScrollTo(x,y, false);
		MoveCaretTo(document->docSize.w,document->docSize.h-1,0,0); 
	    document->Selection_CreateNew();
    	document->forwardCaret = true;
	    Refresh();
		return true;
    }
    
// Scroll text up, down, left, right, page up, down
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::AltOptionState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_UP))
	{	
		return Action_ScrollUp();
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::AltOptionState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_DOWN))
	{
		return Action_ScrollDown();
	}
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::AltOptionState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_LEFT))
	{
		return Action_ScrollLeft();
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::AltOptionState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_RIGHT))
	{
		return Action_ScrollRight();
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::AltOptionState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_PRIOR))
	{					
		return Action_ScrollPageUp(); 
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::AltOptionState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_NEXT))
	{					 
		return Action_ScrollPageDown();
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		((UIEvent::AltOptionState|UIEvent::CtrlState) == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_PRIOR))
	{					 
		Rect2D bounds = GetSkinContentBounds();
		return ScrollTo(0,0);
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		((UIEvent::AltOptionState|UIEvent::CtrlState) == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_NEXT))
	{					 
		Rect2D bounds = GetSkinContentBounds();
		return ScrollTo(document->docSize.w-bounds.w, document->docSize.h-bounds.h);
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::AltOptionState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_HOME))
	{					 
		return ScrollTo(0, docPos.y);
    }
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::AltOptionState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_END))
	{					 
		Rect2D bounds = GetSkinContentBounds();
		sint32 x = std::max(0L, document->docSize.w-bounds.w);
		return ScrollTo(x,docPos.y, true);
    }


// DEL
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_DELETE))
	{			
		Action_Delete();		
		return true;
    }
// BackSpace
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_BACK))
	{	
		Action_DeleteBack();
		return true;
    }

// CTRL+A select all
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::CtrlState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == 'A'))
	{		
		Action_SelectAll();
		return true;
    }
// CTRL+L select line
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::CtrlState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == 'L'))
	{					
		if (document->Selection_Unselect())
			Refresh();
		// end of line
		MoveCaretTo(0x70000000/*document->docSize.w*/,document->caretVPos.y,0,0); 
		document->Selection_MarkTail();
    	document->forwardCaret = true;
		// start of line
		MoveCaretTo(0,document->caretVPos.y,0,0); 
	    document->Selection_CreateNew();
    	document->forwardCaret = false;
	    Refresh();
		return true;
    }
// CTRL+V paste
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::CtrlState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == 'V'))
	{					
		Action_Paste();
		return true;
    }
// SHIFT INS paste
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::ShiftState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_INSERT))
	{					
		Action_Paste();
		return true;
    }
// CTRL+C copy
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::CtrlState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == 'C'))
	{					
		Action_Copy();
		return true;
    }
// CTRL+X cut
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::CtrlState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == 'X'))
	{					
		Action_Cut();
		return true;
    }
// CTRL INS copy
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::CtrlState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_INSERT))
	{					
		Action_Copy();
		return true;
    }
// CTRL DEL cut
	if ((ev.origin == UIEvent::KeyDown) && 
		(UIEvent::CtrlState == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_DELETE))
	{					
		Action_Cut();
		return true;
    }

// TAB = 8 spaces
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_TAB))
	{	
		if (!document->acceptKeyTAB)				
			return false;
		// the rest is just like for the ASCII key 
		
		if (!document->Selection_Empty())
			document->Selection_DeleteText(); // this will move the caret too

		// now  do insert the characters typed at the caret pos
	    char charcode[1] = { ev.code };
		document->InsertText(charcode,charcode+1); // this wil move the caret too

		GoToVisibleCaret(false);
		if (IsFocused())
			SetCaret(document->caretRect);

		document->Selection_MarkTail();
		Refresh();
		return true;
	}
	
// ENTER, RETURN
	if ((ev.origin == UIEvent::KeyDown) && 
		(0 == (ev.modifiers & UIEvent::KModifMask)) && 
		(ev.code == VK_RETURN))
	{	
		if (!document->acceptKeyENTER)				
			return false;
		// the rest is just like for the ASCII key 
		
		if (!document->Selection_Empty())
			document->Selection_DeleteText(); // this will move the caret too

		// now  do insert the characters typed at the caret pos
	    char charcode[1] = { ev.code };
		document->InsertText(charcode,charcode+1); // this wil move the caret too

		GoToVisibleCaret(false);
		if (IsFocused())
			SetCaret(document->caretRect);

		document->Selection_MarkTail();
		Refresh();
		return true;
    }
//ASCII
	if (ev.origin == UIEvent::AsciiCode)
	{
		if (!document->Selection_Empty())
			document->Selection_DeleteText(); // this will move the caret too

		// now  do insert the characters typed at the caret pos
	    char charcode[1] = { ev.code };
		document->InsertText(charcode, charcode+1); // this wil move the caret too

		GoToVisibleCaret(false);
		if (IsFocused())
			SetCaret(document->caretRect);

		document->Selection_MarkTail();
		Refresh();

//	    UnitTest::Log("Received ASCII key %d %c", ev.code, ev.code);
		return true;
    }
    
	return false;
}

void TextView::HandleFocused(bool on)
{
	// while scrolling the carret may go out of the visible range
	// use GoToCaret() to bring the visible range aroung the caret
	if (on) 
	{
		//ToDo: based on text size determine the caretRect size first
		SetCaret(document->caretRect);
	}
	Refresh();
}

void TextView::Action_DeleteBack()
{
	if (document->Selection_Empty())
	{	
		if (document->IsCaretAtBOF())
		{   // do nothing 
		}
		else if (document->IsCaretAtBOP())
		{   // select the previous NL separator by moving to the end of prev line
			document->MoveCaretTo(document->docSize.w,document->caretRect.y,0,-1); 
		    document->Selection_CreateNew();
	    	document->forwardCaret = true;
		}	
		else
		{	// select one char left
			document->MoveCaretTo(document->caretRect.x,document->caretRect.y,-1,0); 
		    document->Selection_CreateNew();
	    	document->forwardCaret = true;
	    }
	}

	document->Selection_DeleteText(); // this will move the carret too

	GoToVisibleCaret(false);
	if (IsFocused())
		SetCaret(document->caretRect);

	Refresh();
}
void TextView::Action_Delete()
{
	if (document->Selection_Empty())
	{	// select one char right
		document->MoveCaretTo(document->caretRect.x,document->caretRect.y,1,0); 
	    document->Selection_CreateNew();
    	document->forwardCaret = false;
	}

	document->Selection_DeleteText(); // this will move the carret too

	GoToVisibleCaret(false);
	if (IsFocused())
		SetCaret(document->caretRect);

	Refresh();
}
void TextView::Action_Paste()
{
	DragDropType::TypeSet availableTypes;
	DragDropType::FatData fatData;
	if (!PasteFromClipboard(availableTypes, false, fatData))
		return;

	DragDropType::Effect effect = DragDropType::Copy;
    document->PasteToDocument(availableTypes, effect, false, fatData);

	if (effect != 0)
	{
		GoToVisibleCaret(false);
		if (IsFocused())
			SetCaret(document->caretRect);

		document->Selection_MarkTail();
		Refresh();
	}
}
void TextView::Action_Copy()
{
	if (document->Selection_Empty()) 
		return; // nothing to do
	DragDropType::TypeSet availableTypes;
	DragDropType::FatData fatData;
    document->CopyFromDocument(availableTypes, fatData);
	CopyToClipboard(availableTypes, fatData);
}
void TextView::Action_Cut()
{
	// you don't want to delete the next char, only the selection
	// therefore a selection must be available
	if (document->Selection_Empty()) 
		return;
	Action_Copy();
	Action_Delete();
}
void TextView::Action_SelectAll()
{
	document->Selection_SelectAll();
	Refresh();
}

void TextView::SetText(const String& text)
{
	document->Selection_SelectAll();
	document->Selection_DeleteText();
	GoToVisibleCaret(false); // in case the text that comes in is much smaller than the one deleted
	document->InsertText(text.begin(), text.end());
	GoToVisibleCaret(false);
	if (IsFocused())
		SetCaret(document->caretRect);

	document->Selection_MarkTail();
}
String TextView::GetText()
{
	return document->GetText();
}


void TextView::HandleDrop(DragDropType::TypeSet& availableTypes, 
						  DragDropType::Effect&  effect,
						  const Point2D& 		 mousePos,
						  bool 					 onlySimulate,
						  DragDropType::FatData& fatData )
{
	// ToDo if threading interferes with this and it probably does,
	// just make an agent who's task is to insert this at (mousePos)
	// and put it into the synchronized queue

	if (IsDisabled())
		effect = 0;
	Rect2D sc(GetSkinContentBounds());
	if (!sc.Contains(mousePos))
		effect = 0;
	if (effect == 0)
		return;

	Point2D docMouse(docPos.x+mousePos.x-sc.x, docPos.y+mousePos.y-sc.y);
	document->forwardCaret = false;
	MoveCaretTo(docMouse.x, docMouse.y,	0,0);

    document->PasteToDocument(availableTypes, effect, onlySimulate, fatData);

	if ((effect != 0) && !onlySimulate)
	{
		GoToVisibleCaret(false);
		if (IsFocused())
			SetCaret(document->caretRect);

		document->Selection_MarkTail();
		Refresh();
	}
}



#if 0
#pragma mark -
#endif

MeasuredText::MeasuredText()
: wdth(0)
{}

MeasuredText::MeasuredText(const MeasuredText& t)
: wdth(t.wdth), font(t.font), colr(t.colr), text(t.text)
{
}

MeasuredText::~MeasuredText()
{
}

MeasuredText& MeasuredText::operator = (const MeasuredText& t)
{
	wdth = t.wdth;
	font = t.font;
	colr = t.colr;
	text = t.text;
	return *this;
}

void MeasuredText::SetText(const String& t)
{
	if (text != t)
		wdth = 0;
	text = t;
}

void MeasuredText::SetFont(const TextFont& t)
{
	if (font != t)
		wdth = 0;
	font = t;
}

void MeasuredText::Measure(Graphics& g)
{
	wdth = g.MeasureTextWidth(text.begin(), text.end(), font);
}

void MeasuredText::Draw(Graphics& g, const Point2D& target)
{
//	Rect2D c = g.GetClipRect();
//	g.SetClipRect(target);
	g.DrawText(text, colr, font, target); //.GetNW());
//	g.SetClipRect(c);
}


#if 0
#pragma mark -
#endif

AlignedText::AlignedText()
: algn(Rect2D::NW)
, ovrf(Rect2D::NW)
{
}
AlignedText::~AlignedText()
{
}
AlignedText::AlignedText(const AlignedText& t)
: algn(t.algn)
, ovrf(t.ovrf)
, MeasuredText(t)
{
}
AlignedText& AlignedText::operator = (const AlignedText& t)
{
	algn = t.algn;
	ovrf = t.ovrf;
	*(MeasuredText*)this = t;
	return *this;
}

void AlignedText::Draw(Graphics& g, const Rect2D& target)
{
	if (wdth == 0)
		Measure(g);
	Rect2D r = target.Align( Rect2D(Size2D(wdth, font.GetHeight())), algn, ovrf);
	MeasuredText::Draw(g, r.GetNW());
}


} // namespace XSP
