/*
Copyright (C) 2003 Hotsprings Inc.
For conditions of distribution and use, see copyright notice

Location: 
	www.HotspringsInc.com 

History:
	2003Apr14-GiuseppeG: code write

Description:
	This module offers platform independent file manipulation objects.

	FileName encapsulates the concept of a file name. It is not just a string. 
	The correct operators on filenames allow accessing the parent dir name, 
	the file extension, the base name and of course a file name can be used to 
	query properties of the file itself and to open it. FileNames can be converted
	to and from String objects according to their representation on one OS or another.
	
	SpecialFileNames: each operating system has a set of special folder names, 
	each user may have preferred places where he stores his downloads, each 
	application has a folder for its binary and for its resources. The SpecialFileNames
	object stores in a platform  independent way these settings.
	
	FileProperties: a file has a size, some access rights, a name, creation time, 
	modification time and some other properties. This class allows you to query and 
	manipulate those properties.
	
	FileList: The result of a folder content query, or a file search query. Instances of 
	this class can store collections of file names.
	
	FileFind: an algorithm to find files based on a pattern. Deep (recursive) searches 
	are supported.
	
	FileForkList: Enumerates the forks available in a file. Linux based and Win-FAT 
	file systems do not support forks so only the main fork is listed for those. On 
	MacOS we get the data fork and the resource fork. On Win-NTFS we get any number of
	file streams.  
	
	OpenedFile: after a file is opened an object of this type is available to allow 
	querries and manipulation of the file content. Read, Write access is atomic and 
	requires the position where in the file the operation should act (there is no 
	independent seek/pos operation). 
	
	FileModule: a singleton class with the static objects in this module ( an 
	initialized instance of the special file names object is available here ).
*/


namespace XSP
{

class FileProperties;
class FileForkList;
class FileList;
class OpenedFile;


#if 0
#pragma mark class FileName
#endif
// a platform independent path/filename encapsulator
// from platform specific string
class FileName
{
public:
	enum NameType { Root, Relative, Drive, Machine };

private:
	class _Data;
	refc<_Data> ref;

public:
	FileName();
	~FileName();

	FileName(const FileName& fn);
	FileName& operator=(const FileName& fn);

    bool operator == (const FileName& fn) const;
    bool operator != (const FileName& fn) const { return !(*this == fn); }

	int cmp(const FileName& fn) const;
    bool operator <= (const FileName& fn) const { return 0<=cmp(fn); }
    bool operator >= (const FileName& fn) const { return 0>=cmp(fn); }
    bool operator < (const FileName& fn) const { return 0<cmp(fn); }
    bool operator > (const FileName& fn) const { return 0>cmp(fn); }

	FileName& operator=(_Data* fn);
    bool operator == (_Data* fn) const;
    bool operator != (_Data* fn) const;
	 
public:
	const String& GetName() const;
	bool SplitNameExtension(String& baseName, String& extension) const;

public:
	bool HasParent() const;
	const FileName& GetParent() const;

public:
	typedef std::deque< FileName > Path;		
	void ToPath(Path& p) const;

public:
	static FileName From(const FileName& path, const String& fname);
	static FileName From(const FileName& path, const String& basename, const String& extension);

public:
	// mac =    name:name:name
	// linux=   /name/name/name
	// win =    \name\name\name
	String GetFullPathNameOS(bool dir=false) const;
	static FileName FromPathNameOS(const String& fullPathNameOS);

	String GetFullPathNameWin32(bool dir=false) const;
	static FileName FromPathNameWin32(const String& fullPathNameWin32);
	String GetFullPathNameLinux(bool dir=false) const;
	static FileName FromPathNameLinux(const String& fullPathNameLinux);
	String GetFullPathNameMac(bool dir=false) const;
	static FileName FromPathNameMac(const String& fullPathNameMac);

public:
	bool IsDir() const;
	bool IsLink() const;
	bool Exists() const;
	void Delete() const;
	void CreateDir(bool deep = true) const;
	void OpenWithOS();
	
	void GetFileProperties(); // time, read_only, owner, creator, type, ... whatever
						 // mime-type (check signature)
	// failIfNotExist, createIfNotExist, createDeepIfNotExist,
	// truncateOnOpen,
	// openExclusive, openReadShare 
	// OpenedFile OpenFile();
	FileForkList ListForks();
	
public:
	static void RunUnitTest();
};

#if 0
#pragma mark class SpecialFileNames
#endif
class SpecialFileNames 
{
public:
	class FileID 
	{
	public:
		inline FileID(){};
	};
	static const FileID SystemDir;	
	static const FileID MachineRoot;
	static const FileID UserHomeDir;
	static const FileID TempDir;
	static const FileID ProcessWorkingDir;
	static const FileID AppName;
	static const FileID AppDir;
	static const FileID UserPreferencesDir;
	static const FileID MachinePreferencesDir;
	static const FileID UserDesktopDir;
	static const FileID RecycleBinDir;
	static const FileID StartupDir;
	static const FileID ShutdownDir;
	static const FileID MenuDir;
private:
	typedef std::map<const FileID*, FileName> Specials;

	class _Data : public ref_obj
	{
	public:
		Specials specials;
	};
	refc<_Data> ref;

    void SetFileName(const FileID& pid, const FileName& prop);

public:
	SpecialFileNames();
	~SpecialFileNames();

	bool HasFileName(const FileID& pid) const;
	const FileName& operator[] (const FileID& pid) const;
}; // class SpecialFileNames	


#if 0
#pragma mark class FileProperties
#endif
// platform specific properties in a multiplatform presentation
// contains filename,size,dates, other properties(dynamic map of strings)
class FileProperties
{
public:
	class PropertyID 
	{
	public:
		inline PropertyID(){};
	};

	static const PropertyID Size;
	static const PropertyID CreationTime;
	static const PropertyID ModifyTime;
	static const PropertyID AccessTime;
	static const PropertyID BackupTime;
	static const PropertyID MacType;
	static const PropertyID MacCreator;
	static const PropertyID IsHidden;
	static const PropertyID IsDir;
	static const PropertyID IsLink;
	static const PropertyID IsReadOnly;
	static const PropertyID UID;
	static const PropertyID GID;
	static const PropertyID Mode; // linux access rights
	static const PropertyID IconLocation;
	static const PropertyID ResourceForkSize;
	static const PropertyID ShortName;

private:
	typedef std::map<const PropertyID*, String> Properties;

	class _Data
	{
	public:
		uint32 refcount;
		FileName filename;
		Properties properties;

	    _Data();
	    ~_Data();
	    void addref();
	    void release();
	};
	refc<_Data> ref;

private:
	void CopyOnWrite();
	
public:
	FileProperties();
	~FileProperties();
	FileProperties(const FileName& fname);
	FileProperties(const FileProperties& fn);
	FileProperties& operator=(const FileProperties& fn);

    void SetProperty(const PropertyID& pid, const String& prop);
	bool HasProperty(const PropertyID& pid) const;
	const String& operator[] (const PropertyID& pid) const;
	
    void SetFileName(const FileName& fname);
	const FileName& GetFileName() const;
}; // class FileProperties	

#if 0
#pragma mark class FileList
#endif

class FileList
{
public:
	typedef FileProperties value_type;
	typedef std::deque<value_type> ContentType;
	typedef ContentType::const_iterator iterator;
	typedef ContentType::size_type size_type;

private:	
	class _Data : public ref_obj
	{
	public:
		ContentType items;
	}; // class Data

	refc<_Data> ref;

public:
	FileList();
	~FileList();
	FileList(const FileList& fn);
	FileList& operator=(const FileList& fn);

	 // if folder fill with content of the folder
	 // if file just quote the file as the content of this list
	FileList(const FileName& fname);

	void Add(const FileName& fname);
	void Add(const FileProperties& entry);
	void ApplyFilter(const String& filter);
	
	const ContentType& GetContent() const;

	iterator begin() const;
	iterator end() const;
	size_type   size() const;
}; // class FileList

#if 0
#pragma mark class FileFilterPredicate
#endif
class FileFilterPredicate
{
	String pattern; // foo*bar??.*
	static bool Match(const char* pattern, const char* fname);
public:
	FileFilterPredicate(const String& pat)
	: pattern(pat)
	{}
	bool operator() (const FileProperties& fp) const
	{
		return Match(pattern.c_str(), fp.GetFileName().GetName().c_str());
	}
}; // class FilePatternPredicate

template <class FilterPredicate>
void FileFind(const FileName& home, 
			  bool deep, 
			  FilterPredicate& filter,
			  FileList& r)
{
	if (filter(home)) 
		r.Add(home);
	if (home.IsDir())
	{
		FileList d(home);
		FileFind(d,deep,filter,r);
/*		FileList::iterator b = d.begin();
		FileList::iterator e = d.end();
		for(FileList::iterator i=b; i!=e; ++i)
		{
			if (filter(*i))
				r.Add(*i);
			if (deep && i->isDir())
				FileFind(*i, deep, f, r);
		} 
*/
	}
}

template <class FilterPredicate>
void FileFind(const FileList& d, 
			  bool deep, 
			  FilterPredicate& filter,
			  FileList& r)
{
	FileList::iterator b = d.begin();
	FileList::iterator e = d.end();
	for(FileList::iterator i=b; i!=e; ++i)
	{
		if (filter(*i))
			r.Add(*i);
		if (deep && i->HasProperty(FileProperties::IsDir))
			FileFind(i->GetFileName(), deep, filter, r);
	}
}


#if 0
#pragma mark class FileForkList
#endif
class FileForkList
{
public:
	typedef String value_type;
	typedef std::deque<value_type> ContentType;
	typedef ContentType::const_iterator iterator;
	typedef ContentType::size_type size_type;
private:
	class _Data : public ref_obj
	{
	public:
		FileName filename;
		ContentType items;
	};

	refc<_Data> ref;

public:
	FileForkList();
	~FileForkList();
	FileForkList(const FileForkList& fn);
	FileForkList& operator=(const FileForkList& fn);

	FileForkList(const FileName& fname);
	
	const FileName& GetFileName() const;
	const ContentType& GetContent() const;

	iterator begin() const;
	iterator end() const;
	size_type size() const;
}; // class FileForkList	

#if 0
#pragma mark class OpenedFile
#endif
class OpenedFile
{
public:
	class _Data;
	typedef uint64 file_size;
	typedef uint32 block_size;
	typedef void*  data_type;

    class AccessType { public: AccessType() {} };
    static const AccessType ReadOnlyShared;
    static const AccessType ReadWriteExclusive;
    static const AccessType ReadWriteShared;
private:
	refc<_Data> ref;
public:
	OpenedFile();
	OpenedFile(const OpenedFile& of);
	OpenedFile& OpenedFile::operator= (const OpenedFile& of);
	~OpenedFile();
	
	bool operator == (const OpenedFile& of) const;
	bool operator != (const OpenedFile& of) const;
	
	OpenedFile( const FileName& fn, 
				const AccessType& acc = ReadOnlyShared, 
				const String& fork = String::kEmpty);

	file_size GetSize() const;
	void SetSize(file_size z) const;
	block_size Read(file_size pos, void* bf, block_size z) const;
	block_size Write(file_size pos, const void* bf, block_size z) const;
}; // class OpenedFile


#if 0
#pragma mark class FileModule
#endif
class FileModule
{
public:
	static FileModule* _instance;
	
public:
	FileModule();
	~FileModule();
public:
	SpecialFileNames specialFileNames;
}; // class FileModule

	
} // namespace XSP
