/* NSC 32000 ROM debugger.
 * Bruce Culbertson  Bob Krause
 *
 * Main, command table, expression parser, many command handlers.
 *
 *
 * added commands to do minimal tape support: backup, restore, wrf, fsf,
 * tread, twrite, and rewind.  --- Charles Rennolet 7/2/91
 *
 * Chanaged backup/restore to use default addresses.
 * Added the boot command.  -- Phil Nelson 8/28/91
 *
 * changed fsf to fspace.  Phil
 *
 * Changed disk label formats to the BSD disk label.  11/5/91
 *
 *
 */

#include "debugger.h"
#if LSC
#include <stdio.h>
#include <unix.h>
#include <storage.h>
#endif
/* GCC considers "inline" to be keyword */
#define inline _inline

char help_str [] =
"Command arguments may be expressions.  Type HELP = for expression syntax.\n";

long debug = 0;
long defaultBase = 16;
long timeout = -1;			/* support for auto boot. PAN */
#if !STANDALONE
char *fileBase = 0;			/* beginning of file buffer */
long fileTop = 0;			/* offset to end of file buffer */
#else
long rambuffAdr = 0x2000;		/* The "large" ram buffer. */
long rambuffSize = 64;			/* The size of the buffer. */
#endif
unsigned char *Dot = 0;                 /* current point in virtual space */

struct MachState machState;

/* Command interpreters */
int baseConverter(), disassemble(), dump(), quitHandler(),
 edit(), help(), help_cmds(),
 search(), mode(), printBkpt(),
 stackTrace(), baud(),
 backup(), restore(), sc_fspace(), sc_wrf(), /* CLR 7/2/91 */
 sc_rewind(), scsitapeRead(), scsitapeWrite(), /* CLR 7/2/91 */
 sc_bspace(), sc_espace(), /* PAN 9/15/91 */
#ifdef AUTOBOOT
 boot(), images(), partitions(),
#endif
 initialize(), init_machState(), run(), single_step(),
 fill(), move(), set(), show(), gpr(), cpu(), fpu(), mmu(),
 scsiRaw(), scsiRead(), scsiWrite(), crc(), download();

#if !STANDALONE
int getfile(), putfile();
#endif /* STANDALONE */

struct cmd {				/* The commands, their names, help */
  int (*fn)();
  char *name;
  char *help;
};

CONST
struct cmd cmd_tbl [] = {		/* Command Table */

#if STANDALONE
{ backup, "backup",
"Syntax: BACKUP <block> <count>.  Dumps blocks from disk to tape.  <Block>\n\
is the starting disk block and <count> is the number of blocks to dump.  Use\n\
fspace or rewind to position tape.  Backup uses ram starting at buf_adr and\n\
reads buf_size blocks at a time.  Disk and tape are defined by by disk_adr,\n\
disk_lun, tape_adr, and tape_lun."
},

{ baud, "baud",
"Syntax: BAUD <rate> [<uart>].  Set UART baud rate.  Console UART is 0 and\n\
is the default."
},

#ifdef AUTOBOOT
{ boot, "boot",
"Syntax: BOOT [<image id> [<SCSI adr> [<SCSI lun>]]].  Boot an image\n\
from the scsi disk.  Image id is a number or a name which selects the\n\
boot image to use.  Use the \"images\" command to list the images.  Default\n\
image is set in the disk header, SCSI adr and SCSI lun are as in read."
},
#endif /* AUTOBOOT */
#endif /* STANDALONE */

{ printBkpt, "breakpoint",
"List software breakpoints.  Use SET to set breakpoints, SET to 0 to clear."
},

#if STANDALONE
{ sc_bspace, "bspace",
"Syntax: BSPACE [<count> [<SCSI adr> [<SCSI lun>]]].  Skip past count\n\
blocks on tape. Count may be negative.  Default count is 1.\n\
Default tape parameters are tape_adr and tape_lun."
},
#endif

{ cpu, "cpu",
"Show CPU registers."
},

{ crc, "crc",
"Syntax: CRC <address> <length>.  Compute CCITT CRC on a section of memory."
},

{ disassemble, "disassemble",
"Syntax: DISASSEMBLE <address> <count>.  Disassembles <count> instructions\n\
starting at <address>."
},

{ download, "download",
"Syntax: DOWNLOAD <address>.  Download via serial line into memory.  Hit\n\
control-C to abort.  After transfer, hit <return> for status."
},

{ dump, "dump",
"Syntax: DUMP [<address> [<count>]].  Displays <count> bytes starting at\n\
<address>.  Default address is \".\" and default count is 16."
},

{ edit, "edit",
"Syntax: EDIT <address> [<size>].  Edits memory starting at <address>.  Give\n\
one or more expressions when prompted; the values will be placed in successive\n\
locations.  After hitting return, you will be prompted for more values.\n\
Editting stops when you hit return without a value.  Typing \"=\" instead of\n\
an expression causes the old value to be retained and advances the editor to\n\
the next location.  Bytes are written by default; bytes, words, or doubles are\n\
written if <size> is B, W, or D."
},

#if STANDALONE
{ sc_espace, "espace",
"Syntax: ESPACE [<SCSI adr> [<SCSI lun>]].  Skip to the end of the\n\
recorded tape. Default tape parameters are tape_adr and tape_lun."
},
#endif

{ fill, "fill",
"Syntax: FILL <address> <count> <pattern>.  Fills memory with byte pattern."
},

{ fpu, "fpu",
"Show FPU registers."
},

#if STANDALONE
{ sc_fspace, "fspace",
"Syntax: FSPACE [<count> [<SCSI adr> [<SCSI lun>]]].  Skip past count\n\
file marks on tape. Count may not be negative.  Default count is 1.\n\
Default tape parameters are tape_adr and tape_lun."
},
#endif

{ gpr, "gpr",
"Show general purpose registers."
},

{ help, "help",
"Syntax: HELP <command>.  Provides help for <command>."
},

#ifdef AUTOBOOT
{ images, "images",
"Syntax: IMAGES [<SCSI adr> [<SCSI lun>]]. Shows boot images that\n\
are available on the disk."
},
#endif

#if !STANDALONE
{ init_machState, "initialize",
"Reset stacks, mod table, interrupt table, initialize machine registers\n\
to handy values."
},
#else /* STANDALONE */
{ initialize, "initialize",
"Restart the monitor."
},
#endif

{ mmu, "mmu",
"Show MMU registers."
},

{ mode, "mode",
"Print mode variables."
},

{ move, "move",
"Syntax: MOVE <source adr> <dest adr> <cnt>.  Move bytes in RAM."
},

#ifdef AUTOBOOT
{ partitions, "partitions",
"Syntax: PARTITIONS [<SCSI adr> [<SCSI lun>]]. Shows partitions that\n\
are available on the disk. Sizes in 1K blocks, numbers in decimal."
},
#endif

#if !STANDALONE
{ quitHandler, "quit",
"Exit program."
},
#endif /* !STANDALONE */

#if STANDALONE
{ scsiRaw, "raw",
"Syntax: RAW <parameters> [<SCSI adr>].  Issues a command to a SCSI device.\n\
<Parameters> is the address of an array of eight pointers to the various\n\
SCSI buffers: data-out, data-in, command, status, dummy, dummy, message-out,\n\
message-in.  Default SCSI address is the variable disk_adr (power of 2)."
},
#endif

#if !STANDALONE
{ getfile, "read",
"Syntax: READ <file> [<address>].  Reads a file into the buffer starting\n\
at <address> or 0 if no address is given."
},
#else
{ scsiRead, "read",
"Syntax: READ <block> <RAM address> [<len> [<SCSI adr> [<SCSI lun>]]].\n\
Read 1 or <len> blocks from disk starting at <block> to RAM at <RAM address>.\n\
Default SCSI address and LUN are disk_adr (power of 2) and disk_lun.  Block\n\
length is same as used by disk."
},

{ restore, "restore",
"Syntax: RESTORE <block> <count>.  Restores <count> blocks from tape to\n\
disk.  <Block> is the starting block on the disk.  Use fspace or rewind\n\
to position tape.  Buffering and scsi addresses are the same as backup."
},

{ sc_rewind, "rewind",
"Syntax: REWIND [<SCSI adr> [<SCSI lun>]].  Rewind tape.  Default parameters\n\
are tape_adr and tape_lun."
},
#endif /* STANDALONE */

{ run, "run",
"Syntax: RUN [<address>].  Start user program running.  Default address is\n\
current PC."
},

{ search, "search",
"Syntax: SEARCH <start> <cnt> <byte> [<byte> ...].  Searches from <start> to\n\
<start>+<cnt> for all occurrences of the given pattern.  Eventually quits if\n\
many matches."
},

{ set, "set",
"Syntax: SET {<register> | <size>:<address>} <expression>.  Write a variable,\n\
register, or memory location with the value of <expression>.  The size of the\n\
memory location written will be a byte, word or double if <size> is B, W or\n\
D.  Type SHOW to get a complete list of register and variable names."
},

{ show, "show",
"List all registers and variables with their values."
},

{ single_step, "step",
"Syntax: STEP [<count>].  Execute one or <count> instructions from user\n\
program."
},

{ stackTrace, "trace",
"Syntax: TRACE <count> [<fp value>].  Display a trace of the top <count>\n\
stack frames, starting at FP or <fp value>."
},

#if !STANDALONE
{ putfile, "write",
"Syntax: WRITE <file> <address> <count>.  Writes <count> bytes to\n\
<file> from the buffer, starting at offset <address>."
},
#else
{ scsitapeRead, "tread",
"Syntax: TREAD <RAM address> [<len> [<SCSI adr> [<SCSI lun>]]].  Read 1 or\n\
<len> blocks from current position on tape to RAM at <RAM address>.  Default\n\
SCSI address and LUN are tape_adr (power of 2) and tape_lun.  Block length is\n\
same as used by tape drive."
},

{ scsitapeWrite, "twrite",
"Syntax: TWRITE <RAM address> [<len> [<SCSI adr> [<SCSI lun>]]].  Write 1 or\n\
<len> blocks to current position on tape from RAM at <RAM address>.  Default\n\
SCSI address and LUN are tape_adr (power of 2) and tape_lun.  Block length is\n\
same as used by the tape drive."
},

{ scsiWrite, "write",
"Syntax: WRITE <block> <RAM address> [<len> [<SCSI adr> [<SCSI lun>]]].\n\
Write 1 or <len> blocks from RAM at <RAM address> to disk starting at\n\
<block>.  Default SCSI address and LUN are disk_adr (power of 2) and disk_lun\n\
Block length is same as used by the disk."
},

{ sc_wrf, "wrmark",
"Syntax: WRMARK [<SCSI adr> [<SCSI lun>]].  Write one file mark at current\n\
position of tape."
},
#endif /* STANDALONE */

{ baseConverter, "=",
"Syntax: = <expression> [<base>].  Display the value of the expression in\n\
the given base or the default base.  Operators in order of precedence are:\n\
\t- ~ b: w: d:\n\
\t* / % << >> &\n\
\t+ - |\n\
()'s may be used to override precedence.  b'101, d'99 and h'ff are binary,\n\
decimal, and hex numbers.  A number without a prefix has the default radix.\n\
Registers may be used in expressions.  The prefix r' specifies a register\n\
for ambigious cases, e.g. f0 is a hex number but r'f0 is a register.\n\
b: w: and d: fetch bytes, words, and doubles from memory.  '<character> is\n\
the ASCII value of <character>.  V(<adr>[,<ptb>]) translates a virtual\n\
address to a physical address."
},

{ help, "?",
"Prints a list of commands."
},
};
#define CMDLEN (sizeof (cmd_tbl) / sizeof (struct cmd))

extern char end;

main(argc, argv)
int argc;
char **argv;
{
#if !STANDALONE
#else /* STANDALONE */
  zero_bss();	/* note: this needs to happen before almost anything */
  copy_dataseg();
  db_setbaud (DEFAULT_BAUD, DEFAULT_UART);
  myPrintf ("\nNS32000 ROM Debugger\nVersion: %s\n", version);
#ifdef AUTOBOOT
  myPrintf ("Auto-boot enabled.\n");
#endif
  myPrintf ("RAM free above 0x%lx\n\n", (long)(&end));
#endif
#if UNIX
  ttopen();
  if (argc > 1) getfile (argv [1]);
#endif
  init_machState();
  initBreaks();
  command_loop();
#if UNIX
  ttclose();
#endif
#if !STANDALONE
  exit (0);
#endif /* STANDALONE */
}

command_loop ()
{
  CONST struct cmd *q, *find_cmd();
  char *p, inline [LNLEN];

#ifdef AUTOBOOT
  timeout = 20000000;  /* About ? minutes? PAN */
#endif

  for (;;) {
    myprompt(inline, LNLEN, "Command (? for help): ");
#ifdef AUTOBOOT
    if (timeout == 0) {
       myPrintf ("\nAuto boot in progress.\n");
       boot ("");
       timeout = -1;
       continue;
    }
    timeout = -1;
#endif
    p = inline;
    scan (&p);
    if (*p == '\0' || *p == '\n') continue;
    /* if command is found in command table, call command handler */
    if (NULL != (q = find_cmd (&p))) (*(q->fn)) (p);
    else morePrintf (1, "Unknown command\n");
  }
}

/* Search cmd_tbl for 1) a command matching what the user typed or 2)
 * a unique command which is a superstring of what the user typed.
 */
CONST struct cmd *
find_cmd (p)
char **p;
{
  char token [LNLEN];
  CONST struct cmd *q, *substr;
  int multisub = 0;

  scanToken (p, token);
  for (q = cmd_tbl; q < cmd_tbl + CMDLEN; ++q)
    switch (myStrCmp (token, q->name)) {
      case CMP_MATCH:
	return q;
      case CMP_SUBSTR:
	++multisub;
	substr = q;
	break;
      case CMP_NOMATCH:
	break;
    }
  return (multisub == 1)? substr: NULL;
}

/* Scan next token from string pointed to by p.  Tokens are delimited
 * by white space.  Shift upper case chars to lower case.  Advance p
 * to point beyond token.
 */
scanToken (p, str)
char **p;
char *str;
{
  scan (p);
  while (**p != '\0' && **p != '\n' && **p != ' ' && **p != '\t')
    if (**p >= 'A' && **p <= 'Z') *str++ = (*(*p)++) - 'A' + 'a';
    else *str++ = *(*p)++;
  *str = '\0';
}

/* Advance p to point to non white space.
 */
scan (p)
char **p;
{
  while (**p == ' ' || **p == '\t') ++(*p);
}

#if UNIX
quitHandler(p)
char *p;
{
    ttclose();
    exit();
}
#endif

/* "?" command handler.  Print help and list of commands.
 */
help_cmds (p)
char *p;
{
    CONST struct cmd *q;

    morePrintf(1, help_str);
    morePrintf(1, "For additional help, type HELP <command>.  Commands are:\n");
    for (q = cmd_tbl; q < cmd_tbl + CMDLEN; ++q) 
        if ((q - cmd_tbl) % 5 == 4)
            morePrintf (1, "%s\n", q -> name);
        else
            morePrintf (0, "%-15s", q -> name);
    if ((q - cmd_tbl) % 5 != 0)
        morePrintf (1, "\n");
}

/* HELP command handler.  Print help on a particular command.
 */
help (p)
char *p;
{
  CONST struct cmd *q;

  if (NULL != (q = find_cmd (&p)))
    morePrintf (1, "%s\n", q -> help);
  else help_cmds (p);
}

/* DUMP command handler.  Hex dump routine.
 */
dump (p)
char *p;
{
    long adr, cnt, i, ret;
    char *q, printchars [24], *pch;

    switch (getIntScan (&p, &adr)) {
      case NO_NUM :
        adr = (long)Dot;
        cnt = 16;
        break;
      case BAD_NUM :
	myPrintf ("Bad argument\n");
        return;
      case GOT_NUM :
        ret = getIntScan(&p, &cnt);
        if (ret == NO_NUM) cnt = 16;
        else if (ret == BAD_NUM) {
	    myPrintf ("Bad argument\n");
            return;
	}
	break;
    }
    Dot = (unsigned char *)(adr + cnt);
    i = 0;
    pch = printchars;
    for (q = (char *)adr; q < (char *)(adr + cnt); ++q, ++i) {
        if ((i & 0xf) == 0) {
	    printhexbyt ((int)((long)q >> 24 & 0xff));
	    printhexbyt ((int)((long)q >> 16 & 0xff));
	    printhexbyt ((int)((long)q >> 8 & 0xff));
	    printhexbyt ((int)((long)q & 0xff));
	    morePrintf (0, " ");
        } else if ((i & 0xf) == 4 || (i & 0xf) == 8 ||(i & 0xf) == 12) {
            morePrintf(0, "| ");
        }
        printhexbyt (*(q + BASE));
        *pch++ = (*(q + BASE) >= 0x20 && *(q + BASE) <= 0x7e)?
          *(q + BASE): '.';
        morePrintf(0, " ");
        if ((i & 0xf) == 0xf) {             /* end of line */
            *pch = '\0';                      /* print printing characters */
            morePrintf(1, "%s\n", printchars);
            pch = printchars;
        }
    }
    if ((i & 0xf) != 0) {
        *pch = '\0';
        morePrintf (1, " %s\n", printchars);
    }
}  

/* EDIT command handler.  Interactively edit memory.
 */
edit (p)
char *p;
{
  char *q, inline [LNLEN];
  int  i, ret, len;
  long offset, mask;

  switch (getIntScan (&p, &offset)) {
    case NO_NUM:
      offset = (long)Dot;
      break;
    case BAD_NUM:
      myPrintf ("Bad address\n");
      return;
    default:;
  }
  scan (&p);
  switch (*p) {
    case 'w': case 'W':	len = T_SHORT;	mask = 0xffff; break;
    case 'd': case 'D':	len = T_LONG;	mask = 0xffffffff; break;
    case 'b': case 'B':
    default:		len = T_CHAR;	mask = 0xff; break;
  }
  q = (char *)offset;
  for (;;) {
    myprompt (inline, LNLEN, "%x(%x): ", q, *((long *)(q + BASE)) & mask);
    p = inline;
    scan (&p);
    if (*p == '\0' || *p == '\n') break;
    for (;;) {
      scan (&p);
      if (*p == '=') ++p;		/* = means leave old value */
      else if (GOT_NUM != (ret = getIntScan (&p, &i))) break;
      else switch (len) {
	case T_CHAR:	CHAR_STAR (q + BASE) = i;	break;
	case T_SHORT:	SHORT_STAR (q + BASE) = i;	break;
	case T_LONG:	LONG_STAR (q + BASE) = i;	break;
      }
      q += len == T_CHAR? 1: (len == T_SHORT? 2: 4);
    }
    if (ret == BAD_NUM) myPrintf ("Bad value\n");
  }
  Dot = (unsigned char *)q;
}

/* FILL command handler.  Fill memory with some value.
 */
fill (p)
char *p;
{
  long cnt, start, pattern;
  unsigned char *endp, *startp;

  if (GOT_NUM != getIntScan (&p, &start) ||
     GOT_NUM != getIntScan (&p, &cnt) ||
     GOT_NUM != getIntScan (&p, &pattern))
  {
    myPrintf ("Bad start address, count or pattern\n");
    return;
  }
  startp = (unsigned char *)start + BASE;
  for (endp = startp + cnt; startp < endp; ++startp)
    *(startp) = pattern;
}

/* MOVE command handler.  Move (really copy) a block of memory.
 */
move (p)
char *p;
{
  long cnt;
  unsigned char *start, *dest, *q;

  if (GOT_NUM != getIntScan (&p, &start) ||
     GOT_NUM != getIntScan (&p, &dest) ||
     GOT_NUM != getIntScan (&p, &cnt))
  {
    myPrintf ("Bad source, destination or count\n");
    return;
  }
  start += BASE;
  dest += BASE;
  if (start > dest)
    for (q = start; q < start + cnt;) *dest++ = *q++;
  else
    for (q = start + cnt, dest += cnt; q > start;)
      *--dest = *--q;
}


/* SEARCH command handler.  Search memory for a pattern.
 */
search (p)
char *p;
{
  char sbuf [LNLEN], *m, *n, *q, *first_fnd, *endp;
  long start, cnt;
  int len, found, i;

  if (GOT_NUM != getIntScan (&p, &start) ||
  GOT_NUM != getIntScan (&p, &cnt)) {
    myPrintf ("Bad start address or count\n");
    return;
  }
  found = len = 0;
  q = sbuf;
  while (q < sbuf + LNLEN && GOT_NUM == getIntScan (&p, &i)) {
    *q++ = i;
    ++len;
  }
  first_fnd = NULL;
  for (m = (char *)start + BASE, endp = (char *)start + BASE + cnt;
    m < endp; ++m)
  {
    if (*m == *sbuf) {
      for (n = m + 1, q = sbuf + 1; n - m < len && *n == *q; ++n, ++q);
      if (n - m == len) {
	if (NULL == first_fnd) first_fnd = m;
        ++found;
        morePrintf (1, "0x%lx\n", (long)(m - BASE));
      }
      if (found > 100) {
        morePrintf (1, "quitting...\n");
        break;
      }
    }
  }
  if (first_fnd != NULL) Dot = (unsigned char *)first_fnd - BASE;
}


/* Print a byte as two hex digits, 0-padded if necessary.
 */
printhexbyt (i)
int i;
{
  printhexnib ((i>>4) & 0xf);
  printhexnib (i & 0xf);
}

/* Print a nibble as a hex digit.
 */
printhexnib (i)
int i;
{
  if (i > 9)
    morePrintf (0, "%c", (char)('A' + i - 10));
  else
    morePrintf (0, "%c", (char)('0' + i));
}

#define ASCII_CH(x) (((x)>=0x20 && (x)<=0x7e)? (x): '.')
/* "=" command handler.  Kind of a calculator.
 */
baseConverter(p)
char *p;
{
  long base, val;

  if (getIntScan (&p, &val) != GOT_NUM) {
    myPrintf ("Bad value\n");
    return;
  }
  switch (getIntScan (&p, &base)) {
    case NO_NUM:
      base = defaultBase;
      break;
    case BAD_NUM:
      base = -1;		/* force error msg */
      break;
  }
  if (base < 2 || base > 16) {
    myPrintf ("Bad base\n");
    return;
  }
  printNumBase (val, (int)base);
  myPrintf (" \"%c%c%c%c\"\n",
    ASCII_CH((val>>24)&0xff),
    ASCII_CH((val>>16)&0xff),
    ASCII_CH((val>>8)&0xff),
    ASCII_CH((val>>0)&0xff));
}

/* TRACE command handler.  Trace back through stack frame chain,
 * printing return addresses.
 */
stackTrace (p)
char *p;
{
  long cnt, *fp;

  switch (getIntScan (&p, &cnt)) {
    case BAD_NUM:
      myPrintf ("Bad count\n");
      return;
    case NO_NUM:
      cnt = 1;
      break;
  }
  switch (getIntScan (&p, (long)(&fp))) {
    case BAD_NUM:
      myPrintf ("Bad fp value\n");
      return;
    case NO_NUM:
      fp = (long *)machState.fp;
      break;
  }
  while (cnt-- > 0) {
    myPrintf ("0x%lx\n", *(fp+1));
    fp = (long *)(*fp);
  }
}

#if STANDALONE
/* BAUD command handler. Change the baud rate.
 */
baud (p)
char *p;
{
  int rate, uart;

  switch (getIntScan (&p, &rate)) {
    case BAD_NUM:
    case NO_NUM:
      myPrintf ("Bad baud rate\n");
      return;
  }
  switch (getIntScan (&p, &uart)) {
    case BAD_NUM:
      myPrintf ("Bad uart number\n");
      return;
    case NO_NUM:
      uart = DEFAULT_UART;
      break;
  }
  db_setbaud (rate, uart);
}
#endif

/* RESTORE: simple restore from tape. CLR 7/2/91 */
/*  NOTE: I have used 0x40000 to 0x5FFFF as a buffer here.
    I tried to make it an automatic variable, but it barfed. */
restore(p)
char *p;
{
    char *buffer;
    long starting_block, blocks, count;
    long size;
    int ret;

    buffer = (char *)0x40000; /* put it up out of harm's way */
    ret = getIntScan(&p, &starting_block);
    if ((ret == NO_NUM) || (ret == BAD_NUM))
    {
	myPrintf ("Bad argument\n");
        return;
    }
    ret = getIntScan(&p, &blocks);
    if ((ret == NO_NUM) || (ret == BAD_NUM))
    {
	myPrintf ("Bad argument\n");
        return;
    }
    buffer = (char *)rambuffAdr; /* put it up out of harm's way */
    for (count = 0; count < blocks; count += rambuffSize)
    {
	size = blocks - count;
	if (size > rambuffSize) size = rambuffSize;
        ret = sc_tape_rdwt( 5, buffer, size, scsitapeAdr, scsitapeLun);
	if (ret)
        {
	    myPrintf("Bad return (%d) from sc_tape_rdwt\n", ret);
	    return;
	} 
	ret = sc_rdwt( 4, starting_block + count, buffer,
                           size, scsiAdr, scsiLun);
	if (ret)
        {
	    myPrintf("Bad return (%d) from sc_rdwt\n", ret);
	    return;
	} 
    }
}

/* BACKUP: simple backup to tape. CLR 7/2/91 */
/*  NOTE: I have used 0x40000 to 0x5FFFF as a buffer here.
    I tried to make it an automatic variable, but it barfed. */
backup(p)
char *p;
{
    char *buffer;
    long starting_block, blocks, count;
    long size;
    int ret;

    ret = getIntScan(&p, &starting_block);
    if ((ret == NO_NUM) || (ret == BAD_NUM))
    {
	myPrintf ("Bad argument\n");
        return;
    }
    ret = getIntScan(&p, &blocks);
    if ((ret == NO_NUM) || (ret == BAD_NUM))
    {
	myPrintf ("Bad argument\n");
        return;
    }
    buffer = (char *)rambuffAdr; /* Use the ram Big buffer. */
    for (count = 0; count < blocks; count += rambuffSize)
    {
    	size = blocks - count;
	if (size > rambuffSize) size = rambuffSize;
	ret = sc_rdwt( 3, starting_block + count, buffer,
                           size, scsiAdr, scsiLun);
	if (ret)
        {
	    myPrintf("Bad return (%d) from sc_rdwt\n", ret);
	    return;
	} 
        ret = sc_tape_rdwt( 6, buffer, size, scsitapeAdr, scsitapeLun);
	if (ret)
        {
	    myPrintf("Bad return (%d) from sc_tape_rdwt\n", ret);
	    return;
	} 
    }
}


#ifdef AUTOBOOT

#define MINIX
#define DKTYPENAMES
#include "disklabel.h"
#include "images.h"

/* Checksum a disk label */
dkcksum(lp)
        struct disklabel *lp;
{
        register unsigned short *start, *end, sum = 0;

        start = (unsigned short *)lp;
        end = (unsigned short*)&lp->d_partitions[lp->d_npartitions];
        while (start < end) sum ^= *start++;
        return sum;
}

/* Utility routine for boot and images. */
read_header (dk_label, im_table, scsi_adr, scsi_lun)
    struct disklabel **dk_label;
    struct imageinfo  **im_table;
    int scsi_adr, scsi_lun;
{
    char *disk_info;
    int ret;

    /* Read the header, Since booting supports only 512 or 1024 sector
       sizes, reading 2 blocks will guarantee to get at least 1024 bytes. */
    disk_info = (char *)rambuffAdr;     /* Use the big ram buffer. */
    ret = sc_rdwt( 3, 0, disk_info, 2, scsi_adr, scsi_lun);
    if (ret)
    {
	myPrintf ("Bad return (%d) from sc_rdwt\n", ret);
	return 0;
    }
    *dk_label = (struct disklabel *) &disk_info[LABELOFFSET];
    *im_table = (struct imageinfo *) (&disk_info[LABELOFFSET] 
    				     + sizeof(struct disklabel));
    if ((*dk_label)->d_magic != DISKMAGIC
	|| dkcksum (*dk_label) != 0)
    {
	myPrintf ("Disk has not have a valid disk label.\n");
	return 0;
    }

    if ((*im_table)->ii_magic != IMAGE_MAGIC
         || (*im_table)->ii_boot_partition == -1)
    {
	myPrintf ("Disk has not have boot information.\n");
	return 1;
    }
    return 2;
}


/************************************************************************
 *				boot					*
 ************************************************************************/
boot (p)
char *p;
{
    struct disklabel *dk_label;
    struct imageinfo *im_table;

    char image_name [80];
    char *name_ptr = image_name;
    int which_image; 	/* Image number. */
    int scsi_adr, scsi_lun; 
    int im_adr; 	/* Sector address of image. */
    int im_size;	/* Number of Sectors. */
    int im_ram;		/* Ram address of image. */
    int im_entry;	/* Entry address of image. */

    int ret;
    int index;

    /* Get parameters. */
    scanToken (&p, image_name);
    if (image_name [0] != 0) {
	which_image = -2;	/* Boot by name first. */
    } else
	which_image = -1;	/* No parameter, boot default. */

    if (!sc_get_adr_lun (&p, &scsi_adr, &scsi_lun, scsiAdr, scsiLun))
	return;

    /* Get the header. */
    if (read_header (&dk_label, &im_table, scsi_adr, scsi_lun) < 2)
	return;

    /* Do some checks on the data. */
    if (which_image == -1)
	which_image = im_table->ii_boot_default;
    if (which_image == -1)
    {
	myPrintf ("No default boot image.\n");
	return;
    }
    if (which_image == -2)
    {
	for (index=0; index < im_table->ii_boot_used && which_image == -2;
	     index++)
            if (myStrCmp(image_name,im_table->ii_images[index].boot_name)
	        == CMP_MATCH)
	        which_image = index;
    }
    if (which_image == -2)
    {
        ret = getIntScan(&name_ptr, &which_image);
        if (ret == BAD_NUM)
	{
	    myPrintf ("No image named %s.\n", image_name);
	    return;
	}
    }
    if (which_image < 0 || which_image >= im_table->ii_boot_used)
    {
	myPrintf ("Bad image number.\n");
	return;
    }

    /* Everything looks ready.  Do the boot! */
    im_adr = dk_label->d_partitions[im_table->ii_boot_partition].p_offset
	      + im_table->ii_images [which_image].boot_address 
    	      / dk_label->d_secsize;
    im_size = im_table->ii_images [which_image].boot_size
    	      / dk_label->d_secsize;
    im_ram  = im_table->ii_images [which_image].boot_load_adr;
    im_entry= im_table->ii_images [which_image].boot_run_adr;
    myPrintf ("Booting image %s.\n", 
    		im_table->ii_images[which_image].boot_name);

    if (debug) 
      myPrintf ("im_adr = %lx, im_size = %lx, im_ram = %lx, im_entry = %lx\n",
      		im_adr, im_size, im_ram, im_entry);

    ret = sc_rdwt( 3, im_adr, im_ram, im_size, scsi_adr, scsi_lun);
    if (ret)
    {
	myPrintf ("Bad return (%d) from sc_rdwt\n", ret);
	return;
    }

    /* Call the run routine! */
    machState.pc = im_entry;
    run ("");
}

/************************************************************************
 *				images					*
 ************************************************************************/
/* List all images on a disk. */
images (p)
char *p;
{
    char *disk_info;
    struct disklabel *dk_label;
    struct imageinfo *im_table;

    int index;
    int secsize;
    int scsi_adr, scsi_lun; 

    /* Get the scsi addresses */
    if (!sc_get_adr_lun (&p, &scsi_adr, &scsi_lun, scsiAdr, scsiLun))
	return;

    /* Read the header. */
    if (read_header (&dk_label, &im_table, scsi_adr, scsi_lun) < 2)
	return;

    /* Print other information about the disk. */
    secsize = dk_label->d_secsize;
    if (im_table->ii_boot_partition != -1)
      myPrintf ("Boot partition address (in sectors) = %x\n",
      	      dk_label->d_partitions[im_table->ii_boot_partition].p_offset);
    if (im_table->ii_boot_default != -1)
      myPrintf ("Default boot image  = %d\n", im_table->ii_boot_default);

    /* Print the images. */
    if (im_table->ii_boot_used > 0) {
	myPrintf ("Boot Images: total of %d\n",im_table->ii_boot_count);
	myPrintf ("  (image address and size in sectors.)\n");
	myPrintf ("Image  address   size  load addr  run addr   name\n");
	for (index = 0; index < im_table->ii_boot_used; index++) {
	    printf ("%5d %8lx %6lx  %9lx %9lx   %s\n", index,
		im_table->ii_images[index].boot_address/secsize,
		im_table->ii_images[index].boot_size/secsize,
		im_table->ii_images[index].boot_load_adr,
		im_table->ii_images[index].boot_run_adr,
		im_table->ii_images[index].boot_name );
	}
    } else
        myPrintf ("The disk has no boot images.\n");
    myPrintf ("\n");
}

/************************************************************************
 *				partitions				*
 ************************************************************************/
/* List all images on a disk. */
partitions (p)
char *p;
{
    char *disk_info;
    struct disklabel *dk_label;
    struct imageinfo *im_table;

    int index;
    int scsi_adr, scsi_lun; 

    /* Get the scsi addresses */
    if (!sc_get_adr_lun (&p, &scsi_adr, &scsi_lun, scsiAdr, scsiLun))
	return;

    /* Read the header. */
    if (read_header (&dk_label, &im_table, scsi_adr, scsi_lun) < 1)
	return;

    /* Disk Partitions. */
    myPrintf ("\nDisk: %s   Type: %s\n", dk_label->d_packname,
		dk_label->d_typename);
    myPrintf ("Physical Sector Size = %d\n", dk_label->d_secsize);
    myPrintf ("Disk Size = %ld\n", dk_label->d_secperunit);

    myPrintf (" Number            type  sector start  length in sectors\n");
    for (index = 0; index < dk_label->d_npartitions; index++) {
      if (dk_label->d_partitions[index].p_fstype != FS_UNUSED) {
        myPrintf (" %5d  ", index);
        myPrintf ("%14s", fstypenames[dk_label->d_partitions[index].p_fstype]);
        myPrintf ("%14ld  %17ld\n",
		dk_label->d_partitions[index].p_offset,
		dk_label->d_partitions[index].p_size);
      }
    }
    myPrintf ("\n");
}
#endif 
