/* Ranlib.c --  NS32000 random library builder.
 *
 * Random libraries are used by the linker.  They have a table of contents
 * at the beginning which lists the symbols defined by each file in the
 * library.  Using the table, the linker can link files from the library
 * in random order.  The files, including the table of contents, are all
 * files in an ar format archive.  This program is passed an archive,
 * containing only relocatable files, and builds a new archive, identical to
 * the original except for a ranlib inserted at the beginning.
 */
#include <stdio.h>
#include <fcntl.h>
#include "magic.h"
#ifdef MSDOS
#include "a_out.h"
#else
#include "a.out.h"
#endif
#include "ar.h"
#include "ranlib.h"
#include "conv.h"

#define SBUFSZ 0x400
#define AR_HEADERSZ (sizeof (struct ar_header))
#define EXECSZ (sizeof (struct exec))
#define COPYBUFSZ 0x1000

typedef struct strnode *strptr;
typedef struct filenode *fileptr;
typedef struct symnode *symptr;
typedef struct nlist *nlptr;

struct strnode {                /* string table is a queue of these */
  strptr link;
  long size;                    /* characters actually in sbuf */
  char sbuf [SBUFSZ];
};
#define STRNODESZ (sizeof (struct strnode))
  
struct filenode {               /* files recorded in a queue of these */
  fileptr link;
  long old_offset;              /* offset in old archive */
  int numsym;                   /* number of symbols */
  symptr sym;                   /* list of symbols defined in file */
};

struct symnode {                /* entry for list of symbols defined in file */
  symptr link;
  long stroff;                  /* offset in string table */
};

strptr strhead = NULL,          /* string table queue head and tail */
       strtail = NULL;

fileptr filehead = NULL,        /* file queue head and tail */
        filetail = NULL;

int debug = 0,                  /* true for ranlib debugging */
    infile;                     /* input file */

FILE *tmp,                      /* output file */
     *fopen();

long arhead_off =               /* offset of current ar_header in infile */
       sizeof (MAGIC_TYPE),
     rllen;                     /* length of table of contents file */

char usage [] = 
     "usage: ranlib [-d] archive\n",
     tmpname [100] = "",        /* name for temporary file */
     *arname,                   /* archive name */
     *myalloc(),
     *malloc (),
     *getseg(),
     copybuf [COPYBUFSZ];       /* for copying old archive to new */

struct rl_head rlhead           /* ranlib header for new table of contents */
  = {0, 0};

#define RL_LEN(x) (sizeof (struct rl_head) + x.rl_ftabsz + x.rl_stabsz);

main (argc, argv)
int argc;
char **argv;
{
  get_args (argc, argv);
  read_archive();
  write_archive();
  exit (0);
}

/* Get arguements, open the archive, check magic number.
 */
get_args (argc, argv)
int argc;
char **argv;
{
  MAGIC_TYPE magic;

  ++argv;
  while (argc >= 3 && **argv == '-' && *++*argv == 'd') {
    ++debug;
    ++argv;
    --argc;
  }
  if (argc != 2) error (usage);
  arname = *argv;
#ifdef MSDOS
  if (-1 == (infile = (open (arname, O_RDONLY | O_BINARY))))
#else
  if (-1 == (infile = (open (arname, O_RDONLY))))
#endif
    error ("could not open %s", arname);
  myread ((char *)&magic, sizeof (magic));
  CM2L (magic);
  if (magic != AR_MAGIC)
    error ("%s has bad magic number", arname);
}

/* Reads the relocatables from the archive.  Checks the magic numbers.
 * calls proc_reloc to build the data structure.  This routine alone is
 * responsible for reading the archive and keeping arhead_off in sync.
 */
read_archive()
{
  struct ar_header h;
  struct exec e;
  char *symtab, *strtab;

  for (;;) {
    lseek (infile, arhead_off, 0);
    if (AR_HEADERSZ != read (infile, &h, AR_HEADERSZ)) break;
    if (debug) printf ("ranlib: getting %s\n", h.ar_name);
    myread ((char *)&e, EXECSZ);
    CM2L (e.a_magic);
    CM2L (e.a_text);
    CM2L (e.a_data);
    CM2L (e.a_trsize);
    CM2L (e.a_drsize);
    CM2L (e.a_sym);
    CM2L (e.a_str);
    if (e.a_magic != RELOC_MAGIC)
      error ("archive member %s is not a relocatable file", h.ar_name);
    symtab = getseg (e.a_sym, (long)(arhead_off + AR_HEADERSZ + SYMPOS (e)));
    strtab = getseg (e.a_str, (long)(arhead_off + AR_HEADERSZ + STRPOS (e)));
    proc_reloc ((struct nlist *)symtab, (long)(e.a_sym / sizeof (struct nlist)),
      strtab, (long)(arhead_off + AR_HEADERSZ));
    free (strtab);
    free (symtab);
    arhead_off += AR_HEADERSZ + WM2L (h.ar_len);
  }
  rllen = RL_LEN (rlhead);
}

/* Passed pertinent information from a relocatable, this routine builds
 * the data structure.
 */
proc_reloc (symtab, numsym, strtab, old_offset)
struct nlist *symtab;
long numsym;
char *strtab;
long old_offset;
{
  nlptr np;
  fileptr fp;
  symptr sp;

  rlhead.rl_ftabsz += sizeof (struct rl_file) - sizeof (long);
  fp = (fileptr) myalloc (sizeof (struct filenode));
  fp->link = NULL;
  fp->numsym = 0;
  fp->sym = NULL;
  fp->old_offset = old_offset;
  if (filehead == NULL) filehead = fp;
  else filetail->link = fp;           /* insert into file queue */
  filetail = fp;
  for (np = symtab; np < symtab + numsym; ++np)
    if (T_UNDF != WM2S (np->n_type)) {
      rlhead.rl_ftabsz += sizeof (long);
      ++fp->numsym; 
      sp = (symptr) myalloc (sizeof (struct symnode));
      sp->link = fp->sym;             /* insert in sym linked list */
      fp->sym = sp;
      insertstr (strtab + WM2L (np->n_stroff), &sp->stroff);
      if (debug) printf ("  found %s\n", strtab + WM2L (np->n_stroff));
    }
}

/* Write the new archive and link it to the old one.
 */
write_archive()
{
  create_tmp();
  write_arhead();
  write_ranlib();
  copy_file();
  fclose (tmp);
  close (infile);
  unlink (arname);
#ifndef MSDOS
  link (tmpname, arname);
  unlink (tmpname);
#else
  close (tmp);
  rename (tmpname, arname);
#endif
}

/* Create a temporary file for writing.  Put its name in
 * tmpname.  Write magic number.
 */
create_tmp()
{
  MAGIC_TYPE magic;

  sprintf (tmpname, "ar.%d", getpid());
#ifdef MSDOS
  if (NULL == (tmp = fopen (tmpname, "wb")))
#else
  if (NULL == (tmp = fopen (tmpname, "w")))
#endif
    error ("could not open temporary file");
  WL2M (magic, AR_MAGIC);
  mywrite ((char *)&magic, sizeof magic);
}

/* Write an ar_header for the table of contents.
 */
write_arhead()
{
  struct ar_header h;
  long tlong;

  WL2M (h.ar_len, rllen);
  strcpy (h.ar_name, RL_NAME);
  time (&tlong);
  WL2M (h.ar_mtime, tlong);
  mywrite (&h, sizeof (struct ar_header));
}

/* Write the ranlib file to the archive.
 */
write_ranlib()
{
  long l;

  CL2M (rlhead.rl_ftabsz, l);          /* write the header */
  CL2M (rlhead.rl_stabsz, l);  
  mywrite (&rlhead, sizeof (rlhead));
  write_ftab();                        /* write the tables */
  write_stab();
}

/* Write the file table for the ranlib.
 */
write_ftab()
{
  fileptr fp;
  symptr sp;
  long l;
  struct rl_file rl_file;

  for (fp = filehead; fp != NULL; fp = fp->link) {
    WL2M (rl_file.rl_offset,
      fp->old_offset + sizeof (struct ar_header) + rllen);
    WS2M (rl_file.rl_numsym, fp->numsym);
    mywrite (&rl_file, sizeof (struct rl_file) - sizeof (long));
    for (sp = fp->sym; sp != NULL; sp = sp->link) {
      WL2M (l, sp->stroff);
      mywrite ((char *)(&l), sizeof l);
    }
  }
}

/* Write the string table to tmp.
 */
write_stab()
{
  strptr sp;

  for (sp = strhead; sp != NULL; sp = sp->link)
    mywrite (sp->sbuf, (int)(sp->size));
}

/* Copy the old archive, without magic number, to tmp.
 */
copy_file()
{
  int len;

  mylseek ((long)(sizeof (MAGIC_TYPE)));
  while (len = read (infile, copybuf, COPYBUFSZ))
    mywrite (copybuf, len);
}

/* Copy a string into the string table, which is a queue of struct strnode.
 * Return a pointer to copy (id1) and offset in string table (stroff).
 */
insertstr (id, stroff)
long *stroff;
char *id;
{
  int len;
  struct strnode *str;

  if (SBUFSZ < (len = strlen (id) + 1)) 
    error ("!#identifier too large");
  if (strtail == NULL                   /* make room in strbuf */
  || strtail->size + len > SBUFSZ) {
    str = (struct strnode *) myalloc (STRNODESZ);
    if (strtail == NULL) strhead = strtail = str;
    else {
      strtail->link = str;
      strtail = str;
    }
    str->size = 0;
    str->link = NULL;
  }
  strcpy (strtail->sbuf + strtail->size, id);   /* copy string */
  strtail->size += len;
  *stroff = rlhead.rl_stabsz;
  rlhead.rl_stabsz += len;
}

/* Malloc, lseek, read combination.  Returns pointer to allocated buffer.
 * Reads infile. Dies on error.
 */
char *
getseg (len, seek)
long len;			/* changed from int 1/10/89 */
long seek;
{
  char *p;

  p = myalloc ((unsigned)len);	/* casts added 1/10/89 */
  mylseek (seek);
  myread (p, (unsigned)len);
  return p;
}

/* Allocate memory, die if fails.  This may be in trouble if MSDOS
 * where int may not be big enough.
 */
char *
myalloc (size)
unsigned size;
{
  char *p;

  if (NULL == (p = malloc (size))) error ("out of memory");
  return p;
}

/* Read from a file into a buffer, die on error.
 */
myread (ptr, size)
char *ptr;
unsigned size;			/* 1/10/89 */
{
  if (size != read (infile, ptr, size)) error ("read failed");
}

/* Seek infile, die if error.
 */
mylseek (offset)
long offset;
{
  if (-1 == lseek (infile, offset, 0)) {
    if (debug) printf ("  seeking to 0x%lx\n", offset);
    error ("lseek failed");
  }
}

/* Write to tmp.  Print error message and die if unsuccessful.
 */
mywrite (buf, size)
char *buf;
int size;
{
  if (size != fwrite (buf, 1, size, tmp)) error ("write failed");
}

/* Print error message and die.
 */
error (fmt, arg1, arg2, arg3, arg4)
int arg1, arg2, arg3, arg4;
char *fmt;
{
  fputs ("ranlib: ", stderr);
  fprintf (stderr, fmt, arg1, arg2, arg3, arg4);
  fputs ("\nranlib: (warning) library not created\n", stderr);
  unlink (tmpname);
  exit (-1);
}
