/* Ld1.c
 * NS32000 linker
 */
#include "ld.h"

struct exec head = {                  /* header for output file */
   0, 0, 0, 0,
   0, 0, 0, 0,
   0, 0, 0, 0};

hashptr hash [HASHLEN];               /* the hash table */

struct strnode *strhead = NULL,       /* string table queue head and tail */
               *strtail = NULL;

fileptr filehead = NULL,              /* file queue head and tail */
        filetail = NULL;

struct nlist
     seed,			      /* entry label, pulls in runtime code */
     btext_,			      /* text beginning */
     etext_,			      /* text end */
     bdata_,			      /* data beginning */
     edata_,			      /* data end */
     end_;			      /* bss end */
	
int  num_errors = 0,                  /* error count */
     num_undef = 0,                   /* number of undefined symbols */
     sym_index = 0,                   /* used to calc indices in new sym tab */
     debug = 0,                       /* true for linker debugging */
     reloc;                           /* true if Ok to build relocatable */

char usage [] = 
  "!usage: ld [-r] [-e <entry id>] [-{T|D} <offset>] [-o <obj file>] files...",
     *curfile,                        /* current file name */
     *curstrtab = NULL,               /* string table from current file */
     outname [100] = DEFAULT_OUT,     /* output file name */
     entry_id [100] = ENTRY_ID;       /* entry identifier */

FILE *infile;                         /* currently open input file */

unsigned long
     tstart_ = DEFAULT,               /* first address of text segment */
     dstart_ = DEFAULT,               /* first address of data segment */
     bstart_;

main (argc, argv)
int argc;
char **argv;
{
  hashinit();
  pass1 (argc, argv);
  between_passes();
  pass2 ();
  exit (0);
}

/* Pass1.  Processes options.  Creates linked list of files.
 * Creates symbol table.
 */
pass1 (argc, argv)
int argc;
char **argv;
{
  char *p;
  int linking;			/* check options which must precede files */

  linking = FALSE;			/* no files yet */
  if (argc == 1) error (usage);
  ++argv;
  while (--argc) {
    if (**argv == '-') {
      switch (*++*argv) {
        case 'o':			/* output file name */
          strncpy (outname, *++argv, 99);
          --argc;
          break;
        case 'e':			/* entry label */
          if (linking) error (usage);	/* must preceed files */
          strncpy (entry_id, *++argv, 99);
          --argc;
          break;
        case 'T':			/* text start address */
	  if (reloc) error ("-T and -D may not be used with -r");
          if (linking) error (usage);
          sscanf (*++argv, "%lx", &tstart_);
          --argc;
          break;
        case 'D':			/* data start address */
	  if (reloc) error ("-T and -D may not be used with -r");
          if (linking) error (usage);
          sscanf (*++argv, "%lx", &dstart_);
          --argc;
          break;
        case 'd':			/* verbose diagnostics */
          ++debug;
          break;
        case 'r':			/* output relocatable file */
	  if (tstart_ != DEFAULT || dstart_ != DEFAULT)
            error ("-T and -D may not be used with -r");
          if (linking) error (usage);
          ++reloc;
          break;
        case 'l':			/* link from lib directory */
          if (!linking) {
            linking = TRUE;		/* have files */
            if (!reloc) proc_special_id (T_UNDF, entry_id, 0L, &seed);
          }
          p = myalloc (50);
          strcpy (p, LIBDIR);
          strncat (p, ++*argv, 15);
          strncat (p, ".a", 3);
          procfile1 (p);
          break;
        default:
          error (usage);
      }
    } else {
      if (!linking) {
        linking = TRUE;			/* have files */
        if (!reloc) proc_special_id (T_UNDF, entry_id, 0L, &seed);
      }
      procfile1 (*argv);
    }
    ++argv;
  }
}

/* Process a file specified in arg list during pass 1.
 */
procfile1 (filename)
char *filename;
{
  fileptr fp;

  myfopen (filename);
  fp = (fileptr) myalloc (FILENODESZ);
  fp->name = filename;
  myfread (&(fp->h.e), max (sizeof (MAGIC_TYPE) + sizeof (struct ar_header),
    sizeof (struct exec)));
  CM2L (fp->h.e.a_magic);
  if (fp->h.e.a_magic == RELOC_MAGIC) proc_reloc1 (fp, 0L);
  else if (fp->h.e.a_magic == AR_MAGIC) proc_ar1 (fp);
  else error ("!#not relocatable or library");
  fclose (infile);	/*BUG FIX*/
}

/* Just before processing the first file put the entry symbol (the seed)
 * into the symbol table as T_UNDF.  The usual use for this is to cause
 * the C runtime routine to be loaded from libc.a.
 */
proc_special_id (type, id, val, nlp)
int type;
char* id;
long val;
nlptr nlp;
{
  hashptr hp;

  nlp->n_type = type;
  if (NULL == (hp = find (id))) {
    nlp->n_stroff = 0;                       /* fool insert */
    curstrtab = id;
    hp = insert (nlp);
    if (type == T_UNDF) {
      ++num_undef;
      nlp->n_value = (long) hp;
    } else nlp->n_value = val;
  } else {
    nlp->n_stroff = hp->nlp->n_stroff;
    if (type == T_UNDF) nlp->n_value = (long)hp;
    else {
      nlp->n_value = val;
      if (hp->nlp->n_type != T_UNDF)
	error ("#%s multiply defined", hp->id);
      else {
	--num_undef;
	hp->nlp = nlp;
      }
    }
  }
}

/* Do pass1 processing on a relocatable file.  Offset is used for files
 * which are members of archives.  Puts file node on queue.  Enters symbols
 * into symbol table.  Accumulates sizes of new segments.
 */
proc_reloc1 (fp, offset)
long offset;
fileptr fp;
{
  nlptr p;

  CM2L (fp->h.e.a_text);
  CM2L (fp->h.e.a_data);
  CM2L (fp->h.e.a_bss);
  CM2L (fp->h.e.a_trsize);
  CM2L (fp->h.e.a_drsize);
  CM2L (fp->h.e.a_sym);
  CM2L (fp->h.e.a_str);
  fp->offset = offset;                /* fill in structure */
  fp->link = NULL;
  if (filehead == NULL) filehead = fp;
  else filetail->link = fp;           /* insert into file queue */
  filetail = fp;
  fp->nlp = (nlptr) getseg            /* get symbol table */
    ((int)fp->h.e.a_sym, (long)(offset + SYMPOS(fp->h.e)));
  curstrtab =                         /* get string table */
    getseg ((int)fp->h.e.a_str, (long)(offset + STRPOS (fp->h.e)));
  for (p = fp->nlp;                   /* insert symbols into hash table */
  p < fp->nlp + fp->h.e.a_sym/sizeof (struct nlist); ++p)
    proc_nlist1 (p);
  fp->new_text = head.a_text;         /* segment offsets in new segments */
  fp->new_data = head.a_data;
  fp->new_bss = head.a_bss;
  head.a_text += fp->h.e.a_text;      /* update segment sizes */
  head.a_data += fp->h.e.a_data;
  head.a_bss += fp->h.e.a_bss;
  head.a_trsize += fp->h.e.a_trsize;
  head.a_drsize += fp->h.e.a_drsize;
}

/* Process nlist structure.  If not in hash table, insert and point hash
 * entry at this nlist.  Else, point hash entry to this nlist only if this
 * nlist is not T_UNDF.  If T_UNDF, use n_value to point to hash entry.
 * Offset in new string table is stored in hashptr->nlp->n_stroff.
 */
proc_nlist1 (p)
nlptr p;
{
  hashptr hp;

  CM2L (p->n_value);
  CM2L (p->n_stroff);
  CM2S (p->n_type);
  if (NULL == (hp = find (curstrtab + p->n_stroff))) {
    hp = insert (p);
    if (p->n_type == T_UNDF) ++num_undef;
  } else if (p->n_type != T_UNDF) {
    if (hp->nlp->n_type != T_UNDF)
      error ("#%s multiply defined", hp->id);
    else --num_undef;
    p->n_stroff = hp->nlp->n_stroff;
    hp->nlp = p;
  }
  /* Here is the actual relocation of labels.  Initially, value is offset 
   * from beginning of old module.  Temporarily, make it offset in current
   * segment.  Later, when we know how long the new segments are, add
   * new segment lengths back in (see fixup_syms) to get offset in new module.
   */
  switch (p->n_type) {
    case T_UNDF:
      p->n_value = (long)hp;     /* kludge, pointer to hash node */
      break;
    case T_TEXT:
      p->n_value += head.a_text;
      break;
    case T_DATA:
      p->n_value += head.a_data - filetail->h.e.a_text;
      break;
    case T_BSS:
      p-> n_value += 
        head.a_bss - filetail->h.e.a_text - filetail->h.e.a_data;
      break;
  }
}

/* Process a ranlib'ed archive file.  This means: go through its table
 * of contents repeatedly.  If the archive contains a file which defines
 * a currently undefined symbol, link it in.  Quit when a search of the
 * ranlib finds no more undefined symbols.
 */
proc_ar1 (fp)
fileptr fp;
{
  int found;
  char *strtab;
  struct rl_head *hp;
  struct rl_file *rfp;

  if (strcmp (fp->h.ar.ar.ar_name, RL_NAME) != 0)
    error ("!#need to run ranlib");
  CM2L (fp->h.ar.ar.ar_len);
  hp = (struct rl_head *) getseg ((int)fp->h.ar.ar.ar_len, /* read __.SYMDEF */
    (long) sizeof (MAGIC_TYPE) + sizeof (struct ar_header));
  strtab = (char *)hp + sizeof (struct rl_head) + WM2L (hp->rl_ftabsz);
  do {
    found = 0;
    rfp = (struct rl_file *)((char *)hp + sizeof (struct rl_head));
    while (num_undef && rfp < (struct rl_file *) strtab)
      rfp = proc_rl_file (rfp, strtab, &found, fp->name);
  } while (found && num_undef);
}

/* Process an rl_file structure from a ranlib table of contents.
 * This means: for each symbol the file defines, see if it is currently
 * T_UNDF.  If so, link it in and return found = TRUE.
 */
struct rl_file *
proc_rl_file (rfp, strtab, found, name)
struct rl_file *rfp;
char *strtab, *name;
int *found;
{
  long *stroff, *endp;
  hashptr hp;
  fileptr fp;

/*  if (debug > 2) printf ("offset=0x%x numsyn=0x%x: ",
    WM2L(rfp->rl_offset), WM2S (rfp->rl_numsym)); */
  endp = rfp->rl_sym + WM2S (rfp->rl_numsym);
  if (rfp->rl_offset)  /* already linked it? */
    for (stroff = rfp->rl_sym; stroff < endp; ++stroff) {
      if (debug > 2) printf ("%s ", strtab + WM2L(*stroff));
      hp = find (strtab + WM2L(*stroff));
      if (hp != NULL) {
        if (hp->nlp->n_type == T_UNDF) {
          /*if (debug > 1) printf ("linking %s(0x%lx)\n", curfile,
            WM2L(rfp->rl_offset));*/
          *found = 1;
          myfseek ((long) WM2L (rfp->rl_offset));
          fp = (fileptr) myalloc (FILENODESZ);
          fp->name = name;
          myfread ((char *)&fp->h.e, sizeof (struct exec));
          CM2L (fp->h.e.a_magic);
          if (fp->h.e.a_magic != RELOC_MAGIC) 
            error ("!#archive has nonrelocatable member");
          proc_reloc1 (fp, (long)(WM2L (rfp->rl_offset)));
        }
        rfp->rl_offset = 0;       /* prevent further linking */
        break;
      }
    }
/*  if (debug > 2) printf ("\n");*/
  return (struct rl_file *)endp;
}

/* Check if pass1 was sufficiently successful to continue to pass2.
 * Do other odds and ends before pass2.
 */
between_passes()
{
  long old_a_text;

  if (num_errors) quit();
  /* the order of the following items is important! */
  if (!reloc) {
    old_a_text = head.a_text;		/* save actual text length */
    head.a_text =                       /* round text to even page */
      ((head.a_text + PAGESZ - 1) / PAGESZ) * PAGESZ;
    head.a_trsize = head.a_drsize = 0;  /* strip relocation info */
    head.a_magic = EXEC_MAGIC;
  } else head.a_magic = RELOC_MAGIC;
  if (tstart_ == DEFAULT)		/* set segment start adr's */
    tstart_ = DEFAULT_TSTART;
  if (dstart_ == DEFAULT)
    dstart_ = tstart_ + head.a_text;
  bstart_ = dstart_ + head.a_data;	/* bss at end of data segment */
  if (!reloc) {				/* create special labels */
    proc_special_id (T_TEXT, BTEXT_ID, 0L, &btext_);
    proc_special_id (T_TEXT, ETEXT_ID, old_a_text, &etext_);
    proc_special_id (T_DATA, BDATA_ID, 0L, &bdata_);
    proc_special_id (T_DATA, EDATA_ID, head.a_data, &edata_);
    proc_special_id (T_BSS, END_ID, head.a_bss, &end_);
  }
  if (num_undef && !reloc) {
    error ("undefined external -");
    hashnav (printundef);
    quit();
  }
  hashnav (fixup_syms);                 /* set indices, fix n_value's */
  fix_new_segs();                       /* fix filenode.new_seg values */
  head.a_sym = sym_index * sizeof (struct nlist);
  if (!reloc) {                         /* get entry location */
    head.a_entry = ((hashptr)seed.n_value)->nlp->n_value;
    head.a_tstart = tstart_;
    head.a_dstart = dstart_;
  }
}

/* When called, symbol values are offsets in their own segments.  This
 * routine makes them offsets from beginning of text segment or from user
 * supplied values.  Also sets index field in hash entries to index symbol
 * will have in new symbol table.
 */
fixup_syms(p)
hashptr p;
{
  switch (p->nlp->n_type) {
    case T_TEXT:
      p->nlp->n_value += tstart_;
      break;
    case T_DATA:
      p->nlp->n_value += dstart_;
      break;
    case T_BSS:
      p->nlp->n_value += bstart_;
      break;
    default:
      break;
  }
  p->index = sym_index++;
}

/* The new_xxx fields in the filenodes are the positions of the old segments
 * in the new segments.  This changes them so that they are the positions
 * of the old segments in the new module, i.e. their position relative to
 * the first byte in the new text module if the user has not supplied
 * special segment addresses.
 */
fix_new_segs()
{
  fileptr fp;

  for (fp = filehead; fp != NULL; fp = fp->link) {
    fp->new_text += tstart_;   /* bug fix 5/16/88 */
    fp->new_data += dstart_;
    fp->new_bss += bstart_;
  }
}

/* Passed a hashnode, print the symbol if it is undefined.
 */
printundef (p)
hashptr p;
{
  if (p->nlp->n_type == T_UNDF)
    fprintf (stderr, "\t%s\n", p->id);
}

/* Hash function.  Given a string, returns a number between 0 and
 * HASHLEN - 1.
 */
int
hashfn (id)
char *id;
{
  U16 ret;

  for (ret = 0; *id; id++)
    ret = (ret<<1 | ret>>15) ^ *id;
  return ret & (HASHLEN - 1);
}

/* Hash table lookup.  Returns a pointer
 * to the entry if found, NULL otherwise.
 */
hashptr
find (id)
char *id;
{
  hashptr p;

  for (p = hash [hashfn (id)]; p != NULL; p = p->link)
    if (!strcmp (id, p->id)) break;
  return p;
}

/* Inserts an identifier into the hash table and returns a pointer to 
 * the hash table entry.  ID is copied into a strnode buffer so that
 * only one copy of each ID is needed.
 */
hashptr
insert (nlp)
nlptr nlp;
{
  int i;
  hashptr hp;

  hp = (hashptr) myalloc (HASHNODESZ);
  hp->nlp = nlp;
  insertstr (curstrtab + nlp->n_stroff, &hp->id, &nlp->n_stroff);
  i = hashfn (hp->id);
  hp->link = hash[i];
  return hash[i] = hp;
}

/* Initialize the hash table.
 */
hashinit ()
{
  hashptr *p;

  for (p = hash; p < hash + HASHLEN;) *p++ = NULL;
}

/* Navigate the hash table, passing each to the function f.
 */
hashnav (f)
int (*f)();
{
  hashptr p, *q;

  for (q = hash; q < hash + HASHLEN; q++)
    for (p = *q; p != NULL; p = p->link) (*f)(p);
}

/* Copy a string into the string table, which is a queue of struct strnode.
 * Return a pointer to copy (id1) and offset in string table (stroff).
 */
insertstr (id, id1, stroff)
long *stroff;
char *id, **id1;
{
  int len;
  struct strnode *str;

  if (SBUFSZ < (len = strlen (id) + 1)) 
    error ("!#identifier too large");
  if (strtail == NULL                   /* make room in strbuf */
  || strtail->size + len > SBUFSZ) {
    str = (struct strnode *) myalloc (STRNODESZ);
    if (strtail == NULL) strhead = strtail = str;
    else {
      strtail->link = str;
      strtail = str;
    }
    str->size = 0;
    str->link = NULL;
  }
  *id1 = strtail->sbuf + strtail->size; /* pointer into string table */
  strcpy (*id1, id);                    /* copy string */
  strtail->size += len;
  *stroff = head.a_str;
  head.a_str += len;
}

/* Malloc, lseek, read combination.  Returns pointer to allocated buffer.
 * Reads infile.
 */
char *
getseg (len, seek)
int len;
long seek;
{
  char *p;

  p = myalloc ((unsigned)len);
  myfseek (seek);
  myfread (p, len);
  return p;
}

/* Allocate memory, die if fails.  This may be in trouble if MSDOS
 * where int may not be big enough.
 */
char *
myalloc (size)
unsigned size;
{
  char *p;

  if (NULL == (p = malloc (size))) error ("!#out of memory");
  return p;
}

/* Open a file, check for errors.  Sets curfile for error reporting.
 */
myfopen (name)
char *name;
{
  curfile = name;
#ifdef MSDOS
  if (NULL == (infile = fopen (name, "rb")))
#else
  if (NULL == (infile = fopen (name, "r")))
#endif
    error ("#!open failed");
}

/* Read from a file into a buffer, die on error.
 */
myfread (ptr, size)
char *ptr;
int size;
{
    if (size != fread (ptr, 1, size, infile)) error ("!#read failed");
}

/* Seek infile, die if error.
 */
myfseek (offset)
long offset;
{
  if (-1 == fseek (infile, offset, 0)) error ("!#fseek failed");
}

/* Print error message.  Message may have # or ! in the first two positions:
 * # means print file name, ! means die.
 */
/*VARARGS1*/
error (fmt, arg1, arg2, arg3, arg4)
int arg1, arg2, arg3, arg4;
char *fmt;
{
  int name, die;

  if (*fmt == '#' || *(fmt+1) == '#') name = 1;
  else name = 0;
  if (*fmt == '!' || *(fmt+1) == '!') die = 1;
  else die = 0;
  fprintf (stderr, name? "ld: (%s) ": "ld: ", curfile);
  fprintf (stderr, fmt + name + die, arg1, arg2, arg3, arg4);
  fputs ("\n", stderr);
  if (die) quit();
  ++num_errors;
}

quit()
{
  fprintf (stderr, "ld: (warning) no output file written\n");
  unlink (outname);
  exit (-1);
}
