/* NS32000 Assembler
 * Pseudo.c
 * Code to process the various pseudo operators.
 */
#include <stdio.h>
#ifdef MSDOS
#  include "a_out.h"
#else
#  include "a.out.h"
#endif
#include "conv.h"
#include "glob.h"

/* Called after finding label: or label::.  This routines inserts
 * the label with its value into the hash table.
 */
proc_lbl(lp)
register lblptr lp;
{
  if ((lp->typ&T_LBL) != T_UNDF || lp->exp != NULL) {
    error ("#symbol multiply defined: %s", lp->id);
    return;
  }
  lp->lnnum = lnnum;                   /* for error reporting */
  lp->typ = curseg | lp->typ & ~T_LBL;
  lp->val = *curlocptr;
  if (curseg == T_TEXT) {
    if (tlbl_head == NULL)             /* insert in text label queue */
      tlbl_head = lp;
    else tlbl_tail->llink = lp;
    tlbl_tail = lp;
    lp->llink = NULL;
  } else if (curseg == T_DATA) {
    if (dlbl_head == NULL)             /* insert in data label queue */
      dlbl_head = lp;
    else dlbl_tail->llink = lp;
    dlbl_tail = lp;
    lp->llink = NULL;
  } else {
    if (blbl_head == NULL)             /* insert in bss label queue */
      blbl_head = lp;
    else blbl_tail->llink = lp;
    blbl_tail = lp;
    lp->llink = NULL;
  }
  if (debug && lp->id != NULL) printf ("PROC_LBL lbl=%s val=0x%lx\n",
    lp->id, lp->val);
}

/* Process .EQU directive.
 */
proc_equ ()
{
  exptyp exp;
  int len;

  if (curlbl == NULL) {
    error ("#.EQU without label");
    scannl();
    return;
  }
  curlbl->typ |= T_STATIC;             /* .equ labels are never global */
  curlbl->lnnum = lnnum;
  curlbl->exp = exp;
  compile_exp (exp, &len);             /* compile expression into token form */
  eval_lbl (curlbl);                   /* evaluate expression */
  if (!(curlbl->typ & T_RESOLVED))     /* if not resolved save expression */
    copy_exp (&curlbl->exp, len);
  if (debug) printf ("PROC_EQU lbl=%s len=%d\n", curlbl->id, len);
}

/* Handler for .EXPORT.  This was added as an after thought and, consequently,
 * is an unfortunate kludge.
 */
export ()
{
  lblptr lp;

  for (;;) {
    if (curtok != tLBL) {
      error ("#expected label in .EXPORT directive");
      return;
    }
    if (curhash != NULL) {
      ((lblptr)curhash)->typ &=        /* makes label global */
        ~T_STATIC;
      ((lblptr)curhash)->typ |=        /* keep T_STATIC from being set later */
        T_EXPORTDIR;
    } else {                           /* label not yet defined */
      lp = insert_lbl (curstr, T_UNDF | T_EXPORTDIR);
      lp->exp = NULL;
    }
    scan();
    if (curtok != ',') break;
    scan();
  }
}

/* Handler for .PROGRAM, .STATIC, and .BSS pseudo ops.
 */
setseg (p)
opptr p;
{
  switch (curseg = p->inst) {
    case T_TEXT:
      curlocptr = &header.a_text;
      break;
    case T_DATA:
      curlocptr = &header.a_data;
      break;
    case T_BSS:
      curlocptr = &header.a_bss;
      break;
  }
}

/* Handler for .ALIGN
 */
align()
{
  register U32 m;
  register int inc;
  patchptr pch;

  m = get_imm();
  if (m > SEGALIGN) {
    error ("#.ALIGN parameter is too large");
    return;
  }
  if (m <= 1) return;
  if (curseg == T_TEXT) {
    /* because of displacement minimization which occurs later in text seg,
     * must defer alignment until later. */
    pch = (patchptr) myalloc (sizeof (struct patch));
    pch->flags = pALIGN;
    pch->val = m;
    pch->size = 4;
    pch->pcoff = 0;
    pch->typ = 0;
    pch->exp = NULL;
    insert_patch (pch, 0);
    update_lnmap (4);
  } else {				/* data or bss */
    if (!(inc = *curlocptr % m)) return;
    inc = m - inc;
    if (curseg == T_BSS) header.a_bss += inc;
    else {
      update_lnmap (inc);
      while (inc--) myputc (0, tmpd);
    }
  }
}

/* Declare constant, handler for .BYTE, .WORD, and .DOUBLE.  p->inst is size.
 */
dc (p)
opptr p;
{
  U32 val;

  if (curseg == T_BSS) {
    error ("#%s not valid in BSS segment", p->id);
    return;
  }
  if (curtok == '\n') {			/* default: one unit of 0 */
    val = 0;
    write_tmp ((char *)&val, (int)p->inst);
    return;
  }
  for (;;) {
    if (p->inst == 1 &&			/* if .BYTE, might be string */
    curtok == tSTR && toklen > 1) {
      write_tmp (curstr, toklen);
      scan();
    } else wr_exp_imm ((int)p->inst);
    if (curtok == ',') scan();
    else if (curtok != tSTR) break;
  }
}

/* Declare floating point constant, handler for .FLOAT, .LONG.
 */
df (p)
opptr p;
{
  float f;

  if (curseg == T_BSS) {
    error ("#%s not valid in BSS segment", p->id);
    return;
  }
  if (curtok == '\n') {			/* default: one unit of 0 */
    curfloat = 0;
    write_tmp ((char *)&curfloat, (int)p->inst);
    return;
  }
  for (;;) {
    if (curtok != tFLOAT) {
      error ("#expected floating point constant");
      break;
    }
    if ((int)p->inst == 4) {
      f = curfloat;
#     ifdef BIG_ENDIAN
	{
          union {
	    unsigned long l;
	    float flt;
	  } F;
	  unsigned long t;

	  F.flt = f;
	  CL2M (F.l, t);
	  f = F.flt;
	}
#     endif
      write_tmp ((char *)&f, 4);
    } else {
#     ifdef BIG_ENDIAN
	union {
	  struct { unsigned long l, h} ints;
	  double flt;
	} d;
	unsigned long t;

	d.flt = curfloat;
	t = d.ints.l;
	d.ints.l = d.ints.h;
	d.ints.h = t;
	CL2M (d.ints.h, t);
	CL2M (d.ints.l, t);
	curfloat = d.flt;
#     endif
      write_tmp ((char *)&curfloat, 8);
    }
    scan();
    if (curtok != ',') break;
    scan();
  }
}

/* For GCC strings.  Parser must turn on "gcc" before this routine is
 * called so that first string is scanned with gcc syntax.
 */
ascii (p)
opptr p;
{
  U32 val;

  if (curseg == T_BSS) error ("#%s not valid in BSS segment", p->id);
  else while (curtok == tSTR) {
    write_tmp (curstr, toklen);
    scan();
  }
  gcc = FALSE;				/* turn off gcc string syntax */
}

/* Declare space, handler for .BLKB, .BLKW, .BLKD.  p->inst is size.
 */
ds (p)
opptr p;
{
  U32 inc;

  inc = get_imm() * p->inst;
  if (curseg == T_BSS) header.a_bss += inc;
  else {
    update_lnmap ((int)inc);
    while (inc--) myputc (0, curseg == T_TEXT? tmpt: tmpd);
  }
}
  
/* Called when an expression is next in input stream.  Compiles and evaluates 
 * expression.  If expression is resolved, it is written in immediate form
 * (msbyte last) to the current segment; otherwise sz zero bytes are written.
 * If the expression is unresolved, or resolved but needs relocating, a
 * patch record is created and its address returned.
 */
patchptr
wr_exp_imm (sz)
int sz;
{
  exptyp exp;
  int len;
  U32 val, tmp;
  U16 typ;
  lblptr undf;
  register patchptr p;
  
  compile_exp (exp, &len);
  eval_exp (exp, &typ, &val, &undf);
  if (!(typ & T_RESOLVED) ||           /* need patch record? */
  (typ & T_LBL) != T_IMM) {            /* create patch, link into list */
    p = (patchptr) myalloc (sizeof (struct patch));
    p->size = sz;
    p->flags = pIMM;
    p->pcoff = 0;
    if (typ & T_RESOLVED) {
      p->exp = NULL;
      p->flags |= pWRITTEN;
    }
    else p->exp = exp;
    p->val = val;
    p->typ = typ;
    p->undf = undf;
    insert_patch (p, len);
  }
  if (typ & T_RESOLVED) {
    CL2M (val, tmp);                   /* write to tmp file */
    write_tmp ((char *)&val, sz);
  } else update_lnmap (sz);
  return p;
}

/* Called when an expression is next in input stream.  Compiles and evaluates 
 * expression which must be immediate and resolved.  Returns value.
 */
U32
get_imm ()
{
  exptyp exp;
  U16 typ;
  lblptr undf;
  U32 val;
  int len;

  compile_exp (exp, &len);
  eval_exp (exp, &typ, &val, &undf);
  if (!(typ & T_RESOLVED) || ((typ & T_LBL) != T_IMM)) {
    error ("#expression does not evaluate to immediate in phase 1");
    return 0;
  }
  return val;
}

/* Writes bytes to current segment's temp file.  Keeps the line map structure
 * up-to-date if creating a listing.  Update location pointer.
 */
write_tmp (ptr, sz)
char *ptr;
int  sz;
{
  update_lnmap (sz);
  myfwrite (ptr, sz, curseg == T_TEXT? tmpt: tmpd);
}

/* Keep the mapping of line numbers to object code up-to-date.
 * Also, updates current location pointer.
 */
update_lnmap (sz)
int sz;
{
  struct melt *meltp;

  *curlocptr += sz;                    /* update current location pointer */
  if (listing) {                       /* update line map */
    if (ln_tail == NULL) {             /* check if there is a lnmap */
      ln_tail = ln_head =              /* alloc first lnmap */
        (lnptr) myalloc (sizeof (struct lnmap));
      ln_tail->link = NULL;
      ln_tail->len = 1;
      meltp = ln_tail->map;
      meltp->lnnum = lnnum;
      meltp->size = 0;
    } else {
      meltp = ln_tail->map +           /* get pointer to current map */
        ln_tail->len - 1;
      if (meltp->lnnum != lnnum) {     /* was last call for same lnnum? */
        if (ln_tail->len == MAXMAP) {  /* room in current lnmap? */
          ln_tail->link = (lnptr)      /* get new lnmap */
            myalloc (sizeof (struct lnmap));
          ln_tail = ln_tail->link;
          ln_tail->link = NULL;
          ln_tail->len = 1;
          meltp = ln_tail->map;
        } else {
          ++meltp;
          ++ln_tail->len;
        }
        meltp->lnnum = lnnum;
        meltp->size = 0;
      }
    }
    meltp->size = ((meltp->size & 0x7fffffff) + sz) |
      (curseg == T_DATA? lDATA: 0);
  }
}

/* Called when we have an expression in a local variable and we want to
 * save it.  Space is alloc'd, the expression is copied, and the pointer is
 * updated.
 */
copy_exp (e, len)
expptr *e;
int len;                               /* expression length in bytes */
{
  register expptr p, q;

  q = *e;
  p = *e = (expptr) myalloc (len);
  for (; p < *e + len;) *p++ = *q++;
}

/* Passed a patch record, writes the corresponding bytes to memory.
 * If type of patch is pSHORT, | the value into ival (an instruction)
 * and write the result.
 */
fmt_patch (pch, ival, outstr)
patchptr pch;
U32 ival;
register U8 *outstr;
{
  register U32 val;

  val = pch->val;
  if (pch->flags & pBYTEREV) val = byte_reverse (val);
  if (pch->flags & pPCREL) val -= *curlocptr;
  if (pch->flags & pIMM) {		/* immediate */
    switch (pch->size) {
      case 1:
        *outstr = val;
        break;
      case 2:
        WS2M (*outstr, val);
        break;
      case 4:
        WL2M (*outstr, val);
        break;
    }
  } else if (pch->flags & pGENIMM) {	/* gen operand immediate */
    switch (pch->size) {
      case 1:
        *outstr = B0 (val);
        break;
      case 2:
        *outstr = B1 (val);
        *(outstr+1) = B0 (val);
        break;
      case 4:
        *outstr++ = B3 (val);
        *outstr++ = B2 (val);
        *outstr++ = B1 (val);
        *outstr = B0 (val);
        break;
    }
  } else if (pch->flags & pDISP) {	/* displacement */
    if (pWORDSIZE & pch->flags) {	/* check for movm, cmpm */
      switch (pWORDSIZE & pch->flags) {
	case pBYTE: --val; break;
	case pWORD: val = 2 * (val - 1); break;
	case pDOUBLE: val = 4 * (val - 1); break;
      }
      if (val > 15) error ("#length exceeds 16 bytes");
    }
    switch (pch->size) {
      case 1:
        *outstr = B0 (val) & 0x7f;
        break;
      case 2:
        *outstr = B1 (val) & 0x3f | 0x80;
        *(outstr+1) = B0 (val);
        break;
      case 4:
        *outstr++ = B3 (val) & 0x3f | 0xc0;
        *outstr++ = B2 (val);
        *outstr++ = B1 (val);
        *outstr = B0 (val);
        break;
    }
  } else if (pch->flags & pALIGN)	/* alignment */
    *((long *) outstr) = 0;
  else if (pch->flags & pSHORT) {	/* short, quick */
    insert_short (pch, &ival);
    *outstr++ = B0 (ival);
    *outstr++ = B1 (ival);
    if (pch->size == 3) *outstr++ = B2 (ival);
  } else error ("!#internal error in fmt_patch");
}

/* Check short value and type, insert it into ival.
 */
insert_short (pch, ival)
register patchptr pch;
U32 *ival;
{
  register U32 val;

  if ((val = pch->val) > 0x7 && val < 0xfffffff0)
    error ("#short value out of range");
  if (!(pch->typ & T_IMM))
    error ("#short has relocatable type");
  *ival |= (val & 0xf) << (pch->size == 2? 7: 15);
}



/* Reverses the order of the bits in the least significant byte of val.
 * Enter/save use the bitmap r7|r6|...|r0 while exit/restore use the bitmap
 * r0|r1|...|r7.  If the assember reverses the arguement of exit/restore, the 
 * user can use the same constant for both.
 */
U32
byte_reverse (val)
register U32 val;
{
  register U32 outval;
  int i;

  outval = 0;
  for (i = 0; i < 8; ++i) {
    outval <<= 1;
    outval |= val & 1;
    val >>= 1;
  }
  return outval;
}
