#ifndef lint
static char *sccsid = "@(#)%M%  %I%  Teemu Torma %H%";
#endif lint

/* Receive file(s) using Xmodem/TeLink/MODEM7 Batch protocols.
   
   @(#)Copyright (c) 1987 by Teemu Torma
   
   Permission is given to distribute this program and alter this code as
   needed to adapt it to forign systems provided that this header is
   included and that the original author's name is preserved. */

#include <stdio.h>
#include <sys/types.h>
#include "fnet.h"
#include "fio.h"
#include "crc.h"

extern time_t time();

/* Macros to check timeouts */

#define Timeout(t)      (time((long *) NULL) - stime > t)
#define SetStart()      (stime = time((long *) NULL))

/* General error states to break the loops. */

#define Error           (-1)
#define Done            (-2)

/* States for XMODEM/TeLink receiver. */

#define RecStart        (0)
#define WaitFirst       (1)
#define WaitBlock       (2)

/* States for BATCH file receiver. */

#define RecvName        (0)
#define CheckFNm        (1)
#define CheckFile       (2)

/* States for MODEM7 filename receiver. */

#define SendNak         (0)
#define WaitAck         (1)
#define WaitChar        (2)
#define WaitOkCk        (3)

bool
xtrecblk(rxbuf, blocknum, crcmode)
     char *rxbuf;
     int blocknum;
     bool crcmode;
{
  register int c, cnt, checksum;
  register unsigned short crc;
  
  debug(1, "Receiving block %d", blocknum);
  if ((c = readline(10)) == TIMEOUT || c != (blocknum & 0377))
    {
      if (c == TIMEOUT)
        debug(1, "Timeout on block number receive");
      else
        debug(1, "Bad block number %d, expected %d", c, blocknum & 0377);
      goto blidge;
    }
  if ((c = readline(10)) == TIMEOUT || c != ~(blocknum & 0377))
    {
      if (c == TIMEOUT)
        debug(1, "Timeout on complement block number receive");
      else
        debug(1, "Bad ~block number %d, expected %d", c,
              ~(blocknum & 0377));
      goto blidge;
    }
  for (crc = 0, checksum = 0, cnt = 0; cnt < BlockSize; cnt++)
    if ((c = readline(10)) == TIMEOUT)
      {
        debug(1, "Timeout while receiving block");
        goto blidge;
      }
    else
      {
        checksum += c;
        crc = updcrc(c, crc);
        rxbuf[cnt] = c;
      }
  if (crcmode)
    {
      if (readline(10) != (int) ((crc >> 8) & 0377)
          || readline(10) != (int) (crc & 0377))
        {
          debug(1, "Crc error");
          goto blidge;
        }
    }
  else
    {
      if (readline(10) != (checksum & 0377))
        {
          debug(1, "Checksum error");
          goto blidge;
        }
    }
  return True;

 blidge:
  while (readline(1) != TIMEOUT)
    /* skip rest of the block */;
  return False;
}

bool
xtrec(filename)
     char *filename;
{
  int state = RecStart;
  time_t stime;
  int tries = 0;
  bool crcmode = True;
  FILE *fp;
  char rxbuf[BlockSize];
  int cnt, block = 0, c;
  
  if ((fp = fopen(filename, "w")) == NULL)
    {
      log("$Unable to open %s for writing", filename);
      return False;
    }
  log("Receive file %s", filename);
  SetStart();
  while (state >= RecStart && state <= WaitBlock)
    switch (state)
      {
      case RecStart:
        sendline(crcmode ? 'C' : NAK);
        state = WaitFirst;
        break;
      case WaitFirst:
        if (tries > 10 || Timeout(60))
          {
            if (tries > 10)
              log("Too many tries on xmodem receive");
            else
              log("Timeout on xmodem receive start");
            state = Error;
          }
        else if (tries > 3 || Timeout(30))
          {
            crcmode = False;
            state = RecStart;
          }
        else if ((c = readline(10)) == EOT)
          {
            sendline(ACK);
            log("No file to receive");
            state = Done;
          }
        else if (c == SYN || c == SOH)
          {
            debug(1, "Startup in %s mode", crcmode ? "crc" : "checksum");
            if (!xtrecblk(rxbuf, c == SYN ? 0 : 1, crcmode))
              {
                tries++;
                state = RecStart;
              }
            else
              {
                if (c == SOH)
                  {
                    for (cnt = 0; cnt < BlockSize; cnt++)
                      (void) putc(c, fp);
                    block = 2;
                    debug(2, "Block written onto disk");
                  }
                else
                  {
                    block = 1;
                    debug(1, "TeLink block ignored");
                  }
                state = WaitBlock;
              }
          }
        else if (c == TIMEOUT)
          {
            debug(1, "Timout on Xmodem rcv start");
            tries++;
            state = RecStart;
          }
        break;
      case WaitBlock:
        SetStart();
        for (tries = 0; state == WaitBlock; tries++)
          if (tries > 10 || Timeout(60))
            {
              if (tries > 10)
                log("Too many retries on %s", filename);
              else
                log("Timeout on receive %s", filename);
              state = Error;
            }
          else if ((c = readline(10)) == EOT)
            {
              log("File %s received ok", filename);
              state = Done;
            }
          else if (c == SOH)
            {
              if (xtrecblk(rxbuf, block, crcmode))
                {
                  for (cnt = 0; cnt < BlockSize; cnt++)
                    (void) putc(c, fp);
                  debug(2, "Block written onto disk");
                  sendline(ACK);
                  block++;
                  SetStart();
                  tries = 0;
                }
              else
                {
                  sendline(NAK);
                  tries++;
                }
            }
          else if (c == TIMEOUT)
            {
              debug(1, "Timeout on block %d", block);
              sendline(NAK);
              tries++;
            }
        break;
      }
  
  (void) fclose(fp);
  
  return state != Error;
}

int
recmdmfn(filename)
     char *filename;
{
  int state = SendNak;
  time_t stime;
  int tries = 0, c, pos;
  
  SetStart();
  while (state >= SendNak && state <= WaitOkCk)
    switch (state)
      {
      case SendNak:
        if (tries > 20)
          {
            log("Too many tries to get filename");
            state = Error;
          }
        else if (Timeout(60))
          {
            log("Timeout while getting filename");
            state = Error;
          }
        else
          {
            sendline(NAK);
            state = WaitAck;
            tries++;
          }
        break;
      case WaitAck:
        switch (readline(5))
          {
          case ACK:
            pos = 0;
            state = WaitChar;
            break;
          case EOT:
            pos = 0;
            state = Done;
            break;
          case TIMEOUT:
            debug(2, "Timout while waiting ACK/EOT on MDM7");
            state = SendNak;
            break;
          default:
            state = SendNak;
            debug(2, "Garbage on line while getting filename");
            (void) sleep((unsigned) 1);
            flush();
            break;
          }
        break;
      case WaitChar:
        switch (c = readline(1))
          {
          case EOT:
            pos = 0;
            debug(2, "Got EOT in middle of filename, no files left");
            state = Done;
            break;
          case SUB:
            filename[pos] = 0;
            debug(2, "Got fn %s, sending checksum", filename);
            for (pos = 0, c = SUB; filename[pos]; pos++)
              c += filename[pos];
            sendline(c & 0377);
            state = WaitOkCk;
            break;
          case 'u':
            debug(2, "Got 'u', send NAK again");
            state = SendNak;
            break;
          case TIMEOUT:
            debug(2, "Timeout while waiting char of filename");
            state = SendNak;
            break;
          default:
            filename[pos++] = c;
            debug(3, "Got char '%c' of filename", c);
            break;
          }
        break;
      case WaitOkCk:
        if (readline(1) == ACK)
          {
            debug(1, "Got filename %s ok", filename);
            state = Done;
          }
        else
          {
            debug(1, "Checksum faiure in filename %s", filename);
            state = SendNak;
          }
        break;
      }
  
  return state == Error ? -1 : pos ? 1 : 0;
}

bool
batchrec()
{
  int state = RecvName;
  char filename[16];
  int ok;
  
  while (state >= RecvName && state <= CheckFile)
    switch (state)
      {
      case RecvName:
        ok = recmdmfn(filename);
        state = CheckFNm;
        break;
      case CheckFNm:
        switch (ok)
          {
          case -1:
            debug(1, "Abort batch receive");
            state = Error;
            break;
          case 0:
            log("All files received successfully");
            state = Done;
            break;
          case 1:
            ok = xtrec(filename);
            state = CheckFile;
          }
        break;
      case CheckFile:
        if (ok)
          {
            debug(1, "%s received successfully", filename);
            state = RecvName;
          }
        else
          {
            log("Batch receive aborted, %s not received", filename);
            state = Error;
          }
        break;
      }
  
  return state != Error;
}
