#ifdef COMMENT

	The purpose of these routines is to provide a way to hand arguments
	to a program via the standard input.  The shell syntax
		chmod a-w `find . -print`
	can be used to feed arguments to a program, but you will often run
	into the 'Arg list too long' problem this way.  After chmod was
	modified to use the arg routines, the previous command could be
	rewritten as
		find . -print | chmod a-w =

	This approach has proven to be a more efficient alternative to
	using the '-exec' option to the 'find' command.  The previous
	example was measured to run about 10 times faster than this:
		find . -exec chmod a-w {} \;

        When you use these argument scanning routines, the first occurrance
        of "=" in the argument list is replaced by a list of newline-
	terminated arguments read from the standard input.  Any subsequent
	"=" argument in the argument list or in the standard input has
	no special effect.

        Instead of referencing argv and argc directly, you call
	"initargs(mode, arglist, argcount)" to initialize the argument
        scanning.  Initargs returns a char * pointing at the first
	argument.  The 'mode' argument is 1 to initalize args, 2 to initialize
        with intent to rescan, and 0 to reset for rescanning (only the "1"
        case is currently implemented.)
	Then for subsequent arguments, you call "nextarg()", which
	returns a char * pointing at the next argument.
	NULL is returned if at end of arg list.
	The string pointed to may go away, so if you want to keep it,
	you must make a copy of it.
	Newlines in the standard input act to separate arguments unless
	preceded by a backslash, in which case the backslash is discarded
	and the newline becomes part of the argument.

	Dave Yost, Rand Corp.

#endif

#include <stdio.h>
#include <args.h>

#define YES 1
#define NO  0

#define NAMSIZ 512

char *args_stdinstr = "=";

static char **curargv;          /* current argv */
static int    curargc;          /* current argc */
static char   argstdin;         /* YES if input coming from stdin */
static char   argstr[NAMSIZ];   /* current stdin arg string */
static char  *argcp;            /* NULL or points to argstr */
static int    stdinok;          /* stdin redirection allowed only once */

#define Block

/* initialize the current argument and arg count */
/* VARARGS 1 */
char *
initargs (mode, allow_stdin, argv, argc)
int mode;
int allow_stdin;
char **argv;
int argc;
{
    switch (mode) {
    case ARGS_SCAN:             /* initialize without intent to rescan */
	curargv = argv;
	curargc = argc;
	break;
    default:
	return NULL;
    }
    stdinok = allow_stdin;
    return curarg ();
}

/* return a (char *) pointing to the next argument or NULL if none */
char *
nextarg ()
{
    if (!argstdin) {
	if (   curargc <= 0
	    || --curargc <= 0
	   )
	    return NULL;
	if (*curargv)
	    curargv++;
    }
    else
	argcp = NULL;
    return curarg ();
}

/* return a char * pointing to the current argument or NULL if none */
char *
curarg ()
{
    extern char *getfarg ();
    if (curargc <= 0)
	return NULL;
    if (!argstdin) {
	if (!*curargv)
	    return NULL;
	else if (   !stdinok
		 || strcmp (*curargv, args_stdinstr)
		)
	    return *curargv;
	argstdin = YES;
	stdinok = NO;
    }
    if (   argcp
	|| (   !feof (stdin)
	    && (argcp = getfarg ())
	   )
       )
	return argcp;
    argstdin = NO;
    if (--curargc <= 0)
	return NULL;
    return *++curargv;
}

/* get the next argument from the standard input */
static char *
getfarg ()
{
    register char *cp;
    register int chr;
    register int esc;

    esc = NO;
    for (cp = argstr; ;) {
	if ((chr = getchar ()) == EOF) {
	    if (cp == argstr)
		return NULL;
	    break;
	}
	if (chr == '\n') {
	    if (!esc)
		break;
	    cp--;
	}
	if (cp >= &argstr[NAMSIZ - 1]) {
	    *cp = '\0';
	    fprintf (stderr, "%s", argstr);
	    do {
		if (chr == '\n')
		    break;
		putchar (chr);
	    } while ((chr = getchar ()) != EOF);
	    fprintf (stderr, ": string too long\n");
	    cp = argstr;
	}
	if (esc)
	    esc = NO;
	else if (chr == '\\')
	    esc = YES;
	*cp++ = chr;
    }
    *cp = '\0';
    return argstr;
}
