/* xwdtopbm.c - read an X11 window dump file and write a portable bitmap
**
** Copyright (C) 1988 by Jef Poskanzer.
**
** Permission to use, copy, modify, and distribute this software and its
** documentation for any purpose and without fee is hereby granted, provided
** that the above copyright notice appear in all copies and that both that
** copyright notice and this permission notice appear in supporting
** documentation.  This software is provided "as is" without express or
** implied warranty.
*/

#include <stdio.h>
#include "pbm.h"

main( argc, argv )
int argc;
char *argv[];
    {
    FILE *ifd;
    bit **bits, getbit();
    int rows, cols, padR, row, col;

    if ( argc > 2 )
	{
	fprintf( stderr, "usage:  %s [xwdfile]\n", argv[0] );
	exit( 1 );
	}

    if ( argc == 2 )
	{
        ifd = fopen( argv[1], "r" );
        if ( ifd == NULL )
	    {
	    fprintf( stderr, "%s: can't open.\n", argv[1] );
	    exit( 1 );
	    }
	}
    else
	ifd = stdin;

    getinit( ifd, &cols, &rows, &padR );

    bits = pbm_allocarray( cols, rows );

    for ( row = 0; row < rows; row++ )
	{
        for ( col = 0; col < cols; col++ )
	    bits[row][col] = getbit( ifd );
        for ( col = 0; col < padR; col++ )
	    (void) getbit( ifd );
	}

    if ( ifd != stdin )
	fclose( ifd );
    
    pbm_writepbm( stdout, bits, cols, rows );

    exit( 0 );
    }


unsigned char bitem;
short sitem;
int bits_per_item, bits_used, bit_shift, bit_order, bit_invert;

/* The following defs are taken from various X10 header files. */
#define X10WD_FILE_VERSION 6
typedef struct {
    int header_size;		/* Size of the entire file header (bytes). */
    int file_version;		/* X10WD_FILE_VERSION */
    int display_type;		/* Display type. */
    int display_planes;		/* Number of display planes. */
    int pixmap_format;		/* Pixmap format. */
    int pixmap_width;		/* Pixmap width. */
    int pixmap_height;		/* Pixmap height. */
    short window_width;		/* Window width. */
    short window_height;	/* Window height. */
    short window_x;		/* Window upper left X coordinate. */
    short window_y;		/* Window upper left Y coordinate. */
    short window_bdrwidth;	/* Window border width. */
    short window_ncolors;	/* number of Color entries in this window */
    } X10WDFileHeader;

typedef struct {
    int pixel;
    unsigned short red, green, blue;
    } X10Color;


/* The following defs are taken from various X11 header files. */
typedef unsigned long xwdval;
#define X11WD_FILE_VERSION 7
typedef struct {
    xwdval header_size;		/* Size of the entire file header (bytes). */
    xwdval file_version;	/* X11WD_FILE_VERSION */
    xwdval pixmap_format;	/* Pixmap format */
    xwdval pixmap_depth;	/* Pixmap depth */
    xwdval pixmap_width;	/* Pixmap width */
    xwdval pixmap_height;	/* Pixmap height */
    xwdval xoffset;		/* Bitmap x offset */
    xwdval byte_order;		/* MSBFirst, LSBFirst */
    xwdval bitmap_unit;		/* Bitmap unit */
    xwdval bitmap_bit_order;	/* MSBFirst, LSBFirst */
    xwdval bitmap_pad;		/* Bitmap scanline pad */
    xwdval bits_per_pixel;	/* Bits per pixel */
    xwdval bytes_per_line;	/* Bytes per scanline */
    xwdval visual_class;	/* Class of colormap */
    xwdval red_mask;		/* Z red mask */
    xwdval green_mask;		/* Z green mask */
    xwdval blue_mask;		/* Z blue mask */
    xwdval bits_per_rgb;	/* Log base 2 of distinct color values */
    xwdval colormap_entries;	/* Number of entries in colormap */
    xwdval ncolors;		/* Number of Color structures */
    xwdval window_width;	/* Window width */
    xwdval window_height;	/* Window height */
    long window_x;		/* Window upper left X coordinate */
    long window_y;		/* Window upper left Y coordinate */
    xwdval window_bdrwidth;	/* Window border width */
    } X11WDFileHeader;

typedef struct {
    unsigned long pixel;
    unsigned short red, green, blue;
    char flags;			/* do_red, do_green, do_blue */
    char pad;
    } X11XColor;


getinit( file, colP, rowP, padRP )
FILE *file;
int *colP, *rowP, *padRP;
    {
    int header_size;
    int file_version;
    X10WDFileHeader h10;
    X11WDFileHeader h11;
    char junk[10000];

    if ( fread( &header_size, sizeof( header_size ), 1, file ) != 1 )
	{
	fprintf( stderr, "Couldn't read XWD header size.\n" );
	exit( 1 );
	}
    if ( fread( &file_version, sizeof( file_version ), 1, file ) != 1 )
	{
	fprintf( stderr, "Couldn't read XWD file version.\n" );
	exit( 1 );
	}
    if ( file_version == X10WD_FILE_VERSION )
	{
	if ( fread( &h10.display_type, sizeof( h10 ) - 2 * sizeof( int ), 1,
                    file ) != 1 )
	    {
	    fprintf( stderr, "Couldn't read X10 XWD file header.\n" );
	    exit( 1 );
	    }
	if ( fread( junk, header_size - sizeof( h10 ), 1, file ) != 1 )
	    {
	    fprintf( stderr, "Couldn't read rest of X10 XWD file header.\n" );
	    exit( 1 );
	    }
	if ( fread( junk, sizeof(X10Color), h10.window_ncolors, file ) !=
	     h10.window_ncolors )
	    {
	    fprintf( stderr, "Couldn't read X10 XWD colormap.\n" );
	    exit( 1 );
	    }

	/* Check whether we can handle this dump. */
	if ( h10.window_ncolors != 0 )
	    {
	    fprintf( stderr, "Can't handle X10 window_ncolors != 0.\n" );
	    exit( 1 );
	    }
	if ( h10.pixmap_format != 0 )
	    {
	    fprintf( stderr, "Can't handle X10 pixmap_format %d.\n",
		     h10.pixmap_format );
	    exit( 1 );
	    }

	*colP = h10.pixmap_width;
	*rowP = h10.pixmap_height;
	*padRP = ( h10.pixmap_width + 15 ) / 16 * 16 - h10.pixmap_width;
	bits_per_item = 16;
	bit_order = 0;
	bit_invert = 1;
	bits_used = 16;
	}
    else if ( file_version == X11WD_FILE_VERSION )
	{
	if ( fread( &h11.pixmap_format, sizeof( h11 ) - 2 * sizeof( xwdval ),
		    1, file ) != 1 )
	    {
	    fprintf( stderr, "Couldn't read X11 XWD file header.\n" );
	    exit( 1 );
	    }
	if ( fread( junk, header_size - sizeof( h11 ), 1, file ) != 1 )
	    {
	    fprintf( stderr, "Couldn't read rest of X11 XWD file header.\n" );
	    exit( 1 );
	    }
	if ( fread( junk, sizeof( X11XColor ), h11.ncolors, file ) !=
	     h11.ncolors )
	    {
	    fprintf( stderr, "Couldn't read X11 XWD colormap.\n" );
	    exit( 1 );
	    }

	/* Check whether we can handle this dump. */
	if ( h11.pixmap_depth != 1 )
	    {
	    fprintf( stderr, "Can't handle X11 pixmap_depth > 1.\n" );
	    exit( 1 );
	    }
	if ( h11.colormap_entries != 2 )
	    {
	    fprintf( stderr, "Can't handle X11 colormap_entries != 2.\n" );
	    exit( 1 );
	    }
	if ( h11.ncolors != 2 )
	    {
	    fprintf( stderr, "Can't handle X11 ncolors != 2.\n" );
	    exit( 1 );
	    }
	if ( h11.pixmap_format != 2 )
	    {
	    fprintf( stderr, "Can't handle X11 pixmap_format %d.\n",
		     h11.pixmap_format );
	    exit( 1 );
	    }

	*colP = h11.pixmap_width;
	*rowP = h11.pixmap_height;
	*padRP = h11.bytes_per_line * 8 - h11.pixmap_width;
	bits_per_item = 8;
	bit_order = h11.bitmap_bit_order;
	bit_invert = 0;
	bits_used = 8;
	}
    else
	{
	fprintf( stderr, "Unknown XWD file version: %d.\n", file_version );
	exit( 1 );
	}
    }

bit
getbit( file )
FILE *file;
    {
    bit b;

    if ( bits_used == bits_per_item )
	{
	if ( bits_per_item == 16 )
	    {
	    if ( fread( &sitem, 2, 1, file ) != 1 )
		{
		fprintf( stderr, "Couldn't read short bits.\n" );
		exit( 1 );
		}
	    }
	else
	    {
	    if ( fread( &bitem, 1, 1, file ) != 1 )
		{
		fprintf( stderr, "Couldn't read byte bits.\n" );
		exit( 1 );
		}
	    }
	bits_used = 0;

	if ( bit_order == 1 )
	    bit_shift = bits_per_item - 1;
	else
	    bit_shift = 0;
	}

    bits_used++;
    if ( bits_per_item == 16 )
	b = ( sitem >> bit_shift) & 1;
    else
	b = ( bitem >> bit_shift) & 1;
    if ( bit_invert )
	b = 1 - b;

    if ( bit_order == 1 )
	bit_shift--;
    else
	bit_shift++;

    return ( b );
    }
