#ifdef RCSID
static char RCSid[] =
"$Header: d:/tads/tads2/RCS/osgen.c 1.5 96/10/14 16:11:53 mroberts Exp $";
#endif

/* Copyright (c) 1990 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  osgen  - Operating System dependent functions, general implementation
Function
  This module contains certain OS-dependent functions that are common
  between several systems.  Routines in this file are selectively enabled
  according to macros defined in os.h:

    USE_STDIO     - implement os_print, os_flush, os_gets with stdio functions
    USE_DOSEXT    - implement os_remext, os_defext using MSDOS-like filename
                    conventions
    USE_NULLINIT  - implement os_init and os_term as do-nothing routines
    USE_NULLPAUSE - implement os_expause as a do-nothing routine
    USE_EXPAUSE   - use an os_expause that prints a 'strike any key' message
                    and calls os_waitc
    USE_TIMERAND  - implement os_rand using localtime() as a seed
    STD_ASKFILE   - use an os_askfile that just prints a prompt and gets
                    a line of text for the filename
    USE_NULLSTAT  - use a do-nothing os_status function
    USE_NULLSCORE - use a do-nothing os_score function
    RUNTIME       - enable character-mode console implementation  
    USE_STATLINE  - implement os_status and os_score using character-mode
                    status line implementation
    USE_OVWCHK    - implements default saved file overwrite check
    USE_NULLSTYPE - use a dummy os_settype routine

    If USE_STATLINE is defined, certain subroutines must be provided for
    your platform that handle the character-mode console:
        ossclr - clears a portion of the screen
        ossdsp - displays text in a given color at a given location
        ossscr - scroll down (i.e., moves a block of screen up)
        ossscu - scroll up (i.e., moves a block of screen down)
        ossloc - locate cursor

    If USE_STATLINE is defined, certain sub-options can be enabled:
        USE_SCROLLBACK - include output buffer capture in console system
        USE_HISTORY    - include command editing and history in console system
        USE_LDESC      - include long room description in status region
Modified
  04/24/93 JEras         - add os_locate() for locating tads-related files
  04/12/92 MJRoberts     - add os_strsc (string score) function
  03/26/92 MJRoberts     - add os_setcolor function
  09/26/91 MJRoberts     - os/2 user exit support
  09/04/91 MJRoberts     - stop reading resources if we find '$eof' resource
  08/28/91 MJRoberts     - debugger bug fix
  08/01/91 MJRoberts     - make runstat work correctly
  07/30/91 MJRoberts     - add debug active/inactive visual cue
  05/23/91 MJRoberts     - add user exit reader
  04/08/91 MJRoberts     - add full-screen debugging support
  03/10/91 MJRoberts     - integrate John's qa-scripter mods
  11/27/90 MJRoberts     - use time() not localtime() in os_rand; cast time_t
  11/15/90 MJRoberts     - created (split off from os.c)
*/

#define OSGEN_INIT
# include "os.h"
#undef OSGEN_INIT

#include "lib.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined(USE_STDARG) && !defined(USE_GCC_STDARG)
# include <stdarg.h>
#endif

#ifdef RUNTIME
# ifdef USE_SCROLLBACK
int osssbmode();
void scrpgup();
void scrpgdn();
void scrlnup();
void scrlndn();
static void ossdosb();
int os_f_plain = 0;
# endif /* USE_SCROLLBACK */
#endif /* RUNTIME */

/* define some dummy routines if not using graphics */
#ifndef USE_GRAPH
void os_mouhide() {}
void os_moushow() {}
#endif /* !USE_GRAPH */
    
/*
 *   Provide prototypes for some functions used herein
 */
#ifndef USE_DOOR
# ifdef USE_STDARG
void  os_printf(const char *f, ...);
#else
void  os_printf();
# endif /* USE_STDARG */
#endif /* USE_DOOR */

void  os_flush();
char *os_gets();
void  os_score();

/*
 *   If this port is to use the default saved file overwrite check, define
 *   USE_OVWCHK.  This routine tries to open the file; if successful, the
 *   file is closed and we ask the user if they're sure they want to overwrite
 *   the file.
 */
#ifdef USE_OVWCHK
int os_chkovw( filename )
char *filename;
{
    FILE *fp;
    
    if ( fp = fopen( filename, "r" ))
    {
        char buf[128];
        
        fclose( fp );
        getstring("That file already exists.  Overwrite it? (y/n) >", buf,
                  (int)sizeof(buf));
        if ( buf[0] != 'y' && buf[0] != 'Y' ) return( 1 );
    }
    return( 0 );
}
#endif /* USE_OVWCHK */

#ifdef USE_NULLSTYPE
void os_settype( f, t )
char *f;
int   t;
{
    /* nothing needs to be done on this system */
}
#endif /* USE_NULLSTYPE */

/******************************************************************************
* Ports can implement os_printf, os_flush, and os_gets as calls to the stdio
* routines of the same name by defining USE_STDIO.  These definitions can be
* used for any port for which the standard C run-time library is available.
******************************************************************************/

#ifdef USE_STDIO

/* os_printf is generally just printf(), but we broke it out to make things
*  easier for weird ports.
*/
# ifdef USE_STDARG
void os_printf(const char *f, ...)
{
    va_list argptr;

    va_start(argptr, f);
    vprintf(f, argptr);
    va_end(argptr);
}
# else /* USE_STDARG */
void os_printf(f,a1,a2,a3,a4)
char *f;
long a1,a2,a3,a4;
{
    printf( f, a1, a2, a3, a4 );
}
# endif /* USE_STDARG */

/*
 *   os_flush forces output of anything buffered for standard output.  It
 *   is generally used prior to waiting for a key (so the normal flushing
 *   may not occur, as it does when asking for a line of input).  
 */
void os_flush()
{
    fflush( stdout );
}

/*
 *   os_gets performs the same function as gets().  It should get a
 *   string from the keyboard, echoing it and allowing any editing
 *   appropriate to the system, and return the null-terminated string as
 *   the function's value.  The closing newline should NOT be included in
 *   the string.  
 */
char *os_gets( s )
char *s;
{
    return(gets( s ));
}
#endif /* USE_STDIO */

/*---------------------------------------------------------------------------*/
/*
 *   Door implementation.  Allows the run-time to be used as a door from
 *   a BBS.  
 */
#ifdef USE_DOOR

int os_comport;       /* -1 indicates CON, 0 indicates COM1, 1 is COM2, etc */

static int getoutstat()
{
    unsigned char port_stat;
    unsigned char modem_stat;
    int           port = os_comport;
    
    if (os_comport == -1) return(0xff);
    
    asm {
        mov ah, 3
        mov dx, port
        push bp
        push es
        push ds
        push si
        push di
        int  14h
        pop  di
        pop  si
        pop  ds
        pop  es
        pop  bp
        mov port_stat, ah
        mov modem_stat, al
    }
    if (!(modem_stat & 0x80))
    {
        /* dropped carrier - abort */
        if (os_comport != -1) com_close();
        exit(0);
    }
    return(modem_stat & (1 << 4));                  /* return clear-to-send */
}

void os_printf(char *f, ...)
{
    va_list argptr;
    char    buf[256];
    char   *p;

    va_start(argptr, f);
    vsprintf(buf, f, argptr);
    va_end(argptr);

    fputs(buf, stdout);
    if (os_comport != -1)
    {
        for (p = buf ; *p ; ++p)
        {
            while (!getoutstat());
            com_out(*p);
            if (*p == '\n')
            {
                while (!getoutstat());
                com_out('\r');
            }
        }
    }
}

void os_flush()
{
    if (os_comport == -1) fflush(stdout);
}

#include <time.h>

char os_getc()
{
    time_t last_key_time = time(NULL);
    char   c;

    for (;;)
    {
        if (time(NULL) - last_key_time > 300)
        {
            os_printf("\n\n\007\007You have been idle too long.  Exiting.\n");
            if (os_comport != -1) com_close();
            exit(0);
        }
        
        if (os_comport == -1)
            c = getch();
        else
        {
            getoutstat();                      /* check for dropped carrier */
            if (!com_ready()) continue;
            c = com_in();
        }

        return(c);
    }
}

void os_waitc()
{
    os_getc();
}

char *os_gets( s )
char *s;
{
    unsigned char *p;
    unsigned char  c;
    long           t;
    
    for (p = s ;; )
    {
        c = os_getc();
        switch(c)
        {
        case '\r':
            *p = '\0';
            os_printf("\n");
            return(s);

        case 8:                                           /* backspace (^H) */
        case 127:                                                 /* delete */
            if (p > s)
            {
                --p;
                os_printf("\010 \010");
            }
            break;

        case 21:                                               /* control U */
            while (p > s)
            {
                --p;
                os_printf("\010 \010");
            }
            break;

        default:
            if (c >= ' ' && p < s+127)
            {
                *p++ = c;
                os_printf("%c", c);
            }
            break;
        }
    }
}

/*
 *   To prevent security problems, we implement our own directory
 *   management system.  First, the BBS must pass the user name in the
 *   environment variable TADSUSER; the format of the TADSUSER value is
 *   arbitrary -- spaces and other punctuation characters are OK.  Second,
 *   we maintain a directory file.  The format of each line in this file
 *   is simple: Username:num:apparent-name.  The num is a three-digit
 *   number which gives the actual DOS filename (of the format
 *   TADS_xxx.SAV).  The apparent-name is an arbitrary string typed by the
 *   user to identify the file; it can contain pretty much any characters
 *   other than CR.  
 */
int os_askfile(prompt, reply, replen)
char *prompt;
char *reply;
int   replen;
{
    char    buf[128];
    char    buf2[128];
    char   *getenv();
    char   *uname;
    size_t  ulen;
    FILE   *fp;
    int     retval;
    long    pos;
    char   *p;
    int     id;
    int     l;

    uname = getenv("TADSUSER");
    if (!uname)
    {
        os_printf("Error:  No user information available to TADS.\n\
Please inform your SysOp.\n");
        return(1);
    }
    ulen = strlen(uname);

    fp = fopen("TADSFILE.DIR", "r+");
    if (!fp)
    {
        fp = fopen("TADSFILE.DIR", "w+");
        if (!fp)
        {
            os_printf("Error:  Unable to create TADS directory file.\n\
Please inform your SysOp.\n");
            return(1);
        }
    }

    for (;;)
    {
        os_printf("Type /L to list files, /D to delete a file\n");
        os_printf("%s > ", prompt);
        os_gets(buf);

        if (!stricmp(buf, "/L"))
        {
            fseek(fp, 0L, SEEK_SET);
            os_printf("\nListing files for user %s\n", uname);
            for (;;)
            {
                if (!fgets(buf, sizeof(buf), fp)) break;
                if ((l = strlen(buf)) && buf[l-1] == '\n') buf[l-1] = '\0';

                if (strlen(buf) > ulen && !memcmp(buf, uname, ulen)
                    && buf[ulen] == ':'
                    && memcmp(buf + ulen + 1, "xxx", 3) != 0)
                {
                    FILE *fp2;

                    sprintf(buf2, "TADS_%03d.SAV", atoi(buf + ulen + 1));
                    fp2 = fopen(buf2, "r");
                    if (fp2)
                    {
                        os_printf("   %s\n", buf + ulen + 5);
                        fclose(fp2);
                    }
                }
            }
            os_printf("\n");
        }
        else if (!stricmp(buf, "/D"))
        {
            int found;
            
            os_printf("Enter name of file to delete: ");
            os_gets(buf2);
            buf[40] = '\0';
            if (!buf[0]) continue;
            
            fseek(fp, 0L, SEEK_SET);
            for (found = 0 ;;)
            {
                pos = ftell(fp);
                if (!fgets(buf, sizeof(buf), fp)) break;
                if ((l = strlen(buf)) && buf[l-1] == '\n') buf[l-1] = '\0';

                if (strlen(buf) > ulen && !memcmp(buf, uname, ulen)
                    && buf[ulen] == ':'
                    && memcmp(buf + ulen + 1, "xxx", 3) != 0
                    && !strcmp(buf + ulen + 5, buf2))
                {
                    fseek(fp, pos, SEEK_SET);
                    memcpy(buf + ulen + 1, "xxx", (size_t)3);
                    fputs(buf, fp);
                    os_printf("File deleted.\n");
                    found = 1;
                    break;
                }
            }
            if (!found) os_printf("File not found - nothing deleted.\n");
        }
        else if (buf[0] == '\0')
        {
            retval = 1;
            break;
        }
        else
        {
            int found;

            buf[40] = '\0';
            fseek(fp, 0L, SEEK_SET);
            for (found = 0, id = 0 ;;)
            {
                if (!fgets(buf2, sizeof(buf2), fp)) break;
                if ((l = strlen(buf2)) && buf2[l-1] == '\n') buf2[l-1] = '\0';

                p = strchr(buf2, ':');
                if (p && atoi(p+1) > id) id = atoi(p+1);
                
                if (strlen(buf2) > ulen && !memcmp(buf2, uname, ulen)
                    && buf2[ulen] == ':'
                    && memcmp(buf2 + ulen + 1, "xxx", 3) != 0
                    && !strcmp(buf2 + ulen + 5, buf))
                {
                    id = atoi(p+1);
                    found = 1;
                    break;
                }
            }

            if (!found)
            {
                ++id;                                /* create a new max ID */
                sprintf(buf2, "%s:%03d:%s\n", uname, id, buf);
                fputs(buf2, fp);
            }
            sprintf(reply, "TADS_%03d.SAV", id);
            retval = 0;
            break;
        }
    }

    fclose(fp);
    return(retval);
}

#endif /* USE_DOOR */


/******************************************************************************
* Ports with MS-DOS-like file systems (Atari ST, OS/2, Macintosh, and, of
* course, MS-DOS itself) can use the os_defext and os_remext routines below
* by defining USE_DOSEXT.  Unix and VMS filenames will also be parsed correctly
* by these implementations, but untranslated VMS logical names may not be.
******************************************************************************/

#ifdef USE_DOSEXT
/* os_defext(fn, ext) should append the default extension ext to the filename
*  in fn.  It is assumed that the buffer at fn is big enough to hold the added
*  characters in the extension.  The result should be null-terminated.  When
*  an extension is already present in the filename at fn, no action should be
*  taken.  On systems without an analogue of extensions, this routine should
*  do nothing.
*/
void os_defext( fn, ext )
char *fn;
char *ext;
{
    char *p = fn+strlen(fn);
    while ( p>fn )
    {
        p--;
        if ( *p=='.' ) return;      /* already has an extension */
        if ( *p=='/' || *p=='\\' || *p==':'
# ifdef VMS
          || *p==']' || *p=='['     /* VMS has '[ ]' around path */
# endif /* VMS */
           ) break;    /* found a path */
    }
    strcat( fn, "." );              /* add a dot */
    strcat( fn, ext );              /* add the extension */
}

/*
 *   Add an extension, even if the filename currently has one 
 */
void os_addext(fn, ext)
char *fn;
char *ext;
{
    strcat(fn, ".");
    strcat(fn, ext);
}

/* os_remext(fn) removes the extension from fn, if present.  The buffer at
*  fn should be modified in place.  If no extension is present, no action
*  should be taken.  For systems without an analogue of extensions, this
*  routine should do nothing.
*/
void os_remext( fn )
char *fn;
{
    char *p = fn+strlen(fn);
    while ( p>fn )
    {
        p--;
        if ( *p=='.' )
        {
            *p = '\0';
            return;
        }
        if ( *p=='/' || *p=='\\' || *p==':'
# ifdef VMS
          || *p==']' || *p=='['     /* VMS has '[ ]' around path */
# endif /* VMS */
            ) return;
    }
}
#endif /* USE_DOSEXT */

/******************************************************************************
* Ports without any special initialization/termination requirements can define
* USE_NULLINIT to pick up the default definitions below.  These do nothing, so
* ports requiring special handling at startup and/or shutdown time must define
* their own versions of these routines.
******************************************************************************/

#ifdef USE_NULLINIT
/* os_init returns 0 for success, 1 for failure.  The arguments are &argc, the
*  address of the count of arguments to the program, and argv, the address of
*  an array of up to 10 pointers to those arguments.  For systems which don't
*  pass a standard command line (such as the Mac Finder), the arguments should
*  be read here using some alternate mechanism (an alert box, for instance),
*  and the values of argc and argv[] updated accordingly.  Note that a maximum
*  of 10 arguments are allowed to be stored in the argv[] array.  The command
*  line itself can be stored in buf, which is a buffer passed by the caller
*  guaranteed to be bufsiz bytes long.
*
*  Unix conventions are followed, so argc is 1 when no arguments are present.
*  The final argument is a prompt string which can be used to ask the user for
*  a command line; its use is not required, but may be desirable for producing
*  a relevant prompt message.  See the Mac implementation for a detailed
*  example of how this mechanism is used.
*/
int os_init(argc, argv, prompt, buf, bufsiz)
int *argc;
char *argv[];
char *prompt;
char *buf;
int bufsiz;
{
        return( 0 );
}

/* os_term should perform any necessary cleaning up, then terminate the
*  program.  The int argument is a return code to be passed to the caller,
*  generally 0 for success and other for failure.
*/
int os_term(rc)
int rc;
{
    qasclose();
    exit( rc );
}
#endif /* USE_NULLINIT */

/******************************************************************************
* Ports can define USE_NULLPAUSE if no exit pause on ti is needed.
*
* Ports needing an exit pause, and can simply print a message (with os_printf)
* and wait for a key (with os_getc) can define USE_EXPAUSE.
******************************************************************************/

#ifdef USE_NULLPAUSE
void os_expause()
{
    /* does nothing */
}
#endif /* USE_NULLPAUSE */

#ifdef USE_EXPAUSE
void os_expause()
{
    os_printf( "(Strike any key to exit...)" );
    os_flush();
    os_waitc();
}
#endif /* USE_EXPAUSE */

/*
 *   A port can define USE_TIMERAND if it wishes to randomize from
 *   the system clock.  This should be usable by most ports.
 */
#ifdef USE_TIMERAND
# include <time.h>

void os_rand( seed )
unsigned long *seed;
{
    time_t t;

    time( &t );
    *seed = (long)t;
}
#endif /* USE_TIMERAND */


#ifdef STD_ASKFILE
/*
 *   The default version of os_askfile(), which can be picked up by
 *   defining STD_ASKFILE, is rather dull; it merely uses os_printf() to
 *   display the prompt, and os_gets() to read the reply.
 *
 *   This routine is to return 0 for success, other for failure.
 *   Generally, the only way this routine can fail is if the user selects
 *   a "Cancel" button, if provided.  The default version always returns 0
 *   (success).
 */

int os_askfile( prompt, reply, replen )
char *prompt;
char *reply;
int   replen;
{
    outformat(prompt);
    getstring(" >", reply, replen);
    return(0);
}
#endif /* STD_ASKFILE */

#ifdef USE_NULLSTAT
/*
 *   USE_NULLSTAT defines a do-nothing version of os_status.
 */
void os_status( stat )
int stat;
{
    /* ignore the new status */
}
#endif /* USE_NULLSTAT */

#ifdef USE_NULLSCORE
/*
 *   USE_NULLSCORE defines a do-nothing version of os_score.
 */
void os_score( cur, turncount )
int cur;
int turncount;
{
    /* ignore the score information */
}

void os_strsc(p)
char *p;
{
    /* ignore */
}
#endif /* USE_NULLSCORE */

#ifdef USE_STATLINE
static int ldesc_savecol;

/*
 *  The special run-time version displays a status line and the ldesc before
 *  each command line is read.  To accomplish these functions, we need to
 *  redefine os_gets and os_printf with our own special functions.
 *
 *  Note that os_init may have to modify some of these values to suit, and
 *  should set up the screen appropriately.
 */
void os_status( stat )
int stat;
{
    if ( status_mode == 2 && stat != 2 )
    {
# ifdef USE_LDESC
#  ifndef USE_GRAPH
/* #  ifndef USE_FSDBUG */
        while ( ldesc_curline < text_line-1 )
        {
            ossclr( ldesc_curline, ldesc_column, text_line-2, max_column,
             ldesc_color );
            ldesc_curline++;
        }
/* #  endif */ /* USE_FSDBUG */
#  endif /* USE_GRAPH */
# endif /* USE_LDESC */
        text_lastcol = ldesc_savecol;
    }
    status_mode = stat;
    if ( status_mode == 2 )
    {
# ifndef USE_FSDBUG
        ldesc_curline = ldesc_line;
        ldesc_savecol = text_lastcol;
# endif /* USE_FSDBUG */
    }
}

/* Set score to a string value provided by the caller */
void os_strsc(p)
char *p;
{
    static char lastbuf[135];
           int  i;
           int  x = score_column;

    if (os_f_plain) return;
    
    if (p) strcpy(lastbuf, p);                 /* save new value for redraw */
    else p = lastbuf;        /* null pointer ==> redraw from previous value */

    /* display enough spaces to right-justify the value */
    for (i = strlen(p) ; i + score_column <= max_column ; ++i)
        ossdsp(sdesc_line, x++, sdesc_color, " ");
    if (x + strlen(p) > max_column)
        score_column = x = max_column - strlen(p);
    ossdsp(sdesc_line, x, sdesc_color, p);
    ossdsp(sdesc_line, max_column, sdesc_color, " ");
}

/*
 *   Set the score.  If cur == -1, the LAST score set with a non-(-1)
 *   cur is displayed; this is used to refresh the status line without
 *   providing a new score (for example, after exiting scrollback mode).
 *   Otherwise, the given current score (cur) and turncount are displayed,
 *   and saved in case cur==-1 on the next call.
 */
void os_score( cur, turncount )
int cur;
int turncount;
{
    char buf[20];
    
    if (turncount == -1)
    {
        os_strsc((char *)0);
    }
    else
    {
        sprintf( buf, "%d/%d", cur, turncount );
        os_strsc(buf);    /* call direct string-setting routine with buffer */
    }
}

# ifdef USE_SCROLLBACK
/*
 *   Pointers for scrollback buffers.
 */
static char *scrbuf, *scrbufp;
static unsigned scrbufl;

void osssbini( size )
unsigned int size;
{
    scrbufl = size;
    if ( !( scrbufp = scrbuf = calloc( scrbufl, 1 )))
    {
        os_printf( "\nSorry, there is no memory for review mode.\n\n" );
    }
}

void osssbdel()
{
    if ( scrbuf ) free( scrbuf );
}

static char *ossadvsp( p )
char *p;
{
    p++;
    if ( p >= scrbuf + scrbufl ) p = scrbuf;
    return( p );
}

static char *ossdecsp( p )
char *p;
{
    if ( p == scrbuf ) p = scrbuf + scrbufl;
    return( p-1 );
}

/* add a color control sequence to start of line if necessary */
static void ossaddbold(sb_text_color)
int sb_text_color;
{
    if (sb_text_color != text_normal_color)
    {
        *scrbufp = 2;              /* store bold code at start of next line */
        scrbufp = ossadvsp(scrbufp);
    }
}

int   os_line_count = 1;             /* count of lines in scrollback buffer */
int   os_top_line;   /* if scrtop if non-null, line # of top line on screen */
static int sb_column;

static void ossaddsb( p )
char *p;
{
    int sb_text_color;

    if ( !scrbufp ) return;

    /*
     *   Copy the text into the screen buffer, respecting the circular
     *   nature of the screen buffer.  If the given text wraps lines,
     *   enter an explicit carriage return into the text to ensure that
     *   users can correctly count lines.
     */
    sb_text_color = text_color;
    while( *p )
    {
        switch(*p)
        {
        case 1:
            sb_text_color = text_normal_color;
            break;
            
        case 2:
            sb_text_color = text_bold_color;
            break;
        }

        /* if overwriting a newline, decrement total line counter */
        if (*scrbufp == '\n') --os_line_count;

        *scrbufp = *p;
        scrbufp = ossadvsp( scrbufp );
        if (*p >= 10) sb_column++;           /* ignore our special controls */
        if ( *p == '\n' )
        {
            ++os_line_count;
            ossaddbold(sb_text_color);
            sb_column = 0;
        }
        if ( *p == '\r' )
        {
            /*
             *   We have a plain carriage return, which indicates that we
             *   should go back to the start of the current line and
             *   overwrite it.  (This is most likely to occur for the
             *   "[More]" prompt.)
             */
            sb_column = 0;
            do
            {
                scrbufp = ossdecsp( scrbufp );      /* back up one character */
            } while ( *scrbufp && *scrbufp != '\n' ); /* go to start of line */
            scrbufp = ossadvsp( scrbufp );             /* move onto the line */
        }
        if ( sb_column > max_column )
        {
            *scrbufp = '\n';
            ++os_line_count;
            sb_column = 0;
            scrbufp = ossadvsp( scrbufp );
            ossaddbold(sb_text_color);
        }
        p++;
    }

    /*
     *   If we've wrapped around into existing screen buffer space,
     *   we must delete it.  Enter nulls up to the next line break.
     */
    if ( *scrbufp != '\0' )
    {
        char *p;
        for ( p=scrbufp ; *p && *p != '\n' ; )
        {
            if (*p == '\n') --os_line_count;             /* deleting a line */
            *p = '\0';
            p = ossadvsp( p );
        }
        if ( *p == '\n' ) --os_line_count, *p = '\0';
    }
}

/*
 *   A slight complication:  if we're not on the
 *   very bottom line, and we don't even have any
 *   text displayed on the bottom line (example:
 *   the input line grew to over 80 characters,
 *   scrolling up a line, but shrunk again due to
 *   deletions), we must add "\n"'s to fill out the
 *   unused space, so the scrollback line counter
 *   doesn't get confused (it counts backwards from
 *   the bottom of the screen, where it assumes we
 *   are right now).  Figure out if the current
 *   buffer will go to the end of the screen.
 *   "i" contains the number of "\n"'s left to do;
 *   each time the buffer overflows the screen
 *   width, we know we are actually using one more
 *   line, so decrement "i".
 *
 *   This routine, ossaddsbe, adds the given text to the screen
 *   buffer, then adds as many blank lines as are necessary to
 *   fill the screen to the bottom line.  buf is the buffer to
 *   add; p is a pointer into the buffer corresponding to screen
 *   position (x,y).
 */
static void ossaddsbe( buf, p, x, y )
char *buf;
char *p;
int   x;
int   y;
{
    int i = max_line - y;                    /* number of blank lines to add */

    for ( ; *p ; p++)
    {
        if ( x > max_column )                 /* we've wrapped to a new line */
        {
            i--;                                     /* one less line to add */
            x=0;                                     /* back to first column */
        }
        if (*p >= 10) ++x;          /* don't count controls in column count */
    }
    ossaddsb( buf );                                /* add the buffer itself */
    while ( i-- ) ossaddsb( "\n" );   /* and any blank lines that are needed */
}

/*
 *   Scrollback mode variables.  These track the current position while
 *   in scrollback mode.
 */
char *scrtop;                         /* first line displayed on the screen */
static char *scrbot;                         /* last line displayed on sreen */
static char *scrlast;                        /* first line of last screenful */

/*
 *   scrdsp  - display a line of text from the scrollback buffer on the
 *   screen.  It is the responsibility of the caller to ensure that the
 *   line on the screen has been cleared beforehand.
 */
static void scrdsp( p, y )
char *p;
int   y;
{
    char  buf[135];
    char *q = buf;
    int   color = text_normal_color;
    int   x = text_column;
    int   newcolor;

    while(*p && *p != '\n' && q < buf + sizeof(buf) - 1)
    {
        switch(*p)
        {
        case 2:                                       /* activate bold mode */
            newcolor = text_bold_color;
            goto bold_on_off;
        case 1:                                    /* disactivate bold mode */
            newcolor = text_normal_color;
        bold_on_off:
            *q = '\0';
            ossdsp(y, x, color, buf);
            color = newcolor;
            
            /* go back to start of buffer and proceed */
            q = buf;
            x += strlen(buf);
            break;
            
        default:
            *q++ = *p;
        }
        
        /* advance to next character no matter what happened */
        p = ossadvsp(p);
    }
    *q = '\0';
    ossdsp(y, x, color, buf);
}

/*
 *   scrdspscr - display a screenful of text starting at a given location
 *   in the scrollback buffer.
 */
static void scrdspscr( p )
char *p;
{
    int y;

    ossclr( text_line, text_column, max_line, max_column, text_color );
    for ( y=text_line ; y<=max_line ; y++ )
    {
        scrdsp( p, y );
        while( *p && *p != '\n' )
            p = ossadvsp( p );
        if ( *p ) p = ossadvsp( p );
    }
}

/* move forward by a number of lines, and redisplays the sreen */
static void scrfwd(n)
int n;
{
    if (scrtop == scrlast) return;

    while(n)
    {
        if (scrtop == scrlast) break;
        if (*scrtop == '\n')
        {
            ++os_top_line;
            --n;
            while(*scrbot && *scrbot != '\n') scrbot = ossadvsp(scrbot);
            if (*scrbot) scrbot = ossadvsp(scrbot);
        }
        scrtop = ossadvsp(scrtop);
    }
    scrdspscr(scrtop);
}

/* move back by a number of lines, redisplaying the screen */
static void scrback(n)
int n;
{
    if (!*ossdecsp(scrtop)) return;

    scrtop = ossdecsp(scrtop);           /* back up to end of previous line */
    while(n)                                 /* go until we've seen n lines */
    {
        scrtop = ossdecsp(scrtop);                   /* back up a character */
        if (*scrtop == '\n' || !*scrtop)  /* start of line - back up scrbot */
        {
            --n;                                 /* ... and count this line */
            --os_top_line;
            scrbot = ossdecsp(scrbot);            /* back up scrbot onto \n */

            /* and now find the \n before this line */
            do { scrbot = ossdecsp(scrbot); } while (*scrbot != '\n');
            scrbot = ossadvsp(scrbot);          /* advance to start of line */
        }
        if (!*scrtop) break;                    /* top of buffer - stop now */
    }
    scrtop = ossadvsp(scrtop);                /* advance onto start of line */
    scrdspscr(scrtop);                            /* display the screenfull */
}

/*
 *   scrto moves to a selected line
 */
void scrto(lin)
int lin;
{
    if (lin > os_top_line)
        scrfwd(lin - os_top_line);
    else if (lin < os_top_line)
        scrback(os_top_line - lin);
}

/*
 *   scrpgup scrolls back a page
 */
void scrpgup()
{
    scrback(max_line - text_line);
}

/*
 *   scrpgdn scrolls forward a page
 */
void scrpgdn()
{
    scrfwd(max_line - text_line);
}

/*
 *   scrlnup scrolls up one line
 */
void scrlnup()
{
    if ( *ossdecsp( scrtop ) == '\0' ) return;

    scrtop = ossdecsp( scrtop );
    scrbot = ossdecsp( scrbot );
    do { scrtop = ossdecsp( scrtop ); } while ( *scrtop && *scrtop != '\n' );
    do { scrbot = ossdecsp( scrbot ); } while ( *scrbot && *scrbot != '\n' );
    scrtop = ossadvsp( scrtop );
    scrbot = ossadvsp( scrbot );
    --os_top_line;

    ossscu( text_line, text_column, max_line, max_column, text_color );
    scrdsp( scrtop, text_line );
}

/*
 *   scrlndn scrolls down one line
 */
void scrlndn()
{
    if ( scrtop == scrlast ) return;

    while( *scrtop != '\n' ) scrtop = ossadvsp( scrtop );
    while( *scrbot != '\n' && *scrbot ) scrbot = ossadvsp( scrbot );
    scrtop = ossadvsp( scrtop );
    if ( *scrbot ) scrbot = ossadvsp( scrbot );
    ++os_top_line;

    ossscr( text_line, text_column, max_line, max_column, text_color );
    scrdsp( scrbot, max_line );
}

/* redraw the screen's text region */
void os_redraw()
{
    int   y = max_line;
    int   i;
    char *p;

#ifdef USE_GRAPH
    os_drawcontrols();
#endif /* USE_GRAPH */

    ossclr(text_line, text_column, max_line, max_column, text_color);
    p = scrbufp;
    while (y >= text_line)
    {
        p = ossdecsp(p);
        if (*p == '\0')
        {
            int old_text_line = text_line;
            text_line = y;
            p = ossadvsp(p);
            scrdspscr(p);
            text_line = old_text_line;
            return;
        }
        if (*p == '\n') --y;
    }
    p = ossadvsp(p);
    scrdspscr(p);
}

/* change colors */
void os_setcolor(stat, text, ldesc)
int stat;
int text;
int ldesc;
{
    sdesc_color = stat;
    text_color = text;
    ldesc_color = ldesc;
    os_redraw();
}

/* vertically resize the text window, redisplaying text */
int ossresize(newtop, newbot)
int newtop;
int newbot;
{
    extern int pagelength;
    
    if (newtop != -1) text_line = newtop;
    if (newtop != -1) max_line = newbot;
    pagelength = max_line - text_line - 1;
    os_redraw();
    
    return(0);
}

/*
 *   osssbmode toggles scrollback mode.  Once in scrollback mode, calls
 *   can be made to osssbup, osssbdn, osssbpgup, and osspgdn to scroll
 *   through the saved text.  We also put up a scrollback-mode version of
 *   the status line, showing keys that should be used for scrollback.
 *   When the user wants to exit scrollback mode, this routine should
 *   be called again.
 *
 *   Returns:  0 if scrollback mode is entered/exited successfully, 1 if
 *   an error occurs.  Note that it may not be possible to enter scrollback
 *   mode, since insufficient saved text may have been accumulated.
 */
int osssbmode(mode_line)
int mode_line;                    /* TRUE ==> display mode line if entering */
{
    if ( !scrbufp ) return( 1 );  /* no buffer - can't enter scrollback mode */

    if ( !scrtop )
    {
        /*
         *   Enter scrollback mode.  Figure out what scrtop should be, and
         *   put up the scrollback status line.  If insufficient saved text
         *   is around for scrollback mode, return 1.
         */
        int  y = max_line;
        char buf[135];
        int  i;

        scrtop = scrbufp;                          /* start at end of buffer */
        os_top_line = os_line_count;      /* start at line count for buffer */
        while ( y >= text_line )           /* keep going until top of screen */
        {
            scrtop = ossdecsp( scrtop );                 /* back up one byte */
            if ( *scrtop == '\0' )                     /* end of the buffer? */
            {
                scrtop = 0;                /* there wasn't enough saved text */
                return( 1 );            /* ... so we can't do the scrollback */
            }
            if ( *scrtop == '\n' ) --y, --os_top_line;        /* count line */
        }
        scrtop = ossadvsp( scrtop );   /* forward over '\n' to start of line */
        scrlast = scrtop;            /* remember where current screen starts */

        /*
         *   Note the last line on the screen as well.  Just find the
         *   beginning of the last line currently in the buffer.
         */
        scrbot = ossdecsp( scrbufp );
        while ( *scrbot != '\n' ) scrbot = ossdecsp( scrbot );
        scrbot = ossadvsp( scrbot );

        /*
         *   Construct the appropriate status line, and display it.
         */
        if (mode_line)
        {
            strcpy( buf, OS_SBSTAT );
            for ( i=strlen( buf ) ; i<max_column ; buf[i++] = ' ' );
            buf[i] = '\0';
            ossdsp( sdesc_line, sdesc_column+1, sdesc_color, buf );
        }

        return( 0 );                 /* successfully entered scrollback mode */
    }
    else
    {
        /*
         *   Exit scrollback mode.  Show the last page of text, and put
         *   up the normal status bar.  Also clear our scrollback mode
         *   variables.
         */

        if ( scrlast != scrtop ) scrdspscr( scrlast );

# ifdef USE_LDESC
#  ifndef USE_FSDBUG
#   define STAT_TYPE 1
#  endif /* USE_FSDBUG */
# endif /* USE_LDESC */
# ifndef STAT_TYPE
#  define STAT_TYPE 0
# endif /* STAT_TYPE */
        if (mode_line)
        {
            runstat( STAT_TYPE );
            os_score( -1, -1 );
        }
        scrtop = 0;
        return( 0 );
    }
}

/*
 *   ossdosb runs a scrollback session.  When entered, osssbmode()
 *   must already have been called.  When we're done, we'll be out
 *   of scrollback mode.
 */
static void ossdosb()
{
    for ( ;; )
    {
        if ( !os_getc())
        {
            switch( os_getc())
            {
                case CMD_SCR:
                case CMD_KILL:     /* leave scrollback via 'escape' as well */
                    osssbmode(1);
                    return;
                case CMD_UP:
                    scrlnup();
                    break;
                case CMD_DOWN:
                    scrlndn();
                    break;
                case CMD_PGUP:
                    scrpgup();
                    break;
                case CMD_PGDN:
                    scrpgdn();
                    break;
            }
        }
    }
}

# else /* USE_SCROLLBACK */
static void ossaddsb( p )
char *p;
{
    /* We're not saving output - do nothing */
}
# endif /* USE_SCROLLBACK */

/* display with no highlighting - intercept and ignore highlight codes */
void ossdspn(y, x, color, p)
char *p;
{
    char *q;
    
    for (q = p ; *q ; ++q)
    {
        switch(*q)
        {
        case 1:                                          /* set mode normal */
        case 2:                                       /* set mode highlight */
            *q = '\0';
            ossdsp(y, x, color, p);
            x += strlen(p);
            p = q + 1;
            break;

        default:
            break;
        }
    }
    if (q != p) ossdsp(y, x, color, p);             /* display last portion */
}

/* display with highlighting enabled */
int ossdsph(y, x, color, p)
char *p;
{
    char *q;
    int   newcolor;
    int   len;

    /* get length of entire string */
    len = strlen(p);
    
    for (q = p ; *q ; ++q)
    {
        switch(*q)
        {
        case 1:                                          /* set mode normal */
            newcolor = text_normal_color;
            goto bold_on_off;
        case 2:                                       /* set mode highlight */
            newcolor = text_bold_color;
        bold_on_off:
            *q = '\0';
            ossdsp(y, x, text_color, p);
            x += strlen(p);
            text_color = newcolor;
            p = q + 1;

            /* don't count the switch character in the lenght */
            --len;
            break;

        default:
            break;
        }
    }
    if (q != p) ossdsp(y, x, text_color, p);        /* display last portion */
    return len;
}

/*
 *   Set the terminal into 'plain' mode: disables status line,
 *   scrollback, command editing.
 */
void os_plain()
{
    os_f_plain = 1;
}

# ifdef USE_STDARG
void os_printf(const char *fmt, ...)
# else /* USE_STDARG */
void os_printf( fmt, a1, a2, a3, a4, a5, a6, a7, a8 )
char *fmt;
long  a1, a2, a3, a4, a5, a6, a7, a8;
# endif /* USE_STDARG */
{
    char buf[256];

# ifdef USE_STDARG
    va_list argptr;

    va_start( argptr, fmt );
    vsprintf( buf, fmt, argptr );
    va_end( argptr );
# else /* USE_STDARG */
    sprintf( buf, fmt, a1, a2, a3, a4, a5, a6, a7, a8 );
# endif /* USE_STDARG */
    if ( status_mode == 0 && !os_f_plain)
        ossaddsb( buf );                             /* save general output */
    switch( status_mode )
    {
        case 2:                                        /* ldesc display mode */
# ifdef USE_LDESC
            if (os_f_plain) break;                /* ignore in 'plain' mode */
            if ( ldesc_curline >= text_line-1 ) break;   /* too much already */
            /* otherwise, fall through to normal display code */
# else /* USE_LDESC */
            break;                       /* don't allow ldesc display at all */
# endif /* USE_LDESC */
        case 0:
        {
            char *p;

            for ( p=buf ; *p ; )
            {
                char *p1;

                for (p1 = p ; *p1 && *p1 != '\n' && *p1 != '\r' ; p1++);
                if (*p1 == '\n' || *p1 == '\r')
                {
                    int c = *p1;

                    *p1 = '\0';
                    if (status_mode == 0)
                    {
                        if (os_f_plain)
                        {
                            char cbuf[2];
                            
                            fputs(p, stdout);
                            cbuf[0] = c;
                            cbuf[1] = '\0';
                            fputs(cbuf, stdout);
                        }
                        else
                        {
                            ossdsph(max_line, text_lastcol, text_color, p);
                            if (c == '\n')
                                ossscr(text_line, text_column, max_line,
                                       max_column, text_color);
                        }
                    }
# ifdef USE_LDESC
                    else
                    {
                        ossdspn(ldesc_curline, text_lastcol, ldesc_color, p);
                        ossclr(ldesc_curline, text_lastcol + strlen(buf),
                               ldesc_curline, max_column, ldesc_color);
                        ++ldesc_curline;
                    }
# endif /* USE_LDESC */
                    text_lastcol = text_column;
                    p = p1 + 1;
                }
                else
                {
                    int len;
                        
                    if (status_mode == 0)
                    {
                        if (os_f_plain)
                        {
                            len = strlen(p);
                            fputs(p, stdout);
                        }
                        else
                        {
                            len = ossdsph(max_line, text_lastcol,
                                          text_color, p);
                        }
                    }
                    else
                    {
                        len = strlen(p);
                        ossdspn(ldesc_curline++, text_lastcol, ldesc_color,p);
                    }

                    text_lastcol += len;
                    p = p1;
                }
                if (status_mode == 0 && !os_f_plain)
                    ossloc(max_line, text_lastcol);
            }
            break;
        }
        case 1:                                          /* status-line mode */
        {
            int   i;
            char *p;

            /* ignore status line in 'plain' mode */
            if (os_f_plain) break;
            
            for ( p=buf ; *p == '\n' ; p++ );
            if (( i=strlen( p )) && p[i-1] == '\n' ) p[--i] = '\0';
            while ( i<score_column-1 ) buf[i++] = ' ';
            buf[i] = '\0';
#ifndef USE_GRAPH
            ossdspn(sdesc_line, sdesc_column, sdesc_color, " ");
#endif /* USE_GRAPH */
            ossdspn(sdesc_line, sdesc_column+1, sdesc_color, buf);
            os_status( 2 );
            break;
        }
    }
}

void os_flush()
{
}

# ifdef USE_HISTORY
/*
 *   For command line history, we must have some buffer space to store
 *   past command lines.  We will use a circular buffer:  when we move
 *   the pointer past the end of the buffer, it wraps back to the start
 *   of the buffer.  A "tail" indicates the oldest line in the buffer;
 *   when we need more room for new text, we advance the tail and thereby
 *   lose the oldest text in the buffer.
 */
unsigned char  histbuf[HISTBUFSIZE];
unsigned char *histhead = histbuf;
unsigned char *histtail = histbuf;

/*
 *   ossadvhp advances a history pointer, and returns the new pointer.
 *   This function takes the circular nature of the buffer into account
 *   by wrapping back to the start of the buffer when it hits the end.
 */
char *ossadvhp( p )
char *p;
{
    if ( ++p >= histbuf+sizeof( histbuf )) p=histbuf;
    return( p );
}

/*
 *   ossdechp decrements a history pointer, wrapping the pointer back
 *   to the top of the buffer when it reaches the bottom.
 */
char *ossdechp( p )
char *p;
{
    if ( p == histbuf ) p = histbuf+sizeof( histbuf );
    return( p-1 );
}

/*
 *  osshstcpy copies from a history buffer into a contiguous destination
 *  buffer, wrapping the history pointer if need be.  One null-terminated
 *  string is copied.
 */
void osshstcpy( dst, hst )
char *dst;
char *hst;
{
    while( *hst )
    {
        *dst++ = *hst;
        hst = ossadvhp( hst );
    }
    *dst = '\0';
}

/*
 *   ossprvcmd returns a pointer to the previous history command, given
 *   a pointer to a history command.  It returns a null pointer if the
 *   given history command is the first in the buffer.
 */
char *ossprvcmd( hst )
char *hst;
{
    if ( hst == histtail ) return( 0 );         /* no more previous commands */
    hst = ossdechp( hst );                                  /* back onto nul */
    do
    {
        hst = ossdechp( hst );                         /* back one character */
    } while ( *hst && hst != histtail );        /* go until previous command */
    if ( !*hst ) hst = ossadvhp( hst );               /* step over null byte */
    return( hst );
}

/*
 *   ossnxtcmd returns a pointer to the next history command, given
 *   a pointer to a history command.  It returns a null pointer if the
 *   given command is already past the last command.
 */
char *ossnxtcmd( hst )
char *hst;
{
    if ( hst == histhead ) return( 0 );         /* past the last one already */
    while( *hst ) hst = ossadvhp( hst );             /* scan forward to null */
    hst = ossadvhp( hst );                /* scan past null onto new command */
    return( hst );
}
# endif /* USE_HISTORY */

int  osqstate;
unsigned char osqbuf[64];

char *os_gets( buf )
unsigned char *buf;
{
    unsigned char *p = buf;
    unsigned char *eol = buf;
    unsigned char *eob = buf + 127;
    int   x = text_lastcol;
    int   y = max_line;
    int   origx = x;
# ifdef USE_HISTORY
    unsigned char *curhist = histhead;
    unsigned char  savbuf[135];
# endif /* USE_HISTORY */
    extern int dbgactive;

    /* disable command editing in 'plain' mode */
    if (os_f_plain) return gets(buf);

/* # ifndef USE_FSDBUG */
#  ifdef USE_LDESC
    text_lastcol = 0;
#  endif /* USE_LDESC */
    if (!dbgactive) runstat( STAT_TYPE );
/* # endif */ /* USE_FSDBUG */

    ossloc( y, x );
    buf[0] = '\0';
    osqstate = 1;                               /* ready to accept commands */

    for ( ;; )
    {
        unsigned char c;
        
        /* if there's a queued command, use it */
        if (osqstate == 2)
        {
            c = CMD_QUEUE;
            goto replace_line;
        }

        ossloc( y, x );
        switch( c = os_getc() )
        {
            case 8:
                if ( p>buf )
                {
                    char *q;
                    char  tmpbuf[2];
                    int   thisx, thisy;

                    for ( q=(--p) ; q<eol ; q++ ) *q = *( q+1 );
                    eol--;
                    if ( --x < 0 )
                    {
                        x = max_column;
                        y--;
                    }
                    *eol = ' ';
                    thisx = x;
                    thisy = y;
                    for ( q=p, tmpbuf[1]='\0' ; q<=eol ; q++ )
                    {
                        tmpbuf[0] = *q;
                        ossdsp( thisy, thisx, text_color, tmpbuf );
                        if ( ++thisx > max_column )
                        {
                            thisx = 0;
                            thisy++;
                        }
                    }
                    *eol = '\0';
                }
                break;
            case 13:
            return_line:
                /*
                 *   Scroll the screen to account for the carriage return,
                 *   position the cursor at the end of the new line, and
                 *   null-terminate the line.
                 */
                *eol = '\0';
                while( p != eol )
                {
                    p++;
                    if ( ++x > max_column )
                    {
                        y++;
                        x = 0;
                    }
                }
                ossloc( y, x );
                if ( y == max_line )
                    ossscr( text_line, text_column, max_line, max_column,
                     text_color );
                text_lastcol = 0;

# ifdef USE_HISTORY
                /*
                 *   Save the line in our history buffer.  If we don't
                 *   have enough room, lose some old text by advancing
                 *   the tail pointer far enough.  Don't save it if it's
                 *   a blank line, though, or if it duplicates the most
                 *   recent previous command.
                 */
                if ( strlen( buf ))
                {
                    char *q;
                    int   advtail;
                    int   saveflag = 1;       /* assume we will be saving it */

                    if ( q = ossprvcmd( histhead ))
                    {
                        char *p = buf;

                        while ( *p == *q && *p && *q )
                        {
                            p++;
                            q = ossadvhp( q );
                        }
                        if ( *p == *q )      /* is this a duplicate command? */
                            saveflag = 0;            /* if so, don't save it */
                    }

                    if ( saveflag )
                    {
                        for ( q=buf, advtail=0 ; q<=eol ; q++ )
                        {
                            *histhead = *q;
                            histhead = ossadvhp( histhead );
                            if ( histhead == histtail )
                            {
                                histtail = ossadvhp( histtail );
                                advtail = 1;
                            }
                        }

                        /*
                         *   If we have encroached on space that was
                         *   already occupied, throw away the entire
                         *   command we have partially trashed; to do
                         *   so, advance the tail pointer to the next
                         *   null byte.
                         */
                        if ( advtail )
                        {
                            while( *histtail )
                                histtail = ossadvhp( histtail );
                            histtail = ossadvhp( histtail );
                        }
                    }
                }
# endif /* USE_HISTORY */

                /*
                 *   Finally, copy the buffer to the screen save buffer
                 *   (if applicable), and return the contents of the buffer.
                 *   Note that we add an extra carriage return if we were
                 *   already on the max_line, since we scrolled the screen
                 *   in this case; otherwise, ossaddsbe will add all the
                 *   blank lines that are necessary.
                 */
                ossaddsbe( buf, p, x, y );
                if ( y == max_line ) ossaddsb( "\n" );
                osqstate = 0;
                return( buf );

            case 0:
                switch( c = os_getc())
                {
# ifdef USE_SCROLLBACK
                    case CMD_SCR:
                    {
                        char *savbufp = scrbufp;

                        /*
                         *   Add the buffer, plus any blank lines, to the
                         *   screen buffer, filling the screen to the bottom.
                         */
                        ossaddsbe( buf, p, x, y );

                        if ( !osssbmode(1)) ossdosb();
                        scrbufp = savbufp;
                        *scrbufp = '\0';
                        sb_column = origx;
                        break;
                    }
# endif /* USE_SCROLLBACK */

                    case CMD_LEFT:
                        if ( p>buf )
                        {
                            p--;
                            x--;
                            if ( x < 0 )
                            {
                                x = max_column;
                                y--;
                            }
                        }
                        break;
                    case CMD_WORD_LEFT:
                        if (p > buf)
                        {
                            --p;
                            --x;
                            if (x < 0) x = max_column, --y;
                            while (p > buf && isspace(*p))
                            {
                                --p, --x;
                                if (x < 0) x = max_column, --y;
                            }
                            while (p > buf && !isspace(*(p-1)))
                            {
                                --p, --x;
                                if (x < 0) x = max_column, --y;
                            }
                        }
                        break;
                    case CMD_RIGHT:
                        if ( p<eol )
                        {
                            p++;
                            x++;
                            if ( x > max_column )
                            {
                                x = 0;
                                y++;
                            }
                        }
                        break;
                    case CMD_WORD_RIGHT:
                        while (p < eol && !isspace(*p))
                        {
                            ++p, ++x;
                            if (x > max_column) x = 0, ++y;
                        }
                        while (p < eol && isspace(*p))
                        {
                            ++p, ++x;
                            if (x > max_column) x = 0, ++y;
                        }
                        break;
                    case CMD_DEL:
                        if ( p<eol )
                        {
                            char *q;
                            char  tmpbuf[2];
                            int   thisx=x, thisy=y;

                            for ( q=p ; q<eol ; q++ ) *q = *(q+1);
                            eol--;
                            *eol = ' ';
                            for ( q=p, tmpbuf[1]='\0' ; q<=eol ; q++ )
                            {
                                tmpbuf[0] = *q;
                                ossdsp( thisy, thisx, text_color, tmpbuf );
                                if ( ++thisx > max_column )
                                {
                                    thisx = 0;
                                    thisy++;
                                }
                            }
                            *eol = '\0';
                        }
                        break;
#ifdef UNIX               
                case CMD_WORDKILL:
                    {
                    char *q;
                    char tmpbuf[2];
                    int  thisx, thisy;

                    /* remove spaces preceding word */
                    while (p >= buf && *p <= ' ') {
                            for (q=(--p); q<eol; q++) *q = *(q+1);
                            eol--;
                            
                            if (--x < 0) {
                                    x = max_column;
                                    y--;
                            }
                            *eol = ' ';
                            thisx = x;
                            thisy = y;
                    }
                    
                    /* remove previous word (i.e., until we get a space) */
                    while (p >= buf && *p > ' ') {
                            for (q=(--p); q<eol; q++) *q = *(q+1);
                            eol--;
                            
                            if (--x < 0) {
                                    x = max_column;
                                    y--;
                            }
                            *eol = ' ';
                            thisx = x;
                            thisy = y;
                    }
                    
                    for ( q=p, tmpbuf[1]='\0' ; q<=eol ; q++ )
                    {
                        tmpbuf[0] = *q;
                        ossdsp( thisy, thisx, text_color, tmpbuf );
                        if ( ++thisx > max_column )
                        {
                            thisx = 0;
                            thisy++;
                        }
                    }
                    *eol = '\0';
                    break;
            }
#endif /* UNIX */
                    case CMD_KILL:
                    case CMD_HOME:
# ifdef USE_HISTORY
                    case CMD_UP:
                    case CMD_DOWN:
                    replace_line:
                        if ( c == CMD_UP && !ossprvcmd( curhist ))
                            break;         /* no more history - ignore arrow */
                        if ( c == CMD_DOWN && !ossnxtcmd( curhist ))
                            break;         /* no more history - ignore arrow */
                        if ( c == CMD_UP && !ossnxtcmd( curhist ))
                        {
                            /* first Up arrow - save current buffer */
                            strcpy( savbuf, buf );
                        }
# endif /* USE_HISTORY */
                        while( p>buf )
                        {
                            p--;
                            if ( --x < 0 )
                            {
                                x = max_column;
                                y--;
                            }
                        }
                        if ( c == CMD_HOME ) break;
                        /*
                         *   We're at the start of the line now; fall
                         *   through for KILL, UP, and DOWN to the code
                         *   which deletes to the end of the line.
                         */
                    case CMD_DEOL:
                        if ( p<eol )
                        {
                            char *q;
                            int   thisx=x, thisy=y;

                            for ( q=p ; q<eol ; q++ )
                            {
                                ossdsp( thisy, thisx, text_color, " " );
                                if ( ++thisx > max_column )
                                {
                                    thisx = 0;
                                    thisy++;
                                }
                            }
                            eol = p;
                            *p = '\0';
                        }
# ifdef USE_HISTORY
                        if ( c == CMD_UP )
                        {
                            curhist = ossprvcmd( curhist );
                            osshstcpy( buf, curhist );
                        }
                        else if ( c == CMD_DOWN )
                        {
                            if ( !ossnxtcmd( curhist )) break;    /* no more */
                            curhist = ossnxtcmd( curhist );
                            if ( ossnxtcmd( curhist )) /* on a valid command */
                                osshstcpy( buf, curhist );  /* ... so use it */
                            else
                            {
                                /* no more history - restore original line */
                                strcpy( buf, savbuf );
                            }
                        }
                        if (( c == CMD_UP || c == CMD_DOWN ) && strlen( buf ))
                        {
                            char tmpbuf[2];

                            tmpbuf[1] = '\0';
                            eol = buf + strlen( buf );
                            while( p<eol )
                            {
                                tmpbuf[0] = *p++;
                                ossdsp( y, x, text_color, tmpbuf );
                                if ( ++x > max_column )
                                {
                                    x = 0;
                                    if ( y == max_line )
                                        ossscr( text_line, text_column,
                                         max_line, max_column, text_color );
                                    else y++;
                                }
                            }
                        }
# endif /* USE_HISTORY */
                        if (c == CMD_QUEUE)
                        {
                            strcpy(buf, osqbuf);
                            ossdsp(y, x, text_color, buf);
                            p = eol = buf;
                            eol += strlen(eol);
                            goto return_line;
                        }
                        break;
                    case CMD_END:
                        while ( p<eol )
                        {
                            p++;
                            if ( ++x > max_column )
                            {
                                x = 0;
                                y++;
                            }
                        }
                        break;
                }
                break;
            default:
                if ( c >= ' ' && eol<eob )
                {
                    if ( p != eol )
                    {
                        char *q;
                        int   thisy=y, thisx=x;
                        char  tmpbuf[2];

                        for ( q=(++eol) ; q>p ; q-- ) *q=*(q-1);
                        *p = c;
                        for ( q=p++, tmpbuf[1] = '\0' ; q<eol ; q++ )
                        {
                            tmpbuf[0] = *q;
                            ossdsp( thisy, thisx, text_color, tmpbuf );
                            thisx++;
                            if ( thisx > max_column )
                            {
                                thisx = 0;
                                if ( thisy == max_line )
                                {
                                    y--;
                                    ossscr( text_line, text_column, max_line,
                                     max_column, text_color );
                                }
                                else thisy++;
                            }
                        }
                        if ( ++x > max_column )
                        {
                            y++;
                            x = 0;
                        }
                    }
                    else
                    {
                        *p++ = c;
                        *p = '\0';
                        eol++;
                        ossdsp( y, x, text_color, p-1 );
                        if ( ++x > max_column )
                        {
                            x = 0;
                            if ( y == max_line )
                                ossscr( text_line, text_column, max_line,
                                 max_column, text_color );
                            else y++;
                        }
                    }
                }
                break;
        }
    }
}
#else /* USE_STATLINE */

int ldesc_color;
void os_setcolor(a, b, c)
{
    /* does nothing if not in full-screen mode */
}

#endif /* USE_STATLINE */

#ifdef RUNTIME
/*
 *   Return character to insert to turn hilighting on (mode==2) or off
 *   (mode==1); a return of zero indicates no highlighting is supported.
 */
int os_hilite(mode)
int mode;
{
    return(os_f_plain ? 0 : mode);   /* just use same character as they use */
}
#else /* RUNTIME */
int os_hilite(mode)
{
    return(0);                                 /* no highlighting supported */
}
#endif /* RUNTIME */

#ifdef USE_PATHSEARCH
/* search a path specified in the environment for a tads file */
static int pathfind(fname, flen, pathvar, buf, bufsiz)
char   *fname;                                      /* name of file to find */
int     flen;                                            /* length of fname */
char   *pathvar;                                          /* path to search */
char   *buf;                                     /* result file name buffer */
size_t  bufsiz;                         /* size of result buffer (not used) */
{
    char   *e;

    if ( !(e = getenv(pathvar)) )
        return(0);
    for ( ;; )
    {
        char   *sep;
        size_t  len;

        if ( (sep = strchr(e, OSPATHSEP)) )
        {
            len = (size_t)(sep-e);
            if (!len) continue;
        }
        else
        {
            len = strlen(e);
            if (!len) break;
        }
        memcpy(buf, e, len);
        if (buf[len-1] != OSPATHCHAR && !strchr(OSPATHALT, buf[len-1]))
            buf[len++] = OSPATHCHAR;
        memcpy(buf+len, fname, flen);
        buf[len+flen] = 0;
        if (osfacc(buf) == 0) return(1);
        if (!sep) break;
        e = sep+1;
    }
    return(0);
}
#endif /* USE_PATHSEARCH */

/*
 *  Look for a tads-related file in the standard locations and, if
 *  the search is successful, store the result file name in the given
 *  buffer.  Return 1 if the file was located, 0 if not.
 *
 *  Search the following areas for the file:  current directory, program
 *  directory (as derived from argv[0]), and the TADS path.
 */
int os_locate(fname, flen, arg0, buf, bufsiz)
char   *fname;                                    /* name of file to locate */
int     flen;                                            /* length of fname */
char   *arg0;                 /* argv[0] - for looking in program directory */
char   *buf;                                     /* result file name buffer */
size_t  bufsiz;                         /* size of result buffer (not used) */
{
    /* Check the current directory */
    if (osfacc(fname) == 0)
    {
        memcpy(buf, fname, flen);
        buf[flen] = 0;
        return(1);
    }

    /* Check the program directory */
    if (arg0 && *arg0)
    {
        char   *p;

        /* find the end of the directory name of argv[0] */
        for ( p = arg0 + strlen(arg0);
              p > arg0 && *(p-1) != OSPATHCHAR && !strchr(OSPATHALT, *(p-1));
              --p )
            ;

        /* don't bother if there's no directory on argv[0] */
        if (p > arg0)
        {
            size_t  len = (size_t)(p - arg0);

            memcpy(buf, arg0, len);
            memcpy(buf+len, fname, flen);
            buf[len+flen] = 0;
            if (osfacc(buf) == 0) return(1);
        }
    }

#ifdef USE_PATHSEARCH
    /* Check TADS path */
    if ( pathfind(fname, flen, "TADS", buf, bufsiz) )
        return(1);
#endif /* USE_PATHSEARCH */

    return(0);
}

/* clear the screen - loses the scrollback buffer */
void oscls()
{
#ifdef RUNTIME
    if (os_f_plain) return;
    scrbufp = scrbuf;
    scrtop = scrbot = scrlast = 0;
    memset(scrbuf, 0, (size_t)scrbufl);
    os_redraw();
#endif
}

/*
 *   Create and open a temporary file 
 */
osfildef *os_create_tempfile(swapname, buf)
char *swapname;
char *buf;
{
    osfildef *fp;
    
    /* if a name wasn't provided, generate a name */
    if (swapname == 0)
    {
        int     try;
        size_t  len;
        time_t  timer;
        int     found;
        
        /* get the appropriate path for a temp file */
        os_get_tmp_path(buf);
        len = strlen(buf);

        /* get the current time, as a basis for a unique identifier */
        time(&timer);

        /* try until we find a non-existent filename */
        for (try = 0, found = FALSE ; try < 100 ; ++try)
        {
            /* generate a name based on time and try number */
            sprintf(buf + len, "SW%06ld.%03d", (long)timer % 999999, try);

            /* if this file doesn't exist, we're done */
            if (osfacc(buf))
            {
                found = TRUE;
                break;
            }
        }

        /* if all the files we tried existed already, give up */
        if (!found)
            return 0;

        /* use the buffer's contents as the name */
        swapname = buf;
    }

    /* open the file */
    fp = osfoprwtb(swapname);

    /* set the file's type in the OS, if necessary */
    os_settype(swapname, OSFTSWAP);

    /* return the file pointer */
    return fp;
}

