#ifdef RCSID
static char RCSid[] =
"$Header: d:/tads/tads2/RCS/fiowrt.c 1.7 96/10/14 16:10:30 mroberts Exp $";
#endif

/* Copyright (c) 1992 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  fiowrt.c - write game to binary file
Function
  Writes a game to binary file.  Separated from fio.c so that run-time
  doesn't have to link in this function, which is unnecessary for playing
  a game.
Notes
  None
Modified
  09/14/92 MJRoberts     - note location of undefined objects in errors
  04/11/92 MJRoberts     - creation
*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "std.h"
#include "os.h"
#include "mch.h"
#include "mcm.h"
#include "mcl.h"
#include "tok.h"
#include "obj.h"
#include "voc.h"
#include "fio.h"
#include "dat.h"
#include "sup.h"


/* write a resource header; returns pointer to next-res field in file */
static ulong fiowhd(fp, ec, resname)
osfildef *fp;
errcxdef *ec;
char     *resname;
{
    ulong ret;

    if (osfwb(fp, resname, (int)(resname[0] + 1))) errsig(ec, ERR_WRTGAM);
    ret = osfpos(fp);
    if (osfwb(fp, "\000\000\000\000", 4)) errsig(ec, ERR_WRTGAM);
    return(ret);
}

/* close a resource by writing next-res pointer */
static void fiowcls(fp, ec, respos)
osfildef *fp;
errcxdef *ec;
ulong     respos;
{
    ulong endpos;
    char  buf[4];
    
    endpos = osfpos(fp);
    osfseek(fp, respos, OSFSK_SET);
    oswp4(buf, endpos);
    if (osfwb(fp, buf, 4)) errsig(ec, ERR_WRTGAM);
    osfseek(fp, endpos, OSFSK_SET);
}

/* write a required object (just an object number) */
static void fiowrq(ec, fp, objn)
errcxdef *ec;
osfildef *fp;
objnum    objn;
{
    uchar buf[2];
    
    oswp2(buf, objn);
    if (osfwb(fp, buf, 2)) errsig(ec, ERR_WRTGAM);
}

/* context for fiowrtobj callback */
struct fiowcxdef
{
    mcmcxdef *fiowcxmem;
    errcxdef *fiowcxerr;
    osfildef *fiowcxfp;
    uint      fiowcxflg;
    int       fiowcxund;
    osfildef *fiowcxffp;
    uint      fiowcxseed;
    uint      fiowcxinc;
};
typedef struct fiowcxdef fiowcxdef;

/* write out a symbol table entry */
static void fiowrtsym(ctx, t)
fiowcxdef *ctx;
toksdef   *t;
{
    uchar     buf[TOKNAMMAX + 50];
    errcxdef *ec = ctx->fiowcxerr;
    osfildef *fp = ctx->fiowcxfp;

    buf[0] = t->tokslen;
    buf[1] = t->tokstyp;
    oswp2(buf + 2, t->toksval);
    memcpy(buf + 4, t->toksnam, (size_t)buf[0]);
    if (osfwb(fp, buf, 4 + t->tokslen)) errsig(ec, ERR_WRTGAM);
}

/* write an object given by a symbol table entry */
static void fiowrtobj(ctx, t)
fiowcxdef *ctx;
toksdef   *t;
{
    uchar     buf[TOKNAMMAX + 50];
    mcmon     obj;
    mcmcxdef *mctx = ctx->fiowcxmem;
    errcxdef *ec = ctx->fiowcxerr;
    osfildef *fp = ctx->fiowcxfp;
    uint      flags = ctx->fiowcxflg;
    uchar    *p;
    uint      siz;
    uint      used;
    int       err = 0;
    ulong     startpos = osfpos(fp);
    
    /* set up start of buffer to write */
    buf[0] = t->tokstyp;
    obj = t->toksval;
    oswp2(buf + 1, obj);
    
    switch(t->tokstyp)
    {
    case TOKSTOBJ:
        /* mark object as finished with compilation, get sizes */
        objcomp(mctx, (objnum)obj);
        p = mcmlck(mctx, (mcmon)obj);
        siz = mcmobjsiz(mctx, (mcmon)obj);         /* size in cache */
        used = objfree(p);          /* size actually used in object */
	if (objflg(p) & OBJFINDEX) used += objnprop(p) * 4;
        goto write_func_or_obj;
                
    case TOKSTFUNC:
        /* size of function is entire object */
        p = mcmlck(mctx, (mcmon)obj);
        siz = used = mcmobjsiz(mctx, (mcmon)obj);

    write_func_or_obj:
        /* write type(OBJ) + objnum + size + sizeused */
        oswp2(buf+3, siz);
        oswp2(buf+5, used);
        if (osfwb(fp, buf, 7)) err = ERR_WRTGAM;
                
        /* write contents of object */
        if (flags & FIOFCRYPT)
	    fioxor(p, used, ctx->fiowcxseed, ctx->fiowcxinc);
        if (osfwb(fp, p, used)) err = ERR_WRTGAM;
	
	/* write fast-load record if applicable */
	if (ctx->fiowcxffp)
	{
	    oswp4(buf + 7, startpos);
	    if (osfwb(ctx->fiowcxffp, buf, 11)) err = ERR_WRTGAM;
	}
                
        /*
         *   We're done with the object - delete it so that
         *   it doesn't have to be swapped out (which would
         *   be pointless, since we'll never need it again).
         */
        mcmunlck(mctx, (mcmon)obj);
        mcmfre(mctx, (mcmon)obj);
        break;
                
    case TOKSTEXTERN:
        /* all we must write is the name & number of ext func */
        buf[3] = t->tokslen;
        memcpy(buf + 4, t->toksnam, (size_t)t->tokslen);
        if (osfwb(fp, buf, t->tokslen + 4)) err = ERR_WRTGAM;
	
	/* write fast-load record if applicable */
	if (ctx->fiowcxffp
	    && osfwb(ctx->fiowcxffp, buf, t->tokslen + 4)) err = ERR_WRTGAM;
        break;
                
    case TOKSTFWDOBJ:
    case TOKSTFWDFN:
        {
            int  e = (t->tokstyp == TOKSTFWDFN ? ERR_UNDEFF : ERR_UNDEFO);

            if (flags & FIOFBIN)
            {
                /* write record for the forward reference */
		p = mcmlck(mctx, (mcmon)obj);
		siz = mcmobjsiz(mctx, (mcmon)obj);
		oswp2(buf+3, siz);
                if (osfwb(fp, buf, 5)
		    || osfwb(fp, p, siz))
		    err = ERR_WRTGAM;
            }
            else
            {
		/* log the undefined-object error */
		sup_log_undefobj(mctx, ec, e,
				 t->toksnam, (int)t->tokslen, obj);

		/* count the undefined object */
                ++(ctx->fiowcxund);

		/*
		 *   we don't need this object any longer - delete it so
		 *   that we don't have to bother swapping it in or out
		 */
		mcmfre(mctx, (mcmon)obj);
            }
        }
        break;
    }
                    
    /* if an error occurred, signal it */
    if (err) errsig(ec, err);
}

/* fast-load temporary file name */
#define FIOWRTFSTNM "TADSFAST.TMP"

/* write game to binary file */
static void fiowrt1(mctx, vctx, tokctx, tab, fmts, fmtl, fp, flags,
		    preinit, extc, prpcnt, filever)
mcmcxdef *mctx;
voccxdef *vctx;
tokcxdef *tokctx;
tokthdef *tab;
uchar    *fmts;
uint      fmtl;
osfildef *fp;
uint      flags;
objnum    preinit;
int       extc;
uint      prpcnt;
char     *filever;
{
    int         i;
    int         j;
    int         k;
    uint        used;
    uint        siz;
    uchar      *p;
    uchar       buf[TOKNAMMAX + 50];
    errcxdef   *ec = vctx->voccxerr;
    ulong       fpos;
    ulong       endpos;
    int         obj;
    vocidef  ***vpg;
    vocidef   **v;
    objnum     *sc;
    vocdef     *voc;
    vocwdef    *vw;
    vocdef    **vhsh;
    struct tm  *tblock;
    time_t      timer;
    fiowcxdef   cbctx;                    /* callback context for toktheach */
    uint        xor_seed = 63;
    uint        xor_inc = 64;
    char       *vsnhdr;
    uint        vsnlen;

    /* generate appropriate file version */
    switch(filever[0])
    {
    case 'a':          /* generate .GAM compatible with pre-2.0.15 runtimes */
	vsnhdr = FIOVSNHDR2;
	vsnlen = sizeof(FIOVSNHDR2);
	xor_seed = 17;                                /* use old xor values */
	xor_inc = 29;
	break;

    case 'b':                                    /* generate 2.0.15+ format */
	vsnhdr = FIOVSNHDR3;
	vsnlen = sizeof(FIOVSNHDR3);
	break;

    case 'c':
    case '*':                                            /* current version */
	vsnhdr = FIOVSNHDR;
	vsnlen = sizeof(FIOVSNHDR);
	break;

    default:
	errsig(ec, ERR_WRTVSN);
    }

    /* write file header, version header, flags, and timestamp */
    timer = time(NULL);
    tblock = localtime(&timer);
    strcpy(vctx->voccxtim, asctime(tblock));
    
    oswp2(buf, flags);
    if (osfwb(fp, FIOFILHDR, sizeof(FIOFILHDR))
        || osfwb(fp, vsnhdr, vsnlen)
        || osfwb(fp, buf, 2)
        || osfwb(fp, vctx->voccxtim, (size_t)26))
        errsig(ec, ERR_WRTGAM);

    /* write xor parameters if generating post 2.0.15 .GAM file */
    if (filever[0] != 'a')
    {
	fpos = fiowhd(fp, ec, "\003XSI");
	buf[0] = xor_seed;
	buf[1] = xor_inc;
	if (osfwb(fp, buf, 2)) errsig(ec, ERR_WRTGAM);
	fiowcls(fp, ec, fpos);
    }
    
    /* write count of external functions */
    if (extc)
    {
        fpos = fiowhd(fp, ec, "\006EXTCNT");
        oswp2(buf, extc);              /* write the external function count */
        if (osfwb(fp, buf, 2)) errsig(ec, ERR_WRTGAM);

	/* write XFCN-seek-location header if post 2.0.15 file version */
	if (filever[0] != 'a')
	{
	    oswp4(buf, 0);      /* placeholder - TADSRSC sets this up later */
	    if (osfwb(fp, buf, 4)) errsig(ec, ERR_WRTGAM);
	}

	/* close the resource */
	fiowcls(fp, ec, fpos);
    }
    
    /* write OBJ resource header */
    fpos = fiowhd(fp, ec, "\003OBJ");
    
    /* go through symbol table, and write each object */
    cbctx.fiowcxmem = mctx;
    cbctx.fiowcxerr = ec;
    cbctx.fiowcxfp  = fp;
    cbctx.fiowcxflg = flags;
    cbctx.fiowcxund = 0;
    cbctx.fiowcxseed = xor_seed;
    cbctx.fiowcxinc = xor_inc;
    if (flags & FIOFFAST)
	cbctx.fiowcxffp = osfoprwtb(FIOWRTFSTNM);
    else
	cbctx.fiowcxffp = (osfildef *)0;
    toktheach((toktdef *)tab, fiowrtobj, &cbctx);
                    
    /* close the resource */
    fiowcls(fp, ec, fpos);
    
    /* copy fast-load records to output file if applicable */
    if (cbctx.fiowcxffp)
    {
	osfildef *fp2 = cbctx.fiowcxffp;
	char      copybuf[1024];
	ulong     len;
	ulong     curlen;
	
	/* start with resource header for fast-load records */
	fpos = fiowhd(fp, ec, "\003FST");

	/* get length of temp file, then rewind it */
	len = osfpos(fp2);
	osfseek(fp2, 0L, OSFSK_SET);

	/* copy the whole thing to the output file */
	while (len)
	{
	    curlen = (len > sizeof(copybuf) ? sizeof(copybuf) : len);
	    if (osfrb(fp2, copybuf, (size_t)curlen)
		|| osfwb(fp, copybuf, (size_t)curlen))
		errsig(ec, ERR_WRTGAM);
	    len -= curlen;
	}

	/* close the fast-load resource */
	fiowcls(fp, ec, fpos);
	
	/* close and delete temp file */
	osfcls(fp2);
	osfdel(FIOWRTFSTNM);
    }
    
    /* write vocabulary inheritance records - start with header */
    fpos = fiowhd(fp, ec, "\003INH");
    
    /* go through inheritance records and write to file */
    for (vpg = vctx->voccxinh, i = 0 ; i < VOCINHMAX ; ++vpg, ++i)
    {
        if (!*vpg) continue;
        for (v = *vpg, obj = (i << 8), j = 0 ; j < 256 ; ++v, ++obj, ++j)
        {
            if (*v)
            {
                buf[0] = (*v)->vociflg;
                oswp2(buf + 1, obj);
                oswp2(buf + 3, (*v)->vociloc);
                oswp2(buf + 5, (*v)->vociilc);
                oswp2(buf + 7, (*v)->vocinsc);
                for (k = 0, sc = (*v)->vocisc ; k < (*v)->vocinsc ; ++sc, ++k)
                {
                    if (k + 9 >= sizeof(buf)) errsig(ec, ERR_FIOMSC);
                    oswp2(buf + 9 + (k << 1), (*sc));
                }
                if (osfwb(fp, buf, 9 + (2 * (*v)->vocinsc)))
                    errsig(ec, ERR_WRTGAM);
            }
        }
    }
    
    /* close resource */
    fiowcls(fp, ec, fpos);
    
    /* write format strings */
    if (fmtl)
    {
        fpos = fiowhd(fp, ec, "\006FMTSTR");
        oswp2(buf, fmtl);
        if (flags & FIOFCRYPT) fioxor(fmts, fmtl, xor_seed, xor_inc);
        if (osfwb(fp, buf, 2) || osfwb(fp, fmts, fmtl))
            errsig(ec, ERR_WRTGAM);
        fiowcls(fp, ec, fpos);
    }
    
    /* write preinit if preinit object was specified */
    if (flags & FIOFPRE)
    {
        fpos = fiowhd(fp, ec, "\007PREINIT");
        oswp2(buf, preinit);
        if (osfwb(fp, buf, 2)) errsig(ec, ERR_WRTGAM);
        fiowcls(fp, ec, fpos);
    }
    
    /* write required objects out of voccxdef */
    fpos = fiowhd(fp, ec, "\003REQ");
    fiowrq(ec, fp, vctx->voccxme);
    fiowrq(ec, fp, vctx->voccxvtk);
    fiowrq(ec, fp, vctx->voccxstr);
    fiowrq(ec, fp, vctx->voccxnum);
    fiowrq(ec, fp, vctx->voccxprd);
    fiowrq(ec, fp, vctx->voccxvag);
    fiowrq(ec, fp, vctx->voccxini);
    fiowrq(ec, fp, vctx->voccxpre);
    fiowrq(ec, fp, vctx->voccxper);
    fiowrq(ec, fp, vctx->voccxprom);
    fiowrq(ec, fp, vctx->voccxpdis);
    fiowrq(ec, fp, vctx->voccxper2);
    fiowrq(ec, fp, vctx->voccxpdef);
    fiowrq(ec, fp, vctx->voccxpask);
    fiowrq(ec, fp, vctx->voccxppc);
    fiowrq(ec, fp, vctx->voccxpask2);
    fiowcls(fp, ec, fpos);
    
    /* write compound words */
    if (vctx->voccxcpl)
    {
        fpos = fiowhd(fp, ec, "\004CMPD");
        oswp2(buf, vctx->voccxcpl);
        if (flags & FIOFCRYPT)
            fioxor(vctx->voccxcpp, (uint)vctx->voccxcpl, xor_seed, xor_inc);
        if (osfwb(fp, buf, 2)
            || osfwb(fp, vctx->voccxcpp, (uint)vctx->voccxcpl))
            errsig(ec, ERR_WRTGAM);
        fiowcls(fp, ec, fpos);
    }
    
    /* write special words */
    if (vctx->voccxspl)
    {
	fpos = fiowhd(fp, ec, "\010SPECWORD");
	oswp2(buf, vctx->voccxspl);
	if (flags & FIOFCRYPT)
	    fioxor(vctx->voccxspp, (uint)vctx->voccxspl, xor_seed, xor_inc);
	if (osfwb(fp, buf, 2)
	    || osfwb(fp, vctx->voccxspp, (uint)vctx->voccxspl))
	    errsig(ec, ERR_WRTGAM);
	fiowcls(fp, ec, fpos);
    }
    
    /* write vocabulary */
    fpos = fiowhd(fp, ec, "\003VOC");

    /* go through each hash chain */
    for (i = 0, vhsh = vctx->voccxhsh ; i < VOCHASHSIZ ; ++i, ++vhsh)
    {
	/* go through each word in this hash chain */
        for (voc = *vhsh ; voc ; voc = voc->vocnxt)
        {
	    /* encrypt the word if desired */
	    if (flags & FIOFCRYPT)
		fioxor(voc->voctxt, (uint)(voc->voclen + voc->vocln2),
		       xor_seed, xor_inc);

	    /* go through each object relation attached to the word */
	    for (vw = vocwget(vctx, voc->vocwlst) ; vw ;
		 vw = vocwget(vctx, vw->vocwnxt))
	    {
		oswp2(buf, voc->voclen);
		oswp2(buf+2, voc->vocln2);
		oswp2(buf+4, vw->vocwtyp);
		oswp2(buf+6, vw->vocwobj);
		oswp2(buf+8, vw->vocwflg);
		if (osfwb(fp, buf, 10)
		    || osfwb(fp, voc->voctxt, voc->voclen + voc->vocln2))
		    errsig(ec, ERR_WRTGAM);
	    }
        }
    }
    fiowcls(fp, ec, fpos);
    
    /* write the symbol table, if desired */
    if (flags & FIOFSYM)
    {
        fpos = fiowhd(fp, ec, "\006SYMTAB");
        toktheach((toktdef *)tab, fiowrtsym, &cbctx);

        /* indicate last symbol with a length of zero */
        buf[0] = 0;
        if (osfwb(fp, buf, 4)) errsig(ec, ERR_WRTGAM);
        fiowcls(fp, ec, fpos);
    }
    
    /* write line source chain, if desired */
    if ((flags & FIOFLIN) && vctx->voccxrun->runcxdbg)
    {
        lindef *lin;
        
        fpos = fiowhd(fp, ec, "\003SRC");
        
        for (lin = vctx->voccxrun->runcxdbg->dbgcxlin ; lin ;
             lin = lin->linnxt)
        {
            if (linwrt(lin, fp))
                errsig(ec, ERR_WRTGAM);
        }
        
        fiowcls(fp, ec, fpos);
    }

    /* if writing a precompiled header, write some more information */
    if (flags & FIOFBIN)
    {
	/* write property count */
        fpos = fiowhd(fp, ec, "\006PRPCNT");
        oswp2(buf, prpcnt);
        if (osfwb(fp, buf, 2)) errsig(ec, ERR_WRTGAM);
        fiowcls(fp, ec, fpos);

	/* write preprocessor symbol table */
	fpos = fiowhd(fp, ec, "\006TADSPP");
	tok_write_defines(tokctx, fp, ec);
	fiowcls(fp, ec, fpos);
    }

    /* write end-of-file resource header */
    (void)fiowhd(fp, ec, "\004$EOF");
    osfcls(fp);
    
    /* if there are undefined functions/objects, signal an error */
    if (cbctx.fiowcxund) errsig(ec, ERR_UNDEF);
}

/* write game to binary file */
void fiowrt(mctx, vctx, tokctx, tab, fmts, fmtl, fname, flags, preinit,
	    extc, prpcnt, filever)
mcmcxdef *mctx;
voccxdef *vctx;
tokcxdef *tokctx;
tokthdef *tab;
uchar    *fmts;
uint      fmtl;
char     *fname;
uint      flags;
objnum    preinit;
int       extc;
uint      prpcnt;
char     *filever;
{
    osfildef *fp;
    
    /* open the file */
    if (!(fp = osfoprwtb(fname))) errsig(vctx->voccxerr, ERR_OPWGAM);
    
    ERRBEGIN(vctx->voccxerr)
    
    /* write the file */
    fiowrt1(mctx, vctx, tokctx, tab, fmts, fmtl, fp, flags, preinit,
	    extc, prpcnt, filever);
    os_settype(fname, OSFTGAME);
   
    ERRCLEAN(vctx->voccxerr)
        /* clean up by closing the file */
        osfcls(fp);
    ERRENDCLN(vctx->voccxerr)
}
