/*
 * table.c
 *
 * Table handling opcodes
 *
 */

#include <math.h>
#include "frotz.h"

/*
 * z_copy_table
 *
 * Copy a table or fill it with zeroes. There are three cases:
 *
 * 1) Destination is 0: Source is filled with zeroes.
 * 2) Size is positive: Source is copied to destination in either
 * forward or backward fashion. Both tables may overlap ("memmove").
 * 3) Size is negative: Source is copied forwards to destination.
 *
 */

void z_copy_table (zword first, zword second, zword size)
{
    zword addr;
    zbyte value;
    int i;

    if (second == 0)

	/* Fill source with zeros */

	for (i = 0; i < size; i++)
	    z_storeb (first, i, 0);

    else if ((short) size < 0 || first > second)

	/* Copy forwards */

	for (i = 0; i < abs ((short) size); i++) {
	    addr = first + i;
	    LOW_BYTE (addr, value)
	    z_storeb (second, i, value);
	}

    else

	/* Copy backwards */

	for (i = size - 1; i >= 0; i--) {
	    addr = first + i;
	    LOW_BYTE (addr, value)
	    z_storeb (second, i, value);
	}

}/* z_copy_table */

/*
 * z_loadb
 *
 * Store a byte from a table of bytes.
 *
 */

void z_loadb (zword array, zword byte_index)
{
    zbyte value;

    array += byte_index;
    LOW_BYTE (array, value)

    store (value);

}/* z_loadb */

/*
 * z_loadw
 *
 * Store a word from a table of words.
 *
 */

void z_loadw (zword array, zword word_index)
{
    zword value;

    array += 2 * word_index;
    LOW_WORD (array, value)

    store (value);

}/* z_loadw */

/*
 * z_scan_table
 *
 * Scan a table of bytes or words looking for a target byte or word.
 * The optional 4th parameter gives the address step (bottom 7 bits)
 * and the type of the table (top bit: 1 for word, 0 for byte table).
 * (The default is word table and address step 2.) If the search is
 * successful, the address of the first match in the table is stored
 * and a branch is made. Otherwise, 0 is stored.
 *
 */

void z_scan_table (int argc, zword x, zword table, zword len, zword form)
{
    zword wvalue;
    zbyte bvalue;
    int i;

    /* Supply default arguments */

    if (argc < 4)
	form = 0x82;

    /* Scan byte or word array */

    for (i = 0; i < len; i++) {

	if (form & 0x80) { /* Scan word array */

	    LOW_WORD (table, wvalue)

	    if (wvalue == x)
		goto finished;

	} else { /* Scan byte array */

	    LOW_BYTE (table, bvalue)

	    if (bvalue == x)
		goto finished;
	}

	table += form & 0x7f;
    }

    table = 0;

finished:

    store (table);
    branch (table);

}/* z_scan_table */

/*
 * z_storeb
 *
 * Write a byte into a table of bytes.
 *
 */

void z_storeb (zword array, zword byte_index, zword value)
{

    array += byte_index;

    /* Check if this toggles the scripting or fixed font flags */

    if (array == H_FLAGS + 1) {

	h_flags &= ~(SCRIPTING_FLAG | FIXED_FONT_FLAG);
	h_flags |= value & (SCRIPTING_FLAG | FIXED_FONT_FLAG);

	z_output_stream (1, (value & SCRIPTING_FLAG) ? 2 : -2, 0, 0);
	check_fixed_font ();
    }

    /* Are we writing outside of the dynamic memory? */

    if (array >= h_dynamic_size)
	os_fatal ("Store out of dynamic memory");

    /* Store the byte */

    SET_BYTE (array, value)

}/* z_storeb */

/*
 * z_storew
 *
 * Write a word into a table of words.
 *
 */

void z_storew (zword array, zword word_index, zword value)
{

    array += 2 * word_index;

    z_storeb (array, 0, hi (value));
    z_storeb (array, 1, lo (value));

}/* z_storew */
