/*
 * screen.c
 *
 * Generic screen manipulation
 *
 */

#include "frotz.h"

extern int replaying;

static int font_height = 1;
static int font_width = 1;

static struct {
    zword y_pos;
    zword x_pos;
    zword y_size;
    zword x_size;
    zword y_cursor;
    zword x_cursor;
    zword left;
    zword right;
    zword nl_routine;
    zword nl_countdown;
    zword style;
    zword colour;
    zword font;
    zword font_size;
    zword attribute;
    zword line_count;
} wp[8];

/*
 * check_fixed_font
 *
 * Set the fixed width style when the fixed font flag is set or
 * when the upper window is selected. This does not apply in V6.
 *
 */

void check_fixed_font (void)
{

    if (h_version == V6)
	return;

    set_text_style (wp[0].style);

}/* check_fixed_font */

/*
 * update_cursor
 *
 * Move the hardware cursor to make it match the current window
 * properties.
 *
 */

static void update_cursor (void)
{
    int y, x;

    y = wp[cwin].y_pos + wp[cwin].y_cursor - 1;
    x = wp[cwin].x_pos + wp[cwin].x_cursor - 1;

    os_set_cursor (y, x);

}/* update_cursor */

/*
 * reset_cursor
 *
 * Small helper function that resets the cursor for a given window
 * to its initial position.
 *
 */

static void reset_cursor (zword window)
{

    wp[window].y_cursor = 1;
    wp[window].x_cursor = wp[window].left + 1;

    if (h_version <= V4 && window == 0)
	wp[0].y_cursor = (wp[0].y_size / font_height - 1) * font_height + 1;

    if (window == cwin)
	update_cursor ();

}/* reset_cursor */

/*
 * display_char
 *
 * Display a character on the screen.
 *
 */

void display_char (int c)
{
    int letter_width;

    letter_width = os_char_width (c);

    /* When the character doesn't fit the current line, then either
       insert a newline (if wrapping is on) or discard the character
       altogether (clipping). */

    if (wp[cwin].x_cursor + letter_width - 1 > wp[cwin].x_size - wp[cwin].right)

	if (enable_wrapping)
	    new_line ();
	else
	    return;

    os_display_char (c);

    /* Adjust cursor position */

    wp[cwin].x_cursor += letter_width;

}/* display_char */

/*
 * display_new_line
 *
 * Print a newline to the screen.
 *
 */

void display_new_line (void)
{

    /* When the cursor has not already reached the bottom line, then
       it is simply moved to the start of the next line. Otherwise
       the lower window is scrolled up one line (if scrolling is on)
       or the cursor is reset to the top left. */

    if (wp[cwin].y_cursor + 2 * font_height - 1 > wp[cwin].y_size)

	if (enable_scrolling)
	    z_scroll_window (cwin, font_height);
	else
	    wp[cwin].y_cursor = 1;

    else wp[cwin].y_cursor += font_height;

    wp[cwin].x_cursor = wp[cwin].left + 1;

    update_cursor ();

    /* See if we have filled the screen. The spare line is for the
       MORE prompt. If a MORE has been displayed, the line counter
       must be reset. Note that the line counter is supposed to be
       a signed value; the game may set the line counter to -999 in
       order to suppress MORE prompts for a while. */

    if (enable_scrolling) {

	wp[cwin].line_count++;

	if ((short) wp[cwin].line_count >= (short) wp[cwin].y_size / font_height - 1) {

	    if (!replaying)
		os_more_prompt ();

	    wp[cwin].line_count = option_context_lines;
	}
    }

    /* Handle newline interrupts. If the newline countdown of the
       current window is set to a non-zero value then decrement it.
       Call the interrupt function when we hit zero. */

    if (wp[cwin].nl_countdown != 0 && --wp[cwin].nl_countdown == 0)
	call_interrupt (wp[cwin].nl_routine);

}/* display_new_line */

/*
 * display_string
 *
 * Print a string of characters to the screen.
 *
 */

void display_string (const char *s)
{
    int i;

    for (i = 0; s[i] != 0; i++)

	if (s[i] != '\n')
	    display_char ((unsigned char) s[i]);
	else
	    display_new_line ();

}/* display_string */

/*
 * display_rewind
 *
 * Remove a string that has already been printed from the screen
 * and place the cursor at the start of the string.
 *
 */

void display_rewind (const char *s)
{
    int word_len;

    if (s[0] == 0)
	return;

    word_len = os_string_width (s);

    wp[cwin].x_cursor -= word_len;

    update_cursor ();

    z_erase_line (word_len + 1);

}/* display_rewind */

/*
 * font_available
 *
 * Return the current font number if the given font is available,
 * or return 0 if it is not.
 *
 */

zword font_available (zword font)
{
    int window = 0;

    if (h_version == V6)
	window = cwin;

    if (font != 0 && !os_font_available (font))
	return 0;

    return wp[window].font;

}/* font_available */

/*
 * get_line_width
 *
 * Return the total space on the current line of the given window.
 *
 */

zword get_line_width (zword window)
{

    return wp[window].x_size - wp[window].left - wp[window].right;

}/* get_line_width */

/*
 * get_units_left
 *
 * Return the number of screen units between the cursor and the right
 * margin of the current window.
 *
 */

zword get_units_left (void)
{

    return wp[cwin].x_size - wp[cwin].right - wp[cwin].x_cursor + 1;

}/* get_units_left */

/*
 * reset_line_count
 *
 * Reset the line counter after an input action took place.
 *
 */

void reset_line_count (void)
{

    wp[cwin].line_count = 0;

}/* reset_line_count */

/*
 * shift_cursor
 *
 * Adjust the x coordinate of the cursor in the current window
 * properties. This is used to adjust the coordinate after the
 * input routine has been called.
 *
 */

void shift_cursor (int units)
{

    wp[cwin].x_cursor += units;

}/* shift_cursor */

#ifdef AMIGA

/*
 * resize_screen
 *
 * Try to adapt the window properties to a new screen size.
 *
 */

void resize_screen (int height, int width)
{
    int i;

    for (i = 0; i < 8; i++) {

	if (wp[i].y_pos > height)
	    wp[i].y_pos = height;
	if (wp[i].x_pos > width)
	    wp[i].x_pos = width;
	if (wp[i].y_pos + wp[i].y_size - 1 > height)
	    wp[i].y_size = height - wp[i].y_pos + 1;
	if (wp[i].x_pos + wp[i].x_size - 1 > width)
	    wp[i].x_size = width - wp[i].x_pos + 1;
    }

    h_screen_height = height;
    h_screen_width = width;
    h_screen_rows = height / h_font_height;
    h_screen_cols = width / h_font_width;

    if (h_version == V6)
	h_flags |= REFRESH_FLAG;

    restart_header ();

    if (h_version == V6)
	h_flags &= ~REFRESH_FLAG;

    if (h_version <= V3)
	z_show_status ();

}/* resize_screen */

#endif

/*
 * restart_screen
 *
 * Prepare the screen for a new game.
 *
 */

void restart_screen (void)
{
    int i;

#ifdef AMIGA
    CheckReset ();
#endif

    /* Standard colours, font and text style */

    os_set_font (TEXT_FONT, &font_height, &font_width);
    os_set_colour (DEFAULT_COLOUR, DEFAULT_COLOUR);
    os_set_text_style (0);

    os_cursor_on ();

    /* Initialise window properties */

    for (i = 0; i < 8; i++) {
	wp[i].y_pos = 1;
	wp[i].x_pos = 1;
	wp[i].y_size = 0;
	wp[i].x_size = 0;
	wp[i].y_cursor = 1;
	wp[i].x_cursor = 1;
	wp[i].left = 0;
	wp[i].right = 0;
	wp[i].nl_routine = 0;
	wp[i].nl_countdown = 0;
	wp[i].style = 0;
	wp[i].colour = (h_default_background << 8) | h_default_foreground;
	wp[i].font = TEXT_FONT;
	wp[i].font_size = (font_height << 8) | font_width;
	wp[i].attribute = BUFFERING_ATTR;
    }

    /* Prepare lower window */

    wp[0].x_size = h_screen_width;

    wp[0].left = option_left_margin;
    wp[0].right = option_right_margin;

    wp[0].attribute = WRAPPING_ATTR | SCROLLING_ATTR | SCRIPTING_ATTR | BUFFERING_ATTR;

    /* Prepare upper window */

    wp[1].x_size = h_screen_width;

    if (h_version != V6)
	wp[1].attribute = 0;

    /* Prepare status line */

    if (h_version <= V3) {
	wp[7].x_size = h_screen_width;
	wp[7].attribute = 0;
    }

    /* Clear the screen, unsplit it and select window 0 */

    z_erase_window (-1);
    z_set_window (0);

}/* restart_screen */

/*
 * set_font
 *
 * Set the font (for the current window in V6, for all windows
 * otherwise).
 *
 */

void set_font (zword font)
{
    int window = 0;

    if (h_version == V6)
	window = cwin;

    wp[window].font = font;

    os_set_font (font, &font_height, &font_width);

    wp[window].font_size = (font_height << 8) | font_width;

}/* set_font */

/*
 * set_text_style
 *
 * Set the text style (for the current window in V6, for all windows
 * otherwise).
 *
 */

void set_text_style (zword style)
{
    int window = 0;

    if (h_version == V6)
	window = cwin;

    /* Add old style flags to new style flags */

    if (style != 0)
	style |= wp[window].style;

    wp[window].style = style;

    /* Set the fixed width style when the fixed font flag is set, or
       when the upper window is selected in a pre-V6 game */

    if (h_version != V6 && (cwin != 0 || (h_flags & FIXED_FONT_FLAG)))
	style |= FIXED_WIDTH_STYLE;

    os_set_text_style (style);

}/* set_text_style */

/*
 * z_buffer_mode
 *
 * Toggle buffering on/off (for the current window in V6, for the
 * lower window otherwise).
 *
 */

void z_buffer_mode (zword flag)
{
    zword window = 0;

    if (h_version == V6)
	window = cwin;

    flush_buffer ();

    if (flag != 0)
	wp[window].attribute |= BUFFERING_ATTR;
    else
	wp[window].attribute &= ~BUFFERING_ATTR;

    if (window == cwin)
	enable_buffering = flag;

}/* z_buffer_mode */

/*
 * z_draw_picture
 *
 * Display a picture at the given coordinates.
 *
 */

void z_draw_picture (zword picture_number, zword y, zword x)
{

    flush_buffer ();

    /* If the given coordinate is 0 use the cursor coordinate instead */

    if (x == 0)
	x = wp[cwin].x_cursor;
    if (y == 0)
	y = wp[cwin].y_cursor;

    /* Draw picture */

    os_draw_picture (picture_number, wp[cwin].y_pos + y - 1, wp[cwin].x_pos + x - 1);

}/* z_draw_picture */

/*
 * z_erase_line
 *
 * Clear (pixels - 1) screen units starting at the current cursor
 * position, or clear the whole line if pixels is 1.
 *
 */

void z_erase_line (zword pixels)
{
    int y, x;

    flush_buffer ();

    /* Clipping at the right border of the current window */

    pixels--;

    if (pixels == 0 || pixels > wp[cwin].x_size - wp[cwin].x_cursor)
	pixels = wp[cwin].x_size - wp[cwin].x_cursor + 1;

    /* Erase from cursor position */

    y = wp[cwin].y_pos + wp[cwin].y_cursor - 1;
    x = wp[cwin].x_pos + wp[cwin].x_cursor - 1;

    os_erase_area (y,
		   x,
		   y + font_height - 1,
		   x + pixels - 1);

}/* z_erase_line */

/*
 * z_erase_picture
 *
 * Erase a picture at the given coordinates.
 *
 */

void z_erase_picture (zword picture_number, zword y, zword x)
{
    int height, width;

    flush_buffer ();

    /* If the given coordinate is 0 use the cursor coordinate instead */

    if (x == 0)
	x = wp[cwin].x_cursor;
    if (y == 0)
	y = wp[cwin].y_cursor;

    /* Erase picture to the background colour */

    os_picture_data (picture_number, &height, &width);

    y += wp[cwin].y_pos - 1;
    x += wp[cwin].x_pos - 1;

    os_erase_area (y,
		   x,
		   y + height - 1,
		   x + width - 1);

}/* erase_picture */

/*
 * z_erase_window
 *
 * Clear one or all windows on the screen.
 *
 */

void z_erase_window (zword window)
{
    int y, x;
    int i;

    flush_buffer ();

    /* Window -3 is the current window */

    if ((short) window == -3)
	window = cwin;

    /* Clear the screen or a window */

    if ((short) window == -2) {

	os_erase_area (1,
		       1,
		       h_screen_height,
		       h_screen_width);

	for (i = 0; i < 8; i++)
	    wp[i].line_count = 0;

    } else if ((short) window == -1) {

	z_mouse_window (-1);
	z_erase_window (-2);
	z_split_window (0);

	reset_cursor (0);

    } else {

	if (h_version == V6 && window != cwin)
	    os_set_colour (lo (wp[window].colour), hi (wp[window].colour));

	y = wp[window].y_pos;
	x = wp[window].x_pos;

	os_erase_area (y,
		       x,
		       y + wp[window].y_size - 1,
		       x + wp[window].x_size - 1);

	if (h_version == V6 && window != cwin)
	    os_set_colour (lo (wp[cwin].colour), hi (wp[cwin].colour));

	wp[window].line_count = 0;

	reset_cursor (window);
    }

}/* z_erase_window */

/*
 * z_get_cursor
 *
 * Write the cursor coordinates into a table. In V5+ the coordinates
 * should refer to character grid positions unless the graphics flag
 * is set.
 *
 */

void z_get_cursor (zword table)
{
    zword line, column;

    flush_buffer ();

    /* Get cursor position */

    line = wp[cwin].y_cursor;
    column = wp[cwin].x_cursor;

    /* Convert screen units to grid positions if necessary */

    if (h_version != V6 && !(h_flags & GRAPHICS_FLAG)) {
	line = (line - 1) / font_height + 1;
	column = (column - 1) / font_width + 1;
    }

    /* Store the cursor position */

    z_storew (table, 0, line);
    z_storew (table, 1, column);

}/* z_get_cursor */

/*
 * z_get_wind_prop
 *
 * Store the value of a window property.
 *
 */

void z_get_wind_prop (zword window, zword property_number)
{

    flush_buffer ();

    /* Window 3 is the current window */

    if ((short) window == -3)
	window = cwin;

    /* Return window property to Z-machine */

    store (((zword *) (wp + window)) [property_number]);

}/* z_get_wind_prop */

/*
 * z_mouse_window
 *
 * Restrict mouse cursor movement to the given window or remove all
 * restrictions (window == -1).
 *
 */

void z_mouse_window (zword window)
{
    int y, x;

    /* Window 3 is the current window */

    if ((short) window == -3)
	window = cwin;

    /* Set mouse window */

    if ((short) window == -1)

	os_mouse_area (1,
		       1,
		       h_screen_height,
		       h_screen_width);

    else {

	y = wp[window].y_pos;
	x = wp[window].x_pos;

	os_mouse_area (y,
		       x,
		       y + wp[window].y_size - 1,
		       x + wp[window].x_size - 1);
    }

}/* z_mouse_window */

/*
 * z_move_window
 *
 * Set the top left coordinates of the given window.
 *
 */

void z_move_window (zword window, zword y, zword x)
{

    flush_buffer ();

    /* Window 3 is the current window */

    if ((short) window == -3)
	window = cwin;

    /* Move window (and the cursor) */

    wp[window].y_pos = y;
    wp[window].x_pos = x;

    if (window == cwin)
	update_cursor ();

}/* z_move_window */

/*
 * z_picture_data
 *
 * Write the width and height of the given picture into a table; or, if
 * the picture number is zero, write the highest legal picture number
 * and the picture file version into the table. Branch if successful.
 *
 */

void z_picture_data (zword picture_number, zword table)
{
    int available;
    int height, width;

    available = os_picture_data (picture_number, &height, &width);

    z_storew (table, 0, height);
    z_storew (table, 1, width);

    branch (available);

}/* z_picture_data */

/*
 * z_picture_table
 *
 * The given table holds numbers of pictures which should be loaded to
 * memory for faster display.
 *
 */

void z_picture_table (zword table)
{

    /* Not implemented */

}/* z_picture_table */

/*
 * z_print_table
 *
 * Write text into a rectangular window.
 *
 */

void z_print_table (int argc, zword ascii_text, zword width, zword height, zword skip)
{
    zword x_cursor;
    zbyte c;
    int first;
    int i, j;

    /* Supply default arguments */

    if (argc < 3)
	height = 1;
    if (argc < 4)
	skip = 0;

    /* Get coordinates of top left corner of rectangle */

    x_cursor = wp[cwin].x_cursor;

    /* Write text in width x height rectangle */

    first = 1;

    for (i = 0; i < height; i++) {

	flush_buffer ();

	if (first == 0) {

	    wp[cwin].y_cursor += font_height;
	    wp[cwin].x_cursor = x_cursor;

	    update_cursor ();

	    ascii_text += skip;

	} else first = 0;

	for (j = 0; j < width; j++) {

	    LOW_BYTE (ascii_text, c)
	    ascii_text++;

	    z_print_char (c);
	}
    }

}/* z_print_table */

/*
 * z_put_wind_prop
 *
 * Set the value of a window property.
 *
 */

void z_put_wind_prop (zword window, zword property_number, zword value)
{

    flush_buffer ();

    /* Window 3 is the current window */

    if ((short) window == -3)
	window = cwin;

    /* Set property */

    ((zword *) (wp + window)) [property_number] = value;

}/* z_put_wind_prop */

/*
 * z_scroll_window
 *
 * Scroll the given window up or down.
 *
 */

void z_scroll_window (zword window, zword pixels)
{
    int y, x;

    flush_buffer ();

    /* Window -3 is the current window */

    if ((short) window == -3)
	window = cwin;

    if (h_version == V6 && window != cwin)
	os_set_colour (lo (wp[window].colour), hi (wp[window].colour));

    y = wp[window].y_pos;
    x = wp[window].x_pos;

    os_scroll_area (y,
		    x,
		    y + wp[window].y_size - 1,
		    x + wp[window].x_size - 1,
		    (short) pixels);

    if (h_version == V6 && window != cwin)
	os_set_colour (lo (wp[cwin].colour), hi (wp[cwin].colour));

}/* z_scroll_window */

/*
 * z_set_colour
 *
 * Set the colour for text output which can be:
 *
 *    0 = current colour
 *    1 = machine default
 *    2 = black
 *    3 = red
 *    4 = green
 *    5 = brown
 *    6 = blue
 *    7 = magenta
 *    8 = cyan
 *    9 = white
 *
 */

void z_set_colour (zword foreground, zword background)
{
    int window = 0;

    if (h_version == V6)
	window = cwin;

    flush_buffer ();

    /* Handle colour -1 */

    if ((short) foreground == -1)
	foreground = os_peek_colour ();
    if ((short) background == -1)
	background = os_peek_colour ();

    /* Handle colour 0 */

    if (foreground == 0)
	foreground = lo (wp[window].colour);
    if (background == 0)
	background = hi (wp[window].colour);

    wp[window].colour = (background << 8) | foreground;

    os_set_colour (foreground, background);

}/* z_set_colour */

/*
 * z_set_cursor
 *
 * Set the cursor position. The game expects either character grid
 * positions (V1 to V5) or screen units (V6). This is not the full
 * story since V5 games can also set the graphics flag to indicate
 * use of screen units ('Beyond Zork').
 *
 * V6 games also use "set_cursor -2" and "set_cursor -1" to turn the
 * cursor on and off. In any case, the cursor need only be visible
 * during input actions such as read and read_char.
 *
 */

void z_set_cursor (int argc, zword line, zword column, zword window)
{

    flush_buffer ();

    /* Supply default arguments */

    if (argc < 3)
	window = cwin;

    /* Handle cursor on/off */

    if ((short) line == -2) {
	os_cursor_on ();
	return;
    }
    if ((short) line == -1) {
	os_cursor_off ();
	return;
    }

    /* Window 3 is the current window */

    if ((short) window == -3)
	window = cwin;

    /* In V1 to V5, cursor movement is restricted to the upper window */

    if (h_version != V6)
	window = 1;

    /* The German translation of Zork contains some unfinished support
       for screen units which causes trouble when the font dimensions
       are different from 1x1 */

    if (german_zork_flag != 0) {
	line /= font_height;
	column /= font_width;
    }

    /* Convert grid positions to screen units if necessary */

    if (h_version != V6 && !(h_flags & GRAPHICS_FLAG)) {
	line = (line - 1) * font_height + 1;
	column = (column - 1) * font_width + 1;
    }

    /* Protect the left margin */

    if (column <= wp[window].left)
	column = wp[window].left;

    /* Move the cursor */

    wp[window].y_cursor = line;
    wp[window].x_cursor = column;

    if (window == cwin)
	update_cursor ();

}/* z_set_cursor */

/*
 * z_set_margins
 *
 * Set the left and right margins of the given window.
 *
 */

void z_set_margins (int argc, zword left, zword right, zword window)
{

    flush_buffer ();

    /* Supply default arguments */

    if (argc < 3)
	window = cwin;

    /* Window -3 is the current window */

    if ((short) window == -3)
	window = cwin;

    /* Adjust margins */

    wp[window].left = left;
    wp[window].right = right;

    /* Push the cursor to the right of the left margin */

    if (wp[window].x_cursor <= left)
	wp[window].x_cursor = left + 1;

    if (window == cwin)
	update_cursor ();

}/* z_set_margins */

/*
 * z_set_window
 *
 * Select the current window. Except for V6, there are only two
 * windows, the lower and the upper window. Activating the upper
 * one moves the cursor to its top left. Activating the lower one
 * returns the cursor to its previous position. In V6, there are
 * up to eight windows; each window remembers its cursor position.
 *
 */

void z_set_window (zword window)
{
    zword attribute;

    flush_buffer ();

    /* Window 3 is the current window */

    if ((short) window == -3)
	window = cwin;

    /* Select new window */

    cwin = window;

    if (h_version == V6) {

	os_set_font (wp[window].font, &font_height, &font_width);
	os_set_colour (lo (wp[window].colour), hi (wp[window].colour));
	os_set_text_style (wp[window].style);

    } else check_fixed_font ();

    /* Before V6, only window 0 remembers its cursor position */

    if (h_version != V6 && window != 0)
	reset_cursor (window);
    else
	update_cursor ();

    /* Set window attributes */

    attribute = wp[window].attribute;

    enable_wrapping = attribute & WRAPPING_ATTR;
    enable_scripting = attribute & SCRIPTING_ATTR;
    enable_scrolling = attribute & SCROLLING_ATTR;
    enable_buffering = attribute & BUFFERING_ATTR;

}/* z_set_window */

/*
 * pad_line
 *
 * Pad the status line with spaces up to the given position.
 *
 */

static void pad_line (int column)
{
    int i;

    i = (wp[7].x_size - wp[7].x_cursor + 1) / font_width - column;

    while (--i >= 0)
	display_char (' ');

}/* pad_line */

/*
 * z_show_status
 *
 * Output the status line for V1 to V3 games.
 *
 */

void z_show_status (void)
{
    zword global0;
    zword global1;
    zword global2;
    zword addr;
    zword hours;
    int brief;

    /* One V5 game (Wishbringer Solid Gold) contains this opcode by
       accident, so just return if the version number does not fit */

    if (h_version >= V4)
	return;

    /* Read all relevant global variables from the memory of the
       Z-machine into local variables */

    addr = h_globals;
    LOW_WORD (addr, global0)
    addr += 2;
    LOW_WORD (addr, global1)
    addr += 2;
    LOW_WORD (addr, global2)

    /* Frotz uses window 7 for the status line */

    z_set_window (7);

    set_text_style (REVERSE_STYLE);

    /* If the screen width is below 55 characters then we have to use
       the brief status line format */

    brief = (h_screen_cols < 55);

    /* Print the object description for the global variable 0 */

    display_char (' ');
    z_print_obj (global0);

    /* A header flag tells us whether we have to display the current
       time or the score/moves information */

    if (h_config & CONFIG_TIME) {

	/* Print hours and minutes from global variables 1 and 2 */

	pad_line (brief ? 15 : 20);

	display_string ("Time: ");

	hours = (global1 + 11) % 12 + 1;
	display_char (hours < 10 ? ' ' : '1');
	display_char ('0' + hours % 10);
	display_char (':');

	display_char ('0' + global2 / 10);
	display_char ('0' + global2 % 10);
	display_char (' ');

	display_char (global1 >= 12 ? 'p' : 'a');
	display_char ('m');

    } else {

	/* Print score and moves from global variables 1 and 2 */

	pad_line (brief ? 15 : 30);

	display_string (brief ? "S: " : "Score: ");
	z_print_num (global1);

	pad_line (brief ? 8 : 14);

	display_string (brief ? "M: " : "Moves: ");
	z_print_num (global2);
    }

    /* Pad the end of status line with spaces */

    pad_line (0);

    /* Return to the lower window */

    z_set_window (0);

    set_text_style (0);

}/* z_show_status */

/*
 * z_split_window
 *
 * Set the size of the upper window. The upper window appears at the
 * top of the screen (below the status line in V3 games -- Frotz
 * implements the status line as window number 7). The upper window
 * overlays the contents of the lower window (except in V3); the upper
 * window always uses a fixed width font; buffering is turned off in
 * the upper window.
 *
 */

void z_split_window (zword lines)
{
    zword status;

    flush_buffer ();

    /* Convert lines to screen units */

    if (h_version != V6)
	lines *= font_height;

    /* Take care of the status line in V3 */

    if (h_version <= V3)
	status = font_height;
    else
	status = 0;

    /* Cursor of upper window does not move unless it would be swallowed
       by the lower window */

    wp[1].y_cursor += wp[1].y_pos - 1 - status;

    wp[1].y_pos = 1 + status;
    wp[1].y_size = lines;

    if ((short) wp[1].y_cursor > wp[1].y_size)
	reset_cursor (1);

    /* Cursor of lower window does not move unless it would be swallowed
       by the upper window */

    wp[0].y_cursor += wp[0].y_pos - 1 - status - lines;

    wp[0].y_pos = 1 + status + lines;
    wp[0].y_size = h_screen_height - status - lines;

    if ((short) wp[0].y_cursor < 1)
	reset_cursor (0);

    /* Erase the upper window in V3 only */

    if (h_version == V3 && lines != 0)
	z_erase_window (1);

}/* z_split_window */

/*
 * z_window_size
 *
 * Set the width and height of the given window.
 *
 */

void z_window_size (zword window, zword y, zword x)
{

    flush_buffer ();

    /* Window -3 is the current window */

    if ((short) window == -3)
	window = cwin;

    /* Resize the window */

    wp[window].y_size = y;
    wp[window].x_size = x;

    /* Keep the cursor within the window */

    if (wp[window].y_cursor > y || wp[window].x_cursor > x)
	reset_cursor (window);

}/* z_window_size */

/*
 * z_window_style
 *
 * Change the attribute flags of a window. Operation can be:
 *
 *     0 = replace old flags by new flags
 *     1 = add new flags to old flags
 *     2 = clear new flags
 *     3 = reverse new flags
 *
 * Following attribute flags are used:
 *
 *     1 = character wrapping (instead of clipping)
 *     2 = transcription
 *     4 = scrolling
 *     8 = text buffering and word wrapping
 *
 */

void z_window_style (int argc, zword window, zword flags, zword operation)
{
    zword attribute;

    flush_buffer ();

    /* Supply default arguments */

    if (argc < 3)
	operation = 0;

    /* Window -3 is the current window */

    if ((short) window == -3)
	window = cwin;

    /* Set window style */

    attribute = wp[window].attribute;

    if (operation == 0)
	attribute = flags;
    if (operation == 1)
	attribute |= flags;
    if (operation == 2)
	attribute &= ~flags;
    if (operation == 3)
	attribute ^= flags;

    wp[window].attribute = attribute;

    /* Set window attributes */

    enable_wrapping = attribute & WRAPPING_ATTR;
    enable_scripting = attribute & SCRIPTING_ATTR;
    enable_scrolling = attribute & SCROLLING_ATTR;
    enable_buffering = attribute & BUFFERING_ATTR;

}/* z_window_style */
