/*
 * buffer.c
 *
 * Text buffering and word wrapping
 *
 */

#include "frotz.h"

#define TEXT_BUFFER_SIZE 120

extern zword script_width;
extern zword redirect_size;

extern int stream_memory;
extern int stream_display;
extern int stream_script;
extern int message;

static int locked = 0;

static char buffer[TEXT_BUFFER_SIZE];
static int buffer_pos = 0;

/*
 * add_to_buffer
 *
 * Insert a character into the output buffer.
 *
 */

static void add_to_buffer (int c)
{

    buffer[buffer_pos++] = c;

    if (buffer_pos == TEXT_BUFFER_SIZE)
	os_fatal ("Text buffer overflow");

}/* add_to_buffer */

/*
 * flush_buffer
 *
 * Copy the contents of the output buffer to the output streams.
 *
 */

void flush_buffer (void)
{
    int left;
    int word_width;
    int code;
    int arg;
    int i;

    /* Return if buffer empty */

    if (buffer_pos == 0)
	return;

    /* Make sure we stop when flush_buffer is called from flush_buffer.
       Note that this is difficult to avoid as we might print a newline
       during flush_buffer, which might cause a newline interrupt, that
       might execute any arbitrary opcode, which might flush the buffer. */

    if (locked != 0)
	return;

    locked = 1;

    /* Get remaining free screen units on current line */

    if (stream_memory)
	left = redirect_size - h_line_width;
    else if (stream_display)
	left = get_units_left ();
    else if (stream_script)
	left = get_line_width (cwin) - script_width;
    else /* all streams closed */
	left = 9999;

    /* Get length of new word */

    buffer[buffer_pos] = 0;

    word_width = os_string_width (buffer);

    /* Print a newline if the word does not fit */

    i = 0;

    if (enable_buffering && word_width > left) {

	if (buffer[0] == ' ' || buffer[0] == 11)
	    i = 1;

	new_line ();
    }

    /* Print the buffer */

    while (i < buffer_pos) {

	code = (unsigned char) buffer[i++];

	if (code == NEW_STYLE || code == NEW_FONT) {

	    arg = (unsigned char) buffer[i++];

	    if (code == NEW_STYLE)
		set_text_style (arg);
	    if (code == NEW_FONT)
		set_font (arg);

	} else print_char (code);

    }

    /* Empty buffer */

    buffer_pos = 0;

    /* Remove lock */

    locked = 0;

}/* flush_buffer */

/*
 * z_new_line
 *
 * High level newline function.
 *
 */

void z_new_line (void)
{

    flush_buffer ();

    /* Pass newline to the output streams */

    new_line ();

}/* z_new_line */

/*
 * z_print_char
 *
 * High level character output function.
 *
 */

void z_print_char (zword code)
{

    /* ASCII value 0 prints nothing */

    if (code == 0)
	return;

    /* ASCII value 13 is newline */

    if (code == 13) {
	z_new_line ();
	return;
    }

    /* ASCII values from 127 to 251 (except for the accented European
       characters) are undefined */

    if (code >= 127 && code <= 251 && (code < EURO_MIN || code > EURO_MAX))
	code = '?';

    /* Check if buffering is on */

    if (enable_buffering) {

	/* Flush the buffer if this is a space */

	if (code == ' ' || code == 11)
	    flush_buffer ();

	/* Add character to the buffer */

	add_to_buffer (code);

	/* Flush the buffer if this is a hyphen */

	if (code == '-')
	    flush_buffer ();

    } else print_char (code);

}/* z_print_char */

/*
 * z_set_font
 *
 * Set font for text output which can be:
 *
 *    0 = current font
 *    1 = text font
 *    2 = picture font (never used)
 *    3 = graphics font
 *    4 = fixed width font
 *
 * If the font isn't available return 0, otherwise return the number
 * of the current font. The specification demands that fonts may be
 * changed in the middle of a word, so font changes must be buffered
 * as well.
 *
 */

void z_set_font (zword font)
{
    int available;

    /* See if the font is available */

    available = font_available (font);

    store (available);

    if (available == 0 || font == 0)
	return;

    /* Set the new font */

    if (enable_buffering && buffer_pos != 0) {

	add_to_buffer (NEW_FONT);
	add_to_buffer (font);

    } else set_font (font);

}/* z_set_font */

/*
 * z_set_text_style
 *
 * Set the text style which can be
 *
 *    0 = reset to roman style (plain style)
 *    1 = reverse mode
 *    2 = boldface
 *    4 = emphasis (italics or underlining)
 *    8 = fixed width
 *
 * Since the text style may change in the middle of a word, it must be
 * buffered like text.
 *
 */

void z_set_text_style (zword style)
{

    if (enable_buffering && buffer_pos != 0) {

	add_to_buffer (NEW_STYLE);
	add_to_buffer (style);

    } else set_text_style (style);

}/* z_set_text_style */
