/* Copyright (c) 1998 by Michael J. Roberts.  All Rights Reserved. */
/*
Name
  plantsur.t - plant interior, above ground
Function
  
Notes
  
Modified
  08/15/98 MJRoberts  - Creation
*/


/* ------------------------------------------------------------------------ */
/*
 *   Generic add-in class for outdoor plant locations with random
 *   lightning strikes.  
 */
class inPlantLightningRoom: object
    roomLightning =
    {
        switch(rand(4))
        {
        case 1:
            "\bA bolt of lightning streaks down from the sky and strikes
            the ground a short distance to the west. ";
            break;

        case 2:
            "\bA lightning bolt rips through the sky nearby to the east. ";
            break;

        case 3:
            "\bLightning flashes across the sky directly above. ";
            break;

        case 4:
            "\bA lightning bolt flashes through the sky a short distance
            to the north. ";
            break;
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Canal, inside the plant, at the fence 
 */

canalInPlant: room, inPlantLightningRoom
    sdesc = "Canal near gate"
    ldesc =
    {
        "Vertical concrete walls line this deep canal.  A chain-link
        fence crosses over the top of the canal above; under the fence,
        a steel gate << ditchAtFence_gate.isRaised
        ? "hangs several feet above the water, allowing passage
           through a low opening under the gate"
        : "is positioned across the canal, blocking travel to
           the north" >>.  To the south, the canal stops at a concrete
        wall perforated by an array of pipes.
        \n\tA ladder is set into the west wall. ";

        if (fireHoseEnd.isIn(self))
        {
            "\n\tThe fire hose stretches up the wall of the canal to
            the top of the ladder. ";
            if (fireHoseEnd.isInPipe)
                "You're holding the nozzle of the fire hose in the
                rusty pipe. ";
        }
    }
    north =
    {
        if (ditchAtFence_gate.isRaised)
        {
            /*
             *   mention that we have to duck under the gate, unless we're
             *   not going to get to travel this way because of the hose 
             */
            if (!fireHoseEnd.isIn(self))
                "You have to bend down slightly as you cross under
                the gate. ";
            return ditchAtFence;
        }
        else
        {
            "The gate is in the way. ";
            return nil;
        }
    }
    up =
    {
        /* 
         *   we'll have another message if the fire hose is in the pipe,
         *   so don't bother with this message 
         */
        if (!fireHoseEnd.isInPipe)
        {
            "You cautiously climb up the rungs.  It's quite a ways up and
            you feel a little anxious about the height, but you reach the
            top";
            if (teeterwaller.location = self)
            {
                " and peer over the edge to see if there are any guards about.
                You can't see anyone, so you pull yourself over the edge.
                \bYou motion to Teeterwaller to join you.  He nervously
                climbs the ladder, looking down several times";
            }
            ".\b";
        }

        return canalBank;
    }
    teeterwallerPreFollow(rm) =
    {
        /* 
         *   we've already said what happens for the canal bank; use the
         *   default in other cases, though 
         */
        if (rm != canalBank)
            inherited.teeterwallerPreFollow(rm);
    }
    teeterNoted = nil
    teeterEntered = 0
    teeterwallerPostFollow =
    {
        /* do the default stuff */
        inherited.teeterwallerPostFollow;

        /* if this is my first time here, remark on it */
        if (!self.teeterNoted)
        {
            "\b<q>Wonderful job!</q> Teeterwaller says excitedly. <q>We're
            inside the complex!  Now we just need to find the trucks.</q> ";
            self.teeterNoted := true;
        }

        /* see if we need to provide some additional info */
        self.teeterEntered++;
        if (self.teeterEntered = 3)
        {
            /* 
             *   if they haven't yet noticed the thing in the pipe, have
             *   teeterwaller point it out 
             */
            if (!canalInPlant_rustyPipe.isIn(self))
            {
                /* have teeterwaller notice the rusty pipe */
                "\b<q>Look in here,</q> Teeterwaller says, looking into
                a rusty pipe.  <q>There's something stuck in there.</q> ";

                /* show the pipe */
                canalInPlant_rustyPipe.moveInto(self);
                
                /* 
                 *   reset the entrance counter, so that we provide
                 *   another hint on a subsequent trip in here
                 */
                self.teeterEntered := 1;
            }
            else if (canalInPlant_rustyPipeSomething.
                     isIn(canalInPlant_rustyPipe))
            {
                /* provide a hint on extracting the object */
                "\bTeeterwaller looks into the rusty pipe. <q>There
                must be some way we could get that out,</q> he says,
                looking at the object stuck in the pipe.  <q>Maybe
                we could flush it out, if we could get some water
                up into the pipe somehow.</q> ";

                /* note that he's given the hint */
                teeterwaller.hintedAboutPipe := true;
            }
        }
    }
    fireHosePreTravelTo(rm) =
    {
        if (rm = canalBank)
        {
            /* make teeterwaller follow again if necessary */
            if (teeterwaller.location = rm)
                teeterwaller.isFollowingMe := true;

            if (fireHoseEnd.isInPipe)
            {
                /* the hose is in the pipe - remove it */
                "You remove the nozzle from the pipe and return to the
                ladder; the hose retracts as you start climbing.\b";

                fireHoseEnd.isInPipe := nil;
            }
            else
            {
                /* mention the hose and allow the travel */
                "The fire hose retracts as you go.\b";
            }

            return true;
        }
        else if (rm != nil)
        {
            /* don't allow travel elsewhere */
            "The fire hose won't stretch any further.  You'll have to
            drop it before you can travel that way. ";
            return nil;
        }
        else
            return nil;
    }
    doDropFireHose(actor) =
    {
        "As soon as you let go of the hose, you hear a surprised yelp
        and a crash from the top of the canal wall.  The fire hose
        quickly retracts up the canal wall and disappears.
        \bTeeterwaller appears at the top of the canal.  He dusts
        himself off elaborately, then climbs down the ladder.
        <q>Sorry about that!</q> he says. <q>I must have lost my
        grip on the hose, and the reel pulled it back before I could
        get ahold of it again.</q> ";
        fireHoseEnd.retractHose;

        /* bring teeterwaller here and set him following again */
        teeterwaller.isFollowingMe := true;
        teeterwaller.moveInto(self);
    }
;

canalInPlant_pipes: fixeditem
    noun = 'pipe' 'pipes' 'array'
    location = canalInPlant
    sdesc = "array of pipes"
    adesc = "an array of pipes"
    ldesc =
    {
        "The pipes vary in size; most are about a foot in diameter,
        but many are only a few inches and a few are two feet wide.
        The pipes all open into the canal, and most are dripping
        slightly. ";

        if (canalInPlant_rustyPipe.location = nil
            || canalInPlant_rustyPipeSomething.isIn(canalInPlant_rustyPipe))
        {
            "\bSomething deep inside a large rusty pipe catches your
            eye. ";

            canalInPlant_rustyPipe.moveInto(self.location);
        }
    }
    verDoLookin(actor) = { }
    doLookin(actor) =
    {
        if (canalInPlant_rustyPipe.location = nil
            || canalInPlant_rustyPipeSomething.isIn(canalInPlant_rustyPipe))
        {
            "You peer in a few of the pipes.  Most of them appear empty
            except for some brackish water dripping out, but something
            deep inside a large rusty pipe catches your eye. ";

            canalInPlant_rustyPipe.moveInto(self.location);
        }
        else
            "The pipes seem empty, except for some brackish water (or some
            other effluent) dripping out. ";
    }
    doSynonym('Lookin') = 'Search'
    verDoClimb(actor) =
    {
        "The pipes are too slippery, and they're not spaced regularly
        enough. ";
    }
    doSynonym('Climb') = 'Climbup'
    verIoPutIn(actor) =
    {
        "It wouldn't be a good idea to put anything into the pipes;
        it would get all covered with slime. ";
    }

    /* 
     *   there's nothing in the regular pipes; even though we don't have
     *   to do this, generate a verIoTakeOut message so that there's no
     *   ambiguity when the player wants to "take hose out of pipe" 
     */
    verIoTakeOut(actor) =
    {
        "There's nothing in the pipe. ";
    }
;

canalInPlant_rustyPipe: fixeditem
    noun = 'pipe'
    adjective = 'rusty'
    sdesc = "rusty pipe"
    isObjFound = nil
    ldesc =
    {
        "The pipe is over two feet in diameter and is very rusty,
        whereas most of the other pipes look clean and new. ";
        if (!self.isObjFound)
            "You can see something inside the pipe, but you can't
            tell what it is from here; it's too far inside the
            pipe for you to reach it. ";
        if (fireHoseEnd.isInPipe)
            "\bYou're holding the fire hose so that it points into
            the pipe. ";
    }
    doSynonym('Inspect') = 'Lookin' 'Search'
    doSynonym('Lookin') = 'Search'
    verIoPutIn(actor) = { }
    ioPutIn(actor, dobj) =
    {
        if (dobj = fireHoseEnd || dobj = fireHoseNozzle)
        {
            if (fireHoseEnd.isInPipe)
                "You're already holding it in the pipe. ";
            else
            {
                "You shove the fire hose nozzle into the pipe.  You
                still have to hold the hose to keep it from retracting. ";
                fireHoseEnd.isInPipe := true;
            }
        }
        else
            "You don't want to put anything into the pipe; it would
            just get all covered with slime. ";
    }
    ioSynonym('PutIn') = 'PointAt'
    verIoTakeOut(actor) = { }
    ioTakeOut(actor, dobj) =
    {
        if ((dobj = fireHoseEnd || dobj = fireHoseNozzle)
            && fireHoseEnd.isInPipe)
        {
            "You remove the nozzle from the pipe, keeping hold of
            the end of the hose. ";
            fireHoseEnd.isInPipe := nil;
        }
        else
            "That's not in the pipe. ";
    }
;

canalInPlant_rustyPipeSomething: distantItem
    noun = 'something' 'object' 'thing'
    location = canalInPlant_rustyPipe
    sdesc = "something in the pipe"
    adesc = "something in the pipe"
    thedesc = "something in the pipe"
    ldesc = "You can't tell what it is. "
    verIoPointAt(actor) = { }
    ioPointAt(actor, dobj) =
    {
        if (dobj = fireHoseEnd || dobj = fireHoseNozzle)
            self.location.ioPutIn(actor, dobj);
        else
            "The object remains obscure. ";
    }
;

canalInPlant_ladder: fixeditem
    noun = 'ladder' 'rung' 'rungs'
    adjective = 'metal'
    sdesc = "ladder"
    location = canalInPlant
    ldesc = "The ladder is just a set of metal rungs set into
             the concrete wall at regular intervals. "
    verDoClimb(actor) = { }
    doClimb(actor) = { actor.travelTo(self.location.up); }
    doSynonym('Climb') = 'Climbup'
;

canalInPlant_walls: fixeditem
    noun = 'wall' 'walls'
    adjective = 'concrete' 'vertical' 'west' 'east' 'south'
    location = canalInPlant
    sdesc = "concrete wall"
    ldesc = "The walls are around twenty feet high and vertical. "
    verDoClimb(actor) = { "The walls are too steep to climb. "; }
    doSynonym('Climb') = 'Climbup'
;

canalInPlant_canal: fixeditem
    noun = 'canal' 'water'
    sdesc = "canal"
    location = canalInPlant
    ldesc = "The canal runs north and south. "
    verDoLookin(actor) = { "You see nothing hiding under the water. "; }
    verDoEnter(actor) = { "You hae no desire to swim in the water. "; }
    doSynonym('Enter') = 'Swim' 'Swimin' 'Board'
    verDoDrink(actor) = { }
    doDrink(actor) = { "You can't be serious. "; }
;

canalInPlant_gate: fixeditem
    noun = 'gate' 'bar' 'bars'
    adjective = 'steel' 'closely-spaced' 'spaced'
    sdesc = "steel gate"
    location = canalInPlant
    ldesc = "The gate is currently << ditchAtFence_gate.isRaised
             ? "raised up several feet above the water"
             : "immersed in the water, blocking travel to the north" >>. "
;

canalInPlant_rails: fixeditem
    noun = 'rail' 'rails'
    adjective = 'left' 'right'
    sdesc = "rail"
    location = canalInPlant
    ldesc = "The rails guide the gate so that it can move up and down. "
;

canalInPlant_fence: distantItem
    sdesc = "fence"
    noun = 'fence'
    adjective = 'chain' 'link' 'chain-link'
    location = canalInPlant
    ldesc = "You can't see much detail from here; the fence is above
             the tops of the banks of the canal. "
;

/* ------------------------------------------------------------------------ */
/*
 *   bank of canal, at top of ladder 
 */

canalBank: room, noFloorRoom, inPlantLightningRoom
    sdesc = "Bank of canal"
    ldesc =
    {
        "A deep, concrete-lined canal is to the east; metal rungs
        are set into the wall, forming a ladder down into the canal.
        A tall fence topped with razor wire is to the north.
        \n\tPipes and conduits are everywhere, packed so tightly
        together that they form walls.  Narrow openings in the pipes
        allow travel to the southeast and southwest.  To the west, a
        cylindrical tower rises up out of the pipes and looms several
        stories overhead; a spiral stairway encircling the tower
        leads up.  The ground is replaced by a metal grating for the
        ten feet around the tower. ";

        if (fireHoseEnd.isIn(self))
            "\n\tThe fire hose stretches along the ground through the
            opening to the southwest. ";
    }
    down = canalInPlant
    east = canalInPlant
    sw = safetyStation
    se = narrowPassage
    up =
    {
        /* 
         *   leave off the transitional message if the fire hose end is
         *   here, since that means we'll fail to travel (in which case
         *   we'll display the fire hose message in fireHosePreTravel) 
         */
        if (!fireHoseEnd.isIn(self))
            "You cross the grating, and climb the gentle slope of the
            stairway through one revolution around the tower.\b";
        return towerStair2;
    }
    west = { return self.up; }
    fireHosePreTravelTo(rm) =
    {
        /* check where we're going */
        if (rm = canalInPlant)
        {
            /* 
             *   allow traveling down into the canal, but make teeterwaller
             *   stay up on the bank to let the hose down 
             */
            if (teeterwaller.location = self)
            {
                if (self.teeterwallerAssistedBefore)
                    "<q>I guess we'll try it again,</q> Teeterwaller
                    says, sounding a little impatient. <q>I'll stay
                    here and help guide the hose over the edge for you.</q>
                    He stands at the top of the ladder as you start to
                    climb down.\b";
                else
                    "Teeterwaller looks down into the canal. <q>I don't think
                    the hose will make it over the edge without some help,</q>
                    he says.  <q>I'll stay up here and guide it over the
                    edge for you.</q>  He situates himself at the top of
                    the ladder as you start to climb down.\b";

                /* make teeterwaller stop following for now */
                teeterwaller.isFollowingMe := nil;
                self.teeterwallerAssistedBefore := true;
            }
            else
            {
                "The hose won't make it over the ledge without someone
                to guide it. ";
                return nil;
            }
        }
        else if (rm = safetyStation)
        {
            /* mention that the hose is reeling back up */
            "The fire hose retracts onto its reel as you move back
            toward the safety station.\b";
        }
        else if (rm = narrowPassage)
        {
            "You can't get the hose around the bend&mdash;there are
            too many pipes at odd angles. ";
            return nil;
        }
        else if (rm = towerStair2)
        {
            "You won't be able to drag the hose up the stairs; it would
            get stuck wrapping around the tower wall. ";
            return nil;
        }
        else if (rm != nil)
        {
            /* we can't take the hose elsewhere - disallow it */
            "The hose won't go any further.  You'll have to let go of
            the hose if you want to go that way. ";
            return nil;
        }

        /* if we got here, allow it */
        return true;
    }
    doDropFireHose(actor) =
    {
        "As soon as you let go of the hose, it slides away to the southeast
        as the reel winds the hose back onto the coil. ";
        fireHoseEnd.retractHose;
    }
;

canalBank_grating: towerGratingItem
    sdesc = "floor grating"
    ldesc = "It's a sturdy metal grating covering a shaft around the
             tower. "
    location = canalBank
    doLookthru(actor) =
    {
        "The tower appears to continue down into the shaft for some
        distance.  You cannot see the bottom.  Fortunately, the grating
        allows you to avoid falling into the shaft. ";
    }
;

canalBank_tower: fixeditem
    noun = 'tower'
    adjective = 'cylindrical'
    location = canalBank
    sdesc = "tower"
    ldesc = "The tower is several stories high.  The bottom of a
             spiral stairway that encircles the tower is to the
             west. "
    verDoEnter(actor) = { "You see no entrance here. "; }
    verDoClimb(actor) = { "You might want to try the stairs. "; }
    doSynonym('Climb') = 'Climbup'
;

canalBank_towerStairs: fixeditem
    noun = 'stair' 'stairs' 'stairway' 'staircase' 'bottom'
    adjective = 'spiral' 'metal'
    location = canalBank
    sdesc = "spiral stairway"
    ldesc = "The stairs are made of a metal grating.
             The bottom of the stairway is to the west; the stairs
             spiral around the tower several times, and look like they
             go to the top. "
    verDoClimb(actor) = { }
    doClimb(actor) = { actor.travelTo(self.location.up); }
    doSynonym('Climb') = 'Climbup' 'Enter'
;

canalBank_ladder: fixeditem
    noun = 'ladder' 'rung' 'rungs'
    adjective = 'metal'
    sdesc = "ladder"
    location = canalBank
    ldesc = "The ladder is set into the wall of the canal. "
    verDoClimb(actor) = { }
    doClimb(actor) = { actor.travelTo(self.location.down); }
    doSynonym('Climb') = 'Climbdown'
;

canalBank_fence: plantFenceItem
    location = canalBank
    doLookthru(actor) =
    {
        "Beyond the fence lies only barren landscape. ";
    }
;

canalBank_pipes: fixeditem
    noun = 'pipe' 'pipes' 'conduits' 'wall'
    location = canalBank
    sdesc = "pipes and conduits"
    adesc = "a wall of pipes and conduits"
    ldesc = "The pipes are all sparkling clean and painted white.  Many
             of the pipes have little arrows pointing in one direction
             or the other along their length, and most have some obscure
             stenciled marking: LX9, ME-OX-TEC, HPS, BYR/BYC, H3Z, CWR. "
    isThem = true
    verDoClimb(actor) = { "They don't provide a good enough handhold. "; }
    doSynonym('Climb') = 'Climbup'
    verDoLookin(actor) = { "The pipes are opaque. "; }
;

canalBank_canal: fixeditem
    noun = 'canal' 'ditch'
    location = canalBank
    sdesc = "canal"
    ldesc = "The canal is deep&mdash;around twenty feet&mdash;and is
             immediately to the east. "
    verDoEnter(actor) = { }
    doEnter(actor) = { actor.travelTo(self.location.down); }
    doSynonym('Enter') = 'Board'
;

/* ------------------------------------------------------------------------ */
/*
 *   Generic class for tower stair locations.  These locations have a
 *   number of associated floating items for scenery - the ditch, the
 *   tower, the stairs themselves.
 *   
 *   Note that we don't want the standard floor object on the stairs, so
 *   we'll mark the stars as noFloorRoom's.  
 */
class towerStairRoom: room, noFloorRoom, inPlantLightningRoom
    /* adjacent levels */
    adjacentAbove = nil
    adjacentBelow = nil
    
    /*
     *   Describe items on the adjacent levels 
     */
    adjacentDesc =
    {
        local disp := nil;
        local guardsAbove, guardsBelow;
        
        if (self.adjacentAbove != nil
            && itemcnt(self.adjacentAbove.contents) != 0)
        {
            "Sitting above you, on top of the grating of the stairs
            one level up, you see <<listcont(self.adjacentAbove)>>";

            disp := true;
        }

        if (self.adjacentBelow != nil
            && itemcnt(self.adjacentBelow.contents) != 0)
        {
            if (disp)
                "; in addition, t";
            else
                "T";
            "hrough the grating below, you
            see <<listcont(self.adjacentBelow)>> on the stairs
            one level down. ";
        }
        else if (disp)
            ". ";

        /* see if we have guards above or below */
        guardsAbove := self.areGuardsPresent(self.adjacentAbove);
        guardsBelow := self.areGuardsPresent(self.adjacentBelow);

        /* show a message if they're present */
        if (guardsAbove && guardsBelow)
            "\n\tSeveral guards are watching you through the grating
            of the landings above and below. ";
        else if (guardsAbove || guardsBelow)
            "\n\tSeveral guards are watching you through the grating
            of the landing one level <<guardsAbove ? "up" : "down">>. ";
    }

    areGuardsPresent(adjacentRoom) =
    {
        return (adjacentRoom != nil
                && adjacentRoom.myGuards != nil
                && adjacentRoom.myGuards.isIn(adjacentRoom));
    }

    /*
     *   Items on the next levels above and below are visible, although
     *   not reachable. 
     */
    isVisible(vantage) =
    {
        /* 
         *   we can see our contents from the adjacent rooms above and
         *   below 
         */
        if (vantage = self.adjacentAbove || vantage = self.adjacentBelow)
            return true;

        /* in other cases, inherit the default behavior */
        return inherited.isVisible(vantage);
    }

    /*
     *   Compute the list of objects visible from this location.  Include
     *   objects that are normally visible in this location, plus the
     *   objects that are visible in my adjacent rooms. 
     */
    getVisibleList(actor) =
    {
        local ret;
        
        /* compute the normal list */
        ret := inherited.getVisibleList(actor);

        /* add in the list of objects visible in the adjacent rooms */
        if (self.adjacentAbove != nil)
            ret += visibleList(self.adjacentAbove, actor);
        if (self.adjacentBelow != nil)
            ret += visibleList(self.adjacentBelow, actor);

        /* return the result */
        return ret;
    }

    /*
     *   Modify the cantReach message to display something sensible if the
     *   actor is on one of the adjacent levels. 
     */
    cantReach(actor) =
    {
        if ((self.adjacentAbove != nil && actor.isIn(self.adjacentAbove))
            || (self.adjacentBelow != nil && actor.isIn(self.adjacentBelow)))
            "That's on the other side of the grating. ";
        else
            inherited.cantReach(actor);
    }
    roomAction(actor, v, dobj, prep, iobj) =
    {
        if (v = jumpVerb)
        {
            "That would be suicidal. ";
            exit;
        }
        else
            inherited.roomAction(actor, v, dobj, prep, iobj);
    }

    roomGuardDaemon =
    {
        "\bThe guards watch you through the stairs. ";
    }
;

class towerStairRoom_item: fixeditem, floatingItem
    location =
    {
        if (isclass(parserGetMe().location, towerStairRoom))
            return parserGetMe().location;
        else
            return nil;
    }
;

towerStairRoom_grateAbove: distantItem, towerStairRoom_item
    sdesc = "top grating"
    noun = 'grate' 'grating' 'stair' 'stairs' 'landing' 'stairway' 'staircase'
    adjective = 'top' 'metal' 'spiral'
    ldesc =
    {
        "The grating is the bottom of the stairs one level above. ";
        if (parserGetMe().location.adjacentAbove != nil
            && itemcnt(parserGetMe().location.adjacentAbove.contents) != 0)
            "Through the grating, you can
            see <<listcont(parserGetMe().location.adjacentAbove)>>. ";
    }
    location =
    {
        local ret;

        /* figure the location normally */
        ret := inherited.location;

        /* 
         *   if the location doesn't have an adjacentAbove setting, don't
         *   give it stairs above
         */
        if (ret != nil && ret.adjacentAbove = nil)
            return nil;

        /* return the result */
        return ret;
    }
    verDoClimb(actor) =
    {
        "You can't reach the next level of the stairs from here. ";
    }
    verDoInspect(actor) = { }
    doSynonym('Climb') = 'Climbup' 'Climbdown' 'Enter'
    doSynonym('Inspect') = 'Lookthru' 'Lookin'
;

towerStairRoom_grateBelow: distantItem, towerStairRoom_item
    sdesc = "bottom grating"
    noun = 'grate' 'grating' 'stair' 'stairs' 'landing' 'stairway' 'staircase'
    adjective = 'bottom' 'metal' 'spiral'
    ldesc =
    {
        "The stairs are made of a metal grating";
        if (parserGetMe().location.adjacentBelow != nil)
        {
            ", through which you can see the next level of stairs below. ";

            if (itemcnt(parserGetMe().location.adjacentBelow.contents) != 0)
                "You can
                see <<listcont(parserGetMe().location.adjacentBelow)>> on
                the stairs one level down. ";
        }
        else
            "; you see only the ground below. ";
    }
    verDoClimb(actor) =
    {
        "You must specify whether you want to go up or down. ";
    }
    verDoClimbup(actor) = { }
    doClimbup(actor) = { actor.travelTo(actor.location.up); }
    verDoClimbdown(actor) = { }
    doClimbdown(actor) = { actor.travelTo(actor.location.down); }
    verDoInspect(actor) = { }
    doSynonym('Inspect') = 'Lookthru' 'Lookin'
;

towerStairRoom_tower: towerStairRoom_item
    sdesc = "tower"
    noun = 'tower' 'wall'
    adjective = 'cylindrical' 'white' 'tower'
    ldesc = "From here, it just looks like a curved metal wall
             painted white. "
    verDoKnock(actor) = { }
    doKnock(actor) =
    {
        "You rap your kunckles on the side of the tower. The metal
        feels very thick; it makes only a dull ringing sound. ";
    }
    verDoEnter(actor) =
    {
        if (self.location != towerStair4
            && self.location != towerStair4West)
            "You see no entrance here. ";
    }
    doEnter(actor) = { actor.travelTo(self.location.in); }
;

towerStairRoom_canal: distantItem, towerStairRoom_item
    sdesc = "canal"
    noun = 'canal' 'ditch'
    ldesc = "You can't see much detail from here. "
;

towerStairRoom_plant: distantItem, towerStairRoom_item
    sdesc = "plant complex"
    noun = 'plant' 'complex'
    adjective = 'plant'
    ldesc = "The plant is all around you. "
;


/* ------------------------------------------------------------------------ */
/*
 *   Class for doors on the outsides of the tower.  
 */
class towerDoorItem: doorway
    sdesc = "rounded metal door"
    noun = 'door'
    adjective = 'rounded' 'metal'
    doordest = inTowerEast
    otherside = inTowerEast_door
    ldesc = "The <<self.isopen ? "open" : "closed" >> door looks like
             something you'd find on a submarine:
             its bottom is a foot or so above the landing, and it's
             only four feet high.  The door is made of the same thick
             metal as the tower.  In its center is a wheel. "
    isopen = nil
    noAutoOpen = true
    issealed = true
    experienced = nil
    setIsopen(setting) =
    {
        inherited.setIsopen(setting);
        towerStair_gas.moveInto(setting ? self.location : nil);
    }
    doOpen(actor) =
    {
        if (self.issealed)
            "The door won't budge. ";
        else if (!olderGasMask.isworn && !newerGasMask.isworn)
        {
            if (self.experienced)
                "You have no desired to open the door again without
                some sort of protection from the acrid fumes beyond it. ";
            else
            {
                "As soon as you start to move the door, a thick, acrid
                vapor oozes out of the opening, and your eyes and lungs
                start burning.  You quickly shut the door again. ";
                self.experienced := true;
            }
        }
        else
        {
            inherited.doOpen(actor);

            "\bWisps of viscous green vapor waft slowly out from
            the door.  Fortunately, the gas mask seems to be protecting
            you from the gas. ";

            towerStair_gas.moveInto(self.location);
        }
    }
;

class towerDoorWheelItem: fixeditem
    noun = 'wheel'
    adjective = 'large'
    sdesc = "wheel"
    ldesc = "It's a big metal wheel, about a foot in diameter, in
             the middle of the door. "
    verDoTurn(actor) = { }
    doTurn(actor) =
    {
        if (self.location.isopen)
            "The wheel won't move. ";
        else
        {
            "The wheel is hard to move, but you manage to turn it a
            couple of revolutions <<self.location.issealed ? "clockwise"
                : "counterclockwise">>, where it stops. ";
            
            if (self.location.issealed)
                "A hissing sound escapes from the door for a few moments,
                then fades away. ";
            
            self.location.issealed := !self.location.issealed;
            if (self.location.otherside != nil)
                self.location.otherside.issealed := self.location.issealed;
        }
    }
    doSynonym('Turn') = 'Move'
;

towerStair_gas: fixeditem
    noun = 'gas' 'fume' 'fumes' 'vapor' 'vapors'
    adjective = 'viscous' 'green' 'acrid'
    sdesc = "green vapor"
    ldesc = "The green vapor wafts lazily out of the door. "
    verDoLookthru(actor) =
    {
        "The vapor is thick, but it's not obscuring anything. ";
    }
    doSynonym('Lookthru') = 'Lookin' 'Search'
;


/* ------------------------------------------------------------------------ */
/*
 *   Tower stairs 
 */


/*
 *   end game rooms for tower stairs must use the same teeterwaller
 *   processing as normal end game rooms 
 */
class towerStairEndGameRoom: towerStairRoom
    teeterwallerPreFollow(rm) =
    {
        /* 
         *   If the disk is about to be left behind, make teeterwaller
         *   grab it; otherwise, just do the normal work. 
         */
        if (lab7_disk.isIn(self) && !lab7_disk.isIn(parserGetMe()))
        {
            "Mr.\ Teeterwaller picks up the black disk and hands it
            to you. <q>We don't want to lose track of this,</q> he says
            as he catches up to you.\b";

            lab7_disk.moveInto(parserGetMe());
        }
        else
            inherited.teeterwallerPreFollow(rm);
    }
;

/*
 *   Tower stairs - odd helix, first level - one story up
 */
towerStair1: towerStairEndGameRoom
    sdesc = "Landing, one story up"
    ldesc = "This landing is about one story above the ground, on
             the east side of the tower. The stairs continue up
             and down.
             \n\t<<self.adjacentDesc>> "
    adjacentAbove = towerStair2
    up = towerStair3
    down =
    {
        "As the stairway winds around to the west side of the tower,
        it passes through an opening in the grating surrounding the
        tower and descends into the shaft below.\b";

        return towerShaft;
    }
;

/*
 *   Tower stairs - even helix, first level - two stories up
 */
towerStair2: towerStairRoom
    sdesc = "Landing, two stories up"
    ldesc = "A short landing, about two stories above the ground on the
             east side of the tower,
             interrupts the steady spiral of the stairs, which continue
             up and down.  The canal is visible in the distance below, and
             the tower looms overhead.
             \n\t<<self.adjacentDesc>> "
    adjacentBelow = towerStair1
    adjacentAbove = towerStair3
    down = canalBank
    up = towerStair4
    myGuards = towerStair2_guards
;

towerStair2_guards: guardsItem
    ldesc = "The guards are watching you through the grating of the stairs. "
;

/*
 *   Tower stairs - odd helix, second level - three stories up 
 */
towerStair3: towerStairEndGameRoom
    sdesc = "Landing, three stories up"
    ldesc = "This landing is about three stories above the
             ground, on the east side of the tower.
             The stairs continue wrapping around the
             tower, allowing you to travel up or down.
             \n\t<<self.adjacentDesc>> "
    adjacentBelow = towerStair2
    adjacentAbove = towerStair4
    down = towerStair1
    up =
    {
        "The stairs only make half a turn around the tower before
        you encounter another landing, this time on the west side
        of the tower.\b";
        return towerStair4West;
    }
;

/*
 *   Tower stairs - even helix, second level - four stories up
 */
towerStair4: towerStairRoom
    sdesc = "Landing, four stories up"
    ldesc =
    {
        "The stairs end at this landing, and go no further up.
        This landing is nearly at the top of the tower, around
        four stories above the ground, and on the east side
        of the tower.\n\t";

        if (towerStair4_door.isopen)
            "An open rounded metal door is set into the tower wall
            to the west; a viscous green vapor wafts out of the door. ";
        else
            "A closed rounded metal door is set into the tower wall
            to the west. ";
        "A large wheel is in the center of the door.
        \n\t<<self.adjacentDesc>> ";
    }
    down =
    {
        if (towerStair4_door.isopen)
        {
            "As you start down the stairs, you notice that the door swings
            shut.\b";
            towerStair4_door.setIsopen(nil);
        }
        
        return towerStair2;
    }
    west = towerStair4_door
    in = towerStair4_door
    adjacentBelow = towerStair3
    isGassy = { return towerStair_gas.isIn(self); }
    myGuards = towerStair4_guards
;

towerStair4_guards: guardsItem
    ldesc = "The guards are watching you through the grating of the stairs. "
;

towerStair4_door: towerDoorItem
    location = towerStair4
;

towerStair4_doorWheel: towerDoorWheelItem
    location = towerStair4_door
;

class hardhatItem: duplicatedClothingItem
    genericDesc = "hard-hat"
    location = towerStair3
    noun = 'hat' 'hardhat' 'hard-hat'
    plural = 'hats' 'hardhats' 'hard-hats'
    adjective = 'hard' 'plastic'
;

yellowHardhat: hardhatItem
    sdesc = "yellow hard-hat"
    ldesc = "It's a yellow plastic hard-hat. "
    adjective = 'yellow'
    otherOne = whiteHardhat
;

whiteHardhat: hardhatItem
    sdesc = "white hard-hat"
    adjective = 'white'
    ldesc = "It's a white plastic hard-hat. "
    otherOne = yellowHardhat
;

/*
 *   Tower stairs - top of odd-level staircase (west landing) 
 */
towerStair4West: towerStairEndGameRoom
    sdesc = "Landing, four stories up"
    ldesc =
    {
        "The stairs reach their summit at this landing, which, unlike
        all of the other landings, is on the west side of the tower.
        Handholds set into the tower wall at regular intervals form
        a ladder up the side of the tower. 
        \n\t";

        if (towerStair4West_door.isopen)
            "An open door is set into the tower wall; a viscous
            green vapor wafts out of the door. ";
        else
            "A closed rounded metal door is set into the tower wall. ";
        "In the center of the door is a wheel. ";
    }
    down =
    {
        if (towerStair4West_door.isopen)
        {
            "As you start down the stairs, you notice that the door swings
            shut.\b";
            towerStair4West_door.setIsopen(nil);
        }

        "You follow the stairs halfway around the tower to the next
        landing.\b";

        return towerStair3;
    }
    up =
    {
        /*
         *   If we're in the end-game, bring on the guards 
         */
        if (towerStair4_guards.isIn(towerStair4))
        {
            "As you start up the ladder, the door bursts open and guards
            start pouring out onto the landing.  You climb the ladder
            as quickly as you can.\b";

            return towerRoof;
        }
        
        if (towerStair4West_door.isopen)
        {
            "As you start down the stairs, you notice that the door swings
            shut.\b";
            towerStair4West_door.setIsopen(nil);
        }
        
        "You climb up the side of the tower; you soon reach the dome at
        the top, and follow the handholds up the curved surface until
        you reach the top of the dome.\b";
        return towerRoof;
    }
    east = towerStair4West_door
    in = towerStair4West_door
    adjacentBelow = fake_towerStair3_half
    isGassy = { return towerStair_gas.isIn(self); }

    /*
     *   Check the end-game.  This should be called for any command
     *   handlers for commands not allowed during the end-game, such as
     *   opening the door or turning the wheel.  If we're in the end-game,
     *   abort whatever command got us here and go to the roof.  Returns
     *   true if we moved the player to the roof, nil if not.  
     */
    checkForEndgame(actor) =
    {
        if (towerStair4_guards.isIn(towerStair4))
        {
            "As you reach for the wheel, someone starts opening the
            door from the other side.  You freeze for a moment.  Someone
            grabs your arm.  You turn around and see that it's Teeterwaller,
            pulling you to the ladder.  You climb up the ladder as fast
            as you can.\b";

            /* move the player to the roof */
            actor.travelTo(towerRoof);

            /* indicate that we've aborted the command */
            return true;
        }
        else
        {
            /* 
             *   we're not in the end-game, so tell the caller to proceed
             *   as normal 
             */
            return nil;
        }
    }

    roomGuardDaemon =
    {
        /*
         *   There are no guards visible from this location during the
         *   end-game, so don't bother with a guard message here.
         */
    }
;

towerStair4West_ladder: fixeditem
    location = towerStair4West
    noun = 'ladder' 'handholds' 'holds' 'hold' 'hand-holds' 'rung' 'rungs'
    adjective = 'hand'
    sdesc = "handholds"
    isThem = true
    ldesc = "The handholds form a ladder up the side of the tower. "
    verDoClimb(actor) = { }
    doClimb(actor) = { actor.travelTo(self.location.up); }
    doSynonym('Climb') = 'Climbup'
;

towerStair4West_door: towerDoorItem
    location = towerStair4West
    experienced = true
    doordest = inTowerWest
    otherside = inTowerWest_door
    doOpen(actor) =
    {
        /*
         *   Check to see if the guards are present.  If so, we're in the
         *   end-game, so proceed to the roof. 
         */
        if (towerStair4West.checkForEndgame(actor))
            return;

        /* otherwise, use the inherited behavior */
        inherited.doOpen(actor);
    }
;

towerStair4West_wheel: towerDoorWheelItem
    location = towerStair4West_door
    doTurn(actor) =
    {
        /*
         *   Check to see if the guards are present on the even stairs.
         *   If so, we're in the end-game, so proceed to the roof
         *   directly.  
         */
        if (towerStair4West.checkForEndgame(actor))
            return;

        /* do the normal work */
        inherited.doTurn(actor);
    }
;

/*
 *   fake stairway room - we never go here, but we need something to serve
 *   as the room adjacent to the top of the west platform below, for when
 *   we look through the bottom grating (there's never anything here, of
 *   course; we just need it so we have a non-nil adjacentBelow setting in
 *   towerStair4West) 
 */
fake_towerStair3_half: towerStairRoom
;


/* ------------------------------------------------------------------------ */
/*
 *   Generic room for inside the tower; provides some floating items. 
 */
class inTowerRoom: room
    roomAction(actor, v, dobj, prep, iobj) =
    {
        if (v = jumpVerb)
        {
            "That would be suicidal. ";
            exit;
        }
        else
            inherited.roomAction(actor, v, dobj, prep, iobj);
    }
    isGassy = true
;

class inTowerItem: floatingItem
    location =
    {
        if (isclass(parserGetMe().location, inTowerRoom))
            return parserGetMe().location;
        else
            return nil;
    }
;

inTower_vapor: inTowerItem, fixeditem
    sdesc = "green vapor"
    noun = 'vapor' 'gas'
    adjective = 'green' 'misty'
    ldesc = "The vapor swirls slowly through the air. It's thickest near
             the surface of the liquid below, but it permeates the entire
             interior of the tower. "
    verDoLookthru(actor) =
    {
        "The vapor is thick, but it's not obscuring anything. ";
    }
    doSynonym('Lookthru') = 'Lookin' 'Search'
;

class inTower_catwalk_item: fixeditem
    noun = 'catwalk'
    sdesc = "catwalk"
;

inTower_liquid: inTowerItem, distantItem
    sdesc = "churning liquid"
    adjective = 'churning' 'bubbling' 'viscous' 'yellow-green' 'green'
    noun = 'liquid' 'fluid' 'sea' 'surface'
    ldesc = "The liquid churns and bubbles; green vapor rises slowly
             off the surface ten feet below the catwalk. "
    verDoEnter(actor) =
    {
        "You want to stay as far away from the stuff as possible. ";
    }
    doSynonym('Enter') = 'Swimin' 'Board' 'Jumpin'
;

class inTowerDoorItem: doorway
    noun = 'door'
    sdesc = "rounded metal door"
    adjective = 'rounded' 'metal'
    ldesc = "The <<self.isopen ? "open" : "closed">> door is about
             four feet tall and rounded at the corners. "
    noAutoOpen = { return self.issealed; }
    doOpen(actor) =
    {
        if (self.otherside.issealed)
            "The door won't budge; it seems to be locked from the
            outside. ";
        else
            inherited.doOpen(actor);
    }
;

inTower_tower: inTowerItem, fixeditem
    noun = 'tower' 'wall' 'cylinder'
    adjective = 'cylindrical'
    sdesc = "tower"
    ldesc = "The tower walls curve around you. "
;

inTower_roof: inTowerItem, distantItem
    noun = 'roof' 'ceiling' 'dome' 'top'
    adjective = 'domed'
    sdesc = "domed roof"
    ldesc = "The domed roof of the tower is about twenty feet up.
             In the center, you can just make out what looks like
             an access hatch. "
;

inTower_accessHatch: inTowerItem, distantItem
    noun = 'hatch'
    adjective = 'access'
    sdesc = "access hatch"
    adesc = "an access hatch"
    ldesc = "It's a round hatch in the center of the roof. "
;

/* ------------------------------------------------------------------------ */
/*
 *   Inside the tower - east end of catwalk
 */
inTowerEast: inTowerRoom
    east = inTowerEast_door
    out = inTowerEast_door
    sdesc = "East end of catwalk"
    enterRoom(actor) =
    {
        inherited.enterRoom(actor);

        if (inTowerEast_door.isopen)
        {
            "\bThe rounded metal door swings shut with a dull thud.\b";
            inTowerEast_door.setIsopen(nil);
        }
    }
    ldesc = "A catwalk starts at the rounded door to the
             east (which is <<inTowerEast_door.isopen ? "open" : "closed">>),
             and extends west toward the center of the cylinder of the tower,
             suspended by cables over a churning, bubbling sea of viscous
             yellow-green liquid ten feet below.  A green, misty vapor
             swirls up from the surface of the liquid. "
    west =
    {
        if (inTowerEast_door.isopen)
        {
            "The door swings shut with a thud.\b";
            inTowerEast_door.setIsopen(nil);
        }
        return inTowerCenter;
    }
;

inTowerEast_catwalk: inTower_catwalk_item
    location = inTowerEast
    ldesc = "The catwalk extends to the west. "
;

inTowerEast_cables: distantItem
    noun = 'cable' 'cables'
    location = inTowerEast
    sdesc = "cables"
    isThem = true
    ldesc = "The cables are attached to the catwalk near the center
             of the tower; you'd be able to get a better look from there. "
;

inTowerEast_door: inTowerDoorItem
    location = inTowerEast
    otherside = towerStair4_door
    doordest = towerStair4
;


/* ------------------------------------------------------------------------ */
/*
 *   In tower - center of catwalk 
 */
inTowerCenter: inTowerRoom
    sdesc = "Center of catwalk"
    east = inTowerEast
    west = inTowerWest
    ldesc = "This is roughly the center of the catwalk, which extends
             east and west.  A pair of cables, suspended from above,
             are attached to the catwalk here to provide support. "
    up =
    {
        if (inTowerCenter_cables.okToClimb)
        {
            if (!topOfCable_hatch.isopen)
                "<q>I think I'll wait here,</q> Mr.\ Teeterwaller says
                nervously.\b";

            if (inTowerCenter_cables.isClimbed)
                "You make your way back up the cable.\b";
            else
            {
                "You take a deep breath and grasp the cable, giving it a tug
                to make sure it's as firmly attached as it appears to be.
                Taking care not to look down at the bubbling cauldron of
                who-knows-what below, you start ascending the cable.  It's
                as hard as you expect, but you manage to climb the twenty
                feet to the roof of the dome.\b";

                inTowerCenter_cables.isClimbed := true;
            }
            return topOfCable;
        }
        else
        {
            "There's no way up, unless you're thinking about climbing
            the cable. ";
            return nil;
        }
    }
    teeterCanFollow(destRoom) =
    {
        if (destRoom = topOfCable && !topOfCable_hatch.isopen)
            return nil;
        else
            return inherited.teeterCanFollow(destRoom);
    }
;

inTowerCenter_catwalk: inTower_catwalk_item
    location = inTowerCenter
    ldesc = "The catwalk extends to the east and west. "
;

inTowerCenter_cables: fixeditem
    noun = 'cable' 'cables'
    location = inTowerCenter
    sdesc = "cables"
    isThem = true
    isClimbed = nil
    ldesc = "The cables extend to the domed roof of the tower above,
             and are attached to the catwalk here to support its weight.
             At the top of the cables, you can just make out what looks
             like an access hatch in the roof. "
    verDoClimb(actor) = { }
    doClimb(actor) =
    {
        self.okToClimb := true;
        actor.travelTo(self.location.up);
    }
    doSynonym('Climb') = 'Climbup'
;


/* ------------------------------------------------------------------------ */
/*
 *   At the top of the cable in the tower 
 */
topOfCable: room
    sdesc = "Under the roof, on a cable"
    ldesc = "The cable is attached to a hook bolted to the roof of
             the dome.  Within reach is a round access hatch,
             currently <<topOfCable_hatch.isopen ? "open" : "closed">>,
             in the roof of the dome. "
    down = { return self.checkTravelTo(inTowerCenter); }
    checkTravelTo(dest) =
    {
        if (harness.isIn(parserGetMe())
            && harness.isAttachedTo(topOfCable_hook))
        {
            "You start to leave, but the harness instantly jerks you
            to a stop. ";
            return nil;
        }
        else
        {
            if (dest = inTowerCenter)
            {
                "You slide down the cable.\b";

                /* close the hatch if we left it open */
                if (topOfCable_hatch.isopen)
                {
                    "You notice the hatch slam shut as you leave.\b";
                    topOfCable_hatch.setIsopen(nil);
                }
            }
            else if (dest = towerRoof)
                "You grab onto the edge of the access hatch and
                pull yourself through the opening and out onto the
                roof of the tower.\b";
            
            return dest;
        }
    }
    up = topOfCable_hatch
    out = topOfCable_hatch
    roomDrop(obj) =
    {
        "This is a bad place to drop things. ";
    }
;

topOfCable_hook: harnessHookableItem
    sdesc = "hook"
    noun = 'hook'
    location = topOfCable
    attachmentList = [topOfCable_cable]
    ldesc =
    {
        "The hook protrudes from the roof. ";
        self.attachmentDesc;
    }

    processAttachTo(actor, otherObj, isFirst) =
    {
        /* inherit the default behavior */
        inherited.processAttachTo(actor, otherObj, isFirst);

        /* make sure it's the harness */
        if (otherObj != harness)
        {
            "You see no way to attach <<otherObj.thedesc>> to the hook. ";
            return;
        }

        /* say what happened */        
        "You slip the harness's ring onto the cable hook. ";
        if (otherObj.isworn)
            "The hook and harness seem easily able to support
            your weight. ";
        else
            otherObj.moveInto(self.location);
    }

    processDetachFrom(actor, otherObj, isFirst) =
    {
        /* inherit default behavior */
        inherited.processDetachFrom(actor, otherObj, isFirst);

        "You slip the ring off the hook";
        if (!otherObj.isIn(actor))
        {
            ", taking <<otherObj.thedesc>>. ";
            otherObj.moveInto(actor);
        }
    }
;

topOfCable_hatch: doorway
    sdesc = "access hatch"
    adesc = "an access hatch"
    location = topOfCable
    noun = 'hatch' 'door' 'opening'
    adjective = 'access'
    noAutoOpen = { return self.issealed; }
    otherside = towerRoof_hatch
    doordest = { return self.location.checkTravelTo(towerRoof); }
    issealed = true
    ldesc = "It's a round door about two feet in diameter, with a wheel
             in the center. It's currently <<self.isopen ? "open"
             : "closed">>. "
    doOpen(actor) =
    {
        if (self.issealed)
            "It won't budge. ";
        else
            inherited.doOpen(actor);
    }
;

topOfCable_wheel: towerDoorWheelItem
    location = topOfCable_hatch
    doTurn(actor) =
    {
        if (harness.isIn(actor) && harness.isworn
            && harness.isAttachedTo(topOfCable_hook))
        {
            "You cautiously let go of the cable, letting your weight
            shift to the harness.  The harness easily holds you in
            place, letting you take hold of the wheel. ";

            inherited.doTurn(actor);
        }
        else
            "You can't seem to turn it with just one hand, and there's
            no way you can let go of the cable with both hands without
            something to keep you from falling. ";
    }
;

topOfCable_cable: fixeditem, attachableItem
    sdesc = "cable"
    noun = 'cable'
    location = topOfCable
    ldesc = "The cable is attached to a hook protruding from the
             roof of the dome. It extends to the catwalk twenty
             feet below. "
    verDoClimb(actor) = { }
    doClimb(actor) = { actor.travelTo(self.location.down); }
    doSynonym('Climb') = 'Climbdown'

    attachmentList = [topOfCable_hook]
    verifyAttachTo(actor, otherObj) =
    {
        if (!inherited.verifyAttachTo(actor, otherObj))
            return nil;

        /* we can never attach it to anything new */
        "It's already attached to the hook. ";
        return nil;
    }
    verifyDetachFrom(actor, otherObj) =
    {
        if (!inherited.verifyDetachFrom(actor, otherObj))
            return nil;
        
        /* we can never detach it */
        "The cable is under too much tension to remove from the hook. ";
        return nil;
    }
;

topOfDome_catwalk: distantItem
    sdesc = "catwalk"
    noun = 'catwalk'
    location = topOfCable
    ldesc = "The catwalk is twenty feet below. "
;

topOfCable_roof: fixeditem
    sdesc = "roof"
    noun = 'roof' 'tower'
    adjective = 'tower'
    location = topOfCable
    ldesc = "It's made of the same metal as the tower walls, and
             curves down around you. "
;


/* ------------------------------------------------------------------------ */
/*
 *   In tower - west end of catwalk 
 */
inTowerWest: inTowerRoom
    sdesc = "West end of catwalk"
    ldesc = "The catwalk reaches the west wall of the tower here,
             and extends into the center of the tower to the east.
             A door is set into the west wall; it's
             currently <<inTowerWest_door.isopen ? "open" : "closed">>. "
    east =
    {
        if (inTowerWest_door.isopen)
        {
            "The door slams shut with a thud. ";
            inTowerWest_door.setIsopen(nil);
        }
        return inTowerCenter;
    }
    west = inTowerWest_door
    out = inTowerWest_door
    enterRoom(actor) =
    {
        inherited.enterRoom(actor);

        if (inTowerWest_door.isopen)
        {
            "\bThe rounded metal door swings shut with a dull thud.\b";
            inTowerWest_door.setIsopen(nil);
        }
    }
;

inTowerWest_catwalk: inTower_catwalk_item
    location = inTowerWest
    ldesc = "The catwalk extends to the east. "
;

inTowerWest_cables: distantItem
    noun = 'cable' 'cables'
    location = inTowerWest
    sdesc = "cables"
    isThem = true
    ldesc = "The cables are attached to the catwalk near the center
             of the tower; you'd be able to get a better look from there. "
;

inTowerWest_door: inTowerDoorItem
    location = inTowerWest
    otherside = towerStair4West_door
    doordest = towerStair4West
    issealed = true
;


/* ------------------------------------------------------------------------ */
/*
 *   Roof of the tower 
 */
towerRoof: room
    sdesc = "Roof of tower"
    ldesc =
    {
        "This is the apex of the tower dome, which slopes
        down in all directions.  The air is cool and a steady
        wind flows past the dome.  An access hatch is at
        the top; it's <<towerRoof_hatch.isopen ?
        "precariously propped up and open, and wisps of
        green vapor drift out" :
        "closed">>.  Handholds set into the roof form a ladder
        down the west side of the dome.
        \n\tThe industrial wasteland of the plant complex stretches
        to the north, other towers delineated by colored lights
        twinkling in the distance, a vast spiderweb of pipes and
        conduits below. ";

        if (towerRoof_lightning.isIn(self))
            "\n\tA tongue of lightning from directly overhead is dancing
            around the center of the dome, its light brilliant
            and blinding. ";
    }
    down = towerRoof_hatch
    west =
    {
        /* if we're in the end-game, don't allow leaving the roof */
        if (towerStair4_guards.isIn(towerStair4))
        {
            "It's crawling with guards down there. ";
            return nil;
        }
        
        if (towerRoof_hatch.isopen)
        {
            "The precariously-balanced hatch slams shut as you leave.\b";
            towerRoof_hatch.setIsopen(nil);
        }
        
        "You climb down the handholds, following the curved dome as
        it becomes steeper and finally vertical.  The handholds continue
        down the side of the tower to a landing.\b";
        
        return towerStair4West;
    }
    teeterNoted = nil
    noexit =
    {
        "There's nothing to hold onto in that direction. ";
        return nil;
    }
    up =
    {
        "You're at the highest point already. ";
        return nil;
    }
    enterRoom(actor) =
    {
        inherited.enterRoom(actor);

        /* 
         *   get teeterwaller up here if he isn't already, as long as he's
         *   still following along 
         */
        if (actor != teeterwaller && teeterwaller.isFollowingMe
            && teeterwaller.isIn(inTowerCenter) && towerRoof_hatch.isopen)
        {
            "\bYou hear frightened muttering from the hatch.
            Mr.\ Teeterwaller's head appears in the opening. You reach
            down and help him up onto the roof. ";

            if (!self.teeterNoted)
            {
                "\b<q>Good job getting that hatch open,</q> Teeterwaller
                tells you. <q>That never would have occurred to me.</q> ";
                self.teeterNoted := true;
            }

            teeterwaller.travelTo(self);
        }

        /* score points the first time they make it here */
        maybeIncscore(10, 'opening the access hatch', self);
    }
    addGreenGas = "The gas quickly blows away in the breeze. "
    isGassy = { return towerStair_gas.isIn(self); }

    endGameState = 0
    roomGuardDaemon =
    {
        /*
         *   We're on the roof, and we're in the end-game.  Run the
         *   end-game operations.  
         */
        switch(self.endGameState)
        {
        case 0:
            "\bGuards take up positions at the top of the stairs below
            the dome, but for some reason do not follow you up the
            ladder. ";
            break;

        case 1:
            /* the gray-haired man arrives */
            "\bSomeone starts climbing the ladder from below.  For a moment
            you think it's one of the guards, but as the figure draws
            closer, you see that it's a tall, gray-haired man in a business
            suit.  He reaches the top of the ladder and
            walks toward you and Teeterwaller, his hands open, presumably
            to show he's not carrying a weapon.  His jacket flaps in the
            breeze; he reaches down to button it.
            \b<q>When I was a boy,</q> the man says in a deep, gravelly
            voice, <q>my family lived in a small town in Arizona.  One
            day I rode my bike out into the desert, as I often did, in
            search of dinosaur fossils or Indian relics.  After several
            hours I was still empty-handed and had mostly forgotten what
            I had set out to find anyway,
            so with the sun nearing the horizon I started heading home,
            when a glint in the eastern sky caught my eye.</q> ";
            
            /* move him here */
            grayHairedMan.moveInto(self);
            break;

        case 2:
            "\bThe gray-haired man continues. <q>I watched as the glint
            turned into a streak, and the streak turned into a fireball,
            all in the course of a few heartbeats.  The
            sight transfixed me and I barely recognized that I was
            directly in the path of whatever was falling from the sky,
            but some instinct made me dive under
            a nearby boulder, only an instant before the ground
            shattered from the impact.  I hid under my rock as debris
            rained down.</q> ";
            break;

        case 3:
            "\bThe man continues speaking to you. <q>My fear soon turned
            to curiosity and before long I made my way to the crash site.
            I expected a meteorite, which I pictured as some kind of red
            glowing sphere with miniature moon craters.  What I found
            was obviously not a meteorite, not even anything natural.
            But it wasn't anything man-made either.  It was the size of
            a small airplane, but had no wings; its shape was complex
            and irregular; and it was made of a dull
            gray material that looked like wet clay.</q> ";
            break;

        case 4:
            "\b<q>The wreckage was in a few large pieces, but some small
            bits were scattered around, and I pocketed a few as
            souvenirs.  I was about to start looking at the larger
            pieces, when a group of airplanes came over
            the horizon and made for the crash site.  Something made
            me return to my bicycle and pedal away as fast as I could.
            I took refuge behind some rocks up a hill half a mile away
            and watched as the planes disgorged men and machines by
            parachute.</q> ";
            break;

        case 5:
            "\b<q>I stayed for some time watching the men load the
            pieces of the fallen object onto trucks.  Nightfall forced
            me to abandon my hideout and go home, and when I
            returned the next day I could find nothing left of the
            crash; the crater had been filled and even the
            fragments of rock scattered hundreds of yards away had been
            swept into the sand.</q> ";
            break;

        case 6:
            "\b<q>I spent years looking at the fragments I had kept from
            the crash without recognizing their true nature.  A series
            of chance discoveries, however, led me to realize that the
            object that crashed was not some experimental military craft,
            as I had long imagined, and that indeed it was not even of
            this Earth.</q> ";
            break;

        case 7:
            "\b<q>My souvenirs were in fact products of a highly advanced
            technology, far beyond what was thought possible when I found
            them and beyond even what is commonly believed possible 
            today, and having discovered a few of their secrets I quickly
            learned more.  I realized at once the commercial potential
            of my discoveries and set about finding practical applications.
            You see around you here a small facet of the enterprise that
            we have built, dedicated to the steady improvement of our
            society and the world through the carefully controlled
            introduction of this technology.</q> ";
            break;

        case 8:
            "\b<q>I did not know until many years after the crash the fate
            of the rest of the object.  As our company grew, we began to
            have the resources necessary to cultivate relationships inside
            the government, and we eventually learned where the pieces
            recovered from the crash were being studied.  We have monitored
            the progress of the government's studies carefully, and it has
            been a source of constant frustration for us that these studies
            have not only been almost completely unproductive, but what
            little they have gleaned has been hidden behind a veil
            of military secrecy and national security.</q> ";
            break;

        case 9:
            "\b<q>We have for many years known that the government is
            incapable of putting this technology to proper use, and that
            we must take control of it for the good of mankind.
            Unfortunately, even our resources have proven insufficient
            to correct this problem, at least until now.  When we learned
            that the crash artifacts were to be relocated to a new
            facility, we saw our chance to intervene, taking what is
            rightfully and morally ours.</q>
            \bA lightning bolt strikes one of the nearby towers. ";
            break;

        case 10:
            "\bTeeterwaller, who up to now has been silently listening
            to the gray-haired man, speaks to him. <q><q>Rightfully
            and morally</q> yours,</q> he scoffs.  <q>You're just as bad
            as the government, hiding this technology for decades.
            You're worse, in fact, the way you dole it out a little at
            a time to maximize your profit at everyone else's expense.</q>
            \bThe gray-haired man shakes his head slightly. <q>Profit
            isn't our motive,</q> he says. <q>It is society's readiness
            to absorb change that dictates the rate of our disclosures.
            If we were to reveal all we knew at one time, the world would
            plunge into chaos.</q>
            \b<q>You'll never get away with this,</q> Teeterwaller says.
            <q>You actually think the government won't figure out who
            stole the artifacts?</q>
            \b<q>The government will never even suspect they were taken,</q>
            the gray-haired man responds dismissively. <q>We replaced the
            originals with inert duplicates.  The fools have never been able
            to learn anything from the originals anyway; they'll labor for
            fifty more fruitless years, then shrug, pack the whole mess
            off to a warehouse in Illinois, and move on to their next
            assignments.</q>
            \bThunder rumbles from the sky overhead. ";
            break;

        case 11:
            "\bThe gray-haired man continues. <q>You have something
            that belongs to us,</q> he says. <q>As you can plainly see,
            there is nowhere for you to go.  Despite your weak bargaining
            position, we can afford to be generous, so I would like to
            offer both of you positions as Vice Presidents in one of our
            many exciting subsidiary or affiliated corporations.  All
            you have to do is give me the disk.</q>
            \bA few bright flashes of lightning light the sky
            directly overhead. ";

            /* he's made the offer */
            grayHairedMan.hasMadeOffer := true;
            break;

        case 12:
            "\bThe gray-haired man looks at you expectantly.
            \bStatic electricity crackles through the sky overhead. ";
            break;

        case 13:
            "\bThe gray-haired man watches you expectantly. <q>The perks
            are very nice for Vice Presidents,</q> he says.
            \bYour hair feels like it's standing on end from the static
            charge in the air. ";
            break;

        case 14:
            "\bThe gray-haired man looks at you expectantly.
            \bThunder roars overhead.  Lightning flashes from cloud to
            cloud in the sky above you.  Suddenly, a tongue of lightning
            cuts through the air directly in front of you, striking the
            center of the dome.
            \bSomehow, the lightning bolt stays where it is, dancing
            a little around the top of the dome but continuing to
            discharge electricity. ";

            towerRoof_lightning.moveInto(self);
            break;

        case 15:
            "\bThe lightning bolt continues to dance around in tiny
            random circles at the top of the dome.  It's so bright
            that you can't look at it directly. ";
            break;

        case 16:
            "\bThe gray-haired man looks away from the lightning,
            which continues to discharge into the top of the dome. ";
            break;

        case 17:
            "\bThe gray-haired man yells something to you, but you
            can't hear it over the noise of the electrical discharge
            from the lightning. ";
            break;

        case 18:
            "\bTeeterwaller yells to you, his voice barely audible
            over the noise from the electrical discharge of the
            lightning, <q>The lightning!  It may activate the disk!</q> ";

            /* 
             *   loop back to state 15 (so make it 14, since we'll shortly
             *   increment it) 
             */
            self.endGameState := 14;
            break;
        }

        /* advance to the next state */
        self.endGameState++;
    }

    teeterIdleAction =
    {
        /* 
         *   if we're in the end-game, don't bother with any teeterwaller
         *   idle message, since we have plenty else going on 
         */
        if (towerRoof_guards.isIn(self))
            return;

        /* use the default teeterwaller idle message */
        inherited.teeterIdleAction;
    }

    roomLightning =
    {
        /*
         *   If we're in the end-game, suppress the normal lightning
         *   flashes. 
         */
        if (towerRoof_guards.isIn(self))
            return;

        /* show lightning for the roof */
        switch(rand(3))
        {
        case 1:
            "\bA bolt of lightning strikes the roof of one of the
            nearby towers. ";
            break;

        case 2:
            "\bA lightning bolt slices through the air and strikes the
            roof of the tower only a few feet away, leaving your hair
            standing on end and your skin tingling with electricity. ";
            break;

        case 3:
            "\bLightning flashes from the sky and strikes the roof of
            the next tower over. ";
            break;
        }
    }

    roomAction(actor, v, dobj, prep, iobj) =
    {
        if (v = throwVerb && prep = nil && dobj = lab7_disk
            && grayHairedMan.isIn(self))
        {
            /*
             *   They're throwing the disk off the roof during the
             *   end-game.  
             */
            "You throw the disk.  Teeterwaller and the gray-haired man
            follow the disk with their eyes as it arcs into the night
            sky and disappears into the distance.
            \b<q>I'm disappointed that we couldn't reach an agreement,</q>
            the gray-haired man says.  He motions to the guards, who come
            up onto the roof and arrest you.
            \bThe guards take you to their security office, and later turn
            you over to the local police, who arrest you on trespassing
            charges.  You and Teeterwaller tell your fantastic story of
            hijacked trucks and floating anti-gravity disks, but the police
            just ignore you.
            \bAfter a night in the local jail, the police release you,
            explaining that the company that owns the plant is not pressing
            charges.  In town, you and Teeterwaller find a mechanic to
            fix the car.  You return home, having missed your meeting.
            Life returns to normal.
            \bIn the months that follow, the subject of that strange night
            seems to become taboo, and you and Teeterwaller never again
            discuss it.  Eventually, Teeterwaller leaves the company;
            you hear later that he's working for one of those UFO magazines,
            writing bitter diatribes about government conspiracies and
            military cover-ups.
            \bYou often wonder what might have happened if you had escaped
            with the odd little black disk, but you feel fortunate enough
            to have made it through the experience unscathed.
            \b*** Your life has returned to normal.\ ***\b";

            /* award some final points */
            incscore(5, 'surviving your night of adventuring');

            exitOptions(true);
            abort;
        }
        else if (v = throwVerb && prep = nil && dobj != nil)
        {
            /*
             *   Throwing something off the roof.  Simply warn that it
             *   will fall off the roof, and don't allow it.  
             */
            "You'd rather not; it would fall to the ground and
            you probably wouldn't be able to find it again. ";
            exit;
        }
        else
            inherited.roomAction(actor, v, dobj, prep, iobj);
    }
;

towerRoof_lightning: fixeditem
    noun = 'lightning' 'tongue' 'bolt'
    adjective = 'lightning'
    sdesc = "lightning bolt"
    ldesc = "The lightning is arcing down from directly overhead and
             dancing around in random little circles near the center
             of the dome.  Its light is brilliant and blinding. "
    verIoPutIn(actor) = { }
    ioPutIn(actor, dobj) =
    {
        if (dobj = lab7_disk)
        {
            "You carefully hold out the disk, and very slowly move
            your hand toward the lightning bolt.  Your skin tingles
            with electricity as you get closer, and your hair starts
            to stand on end.
            \bA tiny tendril of brilliant white light leaps out from
            the lightning bolt and touches the disk.  You feel a jab
            and jerk your hand back, but you see that the disk is
            glowing with an intense blue light, and you start to feel
            light-headed and disoriented.  Teeterwaller grabs your
            arm, and the two of you rapidly ascend into the sky.
            The gray haired man reaches his hand out toward you and
            shouts something, but you're already too far away to
            hear what he's saying.
            \bThe wind picks up speed and changes direction several
            times as you climb, and you start drifting to the north.
            Soon the lights of the plant are behind you, and you're
            over open ground.  Silence falls as your own motion
            matches the wind speed and you move along at the same
            rate as the surrounding air.
            \b<q>I can't believe we made it out of there alive!</q>
            Teeterwaller says.  <q><i>This</i> is an amazing find,</q>
            he says, indicating the disk.
            \bYou start descending again, and you see that you're
            headed for the road&mdash;in fact, you can see Teeterwaller's
            car, right where you left it.  The disk's glow fades, and
            you settle gently to the ground near the car.
            \bHeadlights approach from down the road.  They draw nearer,
            and the vehicle starts slowing down and pulling over behind
            Teeterwaller's car.  You almost make a run for it, when you
            see that the vehicle is a van from a local television station.
            A young woman in a smart blue dress gets out from the passenger
            side, and a man in a flannel shirt carrying a video camera follows.
            \b<q>Excuse me,</q> the woman says as she approaches you.
            <q>We've had some reports of strange electrical activity in
            the sky around here, and I wondered if either of you have
            seen anything interesting tonight.</q>
            \bTeeterwaller smiles at you.  <q>Yes,</q> he says. <q>As
            a matter of fact we have.</q> ";

            "\b*** You have made history.\ ***\b";

            /* award the final points */
            incscore(20, 'escaping the plant with the disk');

            /* offer exit options */
            exitOptions(true);
            abort;
        }
        else
            "The lightning would destroy that. ";
    }
    ioSynonym('PutIn') = 'ThrowAt' 'ThrowTo' 'PutOn'
;

grayHairedMan: Actor
    sdesc = "gray-haired man"
    noun = 'man' 'suit'
    adjective = 'gray-haired' 'grey-haired' 'gray' 'grey' 'haired'
                'dark' 'business'
    ldesc = "He's a tall, athletic-looking man with gray hair.  His
             dark business suit is fluttering in the strong wind. "
    hasMadeOffer = nil
    actorAction(v, d, p, i) =
    {
        "He ignores you. ";
        exit;
    }
    disavow = "He ignores your question. "
    isHim = true
    verDoTellAbout(actor, iobj) = { "He does not respond. "; }
    verIoShowTo(actor) = { }
    ioShowTo(actor, dobj) =
    {
        if (dobj = lab7_disk)
        {
            if (self.hasMadeOffer)
                "<q>Yes, that's the property of ours that I would like
                returned.</q> ";
            else
                "<q>I'll get to that shortly,</q> he says. ";
        }
        else
            "He ignores you. ";
    }
    verIoGiveTo(actor) = { }
    ioGiveTo(actor, dobj) =
    {
        if (dobj = lab7_disk)
        {
            if (self.hasMadeOffer)
                "<q>Thank you,</q> he says, taking the disk.  He shakes
                your hand, <q>Welcome aboard, Mr.\ Vice President,</q>
                he says, then turns to Teeterwaller and welcomes him. ";
            else
                "<q>Well, you've saved me a lot of talking,</q> he says.
                <q>In appreciation of your cooperativeness, I'd like to
                make you both Vice Presidents in one of our many subsidiary
                or affiliated corporations.</q> ";

            "\bAs promised, you are immediately appointed a Vice President
            at AmCaustiCo, a maker of caustic chemical speciality products,
            and given a plush corner office overlooking a pleasant little
            woods a safe distance from the smoke-belching factories whose
            maintenance schedule you are assigned to oversee.
            \bYou find your compensation much more than comfortable and
            your duties agreeably undemanding, leaving you plenty of
            time to practice your golf and <q>network</q> with other top
            executives at the country club, trading stock tips and dirty
            jokes.  Years later, looking back,
            you know you made a good career decision that strange night,
            but you can't help wondering from time to time what might have
            happened if you'd somehow escaped with the odd little black
            disk.  It's an idle thought, though; you'd just have ended up
            like Teeterwaller, writing bitter diatribes for some little
            conspiracy-minded UFO magazine. ";

            "\b*** You have a pleasant but vaguely
            unsatisfying career.\ ***\b";

            /* award the final points */
            incscore(10, 'succeeding in business without really trying');

            exitOptions(true);
            abort;
        }
        else
            "He does not seem interested in that. ";
    }
;

towerRoof_guards: distantItem, guardsItem
    ldesc = "Guards are visible just below the rim of the dome, but they're
             not pursuing you onto the roof. "
;

towerRoof_ladder: fixeditem
    noun = 'ladder' 'handholds' 'holds' 'hold' 'hand-holds' 'rung' 'rungs'
    adjective = 'hand'
    sdesc = "handholds"
    isThem = true
    location = towerRoof
    ldesc = "The handholds form a ladder down the west side of the dome. "
    verDoClimb(actor) = { }
    doClimb(actor) = { actor.travelTo(self.location.west); }
    doSynonym('Climb') = 'Climbdown'
;

towerRoof_pipes: distantItem
    noun = 'pipe' 'pipes' 'conduits'
    location = towerRoof
    sdesc = "pipes"
    isThem = true
    ldesc = "The pipes are everywhere, covering the ground like
             matted vines. "
;

towerRoof_plant: distantItem
    noun = 'plant' 'complex'
    adjective = 'plant'
    location = towerRoof
    sdesc = "plant complex"
    ldesc = "The plant complex dominates the view to the north. "
;

towerRoof_tower: fixeditem
    noun = 'tower' 'roof' 'dome' 'light' 'lights'
    adjective = 'domed' 'flashing' 'red' 'orange' 'yellow'
    sdesc = "roof"
    location = towerRoof
    ldesc = "The roof forms a dome; you are in the center, at the top. "
;

towerRoof_towers: distantItem
    noun = 'tower' 'towers'
    adjective = 'other' 'next'
    location = towerRoof
    sdesc = "other towers"
    isThem = true
    ldesc = "Dozens of towers are visible from here, some taller than
             this one, some shorter. "
;

towerRoof_hatch: towerDoorItem
    noun = 'hatch'
    adjective = 'access' 'round'
    location = towerRoof
    ldesc = "It's a small round door with a wheel in the center;
             it's currently <<self.isopen ? "precariously propped
             up in the open position" : "closed">>. "
    experienced = true
    issealed = nil
    doordest =
    {
        "Even though you were able to make it out through the hatch, you
        don't think you'd be able to reach the cable from this direction. ";
        return nil;
    }
    otherside = topOfCable_hatch
;

towerRoof_wheel: towerDoorWheelItem
    location = towerRoof_hatch
;

/* ------------------------------------------------------------------------ */
/*
 *   Narrow passage through pipes, near canal bank
 */
narrowPassage: room, inPlantLightningRoom
    sdesc = "Narrow passage"
    ldesc = "The pipes are just barely far enough apart here for you
             to squeeze through to the northwest, but this is otherwise
             a dead end.
             \n\tA space between some pipes to the east affords a peephole
             view of a large courtyard. "
    enterRoom(actor) =
    {
        local wasseen := self.isseen;
        
        /* do the normal work */
        inherited.enterRoom(actor);

        /* if this is the first time in, trigger the movie */
        if (!wasseen)
        {
            "\b<q>Look!</q> Mr.\ Teeterwaller says, pointing through
            the opening in the pipes.
            \bThrough the opening, you can see several trucks parked
            in the courtyard&mdash;the same trucks that you saw hijacked.
            \bTwo people, wearing baggy white whole-body suits with
            hoods and gas masks, emerge from the back of one of
            the trucks, walking backwards, carrying something.  They
            continue out down a ramp at the back of the truck; two more
            people dressed the same way come into view, also helping to
            carry the object, which you can now see is a large shiny
            silver crate.  Two more people come out of the truck, then
            another two carrying the trailing end of the crate.  The
            eight slowly make their way down the ramp with the obviously
            heavy cargo.
            \bYou wonder what they could be removing from the trucks
            that requires such elaborate protective clothing.  You look
            at Teeterwaller.  He's watching intently, but, strangely,
            does not seem surprised by what he's seeing.
            \bAnother space-suited worker emerges from the truck, slowly
            waving a wand attached by a cord to a box that makes intermittent
            ticking sounds.
            \bYou feel a chill as you realize that the cargo must
            be radioactive.
            \bThe group and their load disappear behind the truck, and
            the clicking of the Geiger counter recedes into inaudibility.
            \bTeeterwaller continues watching for a few moments.
            <q>Interesting,</q> he finally says. ";

            /* mark the opening as seen (for the movie) */
            narrowPassage_space.isseen := true;
        }
    }
    nw = canalBank
;

narrowPassage_pipes: fixeditem, seethruItem
    sdesc = "pipes"
    adesc = "a pipe"
    noun = 'pipe' 'pipes'
    location = narrowPassage
    ldesc = "The pipes are very tightly spaced here, making this
             passage a tight squeeze.  A space between some of the
             pipes to the east affords a view of a courtyard on the
             other side of the pipes. "
    thrudesc = { narrowPassage_space.thrudesc; }
;

narrowPassage_trucks: distantItem
    sdesc = "truck"
    noun = 'truck' 'trucks'
    adjective = 'white'
    location = narrowPassage
    ldesc = "They're the same white, unmarked trucks that you saw
             hijacked earlier tonight. "
;

narrowPassage_space: fixeditem, seethruItem
    sdesc = "space"
    noun = 'space' 'opening' 'peephole' 'hole' 'gap'
    location = narrowPassage
    ldesc = { self.thrudesc; }
    isseen = nil
    thrudesc = "You have a limited view through the narrow space.
                The courtyard has several trucks parked in it&mdash;the
                same trucks that were hijacked. "
    doSynonym('Lookthru') = 'Lookin'
;

narrowPassage_courtyard: distantItem
    sdesc = "courtyard"
    noun = 'courtyard'
    adjective = 'large'
    location = narrowPassage
    ldesc = { narrowPassage_space.thrudesc; }
;


/* ------------------------------------------------------------------------ */
/*
 *   Safety station
 */
safetyStation: room, inPlantLightningRoom
    sdesc = "Safety station"
    ldesc =
    {
        "The forest of pipes opens up a little here into a large
        clear area.  In the center of the clearing stands a tall
        metal enclosure topped with a brightly-lit red sign:
        << safetyStation_sign.readdesc >>
        An opening in the pipes leads northeast, and another
        passage leads south.  You may also be able to squeeze
        through a narrow crawl-way to the southeast. ";

        if (fireHoseEnd.isIn(crawlway_gap) || atrium_fireHose.isIn(atrium))
            "\n\tA fire hose stretches out of the enclosure and
            through the crawl-way to the southeast. ";
    }
    ne = canalBank
    se =
    {
        "You have to crawl underneath a large pipe about three feet
        above the ground, but once you're past the pipe, you're able
        to stand again. ";
        return crawlway;
    }
    south = aboveAtrium
    fireHosePreTravelTo(rm) =
    {
        /* mention that the hose is coming with us */
        if (rm = crawlway)
        {
            "You pull the hose along with you, carefully holding onto
            it as you crawl under the pipe.\b";
        }
        else if (rm != nil)
        {
            "You carefully pull the fire hose along with you.  The reel
            unwinds, letting out the hose for you.\b";
        }
            
        /* allow it */
        return true;
    }
    doDropFireHose(actor) =
    {
        "As you let go of the end of the hose, the reel takes up the slack
        and winds the hose back onto the coil. ";
        fireHoseEnd.retractHose;
    }
;

safetyStation_crawlway: fixeditem
    noun = 'crawlway' 'crawl-way' 'opening'
    adjective = 'narrow'
    sdesc = "crawl-way"
    location = safetyStation
    verDoEnter(actor) = { }
    doEnter(actor) = { actor.travelTo(self.location.se); }
    ldesc = "The opening to the southeast is partially blocked by a large
             pipe about three feet off the ground, but it looks as though
             you could fit through. "
;

safetyStation_pipes: fixeditem
    noun = 'pipe' 'pipes' 'opening'
    location = safetyStation
    sdesc = "pipes"
    adesc = "a pipe"
    isThem = true
    ldesc = "The pipes are spaced further apart here, creating a
             clearing. "
;

safetyStation_sign: fixeditem, readable
    sdesc = "safety station sign"
    noun = 'sign'
    adjective = 'safety' 'station' 'brightly-lit' 'red'
    readdesc = "\b<center><table><tr>
             <td bgcolor='red'><font face='TADS-Sans' color=white>
             <b>&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;SAFETY
              STATION&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;
             </b></font></table></center>\b"
    ldesc = "It's a brightly-lit red sign on the top of the enclosure:
             << self.readdesc >>"
    location = safetyStation
;

safetyStation_enclosure: fixeditem, qcontainer
    sdesc = "enclosure"
    adesc = "an enclosure"
    noun = 'enclosure' 'station'
    adjective = 'safety' 'metal' 'tall'
    location = safetyStation
    ldesc =
    {
        "The enclosure is about seven feet tall and open on one
        side to provide access to the equipment within.
        A gray fire hose is coiled up neatly on a giant reel";

        if (fireHoseEnd.isIn(parserGetMe()))
            " (except for the last few feet, which you're holding)";
        ". ";

        if (itemcnt(self.contents) != 0)
        {
            "In addition, the enclosure contains "; listcont(self); ". ";
        }
    }
    doSynonym('Inspect') = 'Lookin' 'Search'
;

class duplicatedClothingItem: clothingItem
    everTaken = nil
    moveInto(obj) =
    {
        inherited.moveInto(obj);

        if (obj != teeterwaller && !obj.isIn(teeterwaller) && !self.everTaken
            && teeterwaller.isReachable(obj))
        {
            self.everTaken := true;

            "Mr.\ Teeterwaller picks up <<self.otherOne.thedesc>>.
            <q>Good idea,</q> he says. <q>This could come in handy.</q>\b";

            self.otherOne.moveInto(teeterwaller);
            self.otherOne.everTaken := true;
        }
    }
    doWear(actor) =
    {
        inherited.doWear(actor);
        
        if (actor != teeterwaller && teeterwaller.isIn(actor.location)
            && self.otherOne.isIn(teeterwaller)
            && !self.otherOne.isworn)
        {
            "\bMr.\ Teeterwaller puts his <<self.genericDesc>> on as well. ";
            self.otherOne.isworn := true;
        }
    }
    doUnwear(actor) =
    {
        inherited.doUnwear(actor);
        
        if (actor != teeterwaller && teeterwaller.isIn(actor.location)
            && self.otherOne.isIn(teeterwaller)
            && self.otherOne.isworn)
        {
            "\bMr.\ Teeterwaller removes his <<self.genericDesc>> as well. ";
            self.otherOne.isworn := nil;
        }
    }
;

class gasMaskItem: duplicatedClothingItem
    genericDesc = "gas mask"
    noun = 'mask' 'mouthpiece' 'filters' 'strap'
    plural = 'masks'
    adjective = 'gas' 'rubber'
    location = safetyStation_enclosure
    ldesc = "The mask has a pair of filters sticking out of the
             front, behind which is a rubber mouthpiece for breathing;
             a strap is meant to hold it on to the wearer's head. "
    doUnwear(actor) =
    {
        if (actor.location.isGassy)
            "You have no desire to remove the mask while in the presence
            of the acrid vapor. ";
        else
            inherited.doUnwear(actor);
    }
;

olderGasMask: gasMaskItem
    sdesc = "older gas mask"
    adesc = "an older gas mask"
    adjective = 'old' 'older'
    ldesc = "<<inherited.ldesc>> It looks old and worn, but
             appears to be in one piece. "
    otherOne = newerGasMask
;

newerGasMask: gasMaskItem
    sdesc = "newer gas mask"
    adjective = 'new' 'newer'
    ldesc = "<<inherited.ldesc>> It looks brand new. "
    otherOne = olderGasMask
;

fireHose: fixeditem
    noun = 'hose' 'reel' 'coil'
    adjective = 'fire' 'gray' 'giant' 'coiled'
    sdesc = "coiled fire hose"
    location = safetyStation_enclosure
    verDoTake(actor) =
    {
        if (fireHoseEnd.isIn(actor))
            "%You're% already holding the end of the hose. ";
    }
    doTake(actor) =
    {
        if (atrium_fireHose.isIn(atrium))
        {
            "You try pulling the hose, but it seems stuck.  Perhaps
            you could collect it from somewhere nearer the end. ";
            return;
        }
        else if (fireHoseEnd.location = nil)
        {
            "You take hold of the end of the fire hose, and unreel the hose
            a couple of feet.  The reel remains under tension as you pull
            it, but it allows you to unreel the hose. ";
        }
        else if (fireHoseEnd.isIn(crawlway_gap) && actor.isIn(inCage))
        {
            "You take hold of the nozzle, and pull it through the hole in
            the pipes.  The hose is under tension, but you have no trouble
            holding on to it. ";

            /* mark the hose as feeding through the hole */
            fireHoseEnd.cameThroughGap := true;

            /* move the middle piece of hose to the crawl-way area */
            fireHoseMiddle.moveInto(crawlway);
        }
        else
        {
            "You pick up the hose, give it a couple of tugs to get it
            unstuck, and start pulling it back in, the reel taking up
            the slack as you go.  After a little work, the end of the
            hose comes sliding back to you, and you take hold of it. ";
        }
            
        /* give the player the end of the hose */
        fireHoseEnd.moveInto(actor);

        /* 
         *   move all of the end parts into the end, so that they stay
         *   with the player as the player drags the hose around 
         */
        fireHoseEndPart.moveAllEndPartsInto(fireHoseEnd);
    }
    doSynonym('Take') = 'Pull'
    ldesc =
    {
        if (fireHoseEnd.isIn(crawlway_gap) || fireHoseEnd.cameThroughGap)
            "The hose runs from the coil off through the crawl-way
            to the southeast; next to the coil is a valve with
            a brass control wheel. ";
        else
        {
            "The hose is neatly coiled on a giant reel";
            if (fireHoseEnd.isIn(parserGetMe()))
                " (except for the last few feet, which you're holding)";
            "; next to the coil is a valve with a brass control wheel.
            The hose ends in a nozzle, attached to which is a lever. ";
        }
    }
;

fireHoseValve: fixeditem
    location = safetyStation_enclosure
    noun = 'valve' 'wheel'
    adjective = 'brass' 'control'
    sdesc = "control wheel"
    ldesc = "The valve is big, obviously meant to control a substantial
             flow of water.  At the top is a brass control wheel. "
    verDoOpen(actor) =
    {
        if (self.isopen)
            "The valve is already open. ";
    }
    doOpen(actor) =
    {
        "You turn the wheel counterclockwise until it stops. You can
        hear water rush through the valve for a few moments
        as the hose pressurizes. ";
        self.isopen := true;
    }
    verDoClose(actor) =
    {
        if (!self.isopen)
            "The valve is already closed. ";
    }
    doClose(actor) =
    {
        "You turn the wheel clockwise until it stops. The hose gradually
        depressurizes. ";
        self.isopen := nil;
    }
    verDoTurn(actor) = { }
    doTurn(actor) =
    {
        if (self.isopen)
            self.doClose(actor);
        else
             self.doOpen(actor);
    }
;

fireHoseEnd: item
    noun = 'hose' 'end'
    adjective = 'fire' 'hose'
    sdesc = "end of the fire hose"
    adesc = "the end of a fire hose"
    thedesc = "the end of the fire hose"
    cameThroughGap = nil
    doDrop(actor) =
    {
        /* let the room do the work */
        actor.location.doDropFireHose(actor);
    }
    retractHose =
    {
        /* 
         *   get rid of the end - only the main coil exists as long as the
         *   hose is coiled up 
         */
        self.moveInto(nil);

        /* move all of the end parts back into the coil */
        fireHoseEndPart.moveAllEndPartsInto(fireHose);

        /* get rid of the middle */
        fireHoseMiddle.moveInto(nil);
        fireHoseGlassMiddle.moveInto(nil);
        fireHoseEdgeMiddle.moveInto(nil);
        atrium_fireHose.moveInto(nil);

        /* clear any notion of where it came from */
        self.cameThroughGap := nil;

        /* we're not in the pipe any more */
        self.isInPipe := nil;
    }
    ldesc =
    {
        "At the end of the hose is a nozzle and a lever. ";
        if (self.isInPipe)
            "You're holding the hose inside the rusty pipe. ";
        else if (self.isIn(crawlway_gap))
            "The nozzle is sitting in the gap in the pipes. ";
    }
    verDoTakeOut(actor, iobj) =
    {
        if (iobj != canalInPlant_rustyPipe || !self.isInPipe)
            inherited.verDoTakeOut(actor, iobj);
    }
    verDoTake(actor) =
    {
        if (!self.isInPipe && !self.isIn(crawlway_gap))
            inherited.verDoTake(actor);
    }
    doTake(actor) =
    {
        if (self.isInPipe)
            canalInPlant_rustyPipe.ioTakeOut(actor, self);
        else if (self.isIn(crawlway_gap))
            crawlway_gap.ioTakeOut(actor, self);
        else
            inherited.doTake(actor);
    }
    doSynonym('Take') = 'Pull' 'Move'
    verDoPointAt(actor, iobj) = { }
    doPointAt(actor, iobj) =
    {
        if (iobj = parserGetMe())
            "It might be momentarily amusing, but you don't feel the
            need to soak yourself right now. ";
        else if (iobj = teeterwaller)
            "Remembering that your first job review is not too far off,
            you decide not to soak your boss. ";
        else if (iobj = canalInPlant_rustyPipe
                 || iobj = canalInPlant_rustyPipeSomething)
            canalInPlant_rustyPipe.ioPutIn(actor, self);
        else
            "You can't control the hose well enough to keep it pointed
            at anything. ";
    }
    doSynonym('PointAt') = 'SprayAt'
    doPutIn(actor, iobj) =
    {
        "You're having a hard enough time just dragging the hose around,
        especially since it's still under tension from its reel; you
        can't put it inside anything. ";
    }
    verDoPutOn(actor, iobj) =
    {
        "It's still under tension from its reel; it wouldn't stay
        on << iobj.thedesc >> for long. ";
    }
    verDoLookin(actor) = { }
    doLookin(actor) = { "That would be far too hackneyed. "; }
;

/*
 *   fire hose end parts start off as part of the coiled-up hose, but move
 *   to the end of the hose when the player is holding the end 
 */
class fireHoseEndPart: fixeditem
    location = fireHose
    adjective = 'fire' 'hose'
    verDoTake(actor) =
    {
        if (self.location = fireHose)
            fireHose.verDoTake(actor);
        else if (self.isIn(actor))
        {
            if (!fireHoseEnd.isInPipe)
                "You're already carrying the hose. ";
        }
        else
            inherited.verDoTake(actor);
    }
    doTake(actor) =
    {
        if (self.location = fireHose)
            fireHose.doTake(actor);
        else if (self.isIn(actor) && fireHoseEnd.isInPipe)
            canalInPlant_rustyPipe.ioTakeOut(actor, fireHoseEnd);
        else
            inherited.doTake(actor);
    }
    moveAllEndPartsInto(dest) =
    {
        local obj;
        
        /* 
         *   loop through all of the objects of this class and move them
         *   into the given destination 
         */        
        for (obj := firstobj(fireHoseEndPart) ; obj != nil ;
             obj := nextobj(obj, fireHoseEndPart))
            obj.moveInto(dest);
    }
;

fireHoseNozzle: fireHoseEndPart
    noun = 'nozzle'
    sdesc = "fire hose nozzle"
    ldesc = "The nozzle directs the flow of water from the hose.
             A lever is attached to the nozzle. "
    verDoTakeOut(actor, iobj) =
    {
        if (iobj != canalInPlant_rustyPipe || !fireHoseEnd.isInPipe)
            inherited.verDoTakeOut(actor, iobj);
    }
    verDoPutIn(actor, iobj) =
    {
        if (iobj != canalInPlant_rustyPipe)
            inherited.verDoPutIn(actor, iobj);
    }
    doPointAt -> fireHoseEnd
    doPutIn -> fireHoseEnd
;

/* add default handling for adding the green gas to a room */
modify room
    addGreenGas =
    {
        "Eventually the gas dissipates. ";
    }
;

silverBalloon: item
    noun = 'balloon' 'opening'
    adjective = 'silver' 'balloon' 'plastic' 'shiny'
    sdesc = "silver balloon"
    isInflated = nil
    isGassy = nil
    ldesc = "It's an ordinary balloon, made of shiny silver plastic.
             It's currently fully <<
             self.isInflated ? "inflated, and the opening is knotted
             to keep the air from escaping" : "deflated">>. "
    checkWearingMask(actor) =
    {
        return ((olderGasMask.isIn(actor) && olderGasMask.isworn
                 || (newerGasMask.isIn(actor) && newerGasMask.isworn)));
    }
    verDoInflate(actor) =
    {
        if (self.isInflated)
            "It's already as inflated as it can be. ";
        else if (self.checkWearingMask(actor))
        {
            "(with your lips)
            \nYou'd have a hard time fitting the balloon
            under your gas mask. ";

            if (handpump.isIn(actor))
                "If you want to inflate the balloon with the pump,
                though, just say so. ";
        }
    }
    doInflate(actor) =
    {
        "You hold the balloon to your lips and blow several lungfulls of
        air into it, expanding the silver plastic into an oblong ball.
        After you finish filling the balloon, you tie a knot in the
        opening to keep the air from escaping. ";

        self.setInflated(true);
    }

    verDoInflateWith(actor, iobj) =
    {
        if (self.isInflated)
            "It's already fully inflated. ";
        else if (iobj != nil && iobj != handpump)
            "You can't inflate <<self.thedesc>> with <<iobj.adesc>>. ";
    }
    doInflateWith(actor, iobj) =
    {
        "You put <<iobj.thedesc>> into the balloon's opening and
        pump up the balloon";

        if (actor.location.isGassy)
        {
            self.isGassy := true;
            silverBalloon_gas.moveInto(self);
            ", filling the balloon with the toxic green gas. ";
        }
        else
            ", expanding the silver plastic into an oblong ball. ";

        "Once the balloon is inflated, you tie a knot in the opening
        to keep the <<self.isGassy ? "fumes" : "air">> from escaping. ";

        self.setInflated(true);
    }

    verDoTie(actor) =
    {
        if (self.isInflated)
            "The balloon already has a knot in the opening. ";
        else
            "There's no reason to do that until you inflate it. ";
    }

    verDoUntie(actor) =
    {
        if (self.isInflated)
            silverBalloon_knot.verDoUntie(actor);
        else
            "The balloon has no knot. ";
    }
    doUntie(actor) =
    {
        silverBalloon_knot.doUntie(actor);
    }

    doSynonym('Untie') = 'Open'

    verDoDeflate(actor) =
    {
        if (!self.isInflated)
            "It's already deflated. ";
    }
    doDeflate(actor) =
    {
        if (self.isGassy)
        {
            if (self.checkWearingMask(actor))
            {
                /* show the generic message */
                "You carefully untie the knot, allowing the green gas
                to escape.  Your gas mask protects you as the toxic vapors
                swirl around you. ";

                /* do any location-specific work */
                actor.location.addGreenGas;

                /* the balloon is no longer inflated */
                self.setInflated(nil);
            }
            else
                "You have no desire to expose yourself to the toxic
                contents of the balloon without some sort of protection. ";
        }
        else
        {
            "You carefully untie the knot and let the air escape. ";
            self.setInflated(nil);
        }
    }

    setInflated(status) =
    {
        if (status)
        {
            self.isInflated := true;
            silverBalloon_knot.moveInto(self);
        }
        else
        {
            self.isInflated := nil;
            self.isGassy := nil;
            silverBalloon_gas.moveInto(nil);
            silverBalloon_knot.moveInto(nil);
        }
    }

    verDoLookin(actor) = { }
    doLookin(actor) =
    {
        if (self.isInflated)
            "The balloon is inflated<<
            self.isGassy ? " with toxic green gas" : "">>. ";
        else
            "The balloon is empty. ";
    }
    doSynonym('Lookin') = 'Search'

    showUsage =
    {
        "If you want to inflate the balloon with the pump, just
        say so. ";
    }
    verDoAttachTo(actor, iobj) =
    {
        if (iobj = handpump)
            self.showUsage;
        else
            "You see no way to do that. ";
    }
    verIoAttachTo(actor) = { }
    ioAttachTo(actor, dobj) =
    {
        if (dobj = handpump)
            self.showUsage;
        else
            "You see no way to do that. ";
    }
;

silverBalloon_gas: fixeditem
    noun = 'gas' 'fume' 'fumes' 'vapor' 'vapors' 'quantity'
    adjective = 'viscous' 'green' 'acrid'
    sdesc = "quantity of green vapor"
    ldesc = "It's inside the balloon. "
    dobjGen(a, v, i, p) =
    {
        "The quantity of gas is inside the balloon.  You'll need to open
        the balloon to release the gas. ";
        exit;
    }
    iobjGen(a, v, i, p) = { self.dobjGen(a, v, i, p); }
    verDoRelease(actor) = { }
    doRelease(actor) =
    {
        "If you want to release the gas, just open the balloon. ";
    }
;

silverBalloon_knot: fixeditem
    noun = 'knot'
    adjective = 'silver' 'balloon' 'opening'
    sdesc = "balloon opening knot"
    ldesc = "It's a knot in the balloon's opening, to keep the air
             from escaping the balloon. "
    verDoTie(actor) = { "The knot is already tied. "; }
    verDoUntie(actor) = { self.location.verDoDeflate(actor); }
    doUntie(actor) = { self.location.doDeflate(actor); }
    doSynonym('Untie') = 'Open' 
;

fireHoseLever: fireHoseEndPart
    noun = 'lever'
    sdesc = "fire hose lever"
    ldesc = "The lever is attached to the nozzle. "
    verDoPull(actor) = { }
    doPull(actor) =
    {
        if (fireHoseValve.isopen)
        {
            /* if we're in the rusty pipe, flush out the pipe's contents */
            if (fireHoseEnd.isInPipe)
            {
                "You pull the lever, and water immediately starts gushing
                through the nozzle and into the pipe.  The pipe quickly
                fills up, and the water starts shooting back out.  You
                release the lever and jump out of the way as the water
                pours out of the pipe. ";

                /* if we haven't already found the object, do so now */
                if (!canalInPlant_rustyPipe.isObjFound)
                {
                    "A deflated silver balloon falls out of the pipe, carried
                    by the force of the water. ";

                    /* remove the hose from the pipe, as described */
                    fireHoseEnd.isInPipe := nil;

                    /* move the object here */
                    silverBalloon.moveInto(actor.location);

                    /* get rid of the 'something' */
                    canalInPlant_rustyPipeSomething.moveInto(nil);

                    /* note that it's been found and award points */
                    canalInPlant_rustyPipe.isObjFound := true;
                    incscore(10, 'getting the balloon out of the pipe');
                }
                return;
            }

            /* if we're in the gap outside the atrium, spray the glass */
            if (fireHoseEnd.isIn(crawlway_gap))
            {
                "You pull the lever, and water immediately starts
                gushing through the nozzle, spraying all over the glass.
                You release the lever, and the water stops as a spring
                pulls the lever back to the normal position. ";
                inCage_glass.sprayWithWater;

                return;
            }

            /* we're nowhere special; do the normal aimless spraying */
            "You pull the lever, and water immediately starts gushing
            through the nozzle at such pressure that you can't control
            where the hose is pointing. ";

            switch(actor.location)
            {
            case canalInPlant:
                "The water sprays randomly, mixing with the canal
                water. ";
                break;

            case canalBank:
                "The water sprays everywhere. ";
                break;

            case safetyStation:
                "The water shoots randomly. ";
                break;

            case aboveAtrium:
            case inCage:
                "The water sprays all over, and makes a mess of the
                glass. ";
                inCage_glass.sprayWithWater;
                "\b";
                break;
            }

            "You let go of the lever, and the water stops spraying as
            a spring pulls the lever back to its normal position. ";
        }
        else
            "Nothing happens.  You let go of the lever, and a spring
            pulls it back to its normal position. ";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   crawlway 
 */
crawlway: room, inPlantLightningRoom, noFloorRoom
    sdesc = "Dead end"
    ldesc =
    {
        "This opening in the pipes is large enough to stand in,
        but not much more.  To the northwest is a low crawl-way,
        but there are no other openings large enough to travel through.
        The dense wall of pipes to the south is, however, broken in one
        spot by a six-inch gap. ";

        if (fireHoseEnd.isIn(crawlway_gap) || atrium_fireHose.isIn(atrium))
            "\n\tThe fire hose stretches through the crawl-way to
            the northwest; the nozzle is shoved into the gap in
            the pipes to the south. ";
        else if (fireHoseEnd.isIn(self))
            "\n\tThe fire hose stretches through the crawl-way to
            the northwest. ";
    }
    fireHosePreTravelTo(rm) =
    {
        /* mention that the hose is coming with us */
        if (rm != nil)
            "The hose retracts as you travel back toward the reel.\b";
            
        /* allow it */
        return true;
    }
    doDropFireHose(actor) =
    {
        "As soon as you let go of the hose, it slides back through
        the crawl-way as the reel winds the hose back onto the coil. ";
        fireHoseEnd.retractHose;
    }
    nw = safetyStation
;

fireHoseMiddle: fixeditem
    noun = 'hose'
    adjective = 'fire'
    sdesc = "fire hose"
    ldesc = "The fire hose is under considerable tension; it's stretched
             from the crawl-way to the northwest and out through the
             gap in the pipes to the south. "
    verDoTake(actor) = { }
    doTake(actor) =
    {
        "The hose seems snagged on something beyond the gap.  Perhaps
        you could reel it back in from someplace nearer the nozzle end. ";
    }
    doSynonym('Take') = 'Pull' 'Move'
;

crawlway_gap: fixeditem
    isqcontainer = true
    noun = 'gap' 'hole'
    adjective = 'six-inch'
    sdesc = "six-inch gap"
    location = crawlway
    ldesc =
    {
        "The gap is about six inches high and a little wider.  Beyond
        the gap, there appears to be a large open area with a glass floor. ";
        if (fireHoseEnd.isIn(self) || atrium_fireHose.isIn(atrium))
            "The fire hose has been shoved into the gap. ";
    }
    verIoPutIn(actor) = { }
    ioPutIn(actor, dobj) =
    {
        if (dobj = fireHoseEnd)
        {
            if (dobj.isIn(self))
                "It's already in the gap. ";
            else
            {
                "You shove the nozzle into the gap.  It's a tight squeeze,
                but it fits through.  The lever on the side of the nozzle
                catches slightly on the pipes, holding the nozzle in place
                as you let go of the hose. ";

                fireHoseEnd.moveInto(self);
                aboveAtrium_fireHoseNozzle.moveInto(aboveAtrium);
                inCage_fireHoseNozzle.moveInto(inCage_gap);
            }
        }
        else
            "\^<<dobj.thedesc>> would just fall through to the other
            side, and you don't want to lose <<dobj.itobjdesc>>. ";
    }
    ioSynonym('PutIn') = 'PutThru'
    verIoTakeOut(actor) = { }
    ioTakeOut(actor, dobj) =
    {
        if ((dobj = fireHoseEnd || dobj = fireHoseNozzle)
            && fireHoseEnd.isIn(self))
        {
            "You remove the nozzle from the gap, keeping hold of
            the end of the hose. ";
            fireHoseEnd.moveInto(actor);
        }
        else
            "That's not in the gap. ";
    }
    Grab(obj) =
    {
        if (obj = fireHoseEnd)
        {
            aboveAtrium_fireHoseNozzle.moveInto(nil);
            inCage_fireHoseNozzle.moveInto(nil);
            atrium_fireHose.moveInto(nil);
        }
    }
    doSynonym('Inspect') = 'Lookin' 'Lookthru' 'Search'
;

crawlway_openArea: distantItem
    noun = 'area'
    adjective = 'large' 'open'
    location = crawlway
    sdesc = "open area"
    adesc = "an open area"
    ldesc = "The area on the other side of the gap is open, not covered
             with a tangle of pipes like this side.  The floor of the
             area is made of glass. "
;

crawlway_glass: distantItem
    noun = 'glass' 'window' 'floor'
    adjective = 'glass'
    location = crawlway_openArea
    sdesc = "glass floor"
    ldesc = "The gap does not provide a very good view.  The glass
             looks like a giant window set into the ground, stretching
             a hundred feet to the south. "
;

crawlway_pipes: fixeditem
    noun = 'pipe' 'pipes'
    sdesc = "pipes"
    adesc = "a pipe"
    isThem = true
    location = crawlway
    ldesc = "The pipes are the same as all of the others in this place.
             A low opening to the northwest provides a crawl-way. "
;

crawlway_crawlway: fixeditem
    noun = 'crawlway' 'crawl-way' 'opening'
    adjective = 'narrow' 'low'
    sdesc = "crawlway"
    location = crawlway
    verDoEnter(actor) = { }
    doEnter(actor) = { actor.travelTo(self.location.nw); }
    ldesc = "The opening is only about three feet high. "
;
    

/* ------------------------------------------------------------------------ */
/*
 *   At the edge of the glass ceiling above the atrium 
 */
aboveAtrium: room, inPlantLightningRoom
    sdesc = "North edge of glass"
    ldesc =
    {
        "The ground to the south is covered with a huge array
        of glass panels, stretching hundreds of feet to the
        south and as far to the east.  The pipes that
        are everywhere in this compound bend around the sea of
        glass leaving the area above open to the sky; white
        cylindrical towers rising up above the pipes are scattered
        around the perimeter of the glass.
        A passage through the pipes opens to the north, and the pipes
        leave enough space for a walkway southwest along the edge
        of the glass.
        \n\tA metal cage, suspended a few inches off the ground
        by a cable hanging from a beam overhead, is just this side
        of the glass. ";

        if (aboveAtrium_accessHatch.isIn(self))
            "\n\tRoughly in the center of the glass, one of the
            panes has been tilted up, creating an opening in the glass. ";
    }
    fireHoseDesc =
    {
        if (fireHoseEnd.isIn(self))
            "\n\tThe fire hose stretches along the ground to the north. ";
        else if (fireHoseEnd.isIn(crawlway_gap))
            "\n\tAbout fifty feet to the east, you can see the
            nozzle of the fire hose sticking out through the pipes. ";
        else if (fireHoseEnd.cameThroughGap)
            "\n\tA fire hose is sticking through the pipes
            about fifty feet to the east; the hose stretches across
            the glass and disappears down through the access hatch. ";
    }
    nrmLkAround(verbosity) =
    {
        /* do the normal work first */
        inherited.nrmLkAround(verbosity);

        /* mention the fire hose */
        self.fireHoseDesc;
    }
    north = safetyStation
    south =
    {
        "You'd fall through the glass. ";
        return nil;
    }
    se = { return self.south; }
    sw = outsideElevator
    east =
    {
        "Although the pipes bend around the glass, they run right up
        to the edge, leaving no room to walk to the east. ";
        return nil;
    }
    in = inCage
    fireHosePreTravelTo(rm) =
    {
        /* mention that the hose is coming with us */
        if (rm = safetyStation)
            "The hose retracts as you travel back toward the reel.\b";
        else if (rm = outsideElevator)
        {
            "The hose snags on a pipe as you try to get it around
            the bend.  You'll have to drop the hose before you go
            this way. ";
            return nil;
        }
        else if (rm != nil)
            "You carefully pull the fire hose along with you.  The reel
            unwinds, letting out the hose for you.\b";
            
        /* allow it */
        return true;
    }
    doDropFireHose(actor) =
    {
        "As soon as you let go of the hose, it slides away to the north
        as the reel winds the hose back onto the coil. ";
        fireHoseEnd.retractHose;
    }
    enterRoom(actor) =
    {
        /* do the default work */
        inherited.enterRoom(actor);

        /* 
         *   For simplicity, always bring the cage back here whenever they
         *   enter the room.  (This simplifies our state tracking, in that
         *   we can always describe this room as having the cage right
         *   here.)  We won't bother trying to explain this; perhaps one
         *   of the window washing crews is running around parking it back
         *   here periodically while we're off exploring.
         */
        inCage.xcoord := 10;
        inCage.ycoord := 0;
    }
;

fireHoseEdgeMiddle: distantItem
    noun = 'hose'
    adjective = 'fire'
    sdesc = "fire hose"
    ldesc = "The fire hose is stretching from the pipes to the east,
             over the glass, to the open panel. "
;

aboveAtrium_accessHatch: distantItem
    sdesc = "access hatch"
    adesc = "an access hatch"
    noun = 'pane' 'hatch' 'panel' 'glass' 'opening'
    adjective = 'open' 'access'
    ldesc =
    {
        "The pane has been tilted up, creating an opening in the glass. ";
        if (fireHoseEnd.cameThroughGap)
            "A fire hose stretches from the pipes, and runs over the glass
            to the hatch, where it disappears down into the opening. ";
    }
;

aboveAtrium_fireHoseNozzle: distantItem
    sdesc = "fire hose nozzle"
    noun = 'nozzle' 'hose'
    adjective = 'fire' 'hose'
    ldesc = "The nozzle is sticking through the pipes about
             fifty feet to the east. "
;

aboveAtrium_cage: fixeditem
    sdesc = "metal cage"
    noun = 'cage' 'floor' 'pole' 'poles' 'framework' 'cable'
    adjective = 'metal'
    location = aboveAtrium
    ldesc = "The cage isn't the kind that's meant to keep someone from
             getting out; it's simply a metal floor about three feet
             by eight feet with a loose framework of metal poles forming
             the edges of a box; a cable attached at the top of the
             framework holds the whole thing a few inches off the ground. "
    verDoPush(actor) = { }
    doPush(actor) =
    {
        "The cage is very heavy; you can manage to make it swing a little,
        but that's about all. ";
    }
    doSynonym('Push') = 'Pull' 'Move'
    verDoBoard(actor) = { }
    doBoard(actor) =
    {
        actor.travelTo(inCage);
    }
    doSynonym('Board') = 'Enter' 'Standon'
    verDoClimb(actor) = { "That would be too dangerous. "; }
    verIoPutIn(actor) = { }
    ioPutIn(actor, dobj) =
    {
        "You'd best hold onto it; the platform isn't all that stable,
        and <<dobj.thedesc>> would probably get lost. ";
    }
    ioSynonym('PutIn') = 'PutOn'
    verDoLookin(actor) =
    {
        "You'd have an easier time doing that if you got into the cage. ";
    }
    doSynonym('Lookin') = 'Search'
;

aboveAtrium_beam: distantItem
    sdesc = "overhead beam"
    noun = 'beam'
    adjective = 'overhead'
    location = aboveAtrium
    ldesc = "The beam is forty or fifty feet up, so you can't see much
             more than that the cable holding the cage is suspended
             from the beam. "
;

aboveAtrium_towers: distantItem
    noun = 'tower' 'towers'
    adjective = 'white' 'cylindrical'
    location = aboveAtrium
    sdesc = "tower"
    ldesc = "The towers are cylindrical; they're surrounded by pipes,
             especially near the bottom, but a few pipes snake up the
             sides and enter the towers near the top. "
;

aboveAtrium_pipes: fixeditem
    noun = 'pipe' 'pipes'
    sdesc = "pipes"
    adesc = "a pipe"
    isThem = true
    location = aboveAtrium
    ldesc = "The pipes are the same type as everywhere else around
             here.  They bend around the glass, leaving the area above
             the panels open. "
;

class atriumGlassItem: fixeditem, seethruItem
    noun = 'glass' 'panel' 'panels' 'pane' 'panes' 'windows' 'array'
    adjective = 'glass'
    sdesc = "glass"
    ldesc =
    {
        "The glass looks like a giant window set into the
        ground, stretching a hundred feet to the south and
        as far to the east.
        It looks as though a huge chamber is below the glass. ";

        if (aboveAtrium_accessHatch.isIn(aboveAtrium))
        {
            "\bThe pane in the middle of the glass has been tilted up,
            creating an access hatch. ";

            if (fireHoseEnd.cameThroughGap)
                "A fire hose has been fed through the opening, and
                stretches to the pipes. ";
        }
    }
    thrudesc = "It looks as though a huge chamber is below the glass,
                which means that the glass is the chamber's ceiling.
                You can't see any details; the floor of the chamber
                must be a hundred feet or more below. "
    verDoBreak(actor) = { }
    doBreak(actor) =
    {
        "That would create too much of a disturbance; you're afraid
        you'd be noticed. ";
    }
;

aboveAtrium_glass: atriumGlassItem
    location = aboveAtrium
;

aboveAtrium_chamber: distantItem
    noun = 'chamber' 'room'
    adjective = 'underground'
    sdesc = "underground chamber"
    adesc = "an underground chamber"
    location = aboveAtrium
    ldesc = "You can't see any details from here; the floor of the
             chamber must be over a hundred feet below. "
    verDoEnter(actor) = { "You see no way to do that from here. "; }
;

/* ------------------------------------------------------------------------ */
/*
 *   Edge of glass, outside elevator 
 */
outsideElevator: room, inPlantLightningRoom
    sdesc = "West edge of glass"
    ldesc = "This is a narrow walkway along the west edge of an array
             of glass panels covering the ground to the east.  A wall
             of pipes parallels the edge of the glass a few feet back
             from the edge, blocking travel except to the northeast.
             Midway along the wall of pipes is an elevator door. "
    ne = aboveAtrium
    west = outsideElevator_door
    in = outsideElevator_door
    east =
    {
        "You'd fall through the glass. ";
        return nil;
    }
    se = { return self.east; }
;

outsideElevator_glass: fixeditem, seethruItem
    noun = 'glass' 'panel' 'panels' 'pane' 'panes' 'windows' 'array'
    adjective = 'glass'
    sdesc = "glass"
    location = outsideElevator
    ldesc = "The glass looks like a giant window set into the ground,
             stretching a hundred feet to the east. "
    thrudesc = "It looks as though a huge chamber is below the glass,
                which means that the glass is the chamber's ceiling.
                You can't see any details; the floor of the chamber
                must be a hundred feet or more below. "
    verDoBreak(actor) = { }
    doBreak(actor) =
    {
        "That would create too much of a disturbance; you're afraid
        you'd be noticed. ";
    }
;

outsideElevator_chamber: distantItem
    noun = 'chamber' 'room'
    adjective = 'underground'
    sdesc = "underground chamber"
    adesc = "an underground chamber"
    location = outsideElevator
    ldesc = "You can't see any details from here; the floor of the
             chamber must be over a hundred feet below. "
    verDoEnter(actor) = { "You see no way to do that from here. "; }
;

outsideElevator_cage: distantItem
    sdesc = "metal cage"
    noun = 'cage' 'floor' 'pole' 'poles' 'framework' 'cable'
    adjective = 'metal'
    location = outsideElevator
    ldesc = "The cage is suspended over the glass some distance to
             the east. "
;

outsideElevator_door: elevatorDoorItem
    noun = 'elevator'
    sdesc = "elevator"
    adesc = "an elevator"
    location = outsideElevator
    ldesc = "The elevator has a sliding metal door, which is <<
             self.isopen ? "open" : "closed">>, and a round button
             just to the right of the door. "
    doordest = inElevator
;

outsideElevator_button: elevatorButtonItem
    adjective = 'elevator' 'round'
    sdesc = "elevator button"
    adesc = "an elevator button"
    ldesc = "It's a small round button next to the door. <<self.litDesc>> "
    buttonDest = outsideElevator
    location = outsideElevator
;

outsideElevator_pipes: fixeditem
    noun = 'pipe' 'pipes' 'wall'
    sdesc = "pipes"
    adesc = "a pipe"
    isThem = true
    location = outsideElevator
    ldesc = "The pipes are broken here only by a passage to the northeast
             and by the elevator to the west. "
;    

/* ------------------------------------------------------------------------ */
/*
 *   In the cage 
 */
inCage: room, inPlantLightningRoom, noFloorRoom
    sdesc = "In the cage"
    ldesc =
    {
        "The cage is a simple framework of poles connected to
        a metal floor.  Attached to one of the poles, about midway
        up, is a pair of levers. ";

        /* note the note */
        if (inCage_note.isTaped)
            "A few inches above the levers, a yellow note is
            taped to the pole. ";
        
        /* report our position */
        if (self.ycoord = 0 && self.xcoord = 10)
            "\n\tThe cage is above solid ground, at the north edge of the
            glass, and about << spellOutNumber(self.xcoord) >> feet
            from the west edge. ";
        else if (self.ycoord = 0)
            "\n\tThe cage is at the north edge of the glass, and
            about <<spellOutNumber(self.xcoord)>> feet from the
            west edge of the glass. ";
        else
            "\n\tThe cage is hanging over the glass,
            about << spellOutNumber(self.ycoord) >> feet from the north
            edge, and about << spellOutNumber(self.xcoord) >> feet from
            the west edge. ";
    }
    fireHoseDesc =
    {
        /* report the open hatch */
        if (inCage_glass.isopen)
            "\n\tThe glass pane roughly in the center of the glass
            has been tilted up, creating an opening in the glass. ";

        /* note the fire hose */
        if (fireHoseEnd.cameThroughGap)
        {
            if (fireHoseEnd.isIn(self))
                "\n\tThe fire hose stretches off through a gap in the
                pipes to the north. ";
            else
                "A fire hose stretches between a gap in the pipes
                to the north and the open panel. ";
        }
        else if (fireHoseEnd.isIn(self))
            "\n\tThe fire hose stretches off to the north. ";
        else if (fireHoseEnd.isIn(crawlway_gap))
            "\n\tThe nozzle of the fire hose is sticking through
            a gap in the pipes<<inCage_fireHoseNozzle.cageRelDesc>>. ";
        else
            "\n\tA small gap in the pipes is visible<<
            inCage_fireHoseNozzle.cageRelDesc>>. ";
    }
    nrmLkAround(verbosity) =
    {
        /* do the normal work first */
        inherited.nrmLkAround(verbosity);

        /* mention the fire hose */
        self.fireHoseDesc;
    }
    xcoord = 10
    ycoord = 0
    up =
    {
        "There's no way up. ";
        return nil;
    }
    noexit =
    {
        "You'd can't walk on the glass, since it would not support
        your weight. ";
        return nil;
    }
    out =
    {
        if (self.ycoord != 0)
        {
            "The cage is <<spellOutNumber(self.ycoord)>> feet from the
            edge of the glass.  You can't walk on the glass, since it
            would not support your weight. ";
            return nil;
        }
        else if (self.xcoord != 10)
        {
            "The pipes are too close to the edge of the glass
            here&mdash;there's no place to stand except on the glass,
            which you'd fall through.  You need to go
            about <<spellOutNumber(self.xcoord - 10)>> feet west
            to get back to solid ground. ";
            return nil;
        }
        else
            return aboveAtrium;
    }
    north = { return self.out; }
    leverMove(prop, dir, dirname) =
    {
        local amt;
        local oldx := self.xcoord;
        local oldy := self.ycoord;
        local dest;

        /* multiply by the per-move increment */
        amt := dir * 25;
        
        /* try making the adjustment */
        self.(prop) += amt;

        /* see if it's valid */
        if (self.ycoord < 0 || self.ycoord > 90
            || self.xcoord < 10 || self.xcoord > 100)
        {
            /* it's not valid - undo the change */
            self.(prop) -= amt;

            /* report that nothing happened */
            "Nothing seems to happen.  You release the lever, and a
            spring returns it to the upright position. ";
        }
        else
        {
            /* report the start */
            "With a lurch, the cage starts moving to the << dirname >>. ";

            /* drop the hose if necessary */
            if (fireHoseEnd.isIn(self))
            {
                if ((self.xcoord > 35 && !fireHoseEnd.cameThroughGap)
                    || (self.xcoord < 35 && fireHoseEnd.cameThroughGap))
                {
                    "\bThe fire hose suddenly snags on a pipe and refuses
                    to go any further.  You scramble to keep your hold on
                    the hose, but the cage's motor is too strong and forces
                    you to let go.  The nozzle luckily doesn't break the
                    glass, but skitters away as the coil retracts
                    the hose.\b";

                    /* lose the hose */
                    fireHoseEnd.retractHose;
                }
            }

            /* we're there */
            "The cage travels about twenty-five feet, then stops,
            leaving it ";

            /* if we're newly back over the ground, say so */
            if (self.ycoord = 0 && self.xcoord = 10)
                "back over solid ground. ";
            else if (self.ycoord = 0)
                "about << spellOutNumber(self.xcoord) >> feet
                east of the west edge of the glass. ";
            else
                "hanging over the glass,
                about << spellOutNumber(self.ycoord) >> feet south of the
                north edge, and << spellOutNumber(self.xcoord) >> feet east
                of the west edge. ";

            /* finish the description */
            "You release the lever, and
            a spring returns it to the upright position; after a few
            wide swings, the cage gradually settles into its new location. ";

            /* check for fire hose following */
            if (fireHoseEnd.isIn(self))
                "\bYou manage to hold on to the fire hose, which
                drags along the ground as the cage moves. ";

            /* activate or deactivate the openable panel parts */
            dest := ((self.xcoord = 60 && self.ycoord = 50)
                     ? inCage : nil);
            inCage_handle.moveInto(dest);
            inCage_hinge.moveInto(dest);

            /* 
             *   check for a special message about the open pane and/or
             *   fire hose 
             */
            if (dest = self)
            {
                if (inCage_glass.isopen)
                {
                    "\bThe glass pane just outside the cage has been
                    tilted up, creating an opening the glass. ";

                    if (fireHoseEnd.cameThroughGap
                        && !fireHoseEnd.isIn(self))
                        "A fire hose stretches between a gap in the pipes
                        and the open panel. ";
                }
            }
        }
    }
    fireHosePreTravelTo(rm) =
    {
        /* mention that the hose is coming with us */
        if (rm = aboveAtrium)
            "The hose retracts as you move closer to the safety station.\b";
        else if (rm != nil)
            "You drag the fire hose along with you.\b";
            
        /* allow it */
        return true;
    }
    doDropFireHose(actor) =
    {
        "As soon as you let go of the hose, the cage swings away as the
        tension is released, and the hose slides away to the north
        as the reel pulls the hose back in. ";
        fireHoseEnd.retractHose;
    }
    teeterwallerPreFollow(rm) =
    {
        /* we've already mentioned teeter while climbing */
        if (rm != atrium)
            inherited.teeterwallerPreFollow(rm);
    }
    teeterwallerPostFollow =
    {
        inherited.teeterwallerPostFollow;
        if (teeterwaller.location = atrium)
            "\bAfter a few minutes, Mr.\ Teeterwaller comes climbing
            up the hose, panting and exhausted.  You help him into
            the cage. ";
    }
    down =
    {
        if (self.xcoord = 60 && self.ycoord = 50 && inCage_glass.isopen)
        {
            if (fireHoseGlassMiddle.isIn(self))
                "If you want to climb down the hose, just say so. ";
            else
                "You can't walk on the glass, since it wouldn't support
                your weight, and you can't enter the hatch, since the
                floor is a hundred feet below. ";
        }
        else
            "If you want to get out, just say so. ";
            
        return nil;
    }
    roomDrop(obj) =
    {
        "You'd best hold onto it; the platform isn't all
        that steady, and <<obj.thedesc>> would probably slide off
        and get lost. ";
    }
;

inCage_floor: fixeditem
    sdesc = "cage floor"
    noun = 'floor'
    adjective = 'cage'
    ldesc = "The floor of the cage is a few inches above the glass. "
    location = inCage
    verDoSiton(actor) = { "There's no need to do that. "; }
    doSynonym('Siton') = 'Lieon'
    verIoPutOn(actor) = { "Better not; it would fall off. "; }
    ioSynonym('PutOn') = 'PutIn'
;

inCage_note: readable
    location = inCage
    sdesc = "yellow note"
    noun = 'note' 'paper' 'diagram' 'memo'
    adjective = 'yellow'
    isListed = nil
    isTaped = true
    moveInto(obj) =
    {
        /* when I'm first taken, change to use normal listing notation */
        if (self.isTaped)
        {
            self.isTaped := nil;
            self.isListed := true;
        }

        /* do the default work */
        inherited.moveInto(obj);
    }
    ldesc = "<tt><b>Memo</b>
             \bTo: Window washing crew
             \nFrom: Cleaning crew manager
             \nSubject: Water in atrium
             \bThe cleaning crew has been complaining for <u>several</u>
             weeks about water collecting on the atrium floor, making
             cleanup more difficult, etc.  We have determined that the
             source of the water is the <u>window washing crew</u>.

             \bPlease take note of the access hatch (SEE DIAGRAM BELOW)
             and <u>DO NOT</u> open this while the glass is still wet.

             \bThank you for your cooperation.

             \bJ. Stimms
             \nCleaning crew manager
             </tt>
             \bBelow this is a hand-drawn diagram showing a five-by-five
             grid of squares, with the square in the center emphatically
             circled. ";
;

class cageLeverItem: fixeditem
    location = inCage
    noun = 'lever'
    plural = 'levers'
    ldesc = "The lever is standing straight up; it looks like
             you could push it or pull it. "
    verDoPush(actor) = { }
    doPush(actor) =
    {
        inCage.leverMove(self.coordProp, 1, self.plusDirName);
    }
    verDoPull(actor) = { }
    doPull(actor) =
    {
        inCage.leverMove(self.coordProp, -1, self.minusDirName);
    }
    verDoMove(actor) = { "You can either push it or pull it. "; }
;

inCage_leftLever: cageLeverItem
    sdesc = "left lever"
    adjective = 'left'
    coordProp = &xcoord
    plusDirName = 'east'
    minusDirName = 'west'
;

inCage_rightLever: cageLeverItem
    sdesc = "right lever"
    adjective = 'right'
    coordProp = &ycoord
    plusDirName = 'south'
    minusDirName = 'north'
;

inCage_cage: fixeditem
    location = inCage
    sdesc = "metal cage"
    noun = 'cage' 'pole' 'poles' 'framework' 'floor' 'cable'
    adjective = 'metal'
    ldesc = { self.location.ldesc; }
    verDoUnboard(actor) = { }
    doUnboard(actor) = { actor.travelTo(self.location.out); }
    verDoClimb(actor) = { "That would be too dangerous. "; }
    verDoInspect(actor) = { }
    doInspect(actor) = { self.location.lookAround(true); }
    doSynonym('Inspect') = 'Search' 'Lookin'
;

inCage_pipes: fixeditem
    location = inCage
    noun = 'pipe' 'pipes' 'conduits'
    sdesc = "pipes"
    adesc = "any pipes"
    isThem = true
    ldesc = "The network of pipes nearly surrounds the glass area,
             but all of the pipes bend around the glass to leave the
             area above unobstructed. "
;

/*
 *   Class for items that are accessible only when the cage is located at
 *   the gap.  A regular base class must be used in addition to this
 *   class, since it inherits directly from 'object'; it's only purpose is
 *   to filter out all commands (except 'inspect' and the like) when the
 *   cage is not near the gap.  
 */
class inCage_gapItem: object
    cageRelDesc =
    {
        local x := inCage.xcoord;
        local y := inCage.ycoord;
        
        if (x < 60)
            " about <<spellOutNumber(60 - x)>> feet east";
        else if (x > 60)
            " about <<spellOutNumber(x - 60)>> feet west";

        if (y != 0)
        {
            if (x != 60)
                " and ";
            else
                " about ";
            "<<spellOutNumber(y)>> feet north";
        }

        if (x != 60 || y != 0)
            " of the cage";
    }
    dobjCheck(a, v, i, p) =
    {
        if (v != askVerb && v != tellVerb && v != inspectVerb
            && (inCage.xcoord != 60 || inCage.ycoord != 0))
        {
            "You can't reach it from here; it's still<<self.cageRelDesc>>. ";
            exit;
        }
    }
    iobjCheck(a, v, d, p) = { self.dobjCheck(a, v, d, p); }
;    

inCage_fireHoseNozzle: inCage_gapItem, item
    sdesc = "fire hose nozzle"
    noun = 'nozzle' 'hose'
    adjective = 'fire' 'hose'
    ldesc =
    {
        "The nozzle is sticking through a gap in the
        pipes<<self.cageRelDesc>>. ";
    }
    doTake(actor) =
    {
        fireHose.doTake(actor);
    }
;


inCage_gap: inCage_gapItem, fixeditem
    isqcontainer = true
    sdesc = "six-inch gap"
    noun = 'gap' 'hole'
    location = inCage
    adjective = 'six-inch'
    ldesc =
    {
        "It's a small gap, about six inches high, in the pipes
        to the north<<inCage_fireHoseNozzle.cageRelDesc>>. ";
        if (inCage_fireHoseNozzle.isIn(self))
            "The nozzle of a fire hose is poking out of the hole. ";
        else if (fireHoseEnd.cameThroughGap)
            "The fire hose is running through the gap. ";
    }
    doSynonym('Inspect') = 'Lookin' 'Search'
;


inCage_glass: atriumGlassItem
    location = inCage
    ldesc =
    {
        "The glass looks like a giant window set into the
        ground, roughly square and a hundred feet on a side.
        It looks as though a huge chamber is below the glass. ";

        if (inCage.xcoord = 60 && inCage.ycoord = 50)
        {
            "\b";
            if (self.isopen)
                "The glass pane directly outside the cage has
                been tilted up, creating an access hatch. ";
            else
                "The glass pane directly outside the cage is
                different from all of the others; it has a handle
                and hinges, and can clearly be opened. ";
        }
        else if (self.isopen)
        {
            "\bA glass panel <<self.cageRelDesc>> is tilted up,
            creating an opening. ";
        }

        if (atrium_fireHose.isIn(atrium))
            "A fire hose, stretching from a gap in the pipes to
            the north, hangs down through the opening. ";
    }
    cageRelDesc =
    {
        local x := inCage.xcoord;
        local y := inCage.ycoord;
        
        if (x < 60)
            " about <<spellOutNumber(60 - x)>> feet east";
        else if (x > 60)
            " about <<spellOutNumber(x - 60)>> feet west";

        if (y != 50)
        {
            if (x != 60)
                " and ";
            else
                " about ";

            if (y < 50)
                "<<spellOutNumber(50 - y)>> feet south";
            else
                "<<spellOutNumber(y - 50)>> feet north";
        }
        
        if (x != 60 || y != 50)
            " of here";
    }
    thrudesc =
    {
        if (inCage.xcoord = 60 && inCage.ycoord = 50 && self.isopen)
            "You can't see much from here; the floor of the chamber below
            must be a hundred feet or more down. ";
        else
            inherited.thrudesc;
    }
    verDoOpen(actor) =
    {
        if (inCage.xcoord != 60 || inCage.ycoord != 50)
            "This panel does not appear to open. ";
        else if (self.isopen)
            "The panel is already open. ";
    }
    doOpen(actor) =
    {
        "You carefully lean out, grasp the handle, and give it a good
        pull.  The cage sways slightly as you pull.  The glass sticks
        for a moment, but then pops free, and you tilt up the panel
        until it's sitting straight up, where it locks into position. ";
        
        self.isopen := true;
        aboveAtrium_accessHatch.moveInto(aboveAtrium);
        inCage_accessHatch.moveInto(inCage);
    }
    verDoClose(actor) =
    {
        if (inCage.xcoord != 60 || inCage.ycoord != 50)
            "This panel is already closed. ";
        else if (!self.isopen)
            "The panel is already closed. ";
    }
    doClose(actor) =
    {
        if (atrium_fireHose.isIn(atrium))
        {
            "You don't want to close the panel with the hose still
            going through the opening. ";
        }
        else
        {
            "You carefully lean out and lower the panel.  It seals
            tightly. ";
            
            self.isopen := nil;
            aboveAtrium_accessHatch.moveInto(nil);
            inCage_accessHatch.moveInto(nil);
        }
    }
    verIoPutIn(actor) =
    {
        if (inCage.xcoord != 60 || inCage.ycoord != 50)
            inherited.verIoPutIn(actor);
        else if (!self.isopen)
            "You can't put anything in a closed panel. ";
    }
    ioPutIn(actor, dobj) =
    {
        if (dobj = fireHoseEnd || dobj = fireHoseNozzle)
        {
            "You feed the fire hose down through the opening, gradually
            pulling hose off the reel in the distance.  The weight of
            the hose hanging down into the opening eventually starts
            doing most of the work of pulling more down.  Finally,
            after a tremendous length of hose has disappeared into
            the chamber below, the hose seems to reach the floor.
            \bYou carefully let go of the hose, letting it drape over
            the edge of the opening. ";

            fireHoseEnd.moveInto(nil);
            atrium_fireHose.moveInto(atrium);
            fireHoseGlassMiddle.moveInto(inCage);
            fireHoseEdgeMiddle.moveInto(aboveAtrium);
        }
        else
            "There are all sorts of reasons that
            dropping <<dobj.thedesc>> into the opening would be a bad
            idea. ";
    }
    doSynonym('Lookthru') = 'Lookin' 'Search'
    ioSynonym('PutIn') = 'PutThru'

    sprayWithWater =
    {
        if (self.isopen)
            "\bA booming voice echoes from the public address
            system: <q>Attention cleaning crew: <i>please</i> close the
            access hatch before cleaning the glass.  You're getting the
            atrium all wet again, and we're tired of cleaning it up.</q> ";
    }
;

inCage_handle: fixeditem
    noun = 'handle'
    adjective = 'glass' 'pane' 'metal' 'large'
    sdesc = "glass pane handle"
    ldesc = "It's a large metal handle with square corners. "
    verDoPull(actor) = { inCage_glass.verDoOpen(actor); }
    doPull(actor) = { inCage_glass.doOpen(actor); }
;

inCage_hinge: fixeditem
    noun = 'hinge'
    adjective = 'glass' 'pane' 'heavy-duty'
    sdesc = "glass pane hinge"
    ldesc = "A heavy-duty hinge runs along the entire edge of the
             pane opposite the handle. "
;

inCage_accessHatch: fixeditem
    noun = 'hatch' 'opening'
    adjective = 'access'
    sdesc = "access hatch"
    adesc = "an access hatch"
    ldesc =
    {
        "The pane has been tilted up, creating an opening in
        the glass. ";
    }
    ioPutIn -> inCage_glass
    ioPutThru -> inCage_glass
    doLookin -> inCage_glass
    doLookthru -> inCage_glass
    doSearch -> inCage_glass
    dobjCheck(a, v, i, p) =
    {
        if (v != askVerb && v != tellVerb && v != inspectVerb
            && (inCage.xcoord != 60 || inCage.ycoord != 50))
        {
            "It's too far away. ";
            exit;
        }
    }
    iobjCheck(a, v, d, p) = { self.dobjCheck(a, v, d, p); }
    verDoEnter(actor) = { }
    doEnter(actor) = { actor.travelTo(self.location.down); }
    doSynonym('Enter') = 'Board' 'Gothrough'
;

fireHoseGlassMiddle: fixeditem
    noun = 'hose'
    adjective = 'fire'
    sdesc = "fire hose"
    firstClimb = true
    ldesc = "The fire hose is stretched from a gap in the pipes
             to the access hatch, where it disappears into the
             chamber below. "
    verDoPutIn(actor, iobj) =
    {
        if (iobj = inCage_accessHatch || iobj = inCage_glass)
            "The hose is already going through the opening. ";
        else
            inherited.verDoPutIn(actor, iobj);
    }
    verDoPutThru(actor, iobj) =
    {
        if (iobj = inCage_accessHatch || iobj = inCage_glass)
            "The hose is already going through the opening. ";
        else
            inherited.verDoPutThru(actor, iobj);
    }
    verDoTake(actor) = { }
    doTake(actor) =
    {
        "With much effort, you pull the hose out of the hatch.
        Fortunately, the reel helps a little by taking up the slack
        for you, and starts to pull in the hose by itself as you
        get nearer the end.  Finally, the nozzle comes into view,
        and you take hold of it. ";

        self.moveInto(nil);
        atrium_fireHose.moveInto(nil);
        fireHoseEnd.moveInto(actor);
    }
    doSynonym('Take') = 'Pull' 'Move'
    verDoClimb(actor) = { }
    doClimb(actor) =
    {
        if (inCage.xcoord != 60 || inCage.ycoord != 50)
        {
            "The cage must be closer to the opening for you to climb down
            the hose. ";
            return;
        }

        if (self.firstClimb)
        {
            "You have a sudden chill looking at the hose disappearing
            into the darkness below. ";

            if (teeterwaller.isIn(actor.location))
                "\bTeeterwaller gulps. <q>B-b-b-be careful,</q> he
                stutters. <q>I'll wait here until you reach the bottom,
                to make sure we don't put too much weight on the
                hose.</q>";

            "\bSlowly, you maneuver yourself over to the edge nearest
            the opening, and lean out of the cage, holding on to the
            pole with both hands.  It's only a foot or two to the hose,
            but somehow it feels like you're about to try jumping over
            a canyon.
            \bYou realize you've been holding your breath, and force
            yourself to exhale.  You jerk one hand out and grab at the
            hose, slipping at first but finally getting a good grip on
            it.  You carefully swing one leg out onto the framework
            supporting the windows, then the other, taking care not
            to put any weight on the glass.
            \bWith nothing left to do, you let go of the cage with your
            other hand and grab at the hose.  The framework is obviously
            sturdy enough to support you if it can support all of this
            glass, you keep telling yourself, but you can feel it flexing
            under you.
            \bHaving successfully negotiated the chasm between the cage
            and the hose, you feel much less panicked, and start descending
            the hose.  It's a nice rough material and easy to grip as
            you climb down.
            \bThe descent takes less time than you would have thought,
            given the amount of hose you had to feed down through the
            hatch.  You reach the floor, and rest a moment to let the
            adrenaline wear off. ";
            
            self.firstClimb := nil;
        }
        else
        {
            "You carefully maneuver yourself over to the hose, and
            start lowering yourself down its length.  You reach the
            floor after a brief descent. ";
        }

        "\b";
        actor.travelTo(atrium);
    }
    doSynonym('Climb') = 'Climbdown'
    dobjCheck(a, v, i, p) =
    {
        if (v != askVerb && v != tellVerb && v != inspectVerb
            && (inCage.xcoord != 60 || inCage.ycoord > 50))
        {
            "It's too far away. ";
            exit;
        }
    }
    iobjCheck(a, v, d, p) = { self.dobjCheck(a, v, d, p); }
;

inCage_chamber: distantItem
    noun = 'chamber' 'room' 'atrium'
    location = inCage
    sdesc = "chamber"
    ldesc = "You can't see any details from here; the floor of
             the chamber must be over a hundred feet below. "
    verDoEnter(actor) =
    {
        if (inCage.xcoord = 60 && inCage.ycoord = 50
            && atrium_fireHose.isIn(atrium))
            "If you want to climb down the hose, just say so. ";
        else
            "You see no way to do that from here. ";
    }
;

