/* Copyright (c) 1989, 1991 by Michael J. Roberts.  All Rights Reserved. */
/* Requires TADS version 2.0 and the KITCHEN ENCOUNTERS version of adv.t */

#include <adv.t>

/*
 *   Pre-declare all functions, so the compiler knows they are functions.
 *   (This is only really necessary when a function will be referenced
 *   as a daemon or fuse before it is defined; however, it doesn't hurt
 *   anything to pre-declare all of them.)
 */
die: function;
scoreRank: function;
init: function;
terminate: function;
pardon: function;
sleepDaemon: function;
eatDaemon: function;
darkTravel: function;
fightDaemon: function;
weaseldaemon: function;
snoboldaemon: function;
done: function;
AllDone : function;

upstairsPrep: Prep
    preposition = 'upstairs'
    sdesc = "upstairs"
;

awayPrep: Prep
    preposition = 'away'
    sdesc = "away"
;

yellatVerb: deepverb
	sdesc = "yell at"
    verb = 'yell at' 'shout at'
    doAction = 'YellAt'
;

locateVerb: deepverb
    sdesc = "locate"
    verb = 'locate' 'where is' 'find' 'where are'
    doAction = 'Locate'
;

MYuVerb: travelVerb
    action(actor) = { actor.travelTo(self.travelDir(actor)); }
    verb = 'go upstairs' 'upstairs'
    travelDir(actor) = { return(actor.location.up); }
;

/*
 *   The die() function is called when the player dies.  It tells the
 *   player how well he has done (with his score), and asks if he'd
 *   like to start over (the alternative being quitting the game).
 */
die: function
{
	"\b*** You have died ***\b";
	scoreRank();
	done();
}

done : function {
	"\bYou may restore a saved game, start over, or quit.\n";
	while (1) {
		local resp;

		"Please enter RESTORE, RESTART, or QUIT: >";
		resp := upper(input());
		if ( resp = 'RESTORE' ) {
			resp := askfile( 'File to restore' );
			if (resp = nil) "Restore failed. ";
			else if ( restore( resp )) "Restore failed. ";
			else {
				setscore( global.score, global.turnsofar );
				abort;
			}
		}
		else if ( resp = 'RESTART' ) {
			setscore( 0, 0 );
			restart();
		}
		else if ( resp = 'QUIT' ) {
			terminate();
			quit();
			abort;
		}
	}
}

/*
 *   The scoreRank() function displays how well the player is doing.
 *   This default definition doesn't do anything aside from displaying
 *   the current and maximum scores.  Some game designers like to
 *   provide a ranking that goes with various scores ("Novice Adventurer,"
 *   "Expert," and so forth); this is the place to do so if desired.
 *
 *   Note that "global.maxscore" defines the maximum number of points
 *   possible in the game; change the property in the "global" object
 *   if necessary.
 */
scoreRank: function
{
    "In a total of "; say( global.turnsofar );
    " turns, you have achieved a score of ";
    say( global.score ); " points out of a possible ";
    say( global.maxscore ); ".\n";
}

AllDone : function {
	terminate();
	done();
}

/*
 *   The init() function is run at the very beginning of the game.
 *   It should display the introductory text for the game, start
 *   any needed daemons and fuses, and move the player's actor ("Me")
 *   to the initial room, which defaults here to "startroom".
 */
init: function {

	"\bKITCHEN ENCOUNTERS (v 0.2 comments to Andy Jewell : avjewe@cfar.umd.edu)
	\n(a tiny adventure game for Andy)

	\bAs you unlock the door, the alarm starts beeping.  In a flurry
	of movement, you disarm it with practiced ease, while simultaneously
	preventing the black cat from escaping outside.  As you walk into
	the kitchen, you yell, \"Honey, I'm home!\" and drop your pile of
	work cruft on the counter.  A sleepy purr is all you hear...\b";

	setdaemon(turncount, nil);		// start the turn counter daemon
	setdaemon(snoboldaemon, nil);	// start the hunger daemon
	Me.location := startroom;		// move player to initial location
	startroom.lookAround(true);		// show player where he is
	global.maxscore := 8;
	setfuse(AllDone, 120, 0);
}

/*
 *   preinit() is called after compiling the game, before it is written
 *   to the binary game file.  It performs all the initialization that can
 *   be done statically before storing the game in the file, which speeds
 *   loading the game, since all this work has been done ahead of time.
 *
 *   This routine puts all lamp objects (those objects with islamp = true) into
 *   the list global.lamplist.  This list is consulted when determining whether
 *   a dark room contains any light sources.
 */
preinit: function
{
	local o;

	global.lamplist := [];
	o := firstobj();
	while (o <> nil) {
		if (o.islamp ) global.lamplist := global.lamplist + o;
		o := nextobj( o );
	}
	initSearch();
}

/*
 *   The terminate() function is called just before the game ends.  It
 *   generally displays a good-bye message.  The default version does
 *   nothing.  Note that this function is called only when the game is
 *   about to exit, NOT after dying, before a restart, or anywhere else.
 */
terminate: function {
	if (global.score = global.maxscore) {		
		"\nYour score is <<global.maxscore>> out of <<global.maxscore>> points.  That give you the rank of Helpmeet.
		Congratulations!  You can come live in my house anytime!\b";
	}
	else {
		"\n%You% hear a car pull into the driveway. Moments later
		the door opens and %your% mate enters.  The familiar cry of
		\"Honey, I'm home!\" fills the house, and %you% run%s% to offer a warm welcome,
		wondering whether you thought of everything.

		\bAfter a brief silence your mate shouts, \"There were only a few things
		that needed cleaning up! Couldn't you even find time to ";
		
		if (scored.didsweep = nil) {
			"sweep the floor";
		}
		else if (scored.didnewspapers = nil) {
			"put away the newspapers";
		}
		else if (scored.soupaway = nil) {
			"put away the chicken broth";
		}
		else if (scored.didworkcruft = nil) {
			"put your work cruft upstairs";
		}
		else if (scored.didrecycling = nil) {
			"clean up the recycling";
		}
		else if (scored.weaselaway = nil) {
			"put the weasel in her cage";
		}
		else if (scored.worthlesscruft = nil) {
			"throw away the worthless cruft on the counter";
		}
		"?\"";
		"\bYOU LOST! You'll be cleaning the bathroom every day for a month.\b";
	}
}

/*
 *   The pardon() function is called any time the player enters a blank
 *   line.  The function generally just prints a message ("Speak up" or
 *   some such).  This default version just says "I beg your pardon?"
 */
pardon: function {
	"I beg your pardon? ";
}

/*
 *   This function is a daemon, started by init(), that monitors how long
 *   it has been since the player slept.  It provides warnings for a while
 *   before the player gets completely exhausted, and causes the player
 *   to pass out and sleep when it has been too long.  The only penalty
 *   exacted if the player passes out is that he drops all his possessions.
 *   Some games might also wish to consider the effects of several hours
 *   having passed; for example, the time-without-food count might be
 *   increased accordingly.
 */
sleepDaemon: function( parm ) {
	local a, s;

	global.awakeTime := global.awakeTime + 1;
	a := global.awakeTime;
	s := global.sleepTime;

	if (a = s or a = s+10 or a = s+20) {
		"\bYou're feeling a bit drowsy; you should find a
		comfortable place to sleep. ";
	}
	else if (a = s+25 or a = s+30) {
		"\bYou really should find someplace to sleep soon, or
		you'll probably pass out from exhaustion. ";
	}
	else if (a >= s+35) {
		global.awakeTime := 0;
		if (Me.location.isbed or Me.location.ischair) {
			"\bYou find yourself unable to stay awake any longer.
			Fortunately, you are ";
			if ( Me.location.isbed ) "on "; else "in ";
			Me.location.adesc; ", so you gently slip off into
			unconsciousness.
			\b* * * * *
			\bYou awake some time later, feeling refreshed. ";
		}
	}
	else if (a >= s+45) {
		local itemRem, thisItem;

		"\bYou find yourself unable to stay awake any longer.
		You pass out, falling to the ground.
		\b* * * * *
		\bYou awaken, feeling somewhat the worse for wear.
		You get up and dust yourself off. ";
		itemRem := Me.contents;
		while (car(itemRem)) {
			thisItem := car(itemRem);
			if (not thisItem.isworn) {
				thisItem.moveInto(Me.location);
			}
			itemRem := cdr( itemRem );
		}
	}
}

/*
 *   This function is a daemon, set running by init(), which monitors how
 *   long it has been since the player has had anything to eat.  It will
 *   provide warnings for some time prior to the player's expiring from
 *   hunger, and will kill the player if he should go too long without
 *   heeding these warnings.
 */
eatDaemon: function(parm) {
	local e, l;

	global.lastMealTime := global.lastMealTime + 1;
	e := global.eatTime;
	l := global.lastMealTime;

	if (l = e or l = e+5 or l = e+10) {
		"\bYou're feeling a bit peckish. Perhaps it would be a good
		time to find something to eat. ";
	}
	else if (l = e+15 or l = e+20 or l = e+25) {
		"\bYou're feeling really hungry. You should find some food
		soon or you'll pass out from lack of nutrition. ";
	}
	else if (l=e+30 or l = e+35) {
		"\bYou can't go much longer without food. ";
	}
	else if (l >= e+40) {
		"\bYou simply can't go on any longer without food. You perish from
		lack of nutrition. ";
		die();
	}
}

/*
 *   The numObj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numObj, with its "value"
 *   property set to 621.
 */
numObj: basicNumObj;  // use default definition from adv.t

/*
 *   strObj works like numObj, but for strings.  So, a player command of
 *	 type "hello" on the keyboard
 *   will result in a direct object of strObj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 */
strObj: basicStrObj;	 // use default definition from adv.t

/*
 *   The "global" object is the dumping ground for any data items that
 *   don't fit very well into any other objects.  The properties of this
 *   object that are particularly important to the objects and functions
 *   are defined here; if you replace this object, but keep other parts
 *   of this file, be sure to include the properties defined here.
 *
 *   Note that awakeTime is set to zero; if you wish the player to start
 *   out tired, just move it up around the sleepTime value (which specifies
 *   the interval between sleeping).  The same goes for lastMealTime; move
 *   it up to around eatTime if you want the player to start out hungry.
 *   With both of these values, the player only starts getting warnings
 *   when the elapsed time (awakeTime, lastMealTime) reaches the interval
 *   (sleepTime, eatTime); the player isn't actually required to eat or
 *   sleep until several warnings have been issued.  Look at the eatDaemon
 *   and sleepDaemon functions for details of the timing.
 */
 
scored: object
	didworkcruft = nil
	eatmouse = nil
	didsweep = nil
	didrecycling = nil
	weaselaway = nil
	soupaway = nil
	worthlesscruft = nil
	didnewspapers = nil
;

global: object
	turnsofar = 0		// no turns have transpired so far
	score = 0			// no points have been accumulated yet
	maxscore = 100		// maximum possible score
	verbose = nil		// we are currently in TERSE mode
	awakeTime = 0		// time that has elapsed since the player slept
	sleepTime = 400		// interval between sleeping times (longest time awake)
	lastMealTime = 0		// time that has elapsed since the player ate
	eatTime = 200		// interval between meals (longest time without food)
	lamplist = []		// list of all known light providers in the game
	fightTime = 10		// maximum number of turns Cyrano and Fiver will fight
;

/*
 *   The "version" object defines, via its "sdesc" property, the name and
 *   version number of the game.  Change this to a suitable name for your
 *   game.
 */
version: object
	sdesc = "KITCHEN ENCOUNTERS v 0.2
	\b Andy Jewell
	\n avjewe@cfar.umd.edu
	\b"
;

/*
 *   "Me" is the player's actor.  Pick up the default definition, basicMe,
 *   from "adv.t".
 */
Me: basicMe;

/*
 *   darkTravel() is called whenever the player attempts to move from a dark
 *   location into another dark location.  By default, it just says "You
 *   stumble around in the dark," but it could certainly cast the player into
 *   the jaws of a grue, whatever that is...
 */
darkTravel: function {
	"You stumble around in the dark, and don't get anywhere. ";
}

/*
 *   goToSleep - carries out the task of falling asleep.  We just display
 *   a message to this effect.
 */
goToSleep: function {
	"***\bYou wake up some time later, feeling refreshed. ";
}

/*
 * FUNCTIONS AND DAEMONS
 */

doincscore: function (amount, LastPointObj) {
	incscore(amount);
	"\b(Your score just went up!)";
	
	if (global.score = global.maxscore) {
		"\b";
		LastPointObj.lastpoint;
		AllDone();
	}
}

StopFight: function() {
	cyrano.moveInto(nil);
	cyranose.moveInto(nil);
	fiver.moveInto(nil);
	cats.moveInto(nil);
	remdaemon(fightDaemon, 0);
	" All is now quiet. ";
}

weaseldaemon: function(parm) {
	if (weasel.location = startroom) {
		if (snobol.location = startroom) {
			if (snobol.weaselcount = 0) {
				"\b\^<<weasel.thedesc>> annoys <<snobol.thedesc>>. ";
			}
			else if (snobol.weaselcount = 1) {
				"\b\^<<snobol.thedesc>> whacks <<weasel.thedesc>> repeatedly on the head. ";
			}
			else {
				"\b\^<<weasel.thedesc>> win the fight, and <<snobol.sdesc>> flees. ";
				snobol.moveInto(nil);
			}
			snobol.weaselcount := snobol.weaselcount + 1;
		}
		else {
			if (weasel.lastopened) {
				if (weasel.lastopened.isopen) {
					"\b\^<<weasel.thedesc>> scurries into <<weasel.lastopened.thedesc>>. ";
					weasel.moveInto(weasel.lastopened);
					weasel.lastopened := 0;
				}
				else {
					"\b\^<<weasel.thedesc>> chases your feet. ";
				}
			}
			else {
				"\b\^<<weasel.thedesc>> chases your feet. ";
			}
		}
		weasel.wasincage := 0;
	}
	else if (weasel.location = cage) {
		weasel.wasincage := weasel.wasincage + 1;
		if (weasel.wasincage <= 1) {
			"\b\^<<weasel.thedesc>> is heading for the open door of the cage! ";
		}
		else {
			"\b\^<<weasel.thedesc>> escapes! ";
			weasel.moveInto(startroom);
			weasel.wasincage := 0;
		}
	}
	else if (Me.isCarrying(weasel)) {
	}
	else {
		weasel.wasincage := weasel.wasincage + 1;
		if (weasel.location.isopen) {
			if (weasel.wasincage <= 1) {
				"\b\^<<weasel.thedesc>> explores the interior of <<weasel.location.thedesc>>. ";
			}
			else {
				"\b\^<<weasel.thedesc>> quickly escapes from <<weasel.location.thedesc>>. ";
				weasel.moveInto(startroom);
				weasel.wasincage := 0;
			}
		}
		else {
			"\b\^%You% hear%s% scratching noises from <<weasel.location.thedesc>>. ";
		}
	}
}

fightDaemon: function(parm) {
	global.fightTime := global.fightTime - 1;
	if (global.fightTime = 0) {
		"\bThe cats, with one final horrible shriek, upset the recycling boxes
		and dash out of the room. ";
		StopFight();
		pantry.catfight := true;
	}
	else if (global.fightTime = 8) {
		"\bSuddenly, the two cats lunge at each other and start to fight!
		The noise is horrendous as the two animals roll over and over
		and claw at each other.  A variation on an old joke crosses %your%
		mind:  \"What's orange and black and red all over?\"";
	}
	if (global.fightTime <= 8 and global.fightTime > 0) {
		"\bThe cats continue to fight. ";
	}		
}
	
/*
 * ROOMS
*/

dumbwin : object
	lastpoint =
		"\n%You% hear a car pull into the driveway. Moments later
		the door opens and %your% mate enters.  The familiar cry of
		\"Honey, I'm home!\" fills the house, and %you% run%s% to offer a warm welcome,
		wondering whether %you% should tell your mate about the mouse...\b"
;

upstairsRoom : room
	lastpoint =
		"As you mount the stairs, you hear a car pull into the driveway.  You
		hurry into the bedroom and put your work cruft on your dresser just
		as the downstairs door opens and your mate enters.  The familiar
		cry of \"Honey, I'm home!\" fills the house, and you run downstairs,
		wondering whether you should tell your mate about the mouse...\b"

    sdesc = {
		Me.location := startroom;
		if (Me.isCarrying(workcruft)) {
			if (global.score < (global.maxscore-1)) {		
				"You put the work cruft on the dresser in the bedroom,
				and return to the kitchen";
				if (chickenbroth.location = workcruft) {
					" with the chicken broth. ";
					chickenbroth.moveInto(Me);
				}
				else {
					". ";
				}
			}
			workcruft.moveInto(nil);
			doincscore(1, upstairsRoom);
			scored.didworkcruft := true;
		}
		else {
			"You wander about upstairs for a while, and then come down again. ";
		}
	}
	ldesc = ""
;

startroom : room
	noun = 'kitchen'
	sdesc = "kitchen"
	ldesc = {
		"You are in the kitchen.  There is counterspace everywhere<<counter.fullness>>  
		Cabinets line the walls, and there is a pantry at one end.  Standard kitchen appliances 
		(stove, refrigerator, microwave oven, sink) complete the picture. ";
	}
	HasFloor = true
    up = upstairsRoom
;

/*
 *  FIXED ITEMS
 */ 

mybox : container, fixeditem
	nocontents = {}
;
bottlebox : mybox
	sdesc = "bottle box"
	ldesc = {
		"This box holds the bottles that are to be recycled. ";
		inherited.ldesc;
	}
	location = pantry
	noun = 'box'
	adjective = 'bottle' 'glass'
;

counter : surface, fixeditem
	sdesc = "counter"
	ldesc = {
		"This is the counter you eat most of your meals on.  When there's
		room, that is.\b";
		pass ldesc;
	}
	location = startroom
	noun = 'counterspace' // 'counter'
	CanFitBroom = true
	showcontcont = {
		if (itemcnt(self.contents)) {
			"On <<self.thedesc>> you see <<listcont(self)>>. ";
		}
		listfixedcontcont(self);
	}
	fullness = {
		local thenum;
		thenum := itemcnt(self.contents);
		if (thenum = 0) {
			". ";
		}
		else {
			", although ";
			if (thenum = 1) {
				"some";
			}
			else if (thenum = 2) {
				"a bunch";
			}
			else if (thenum = 3) {
				"most";
			}
			else {
				"almost all";
			}
			" of it is covered up at the moment. ";
		}
	}
;


pantry : mybox, openable
	showcontcont = {
		if (self.isopen) {
			"\^<<self.thedesc>> is open. ";
		}
	}
	sdesc = "pantry"
	ldesc = {
		if (self.isopen) {
			"The pantry has space to store things on four shelves as well as
			on the floor. ";
			inherited.ldesc;
		}
		else {
			"It is a very nice pantry.
			It has two doors with doorknobs that don't turn. ";
		}
	}
	verIoPutIn(actor) = {
		"You should put that onto one of the shelves or the floor of <<self.thedesc>>. ";
	}
	location = startroom
	noun = 'pantry'
	isopen = nil
	catfight = nil
	doOpen = {
		inherited.doOpen;
		if (not self.catfight) {
			cyrano.moveInto(self.location);
			cyranose.moveInto(self.location);
			fiver.moveInto(self.location);
			cats.moveInto(self.location);
			pantry.catfight := true;
			cyrano.angry := true;
			"\bAs you open the pantry door, you hear a noise behind you.  As you
			turn to look, you see the orange cat heading toward you with an
			expectant look on his face.  As he reaches the pantry, the black
			cat trots up as well.  The two cats look at each other and growl. ";
			setdaemon(fightDaemon, 0);
		}
	}
;

AnyFloor : fixeditem, surface
	CanFitBroom = true
	location = nil
	verDoSweepWith(actor, io) = {}
	verDoSweep(actor) = {}
	doSweep(actor) = {
		askio(withPrep);
	}
    doSiton(actor) =
    {
        "Okay, %you're% now sitting on "; self.thedesc; ". ";
        self.sitloc := actor.location;
        actor.moveInto(self);
    }
    doLieon(actor) =
    {
        self.doSiton(actor);
    }
;

kitchenFloor : AnyFloor, theFloor
	isswept = nil
	sdesc = "kitchen floor"
	ldesc = {
		"\^<<self.thedesc>> is ";
		if (self.isswept) {
			"cleaner than usual. ";
		}			
		else {
			"covered with fur and other stuff. ";
		}
	}
	location = startroom
	noun = 'floor' 'ground'
	adjective = 'kitchen'
	doSweepWith(actor , io) = {
		if (self.isswept) {
			"It doesn't get any better than this. ";
		}
		else {
			"After a few minutes, the floor looks as good as it ever does with so
			many furry creatures around. ";
			self.isswept := true;
			doincscore(1, dumbwin);
			scored.didsweep := true;
		}
	}
;

leftbottomshelf : LSshelf
	sdesc = "second shelf of the left cabinet"
	location = leftcabinet
	noun = 'shelf'
	adjective = 'bottom' 'second' 'left' '2'
;

leftcabinet : LScabinet
	foundweasel = {
		if (not self.foundeyes) {
			"As you do, you hear a noise in the cabinet to your left. ";
		}
	}
	ldesc = {
		inherited.ldesc;
		if ((not self.foundeyes) and (not self.isopen)) {
			"\b%You% hear a noise in the cabinet. ";
		}
	}
	sdesc = "left lazy susan cabinet"
	noun = 'cabinet' 'susan'
	plural = 'cabinets'
	adjective = 'lazy' 'left'
	location = startroom
;

LSshelf : fixeditem, surface
	showcontcont = {}
;

lefttopshelf : LSshelf
	sdesc = "first shelf of the left cabinet"
	location = leftcabinet
	noun = 'shelf'
	adjective = 'top' 'first' 'left' '1'
;

LScabinet : container, openable, fixeditem
	showcontcont = {
		if (self.isopen) {
			"\^<<self.thedesc>> is open. ";
		}
	}
	foundeyes = nil
	isopen = nil
	ldesc = {
		if (self.isopen) {
			"This is a lazy susan cabinet with two shelves. ";
		}
		else {
			"Looks like an ordinary lazy susan cabinet to me. ";
		}
	}
	verIoPutIn(actor) = {
		"You should put that onto one of the shelves of <<self.thedesc>>. ";
	}
	verDoOpenWith(actor, io) = {inherited.verDoOpen(actor);}
	doOpen(actor) = {
		if (self.foundeyes) {
			inherited.doOpen(actor);
		}
		else {
			"As you start to push it open, a pair of beady eyes looks out at you!  
			You're so startled you let go!

			\bThe cabinet swings shut. ";
		}
	}
	doOpenWith(actor, io) = {
		if (self.foundeyes) {
			inherited.doOpen(actor);
		}
		else if (io = broom) { 
			"As you open the cabinet with the broom, a pair of beady eyes looks
			out at you!  But since you're now so far away from them, you continue
			to push open the cabinet.

			\bAs the cabinet swings open, the beady eyes disappear into the supplies
			contained there.\b";
			inherited.doOpen(actor);
		}
	}
;


newspaperbox : mybox
	sdesc = "newspaper box"
	ldesc = {
		"This box holds the newspapers that are to be recycled. ";
		inherited.ldesc;
	}
	location = pantry
	noun = 'box'
	adjective = 'newspaper'
;

boxes : container, shelfitem
	sdesc = "recycling boxes"
	location = pantryfloor
	adjective = 'recycling'
	noun = 'boxes'
;

recycling : item
    isListed = nil          // not listed in room/inventory displays
	sdesc = "recycling"
	heredesc = "The recycling is now scattered all over the floor"
	location = pantryfloor
	noun = 'recycling' 'newspapers' 'bottles'
	adjective = 'scattered' 'glass'
	glassgone = nil
	papergone = nil
	IsDone = {
		self.moveInto(nil);
		"Ok, the recycling is now back in the boxes. ";
		doincscore(1, dumbwin);
		scored.didrecycling := true;
	}
	doPutIn(actor, io) = { 
		if (io = boxes) {
			self.IsDone;
		}
		else if (io = newspaperbox) {
			if (self.glassgone) {
				self.IsDone;
			}
			else if (self.papergone) {
				"There are no scattered newspapers left to put away, just glass bottles. ";
			}
			else {
				self.papergone := true;
				"The scattered newspapers are back in the box, but there 
				are still glass bottles everywhere";
			}
		}
		else if (io = bottlebox) {
			if (self.papergone) {
				self.IsDone;
			}
			else if (self.glassgone) {
				"There are no glass bottles left to put away, just scattered newspapers. ";
			}
			else {
				self.glassgone := true;
				"The glass bottles are back in the box, but there 
				are still scattered newspapers everywhere";
			}
		}
		else {
			inherited.doPutIn(actor, io);
		}
	}		
;

pantryshelf : fixeditem, surface
	CanFitBroom = true
	showcontcont = {}
;

pantryfirstshelf : pantryshelf
	sdesc = "first shelf of the pantry"
	noun = 'shelf'
	adjective = 'top' 'first' 'pantry' '1'
	location = pantry
;

pantryfloor : AnyFloor
	sdesc = "pantry floor"
	noun = 'floor'
	adjective = 'pantry'
	location = pantry
	doSweepWith(actor , io) = {
		"What with all the cat food and recycling boxes, there isn't much floor left over for sweeping. ";
	}
;

pantryfourthshelf : pantryshelf
	sdesc = "fourth shelf of the pantry"
	noun = 'shelf'
	adjective = 'bottom' 'fourth' '4'
	location = pantry
;

pantrysecondshelf : pantryshelf
	sdesc = "second shelf of the pantry"
	noun = 'shelf'
	adjective = 'second' '2'
	location = pantry
;

pantrythirdshelf : pantryshelf
	sdesc = "third shelf of the pantry"
	noun = 'shelf'
	adjective = '3' 'third'
	location = pantry
;

rightbottomshelf : LSshelf
	sdesc = "second shelf of the right cabinet"
	location = rightcabinet
	noun = 'shelf'
	adjective = 'bottom' 'second' 'right' '2'
;

rightcabinet : LScabinet
	sdesc = "right lazy susan cabinet"
	noun = 'cabinet' 'susan'
	plural = 'cabinets'
	adjective = 'lazy' 'right'
	location = startroom
;

righttopshelf : LSshelf
	sdesc = "first shelf of the right cabinet"
	location = rightcabinet
	noun = 'shelf'
	adjective = 'top' 'first' 'right' '1'
;

sink : fixeditem
	sdesc = "sink"
	noun = 'sink'
	location = startroom
	plural = 'appliances'
	adjective = 'standard' 'kitchen'
    verDoLookunder(actor) = {}
    doLookunder(actor) = {
		"Just the trash can. ";
	}
;

trashcan : fixeditem
	location = sink
	sdesc = "trash can"
	noun = 'can' 'bin'
	adjective = 'trash' 'garbage' 'rubbish'
;

microwave : fixeditem
	sdesc = "microwave oven"
	noun = 'microwave oven' 'oven'
	adjective = 'microwave'
	location = startroom
	plural = 'appliances'
	adjective = 'standard' 'kitchen'
;

refrigerator : fixeditem
	sdesc = "refrigerator"
	noun = 'refrigerator'
	location = startroom
	plural = 'appliances'
	adjective = 'standard' 'kitchen'
;

stove : fixeditem
	sdesc = "stove"
	noun = 'stove'
	location = startroom
	plural = 'appliances'
	adjective = 'standard' 'kitchen'
;


walls : fixeditem
	sdesc = "walls"
	ldesc = "They're just walls. "
	noun = 'walls' 'wall'
	location = startroom
;

/*
 *  ITEMS
 */

cage : transparentItem, openable
	isListed = nil
	heredesc = {
		"In one corner is a small wire cage";
		if (self.location = broom.location) {
			", and ";
		}
		else {
			". ";
		}
	}
	sdesc = "cage"
	location = startroom
	noun = 'cage'
	adjective = 'small' 'wire' 'ferret'
	isopen = nil
	maxbulk = 2
	verDoClose(actor) = {}
	doClose(actor) = {
		inherited.doClose(actor);
		if (weasel.location = cage) {
			"\bThe ferret resignedly curls up to take a nap. ";
			weasel.moveInto(nil);
			sleepingweasel.moveInto(cage);
			remdaemon(weaseldaemon, 0);
			doincscore(1, dumbwin);
			scored.weaselaway := true;
		}
	}
;

broom : item
	isListed = nil
	heredesc = {
		if (self.location = cage.location) {
			"in another ";
		}
		else {
			"In one corner ";
		}
		"is a broom. ";
	}
	location = startroom
	sdesc = "broom"
	ldesc = "Looks like a cheap yellow broom, probably from K-Mart. "
	noun = 'broom'
	adjective = 'yellow' 'cheap'

	verIoAttackWith( actor ) = {
		if ( not actor.isCarrying( self )) {
			"%You're% not carrying <<self.thedesc>>. ";
		}
	}
	verIoSweepWith( actor ) = {
		if (actor.isCarrying(self)) {
		}
		else {
			"%You're% not carrying <<self.thedesc>>. ";
		}
	}
	ioSweepWith(actor, dobj) = {
		dobj.doSweepWith(actor, self);
	}
	ioAttackWith(actor, dobj) = {
		dobj.doAttackWith(actor, self);
	}
	verIoOpenWith(actor) = {
		if ( not actor.isCarrying(self)) {
			"%You're% not carrying "; self.thedesc; ". ";
		}
	}
	ioOpenWith(actor, dobj) = {
		dobj.doOpenWith( actor, self );
	}	
	verDoPutIn(actor, io) = {
		if (io.CanFitBroom = nil) {
			"\^<<self.thedesc>> will not fit in <<io.thedesc>>. ";
		}
	}
	verDoPutOn(actor, io) = {
		if (io.CanFitBroom = nil) {
			"\^<<self.thedesc>> will not fit on <<io.thedesc>>. ";
		}
	}
;

chair : chairitem
	heredesc = {listhere(self.contents);}
	sdesc = "chair"
	location = startroom
	noun = 'chair'
	bulk = 5
;

chickenbroth : soupparent
	sdesc = "can of chicken broth"
	ldesc = "This is a <<self.sdesc>> you just got at the store. "
	noun = 'broth' 'soup'
	adjective = 'chicken'
	location = workcruft
	verDoPutIn(actor, io) = {
		if (not actor.isCarrying(self)) {
			"%You're% not carrying <<self.thedesc>>. ";
		}
		else {
			inherited.verDoPutIn(actor, io);
		}
	}
	verDoPutOn(actor, io) = {
		if (not actor.isCarrying(self)) {
			"%You're% not carrying <<self.thedesc>>. ";
		}
		else {
			inherited.verDoPutOn(actor, io);
		}
	}
	doPutOn(actor, io) = { 
		if (io = rightbottomshelf) {
			"Done. ";
			self.moveInto(nil);
			doincscore(1, dumbwin);
			scored.soupaway := true;
		}
		else {
			inherited.doPutOn(actor, io);
		}
	}		
;

cruft : item
	sdesc = "pile of worthless cruft"
	ldesc = "Out-of-date catalogs, mortgage insurance offers...the usual. "
	noun = 'cruft'
	adjective = 'worthless'
	location = nil
	verDoThrowAway(actor) = {}
	doThrowAway(actor) = {
		"You open the door under the sink and put the cruft in the trash
		can.  <<leftcabinet.foundweasel>>";
		self.moveInto(nil);
		doincscore(1, dumbwin);
		scored.worthlesscruft := true;
	}
	verDoTrash(actor) = {}
	doTrash(actor) = {
		self.doThrowAway(actor);
	}
;

cyrano : catparent
	isHim = true
	isListed = nil
	heredesc = {
		if (not cyrano.angry) {
			"In a chair next to a counter a large orange cat looks up at 
			you sleepily and blinks his eyes. ";
		}
	}
	sdesc = "orange cat"
	ldesc = {
		"The orange cat is very large, but his nose is even larger. ";
		if (cyrano.angry) {
			"He looks mad! ";
		}
		else {
			"He looks sleepy and well-fed. ";	
		}
	}
	adesc = "an <<cyrano.sdesc>>"
	location = chair
	noun = 'cat' 'cyrano' 'george'
	adjective = 'orange' 'large' 'sleepy'
	angry = nil
	verDoTake(actor) = { "You must be joking. "; }
	verDoPet(actor) = {}
	doPet(actor) = {
		if (cyrano.angry) { 
			"\^<<cyrano.thedesc>> flinches away from your touch, angrily 
			twitching his tail. ";
		}
		else {
			"\^<<cyrano.thedesc>> purrs sleepily. ";
		}
	}
	verDoCallGeorge(actor) = {}
	doCallGeorge = {
		"\^<<cyrano.thedesc>> looks at you reproachfully. ";
	}

	verDoHug(actor) = {}
	doHug = {
		if (cyrano.angry) { 
			"\^<<cyrano.thedesc>> flinches away from your touch, angrily 
			twitching his tail. ";
		}
		else {
			"\^<<cyrano.thedesc>> purrs squeekily. ";
		}
	}
;
cyranose : cyrano
    isListed = nil
	heredesc = nil
	sdesc = "orange nose"
	ldesc = "Wow! would you look at the nose on him! "
	location = chair
	noun = 'nose'
	adjective = 'Cyrano\'s' 'large'
;

fiver : catparent
	isHim = true
	sdesc = "black cat"
	ldesc = "\^<<self.thedesc>> circles the orange cat warily, looking for an opening. "
	location = nil
	noun = 'cat' 'Fiver'
	adjective = 'black' 
	verDoTake(actor) = { "He lithely eludes your grip. "; }
	verDoPet(actor) = {}
	doPet = {
		"\^<<self.thedesc>> ignores you.  He seems more interested in
		defending his territory right now. ";
	}
	verDoHug(actor) = {}
	doHug(actor) = {self.doPet(actor);}
;

newspapers : readable
	adesc = "some newspapers"
	sdesc = "stack of newspapers"
	ldesc = "This is a <<self.sdesc>>, all of which you have read (well, at
			least the comics anyway).  They belong in the recycling bin in the
			pantry. "
	location = counter
	noun = 'newspapers' 'newspaper' 'comics' 'stack'
	bulk = 2
	readdesc = "You start to re-read the comics, but they're a little boring the
			second time around, and besides, you're a little uneasy about
			what might happen to you if the kitchen is not cleaned up when
			your mate gets home. "
	verDoPutIn(actor, io) = {
		if (not actor.isCarrying(self)) {
			"%You're% not carrying "; self.thedesc; ". ";
		}
		else if (io <> newspaperbox) {
			"I think the newspapers would go better in the newspaper box
			than in <<io.thedesc>>. ";
		}
		else {
			inherited.verDoPutOn(actor, io);
		}
	}
	doPutIn(actor, io) = { 
		if (io = newspaperbox) {
			self.moveInto(nil);
			"Ok, the newspapers are now in the newspaper box. ";
			oldpapers.moveInto(newspaperbox);
			doincscore(1, dumbwin);
			scored.didnewspapers := true;
		}
		else {
			inherited.doPutIn(actor, io);
		}
	}		
;

oldpapers : item
	location = nil
	sdesc = "old papers"
	ldesc = "%You've% read these. "
	noun = 'newspapers' 'newspaper' 'comics' 'stack'
	doTake(actor) = { "You've read those. ";}
;

randomcruft : item
	issorted = nil
	sdesc = "pile of random cruft"
	ldesc = "This is a pile of bills, catalogs, announcements, and other assorted
			papers that may or may not be important.  Perhaps if you sorted it... "
	noun = 'cruft'
	adjective = 'random'
	location = counter
	verDoSort(actor) = {}
	doSort = {
		"You now have two piles of papers, one of which is <<cruft.adesc>>
		and one of which is <<vipapers.adesc>>";
		self.issorted := true;
		vipapers.moveInto(self.location);
		cruft.moveInto(self.location);
		self.moveInto(nil);
	}
;

note : readable
	sdesc = "note"
	ldesc = {
		if (not Me.isCarrying(self)) {
			"(taking note first)";
			self.moveInto(Me);
		}
		"\b\"Hi sweetie!
	
		\bI had to go out for a bit, could you please straighten up
		the kitchen before people come by for Lower Slobovian dancing
		tonight?  Thanks!\"

		\bThe signature is illegible. ";
	}
	location = counter
	noun = 'note'
;

sleepingweasel : item
	location = nil
	sdesc = "sleeping ferret"
	ldesc = "She's cute, but a little wiggly. "
	noun = 'weasel' 'ferret' 'Gojira'
	verDoPet(actor) = {}
	doPet = { "\^<<self.thedesc>> twitches a little in her sleep. "; }	
	verDoHug(actor) = {}
	doHug(actor) = {self.doPet(actor);}
	verDoTake(actor) = {}
	doTake(actor) = { "You'll never get anything done if you wake her up again! ";}
;

snoboldaemon: function {
	if (snobol.petcount > 0) {
		if (snobol.petrest < 0) {
			snobol.petrest := 10;
		}
		snobol.petrest := snobol.petrest - 1;
		if (snobol.petrest < 0) {
			snobol.petcount := snobol.petcount - 1;
		}
	}
	if (snobol.weaselcount > 0) {
		if (snobol.weaselrest < 0) {
			snobol.weaselrest := 10;
		}
		snobol.weaselrest := snobol.weaselrest - 1;
		if (snobol.weaselrest < 0) {
			snobol.weaselcount := snobol.weaselcount - 1;
		}
	}
}

snobol : catparent
	isHer = true
	petcount = 0
	weaselcount = 0
	sdesc = "gray cat"
	ldesc = "The gray cat is washing her face. "
	heredesc = "A gray cat sits contentedly washing her face. "
	isListed = nil
	location = nil
	petrest = 10
	weaselrest = 10
	noun = 'cat' 'Snobol'
	adjective = 'gray' 'grey'
	verDoTake(actor) = {
		"As you start to pick her up, she growls deep in her throat.
		You decide to put her back down again. ";
	}
	verDoPet(actor) = {}
	doPet = {
		if (self.petcount = 0) {
			"\^<<self.thedesc>> lets you pet her for about five seconds.
			Then she slaps at your hand.  Fortunately for you, her claws were
			in.  This time. ";
		}
		else if (self.petcount = 1) {
			"\^<<self.thedesc>> slaps at your hand.
			Her claws were out this time. Fortunately for you,
			the damage is minimal. ";
		}
		else if (self.petcount = 2) {
			"Realizing that you are a slow learner, <<self.thedesc>>
			gives you a scar to help you remember. ";
		}
		else {
			"You rush to the sink, searching for a towel to staunch the blood flow.
			Blurred as your vision is, you still notice <<self.thedesc>>
			storm out of the kitchen in a huff. ";
			self.moveInto(nil);
		}
		self.petcount := self.petcount + 1;
	}
	verDoHug(actor) = {}
	doHug(actor) = {self.doPet(actor);}
;

soupparent : item
	verDoEat(actor) = {}
	doEat(actor) = {
		"Unfortunately, %you% cannot find the can opener. ";
	}
;

soup : soupparent
	isListed = nil
	sdesc = "soup on the shelf"
	noun = 'soup'
	adjective = 'shelf'
	location = rightbottomshelf
	verDoTake(actor) = {}
	doTake(actor) = { "You're not particularly hungry right now. "; }
;

vipapers : item
	sdesc = "pile of important papers"
	ldesc = "This is <<self.adesc>>. "
	noun = 'papers'
	adjective = 'important'
	location = nil
;

weasel : item
	isHer = true
	lastopened = nil
	location = nil
	sdesc = "ferret"
	ldesc = "She's cute, but a little wiggly. "
	wasincage = 0
	noun = 'weasel' 'ferret' 'Gojira'
	verDoPet(actor) = {}
	doPet = { "\^<<self.thedesc>> lovingly gnaws on your fingertips. "; }	
	verDoHug(actor) = {}
	doHug(actor) = {self.doPet(actor);}
	doPutIn(actor, io) = { 
		inherited.doPutIn(actor, io);
	}		
;

catparent : item
    verDoAttackWith(actor, io) = {
		if (io <> broom) {
			inherited.verDoAttackWith(actor, io);
		}
    }
    doAttackWith(actor, io) = {
		if (io = broom) {
			self.doYellAt(actor);
		}
    }
	verDoYellAt(actor) = {}
	doYellAt(actor)  = {
		if (cyrano.angry and self <> snobol) {
			"That got their attention!  In their haste to get away from you,
			the cats have upset the recycling boxes in the pantry. ";
			StopFight();
		}
		else if (self.location) {
			"How could you do such a mean thing such a friendly kitty?";
		}
	}
;

cats : fixeditem, movableActor, catparent
	actorDesc = {}
	sdesc = "cats"
	noun = 'cats'
	location = nil
	roomCheck(verb) = {}
	actorAction(verb, dobj, prep, io) = {
		if (verb <> stopVerb) {
			"<<self.thedesc>> ignore you. ";
			exit;
		}
	}
;

workcruft : item
	sdesc = "pile of work cruft"
	ldesc = {
		"This is the stuff you are constantly bringing back and forth from
		work.  Everything here can be put upstairs";
		if (chickenbroth.location = workcruft) {
			" except for the chicken
			broth you picked up from the store.  You recall that soup goes in
			the right lazy susan cabinet. ";
		}
		else {
			". ";
		}
	}
	noun = 'cruft' 'workcruft'
	adjective = 'work'
	location = counter
;

/*
 *  VERBS
 */

petVerb : deepverb
	sdesc = "pet"
	verb = 'pet' 'stroke' 'kiss' 'scratch'
	doAction = 'Pet'
;	

hugVerb : deepverb
	sdesc = "hug"
	verb = 'hug' 'squeeze'
	doAction = 'Hug'
;	

goVerb : deepverb
	sdesc = "go"
	verb = 'go'
	doAction = 'Go'
;	

TrashVerb : deepverb
	sdesc = "trash"
	verb = 'trash' 'throw away'
	doAction = 'Trash'
;	

stopVerb : deepverb
	sdesc = "stop"
	verb = 'stop'
	doAction = 'Stop'
	action( actor ) = {
		if (actor = cats) {
			"The cats prick up their ears a bit at this, but sense that you
			don't really mean it. ";
		}
		else {
			"%You% don't stop. ";
		}
	}
;	

sortVerb : deepverb
	sdesc = "sort"
	verb = 'sort'
	doAction = 'Sort'
;
sweepVerb : deepverb
	sdesc = "sweep"
	verb = 'sweep'
	prepDefault = withPrep
	ioAction(withPrep) = 'SweepWith'
;

georgePrep: Prep
    preposition = 'george'
    sdesc = "george"
;

isPrep: Prep
    preposition = 'is' 'are'
    sdesc = "is"
;

callgeorgeVerb : deepverb
	sdesc = "call george"
	verb = 'call george'
	doAction = 'CallGeorge'
    invalidObj(actor, obj, name) = {
        if (not obj.isVisible(actor))
            "I don't see any <<name>> here. ";
        else
            "%You% can't call <<obj.thedesc>> George! ";
	}
;

parseError : function(errnum, errstr) {
	return errstr;
}

preparse : function(str)
{
	return true;
}

shelfitem : item
	adesc = "<<self.sdesc>>"
	isPlural = true
	YouCant = {
		if (self.isPlural) {
			"\^<<self.thedesc>> are right where they belong. ";
		}
		else {
			"\^<<self.thedesc>> is right where it belongs. ";
		}
	}
    verDoTake( actor ) =
    {
        self.YouCant;
    }
    verDoTakeOut( actor, io ) =
    {
        self.YouCant;
    }
    verDoDrop( actor ) =
    {
        self.YouCant;
    }
    verDoTakeOff( actor, io ) =
    {
        self.YouCant;
    }
    verDoPutIn( actor, io ) =
    {
        self.YouCant;
    }
    verDoPutOn( actor, io ) =
    {
        self.YouCant;
    }
    verDoMove( actor ) =
    {
        self.YouCant;
    }
;
cannedveggies : shelfitem
	sdesc = "canned veggies"
	noun = 'veggies' 'vegetables' 'vegetable' 'can'
	adjective = 'canned'
	location = leftbottomshelf
;

driedveggies : shelfitem
	sdesc = "dried veggies"
	noun = 'veggies' 'vegetables' 'vegetable' 'can'
	adjective = 'dried'
	location = leftbottomshelf
;

cannedmeats : shelfitem
	sdesc = "canned meats"
	noun = 'meats' 'can' 'meat'
	adjective = 'canned'
	location = leftbottomshelf
;

tomatoproducts : shelfitem
	sdesc = "tomato products"
	noun = 'tomatos' 'catsup' 'ketchup' 'katchup' 'products' 'can'
	adjective = 'tomato'
	location = leftbottomshelf
;

jams : shelfitem
	sdesc = "jams"
	noun = 'jams' 'jam'
	location = lefttopshelf
;

jellies : shelfitem
	sdesc = "jellies"
	noun = 'jellies' 'jelly'
	location = lefttopshelf
;

honey : shelfitem
	isPlural = nil
	sdesc = "honey"
	noun = 'honey'
	location = lefttopshelf
;

molasses : shelfitem
	isPlural = nil
	sdesc = "molasses"
	noun = 'molasses'
	location = lefttopshelf
;

cannedfruits : shelfitem
	sdesc = "canned fruits"
	noun = 'fruits' 'can' 'fruit'
	adjective = 'canned'
	location = lefttopshelf
;

raisins : shelfitem
	sdesc = "raisins"
	noun = 'raisins' 'raisin'
	location = lefttopshelf
;

noodles : shelfitem
	sdesc = "noodles"
	noun = 'noodles' 'noodle'
	location = rightbottomshelf
;

rice : shelfitem
	isPlural = nil
	sdesc = "rice"
	noun = 'rice'
	location = rightbottomshelf
;

maccheese : shelfitem
	isPlural = nil
	sdesc = "macaroni and cheese"
	noun = 'macaroni' 'cheese'
	location = rightbottomshelf
;

crackers : shelfitem
	sdesc = "crackers"
	noun = 'crackers' 'cracker'
	location = righttopshelf
;

nuts : shelfitem
	sdesc = "nuts"
	noun = 'nuts' 'nut'
	location = righttopshelf
;

ricecakes : shelfitem
	sdesc = "rice cakes"
	noun = 'cakes' 'cake'
	adjective = 'rice'
	location = righttopshelf
;

instantoatmeal : shelfitem
	isPlural = nil
	sdesc = "instant oatmeal"
	noun = 'oatmeal'
	adjective = 'instant'
	location = rightbottomshelf
;

peanutbutter : shelfitem
	isPlural = nil
	sdesc = "peanut butter"
	noun = 'butter'
	adjective = 'peanut'
	location = rightbottomshelf
;

junkfood : shelfitem
	isPlural = nil
	isListed = nil
	heredesc = "This looks like a good place for most types of junk food. "
	sdesc = "junk food"
	noun = 'food' 'types'
	adjective = 'junk' 'most'
	location = pantryfirstshelf
;

bakingmixes : shelfitem
	sdesc = "baking mixes"
	noun = 'mixes' 'mix'
	adjective = 'baking'
	location = pantryfirstshelf
;

moonpies : shelfitem
	sdesc = "moon pies"
	noun = 'pies' 'pie'
	adjective = 'moon'
	location = pantryfirstshelf
;

cookies : shelfitem
	sdesc = "cookies"
	noun = 'cookies' 'cookie'
	location = pantryfirstshelf
;

potatochips : shelfitem
	sdesc = "potato chips"
	noun = 'chips' 'chip'
	adjective = 'potato'
	location = pantryfirstshelf
;

pudding : shelfitem
	isPlural = nil
	sdesc = "pudding"
	noun = 'pudding'
	adjective = 'chocolate' 'vanilla' 'fudge' 'tapioca'
	location = pantryfirstshelf
;

cereal : shelfitem
	isPlural = nil
	sdesc = "cereal"
	noun = 'cereal'
	location = pantryfirstshelf
;

catfood : shelfitem
	isPlural = nil
	sdesc = "cat food"
	noun = 'food' 'fud'
	adjective = 'cat'
	location = pantryfourthshelf
;

catstuff : shelfitem
	isPlural = nil
	sdesc = "cat stuff"
	adesc = "other cat stuff"
	noun = 'stuff'
	adjective = 'cat' 'other'
	location = pantryfourthshelf
;

parchmentpaper : shelfitem
	isPlural = nil
	sdesc = "parchment paper"
	noun = 'paper'
	adjective = 'parchment'
	location = pantryfourthshelf
;

aluminumfoil : shelfitem
	isPlural = nil
	sdesc = "aluminum foil"
	noun = 'foil'
	adjective = 'aluminum'
	location = pantryfourthshelf
;

sterno : shelfitem
	isPlural = nil
	sdesc = "sterno"
	noun = 'sterno'
	location = pantryfourthshelf
;

grocerybags : shelfitem
	sdesc = "mesh grocery bags"
	noun = 'bags' 'bag'
	adjective = 'mesh' 'grocery'
	location = pantryfourthshelf
;

chocolatebar : shelfitem
	isPlural = nil
	sdesc = "HUGE bar of chocolate"
	adesc = "a <<self.sdesc>>"
	noun = 'bar'
	adjective = 'huge' 'chocolate'
	location = pantryfourthshelf
;

driedmilk : shelfitem
	isPlural = nil
	sdesc = "dried milk"
	noun = 'milk'
	adjective = 'dried'
	location = pantrysecondshelf
;

vinegar : shelfitem
	isPlural = nil
	sdesc = "vinegar"
	noun = 'vinegar'
	adjective = 'cider' 'white'
	location = pantrysecondshelf
;

bakingpowder : shelfitem
	isPlural = nil
	sdesc = "baking powder"
	noun = 'powder'
	adjective = 'baking'
	location = pantrysecondshelf
;

cornsyrup : shelfitem
	isPlural = nil
	sdesc = "corn syrup"
	noun = 'syrup'
	adjective = 'corn'
	location = pantrysecondshelf
;

oils : shelfitem
	sdesc = "oils"
	adesc = "various kinds of oils"
	noun = 'oils' 'oil'
	adjective = 'various' 'kinds'
	location = pantrysecondshelf
;

oatmeal : shelfitem
	isPlural = nil
	sdesc = "oatmeal"
	noun = 'oatmeal'
	location = pantrysecondshelf
;

bakingchocolate : shelfitem
	isPlural = nil
	sdesc = "baking chocolate"
	noun = 'chocolate'
	adjective = 'baking' 'unsweetened' 'semisweet' 'semi-sweet' 'milk' 
	location = pantrysecondshelf
;

cannedmilk : shelfitem
	isPlural = nil
	sdesc = "canned milk"
	noun = 'milk'
	adjective = 'canned' 'condensed' 'evaporated'
	location = pantrysecondshelf
;

gelatin : shelfitem
	isPlural = nil
	sdesc = "gelatin"
	noun = 'gelatin'
	location = pantrysecondshelf
;

shortening : shelfitem
	isPlural = nil
	sdesc = "shortening"
	noun = 'shortening' 'crisco'
	location = pantrysecondshelf
;

bakingsupplies : shelfitem
	isPlural = nil
	isListed = nil
	heredesc = "This looks like a good place for most baking supplies. "
	sdesc = "baking supplies"
	noun = 'supplies'
	adjective = 'baking' 'most'
	location = pantrysecondshelf
;

sugars : shelfitem
	sdesc = "various sugars"
	noun = 'sugars' 'sugar'
	adjective = 'brown' 'powdered' 'granulated' 'confectioners' 'various'
	location = pantrythirdshelf
;

flours : shelfitem
	sdesc = "various flours"
	noun = 'flours' 'flour'
	adjective = 'white' 'whole' 'wheat' 'bread' 'cake' 'various'
	location = pantrythirdshelf
;

bakingsoda : shelfitem
	isPlural = nil
	sdesc = "baking soda"
	noun = 'soda'
	adjective = 'baking'
	location = pantrythirdshelf
;

popcorn : shelfitem
	isPlural = nil
	sdesc = "popcorn"
	noun = 'popcorn' 'corn'
	adjective = 'pop'
	location = pantrythirdshelf
;

drinkmixes : shelfitem
	sdesc = "drink mixes"
	noun = 'mixes' 'mix'
	adjective = 'drink' 'kool' 'aid' 'lemonade'
	location = pantrythirdshelf
;

whatever : shelfitem
	isListed = nil
	heredesc = "This looks like a good place for baking supplies and whatever won't fit anywhere else. "
	sdesc = "whatever"
	noun = 'whatever'
	location = pantrythirdshelf
;

nonfood : shelfitem
	isListed = nil
	heredesc = "This looks like a good place for non-food items or items that are too big for the other shelves. "
	sdesc = "non-food items"
	noun = 'items' 'item'
	adjective = 'non-food' 'non' 'food' 'too' 'big'
	location = pantryfourthshelf
;

catfoodbags : shelfitem
	sdesc = "large bags of cat food"
	noun = 'food' 'bag' 'bags'
	adjective = 'cat' 'large'
	location = pantryfloor
;

sodas : shelfitem
	sdesc = "sodas"
	noun = 'sodas' 'soda' 'case' 'cases'
	location = pantryfloor
;

eyes : item
	adesc = "two beady eyes"
	sdesc = "pair of beady eyes"
	verDoTake(actor) = {}
	doTake(actor) = { self.ldesc; }
;

eyesleft : eyes
	ldesc = {
		"Oops!  Those beady eyes belong to your pet ferret, who scurries out
		from between the canned meats and onto the floor.  You hurriedly
		pick her up. ";
		self.moveInto(nil);
		weasel.moveInto(Me);
		leftcabinet.foundeyes := true;
		setdaemon(weaseldaemon, 0);
	}
	location = leftbottomshelf
	noun = 'eyes' 'eye'
	adjective = 'beady' 'two'
;

eyesright : eyes
	ldesc = {
		"Oops!  Those beady eyes belong to a mouse, probably one of the ones you've
		seen footprints of in your workshop.  You hurriedly back up as the mouse
		runs across the floor and straight into the waiting jaws of a gray
		cat who has just come in.

		\b*CRUNCH* \ Well, that's the end of that mouse! ";
		self.moveInto(nil);
		snobol.moveInto(startroom);
		rightcabinet.foundeyes := true;
		doincscore(1, dumbwin);
		scored.eatmouse := true;
	}
	location = rightbottomshelf
	noun = 'eyes' 'eye'
	adjective = 'beady' 'two'
;
