/*
 * MAGNETIC
 * A "Magnetic Scrolls" adventure game emulator.
 *
 * Original version by: Niclas Karlsson, 1997
 * Unix port and cleanup: Jeff Mitchell, 1997
 *
 */

/* unix.c
 * This source module includes the code for the X11 port of Magnetic.
 * The X11 general libraries are in x11lib.c, but most of the game
 * calls should be routed through here.. it is hoped that the x11lib
 * file should know nothing of the game for which it is being used.
 */

/* Yes I'll rewrite a lot of this when I get a chance.. to use sane
 * data structures, more optimized output routines, etc etc. Don't
 * bug me ;)
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "defs.h"
#include "x11lib.h"
#include "unix.h"
#include "main.h"

/* types
 */

typedef struct {
  char *uol_name;    /* option name (ie: the part following dash) */
  int   uol_value;   /* boolean; if true, option requires a value. */
  char *uol_help;    /* help text for -h request */
} _unix_option_list;

_unix_option_list unix_options[] = {
  { "rows",       1, "No. of text rows in scroll window" },
  { "intensity",  1, "Modify (+/-) brightness" },
  { "nointro",    0, "Skip game specific special intro" },
  { "rawtext",    0, "Print to stdout, input from stdin" },
  { "nopics",     0, "No pictures; implies -nointro" },
  /* { "scale",      1, "Scale no more than <arg> times" }, */
  /* { "scalex",     1, "Force image X scaling to be <arg>" }, */
  /* { "scaley",     1, "Force image Y scaling to be <arg>" }, */
  { "display",    1, "Set the display; ie: -display localhost:0.0" },
  { "font",       1, "Specify the font. Default is 9x16." },
  { "root",       0, "Force display on the root window." },
  { "dump",       1, "Start dumping registers after n instructions." },
  /* { "rscript",    1, "Use specified file as a script" }, */
  /* { "transcript", 1, "Use specified file as a transcript." }, */
  /* { "wscript",    1, "Use specified file to write script to." }, */
  { "safety",     1, "Exit after <arg> instructions." },
  { NULL,      0, NULL }
};

typedef struct {
  int uol_option;
  char uol_name [ MAXLINE ];
  char uol_value [ MAXLINE ];
} unix_commandline;

char *preferred_fonts[] = {
  "12x24", "10x20", "9x15", "8x16", "7x14", "6x10", "5x8", NULL
};

char *font_measure = "MMMMMMMMMMMMMMMMMMMM" \
                     "MMMMMMMMMMMMMMMMMMMM" \
                     "MMMMMMMMMMMMMMMMMMMM" \
                     "MMMMMMMMMMMMMMMMMMMM";
char *font_border =  "MMMM";

typedef struct {
  char *ghd_name;                  /* for conveniance */
  int   ghd_offset;                /* offset in file to compare */
  char *ghd_magic;                 /* buffer for comparison */
  int   ghd_length;                /* buffer length */
  void (*ghd_handler)(void);       /* on comparison true, call this func */
} game_handler_detail;

game_handler_detail game_handlers[] = {
  { "The Pawn", 0, "MaSc *   *    ", 16, pawn_intro },
  { "Myth",     0, "MaSc *   *   ", 16, myth_intro },
  { NULL, 0, NULL, 0, NULL }
};

typedef struct {
  unsigned short p_palette [ 16 ];
  unsigned short p_width;
  unsigned short p_height;
  Pixmap         p_pixmap;
} picture;

/* globals
 */

static unix_commandline options [ MAXOPTIONS ];
static int option_count = 0;
static char requests [ MAXOPTIONS ][ MAXLINE ];
static int request_count = 0;

static int desired_x_scale_factor = -1;          /* x image scale, or guess */
static int desired_y_scale_factor = -1;          /* y image scale, or guess */
static int desired_intensity = 0;                /* colour scaling */
static int desired_rows = DESIRED_ROWS;          /* height of scroll window */
static int more_counter = 0;                     /* for [MORE] prompt */
static int more_prompt = 0;                      /* MORE is on currently? */

static char *desired_font = "8x16";              /* default font */

static int calculated_image_width = -1;          /* sized canon graphic */
static int calculated_image_height = -1;         /* sized canon graphic */

static int current_picture = -1;                 /* current picture index */
static picture cached_pictures [ MAXPICTURES ];  /* picture cache */
static char statusbuffer [ MAXLINE ];            /* statusline */
static char textbuffer [ LINEBUF ] [ 90 ];       /* scrolling text buffer */
static int textbufferlength = 0;                 /* no of valid lines in buf */

static char readykeybuffer [ MAXLINE ];          /* an entire ready line */
static char keybuffer [ MAXLINE ];               /* key inbuffer */
static int keybufferlength = 0;                  /* depth of buffer */

/* local library code
 */

/* unix_usage() displays the available commandline options.
 */
void unix_usage ( void ) {
  int iter = 0;

  printf ( "usage: ./magnetic [options] gamename\n" );

  printf ( "\n" );
  printf ( "Available options are:\n" );

  while ( unix_options [ iter ].uol_name ) {
    printf ( "%-15s%-15s%s\n",
	     unix_options [ iter ].uol_name,
	     unix_options [ iter ].uol_value ? "<argument>" : "",
	     unix_options [ iter ].uol_help );
    iter++; /* next option */
  }

  printf ( "\n" );
  printf ( "The interpreter has these additional meta-commands:\n" );
  printf ( "#undo\t\tHas the obviou usage.\n" );
  printf ( "#logoff\t\tTurns off script recording.\n" );
  printf ( "\n" );
  printf ( "Many more options are in the works; scripting, multiple\n" );
  printf ( "windows, shell like command history and aliases.\n" );
  printf ( "\n" );

}

/* runs through the provided commandline options, stuffing them into
 * the option-detail array; returns the number of options identified.
 */
int unix_identify_options ( int argc, char *argv[] ) {
  int iter = 1; /* skip the program init path */
  int argcount = 0;

  while ( iter < argc ) {
    int argmatch = 0;

    if ( argv [ iter ][0] != '-' ) {
      strcpy ( requests [ request_count ], argv [ iter ] );
      iter++;          /* increment counter */
      argcount++;      /* count this as an argument */
      request_count++; /* increment counter */
      continue;        /* skip to next argument */
    }

    /* run through known options and see if one of them matches
     * this particular user provided argument. If not, its an
     * error.
     */
    while ( unix_options [ argmatch ].uol_name ) {

      if ( strcmp ( argv [ iter ] + 1, /* skip the leading dash */
		    unix_options [ argmatch ].uol_name ) == 0 )
      {
	break;
      }

      argmatch++;
    }

    if ( unix_options [ argmatch ].uol_name ) {
      /* copy argument into options buffer */
      options [ option_count ].uol_option = argmatch;
      strcpy ( options [ option_count ].uol_name, argv [ iter ] + 1 );

      /* does this option require an argument? */
      if ( unix_options [ argmatch ].uol_value ) {

	/* yuss! */
	if ( iter < ( argc - 1 ) ) {
	  iter++;
	  strcpy ( options [ option_count ].uol_value, argv [ iter ] );

	} else {
	  printf ( "Option %s requires an argument!\n",
		   argv [ iter ] );
	  return ( 0 );
	}

      }

      /* increment counter */
      option_count++;

    } else {
      printf ( "Bad commandline option -%s specified!\n",
	       argv [ iter ] );
      return ( 0 );
    }

    iter++;
  } /* while iter < argc */

  /* return number of arguments */
  return ( argcount );
}

int unix_option ( char *name ) {
  int iter = 0;

  for ( iter = 0; iter < option_count; iter++ ) {
    if ( strcmp ( options [ iter ].uol_name, name ) == 0 ) {
      return ( 1 );
    }
  }

  return ( 0 );
}

char *unix_option_value ( char *name ) {
  int iter = 0;

  for ( iter = 0; iter < option_count; iter++ ) {
    if ( strcmp ( options [ iter ].uol_name, name ) == 0 ) {
      return ( options [ iter ].uol_value );
    }
  }

  return ( NULL );
}

char *unix_request_value ( int x ) {
  return ( requests [ x ] );
}

void unix_init_graphics_engine ( void ) {
  extern int desired_window_width;
  extern int desired_window_height;
  int iter = 0;

  /* see if we need it at all...
   */
  if ( unix_option ( "rawtext" ) && unix_option ( "nopics" ) ) {
    return; /* no graphics mode needed */
  }

  /* check the unix specific commandline options */
  if ( unix_option ( "font" ) ) {
    desired_font = strdup ( unix_option_value ( "font" ) );
  }

  if ( unix_option ( "intensity" ) ) {
    desired_intensity = atoi ( unix_option_value ( "intensity" ) );
  }

  if ( unix_option ( "rows" ) ) {
    desired_rows = atoi ( unix_option_value ( "rows" ) );
  }

  /* open the initial connection to the display */
  init_graphics_engine();

  /* retrieve the desired system font */
  if ( ! retrieve_font ( FONT_SYSTEM, desired_font ) ) {
    printf ( "Couldn't retrieve requested system font %s!\n", desired_font );
    exit ( 0 );
  }

  if ( ! retrieve_font ( FONT_STATUS, desired_font ) ) {
    printf ( "Couldn't retrieve requested status font %s!\n", desired_font );
    exit ( 0 );
  }

  /* attempt to size the window -- identify font, calculate
   * its size to fit, figure out graphic scaling factor, etc.
   */
  /* Rules:
   * (1) Window must have space for a graphics window (top) and a
   *     textual window (bottom).
   * (2) The textual window must have at least DESIRED_ROWS rows of space.
   * (3) The graphical area should be the highest integral scale
   *     of the canonicle image size that can fit within the width
   *     of the text.
   * (4) Window width must have enough room for 80 characters across,
   *     where the character to measure against is the letter M. There
   *     ought to be a couple of characters padding to each side, as
   *     well.
   */
  desired_window_width =
    query_font_width ( FONT_SYSTEM, font_measure ) + /* 80 chars min */
    query_font_width ( FONT_SYSTEM, font_border );

  /* given the window width, calculate the max image size that can fit
   * horizontally across.
   */
  if ( ! unix_option ( "nopics" ) ) {

    desired_x_scale_factor = desired_window_width / CANONICLE_WIDTH;
    desired_y_scale_factor = desired_x_scale_factor;

    calculated_image_width = CANONICLE_WIDTH * desired_x_scale_factor;
    calculated_image_height = CANONICLE_HEIGHT * desired_y_scale_factor;

  }

  /* with this happy ratio in hand, calculate what height the image
   * would consume. Desirable window height is
   * image_consumption + text_consumption + frame
   */
  if ( unix_option ( "rawtext" ) ) {
    desired_window_height =
      calculated_image_height +                       /* image consumption */
      ( query_font_height ( FONT_SYSTEM ) * 4  );     /* border consumption */
  } else {
    desired_window_height =
      calculated_image_height +                       /* image consumption */
      ( query_font_height ( FONT_SYSTEM ) * desired_rows ) + /* text consump */
      ( query_font_height ( FONT_SYSTEM ) * 4  );     /* border consumption */
  }

  /* start the graphic subsystem */
  open_graphics_engine ( "magnetic/x11" );

  /* create the GC's for the 16 palette entries */
  for ( iter = 0; iter < 16; iter++ ) {
    set_palette_entry ( iter, 1, 1, 1 ); /* default silly value */
  }

  /* set up keyboard handler */
  X11ResetVirtualAlarm();                           /* first time */
  X11InstallVirtualAlarm();                         /* start ticking... */
  handleExpose = unix_expose_event;                 /* Expose event handler */
  handleKeyRelease = unix_keyrelease_event;

}

void unix_expose_event ( void ) {
  unix_refresh_display();
}

void unix_keyrelease_event ( char key, KeySym keysym ) {
  static int historycursor = -1;

  if ( ( keysym == XK_BackSpace ) ||  /* backspace */
       ( keysym == XK_Delete ) )      /* delete */
  {

    if ( keybufferlength ) {
      keybufferlength --;
      keybuffer [ keybufferlength ] = '\0';
      unix_put_text ( "", 0 ); /* draw to screen */
    }

  } else if ( keysym == XK_Up ) {     /* command history */

    if ( historycursor < 0 ) {
      if ( unix_query_history_depth() ) {
	historycursor = unix_query_history_depth() - 1;
      }
    } else if ( historycursor ) {
      historycursor --;
    }

    strncpy ( keybuffer, unix_query_history ( historycursor ), MAXLINE );
    keybufferlength = strlen ( keybuffer );

    unix_put_text ( "", 0 );

  } else if ( keysym == XK_Down ) {  /* command history */

    if ( historycursor < 0 ) {
      historycursor = 0;
    } else if ( historycursor < unix_query_history_depth() ) {
      historycursor ++;
    }

    strncpy ( keybuffer, unix_query_history ( historycursor ), MAXLINE );
    keybufferlength = strlen ( keybuffer );

    unix_put_text ( "", 0 );

  } else if ( isprint ( key ) ) {

    keybuffer [ keybufferlength ] = key;
    keybufferlength++;

    unix_put_text ( "", 0 );

  } else if ( ( key == '\r' ) || ( key == '\n' ) ) {

    unix_append_history ( keybuffer );
    historycursor = -1;

    keybuffer [ keybufferlength ] = '\n';
    keybufferlength++;

    strncpy ( readykeybuffer, keybuffer, keybufferlength );

    bzero ( keybuffer, MAXLINE );
    keybufferlength = 0;

  } else {
    fprintf ( stderr, "failed %d (%c)\n", key, key );
  }

}

void unix_shutdown_graphics_engine ( void ) {

  /* destroy the palette GC's */

  /* shutdown the graphic subsystem */
  shutdown_graphics_engine();

}

void unix_draw_pixel ( int x, int y ) {
  int xiter, yiter;

  /* do the desired scaling */
  for ( xiter = 0; xiter < desired_x_scale_factor; xiter++ ) {

    for ( yiter = 0; yiter < desired_y_scale_factor; yiter++ ) {

      draw_pixel ( ( x * desired_x_scale_factor ) + xiter,
		   ( y * desired_y_scale_factor ) + yiter );

    } /* y scale count */

  } /* x scale count */

}

int unix_set_palette_entry ( int entry, int red, int green, int blue ) {
  return ( set_palette_entry ( entry,
			       red + desired_intensity,
			       green + desired_intensity,
			       blue + desired_intensity ) );
}

/* display wrappers
 */

void unix_refresh_display ( void ) {

  unix_refresh_picture();
  unix_refresh_status();
  unix_refresh_text();

  realize_pixmap ( 0, 0 );
  flush_display();

  return;
}

void unix_refresh_status ( void ) {

  if ( unix_option ( "rawtext" ) ) {
    printf ( "\n>>%s\n", statusbuffer );
    return;
  }

  /* clear the graphic from the display */
  set_target_pixmap ( query_default_pixmap() );

  fill_region ( COLOUR_BLACK, 0, 0,
		query_font_width ( FONT_STATUS, font_measure ),
		query_font_height ( FONT_STATUS ) * 3 );

  /* back to the virtual display; already had status bar cleared,
   * and has had the picture placed onto it.
   */
  set_target_pixmap ( query_default_pixmap() );
  set_font ( COLOUR_STATUS, FONT_STATUS );

  begin_text_draw();

  draw_text ( COLOUR_STATUS,
	      query_font_width ( FONT_STATUS, "MM" ),
	      query_font_height ( FONT_STATUS ) * 2,
	      statusbuffer );

  if ( more_prompt == 1 ) {

    draw_text ( COLOUR_STATUS,
		query_font_width ( FONT_STATUS, "M" ) * 40,
		query_font_height ( FONT_STATUS ) * 2,
		"[[ MORE ]]" );

  }

  end_text_draw();

}

void unix_refresh_picture ( void ) {
  extern int desired_window_width;
  extern int desired_window_height;
  picture *image;
  int iter;

  if ( unix_option ( "nopics" ) ) {
    return;
  }

  /* clear the graphic from the display */
  set_target_pixmap ( query_default_pixmap() );

  fill_region ( COLOUR_BLACK, 0, 0,
		desired_window_width, desired_window_height );

  /* do we have a picture yet? */
  if ( current_picture == -1 ) {
    /* printf ( "[ *** No image *** ]\n" ); */

  } else {
    image = &cached_pictures [ current_picture ];

#ifdef DEBUG
    printf( "\n\n[ Image %d (%dx%d). ]", current_picture,
	    cached_pictures [ current_picture ].p_width,
	    cached_pictures [ current_picture ].p_height );
    printf ( "\n" );
#endif

    /* set the new palette */
    for ( iter = 0; iter < 16; iter ++ ) {
      unix_set_palette_entry ( iter,
			       ( image -> p_palette [ iter ] & 0xf00 ) >> 4,
			       ( image -> p_palette [ iter ] & 0x0f0 ),
			       ( image -> p_palette [ iter ] & 0x00f ) << 4 );
    }

    /* set image to display */
    set_target_pixmap ( image -> p_pixmap );

    /* copy image to virtual, with centering */
    realize_pixmap ( ( desired_window_width -
		       unix_query_pixmap_width ( current_picture ) ) / 2,
		     ( query_font_height ( FONT_SYSTEM ) * 2 ) +
		       ( ( CANONICLE_HEIGHT * desired_y_scale_factor ) -
			 unix_query_pixmap_height ( current_picture ) )
		       / 2 );

  } /* current_picture is valid? */

}

void unix_refresh_text ( void ) {
  extern int desired_window_width;
  extern int desired_window_height;
  int baseline_offset = 0;

  if ( unix_option ( "rawtext" ) ) {
    return;
  }

  /* calculate the top y coord... */
  if ( ! unix_option ( "nopics" ) ) {

    baseline_offset =
      ( query_font_height ( FONT_SYSTEM ) * 2 ) +
      ( CANONICLE_HEIGHT * desired_y_scale_factor );

  } else {

    baseline_offset = ( query_font_height ( FONT_SYSTEM ) * 2 );

  }

  /* draw the main text window; this requires several stages:
   * (1) clear region (of status bar remains, etc.)
   * (2) run through buffer, displaying the visible text (which is
   *     always the last DESIRED_ROWS lines)
   */
  fill_region ( COLOUR_BLACK,
		0, baseline_offset,
		desired_window_width, desired_window_height );
  set_font ( COLOUR_SYSTEM, FONT_SYSTEM );

  /* draw the text window */
  begin_text_draw();
  {
    int iter;

    for ( iter = textbufferlength - desired_rows;
	  iter < textbufferlength;
	  iter++ )
    {
      char temp [ MAXLINE ]; /* to make command history easier */

      /* when running through the text window, be sure to add current
       * keyboard entries onto the last line. These characters are
       * in-edit, and not real, and as such aren't part of the
       * display array.
       */

      if ( iter == ( textbufferlength - 1 ) ) {
	sprintf ( temp, "%s%s_", textbuffer [ iter ], keybuffer );
      } else {
	sprintf ( temp, "%s", textbuffer [ iter ] );
      }

      draw_text ( COLOUR_SYSTEM,
		  query_font_width ( FONT_SYSTEM, "MM" ),
		  baseline_offset +
		    ( query_font_height ( FONT_SYSTEM ) * 2 ) +
		    ( query_font_height ( FONT_SYSTEM ) *
		      ( desired_rows - ( textbufferlength - iter ) ) ),
		  temp );
    }

  }
  end_text_draw();

}

int unix_unpack_picture ( int pic, int mode ) {
  extern int desired_window_height;
  extern int desired_window_width;

  unsigned char *raw;
  picture *image;
  int iter;
  int x, y;

  if ( unix_option ( "rawtext" ) && unix_option ( "nopics" ) ) {
    return ( 1 ); /* no graphics mode needed */
  }

  if ( pic > MAXPICTURES ) {
    printf ( "*** Internal error; not enough pixmaps!\n" );
    return ( 1 );
  }

  if ( cached_pictures [ pic ].p_pixmap != (Pixmap) None ) {
    return ( 1 ); /* already cached */
  }

  if ( ( cached_pictures [ pic ].p_pixmap = create_pixmap() ) ==
       (Pixmap) None )
  {
    printf ( "Couldn't create pixmap!\n" );
    return ( 0 );
  }

  image = &cached_pictures [ pic ];
  set_target_pixmap ( image -> p_pixmap );

  raw = ms_extract ( pic,
		     & ( image -> p_width ), & ( image -> p_height ),
		     image -> p_palette );

  fill_region ( COLOUR_BLACK, 0, 0,
		desired_window_width, desired_window_height );

  /* set the palette for the pixmap to cache */
  for ( iter = 0; iter < 16; iter ++ ) {
    unix_set_palette_entry ( iter,
			     ( image -> p_palette [ iter ] & 0xf00 ) >> 4,
			     ( image -> p_palette [ iter ] & 0x0f0 ),
			     ( image -> p_palette [ iter ] & 0x00f ) << 4 );
  }

  /* draw the image into the pixmap */
  begin_pixel_draw();

  for ( y = 0; y < image -> p_height; y++ ) {
    for ( x = 0; x < image -> p_width; x++ ) {
      set_pixel_draw_colour ( raw [ ( y * image -> p_width ) + x ] );
      unix_draw_pixel ( x, y );
    }
  }

  end_pixel_draw();

  return ( 1 );
}

int unix_query_pixmap_width ( int pic ) {
  return ( cached_pictures [ pic ].p_width * desired_x_scale_factor );
}

int unix_query_pixmap_height ( int pic ) {
  return ( cached_pictures [ pic ].p_height * desired_y_scale_factor );
}

int unix_show_picture ( int pic, int mode ) {
  current_picture = pic; /* for next run through */
  unix_refresh_display(); /* redraw the display */
  return ( 1 );
}

/* unix_put_text() adds ONE LINE to the scrolling text buffer; as such,
 * it may well knock off the oldest line. If 'newline' is true, a
 * new line is appended to the scrollwindow, otherwise the last line
 * in the buffer is appended to.
 */
int unix_put_text ( char *text, int newline ) {

  /* printf ( ">> %s (%2.2X)\n", text, text [ strlen ( text ) - 1 ] ); */

  /* append line to textbuffer */
  if ( newline ) {
    strncpy ( textbuffer [ textbufferlength ], text, 80 );
    textbufferlength++;
    more_counter ++; /* don't increment on empty lines */
  } else {
    strncat ( textbuffer [ textbufferlength - 1 ], text, 80 );
    more_counter = 0; /* flushed */
  }

  /* rotate buffer if necessary */
  if ( textbufferlength == LINEBUF ) {
    int iter = 0;

    /* for each buffer line (beyond fifth), juggle up 5 */
    for ( iter = 5; iter < textbufferlength; iter++ ) {
      strcpy ( textbuffer [ iter - 5 ], textbuffer [ iter ] );
    }

    textbufferlength -= 5;
  }

  /* if there are keys in the keyboard inbuffer, we know we're
   * just inside of a key-flush-loop, so don't update the
   * display until we're all out of characters.
   */
  if ( readykeybuffer [ 0 ] == '\0' ) {
    unix_refresh_text();
    realize_pixmap ( 0, 0 );
    flush_display();
  }

  /* check to see if MORE prompt is required */
  if ( more_counter > ( desired_rows + 1 ) ) {

    more_prompt = 1; /* turn on the more prompt */
    unix_refresh_status();
    flush_display();
    
    while ( unix_query_keybuffer_depth() < 1 ) {
      poll_events();      /* process window system events */
      usleep ( 50000 );   /* waste time until input is available */
    }

    while ( unix_pop_keybuffer() >= 0 ) {
      /* pop until empty, in case they typed a load of junk */
    }

    more_prompt = 0; /* turn off the more prompt */
    more_counter = 0; /* turn off more counter */
    unix_refresh_status();
    flush_display();

  }

  return ( 1 );
}

int unix_put_status ( char *text ) {
  strncpy ( statusbuffer, text, MAXLINE );

  unix_refresh_display(); /* redraw the display */

  return ( 1 );
}

int unix_query_keybuffer_depth ( void ) {
  return ( strlen ( readykeybuffer ) );
}

int unix_pop_keybuffer ( void ) {
  char tempbuffer [ MAXLINE ];
  char key;

  if ( strlen ( readykeybuffer ) == 0 ) {
    return ( -1 );
  }

  key = readykeybuffer [ 0 ];

  strcpy ( tempbuffer, readykeybuffer + 1 );   /* copy remainder */
  bzero ( readykeybuffer, MAXLINE );           /* wipe buffer */
  strcpy ( readykeybuffer, tempbuffer );       /* replace remainder */

  return ( key );
}

static char historybuffer [ MAXHISTORY ] [ MAXLINE ];
static int historybufferlength = 0;

void unix_append_history ( char *command ) {

  if ( historybufferlength ) {

    if ( strcmp ( command, historybuffer [ historybufferlength - 1 ] ) == 0 ) {
      return; /* same as last string in history */
    }

  }

  if ( historybufferlength == MAXHISTORY ) {
    int iter;

    for ( iter = 1; iter < MAXHISTORY; iter++ ) {
      strncpy ( historybuffer [ iter - 1 ],
		historybuffer [ iter ],
		MAXLINE );
    }

    historybufferlength -= 1;
  }

  strncpy ( historybuffer [ historybufferlength ], command, MAXLINE );
  historybufferlength ++;

}

int unix_query_history_depth ( void ) {
  return ( historybufferlength );
}

char *unix_query_history ( int index ) {
  return ( historybuffer [ index ] );
}

/* special game intro handlers, et al.
 */

void unix_special_intro ( void ) {
  int iter = 0;
  extern type8 header[];

  set_font ( COLOUR_SYSTEM, FONT_SYSTEM );

  begin_text_draw();
  draw_text ( COLOUR_SYSTEM, 10, 20, "Magnetic" );
  draw_text ( COLOUR_SYSTEM, 10, 40, "Built:" __DATE__ __TIME__ );
  draw_text ( COLOUR_SYSTEM, 10, 60, "v1.0" );
  end_text_draw();

  realize_pixmap ( 0, 0 );
  flush_display();
  sleep ( 1 );

  while ( game_handlers [ iter ].ghd_magic ) {

    if ( memcmp ( header,
		  game_handlers [ iter ].ghd_magic,
		  game_handlers [ iter ].ghd_length ) == 0 )
    {
      /* looks like we found a match, so call the handler and break
       * the loop.
       */

      printf ( "Looks like this is %s.\n", game_handlers [ iter ].ghd_name );
      (game_handlers [ iter ].ghd_handler)();
      break;

    }

    iter ++; /* inc counter */
  }

  if ( ! game_handlers [ iter ].ghd_magic ) {
    printf ( "Unknown game image.\n" );
  }

}

void pawn_intro ( void ) {
  extern int desired_window_height;
  extern int desired_window_width;

  fill_region ( COLOUR_BLACK, 0, 0,
		desired_window_width, desired_window_height );

  begin_text_draw();
  draw_text ( COLOUR_SYSTEM, 10, 20, "Magnetic" );
  draw_text ( COLOUR_SYSTEM, 10, 40, "Built:" __DATE__ __TIME__ );
  draw_text ( COLOUR_SYSTEM, 10, 60, "v1.0" );
  end_text_draw();

  begin_text_draw();
  draw_text ( COLOUR_SYSTEM, 10,
	      desired_window_height - ( 4 * query_font_height ( FONT_SYSTEM) ),
	      "Loading ." );
  end_text_draw();
  realize_pixmap ( 0, 0 );
  flush_display();
  sleep ( 1 );

  begin_text_draw();
  draw_text ( COLOUR_SYSTEM, 10,
	      desired_window_height - ( 4 * query_font_height ( FONT_SYSTEM) ),
	      "Loading .." );
  end_text_draw();
  realize_pixmap ( 0, 0 );
  flush_display();
  sleep ( 1 );

  begin_text_draw();
  draw_text ( COLOUR_SYSTEM, 10,
	      desired_window_height - ( 4 * query_font_height ( FONT_SYSTEM) ),
	      "Loading ..." );
  end_text_draw();
  realize_pixmap ( 0, 0 );
  flush_display();
  sleep ( 1 );

  begin_text_draw();
  draw_text ( COLOUR_SYSTEM, 10,
	      desired_window_height - ( 2 * query_font_height ( FONT_SYSTEM) ),
	      "Pondering ..." );
  end_text_draw();
  realize_pixmap ( 0, 0 );
  flush_display();
  sleep ( 1 );

}

void myth_intro ( void ) {
}
