
import java.awt.*;

public class Displayer extends Canvas {

	private Font theCurrentFont = null;
	
	private String theFontName = "";
	private int theFontSize = 0;
	private int theFontStyle = 0;
	private String theOldFontName = "";
	private int theOldFontSize = 0;
	private Color theOldFontForeColour = Color.white;
	
	private FontMetrics fm;
	private boolean mIsBusy = false;
	private Color backColour = Color.blue;
	private Color foreColour = Color.white;
	private int lineRenderCount = 0;
	private Image ibuffer;
	private Graphics igraphics;
	private boolean mIsReady = false; // Determines when the component is ready
	private Label statuslabel = null;

	public synchronized boolean getIsReady() {
		return mIsReady;	
	}
	
	public synchronized void setIsReady(boolean b) {
		mIsReady = b;	
	}

	public Displayer() {
		super();
		// Set the initial font
		try {
			// If an error occurs, it's ok - Java will select the
			// default font and size, the finally block will
			// execute and we will use that instead.
			theCurrentFont = new Font("Dialog", 0, 12);
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		finally {
			theFontName = theCurrentFont.getName();
			theFontSize = theCurrentFont.getSize();
			theFontStyle = 0;
			theOldFontName = theCurrentFont.getName();
			theOldFontSize = theCurrentFont.getSize();
			theOldFontForeColour = Color.white;
		}
	}
	
	public void paint(Graphics g) {
		
		// recheck size each time, rebuild if resized
    	if (ibuffer == null || ibuffer.getWidth(this) != getSize().width ||
			ibuffer.getHeight(this) != getSize().height)
    	{
      		resize();
    	}

    	if (ibuffer == null) return;
		
		// We are ready for painting
		setIsReady(true);
		
		// Copy the buffer onto the screen
		g.drawImage(ibuffer, 0, 1, this);
		
	}

	/** Takes a line of text and renders it to the bottom line
	    of the display */
	public synchronized void renderLine(String s, Label sl) {		
		
		statuslabel = sl;
		
		/* Render it to the bottom line,
		   one character at a time. Where HTML tags are encountered, 
		   switch the style accordingly */
		doRender(s);
		
		/* Scroll the buffer screen up by one line */
		scrollUpOneLine();
		
		/* Check to see if we need to do the "More" prompt
	       by counting how many lines we have rendered since
	       the user made some input */
	    incrementRenderCount();
	    
	    /* If the render count is greater than or equal to the number of rows
	       available on the screen minus one, then a whole screenful has
	       gone past since the last user input, so prompt for More */
	    if (getRenderCount() >= (getRows() - 1)) {
	       
	       	setIsBusy(true);
	    	showMore();
	    	repaint();
	    	
	    	// Setting this class to being busy means the PreOutput class
	    	// won't send any more until it has been cleared.
	 	}
		
	}	
	
	public synchronized void incrementRenderCount() {
		lineRenderCount++;
	}
	
	public synchronized void resetRenderCount() {
		lineRenderCount = 0;
	}
	
	public synchronized int getRenderCount() {
		return lineRenderCount;	
	}
	
	public synchronized int getColumnWidth() {
					
		Dimension dm = this.getSize();
		double width = (double) dm.width;
		
		// Now calculate the width of the widest char in the current
		// font (use M).
		double onechar = (double) fm.charWidth('M');
		
		// Work out how many of these we can fit on a line
		int maxonline = (int) (width / onechar) - 2;
		
		// return it to the caller
		return maxonline;			
	}
	
	public synchronized boolean getStringWidth(String s) {
	
		// Calculates the width of the string passed
		// in and returns true if it is longer than
		// one line.
		
		Dimension dm = this.getSize();
		double width = (double) dm.width;
		
		double swidth = (double) fm.stringWidth(s);
		
		if ( swidth >= (width - getColumnWidth()) )
			return true;
		else
			return false;
	}
	
	public synchronized int getRows() {
			
		updateFonts();
		return (this.getSize().height / fm.getHeight()) - 1;
	}
	
	public synchronized int getLineHeight() {
		return fm.getHeight();
	}
	
	public synchronized void scrollUpOneLine() {
		
		try {
		
			Graphics g = igraphics;
			
			// copy the chunk up:
		    int width = getSize().width;
		    int height = getSize().height;
		    int line_height = getLineHeight();
		    g.copyArea(0, line_height, width, height - line_height, 0, -line_height);
		
		    // erase the old chunk:
		    eraseBottomLine();
		    
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	public synchronized void eraseBottomLine() {
		
		try {
		
			Graphics g = igraphics;
			int width = getSize().width;
		    int height = getSize().height;
		    int line_height = getLineHeight();
		    g.setColor(backColour);
		    g.fillRect(0, height - (line_height), width, (line_height));
		    //repaint();
		    
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	    
	}
	
	public synchronized void clearScreen() {
		
		try {
		
			Graphics g = igraphics;
			int width = getSize().width;
		    int height = getSize().height;
		    g.setColor(backColour);
		    g.fillRect(0, 0, width, height);
		    repaint();
		    
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	    
	}
	
	public synchronized void showMore() {		
	
		try {
			statuslabel.setText("[More...]");
			statuslabel.repaint();	
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	public synchronized void showPressAKey() {		
	
		try {
			statuslabel.setText("[Press any key...]");
			statuslabel.repaint();
		}
		catch (Exception e) {
			e.printStackTrace();
		}
	}
	
	public synchronized boolean getIsBusy() {
		return mIsBusy;
	}
	
	public synchronized void setIsBusy(boolean b) {
		
		// If the flag is being cleared, destroy the
		// more prompt ready for more rendering!
		if (mIsBusy && !b) {
	    	resetRenderCount();
			statuslabel.setText("");
	 	}
		
		mIsBusy = b;
		
	}
	
	/** Actually renders the text onto the line, char by char
	    parsing HTML tags and change the font accordingly. */
	public synchronized void doRender(String s) {
		
		try {
		
			Graphics g = igraphics;
		    int height = getSize().height;
			
			int i = 0;
			String buff = "";
			String curchar = "";
			String curtag = "";
			
			int horpos = 0;
			int closetag = 0;
			
			while (i < s.length()) {
			
				curchar = s.substring(i, i + 1);
				
				if (curchar.equals("<")) {
					
					// We have an HTML tag - process everything
					// upto it if we have anything yet
					
					if (!buff.equals("")) {
						// Render upto it
						updateFonts();
						g.drawString(buff, horpos, height - getLineHeight());
						// Calculate the next horizontal position for next render
						horpos = horpos + fm.stringWidth(buff);
						// Reset the buffer
						buff = "";
					}
					
					// Parse the tag - find the > after the tag start
					closetag = s.indexOf(">", i);
					
					// Get the tag
					curtag = s.substring(i + 1, closetag);
					
					// Now parse it and change the style accordingly
					
					// BOLD ON
					if (curtag.equalsIgnoreCase("B")) {
						theFontStyle = theFontStyle | Font.BOLD;
					}
					
					// BOLD OFF
					if (curtag.equalsIgnoreCase("/B")) {
						theFontStyle = theFontStyle - Font.BOLD;
					}
					
					// ITALIC ON
					if (curtag.equalsIgnoreCase("I")) {
						theFontStyle = theFontStyle | Font.ITALIC;
					}
					
					// ITALIC OFF
					if (curtag.equalsIgnoreCase("/I")) {
						theFontStyle = theFontStyle - Font.ITALIC;
					}
					
					// FONT TAG 
					if (curtag.toLowerCase().startsWith("font")) {
					
						// Store previous font settings
						theOldFontForeColour = foreColour;
						theOldFontName = theFontName;
						theOldFontSize = theFontSize;
					
						int start = 0;
						int end = 0;
					
						// Parse Colours
						int idx = curtag.indexOf("color=");
						if (idx != -1) {
						
							// Get the colour
							start = idx + 6;
							end = curtag.indexOf(" ", idx + 7);
							if (end == -1) end = curtag.length();
							String newcolour = curtag.substring(start, end).trim();
							
							// Change according to what it is
							if (newcolour.equals("white")) {
								foreColour = Color.white;
							}
							
							if (newcolour.equals("black")) {
								foreColour = Color.black;
							}
							
							if (newcolour.equals("blue") || newcolour.equals("navy")) {
								foreColour = Color.blue;
							}
							
							if (newcolour.equals("gray")) {
								foreColour = Color.gray;
							}
							
							if (newcolour.equals("yellow")) {
								foreColour = Color.yellow;
							}
							
							if (newcolour.equals("red")) {
								foreColour = Color.red;
							}
							
							if (newcolour.equals("pink")) {
								foreColour = Color.pink;
							}
							
							if (newcolour.equals("orange")) {
								foreColour = Color.orange;
							}
							
							if (newcolour.equals("magenta")) {
								foreColour = Color.magenta;
							}
							
							if (newcolour.equals("lightgray")) {
								foreColour = Color.lightGray;
							}
							
							if (newcolour.equals("darkgray")) {
								foreColour = Color.darkGray;
							}
							
							if (newcolour.equals("green")) {
								foreColour = Color.green;
							}
							
							if (newcolour.equals("cyan")) {
								foreColour = Color.cyan;
							}
								
							
						}
						
						// Parse Size
						idx = curtag.indexOf("size=");
						
						if (idx != -1) {
							
							// Get the size
							start = idx + 5;
							end = curtag.indexOf(" ", idx + 6);
							if (end == -1) end = curtag.length();
							String newsize = curtag.substring(start, end).trim();
							
							if (newsize.equals("1")) {
								theFontSize = 10;	
							}
							if (newsize.equals("2")) {
								theFontSize = 12;	
							}
							if (newsize.equals("3")) {
								theFontSize = 16;	
							}
							if (newsize.equals("4")) {
								theFontSize = 24;	
							}
							if (newsize.equals("5")) {
								theFontSize = 36;	
							}
							if (newsize.equals("6")) {
								theFontSize = 72;	
							}
						}
						
						// Parse Face
						idx = curtag.indexOf("face=");
						if (idx != -1) {
							
							// Get the face
							start = idx + 5;
							end = curtag.indexOf(" ", idx + 6);
							if (end == -1) end = curtag.length();
							String newface = curtag.substring(start, end).trim();
							
							theFontName = newface;
						}
						
					}
					
					// CLOSE FONT TAG
					if (curtag.equalsIgnoreCase("/FONT")) {
						// Restore previous font settings
						foreColour = theOldFontForeColour;
						theFontName = theOldFontName;
						theFontSize = theOldFontSize;	
					}
					
					// CLEAR SCREEN TAG (SPECIAL CUSTOM TAG!)
					if (curtag.equalsIgnoreCase("CLEARSCREEN")) {
						clearScreen();
						resetRenderCount();	
					}
					
					// WAIT FOR KEYPRESS TAG (SPECIAL CUSTOM TOO)
					if (curtag.equalsIgnoreCase("WAITKEY")) {
						showPressAKey();
						setIsBusy(true);
						repaint();
					}
					
					// BACKGROUND COLOUR TAG (ALSO SPECIAL CUSTOM)
					if (curtag.toLowerCase().startsWith("bgcolor")) {
						
						int start = 0;
						int end = 0;
						
						// Background colour is the second word
						start = curtag.indexOf(" ");
						end = curtag.length();
						String newcolour = curtag.substring(start + 1, end).trim();
						
						
						
						if (newcolour.equals("white")) {
							backColour = Color.white;
						}
						
						if (newcolour.equals("black")) {
							backColour = Color.black;
						}
						
						if (newcolour.equals("blue") || newcolour.equals("navy")) {
							backColour = Color.blue;
						}
						
						if (newcolour.equals("gray")) {
							backColour = Color.gray;
						}
						
						if (newcolour.equals("yellow")) {
							backColour = Color.yellow;
						}
						
						if (newcolour.equals("red")) {
							backColour = Color.red;
						}
						
						if (newcolour.equals("pink")) {
							backColour = Color.pink;
						}
						
						if (newcolour.equals("orange")) {
							backColour = Color.orange;
						}
						
						if (newcolour.equals("magenta")) {
							backColour = Color.magenta;
						}
						
						if (newcolour.equals("lightgray")) {
							backColour = Color.lightGray;
						}
						
						if (newcolour.equals("darkgray")) {
							backColour = Color.darkGray;
						}
						
						if (newcolour.equals("green")) {
							backColour = Color.green;
						}
						
						if (newcolour.equals("cyan")) {
							backColour = Color.cyan;
						}
							
					}
					
					
					// Update the font settings
					updateFonts();
					
					// Manouvre to the next piece of text ready to
					// check that
					i = closetag;
									
				}
				else 
				{
					// Append this char to the buffer
					buff = buff + curchar;
				}
				i++;	
			}
	
			// If there is some buffer left over, flush it		
			if (!buff.equals("")) {
				// Render it
				g.drawString(buff, horpos, height - getLineHeight());
			}
			
		}
		catch (Exception e) {
			e.printStackTrace();
		}		
	}
	
	public synchronized void resize() {
		
		// If we already have a buffer, then this is a resize and
		// not the initial create. If this is the case, we should
		// copy the old buffer to the bottom left corner of the new
		// buffer so all the text doesn't vanish!
		
		try {
		
			if (ibuffer != null) {
				// Store reference to original buffer
				Image oldbuff = ibuffer;
				// Create new one
				ibuffer = createImage(this.getSize().width, this.getSize().height);
				igraphics = ibuffer.getGraphics();
				// Clear the new screen
				clearScreen();
				// Copy old buffer onto new
				igraphics.drawImage(oldbuff, 0, this.getSize().height - oldbuff.getHeight(this), this);
			}
			else
			{
				ibuffer = createImage(this.getSize().width, this.getSize().height);
				igraphics = ibuffer.getGraphics();
				// clear the screen
	      		clearScreen();
			}
	    	
	    	updateFonts();
	    	repaint();
	    }
		catch (Exception e) {
			e.printStackTrace();
		}
		
	}
	
	public synchronized void updateFonts() {

		
		if (igraphics != null) {
			
			try {
				
				// If an error occurs setting the font, just carry
				// on with the default font instead.
				theCurrentFont = new Font(theFontName, theFontStyle, theFontSize);
				
			}
			catch (Exception e) {
				e.printStackTrace();
			}
			finally {
				igraphics.setFont(theCurrentFont);
				igraphics.setColor(foreColour);
				fm = igraphics.getFontMetrics();
			}
		}
	}
	
	
}