package zpplet.ops;

import java.io.IOException;
import java.util.Random;

import zpplet.machine.ZMachine;
import zpplet.misc.ZChars;
import zpplet.misc.ZError;
import zpplet.misc.ZState;

public class ZInstruction
	{
	protected ZMachine zm;
	protected int op; // operator
	protected int[] o; // operands
	protected int noperands; // operand count
	protected int startpc; // for debugging; start of instruction
	protected int nargs; // # arguments on last call
	protected boolean interrupting; // true if running in an interrupt routine
	protected IOP[] ops; // array of operator executors
	
	public ZInstruction(ZMachine zm)
		{
		this.zm = zm;
		o = new int[8];
		interrupting = false;
		initOps();
		}

	// call this puppy to "call" the initial routine
	public void callMain()
		{
		zm.pc = zm.hd.getInitialPC();
		}

	// operand classes
	private final static int OP_LARGE = 0;
	private final static int OP_SMALL = 1;
	private final static int OP_VARIABLE = 2;
	private final static int OP_OMITTED = 3;

	protected interface IOP
		{
		void x()
				throws ZError;
		}

	private int getOperand(int type)
			throws ZError
		{
		switch (type)
			{
			case OP_SMALL:
				return zm.getCodeByte();
			case OP_LARGE:
				return zm.getCodeWord();
			case OP_VARIABLE:
				return zm.getVariable(zm.getCodeByte());
			default:
				throw new ZError("Invalid operand type " + type);
			}
		}

	void fetchVarOperands(int typeinfo, int max)
			throws ZError
		{
		while (max-- > 0)
			{
			int optype = (typeinfo >> (max * 2)) & 3;
			if (optype == OP_OMITTED) break;
			o[noperands++] = getOperand(optype);
			}
		}

	public void decode()
			throws ZError
		{
		// does not get store and branch info, nor inline text
		startpc = zm.pc;
		int opcode = zm.getCodeByte();
		int optype;

		if (opcode == 190) // extended (var)
			{
			op = 0x100 + zm.getCodeByte();
			noperands = 0;
			fetchVarOperands(zm.getCodeByte(), 4);
			}

		else if ((opcode & 0xC0) == 0xC0) // variable form
			{
			op = opcode;
			noperands = 0;
			if ((opcode & 0x20) == 0) // 2OP
				op = opcode & 0x1F;
			if ((op == 236) || (op == 250)) // CALL_VS2 or CALL_VN2
				fetchVarOperands(zm.getCodeWord(), 8);
			else
				fetchVarOperands(zm.getCodeByte(), 4);
			}

		else if (((opcode & 0xC0) == 0x80)) // short form
			{
			optype = (opcode >> 4) & 3;
			if (optype == OP_OMITTED) // 0OP
				{
				op = opcode;
				noperands = 0;
				}
			else
				// 1OP
				{
				op = opcode & 0x8F;
				noperands = 1;
				o[0] = getOperand(optype);
				}
			}

		else
			// long form (2OP)
			{
			op = opcode & 0x1F;
			noperands = 2;

			optype = ((opcode & 0x40) >> 6) + 1;
			o[0] = getOperand(optype);
			optype = ((opcode & 0x20) >> 5) + 1;
			o[1] = getOperand(optype);
			}
		}

	protected void doStore(int val)
		{
		zm.setVariable(zm.getCodeByte(), val);
		}

	protected void doBranch(boolean result)
		{
		int b = zm.getCodeByte();
		boolean branchtype = (b & 0x80) != 0;

		int branchoffset = b & 0x3F; // 6 bits (positive)

		if ((b & 0x40) == 0) // 14 bits
			if ((branchoffset & 0x20) == 0) // positive
				branchoffset = (branchoffset << 8) | zm.getCodeByte();
			else
				// negative
				branchoffset = ~0x3FFF | (branchoffset << 8) | zm.getCodeByte();

		if (result == branchtype) switch (branchoffset)
			{
			case 0:
			case 1:
				doReturn(branchoffset);
				break;
			default:
				zm.pc += branchoffset - 2;
			}
		}

	private java.io.BufferedReader debugtrace;

	private void debugLinkToTrace()
		{
		try
			{
			debugtrace = new java.io.BufferedReader(new java.io.FileReader("trace.txt"));
			}
		catch (Exception e)
			{}
		}

	private String debugFindTraceLine(int addr)
		{
		boolean pass1 = true;
		try
			{
			if (debugtrace == null) debugLinkToTrace();
			while (debugtrace.ready() || pass1)
				{
				if (!debugtrace.ready())
					{
					pass1 = false;
					debugtrace.close();
					debugLinkToTrace(); // from the top
					}
				String line = debugtrace.readLine();
				int colon = line.indexOf(':');
				if (colon < 0) continue;
				int thisaddr = -1;
				try
					{
					thisaddr = Integer.parseInt(line.substring(0, colon).trim(), 16);
					}
				catch (Exception e)
					{
					thisaddr = -1;
					}
				if (thisaddr != addr) continue;
				return line;
				}
			return null;
			}
		catch (IOException ad)
			{
			return null;
			}
		}

	protected void debugShowInstruction(boolean trace)
		{
		StringBuffer s = new StringBuffer();
		for (int i = 0; i < noperands; i++)
			{
			s.append(", ");
			s.append(o[i]);
			}
		s.append(" locals [");
		for (int i = 0; i < zm.l.length; i++)
			{
			if (i > 0) s.append(", ");
			s.append(zm.l[i]);
			}
		s.append("]");
		System.out.println("pc = 0x" + Integer.toHexString(startpc) + ", opcode = " + op + s);
		if (trace)
			{
			String traceline = debugFindTraceLine(startpc);
			if (traceline == null)
				System.out.println("No trace found.");
			else
				System.out.println("\t" + traceline);
			}
		}

	final public void execute()
			throws ZError
		{
		//debugShowInstruction(true);

		try
			{
			ops[op].x();
			}
		catch (IndexOutOfBoundsException e)
			{
			throw new ZError("Invalid operation #" + op);
			}
		catch (NullPointerException e)
			{
			throw new ZError("Invalid operation #" + op);
			}
		}

	final public int interrupt(int addr)
		{
		if (interrupting)
			{
			System.err.println("Interrupt interrupted!");
			return 0;
			}
		// store current operands
		for (int i = 0; i < noperands; i++)
			zm.st.push(new Integer(o[i]));
		zm.st.push(new Integer(noperands));

		// invoke interrupt routine
		noperands = 1;
		o[0] = addr;
		doCall(false);

		// process routine
		try
			{
			interrupting = true;
			do
				{
				decode();
				execute();
				}
			while (interrupting);
			}
		catch (Exception e)
			{
			if (e.getMessage() != null)
				{
				e.printStackTrace();
				System.err.println("INTERRUPT FAILED: " + e.getMessage());
				}
			interrupting = false;
			return 0;
			}

		// pop result
		int result = zm.getVariable(0);

		// restore original operands
		noperands = ((Integer)zm.st.pop()).intValue();
		for (int i = noperands - 1; i >= 0; i--)
			o[i] = ((Integer)zm.st.pop()).intValue();

		return result;
		}

	protected void initOps()
		{
		ops = new IOP[285];

		ops[1] = new OP_JE();
		ops[2] = new OP_JL();
		ops[3] = new OP_JG();
		ops[4] = new OP_DEC_CHK();
		ops[5] = new OP_INC_CHK();
		ops[6] = new OP_JIN();
		ops[7] = new OP_TEST();
		ops[8] = new OP_OR();
		ops[9] = new OP_AND();
		ops[10] = new OP_TEST_ATTR();
		ops[11] = new OP_SET_ATTR();
		ops[12] = new OP_CLEAR_ATTR();
		ops[13] = new OP_STORE();
		ops[14] = new OP_INSERT_OBJ();
		ops[15] = new OP_LOADW();
		ops[16] = new OP_LOADB();
		ops[17] = new OP_GET_PROP();
		ops[18] = new OP_GET_PROP_ADDR();
		ops[19] = new OP_GET_NEXT_PROP();
		ops[20] = new OP_ADD();
		ops[21] = new OP_SUB();
		ops[22] = new OP_MUL();
		ops[23] = new OP_DIV();
		ops[24] = new OP_MOD();

		ops[128] = new OP_JZ();
		ops[129] = new OP_GET_SIBLING();
		ops[130] = new OP_GET_CHILD();
		ops[131] = new OP_GET_PARENT();
		ops[132] = new OP_GET_PROP_LEN();
		ops[133] = new OP_INC();
		ops[134] = new OP_DEC();
		ops[135] = new OP_PRINT_ADDR();

		ops[137] = new OP_REMOVE_OBJ();
		ops[138] = new OP_PRINT_OBJ();
		ops[139] = new OP_RET();
		ops[140] = new OP_JUMP();
		ops[141] = new OP_PRINT_PADDR();
		ops[142] = new OP_LOAD();
		ops[143] = new OP_NOT();

		ops[176] = new OP_RTRUE();
		ops[177] = new OP_RFALSE();
		ops[178] = new OP_PRINT();
		ops[179] = new OP_PRINT_RET();
		ops[180] = new OP_NOP();
		ops[181] = new OP_SAVE();
		ops[182] = new OP_RESTORE();
		ops[183] = new OP_RESTART();
		ops[184] = new OP_RET_POPPED();
		ops[185] = new OP_POP();
		ops[186] = new OP_QUIT();
		ops[187] = new OP_NEW_LINE();

		ops[224] = new OP_CALL();
		ops[225] = new OP_STOREW();
		ops[226] = new OP_STOREB();
		ops[227] = new OP_PUT_PROP();
		ops[228] = new OP_READ();
		ops[229] = new OP_PRINT_CHAR();
		ops[230] = new OP_PRINT_NUM();
		ops[231] = new OP_RANDOM();
		ops[232] = new OP_PUSH();
		ops[233] = new OP_PULL();
		}

	class OP_JE
			implements IOP
		{
		public void x()
			{
			boolean result = false;
			for (int i = 1; i < noperands; i++)
				if (o[0] == o[i])
					{
					result = true;
					break;
					}
			doBranch(result);
			}
		}

	class OP_JL
			implements IOP
		{
		public void x()
			{
			doBranch((short)o[0] < (short)o[1]);
			}
		}

	class OP_JG
			implements IOP
		{
		public void x()
			{
			doBranch((short)o[0] > (short)o[1]);
			}
		}

	class OP_DEC_CHK
			implements IOP
		{
		public void x()
			{
			short x = (short)zm.getVariable(o[0]);
			x--;
			zm.setVariable(o[0], x);
			doBranch(x < (short)o[1]);
			}
		}

	class OP_INC_CHK
			implements IOP
		{
		public void x()
			{
			short x = (short)zm.getVariable(o[0]);
			x++;
			zm.setVariable(o[0], x);
			doBranch(x > (short)o[1]);
			}
		}

	class OP_JIN
			implements IOP
		{
		public void x()
			{
			doBranch(zm.objs.getParent(o[0]) == o[1]);
			}
		}

	class OP_TEST
			implements IOP
		{
		public void x()
			{
			doBranch((o[0] & o[1]) == o[1]);
			}
		}

	class OP_OR
			implements IOP
		{
		public void x()
			{
			doStore(o[0] | o[1]);
			}
		}

	class OP_AND
			implements IOP
		{
		public void x()
			{
			doStore(o[0] & o[1]);
			}
		}

	class OP_TEST_ATTR
			implements IOP
		{
		public void x()
			{
			doBranch(zm.objs.getAttribute(o[0], o[1]));
			}
		}

	class OP_SET_ATTR
			implements IOP
		{
		public void x()
			{
			zm.objs.setAttribute(o[0], o[1], true);
			}
		}

	class OP_CLEAR_ATTR
			implements IOP
		{
		public void x()
			{
			zm.objs.setAttribute(o[0], o[1], false);
			}
		}

	class OP_STORE
			implements IOP
		{
		public void x()
			{
			zm.setVariable(o[0], o[1]);
			}
		}

	class OP_INSERT_OBJ
			implements IOP
		{
		public void x()
				throws ZError
			{
			int obj = o[0];
			detachObj(obj);
			int dest = o[1];
			int dchild = zm.objs.getChild(dest);
			zm.objs.setChild(dest, obj);
			zm.objs.setSibling(obj, dchild);
			zm.objs.setParent(obj, dest);
			}
		}

	class OP_LOADW
			implements IOP
		{
		public void x()
			{
			doStore(zm.getWord(o[0] + o[1] * 2));
			}
		}

	class OP_LOADB
			implements IOP
		{
		public void x()
			{
			doStore(zm.getByte(o[0] + o[1]));
			}
		}

	class OP_GET_PROP
			implements IOP
		{
		public void x()
			{
			int p = zm.objs.getProp(o[0], o[1]);
			doStore(p);
			}
		}

	class OP_GET_PROP_ADDR
			implements IOP
		{
		public void x()
			{
			doStore(zm.objs.getPropAddr(o[0], o[1]));
			}
		}

	class OP_GET_NEXT_PROP
			implements IOP
		{
		public void x()
				throws ZError
			{
			doStore(zm.objs.getNextProp(o[0], o[1]));
			}
		}

	class OP_ADD
			implements IOP
		{
		public void x()
			{
			doStore((short)o[0] + (short)o[1]);
			}
		}

	class OP_SUB
			implements IOP
		{
		public void x()
			{
			doStore((short)o[0] - (short)o[1]);
			}
		}

	class OP_MUL
			implements IOP
		{
		public void x()
			{
			doStore((short)o[0] * (short)o[1]);
			}
		}

	class OP_DIV
			implements IOP
		{
		public void x()
				throws ZError
			{
			if (o[1] == 0) throw new ZError("Division by zero");
			doStore((short)o[0] / (short)o[1]);
			}
		}

	class OP_MOD
			implements IOP
		{
		public void x()
				throws ZError
			{
			if (o[1] == 0) throw new ZError("Mod by zero");
			doStore((short)o[0] % (short)o[1]);
			}
		}

	class OP_JZ
			implements IOP
		{
		public void x()
			{
			doBranch(o[0] == 0);
			}
		}

	class OP_GET_SIBLING
			implements IOP
		{
		public void x()
			{
			int v = zm.objs.getSibling(o[0]);
			doStore(v);
			doBranch(v != 0);
			}
		}

	class OP_GET_CHILD
			implements IOP
		{
		public void x()
			{
			int v = zm.objs.getChild(o[0]);
			doStore(v);
			doBranch(v != 0);
			}
		}

	class OP_GET_PARENT
			implements IOP
		{
		public void x()
			{
			doStore(zm.objs.getParent(o[0]));
			}
		}

	class OP_GET_PROP_LEN
			implements IOP
		{
		public void x()
			{
			doStore(zm.objs.getPropLen(o[0]));
			}
		}

	class OP_INC
			implements IOP
		{
		public void x()
			{
			zm.setVariable(o[0], zm.getVariable(o[0]) + 1);
			}
		}

	class OP_DEC
			implements IOP
		{
		public void x()
			{
			zm.setVariable(o[0], zm.getVariable(o[0]) - 1);
			}
		}

	class OP_PRINT_ADDR
			implements IOP
		{
		public void x()
				throws ZError
			{
			zm.printStringAt(o[0]);
			}
		}

	class OP_REMOVE_OBJ
			implements IOP
		{
		public void x()
				throws ZError
			{
			detachObj(o[0]);
			zm.objs.setParent(o[0], 0);
			zm.objs.setSibling(o[0], 0);
			}
		}

	class OP_PRINT_OBJ
			implements IOP
		{
		public void x()
				throws ZError
			{
			zm.printStringAt(zm.objs.getShortNameAddr(o[0]));
			}
		}

	class OP_RET
			implements IOP
		{
		public void x()
			{
			doReturn(o[0]);
			}
		}

	class OP_JUMP
			implements IOP
		{
		public void x()
			{
			zm.pc += (short)o[0] - 2;
			}
		}

	class OP_PRINT_PADDR
			implements IOP
		{
		public void x()
				throws ZError
			{
			zm.printStringAt(zm.unpackSAddr(o[0]));
			}
		}

	class OP_LOAD
			implements IOP
		{
		public void x()
			{
			doStore(zm.getVariable(o[0]));
			}
		}

	class OP_NOT
			implements IOP
		{
		public void x()
			{
			doStore(~o[0]);
			}
		}

	class OP_RTRUE
			implements IOP
		{
		public void x()
			{
			doReturn(1);
			}
		}

	class OP_RFALSE
			implements IOP
		{
		public void x()
			{
			doReturn(0);
			}
		}

	class OP_PRINT
			implements IOP
		{
		public void x()
				throws ZError
			{
			zm.printStringAt(zm.pc);
			while ((zm.getCodeWord() & 0x8000) == 0)
				{} // skip past the string
			}
		}

	class OP_PRINT_RET
			implements IOP
		{
		public void x()
				throws ZError
			{
			zm.printStringAt(zm.pc);
			zm.printAsciiChar(ZChars.RETURN_CHAR);
			doReturn(1);
			}
		}

	class OP_NOP
			implements IOP
		{
		public void x()
			{}
		}

	class OP_SAVE
			implements IOP
		{
		public void x()
			{
			doSave();
			}
		}

	class OP_RESTORE
			implements IOP
		{
		public void x()
			{
			doRestore();
			}
		}

	class OP_RESTART
			implements IOP
		{
		public void x()
			{
			zm.restart();
			}
		}

	class OP_RET_POPPED
			implements IOP
		{
		public void x()
			{
			doReturn(zm.getVariable(0));
			}
		}

	class OP_POP
			implements IOP
		{
		public void x()
			{
			zm.getVariable(0); // pop the stack and toss the result
			}
		}

	class OP_QUIT
			implements IOP
		{
		public void x()
			{
			zm.printAsciiString("***END OF SESSION***\r");
			zm.s.repaint();
			zm.running = false;
			}
		}

	class OP_NEW_LINE
			implements IOP
		{
		public void x()
			{
			zm.printAsciiChar('\r');
			}
		}

	class OP_CALL
			implements IOP
		{
		public void x()
			{
			doCall(true);
			}
		}

	class OP_STOREW
			implements IOP
		{
		public void x()
			{
			zm.setWord(o[0] + o[1] * 2, o[2]);
			}
		}

	class OP_STOREB
			implements IOP
		{
		public void x()
			{
			zm.setByte(o[0] + o[1], o[2]);
			}
		}

	class OP_PUT_PROP
			implements IOP
		{
		public void x()
				throws ZError
			{
			zm.objs.setProp(o[0], o[1], o[2]);
			}
		}

	class OP_READ
			implements IOP
		{
		public void x()
				throws ZError
			{
			doRead();
			}
		}

	class OP_PRINT_CHAR
			implements IOP
		{
		public void x()
			{
			zm.printAsciiChar(o[0]);
			}
		}

	class OP_PRINT_NUM
			implements IOP
		{
		public void x()
			{
			zm.printAsciiString(Short.toString((short)o[0]));
			}
		}

	class OP_RANDOM
			implements IOP
		{
		public void x()
			{
			short n = (short)o[0];
			if (n == 0)
				{
				zm.random = new Random(); // back to random mode
				doStore(0);
				}
			else if (n < 0)
				{
				zm.random = new Random(-n);
				doStore(0);
				}
			else
				doStore(zm.random.nextInt(n) + 1);
			}
		}

	class OP_PUSH
			implements IOP
		{
		public void x()
			{
			zm.setVariable(0, o[0]);
			}
		}

	class OP_PULL
			implements IOP
		{
		public void x()
			{
			zm.setVariable(o[0], zm.getVariable(0));
			}
		}

	void detachObj(int obj)
			throws ZError
		{
		// removes object from the tree, updating relatives,
		// BUT does not update the object itself
		int parent = zm.objs.getParent(obj);
		if (parent == 0) return;

		int cursor = zm.objs.getChild(parent);
		if (cursor == obj)
			{
			zm.objs.setChild(parent, zm.objs.getSibling(obj));
			return;
			}

		while (zm.objs.getSibling(cursor) != obj)
			{
			cursor = zm.objs.getSibling(cursor);
			if (cursor == 0) throw new ZError("Malformed object tree");
			}
		zm.objs.setSibling(cursor, zm.objs.getSibling(obj));
		}

	/*
	 * Overridable methods follow
	 */

	protected void doSave()
		{
		try
			{
			String fname = zm.s.getFileName("Save Game", true);
			if (fname != null)
				{
				(new ZState(zm)).saveQuetzalFile(fname);
				doBranch(true);
				return;
				}
			}
		catch (IOException e)
			{}
		doBranch(false);
		}

	protected void doRestore()
		{
		try
			{
			String fname = zm.s.getFileName("Load Game", false);
			if (fname != null) (new ZState(zm)).loadQuetzalFile(fname);
			doBranch(true);
			}
		catch (Exception e)
			{
			doBranch(false);
			}
		}

	protected void doRead()
			throws ZError
		{ // SREAD
		int tbuf = o[0];
		int parseaddr = o[1];
		int timeout = noperands < 3 ? 0 : o[2];
		int timeoutroutine = noperands < 4 ? 0 : o[3];

		zm.updateStatusLine();
		zm.curw.flush();
		zm.curw.resetLineCount();
		int tsize = zm.getByte(tbuf) + 1;
		if (tsize < 3) throw new ZError("Text buffer < 3 bytes");
		int count = 0;
		while (tsize-- > 0)
			{
			int ch = zm.getInput(true, timeout, timeoutroutine);
			if (ch == ZChars.ABORT) throw new ZError(null); // terminating
			if (zm.zc.isTerminator(ch)) break;
			if ((ch >= 'A') && (ch <= 'Z')) ch += 'a' - 'A';
			zm.setByte(tbuf + 1 + count++, ch);
			}
		zm.setByte(tbuf + 1 + count, 0);
		zm.zd.tokenize(tbuf + 1, count, parseaddr, true);
		}

	protected void doCall(boolean stores)
		{
		if (o[0] == 0)
			{
			if (stores) doStore(0); // calls to zero return false
			return;
			}
		zm.st.push(new ZStackFrame(zm, stores));
		}

	final protected void doReturn(int retval)
		{
		Object tos;
		do
			tos = zm.st.pop();
		while (!(tos instanceof ZStackFrame));
		((ZStackFrame)tos).restore(zm, retval);
		}

	protected void loadParameters(int count)
		{
		for (int i = 0; i < count; i++)
			{
			int initval = zm.getCodeWord();
			if ((i + 1) < noperands)
				zm.l[i] = zm.zi.o[i + 1];
			else
				zm.l[i] = initval;
			}
		}
	}