#include "ztypes.h"
#include "glk.h"

static winid_t text_win, status_win;
static int machineattr;
static int curr_status_ht, mach_status_ht;
static winid_t curr_win;

static int status_x, status_y;

static int line_pending, *line_len;
static winid_t line_win;

static void reset_status_ht(void);

static void update_width(void) {
  glui32 width;

  if (status_win != 0) {
    glk_window_get_size(status_win, &width, NULL);
    screen_cols = width;
    set_byte(H_SCREEN_COLUMNS, width);
    status_line = realloc(status_line, width + 1);
    if (status_x > width)
      move_cursor(1, status_y + 1);
  }
}

static void cancel_pending_line(void) {
  event_t ev;

  glk_cancel_line_event(line_win, &ev);
  *line_len = ev.val1;
  line_pending = 0;
  set_attribute(-1);
}

void display_char(int ch) {
  if (line_pending && (curr_win == line_win)) {
    cancel_pending_line();
    if (curr_win == status_win) {
      status_x = 1;
      status_y++;
    }
    if (ch == '\n')
      return;
  }
  glk_put_char(ch);
  if (curr_win == status_win) {
    if (ch == '\n') {
      status_x = 1;
      status_y++;
    }
    else {
      glui32 len;
      glk_gestalt_ext(gestalt_CharOutput, ch, &len);
      while (len > 0) {
        status_x++;
        if (status_x > screen_cols) {
          status_x = 1;
          status_y++;
        }
        len--;
      }
    }
  }
}

int input_character(int timeout) {
  event_t ev;

  if (line_pending)
    cancel_pending_line();
  glk_request_char_event(curr_win);
  if (timeout != 0)
    glk_request_timer_events(timeout * 100);
  for (;;) {
    glk_select(&ev);
    if (ev.type == evtype_Arrange)
      update_width();
    else if (ev.type == evtype_Timer) {
      glk_cancel_char_event(curr_win);
      glk_request_timer_events(0);
      return -1;
    }
    else if (ev.type == evtype_CharInput) {
      if (ev.val1 < 0 &&
          (ev.val1 > keycode_Func1 || ev.val1 < keycode_Func12) &&
          (ev.val1 != keycode_Left && ev.val1 != keycode_Right &&
           ev.val1 != keycode_Up && ev.val1 != keycode_Down) &&
          ev.val1 != keycode_Return && ev.val1 != keycode_Delete &&
          ev.val1 != keycode_Tab)
        continue;
      else
        break;
    }
  }
  glk_request_timer_events(0);
  if (mach_status_ht < curr_status_ht)
    reset_status_ht();
  curr_status_ht = 0;
  if (ev.val1 <= keycode_Func1 && ev.val1 >= keycode_Func12)
    return (0x85 + keycode_Func1 - ev.val1);
  else
    switch (ev.val1) {
    case keycode_Left:
      return 0x83;
    case keycode_Right:
      return 0x84;
    case keycode_Up:
      return 0x81;
    case keycode_Down:
      return 0x82;
    case keycode_Return:
      return 13;
    case keycode_Delete:
      return 8;
    case keycode_Tab:
      return 9;
    default:
      return ev.val1;
    }
}

int input_line(int buflen, char *buffer, int timeout, int *readpos,
               int firsttime) {
  event_t ev;

  if (firsttime && line_pending)
    cancel_pending_line();
  if (firsttime || ! line_pending) {
    glk_request_line_event(curr_win, buffer, buflen, *readpos);
    if (timeout != 0)
      glk_request_timer_events(timeout * 100);
  }
  line_pending = 0;
  for (;;) {
    glk_select(&ev);
    if (ev.type == evtype_Arrange)
      update_width();
    else if (ev.type == evtype_Timer) {
      line_win = curr_win;
      line_pending = 1;
      line_len = readpos;
      return -1;
    }
    else if (ev.type == evtype_LineInput)
      break;
  }
  glk_request_timer_events(0);
  *readpos = ev.val1;
  if (mach_status_ht < curr_status_ht)
    reset_status_ht();
  curr_status_ht = 0;
  return '\n';
}

void initialize_screen() {
  glui32 width, height;

  /* Get initial values for the screen size. Later Infocom games
     choke without this. */
  status_win = glk_window_open(0, 0, 0, wintype_TextGrid, 0);
  glk_window_get_size(status_win, &width, &height);
  glk_window_close(status_win, NULL);
  screen_cols = width;
  screen_rows = height;

  text_win = glk_window_open(0, 0, 0, wintype_TextBuffer, 0);
  status_win = 0;
  curr_win = text_win;
  line_pending = 0;
  select_text_window();
  machineattr = NORMAL;
}

void restart_screen() {
  zbyte_t val;

  switch (h_type) {
  case V1:
  case V2:
  case V3:
    val = get_byte(H_CONFIG);
    val |= (0x20 | 0x40); /* screen-splitting, variable-pitch font */
    set_byte(H_CONFIG, val);
    break;
  case V4:
    val = get_byte(H_CONFIG);
    val |= (0x04 | 0x08 | 0x10); /* bold, italic, fixed-width */
    if (glk_gestalt(gestalt_Timer, 0))
      val |= 0x80;              /* timed input */
    set_byte(H_CONFIG, val);
    break;
  case V5:
  case V8:
    val = get_byte(H_CONFIG);
    val |= (0x04 | 0x08 | 0x10); /* bold, italic, fixed-width */
    if (glk_gestalt(gestalt_Timer, 0))
      val |= 0x80;
    set_byte(H_CONFIG, val);
    val = get_byte(H_FLAGS + 1);
    val &= ~(0x08 | 0x20 | 0x80); /* no graphics, no mouse, no sound */
    set_byte(H_FLAGS + 1, val);
    break;
  }
}

void reset_screen() {
  glk_exit();
}

void set_attribute(int attr) {
  int finalattr;

  if (attr == 0)
    machineattr = NORMAL;
  else if (attr > 0)
    machineattr |= attr;

  if (line_pending && curr_win == line_win)
    return;

  if (get_word(H_FLAGS) & FIXED_FONT_FLAG)
    finalattr = machineattr | FIXED_FONT;
  else
    finalattr = machineattr;

  if (finalattr & FIXED_FONT)
    glk_set_style(style_Preformatted);
  else if (finalattr & BOLD)
    glk_set_style(style_Subheader);
  else if (finalattr & EMPHASIS)
    glk_set_style(style_Emphasized);
  else
    glk_set_style(style_Normal);
}

void clear_line() {
  glui32 xpos, ypos;
  int i;

  if (curr_win != status_win)
    fprintf(stderr, "Warning: @clear_line called in text window\n");
  else {
    for (i = 0; i < screen_cols + 1 - status_x; i++)
      glk_put_char(' ');
    glk_window_move_cursor(curr_win, status_x - 1, status_y - 1);
  }
}

void clear_screen() {
  clear_status_window();
  clear_text_window();
}

void clear_status_window() {
  if (status_win != 0) {
    glk_window_clear(status_win);
    reset_status_ht();
    curr_status_ht = 0;
  }
}

void clear_text_window() {
  glk_window_clear(text_win);
}

void reset_status_ht(void) {
  if (status_win != 0)
    glk_window_set_arrangement(glk_window_get_parent(status_win),
                               winmethod_Above | winmethod_Fixed,
                               mach_status_ht, NULL);
}

void create_status_window(int lines) {
  if (status_win == 0) {
    curr_status_ht = mach_status_ht = lines;
    status_win = glk_window_open(text_win, winmethod_Above | winmethod_Fixed,
                                 lines, wintype_TextGrid, 0);
    move_cursor(1, 1);
  }
  else {
    if (lines > curr_status_ht) {
      glk_window_set_arrangement(glk_window_get_parent(status_win),
                                 winmethod_Above | winmethod_Fixed,
                                 lines, NULL);
      curr_status_ht = lines;
    }
    mach_status_ht = lines;
    if (status_y > lines)
      move_cursor(1, 1);
  }
  update_width();
}

void delete_status_window() {
  if (status_win != 0) {
    glk_window_close(status_win, NULL);
    status_win = 0;
    select_text_window();
  }
}

void get_cursor_position(int *row, int *col) {
  if (curr_win != status_win)
    fprintf(stderr, "Warning: @get_cursor called in text window\n");
  else {
    *row = status_y;
    *col = status_x;
  }
}

void move_cursor(int row, int col) {
  if (curr_win != status_win) {
    if (row != 1 || col != 1)
      fprintf(stderr, "Warning: @set_cursor %d %d called in text window\n",
              row, col);
  }
  else {
    glk_window_move_cursor(status_win, col - 1, row - 1);
    status_x = col;
    status_y = row;
  }
}

void scroll_line() {
  display_char('\n');
}

void select_status_window() {
  glk_set_window(status_win);
  curr_win = status_win;
}

void select_text_window() {
  glk_set_window(text_win);
  curr_win = text_win;
}

void xio_bell() {
}
