/*
 * input.c
 *
 * High level input functions
 *
 */

#include "frotz.h"
#include "s5api.h"

extern short save_undo (struct sg *g);

extern zchar stream_read_key (struct sg *g, zword, zword, short);
extern zchar stream_read_input (struct sg *g, short, zchar *, zword, zword, short, short);

extern void tokenise_line (struct sg *g, zword, zword, zword, short);

/*
 * is_terminator
 *
 * Check if the given key is an input terminator.
 *
 */

short is_terminator (struct sg *g, short key)
{

    if (key == ZC_TIME_OUT)
	return TRUE;
    if (key == ZC_RETURN)
	return TRUE;
    switch(key)
		  {
		  case 18: case 16:
          case 19: case 21:
          case 14: case 24:
          case 4:  return TRUE;
		  default:break;
		  }

    if (g->h_terminating_keys != 0)

	if (key >= ZC_ARROW_MIN && key <= ZC_MENU_CLICK) {

	    zword addr = g->h_terminating_keys;
	    zbyte c;

	    do {
		LOW_BYTE (addr, c)
		if (c == 255 || key == translate_from_zscii (g, c))
		    return TRUE;
		addr++;
	    } while (c != 0);

	}

    return FALSE;

}/* is_terminator */

/*
 * z_make_menu, add or remove a menu and branch if successful.
 *
 * 	zargs[0] = number of menu
 *	zargs[1] = table of menu entries or 0 to remove menu
 *
 */

void z_make_menu (struct sg *g)
{

    /* This opcode was only used for the Macintosh version of Journey.
       It controls menus with numbers greater than 2 (menus 0, 1 and 2
       are system menus). Frotz doesn't implement menus yet. */

    branch (g,FALSE);

}/* z_make_menu */

/*
 * read_yes_or_no
 *
 * Ask the user a question; return true if the answer is yes.
 *
 */

short read_yes_or_no (struct sg *g, const char *s)
{
    zchar key;

    print_string (g, s);
    print_string (g, "? (y/n) >");

    key = stream_read_key (g, 0, 0, FALSE);

    if (key == 'y' || key == 'Y') {
	print_string (g, "y\n");
	return TRUE;
    } else {
	print_string (g, "n\n");
	return FALSE;
    }

}/* read_yes_or_no */

/*
 * read_string
 *
 * Read a string from the current input stream.
 *
 */

void read_string (struct sg *g, short max, zchar *buffer)
{
    zchar key;

    buffer[0] = 0;

    do {

	key = stream_read_input (g, max, buffer, 0, 0, FALSE, FALSE);

    } while (key != ZC_RETURN);

}/* read_string */

/*
 * read_number
 *
 * Ask the user to type in a number and return it.
 *
 */

short read_number (struct sg *g)
{
    zchar buffer[6];
    short value = 0;
    short i;

    read_string (g, 5, buffer);

    for (i = 0; buffer[i] != 0; i++)
	if (buffer[i] >= '0' && buffer[i] <= '9')
	    value = 10 * value + buffer[i] - '0';

    return value;

}/* read_number */

/*
 * z_read, read a line of input and (in V5+) store the terminating key.
 *
 *	zargs[0] = address of text buffer
 *	zargs[1] = address of token buffer
 *	zargs[2] = timeout in tenths of a second (optional)
 *	zargs[3] = packed address of routine to be called on timeout
 *
 */

void z_read (struct sg *g)
{
    zchar buffer[INPUT_BUFFER_SIZE];
    zword addr;
    zchar key;
    zbyte max, size;
    zbyte c;
    short i;

    /* Supply default arguments */

    if (g->zargc < 3)
		g->zargs[2] = 0;

    /* Get maximum input size */

    addr = g->zargs[0];

    LOW_BYTE (addr, max)

    if (g->h_version <= V4)
	max--;

    if (max >= INPUT_BUFFER_SIZE)
	max = INPUT_BUFFER_SIZE - 1;

    /* Get initial input size */

    if (g->h_version >= V5) {
	addr++;
	LOW_BYTE (addr, size)
    } else size = 0;

    /* Copy initial input to local buffer */

    for (i = 0; i < size; i++) {
	addr++;
	LOW_BYTE (addr, c)
	buffer[i] = (unsigned char)translate_from_zscii (g,c);
    }

    buffer[i] = 0;

    /* Draw status line for V1 to V3 games */

    if (g->h_version <= V3)
	z_show_status (g);

    /* Read input from current input stream */

    key = stream_read_input (g,
	max, buffer,		/* buffer and size */
	g->zargs[2],		/* timeout value   */
	g->zargs[3],		/* timeout routine */
	TRUE,	        	/* enable hot keys */
	g->h_version == V6);	/* no script in V6 */

    if (key == ZC_BAD)
	return;

    /* Perform save_undo for V1 to V4 games */

    if (g->h_version <= V4)
	save_undo (g);

    /* Copy local buffer back to dynamic memory */

    for (i = 0; buffer[i] != 0; i++) {

	if (key == ZC_RETURN) {

	    if (buffer[i] >= 'A' && buffer[i] <= 'Z')
		buffer[i] += 'a' - 'A';
	    if (buffer[i] >= 0xc0 && buffer[i] <= 0xde && buffer[i] != 0xd7)
		buffer[i] += 0x20;

	}

	storeb (g, (zword) (g->zargs[0] + ((g->h_version <= V4) ? 1 : 2) + i), (unsigned char)translate_to_zscii (g, buffer[i]));

    }

    /* Add null character (V1-V4) or write input length into 2nd byte */

    if (g->h_version <= V4)
	storeb (g,(zword) (g->zargs[0] + 1 + i), 0);
    else
	storeb (g,(zword) (g->zargs[0] + 1), (unsigned char)i);

    /* Tokenise line if a token buffer is present */

    if (key == ZC_RETURN && g->zargs[1] != 0)
	tokenise_line (g,g->zargs[0], g->zargs[1], 0, FALSE);

    /* Store key */

    if (g->h_version >= V5)
	store (g, translate_to_zscii (g, key));

}/* z_read */

/*
 * z_read_char, read and store a key.
 *
 *	zargs[0] = input device (must be 1)
 *	zargs[1] = timeout in tenths of a second (optional)
 *	zargs[2] = packed address of routine to be called on timeout
 *
 */

void z_read_char (struct sg *g)
{
    zchar key;

    /* Supply default arguments */

    if (g->zargc < 2)
	g->zargs[1] = 0;

    /* Read input from the current input stream */

    key = stream_read_key (g,
	g->zargs[1],	/* timeout value   */
	g->zargs[2],	/* timeout routine */
	TRUE);  	/* enable hot keys */

    if (key == ZC_BAD)
	return;

    /* Store key */

    store (g, translate_to_zscii (g, key));

}/* z_read_char */

/*
 * z_read_mouse, write the current mouse status into a table.
 *
 *	zargs[0] = address of table
 *
 */

void z_read_mouse (struct sg *g)
{

    storew (g, (zword) (g->zargs[0] + 0), g->hx_mouse_y);
    storew (g, (zword) (g->zargs[0] + 2), g->hx_mouse_x);
    storew (g, (zword) (g->zargs[0] + 4), 1);		/* mouse button bits */
    storew (g, (zword) (g->zargs[0] + 6), 0);		/* menu selection */

}/* z_read_mouse */
