! RAP - Reactive Agent Planner for Inform
! Planbase v 1.1, March 2000
!
! Module maintainer:
! Nate Cull (culln@chchpoly.ac.nz)
!
! Revision history:
! 1.0 - RAP for TADS 1.0, first gmd.de release (rap10.zip), 1998
! 1.1 - Inform port, code cleanup and documentation, March 2000 
!
! General approach is based on a modification and simplification 
! of the HAP reactive planning system as described in publically 
! available white papers for the CMU OZ project 
!
! Notes on my symbol naming convention:
! All globals are prefixed with Rap
! All constants are underscore delimited words, all uppercase
! All variables and functions are underscore delimited words, all lowercase
! All object classes are mixed case delimited words, leading uppercase
! All object instances are mixed case delimited words, leading lowercase


! This module defines a standard library of conditions and actions
! that should be useful for most IF games.  Some of the actions may
! be more applicable to portable animal NPCs, such as Widget the puppy,
! in the first demo world.


! rapInRoom - a basic, general-purpose method for moving from room
! to room in a map. This is bread-and-butter NPC stuff and RAP's 
! original 'killer app'. Out of the box, RAP should be able to traverse
! a map (given a few hints - see the RapRoom class later in this file),
! and the OO design of RAP means that you can easily add hints for
! solving obstacles, like opening doors and getting keys.  The demo
! world doesn't have any lockable doors, but that will come soon.


RapCondition rapInRoom "rapInRoom"
with
	istrue	[ actor object room;

	if (rap_trace_level > 1) {
		print "^rapInRoom istrue: starting^";
		print "^rapInRoom istrue: actor=", (name) actor;
		print ", object=", (name) object;
		print ", room=", (name) room;

	}
! object is in room if it's contained in the object tree

		if (object in room) {

			if (rap_trace_level > 1)
				print "^rapInRoom istrue: returning true^";

			rtrue;
			
		} else {
		
			if (rap_trace_level > 1)
				print "^rapInRoom istrue: returning false^";

		
			rfalse;
		}	
	],
	
	
	get_plans	[ actor object room plan step;


		if (rap_trace_level > 1)
			print "^rapInRoom get_plans: 
				actor=", (name) actor,
				", object=", (name) object,
				", room=", (name) room,
				", plan=", plan,
				", step=", step, "^";

! The plan for making param1 be in room param2 is different depending
! on whether param1 is the actor or an object
!
! For now we're just implementing the case where param1 is actor
! ie, actor wants to be in room param2 and can move there with
! standard move actions


		if (actor == object) {
			
			self.get_move_to(actor, room, plan, step); 
		
		} else {
		
			rap_set_num_plans(0);
			rfalse;
		}
	
	], 


! This method does the standard room traversal lookup.  Currently
! in a bit of flux, being melded into a more OO way of doing things.
! Might need to reduce the number of chained method calls it's doing.
! But it will work fine for a demo situation. 

	get_move_to	[ actor room plan step;


! Small tweak here for portable actors (like Widget the puppy)
! if currently held, do something special - beg to be let down

		if (actor in player) {
			switch (plan)
			{
			
			0:	rap_set_num_plans(1);
			
			1:	switch (step) {
			
				0:	rap_set_num_steps(1);
				
				1:	rap_set_step(	RAP_DO,
							rapBegDown,
							false,
							false	);
			
				}
			
			}
			
			rtrue; 
		} 


! Treat player as a special case room - beg to be picked up

		if (room == player)	switch (plan) {
		
				0:	rap_set_num_plans(1);
				
				1:	switch (step) {
				
					0:	rap_set_num_steps(2);
					
					1:	rap_set_step(	RAP_BE,
								rapInScope,
								player,
								false	);

					2:	rap_set_step(	RAP_DO,
								rapBegUp,
								player,
								false	);
								
					}
		
				} ! plans for player

		
		else

! For normal rooms, get it from a method

			room.rap_get_move_to( actor, plan, step );	

	
	],  ! get_move_to()
	
;


! rapGoTo - a simple 'move to room' action. Does some basic notifications
! to the player, and doesn't care about using exits - it assumes that
! exit sanity checking has already been done by the planbase.  In most
! cases that should be correct. Perhaps it might be useful later on,
! once we add the knowledge engine (or rather, ignorance engine - at the
! moment RAP knows everything about the world instantly, we'd like to filter
! that down and make it deal with incomplete knowledge) to have 'GoTo' 
! actions that fail and force RAP to adjust its knowledge of what exits
! are currently working.  But that's an exercise for the reader at this
! point.

RapAction rapGoTo "rapGoTo"
with
	execute	[ actor room dummy
			
			oldroom
	;


		if (TestScope (player, actor))
	
			print "^",
			 (the) actor,
			" heads off to ",
			(the) room,
			".^";

		oldroom = room;
			
		move actor to room;


		if (TestScope (player, actor))
	
			print "^",
			 (the) actor,
			" arrives from ",
			(the) oldroom,
			".^";

	
		rtrue;
			
	],
	
	
;


! rapInScope - a fairly useful generic kind of condition, check to see
! if an object is nearby and if not go and hunt for it. Lots of different
! NPCs would tend to use this for all sorts of things, it would be the
! cornerstone of various 'find the widget' quests.


RapCondition rapInScope "rapInScope"
with
	istrue	[ actor object dummy;

! Using a very simplistic scope definition here - if the object's
! in the actor's immediate locale, then it's in scope. This is NOT
! the full scope definition we'd want to use in a real library,
! but it should work for demo purposes
	
		if (object in (parent(actor) ) )
		
			rtrue;
		else
		
			rfalse;
	
	],
	
	get_plans	[ actor object dummy plan step;

! How do we make an object get in scope?
! The easiest way is to go to its location.
!
! Note: We'd want to have more alternatives in a full library.
! Depending on the type of object, we might want to use all sorts
! of different strategies to obtain it.  IE, beg / barter / steal it,
! if it's held by another actor; buy it from a vending machine;
! transform another object into it with a magic spell.  A lot of these
! options will be game-specific.
!
! For now, we simply go to its location.

! Check that it does exist

		if (parent(object) == false) {
			
			rap_set_num_plans(0);
			rfalse;
			
		}
			
		switch(plan) {
			
		0:	rap_set_num_plans(1);
			
		1:	switch (step) {
								
			0:	rap_set_num_steps(1);
			
			1:	rap_set_step (	RAP_BE,
						rapInRoom,
						actor,
						parent(object)	);

			}
		} 		
	
		rtrue;
	],
	
;

! rapTake - a simple Take action, without any kind of sanity checking
! Not used in this example, but could be if we needed it.

RapAction rapTake "rapTake"
with	
	execute	[ actor object dummy;
	
		move object to actor;

		if (TestScope (player, actor))
			print "^", (The) actor, " picks up ", (the) " object.";
		
		rtrue;
	
	],
;


! rapBegUp - what a doggy wants to do when it wants to get 'into' the
! player.  In this case, probably because the player is holding something
! (ie, the ball) that Widget wants to be in scope.  It would also be
! possible to have a jumpUp action or similar that forces Widget to
! jump onto the player, but that would get annoying.


RapAction rapBegUp "rapBegUp"
with	
	execute	[ actor begee dummy;

		if (TestScope (player, actor))
			print "^", (The) actor, " whines and begs pleadingly.^";
		
		rtrue;
	
	],
;


! rapBegDown - what a doggy wants to do when it's being held by the player
! and wants to go to a room.  It's degrading to have to beg, but how
! else is the player going to know you want DOWN, now?


RapAction rapBegDown "rapBegDown"
with	
	execute	[ actor begee dummy;

		if (TestScope (player, actor))
			print "^", (The) actor, " whines and struggles 
					to get free.^";
		
		rtrue;
	
	],
;


! rapDoggyFun - a simple top-level desire for Widget the puppy
! What's a dog's idea of fun? In this case, it's simply finding
! and playing with a ball.  We can expand this later on to make
! for more interesting behaviour. A Maslow's Hierarchy of Needs
! could be done quite easily as a series of prioritised conditions 
! (beFed, beAmused, etc) with state variables tracking how hungry,
! bored, etc Widget is. As each condition became unfulfilled, Widget
! would stop whatever else he was doing and go on a quest to solve
! the more important need.  That's for a later version, but all it
! needs is a planbase modification, the core engine can handle it
! already.


RapCondition rapDoggyFun "rapDoggyFun"
with
	istrue	[ actor param1 param2;

! A doggy can't ever have too much fun. 
! If we wanted a dog to get bored with playing with the ball,
! we'd put a counter in here and return true when it was bored/satisfied,
! so it would try something else.
	
		rfalse;
	
	],

	get_plans	[ actor param1 param2 plan step;

		if (rap_trace_level > 1)
			print "^rapDoggyFun plans:
			actor=", (name) actor,
			", param1=", (name) param1,
			", param2=", (name) param2,
			", plan=", plan,
			", step=", step,
			"^";
	
		switch (plan) {
		
		0:	rap_set_num_plans(1);
		
		1:	switch (step) {
			
			0:	rap_set_num_steps(2);
			
			1:	rap_set_step(	RAP_BE,
						rapInScope,
						ball,
						false	);

			2:	rap_set_step(	RAP_DO,
						rapPlayWith,
						ball,
						false	);
		
			} ! steps
		
		} ! plans
		
		rtrue;
	
	],	
	
;

! rapPlayWith - a simple action for Widget the puppy and other animals
! makes him play with a given toy


RapAction rapPlayWith "rapPlayWith"
with	
	execute	[ actor toy dummy;

		if (TestScope (player, actor))
			print "^", (The) actor, " plays with ", (the) toy, ".^";
		
		rtrue;
	
	],
;

! RapRoom class encapsulates a simple method for integrating RAP hints
! into your map.  If you're using the standard rapInRoom condition,
! you'll need to derive all your NPC-accessible rooms from this.
!
! To use it, set rap_from_num to the number of entrances to the room,
! and the rap_from_x properties to each entrance.
!
! If you have special rooms with weird exits that require different
! behaviour, you'll need to override rap_get_move_to for that room.
!

Class RapRoom

with	rap_get_move_to	[ actor plan step
	
			;

		switch (plan) {
			
		0:	rap_set_num_plans( self.rap_from_num);
		
		default: switch (step) {
		
			0:	rap_set_num_steps(2);
		
			1:	switch (plan) {
			
				1:	rap_set_step(	RAP_BE,
							rapInRoom,
							actor,
							self.rap_from_1	);
							
				2:	rap_set_step(	RAP_BE,
							rapInRoom,
							actor,
							self.rap_from_2	);
							
				3:	rap_set_step(	RAP_BE,
							rapInRoom,
							actor,
							self.rap_from_3	);
							
				4:	rap_set_step(	RAP_BE,
							rapInRoom,
							actor,
							self.rap_from_4	);
							
				5:	rap_set_step(	RAP_BE,
							rapInRoom,
							actor,
							self.rap_from_5	);
							
				6:	rap_set_step(	RAP_BE,
							rapInRoom,
							actor,
							self.rap_from_6	);
							
				7:	rap_set_step(	RAP_BE,
							rapInRoom,
							actor,
							self.rap_from_7	);
							
				8:	rap_set_step(	RAP_BE,
							rapInRoom,
							actor,
							self.rap_from_8	);
							
				9:	rap_set_step(	RAP_BE,
							rapInRoom,
							actor,
							self.rap_from_9	);
							
				10:	rap_set_step(	RAP_BE,
							rapInRoom,
							actor,
							self.rap_from_10);
							

				} ! plan
		
			2:	rap_set_step(	RAP_DO,
						rapGoTo,
						self,
						false	);
		
			} ! steps
		
		} ! plan
		
		rtrue;		
	
	],
	
	rap_from_num,
	rap_from_1,
	rap_from_2,
	rap_from_3,
	rap_from_4,
	rap_from_5,
	rap_from_6,
	rap_from_7,
	rap_from_8,
	rap_from_9,
	rap_from_10

;