/*
 * Polyadventure
 *
 * A remake of the various versions of the classic Adventure game by Don
 * Woods and Willie Crowther, based on their sources.  Currently, the 350,
 * 550, and 551-point versions are implemented.  See the file "ccr-help.t"
 * for more information.
 *
 * Please document all changes in the history so we know who did what.
 *
 * This source code is copylefted under the terms of the GNU Public
 * License.  Essentially, this means that you are free to do whatever
 * you wish with this source code, provided you do not charge any
 * money for it or for any derivative works.
 *
 *
 * Contributors (see history.t for current e-mail addresses)
 *
 *      dmb     In real life:   David M. Baggett
 *
 *      djp     In real life:   David J. Picton
 *
 *      bjs     In real life:   Bennett J. Standeven
 *
 * Modification History (this file)
 *
 * CCR
 * ===
 *
 *  1-Jan-93    dmb     rec.arts.int-fiction BETA release (source only)
 *                      For beta testing only -- not for general
 *                      distribution.
 * 20-Apr-93    dmb     Incorporated Mike Roberts' changes to make
 *                      this a lot faster.  Uses the new intersect
 *                      built-in, so CCR must now be built with TADS
 *                      >= 2.0.13.
 *
 * 29-Apr-96    djp     Supplied some suggested enhancements/bugfixes
 *                      to dmb.  These were incorporated in Version 2.0G,
 *                      (now available in compiled form only under
 *                      ftp://ftp.gmd.de/if-archive/games/adventions)
 *
 * AD551
 * =====
 *
 * 16-Oct-98    djp     Moved the code for all NPC's into this file.
 *
 * 14-Apr-99    djp     Initial release of Adventure 551 (1.01)
 *
 * 23-Apr-99    djp     New release (1.10) of Adventure 551.
 *
 * 30-Apr-99    djp     New release (1.11) of Adventure 551.
 *                      Changes since version 1.10:
 *                      *  Changed some responses to the 'rub' verb.
 *                      *  Minor changes to DragonCorpse responses.
 *                      *  Changed Troll.ispaid to indicate location of
 *                         payment - see ccr-room.t for the reason.
 *
 * 17-May-99    djp     New release (1.20)
 *              djp     Changes since version 1.11:
 *                      *  Corrected the code for attacking the troll
 *                         with a non-weapon (which gave no output in
 *                         some cases)
 *                      *  Suppressed messages about dwarves and dwarf
 *                         attacks in dark rooms (however, the pirate is
 *                         now described as carrying a lamp so he can be
 *                         seen, and can attack, in dark rooms.)
 *                      *  Changed messages about the pirate so that he
 *                         doesn't keep saying he'll hide the chest when he's
 *                         already done so.
 *
 * 20-May-99    djp     New release (1.21)
 *
 * 15-Jul-99    djp     New release (2.00)
 *                      Changes in this version:
 *                      * Added missing climb, climbup and climbdown methods
 *                      * to exitlist.
 *                      * Reworked dwarf attacking code for new Rings of
 *                        Protection scheme.
 *                      * Changed the code for ATTACK DWARF WITH weapon so
 *                        that if only one dwarf is present, he won't also
 *                        retaliate at the end of the turn.
 *                      * Added a 'catch-all' (dobjGen/iobjGen to
 *                        dwarfKnives to prevent commands like RUB KNIFE.
 *
 * 3-Aug-99     djp     Bugfix release - Version 2.01
 *
 *              djp     Bugfix Version 2.02 (not released)
 *                      Changes for this version
 *                      * Installed PirateMessage in Dead_End_14 when
 *                        pirate spotted.
 *                      * Split ccr-item.t and ccr-npc.t into two files to
 *                        facilitate integration of ad551 updates into
 *                        Bennett Standeven's 550-point game.
 *
 * 9-Nov-99     djp     New release - Version 2.10
 *                      * Incorporated Bennet Standeven's Chaser class for
 *                        the Wumpus.
 *                      * Incorporated Bennet Standeven's code for the
 *                        Dwarves.doAttackWith method.
 *                      * Added default giving methods for the feedable
 *                        class.
 *
 * POLYADV
 * =======
 *
 * 24-Aug-99    bjs     Pre-release version 0.00
 *              bjs     Split this file into one part for each version,
 *                      generalized Wumpus code, corrected bridge-crossing
 *                      code.
 *
 *          djp+bjs     Incorporated ad551 mods up to 2.20
 *
 * 3-Mar-00     djp     Initial beta release - Version 1.00
 *              djp     Changes since ad551 v2.20:
 *                      * Mapped 'feed x to y' to 'feed y with x'.
 *                      * Defined a doCount method for the dwarves.
 *                      * Test for new weapon class where appropriate.
 *                      * Various bugfixes to attacking/throwing/giving
 *                        to the Bear.  A bug which allowed
 *                        weapons to be thrown at the tame bear has been
 *                        fixed.
 *                      * Reworked attacking code to use the new
 *                        weapon class where appropriate.
 *                      * Reworked the code for 'throw x to enemy'.  When
 *                        weapons get trapped if they land close to the
 *                        enemy (e.g. the Bear), 'throw weapon TO enemy'
 *                        is the same as 'throw weapon AT enemy' but 'throw
 *                        other at enemy' resolves to 'give other to enemy'.
 *                        In other cases the latter choice is made for all
 *                        items. (exception: the Djinn)
 *                      * Ensured that giving methods are inherited from
 *                        'feedable' rather than 'Actor' by adjusting the
 *                        order of classes (feedable comes before Actor etc)
 *                      * Defined a summon method for all Chasers.  Added
 *                        methods for the game-testing 'summon' and
 *                        'banish' verbs.
 *                      * Tidied up NPC interaction code to follow the
 *                        order:
 *                        Attacking
 *                        Kicking
 *                        Throwing
 *                        Giving
 *                        Feeding
 *                        Other
 *
 * 4-Apr-00     djp     Version 1.01: bugfix release
 *                      Changes in this version:
 *                      * The troll now stays after rejecting the eggs.
 *
 * 18-Sep-00    djp     Version 2.00: New version with 701-point game
 *                      Changes in this version:
 *                      * Code for attacks on dwarves extended for
 *                        701-point game (most weapons now use the 550-point
 *                        code, except for the singing sword.)
 *                      * Code for attacks by dwarves changed to support new
 *                        protection level (search and destroy) in 701-point
 *                        game.
 *              
 * 20-Dec-00    djp     Version 2.02: bugfix release
 *                      Changes in this version:
 *                      * Added some default verification methods for the rub
 *                        verb
 *                      * Put the troll code in the standard order and with
 *                        the appropriate headings.
 *
 * 8-Jun-01     djp     Version 2.05: bugfix release
 *                      Changes in this version
 *                      * Changes to class feedable and associated code to
 *                        handle liquids.      
 *
 * 22-Nov-01    djp     Version 2.08: bugfix release
 *                      * LEAVE BEAR or LEAVE BEAR HERE implemented.
 *
 * 13-Jan-03    djp     Version 2.09 rev C: bugfixes and code tidy-up
 *                      * Implemented door destinations in NPC travel code.
 *                      * Added a check to detect when an NPC is stuck with
 *                        no valid exits, and issue a warning message.
 *                      * Added more NPC exit properties.
 *                      * Enhancements to the NPC scatter method.
 *
 * 12-Aug-03    bjs     Version 2.11: added 580-point mode.
 *
 * 6-May-04     djp     Version 2.12 Rev C 
 *                      * Fixed melocs evaluation in dwarf methods to 
 *                        change global.currentActor afterwards, not before.
 *
 * 23-Jul-04    djp     Version 3.00.  Addition of a new game mode (701+).
 *
 * 12-Apr-05    djp     Version 3.10: Bugfixes and enhancements
 *                      * Changes to accomodate new throwHitDest method
 * 7-May-05     djp     Version 3.10 Rev. A
 *                      * New property 'chasehold' added to Chaser class.
 *                        The chase level is 'held' for the specified
 *                        number of turns.
 *
 * 30-Mar-09    djp     Version 3.20 Rev. D
 *                      * Changed the class list for DwarfKnives to cause it
 *                        to be recorded as a floatingItem.  (Items must be in
 *                        class 'thing' for this to happen).
 * 
 * 21-Nov-10    djp     Version 3.21
 *                      * If the game version is such that the pirate locks 
 *                        his treasure in the chest, he won't steal the
 *                        pendants or mithril ring if these may be needed to
 *                        reach the keys.
 */

/*
 * This file handles non-player character movement and characteristics.
 *
 * Be sure to update exitlist and/or npclist if you add any new travel
 * verbs to CCR_Room.
 */

/* Note on 'attacking' methods.
 * (weapon = sword/axe, food = food item)
 * attack enemy:   If player has axe, attack enemy with axe
 *                 If player has sword, attack enemy with sword
 *                 otherwise issue 'bare hands' message or similar.
 *                 Exceptions: snake, bees and Djinn allow no
 *                 attacking at all, dragon and slime issue 'bare hands'
 *                 message regardless of weapons held.
 *
 * attack enemy with weapon
 *                 Often interpreted in the same way as "throw weapon".
 *                 The doAttackWith method is used, but a flag
 *                 (global.saidthrow) indicates whether the player actually
 *                 asked for the weapon to be thrown, so an explanatory
 *                 message e.g. "(throwing the axe)" is given when
 *                 appropriate.  The global.saidthrow property must be cleared
 *                 by the doAttackWith method.  Exceptions: snake, bees,
 *                 Djinn allow no attacking, dwarves, blob and slime allow
 *                 hand-to-hand combat to be attempted (except in a 350-point
 *                 game).
 *
 *
 * attack enemy with hands:
 *                 This should have a similar effect to simply attacking
 *                 the enemy.  (In most cases, a dismissive message).
 *
 * attack enemy with other:
 *                 Dismissed with a message like "that wouldn't be very
 *                 effective".
 *
 * wave weapon     If there is only one enemy present, it is assumed to be
 *                 the target.  Otherwise, the target is prompted for.  The
 *                 code is all in ccr-verb.t.
 *
 * wave weapon at enemy
 *                 Indicates an attempt at hand-to-hand combat with the
 *                 enemy.  The coding is all in ccr-verb.t.  Where attacking
 *                 with the weapon implies hand-to-hand combat, the attacking
 *                 code is used.  Where attacking implies throwing, another
 *                 method is used (e.g. attacking with bare hands.)
 *
 * throw weapon
 *                 If there is only one enemy present, it is assumed to be
 *                 the target.  Otherwise, the target is prompted for.
 *                 Coding in ccr-adv.t has now been changed so that the
 *                 presence of the weapon is checked before the prompting.
 *
 * throw weapon at/to enemy:
 *                 Causes enemy to be attacked.  The effect is different
 *                 from "attack enemy with weapon" only for dwarves (except
 *                 in 350-point mode), Djinn, Blob and Slime.  Weapons bounce
 *                 off the Djinn forcefield (and are said to be dodged by
 *                 the player, so they are dropped).  Weapons cut
 *                 harmlessly through the Blob and are dropped.  The
 *                 gleaming sword is fragile and will shatter in most cases.
 *
 * throw glass vial at/to enemy:
 *                 Transfers control to the glass vial (often effective, but
 *                 the vial can be used only once and is needed to remove the
 *                 slime).
 *
 * throw other at/to enemy:
 *                 Usually treated in the same way as "give other at/to
 *                 enemy".  Objects bounce off the Djinn forcefield or
 *                 the Blob and are caught again by the player.  In the
 *                 case of the Goblins, objects are said to 'miss' and
 *                 are dropped.
 */


initNPC: function
{
    local   o;


    //
    // Construct list of NPC exits for each room
    //
    o := firstobj(room);
    while (o <> nil) {
        if (not o.noNPCs) {  
            //
            // Add this room to the global list of rooms
            // the NPC's can be in.
            //
            global.NPCrooms := global.NPCrooms + o;
            do_exitlist(o);
            do_npclist(o);
        }
        else if (global.debug) {
            //
            // Debugging info:
            //
            "\b\"<< o.sdesc >>\" is off limits to NPC's.";
        }

        o := nextobj(o, room);
    }
}

/*
 * Add standard exits to the list of exits that NPC's should check
 * when they're wandering about randomly.
 */
do_exitlist: function(o)
{
    local   exitlist, i, j, l := [], ex, gotit;
    local   tot1, tot2;

    //
    // List of all exit property names that NPC's will consider.
    // Note that magic words are left out because NPC's don't
    // know them (and don't have magic slippers).  Other words are left
    // out because they don't work anywhere, or only apply to
    // outside areas.
    //
    exitlist := [  // DJP - added commas to avoid compiler warning
        &north, &south, &east, &west,
        &ne, &nw, &se, &sw,
        &up, &down, &in, &out,
        &climb, &climbup, &climbdown,

        &jump, &upstream, &downstream, &forwards, &outdoors,
        &left, &right, &center, &cross, &over, &across, &road, &forest,
        &valley, &stairs, &building, &gully, &stream, &rock, &bed,
        &crawl, &cobble, &tosurface, &dark, &passage, &low, &canyon,
        &awkward, &giant, &view, &pit, &crack, &steps, &dome, &hall,
        &barren, &debris, &hole, &wall, &broken, &y2, &floor, &toroom,
        &slit, &slab, &depression, &entrance, &secret, &cave,
        &bedquilt, &oriental, &cavern, &shell, &reservoir,
        &fork, &chimney, &slide, &pool, &lair,
        &ledge, &ice, &bridge, &altar, &grotto
    ];  // Most 550-point specific words are left out, because
        // the correspondent rooms are off limits to NPCs.
    tot1 := length(exitlist);
    // "\no=",o.sdesc,".\n";

    for (i := 1; i <= tot1; i++) {
        //
        // If this exit property is a simple
        // object (prop 2), NPC's can use it, so
        // add it to the room's NPC exit list.
        //
        // Make sure we don't add the same
        // destination room twice.  Just because
        // there are multiple travel verbs from
        // one place to another doesn't mean the
        // destination should be more likely.
        //
        // DJP - now fixed (the code compared exit properties
        // instead of actual destinations.)
        //
        if (proptype(o, exitlist[i]) = 2) {
            ex := o.(exitlist[i]);
            // "\nex=";ex.sdesc;"\n";
            if (not ex.noNPCs) {
                //
                // Search current exitlist for
                // this exit's destination.
                //
                if(length(l) = 0)
                    gotit := nil;
                else
                    gotit := (find(l,ex) <> nil);
                if (not gotit) {
                    // "\nnot gotit\n";
                    l += ex;
                    o.NPCexits := o.NPCexits + exitlist[i];
                }
                // else "\ngotit\n";
            }
        }
    }
}

/*
 * Add NPC special exits to the list of exits that NPC's should check
 * when they're wandering about randomly.
 */
do_npclist: function(o)
{
    local   npclist, i, tot;

    //
    // NPC exits.  These get considered if even if they're methods.
    // The only way they won't be added to the list of exits to
    // try is if they're = nil and not methods.
    //
    npclist := [ // DJP - added commas to avoid compiler warning
        &NPCexit1, &NPCexit2, &NPCexit3, &NPCexit4,
        &NPCexit5, &NPCexit6, &NPCexit7, &NPCexit8, 
        &NPCexit9, &NPCexit10, &NPCexit11, &NPCexit12
    ];

    tot := length(npclist);
    for (i := 1; i <= tot; i++) {
        //
        // If this NPC exit property is anything but
        // nil (i.e., just nil, and not a method
        // that returns nil). then NPC's can use it.
        // Methods that return nil are fine because
        // they might be conditional on some game
        // events, like the crystal bridge having
        // been created, etc.
        //
        if (proptype(o, npclist[i]) <> 5) // not = nil
            o.NPCexits := o.NPCexits + npclist[i];
    }
}

/*
 * Make sure NPC room connections are sound.  Note that this check is very
 * simplistic; no account is taken of game modes, so it won't notice
 * if all the exits are invalid in (say) the 551-point version.  It also
 * fails to notice if all the exits are locked doors and/or NPCexit methods 
 * which return nil.
 */
check_connections: function
{
    local   o;

    o := firstobj(room);
    while (o <> nil) {
        if (not o.noNPCs)
            do_debug(o);

        o := nextobj(o, room);
    }
}

do_debug: function(o)
{
    local   i, tot;

    if (length(o.NPCexits) = 0) {
        P(); I();
        "Oh dear, my room connection table isn't right.
        The room \"";

        o.sdesc;

        "\" has no exits for NPC's to follow, but it's not
        listed as off limits to NPC's.  Please notify the
        cave management as soon as possible!";
    }
    else if (global.debug) {
        //
        // Debugging info:
        //
        "\b\""; o.sdesc; "\" has "; say(length(o.NPCexits));
        if (length(o.NPCexits) > 1)
            " NPC exits:";
        else
            " NPC exit:";

        tot := length(o.NPCexits);
        for (i := 1; i <= tot; i++) {
            "\b\t-> ";
            if (o.(o.NPCexits[i]))
                o.(o.NPCexits[i]).sdesc;
            else
                "(nil)";
        }
    }
}

class feedable: object
/* DJP - class to set up default methods for feeding NPC's with specific
   items */
/* syntax is feed item to actor, feed actor with item */
/* other methods are defined in advmods.t - see 'thing' and 'item'. */
    verDoFeed (actor) = {}
    verDoFeedWith (actor, io) = {
        if(io = parserGetMe()) {
            "Pull yourself together!  Self-sacrifice is \(not\)
            the object of this game. ";
            return;
        }
        vertesttake(actor,io);
    }
    doFeedWith (actor, io) = {
        if(isclass(io,contliquid) and ((io.myflag = &haswater) or
        (io.myflag = &haswine))){
            caps(); self.thedesc; if (self.isThem)
                " don't appear to be thirsty. ";
            else
                " doesn't appear to be thirsty. ";
        }
        else {
            "I don't think that ";
            self.thedesc; " would want to eat ";io.thedesc;". ";
        }
    }
    doFeed(actor) = {
        "There's nothing here that "; self.thedesc; " would want to
        eat. ";
    }
    verIoGiveTo(actor) = {}
    ioGiveTo(actor, dobj) = {
        if(isclass(dobj,contliquid) and ((dobj.myflag = &haswater) or
        (dobj.myflag = &haswine))){
            self.doFeedWith(actor,dobj);
        }
        else if(isclass(dobj,fooditem)) {
            self.doFeedWith(actor,dobj);
        }
        else {
            caps(); self.thedesc; " doesn't appear to be at all
            interested in <<dobj.thedesc>>. ";
        }
    }
    /* Recommended methods for feedable actors:
    verIoGiveTo(actor)      default normally OK but add special checks
                            here if they're dependent only on the
                            NPC or actor, not the item given.
                            If nothing can be given, this method
                            can be used to issue an appropriate message.

    verDoFeed(actor) -      use default unless checks are needed - see
                            Dog, Wumpus for examples.

    doFeed(actor) -  method for feeding when no food is explicitly named.
    verDoFeedWith(actor,io) - omit unless special checks are needed.
    doFeedWith(actor,io) - method for feeding with specific food (io).
    ioGiveTo(actor, dobj) - method for giving 'dobj' to the actor.
    The method should detect all potential foods and pass them to
    the doFeedWith method.
    */
;

modify contliquid
/* methods to allow us to attempt to feed a liquid to a 'feedable' object. */
/* N.B. special coding must be added if this is to be actually allowed! */
/* syntax is feed item to actor, feed actor with item */
/* These methods work in conjunction with those defined for the feedable
   class in ccr-room.t */

    verIoFeedWith (actor) = {}
    ioFeedWith (actor, dobj) = {dobj.doFeedWith (actor,self);}
;

/*
 * A class defining some common things for dwarves and pirates.
 */
class NPC: object
    //
    // List of current locations.
    //
    oldloclist = []    // where they were -- DJP
    loclist = []        // where they are
    newloclist = []     // where they're going

    //
    // Scatter any NPC's that are currently in
    // the room to random rooms in the cave.  (We
    // have to make sure the new rooms aren't off
    // limits to dwarves, though.)
    //
    // DJP  - added the optional loc argument, prevented NPCs from being
    // scattered to the same room.
    scatter(...) = {
        local   i, dest, len, r, tot, loc;
        local   melocs;
        
        if(argcount = 0) {
            loc := parserGetMe().location;
        }
        else {
            loc := getarg(1);
        }

        // DJP - modified to stop at a room with the istoproom property set
        melocs := []; i := loc;
        while (i) {
            melocs += i;
            if (i.istoproom) i := nil;
            else i := i.location;
        }


        /* DJP - corrected the following 'if' condition which was wrong, 
         * causing this routine to be disabled. 
         */
        if (length(intersect(melocs, self.loclist)) = 0) return;

        self.newloclist := [];
        tot := length(self.loclist);
        len := length(global.NPCrooms);
        for (i := 1; i <= tot; i++) {
            if (find(melocs, self.loclist[i])) {
                //
                // Make sure we get a real location.
                //
                dest := nil;
                while (dest = nil) {
                    r := rand(len);
                    dest := global.NPCrooms[r];
                    if((not dest.NPCvalid) or find(melocs, dest))
                        dest := nil; // DJP
                }

                self.newloclist += dest;
            }
            else {
                self.newloclist += self.loclist[i];
            }

        }
        self.loclist := self.newloclist;
    }


    //
    // Place NPCs in starting locations.
    //
    place = {
        local   i, loc, r;
        local currentsave, travelsave;

        // Define the NPC as the current actor in case any referenced 
        // methods depend on this.  For example, NPCexit or doordest methods 
        // may reference the location of a floatingdecoration item.  For
        // correct results, this must be evaluated from the viewpoint of the
        // relevant NPC, not the player.

        currentsave := global.currentActor;
        travelsave := global.travelActor;
        global.currentActor := self;
        global.travelActor := self;

        self.loclist := [];
        for (i := 1; i <= global.(self.number_to_place); i++) {
            //
            // If there are any fixed starting locations
            // for pirates left, put this pirate in the
            // next one.  Otherwise place him randomly.
            //
            loc := nil;
            if (length(global.(self.NPClocs)) >= i)
                loc := global.(self.NPClocs)[i];

            //
            // Invalidate initial location if it's off limits
            // to NPC's.
            //
            if (loc)
                if (not loc.NPCvalid)
                    loc := nil;

            //
            // Make sure we get a real location.
            //
            while (loc = nil) {
                r := rand(length(global.NPCrooms));
                loc := global.NPCrooms[r];
                if (not loc.NPCvalid)
                    loc := nil;
            }

            //
            // Add this pirate's location to the list.
            //
            self.loclist := self.loclist + loc;
        }
        // reset the current actor
        global.currentActor := currentsave;
        global.travelActor := travelsave;
    }

    //
    // Returns true if any NPC's of this type are in locations
    // adjacent to the player.  (I.e., if any NPC's could take
    // any exit that would bring them to the player's current
    // location.)
    //
    anyadjacent = {
        local   adjacent, i, j, len, dir, dest, melocs;
        local   tot1;
        local   cur;
        local currentsave, travelsave;

        //"\nanyadjacent(enter)\n";

        // Define the NPC as the current actor in case any referenced 
        // methods depend on this.  For example, NPCexit or doordest methods 
        // may reference the location of a floatingdecoration item.  For
        // correct results, this must be evaluated from the viewpoint of the
        // relevant NPC, not the player.

        currentsave := global.currentActor;
        travelsave := global.travelActor;

        // DJP - modified to stop at a room with the istoproom property set
        melocs := []; i := parserGetMe().location;
        while (i) {
            melocs += i;
            if (i.istoproom) i := nil;
            else i := i.location;
        }

        global.currentActor := self;
        global.travelActor := self;

        adjacent := nil;
        tot1 := length(self.loclist);
        for (i := 1; i <= tot1; i++) {
            len := length(self.loclist[i].NPCexits);
            cur := self.loclist[i];
            for (j := 1; j <= len; j++) {
                dest := cur.(cur.NPCexits[j]);
                if (dest)
                     if(dest.isdoor) dest := dest.NPCdest;

                //
                // We need to check the destination
                // to be sure it exists.  It may be
                // nil if we called an NPCexit method.
                //
                if (dest) if (find(melocs, dest)) {
                    adjacent := true;
                    break;
                }
            }

            //
            // If we've found an adjacent NPC we
            // can stop looking.
            //
            if (adjacent)
                break;
        }

        //"\nanyadjacent(exit)\n";
        // reset the current actor
        global.currentActor := currentsave;
        global.travelActor := travelsave;
        return adjacent;
    }

    // DJP - added a method to summon up a dwarf or pirate at will.
    // For game testing only (although we may need to allow players to
    // issue this command in order to complete the 430-point game in
    // nodwarves mode!)
    verDoSummon(actor) = {}
    doSummon(actor) = {
        local toploc := toplocation(actor), i, o,
        l := length(self.loclist);
        if(not toploc.NPCvalid) {
            "I can't do that, because <<self.thedesc>> isn't
            allowed in this room. ";
            return;
        }
        if(self = Pirates and treasure_chest.spotted) {
            "You can't summon the pirate after you've spotted the
            treasure chest. ";
            return;
        }
        // attempt to remove one NPC from another location.  NPC's are
        // not removed from the player's room in order to allow repeated
        // SUMMONs to summon up more dwarves.
        for (i := 1; i <= l; i++) {
            o := self.loclist[i];
            if (o <> toploc) {
                self.loclist -= o;
                break;
            }
        }
        // add this location to the front of the list.
        self.loclist := [toploc] + self.loclist;
        // start the NPCs if necessary
        if(global.nodwarves) global.nodwarves := nil;
        if(not global.NPCstarted) {
            notify(Dwarves, &move, 0);
            // Pirates exist only when the chest hasn't been spotted.
            if (not treasure_chest.spotted) notify(Pirates, &move, 0);
            global.NPCstarted := true;
        }
        if (self = Dwarves) self.noattack := true;
    }
    verDoBanish(actor) = {
        if(global.nodwarves and length(Dwarves.loclist) = 0 and
        length(Pirates.loclist) = 0)
            "The dwarves and pirate have already been banished. ";
    }
    // This is almost the same as nodwarves, but doesn't deduct points.
    // Available only when the game is compiled for debugging.
    doBanish(actor) = {
        "Banishing all dwarves and pirates ... ";
        global.nodwarves := true;
        Dwarves.loclist := [];
        Pirates.loclist := [];
        if(not treasure_chest.spotted) {
             treasure_chest.moveInto(Dead_End_13);
        }
        global.nodwarves := true;
    }
;

/* Dummy actor used for location method evaluation, e.g. in travelTo code */

DummyActor: object
;

/* The NPCs for the original version. */

/*
 * Move the dwarves.  See the global object in ccr-std.t for parameters.
 *
 * Also see the comments near the beginning of ccr-room.t concerning relevant
 * room properties (NoNPCs, deleted, version_NoNPCs etc.)
 *
 */
Dwarves: feedable, NPC, Actor
    rhetoricalturn = -999   // hack -- see yesVerb in ccr-verbs.t
    attackers = 0           // number of dwarves that attack this turn

    sdesc = "threatening little dwarf"
    ldesc = {
        "It's probably not a good idea to get too close.  Suffice
        it to say the little guy's pretty aggressive.";
    }

    noun = 'dwarf' 'dwarves' 'dwarfs' 'guy' //DJP added dwarfs
    adjective = 'threatening' 'nasty' 'little' 'mean'

    disavow = "I don't think the dwarves would be interested in
        answering questions, even if they understood your
        language."
    //
    // We don't use actorDesc for the dwarves because it gets printed
    // too late.  (We want to let the player know that a dwarf is
    // in the room as soon as the dwarf moves into the room, not at
    // the start of the next turn.)
    //
    actorDesc = {}

    oldmelocs = [] // DJP
   
    loc = nil      // DJP

    locationOK = true       // tell compiler OK for location to be method
    location = {
        local   i, melocs;
        // DJP - during NPC movement, the location method returns the 
        // room of interest.
        if(loc and (global.currentActor = self)) {
             return loc;
        }
        // DJP - modified to stop at a room with the istoproom property set
        melocs := []; i := parserGetMe().location;
        while (i) {
            melocs += i;
            if (i.istoproom) i := nil;
            else i := i.location;
        }
        if (length(intersect(melocs, self.loclist)) = 0)
            return(nil);

        //
        // Check each dwarf's location.  If at least one dwarf
        // is in the same location as the player, make our
        // location the same as the player's.
        //
        for (i := 1; i <= length(self.loclist); i++)
            if (find(melocs, self.loclist[i]))
                return self.loclist[i];

        return nil;
    }

    // ATTACKING (complex)
    verDoAttack(actor) = {
        if (not (axe.isIn(actor) or sword.isIn(actor)
             or singing_sword.isIn(actor))) {
            "With what?  Your bare hands?";
            self.rhetoricalturn := global.turnsofar;
            Dwarves.noattack := true;
        }
    }
    doAttack(actor) = {
        // By default, attack with the axe ...
        if (axe.isIn(actor)) {
            "\n(with the axe)\n";
            self.doAttackWith(actor, axe);
        }
        // or the sword ...
        else if (sword.isIn(actor)) {
            "\n(with the sword)\n";
            self.doAttackWith(actor, sword);
        }
        else if (singing_sword.isIn(actor)) {
            "\n(with the singing sword)\n";
            self.doAttackWith(actor, singing_sword);
        }
    }
    //
    // The following method is called when the player attacks with
    // bare hands (except the 550-point game and derivatives)
    //
    nicetry = { "You wish."; }

    verDoAttackWith(actor, io) = {}
    doAttackWith(actor, io) = {
        local kilpct := 25, count, skipnext, pass_to_game551;
        switch(global.vnumber) {
          case 11:
          case 15: // In 701-point mode, we mainly use the 550-point code
                   // except for the gleaming sword.
          case 2:  // In 550-point mode
          case 7:  // In 580-point mode
            count := self.numberhere(parserGetMe());
            switch(io) {
                case axe: // The axe is best off thrown.
                kilpct := 5*(parserGetMe().maxbulk - 
                addbulk(parserGetMe().contents)) + 15*count;
                if(global.saidthrow) kilpct += 45;
                else kilpct += 30;
                if (rand(100) <= kilpct) self.kill;
                else "You attack a little dwarf, but he
                    dodges out of the way.";
                break;

                case sword: // Pass on to 551-point code
                pass_to_game551 := true;
                break;

                case singing_sword: // The sword is best off not thrown.
                kilpct := 5*(parserGetMe().maxbulk - 
                addbulk(parserGetMe().contents)) + 15*count;
                if(global.saidthrow) kilpct += 30;
                else kilpct += 45;
                // added provision for clover in 701-point game
                if (clover.isIn(parserGetMe())) kilpct += 25;
                if (rand(100) <= kilpct) self.kill;
                else "You attack a little dwarf, but he
                    dodges out of the way.";
                break;

                case Hands: // This has different code.
                kilpct:= 10*(parserGetMe().maxbulk - 
                addbulk(parserGetMe().contents)) + 20;
                if (rand(100) <= kilpct) self.kill;
                else "You attack a little dwarf, but he
                    dodges out of the way.";
                break;

                default: "Somehow I doubt that'll be very effective.";
                break;
            }
            if (not pass_to_game551) {
                if (global.saidthrow) 
                    // DJP - updated destination to use new 
                    // throwHitDest property
                    io.moveInto(toplocation(parserGetMe()).throwHitDest);
                global.saidthrow := nil;
                break;
            }
          case 1:
            if (((io = axe) or (io = sword)) and not global.saidthrow) {
                skipnext := true;
                if(clover.isInside(parserGetMe())) kilpct := 50;
                if((io = sword) and (crown.location = parserGetMe()) and 
                (crown.isworn))
                    kilpct := 95;
                // If only one dwarf is present, we disable dwarf attacks
                // at the end of the turn.
                if (self.numberhere(actor) <= 1) self.noattack := true;
                if(rand(100) <= kilpct) self.kill;
                else if(rand(100) <= 25) {
                    if (parserGetMe().protection < 3) {
                        "As you move in for the kill, the dwarf neatly slips a
                        knife between your ribs. ";
                        die();
                    }
                    else if(parserGetMe().protection <= 5) {
                        "As you move in for the kill, the dwarf lunges at
                        you with a knife - but something seems to deflect it
                        away from you. It misses. ";
                    }
                    else if(parserGetMe().protection >= 6) {
                        "As you move in for the kill, the dwarf lunges at
                        you with a knife, but it flies out of his hand
                        and stabs him in the heart!  The body vanishes in a
                        cloud of greasy black smoke. ";
                        self.hoist_petard(1);
                    }
                    else if(parserGetMe().protection >= 9) {
                        "As you move in for the kill, the dwarf lunges at
                        you with a knife, but it flies out of his hand
                        and stabs him in the heart!  The body vanishes in a
                        cloud of greasy black smoke. Normally the knife
                        would also vanish, but it doesn't. ";
                        self.search_and_destroy(1);
                    }
                }
                else {
                    "You can't get close enough for a clean thrust. ";
                    if(rand(100) <= 36) return;
                    "As you approach, the dwarf slashes out with his knife! ";
                    if(parserGetMe().protection < 3) {
                        if(rand(100) <= 61)
                            "It misses! ";
                        else {
                           "It gets you! ";
                            die();
                        }
                    }
                    else if(parserGetMe().protection <= 5)
                        "Something seems to deflect the knife away from
                        you, and it misses! ";
                    else if(parserGetMe().protection >= 6) {
                        "For some reason the knife flies out of the dwarf's
                        hand, does a 180 degree flip, and then shoots
                        towards him, stabbing him in the chest! The body
                        vanishes in a cloud of greasy black smoke. ";
                        if (parserGetMe().protection < 9) {
                            self.hoist_petard(1);
                        }
                        else {
                            "Normally, the knife would also vanish, but it
                            doesn't. ";
                            self.search_and_destroy(1);
                        }
                    }
                }
            }
            if(skipnext) break;

          // Code to handle throwing the axe or gleaming sword in the
          // 350-point or 550-point game.
          case 0:
            switch (io) {
              case axe:
                // DJP - the clover brings luck by making the axe more accurate
                if (not global.saidthrow) {
                    "\n(throwing the axe)\n";
                }
                global.saidthrow := nil;
                if (clover.isInside(parserGetMe()) and 
                rand(100) <= global.luckyhit) {
                    self.kill;
                }
                else if (rand(100) <= global.dwarfhit) {
                    self.kill;
                }
                else {
                    "You attack a little dwarf, but he
                    dodges out of the way.";
                }

                
                // drop the axe
                // DJP - updated destination to use new 
                // throwHitDest property
                axe.moveInto(toplocation(parserGetMe()).throwHitDest); 

                break;

              case sword: // (It's more fragile than it looks)
                global.saidthrow := nil;
                sword.throwsmash;
                break;

              case Hands: self.nicetry; break;

                default: "Somehow I doubt that'll be very effective.";
            }
            break;

            default: "Error! Unrecognized version number!"; break;
        }
    }
    kill = {
        "You killed a little dwarf.  The body
        vanishes in a cloud of greasy black
        smoke. ";
        //
        // Remove this location from our list
        // of locations where dwarves are.
        //
        self.loclist -= self.location;
    }
    // Method to kill off attacking dwarves when Me.protection = 6.  Any
    // other dwarves who witness the event will also 'flee in terror' and
    // won't be seen again.
    hoist_petard(n) = {
        local i,l;
        // kill off 'n' dwarves
        for (i := 1; i <= n; i++) {
            self.loclist -= self.location;
        }
        // work out how many are still left at this location
        l := self.numberhere(parserGetMe());
        if (l = 0) return;
        if (l = 1)
            "The remaining dwarf flees in terror! ";
        else if (l > 1)
            "The remaining dwarves flee in terror! ";
        for (i := 1; i <= l; i++) {
            self.loclist -= self.location;
        }
    }
    search_and_destroy(n) = {
        local i,l;
        // kill off 'n' dwarves
        for (i := 1; i <= n; i++) {
            self.loclist -= self.location;
        }
        // work out how many are still left at this location
        l := self.numberhere(parserGetMe());
        if (l = 0) {
            "The knife flies out of the room";
        }
        else if (l = 1)
            "The knife stabs the remaining dwarf, then flies out of the
            room";
        else if (l > 1)
            "The knife stabs the remaining dwarves, then flies out of the
            room";
        self.loclist := [];
        ", presumably searching for more dwarves.  Anyway, I have the feeling
        that they won't be troubling you again. ";
    }

    // KICKING
    verDoKick(actor) =  {}
    doKick(actor) = {
        "You boot the dwarf across the room.  He curses, then
        gets up and brushes himself off.  Now he's madder
        than ever!";
    }

    // THROWING
    verIoThrowAt(actor) = { self.verIoGiveTo(actor); }
    ioThrowAt(actor, dobj) = {
        if (isclass(dobj,weapon))
        {
            global.saidthrow := true;
            self.doAttackWith(actor, dobj);
        }
        else if (dobj = glass_vial) dobj.doThrowAt(actor, self);
        else self.ioGiveTo(actor, dobj);
    }
    verIoThrowTo(actor) = { self.verIoGiveTo(actor); }
    ioThrowTo(actor, dobj) = {self.ioGiveTo (actor, dobj);}

    // GIVING
    ioGiveTo(actor, dobj) = {
        if (isclass(dobj,fooditem) or (dobj = coal)) {
            self.doFeedWith(actor,dobj);
        }
        else {
            "The dwarf is not at all interested in your
            offer. ";
            // DJP - Taken out message which suggested that
            // dwarves steal your items after killing you -
            // they don't steal your stuff in any currently-implemented
            // version.  (They do in the 440-point game and its derivatives
            // i.e. 660, 770-point).
        }
    }

    // FEEDING
    verDoFeed(actor) = {}
    doFeed(actor) = {
        askio (withPrep);
    }
    doFeedWith(actor,io) = {
        // The coal was stolen from Funadv.  However, its use as dwarf food
        // is well documented.
        if(io = coal){
            if(testtake(actor,io)) {
            "A little dwarf has run off with your coal! ";
            //
            // Remove this location from our list
            // of locations where dwarves are.
            //
            self.loclist -= self.location;
            io.moveInto(nil);
            }
        }
        else if(isclass(io,fooditem))
            "You fool, dwarves eat only coal!  Now you've
            made him *really* mad!! ";
        else pass doFeedWith;
    }


    // OTHER (place dwarves, move dwarves, count dwarves at location etc)

    number_to_place = &dwarves // property indicating number of pirates
    NPClocs = &dwarfloc        // list of locations    
    //
    // Move dwarves.
    // Also see the comments near the beginning of ccr-room.t concerning
    // relevant room properties (NoNPCs, deleted, version_NoNPCs etc.)
    //
    move = {
        local i, j, len, vlen, r, dest, done, dir, count, nd, prevloc;
        local melocs;
        local currentsave, travelsave;
        //"\ndwarves.move(enter)\n";

        //
        // Move each remaining dwarf.
        //
        // If the dwarf is currently in the player's location,
        // he stays where he is.
        //
        // If a dwarf is in a location adjacent to the player's
        // current location, he moves into the player's location
        // if he can.  (We check his possible exits to see if
        // any of them go the player's location.)  A consequence
        // of this is that dwarves will follow the player
        // relentlessly once they've spotted him.  (But the global
        // value dwarftenacity can be set to prevent dwarves
        // from *always following*, of course.)
        //
        // If a dwarf isn't adjacent to the player, he just moves
        // around randomly.
        //

        // DJP - modified to stop at a room with the istoproom
        // property set
        melocs := []; i := parserGetMe().location;
        while (i) {
            melocs += i;
            if (i.istoproom) i := nil;
            else i := i.location;
        }

        self.newloclist := [];
        self.attackers := 0;    // assume no dwarves attack this turn
        nd := length(self.loclist);

        // Define the NPC actor object as the current actor (in case
        // any NPCexit or doordest methods reference the location of a
        // floatingdecoration item - for correct results, this must
        // be calculated from the NPC's vantage point, not the player's.)

        currentsave := global.currentActor; 
        travelsave := global.travelActor; 
        global.currentActor := self;
        global.travelActor := self;

        for (i := 1; i <= nd; i++) {
            //
            // Get a copy of this dwarf's location for speed.
            //
            loc := self.loclist[i];

            //
            // Haven't found a new location yet.
            //
            done := nil;
    
            //
            // In player's current location?
            //
            if (find(melocs, loc)) {
                dest := loc;    // stay put
                done := true;
            }   

            //
            // Try each exit and see if we can reach the
            // player.
            //
            // DJP - The original version didn't check the
            // validity of the destination, but we must now do
            // so because it might depend on the game version.
            //
    
            len := length(loc.NPCexits);
            
            // Count the number of valid exits found in the following
            // loop.  (This may be less than the actual number of exits, but
            // it will be nonzero if at least one valid exit is found).

            vlen := 0;

            if (not done) for (j := len; j > 0; j--) {
                dir := loc.NPCexits[j];
                dest := loc.(dir);
    
                // Handle doors using the NPCdest method in advmods.t
                if (dest) {
                    if (dest.isdoor) { 
                        dest := dest.NPCdest;
                    }
                }
                if (dest) {
                    if (dest.NPCvalid) vlen++;
                }

                //
                // We need to check the destination
                // to be sure it exists.  It may be
                // nil if we called an NPCexit method.
                //
                if (dest <> nil and find(melocs, dest) <> nil) {
                    if(dest.NPCvalid) {
                        //
                        // Is this dwarf tenacious enough
                        // to follow the player?
                        //
                        // DJP - changed to work more like the original
                        // versions.  Dwarves will stick with the player
                        // once they've entered the player's location, but
                        // will have less tendency to 'spot' the player
                        // from an adjacent room.   This should reduce the
                        // 'ganging up' of several dwarves in one room.
                        if (find(self.oldmelocs, loc) <> nil) {
                            if (rand(100) <= global.dtenacity)
                                done := true;
                        }
                        else {
                            if (rand(100) <= global.dspotfromadj)
                            done := true;
                        }
                        break;
                    }
                }
            }
    
            //
            // Have we found a destination yet?  If not,
            // move dwarf to a randomly selected adjacent
            // location.
            //
            // We need to check the destination because
            // the NPCexit methods in the rooms can sometimes
            // return nil.    (For example, when the crystal
            // bridge doesn't exist yet, the giant's door
            // has not been opened, etc.)
            //
            // DJP - we also need to take care to check that the
            // proposed destination is valid in the current
            // game version.  We make two checks for a stuck NPC.
            //
            count := 0;
            while (not done) {
                count += 1;
                if((vlen < 1) or (count > 50)) {
                    debugTrace();
                    if (not loc.npcstuckwarn) {
                        "\bWarning: NPCs appear not to be finding any
                        valid exits in \"<<loc.sdesc>>\"! Please ask 
                        the cave management to check the room connection 
                        table.\b";
                        loc.npcstuckwarn := true;
                    }
                    self.scatter(loc);
                    dest := self.loclist(i);
                    break;
                }
                dir := loc.NPCexits[rand(len)];
                dest := loc.(dir);

                // Handle doors using the NPCdest method in advmods.t
                if (dest) {
                    if (dest.isdoor) { 
                        dest := dest.NPCdest;
                    }
                }

                if (dest) {

                    if(dest.NPCvalid) {
                        // DJP - avoid going to a previous
                        // location in the first 3 attempts.  If no dwarves
                        // have been killed, we look at the previous loc of
                        // the current dwarf.  If a dwarf has been killed,
                        // there won't be a one-to-one correspondence and
                        // we instead look at whether any dwarf was
                        // previously at the proposed location.
                        if (count <= 3) {
                            if (length(self.oldloclist) <>
                            length(self.loclist))
                                prevloc := (find(self.oldloclist,dest)
                                <> nil);
                            else
                                prevloc := (oldloclist[i] = dest);
                            if (not prevloc) done := true;
                        }
                        else done := true;
                    }
                }
            }
    
            //
            // Set new destination.
            //
            self.newloclist += dest;
    
            //
            // If the dwarf didn't move, he has an opportunity
            // to attack.
            //
            if (loc = dest and not self.noattack) {
                if (find(melocs, loc))
                if (rand(100) <= global.dwarfattack)
                    self.attackers++;
    
                //
                // Print some debugging info if in debug mode
                //
                if (global.debug) {
                    P();
                    "Dwarf stays \""; dest.sdesc; ".\"\n";
                }
            }
            else {
                //
                // Print some debugging info if in debug mode
                //
                if (global.debug) {
                     P();
                     "Dwarf moves from \"";
                     self.loclist[i].sdesc; "\" to \"";
                     dest.sdesc; ".\"\n";
                }
            }
        }

        // Reset the actor and location
        global.currentActor :=  currentsave;
        global.travelActor :=  travelsave;
        loc := nil;
    
        //
        // Replace old locations with destinations.
        //
        self.oldloclist := self.loclist; // DJP
        self.loclist := self.newloclist;
    
        self.tell;
        //"\ndwarves.move(exit)\n";
    
        self.noattack := nil; // DJP - clear noattack flag (set when
                              // a weapon is taken, or when a dwarf
                              // is attacked in hand-to-hand combat)
        self.oldmelocs := melocs; // DJP - save melocs list
    }

    //
    // Tell the player what's going on with the dwarves.
    //
    tell = {
        local    i, j, len, r, dest, done, dir, count, adjacent;
        local    melocs;
        // Do nothing if the room is dark
        if (not parserGetMe().location.islit) {
            self.attackers := 0;
            return;
        }
        //
        // Count how many dwarves are in the room with the player.
        //
        /* DJP: count was set to 1 when no. of dwarves was > 1 */
        count := 0;
        // DJP - modified to stop at a room with the istoproom property set
        melocs := []; i := parserGetMe().location;
        while (i) {
            melocs += i;
            if (i.istoproom) i := nil;
            else i := i.location;
        }
        for (i := 1; i <= length(self.loclist); i++){
            if( find(melocs,self.loclist[i]) ) count++;
        }
        //
        // If any dwarves are in the room with the player and
        // the axe hasn't been thrown yet, throw the axe and
        // scatter the dwarves.
        //
        if (count > 0 and axe.location = nil) {
            P(); I();

            "A little dwarf just walked around a corner,
            saw you, threw a little axe at you which
            missed, cursed, and ran away.";

            axe.moveInto(self.location.throwHitDest);

            //
            // Scatter any dwarves in the room.
            //
            self.scatter;

            //
            // No dwarves in the room.  Be sure we take back
            // those attacks too...
            //
            count := 0;
            self.attackers := 0;
        }

        //
        // Tell the player if any dwarves are in the room with him,
        // or if any are nearby.
        //
        if (count = 0) {
            //
            // If no dwarves are in the room, but at least
            // one dwarf is in an adjacent location, tell
            // the player he hears something.
            //
            // (Only the pirate makes noise in the original,
            // which seems a bit strange and not as much fun.)
            //
            if (self.anyadjacent) {
                P(); I(); "You hear the pitter-patter
                of little feet.";
            }
        }
        else if (count = 1) {
            P(); I();
            "There is a threatening little dwarf in the
            room with you.";
        }
        else if (count > 1) {
            P(); I();
            "There are "; say(count); " threatening
            little dwarves in the room with you.";
        }

        //
        // Handle dwarf attacks.
        //
        if (self.attackers > 0) {
            if (self.attackers = 1) {
                if (count = 1)
                    " He throws a knife at you!";
                else
                    " One of them throws a knife
                    at you!";
            }
            else {
                if (self.attackers = count) {
                    if (count = 2)
                        " Both of them throw
                        knives at you!";
                    else
                        " All of them throw
                        knives at you!";
                }
                else {
                    " "; say(self.attackers); " of them throw
                    knives at you!";
                }
            }

            P(); I();

            //
            // Curtains for our hero?!
            //


            if(parserGetMe().protection < 3) {
                count := 0;
                for (i := 1; i <= self.attackers; i++) {
                    if (rand(100) <= global.dwarfaccuracy) {
                        count++;
                    }
                }

                if (count > 0) {
                    if (count = self.attackers) {
                        if (count = 1)
                            "It gets you! ";
                        else if (count = 2)
                            "Both of them get you! ";
                        else
                            "All of them get you!";
                    }
                    else if (count = 1) {
                        "One of them gets you! ";
                    }
                    else {
                        say(count); " of them get
                        you! ";
                    }

                    die();
                }
                else {
                    if (self.attackers = 1)
                        "It misses you!";
                    else if (self.attackers = 2)
                        "Both of them miss you! ";
                    else
                        "They all miss you! ";
                }
            }
            else if (parserGetMe().protection <= 5) {
                if (self.attackers = 1)
                    "Something seems to deflect the knife away from you! ";
                else if (self.attackers > 1)
                    "Something seems to deflect the knives away from
                    you! ";
            }
            else if(parserGetMe().protection >= 6) {
                if (self.attackers = 1) {
                    "Something very strange happens.  The knife stops in
                    mid-air, turns round, and stabs the dwarf in the
                    chest!  The body disappears in a puff of greasy
                    black smoke. ";
                    if (parserGetMe().protection < 9)
                        self.hoist_petard(self.attackers);
                    else {
                        "Normally the knife would also disappear, but it
                        doesn't. ";
                        self.search_and_destroy(self.attackers);
                    }
                    self.hoist_petard(1);
                }
                else if (self.attackers > 1) {
                    "Something very strange happens.  The knives stop in
                    mid-air, turn round, and stab your attackers in the
                    chest!  The bodies disappear in puffs of greasy black
                    smoke. ";
                    if (parserGetMe().protection < 9)
                        self.hoist_petard(self.attackers);
                    else {
                        "Normally the knife would also disappear, but it
                        doesn't. ";
                        self.search_and_destroy(self.attackers);
                    }
                }
            }
        }
    }
    numberhere (actor) = { // DJP - a convenient method to return the
                           // number of dwarves in the player's room.
        local    i, melocs, count;
        //
        // Count how many dwarves are in the room with the player.
        //
        count := 0;
        melocs := []; i := actor.location;
        while (i) {
            melocs += i;
            if (i.istoproom) i := nil;
            else i := i.location;
        }
        for (i := 1; i <= length(self.loclist); i++){
            if( find(melocs,self.loclist[i]) ) count++;
        }
        return count;
    }

    verDoRub(actor) = {"Don't be ridiculous!";}
    doCount(actor) = {
        count := self.numberhere(actor);
        if (count=1)
            "There is 1 nasty little dwarf in the room
            with you! ";
        else
            "There are <<count>> nasty little dwarves
            in the room with you! ";
    }
;
/*
 * The player can never get the dwarves' knives (that would be too easy),
 * but we'll let him examine them anyway.
 */
DwarfKnives: item, floatingItem
    sdesc = "dwarf's knife"
    ldesc = { self.verifyRemove(parserGetMe()); }
    noun = 'knife' 'knives'
    adjective = 'sharp' 'nasty' 'dwarf\'s' 'dwarvish' 'dwarven'
                'dwarfish'

    locationOK = true       // tell compiler OK for location to be method
    location = {
        return Dwarves.location;
    }

/* DJP: 'Take knife' gives the message about knives vanishing. */
    verDoTake( actor ) = {
        self.verifyRemove(actor);
    }
    verifyRemove(actor) = {
        "The dwarves' knives vanish as they strike the walls
        of the cave.";
    }

    verDoAttackWith(actor,dobj) = {
        "You don't have the dwarf's knife!";
    }

    dobjGen(a, v, i, p) =
    {
        if ((v <> inspectVerb) and (v <> gonearVerb) and (v <> countVerb))
        {
            self.verifyRemove(a);
            exit;
        }
    }
    iobjGen(a, v, d, p) = {
        if (v = askVerb or v = tellVerb or v = waveVerb)return;
        self.dobjGen(a, v, d, p);
    }
    verDoCount(actor) = {self.verifyRemove(actor);}
;

/*
 * Move the pirate(s).  See the global object in ccr-std.t for parameters.
 *
 * Also see the comments near the beginning of ccr-room.t concerning relevant
 * room properties (NoNPCs, deleted, version_NoNPCs etc.)
 *
 * This code is quite similar to the Dwarves code, but is simpler because
 * there's never any interaction between the player and the pirates. (The
 * pirates just come in, do their stuff, and vanish.)
 *
 * Note that even if there's more than one pirate, the text printed in
 * in this object will treat all the pirates as a single one. So the
 * only difference having multiple pirates makes is that the more you
 * have, the more likely the player is to run into "him."
 */
Pirates: NPC, Actor, floatingItem
    sdesc = "pirate"
    noun = 'pirate' // (for gonear)

    seen = nil      // has the player seen (one of) us?

    loc = nil  // DJP

    locationOK = true       // tell compiler OK for location to be method

    location = {
        local   i, melocs;
        // DJP - during NPC movement, the location method returns the 
        // room of interest.
        if(loc and (global.currentActor = self)) {
             return loc;
        }
        // The Pirate isn't a real actor, so in other cases we pretend he
        // doesn't exist.
        else return nil;
    }

    //No need for attacking,giving,feeding etc.

    number_to_place = &pirates // property indicating number of pirates
    NPClocs = &pirateloc       // list of locations    
                          
    //
    // Move pirates.
    // Also see the comments near the beginning of ccr-room.t concerning
    // relevant room properties (NoNPCs, deleted, version_NoNPCs etc.)
    //
    move = {
        local i, j, len, vlen, r, dest, done, dir, count, np, prevloc;
        local melocs;
        local currentsave, travelsave;

        //"\npirates.move(enter)\n";
        //
        // Move each remaining pirate.
        //
        // If the pirate is currently in the player's location,
        // he stays where he is.
        //
        // If a pirate is in a location adjacent to the player's
        // current location, he moves into the player's location
        // if he can.  We limit this with the pspotfromadj global.
        //
        // If a pirate isn't adjacent to the player, he just moves
        // around randomly.
        //
        self.newloclist := [];
        currentsave := global.currentActor;
        travelsave := global.travelActor;
        // DJP - modified to stop at a room with the istoproom
        // property set
        melocs := []; i := parserGetMe().location;
        while (i) {
            melocs += i;
            if (i.istoproom) i := nil;
            else i := i.location;
        }

        self.newloclist := [];
        np := length(self.loclist);

        // Define the NPC as the current actor in case any referenced 
        // methods depend on this.  For example, NPCexit or doordest methods 
        // may reference the location of a floatingdecoration item.  For
        // correct results, this must be evaluated from the viewpoint of the
        // relevant NPC, not the player.

        global.currentActor := self;
        global.travelActor := self;

        for (i := 1; i <= np; i++) {
            //
            // Get a copy of this pirate's location for speed.
            //
            loc := self.loclist[i];

            //
            // Haven't found a new location yet.
            //
            done := nil;

            //
            // In player's current location?
            //
            if (find(melocs, loc)) {
                dest := loc;    // stay put
                done := true;
            }

            //
            // Try each exit and see if we can reach the
            // player.
            //
            // DJP - The original version didn't check the
            // validity of the destination, but we must now do
            // so because it might depend on the game version.
            //

            len := length(loc.NPCexits);

            // Count the number of valid exits found in the following
            // loop.  (This may be less than the actual number of exits, but
            // it will be nonzero if at least one valid exit is found).

            vlen := 0;

            if (not done) for (j := len; j > 0; j--) {
                dir := loc.NPCexits[j];
                dest := loc.(dir);

                // Handle doors using the NPCdest method in advmods.t
                if (dest) {
                    if (dest.isdoor) { 
                        dest := dest.NPCdest;
                    }
                }
                if (dest) {
                    if (dest.NPCvalid) vlen++;
                }

                //
                // We need to check the destination
                // to be sure it exists.  It may be
                // nil if we called an NPCexit method.
                //
                if (dest <> nil and find(melocs, dest) <> nil) {
                    if(dest.NPCvalid) {
                        //
                        // DJP - renamed ptenacity to pspotfromadj
                        // for consistency.
                        //
                        // Will this pirate spot the player?
                        // (note: ptenacity is not defined because
                        // the pirate 'scatters' after seeing the
                        // player, rather than continuing to follow
                        // the player.)
                        //
                        if (rand(100) <= global.pspotfromadj)
                            done := true;
                        break;
                    }
                }
            }

            //
            // Have we found a destination yet?  If not,
            // move pirate to a randomly selected adjacent
            // location.
            //
            // We need to check the destination because
            // the NPCexit methods in the rooms can sometimes
            // return nil.    (For example, when the crystal
            // bridge doesn't exist yet, the giant's door
            // has not been opened, etc.)
            //
            // DJP - we also need to take care to check that the
            // proposed destination is valid in the current
            // game version.  We make two checks for a stuck NPC.
            //
            count := 0;
            while (not done) {
                count += 1;
                if((vlen < 1) or (count > 50)) {
                    debugTrace();
                    if (not loc.npcstuckwarn) {
                        "\bWarning: NPCs appear not to be finding any
                        valid exits in \"<<loc.sdesc>>\"! Please ask 
                        the cave management to check the room connection 
                        table.\b";
                        loc.npcstuckwarn := true;
                    }
                    self.scatter(loc);
                    dest := self.loclist(i);
                    break;
                }
                dir := loc.NPCexits[rand(len)];
                dest := loc.(dir);

                // Handle doors using the NPCdest method in advmods.t
                if (dest) {
                    if (dest.isdoor) { 
                        dest := dest.NPCdest;
                    }
                }

                if (dest) {

                    if(dest.NPCvalid) {
                        // DJP - avoid going to a previous
                        // location in the first 3 attempts.  If no pirates
                        // have been removed, we look at the previous loc
                        // of the current pirate.  If a pirate has been
                        // removed, there won't be a one-to-one
                        // correspondence and we instead look
                        // at whether any pirate was previously
                        // at the proposed location.
                        if (count <= 3) {
                            if (length(self.oldloclist) <>
                            length(self.loclist))
                                prevloc := (find(self.oldloclist,dest)
                                <> nil);
                            else
                                prevloc := (oldloclist[i] = dest);
                            if (not prevloc) done := true;
                        }
                        else done := true;
                    }
                }
            }

            //
            // Set new destination.
            //
            self.newloclist += dest;

            //
            // Print some debugging info if in debug mode
            //
            if (self.loclist[i] = dest) {
                if (global.debug) {
                    P();
                    "Pirate stays \""; dest.sdesc; ".\"\n";
                }
            }
            else {
                if (global.debug) {
                    P();
                    "Pirate moves from \"";
                    self.loclist[i].sdesc; "\" to \"";
                    dest.sdesc; ".\"\n";
                }
            }
        }

        // Reset the current actor and location
        global.currentActor :=  currentsave;
        global.travelActor :=  travelsave;
        loc := nil;

        //
        // Replace old locations with destinations.
        //
        self.oldloclist := self.loclist; // DJP
        self.loclist := self.newloclist;

        self.tell;
        //"\npirates.move(exit)\n";
    }

    //
    // Tell the player what's going on with the pirates.
    //
    tell = {
        local    i, t, o, count, snagged, melocs;
        local left_pendants :=0, left_ring := nil;
        //"\npirates.tell(enter)\n";

        //
        // Count how many pirates are in the room with the player.
        // (We really only need to know if there are any at all,
        // but this is just as easy.)
        //
        // DJP - modified to stop at a room with the istoproom property set
        melocs := []; i := parserGetMe().location;
        while (i) {
            melocs += i;
            if (i.istoproom) i := nil;
            else i := i.location;
        }

        //
        // Count how many pirates are in the room with the player.
        //
        /* DJP: count was set to 1 when no. of dwarves was > 1 */
        count := 0;
        // DJP - modified to stop at a room with the istoproom property set
        melocs := []; i := parserGetMe().location;
        while (i) {
            melocs += i;
            if (i.istoproom) i := nil;
            else i := i.location;
        }
        for (i := 1; i <= length(self.loclist); i++){
            if( find(melocs,self.loclist[i]) ) count++;
        }

        //
        // Tell the player if any pirates are nearby.
        //
        if (count = 0) {
            //
            // If no pirates are in the room, but at least
            // one pirate is in an adjacent location, tell
            // the player he hears something.
            //
            if (self.anyadjacent) {
                P(); I();
                "There are faint rustling noises from
                the darkness behind you.";
            }
        }
        else if (count > 0) {
            //
            // A pirate has snagged the player.
            // Move any treasures the player is carring
            // to the pirate's repository, currently
            // hard-coded as treasure_chest or Dead_End_13,
            // depending on the size of the object!
            //
            // Since the player may be keeping his treasures
            // in containers, it's actually easier just
            // to search through the global list of
            // treasures and check each one to
            // see if it's visible.     (This is a change for the
            // 551-point version, in which objects in a closed
            // sack will not be stolen.)
            snagged := 0;
            for (i := 1; i <= length(global.treasurelist); i++) {
                t := global.treasurelist[i] ;
                // a treasure is eligible for taking if it is visible,
                // or (in the case of the glowing stone) its container
                // is visible and contains the treasure.
                if (t.isVisible(parserGetMe()) or ((t = glowing_stone) and
                (t.location = canister) and 
                canister.isVisible(parserGetMe()))) {
                    // DJP - pirate ignores the glowing stone when
                    // the canister is unavailable
                    if ((t = glowing_stone) and not
                    canister.isVisible(parserGetMe())) {
                        self.nostone := true;
                        continue;
                    }
                    // DJP - have the pirate put the glowing stone into
                    // the canister and close it.
                    else if (t = glowing_stone){
                        t.moveInto(canister);
                        canister.isopen := nil;
                    }
                    // DJP - deny hard-to-get treasures to the pirate.
                    // For example, the pirate shouldn't take the pyramid
                    // from the Dark Room unless it is lit.  No check is
                    // needed for the sword or chain because their
                    // locations are NoNPC. BJS: The same thing goes
                    // for most 550-point treasures.
                    if ((t = persian_rug) and
                    (t.location = Dragon.location))
                        continue;
                    else if ((t = platinum_pyramid) and
                    (t.location = In_Dark_Room) and not In_Dark_Room.islit)
                        continue;
                    else if ((t = cloak) and (not t.moved))
                        continue;
                    // In the 551-point game and derivatives, don't let the 
                    // pirate lock the mithril ring in the chest if the keys
                    // are inaccessible, or are located on the far side of
                    // the stone bridge (need the ring to get there)
                    else if ((t = mithril_ring) and global.newgame) {
                         local keytop := toplocation(set_of_keys);
                         if(keytop <> nil) {
                             if(keytop.wino_wsbridge) {
                                 left_ring := true;
                                 continue;
                             }
                         }
                    }
                    // Also don't let the pirate lock any pendant in the
                    // chest if the keys are accessible only via Transindection
                    else if (isclass(t,pendantItem)) {
                         local keytop := toplocation(set_of_keys);
                         if(keytop <> nil) {
                             if((keytop.analevel <> 0) or keytop.isolated) {
                                 left_pendants++;
                                 continue;
                             }
                         }
                    }
                    
                    // BJS: The 550-point version makes an exception
                    // for the ring, but I'm not sure why...
                    // The sword is not a treasure.
                    // decide on actual object to
                    // move - in case of wine, move
                    // the cask.
                    if(t.mycont) o := t.mycont; else o := t;
                    // decide on where the object is
                    // to go - in the chest if possible, otherwise in
                    // Dead_End_13 (N.B. chest is full in old game)
                    if(o = treasure_chest or not
                    treasure_chest.accepts_item(o)
                    or ((treasure_chest.maxbulk -
                    addbulk(treasure_chest.contents)) < o.bulk ))
                        o.moveInto(Dead_End_13);
                    else
                        o.moveInto(treasure_chest);
                    snagged++;
                }
            }

            //
            // Print a message telling the player what happened.
            //
            if (snagged > 0) {
                P();
                I(); "Out from the shadows behind you
                pounces a bearded pirate!  \"Har,
                har,\" he chortles.  \"I'll just take
                all this booty and hide it away ";
                if(global.newgame)"in "; else "with ";
                "me chest deep in the maze!\"  He
                snatches your treasure and vanishes
                into the gloom.  ";
                // check for a change of score in the 701+ point version
                // (possible if the pirate has taken pendants)
                global.gendaemon;
            }
            else {
                //
                // DMB - In the original code, if you weren't
                // holding the lamp, you just wouldn't
                // see the pirate when you weren't
                // carrying any treasures.  This seems
                // bogus, so I've added a conditional here.
                //
                P();
                I(); "There are faint rustling noises
                from the darkness behind you.  As you
                turn toward them, ";

                if (brass_lantern.isIn(parserGetMe()) and brass_lantern.islit)
                    "the beam of your lamp falls
                    across";
                else
                    "you spot";

                " a bearded pirate. He is carrying a
                lamp";
                if (treasure_chest.location <> Dead_End_13) {
                    " and a large chest.  \"Shiver me timbers!\"
                    he cries, \"I've been spotted!
                    I'd best hie meself off to the maze to
                    hide me chest!\" ";
                }
                else {
                    ". \"Shiver me timbers!\" he cries,
                    \"I've been spotted!\" ";
                }
                "With that, he
                vanishes into the gloom. ";
            }
            if (self.nostone) {
                self.nostone := nil; P(); I();
                "You realize that the pirate seems to have
                overlooked the glowing stone.  I wonder if
                he knows something that you don't? ";
            }
            if (left_pendants > 0) {
                "You notice that the pirate seems to have missed your
                pendant";
                if(left_pendants > 1)"s";
                ".  That's just as well - he locks his treasures in the chest,
                and you may need to use the pendant";
                if(left_pendants > 1)
                    "s";
                " to get to the keys. ";
            }
            if (left_ring) {
                "For some reason the pirate hasn't taken your mithril ring.
                That's just as well - he locks his treasures in the chest, and
                the keys are on the far side of the stone bridge! ";
            }
            //
            // Install the treasure chest if it hasn't
            // already been installed.  No worries about
            // the chest appearing out of nowhere when
            // the player's at Dead_End_13, because the
            // pirate can't go there.  (It's off limits
            // to NPC's.)
            //
            if (not self.seen) {
                treasure_chest.moveInto(Dead_End_13);
                /* remember to install the pirate message */
                PirateMessage.moveInto(Dead_End_14);
                self.seen := true;
            }

            //
            // Scatter any pirates in the room.
            //
            self.scatter;
        }
//"\npirates.tell(enter)\n";
    }
;

Snake: feedable, Actor
    sdesc = "snake"
    ldesc = "I wouldn't mess with it if I were %you%."
    location = In_Hall_Of_Mt_King
    noun = 'snake' 'cobra' 'asp'
    adjective = 'huge' 'fierce' 'green' 'ferocious' 'venemous'
        'venomous' 'large' 'big' 'killer'

    actorDesc =  {"A huge green fierce snake bars the way!";}


    // ATTACKING
    verDoAttack(actor) = {}
    doAttack(actor) = {
        "Attacking the snake both doesn't work and is very
        dangerous.";
    }
    verDoAttackWith(actor, io) = { self.verDoAttack(actor); }
    doAttackWith(actor, io) = { self.doAttack(actor); }

    // KICKING
    verDoKick(actor) = {}
    doKick(actor) = { "That would be satisfying, but totally insane."; }

    // THROWING
    verIoThrowAt(actor) = { self.verIoGiveTo(actor); }
    ioThrowAt(actor, dobj) = {
        if (isclass(dobj,weapon))
            self.doAttackWith(actor, dobj);
        else if (dobj = glass_vial) dobj.doThrowAt(actor, self);
        else if (isclass(dobj,fooditem)) self.ioGiveTo(actor, dobj);
        else self.doAttackWith(actor, dobj);
    }
    verIoThrowTo(actor) = { self.verIoGiveTo(actor); }
    ioThrowTo(actor, dobj) = {self.ioGiveTo(actor, dobj);}

    // GIVING
    doGiveTo(actor, io) = {
        if (io = little_bird)
            self.doFeedWith(actor,little_bird);
        else {
            "The snake does not seem interested in ";
            io.thedesc; ".";
        }
    }

    //FEEDING
    doFeed(actor) = {
        if (little_bird.isReachable(parserGetMe()))
            self.doFeedWith(actor,little_bird);
        else
            "There's nothing here it wants to eat (except
            perhaps %you%).";
    }

    doFeedWith(actor, io) = {
        if (io = little_bird) {
            if(testtake(actor,io)){
                "The snake has now devoured %your% bird.";
                little_bird.moveInto(nil);
            }
        }
        else pass doFeedWith;
    }
    // OTHER
    verDoRub(actor) = {"I don't think ";self.thedesc;
        " would care for that.";}

    verifyRemove(actor) = { "Surely you're joking."; }
;

Dragon: feedable, Actor
    rhetoricalturn = -999   // hack -- see yesVerb in ccr-verbs.t

    sdesc = "dragon"
    ldesc = "I wouldn't mess with it if I were you."
    location = In_Secret_Canyon
    noun = 'dragon' 'monster' 'beast' 'lizard'
    adjective = 'huge' 'green' 'fierce' 'scaly' 'giant' 'ferocious'
    heredesc = {P(); I(); "A huge green dragon bars the way! ";}
    actorDesc = {}

    // ATTACKING
    verDoAttack(actor) = {}
    doAttack(actor) = {
        "With what?  %Your% bare hands?";
        self.rhetoricalturn := global.turnsofar;
    }
    verDoAttackWith(actor, io) = {}
    doAttackWith(actor, io) = {
        if (isclass(io, weapon)) {
            if (not global.saidthrow)
                "\n(throwing <<io.thedesc>>)\n";
            global.saidthrow := nil;
            caps; io.thedesc; " bounces harmlessly off the
            dragon's thick scales. ";
            io.moveInto(toplocation(actor));
        }
        else if (io = Hands) {
            self.kill;
        }
        else {
            "You'd probably be better off using
            %your% bare hands than that thing!";
        }
    }

    kill = {
        "Congratulations!  %You% %have% just vanquished a
        dragon with %your% bare hands!  (Unbelievable,
        isn't it?)";

        DragonCorpse.moveInto(self.location);
        if(global.game550) DragonTeeth.moveInto(self.location);
        self.moveInto(nil);
    }

    // KICKING
    verDoKick(actor) = {}
    doKick(actor) = {
        "Right idea, wrong limb. ";
    }

    // THROWING
    verIoThrowAt(actor) = { self.verIoGiveTo(actor); }
    ioThrowAt(actor, dobj) = {
        if (isclass(dobj,weapon))
        {
            global.saidthrow := true;
            self.doAttackWith(actor, dobj);
        }
        else if (dobj = glass_vial) dobj.doThrowAt(actor, self);
        else self.ioGiveTo(actor, dobj);
    }
    verIoThrowTo(actor) = { self.verIoGiveTo(actor); }
    ioThrowTo(actor, dobj) = {self.ioGiveTo (actor, dobj);}

    // GIVING (uses feedable defaults)

    // FEEDING (uses feedablt defaults)

    // Other

    verDoRub(actor) = {"Don't be ridiculous!";}
;
DragonCorpse: fixeditem
    sdesc = "dragon"
    ldesc = {
        "It's the rotting carcass of a huge dragon, lying off to one
        side.  There isn't a mark on it, and I can't really explain
        how you managed to kill it with your bare hands.  \(That\)
        mystery will always defy attempts at a rational explanation. ";
    }
    location = nil
    noun = 'dragon' 'corpse'
    adjective = 'dead'
    heredesc = { P();I();
        "The body of a huge green dead dragon is lying off to
        one side. "; }

    // THROWING
    verIoThrowAt(actor) = {}
    ioThrowAt(actor, dobj) = {
        if (isclass(dobj,weapon))
            self.verDoAttack(actor);
        else {
            "I don't see much point in throwing things at a
            dead dragon. ";
        }
    }

    // KICKING
    verDoKick(actor) = { "You've already done enough damage! "; }
    verDoAttack(actor) = {
        "For crying out loud, the poor thing is already dead! ";
    }
    verDoAttackWith(actor, io) = { self.verDoAttack(actor); }
;

Troll: feedable, Actor
    ispaid = nil

    sdesc = "burly troll"
    ldesc = {
        "Trolls are close relatives with rocks and have skin
        as tough as that of a rhinoceros.";
    }

    actorDesc = {
        if (not self.isIn(Troll_Treasure))
            "A burly troll stands by the bridge and insists you
            throw him a treasure before you may cross.";
        else
            "You see the troll here, counting money and cataloguing
            his treasures. You hear him muttering something about
            golden eggs.";
    }

    noun = 'troll'
    adjective = 'burly'

    location = On_Sw_Side_Of_Chasm

    // ATTACKING

    verDoAttack(actor) = {
        if(actor <> Bear) {
        if ((not axe.isIn(actor)) and (not singing_sword.isIn(actor))
             and (not sword.isIn(actor)))
            "The troll fends off %your% blows effortlessly.";
        }
    }
    doAttack(actor) = {
        if(actor = Bear) {
            "The bear lumbers toward the troll, who lets
            out a startled shriek and scurries away.  The
            bear soon gives up the pursuit and wanders
            back.";
            Troll.moveInto(nil);
            Bear.stayloc := Bear.location;
            Bear.prevloc := parserGetMe().prevloc;
            Bear.isfollowing := nil;
        }
        else {
            // By default, attack with the axe ...
            if (axe.isIn(actor)) {
                "\n(with the axe)\n";
                self.doAttackWith(actor, axe);
            }
            // or the sword ...
            else if (sword.isIn(actor)) {
                "\n(with the sword)\n";
                self.doAttackWith(actor, sword);
            }
            // or the other sword ...
            else if (singing_sword.isIn(actor)) {
                "\n(with the sword)\n";
                self.doAttackWith(actor, singing_sword);
            }
        }
    }
    verDoAttackWith(actor, io) = {}
    doAttackWith(actor, io) = {
        //
        // If the player throws the axe at the troll,
        // he just catches it.
        //
        if (isclass(io,weapon)) {
            "\n(throwing <<io.thedesc>>)\n";
            global.trollcatch := true;
            self.ioGiveTo(actor, io);
        }
        else
            "The troll fends off %your% blows effortlessly.";
    }

    // KICKING

    verDoKick(actor) = {}
    doKick(actor) = {
        "The troll laughs aloud at %your% pitiful attempt
        to injure him.";
    }

    // THROWING

    verIoThrowAt(actor) = { self.verIoGiveTo(actor); }
    ioThrowAt(actor, dobj) = {
        if (dobj = glass_vial)
            dobj.doThrowAt(actor, self);
        else {
            global.trollcatch := true;
            self.ioGiveTo(actor, dobj);
        }
    }
    verIoThrowTo(actor) = { self.verIoGiveTo(actor); }
    ioThrowTo(actor, dobj) = { self.ioThrowAt(actor, dobj); }

    // GIVING

    ioGiveTo(actor, dobj) = {
        local cont,vb1,vb2,adv;
        if (not global.trollcatch){
            vb1 := 'takes';
            vb2 := 'gives';
            adv := '';
        }
        else {
            vb1 := 'catches';
            vb2 := 'tosses';
            adv := 'deftly ';
        }
        global.trollcatch := nil;
        // if a liquid is given, give the container; similarly
        // for the glowing stone if it's in its container.
        if (isclass(dobj,contliquid) or ((dobj = glowing_stone) and
        (dobj.location = dobj.mycont))) {
            cont := dobj.mycont;
            "\n(";cont.thedesc;")\n";
        }
        else cont := dobj;

        if (isclass(dobj, CCR_treasure_item) or
        (dobj = cask and cask.haswine) or
        (dobj = canister and glowing_stone.location = canister)) {
            local letfall;
            if(cont = glowing_stone) {
                "The troll catches the glowing stone, then sets it down and
                steps back.  He says \"Don't you realize what this
                is?  Next time, use a suitable container. \" He then
                scurries off and quickly reappears with
                an ornate lead casket.  He places the stone into the 
                casket, closes it, and disappears out of sight again. ";
                cont.moveInto (lead_box);
                cont := lead_box;
            }
            else if (dobj = golden_eggs and self.isduped) {
                if (vb1 = 'catches') "The troll nimbly steps
                to one side and grins nastily as the nest of
                golden eggs flies past him and plummets into
                the chasm.  \"Fool me once, shame on you;
                fool me twice, shame on me!\" he sneers.
                \"I want something a touch more substantial
                this time!\"";
                else "The troll nimbly moves his hand away and
                grins nastily as the nest of golden eggs falls
                from your hand and plummets into the chasm.
                \"Fool me once, shame on you; fool me twice,
                shame on me!\" he sneers.  \"I want something
                a touch more substantial this time!\"";
                letfall := true;
            }
            else {
                "The troll <<vb1>> %your% treasure and
                scurries away out of sight.";
            }
            // troll closes canister for safety
            if(cont = canister) cont.isopen := nil;
            if(letfall) cont.moveInto(nil);
            else {
                cont.moveInto(Troll_Treasure);
                self.moveInto(Troll_Treasure);
            }
            // DJP - ispaid variable now holds the top-level
            // location where payment was made.
            self.ispaid := toplocation(actor);
            // Troll only accepts eggs once:
            if(letfall) self.ispaid := nil;
        }
        else if(isclass(dobj,fooditem)) {
            self.doFeed(parserGetMe());
        }
        else {
            "The troll <<adv>><<vb1>> <<cont.thedesc>>,
            examines <<cont.itobjdesc>> carefully,
                        and <<vb2>> <<cont.itobjdesc>> back,
            declaring, \"Good workmanship,
                        but <<cont.itisdesc>> not
            valuable enough.\"";
        }
    }

    // FEEDING

    /* simplified doFeed to avoid a complicated check for edible items */
    doFeed(actor) = {
        "Gluttony is not one of the troll's vices.
        Avarice, however, is.";
    }
    doFeedWith(actor,io) = {
        if (isclass(io,fooditem)) self.doFeed(actor);
        else pass doFeedWith;
    }

    // OTHER (troll, take item; ask/tell troll about x)

    verDoRub(actor) = {"I don't think ";self.thedesc;
        " would care for that.";}

    // allow the troll to take items
    actorAction( v, d, p, i ) = {
    if( not (v = takeVerb and p = nil and i = nil) )
        pass actorAction;
    }

    disavow = "He wants treasure, not gab."
    verDoTellAbout(actor,io) = {self.disavow;}

;

/* Dummy object to give a message when troll is absent.  It cannot
   be referred to by the player. */
Absent_troll: fixeditem
    sdesc = "absent troll"
    actorDesc = {
        "The troll is nowhere to be seen.";
    }
    locationOK = true
    location = {
        if ((parserGetMe().isIn(On_Sw_Side_Of_Chasm) or
        parserGetMe().isIn(On_Ne_Side_Of_Chasm)) and 
        (not parserGetMe().isIn(Troll.location)))
            return parserGetMe().location;
        else return nil;
    }
;

Bear: feedable, Actor, floatingItem
    rhetoricalturn = -999   // hack -- see yesVerb in ccr-verbs.t

    exists = true

    istame = nil        // has the bear been fed?
    isfollowing = nil   // is the bear following the player?
    wasreleased = nil   // has the bear been unchained yet?

    sdesc = "large bear"
    ldesc = {
        "The bear is extremely large, ";

        if (self.istame)
            "but appears to be friendly.";
        else
            "and seems quite ferocious!";
    }

    // DJP - We've hacked the nrmLkAround method to use actorDesc even for
    // floating objects.

    stayloc = In_Barren_Room
    actorDesc = {
        if (self.isIn(parserGetMe().location)) {
            if (self.isfollowing) {
                "You are being followed by a very
                large, tame bear.";
            }
            else if (self.istame) {
                if (not self.wasreleased and
                self.isIn(In_Barren_Room)) {

                    "There is a gentle cave bear
                    sitting placidly in one
                    corner.";
                }
                else
                    "There is a contented-looking
                    bear wandering about
                    nearby.";
            }
            else {
                "There is a ferocious cave bear
                eyeing you from the far end of the
                room!";
            }
        }
    }

    noun = 'bear'
    adjective = 'large' 'tame' 'ferocious' 'cave'

    locationOK = true       // tell compiler OK for location to be method
    location = {
        if (self.exists) {
            if (self.isfollowing)
                return toplocation(parserGetMe());
            else
                return self.stayloc;
        }
        else
            return nil;
    }

    // ATTACKING
    verDoAttack(actor) = {
        if (self.istame)
            self.onlyfriend;
        else if ((not axe.isIn(actor)) and (not sword.isIn(actor))
                 and (not singing_sword.isIn(actor)))
            self.bearhands;
    }
    doAttack(actor) = {
        // By default, attack with the axe ...
        if (axe.isIn(actor)) {
            "\n(with the axe)\n";
            self.doAttackWith(actor, axe);
        }
        // or the sword ...
        else if (sword.isIn(actor)) {
            "\n(with the sword)\n";
            self.doAttackWith(actor, sword);
        }
        // or the singing sword ...
        else if (singing_sword.isIn(actor)) {
            "\n(with the sword)\n";
            self.doAttackWith(actor, singing_sword);
        }
    }

    verDoAttackWith(actor, io) = {
        if (self.istame)
            self.onlyfriend;
    }
    doAttackWith(actor, io) = {
        //
        // If the player throws the axe at the bear, the
        // axe misses and becomes inaccessible.  (Doh!)
        //
        if (self.istame) self.onlyfriend;
        else if (io = axe) {
            if (not global.saidthrow)
                "\n(throwing the axe)\n";
            global.saidthrow := nil;
            "The axe misses and lands near the bear where
            you can't get at it.";

            axe.moveInto(self.location);
            axe.nograb := true;         // little hack
        }
        else if (io = singing_sword) {
            if (not global.saidthrow)
                "\n(throwing the sword)\n";
            global.saidthrow := nil;
            "The sword misses, bounces off the wall, and
             lands at your feet.";
            singing_sword.moveInto(self.location);
        }
        else if (io = sword) {
            if (not global.saidthrow)
                "\n(throwing the sword)\n";
            global.saidthrow := nil;
            sword.throwsmash;
        }
        else if (io = Hands)
            self.nicetry;
        else
            "You'd be better off using your bare hands than
            that thing. ";
    }

    onlyfriend = {
        "The bear is confused; he only wants to be
        %your% friend.";
    }
    bearhands = {
        "With what?  %your% bare hands?  Against *his* bear
        hands??";

        self.rhetoricalturn := global.turnsofar;
    }

    // DJP - added adventure 550 responses (player is allowed to attack
    // but bear fights back.)
    nicetry = {
        if (not global.game550)
            "Nice try, but sorry.";
        else {
            if (rand(2) = 1) {
                "The bear dodges away from your attack, growls, and
                swipes at you with his claws.  Fortunately, he
                misses. ";
            }
            else {
                 "The bear snarls, ducks away from your attack and
                 slashes you to death with his claws. ";
                 die();
            }
        }
    }

    // KICKING
    verDoKick(actor) =  {
        if (self.istame)
            self.onlyfriend;
        else
            "You obviously have not fully grasped the
            gravity of the situation.  Do get a grip on
            yourself.";
    }

    // THROWING
    verIoThrowAt(actor) = { self.verIoGiveTo(actor); }
    // DJP - fixed to stop weapons from being thrown at the tame
    // bear.  Also 'throw at' made different from 'throw to' for
    // objects which are neither food nor weapons.
    ioThrowAt(actor, dobj) = {
        if (isclass(dobj,weapon))
        {
            global.saidthrow := true;
            self.doAttackWith(actor, dobj);
        }
        else if (dobj = glass_vial) dobj.doThrowAt(actor, self);
        else if (isclass(dobj,fooditem)) self.ioGiveTo(actor, dobj);
        else self.doAttackWith(actor,dobj);
    }

    verIoThrowTo(actor) = { self.verIoGiveTo(actor); }
    ioThrowTo(actor, dobj) = {
        if (isclass(dobj,weapon))
             self.ioThrowAt(actor, dobj);
        else self.ioGiveTo(actor, dobj);
    }

    // GIVING
    ioGiveTo(actor, dobj) = {
        if(isclass(dobj,contliquid) and ((dobj.myflag = &haswater) or
        (dobj.myflag = &haswine))){
            self.doFeedWith(actor,dobj);
        }
        else if (isclass(dobj,fooditem))
            self.doFeedWith(actor,dobj);
        else {
            if (self.istame)
                "The bear doesn't seem very
                interested in %your% offer.";
            else
                "Uh-oh -- %your% offer only makes the bear
                angrier!";
        }
    }

    // FEEDING
    doFeed(actor) = {
        if (honeycomb.isReachable(parserGetMe())) {
            "(with ";honeycomb.thedesc;")\n";
            self.doFeedWith(actor,honeycomb);
        }
        else if (tasty_food.isReachable(parserGetMe())) {
            "(with ";tasty_food.thedesc;")\n";
            self.doFeedWith(actor,tasty_food);
        }
        else if (self.istame) {
            "%You% %have% nothing left to give the bear.";
        }
        else {
            "The bear seems more likely to eat *you*
            than anything you've got on you!";
        }
    }

    doFeedWith(actor, io) = {
        if(io = honeycomb){
            if(testtake(actor,io)) {
                "The bear eagerly licks up the honeycomb, after
                which he seems to calm down considerably and
                even becomes rather friendly.";
                io.moveInto(nil);
                self.istame := true;
                if(axe.location = Bear.location)
                    axe.nograb := nil;
            }
        }
        else if (io = tasty_food and global.newgame){
            "All %you% %have% are watercress sandwiches.  The bear
            is less than interested.";
        }
        else if (io = tasty_food){
            if(testtake(actor,io)) {
                "The bear eagerly wolfs down your food, after
                which he seems to calm down considerably and
                even becomes rather friendly.";
                io.moveInto(nil);
                self.istame := true;
                if(axe.location = Bear.location)
                    axe.nograb := nil;
            }
        }
        else pass doFeedWith;
    }

    // OTHER (drop/take; bear,verb)

    verDoRub(actor) = {"I don't think ";self.thedesc;
        " would care for that.";}

    verDoDrop(actor) = {
        if (not self.isfollowing)
            "The bear isn't following you.";
    }
    // special method for leave (allowing it to work even though the bear
    // is not carried)
    verDoLeave(actor) = {self.verDoDrop(actor);}
    doDrop(actor) = {
        // code for the bear attacking the troll is now with
        // Troll:
        if (Troll.isIn(parserGetMe().location)) Troll.doAttack(Bear);
        else {
            "OK, the bear is no longer following you around.";
            self.stayloc := self.location;
            self.prevloc := parserGetMe().prevloc;
            self.isfollowing := nil;
        }
    }

    verDoTake(actor) = {
        if (not self.istame)
            "Surely you're joking!";
        else if (not self.wasreleased)
            "The bear is still chained to the wall.";
        else if (self.isfollowing)
            "The bear is already following you.";
    }
    doTake(actor) = {
        self.isfollowing := true;
        "Ok, the bear's now following you around.";
    }
    moveInto(loc) = {
        local oldloc, newloc;
        oldloc := toplocation(self.location);
        newloc := toplocation(loc);
        if(oldloc <> newloc) self.prevloc := oldloc;
        self.isfollowing := nil;
        self.stayloc := loc;
    }
    actorAction( v, d, p, i) = {
        if ((v=eatVerb) and (p=nil) and (i=nil))
        return;
        if(v=takeVerb and p=nil and i=nil) return;
        if(not self.istame) {
        "You obviously have not fully grasped the
        gravity of the situation.  Do get a grip on
        yourself.";
        exit;
        }
        else if (isclass(v,travelVerb) and d=nil and p=nil and i=nil
        and v <> jumpVerb) {
            if (not self.wasreleased) {
                "The bear isn't going anywhere until you've unlocked
                the chain.";
                exit;
            }
            return;
        }
        else if (v = attackVerb and d=Troll and p=nil and i=nil) {
            return;
        }
        else if (v = followVerb and d=parserGetMe() and p=nil and i=nil) {
            return;
        }
        else if (v = stayVerb and d=nil and p=nil and i=nil) {
            return;
        }
        else {
            "The bear isn't quite sure what you want it to do.";
            exit;
        }
    }
    stay = {
        if (not self.isfollowing)
            "The bear isn't following you.";
        else {
            "OK, the bear is no longer following you around.";
            self.stayloc := self.location;
            self.prevloc := parserGetMe().prevloc;
            self.isfollowing := nil;
        }
    }
;

DeadBear: floatingdecoration,distantItem
    sdesc = "bridge wreckage and dead bear"
    ldesc = "The remains of the bridge and the smashed body of a dead bear
        are at the bottom of the chasm.  It's just as well that you
        can't examine them closely - the bear isn't a pleasant sight."
    noun = 'bear' 'bridge' 'wreckage'
    adjective = 'dead' 'wrecked' 'destroyed'
    loclist = []
;

/* The class Chaser is used for things like the Wumpus, blob, etc. that
   follow the player relentlessly. The variable "chase" indicates the
   creature's distance from the player, and is incremented by "moveinc"
   each turn the creature is active. If the player did not move, it is
   further incremented by "stayinc". The variable "ischasing" indicates
   that the creature is on the move. The routine MagicMsg is called
   when the player tries to use a magic word to escape. The routine
   "ChaseMsg" is called each turn to print messages, kill the player,
   or do other tasks that vary from one Chaser to another. The
   "backtrackAct" routine is called when the player returns to the
   previous location. Finally, the "banish" routine sends the Chaser
   away. */
Chaser: Actor

    actorDesc = {  // the move routine takes care of this.
    }

    ischasing = nil    // Until it is set off by something.

    chase = -1  // Should increase with time in the move routine.
    chasehold = 0 // Set nonzero if the chase level is to be held for
                  // a few turns
    moveinc = 1 // Value by which chase should increase
    stayinc = 0 // Extra chase increment if player stays in same room
    move = {
        local msgind,oldloc,newloc,backtrack := nil;
        oldloc := self.location;
        if(self.chasehold > 0)
            self.chasehold--;
        else
            self.chase += self.moveinc;
        if(self.chase < 1) {
            // initialize some variables
            parserGetMe().reflexmove := nil; // see below
            self.locstay := 0;    // for comments
            return;
        }
        newloc := toplocation(parserGetMe());
        if((newloc = Inside_Building and oldloc = At_Y2) or
        (newloc = In_Plover_Room and oldloc = At_Y2) or
        (newloc = Inside_Building and oldloc = In_Debris_Room) or
        (newloc = Volcano_Platform and oldloc = Fake_Y2) or
        (newloc = Fake_Y2 and oldloc = Volcano_Platform)) self.MagicMsg;
        // should explain why chaser follows, or does not. Be
        // sure to include all possible cases here...

        if((newloc = Rainbow_Room and oldloc = Inside_Building))
            self.SlippMsg;
        // self.locstay counts the turns at this location
        // parserGetMe().reflexmove indicates that the player has travelled 
        // through a looping passage to the same room.
        if(newloc <> oldloc or parserGetMe().reflexmove) self.locstay := 1;
        // Note: parserGetMe().moveInto sets self.locstay to 0 if you use a passage
        // which takes you back to the same room - the chaser is
        // presumed to have chased you along the same passage.
        else self.locstay := self.locstay + 1;
        // detect backtracking, i.e. you've gone to the previous
        // location, and you've used the same route and not used
        // a magic word or a passage going back to the same room
        // travel routes are:
        // 0 (default)
        // 1,2,3 (alternative routes e.g. low wide passage in west
        // half of Hall of Mists)
        // 10 (magic)
        // 11 (reflexive - a passage going back to the same room)
        if(newloc = self.prevloc and 
        parserGetMe().previousRoute = parserGetMe().travelRoute
        and not (parserGetMe().travelRoute >= 10))
            backtrack := true;
        // move into the new location if appropriate
        if(newloc <> self.location or parserGetMe().reflexmove)
            self.moveInto(newloc);
        // clear the reflexmove property
        parserGetMe().reflexmove := nil;
        // increment chase variable if we haven't moved
        if(self.locstay > 1) self.chase += self.stayinc;
        // act on backtracking
        if (backtrack) self.backtrackAct;
        // do the rest
        self.ChaseMsg; // Explain motion, kill player, etc.
    }
    banish() = { // Put the Chaser back to bed
        self.moveInto(nil);
        self.prevloc := nil;
        self.ischasing := nil;
        self.chase := -1;
        unnotify(self,&move);
    }
    summon(...) = { // Summon the Chaser.  This is a variable-argument
                    // method which will place the chaser in some default
                    // location if no argument is given; if an argument
                    // is given, it specifies the initial location of
                    // the chaser.  If this routine is called in
                    // a travel method, the argument should be the
                    // player's destination, not the current location.
        if(argcount > 0)
            self.moveInto(getarg(1));
        else
            self.moveInto(toplocation(parserGetMe()));
        self.ischasing := true;
        self.prevloc := nil;  // needed to avoid false BackTrack calls
        notify(self, &move, 0); self.ischasing := true;
    }
    verDoSummon(actor) = {}
    doSummon(actor) = {
        if (self.ischasing) {
            caps(); self.thedesc; " is already chasing you! ";
        }
        else
            self.summon(toplocation(actor));
    }
    verDoBanish(actor) = {
        if (not self.ischasing) {
            caps(); self.thedesc; " is not chasing you! ";
        }
    }
    doBanish(actor) = {
        "Banishing <<self.thedesc>>. ";
        self.banish;
    }
    // Default method for rubbing
    verDoRub(actor) = {"Don't be ridiculous!";}
;
