/*
** Pumping!
**
** Written by Stephen Granade for the TextFire project.
*/

#include <adv.t>
#include "bugs.t"
#include "consist.t"
#include "sysfuncs.t"

#pragma C+

die: function
{
    "\b\(*** You have killed your myocardium ***\)";
    "\bYou may restore a saved game, start over, quit, or undo
        the current command.\n";
    while ( 1 )
    {
        local resp;

	"\nPlease enter RESTORE, RESTART, QUIT, or UNDO: >";
        resp = upper(input());
        if ( resp == 'RESTORE' )
	{
	    resp = askfile( 'File to restore' );
	    if ( resp == nil ) "Restore failed. ";
	    else if ( restore( resp )) "Restore failed. ";
	    else
	    {
	        Me.location.lookAround(true);
	        scoreStatus( global.wongames, global.numgames );
		abort;
	    }
	}
        else if ( resp == 'RESTART' )
	{
	    scoreStatus( 0, 0 );
            restart();
	}
	else if ( resp == 'QUIT' || resp == 'Q' )
        {
	    terminate();
            quit();
	    abort;
        }
	else if (resp == 'UNDO')
	{
	    if (undo())
	    {
		"(Undoing one command)\b";
		Me.location.lookAround(true);
	        scoreStatus(global.wongames, global.numgames);
		abort;
	    }
	    else
		"Sorry, no undo information is available. ";
	}
    }
}

/*
 *   The scoreRank() function displays how well the player is doing.
 *   This default definition doesn't do anything aside from displaying
 *   the current and maximum scores.  Some game designers like to
 *   provide a ranking that goes with various scores ("Novice Adventurer,"
 *   "Expert," and so forth); this is the place to do so if desired.
 *
 *   Note that "global.maxscore" defines the maximum number of points
 *   possible in the game; change the property in the "global" object
 *   if necessary.
 */
scoreRank: function
{
    "You are currently pumping <<global.flow>> ml/sec. ";
}

// Pumping! initialization
pumpinginit: initialization
    preinit_phase = {
         initSearch();
    }
;

replace gameinit: function
{
    // put introductory text here
    "\bYou've never minded the stretching, the pulling, the pumping. It's
        all been part of a day's work. And a night's work, come to think of
        it.
    \bBut now...now you're under more and more stress. Tension's building,
        and something's gotta give.
    \bDoesn't matter. You've got a job to do, and you're going to keep doing
        it, come plaque or high pressure. After all, you are Joe's heart.\n";

    version.sdesc;                // display the game's name and version number

    Me.location = startroom;                // move player to initial location
    startroom.lookAround(true);             // show player where he is
    startroom.isseen = true;                // note that we've seen the room
    setscore('100 ml/sec');
    randomize();
    notify(pumpDaemon, &checkFlow, 0);
}


/*
 *   The terminate() function is called just before the game ends.  It
 *   generally displays a good-bye message.  The default version does
 *   nothing.  Note that this function is called only when the game is
 *   about to exit, NOT after dying, before a restart, or anywhere else.
 */
terminate: function
{
"\bThank you for playing the demo version of \(Pumping!\), a TextFire
release.  On-line registration for TextFire products will be available at
http://www.textfire.com on or before June 30, 1998.  Thanks for your
patience!";
}

/*
 *   The pardon() function is called any time the player enters a blank
 *   line.  The function generally just prints a message ("Speak up" or
 *   some such).  This default version just says "I beg your pardon?"
 */
pardon: function
{
    "Yes? ";
}

/*
 *   The numObj object is used to convey a number to the game whenever
 *   the player uses a number in his command.  For example, "turn dial
 *   to 621" results in an indirect object of numObj, with its "value"
 *   property set to 621.
 */
numObj: basicNumObj  // use default definition from adv.t
;

/*
 *   strObj works like numObj, but for strings.  So, a player command of
 *     type "hello" on the keyboard
 *   will result in a direct object of strObj, with its "value" property
 *   set to the string 'hello'.
 *
 *   Note that, because a string direct object is used in the save, restore,
 *   and script commands, this object must handle those commands.
 */
strObj: basicStrObj     // use default definition from adv.t
;

/*
 *   The "global" object is the dumping ground for any data items that
 *   don't fit very well into any other objects.  The properties of this
 *   object that are particularly important to the objects and functions
 *   are defined here; if you replace this object, but keep other parts
 *   of this file, be sure to include the properties defined here.
 *
 *   Note that awakeTime is set to zero; if you wish the player to start
 *   out tired, just move it up around the sleepTime value (which specifies
 *   the interval between sleeping).  The same goes for lastMealTime; move
 *   it up to around eatTime if you want the player to start out hungry.
 *   With both of these values, the player only starts getting warnings
 *   when the elapsed time (awakeTime, lastMealTime) reaches the interval
 *   (sleepTime, eatTime); the player isn't actually required to eat or
 *   sleep until several warnings have been issued.  Look at the eatDaemon
 *   and sleepDaemon functions for details of the timing.
 */
global: object
    turnsofar = 0                            // no turns have transpired so far
    score = 0                            // no points have been accumulated yet
    maxscore = 100                                    // maximum possible score
    flow = 100                           // In ml/sec
;

/*
 *   The "version" object defines, via its "sdesc" property, the name and
 *   version number of the game.  Change this to a suitable name for your
 *   game.
 */
version: object
    sdesc = "\b\(Pumping!\)\n
\nDEMO version for the TextFire 12-PACK
\n(Type \"about\" for more information)
\nCopyright (c) 1998 by TextFire, Inc. All rights reserved.
\nDeveloped with TADS, the Text Adventure Development System.\b"
;

/*
 *   "Me" is the player's actor.  Pick up the default definition, basicMe,
 *   from "adv.t".
 */
Me: basicMe
    ldesc = "You have always found your epicardium beautiful, no matter what
        that supercilious ascending aorta says. "
    verDoFix(actor) = {
        if (!pumpDaemon.vFib)
            "There's nothing wrong with you. ";
        else "That's the general idea, even if it is lacking in, oh, I don't
            know, SPECIFICS! ";
    }
;

/*
 *   darkTravel() is called whenever the player attempts to move from a dark
 *   location into another dark location.  By default, it just says "You
 *   stumble around in the dark," but it could certainly cast the player into
 *   the jaws of a grue, whatever that is...
 */
darkTravel: function
{
    "You stumble around in the dark, and don't get anywhere. ";
}

/*
 *   goToSleep - carries out the task of falling asleep.  We just display
 *   a message to this effect.
 */
goToSleep: function
{
    "***\bYou wake up some time later, feeling refreshed. ";
}

replace scoreStatus: function(points, turns)
{
    pumpDaemon.avgRate;
    if (global.flow == -1)
        setscore('Too damn fast!');
    else setscore(cvtstr(global.flow)+' ml/sec');
}

fixVerb: deepverb
    verb = 'fix'
    sdesc = "fix"
    doAction = 'Fix'
;

expandVerb: deepverb
    verb = 'expand'
    sdesc = "expand"
    action(actor) = {
        "No need--you automatically expand after each contraction. ";
    }
;

stopVerb: deepverb
    verb = 'stop'
    sdesc = "stop"
    action(actor) = {
        "Hey, you're stopped until you decide to contract again. You run the
            show around here.\b
        Mostly run the show, I guess. That brain keeps bugging you, \"Making
            sure you do your job,\" he says. Huh. Doesn't he have better
            things to worry about? ";
    }
;

modify pushVerb
    verb = 'pump' 'contract' 'compress'
    action(actor) = {
        pumpDaemon.pumpOnce;
    }
;

modify sleepVerb
    action(actor) = {
        "Right. Like the heart ever sleeps. ";
    }
;

telegram: function(msg)
{
    if (pumpDaemon.seenTelegram == nil)
        "\bYou receive a message from the brain:";
    "\b\(--TELEGRAM--\b
        From:\)\ The Brain\t\t\(To:\)\ The Heart\b
        \(Message:\)\ <<msg>>\b\(--END TELEGRAM--\) ";
    if (pumpDaemon.seenTelegram == nil) {
        pumpDaemon.seenTelegram = true;
        "\bThen the message is gone. ";
    }
}

EBS: function(msg)
{
    "\bYou hear a loud, piercing tone which echoes throughout the body.
        Just before you're ready to tear your tricuspid valve out in
        irritation, the tone stops. A message soon follows:\b
        \(--EBS--\b
        From:\)\ The Brain\t\t\(To:\)\ Everyone\b
        \(Message:\)\ <<msg>>\b\(--END EBS--\)\b
        After that, you hear nothing else. ";
}

startroom: room
    sdesc = "Pericardium"
    ldesc = "You are floating in your fibrous home, attached to Joe's
        circulatory system. Occasional vibrations send you jiggling as
        Joe shifts about. "
    allowedVerbs = [ pushVerb, againVerb, waitVerb, iVerb, lookVerb, fixVerb
        sleepVerb, inspectVerb, askVerb, expandVerb, stopVerb ]
    roomAction(a, v, d, p, i) = {
        if (v.isTravelVerb) {
            "You've never been one to travel. Not like some hearts you know.
                Like that heart of Jim's. Why, you heard that that heart was
                pulled out of his sac and placed to one side while some
                surgeons hacked on Jim.\bBrr. Bet that was cold. ";
            exit;
        }
        if (find(self.allowedVerbs, v) == nil && !v.issysverb) {
            "You're a fairly simple muscle with a fairly simple task. Don't
                get too fancy. ";
            exit;
        }
    }
;

brain: distantItem, Actor
    noun = 'brain'
    location = startroom
    sdesc = "brain"
    ldesc = "He's up there somewhere, all twisted and wrinkled. Thinks he
        runs the place. "
    actorDesc = {}
    actorAction(v, d, p, i) = {
        "Communication between the brain and you is strictly one-way. Except
            for all those nerves he's got around you. Claims there for your
            own good. Yeah, right. Little spies, they are. ";
        exit;
    }
    verDoAskAbout(actor, io) = {
        "Communication between the brain and you is strictly one-way. Except
            for all those nerves he's got around you. Claims there for your
            own good. Yeah, right. Little spies, they are. ";
    }
;

lungs: distantItem
    noun = 'lung' 'lungs'
    location = startroom
    sdesc = "lungs"
    ldesc = "Like you can see the lungs from inside your sack.\bCome to think
        of it, like you can see. "
;

arteries: fixedItem
    noun = 'artery' 'arteries'
    location = startroom
    sdesc = "arteries"
    ldesc = "They cling to you like veins, carrying the blood you pump. "
;

veins: fixedItem
    noun = 'vein' 'veins'
    location = startroom
    sdesc = "veins"
    ldesc = "They cling to you like arteries, carrying the blood you pump. "
;

blood: item
    noun = 'blood'
    location = Me
    sdesc = "blood"
    adesc = "some blood"
    thedesc = "some blood"
    ldesc = {
        "You are partially filled with blood, blood which will be sent on its
            way as soon as you contract. ";
        if (global.flow < 50)
            "That blood is looking a little too blue for your tastes. Better
                speed things up. ";
    }
    verDoPush(actor) = {}
    doPush(actor) = { pushVerb.action(actor); }
;

pumpDaemon: object
    pumpTurns = 0
    pumpList = [5, 5, 5, 5]
    fastTimes = 0
    slowTimes = 0
    seenTelegram = nil
    brainDrunk = nil
    vFib = nil
    vFibTurns = 0
    checkFlow = {
        self.pumpTurns++;
        if (self.vFib) {
            self.checkVFib;
            return;
        }
        if (self.pumpTurns >= 22) {
            "\bAll of a sudden, you stop getting any messages from the brain.
                Before you can complain to that stuck-up so-and-so you pass
                out. ";
            die();
            return;
        }
        if (self.pumpTurns == 20) {
            "\bYou feel yourself getting very woozy. ";
        }
        if (self.brainDrunk) return;
        if (self.pumpTurns == 21)
            telegram('You...really need...to...pump now...oooogh....');
        if (self.pumpTurns == 19)
            telegram('I mean, how do you think I feel? What will I say to
                Joe? "Hello, sorry to bother you, but your heart refuses to
                pump"?');
        if (self.pumpTurns == 18)
            telegram('Right. Now I\'m upset. All this time we\'ve worked
                together, and now you choose to mutiny.');
        if (self.pumpTurns == 15)
            telegram('C\'mon already! The calf muscles are whining about folic
                acid build-up, the lower intestine is in some distress,
                and the liver\'s got the gall to complain
                about--never mind. The important thing is, you need to get us
                some oxygenated blood! Now!!');
        if (self.pumpTurns == 12)
            telegram('Any day now!');
        if (self.pumpTurns == 10)
            telegram('Hello! Anyone at home? It\'s time to pump!');
        if (self.pumpTurns == 7)
            telegram('Didn\'t you hear me? Pump!');
        if (self.pumpTurns == 5)
            telegram('Pump!');
    }
    checkVFib = {
        self.vFibTurns++;
        if (self.pumpTurns == 10)
            "\bYou feel something correct itself in one of your nodes--maybe
                the sinuatrial? Hard to tell. Anyway, something's fixed. ";
        if (self.vFibTurns == 22) {
            "\bA light-headed wooziness overtakes you, and you pass out. ";
            die();
            return;
        }
        if (self.vFibTurns == 11)
            EBS('Attention! Attention! This is The Brain, speaking to you
                over the Emergency
                Broadcast System. This is not a drill. I repeat, this is
                not a drill. We are having heart trouble, and are going to
                attempt to conserve oxygenated blood until the danger is
                over. I will not be broadcasting any more past this point--hey!
                You muscles! Stop twitching like that! You want to use up
                all the--');
        if (self.vFibTurns == 10)
            telegram('Looks like I\'m going to have to do what I can to lower
                oxygen use. Good luck:\ you\'re on your own.');
        if (self.vFibTurns == 6)
            "\bThere is a rumbling, pounding noise that shakes you. Joe must
                be thumping his chest. Probably thinks that tight band across
                it is due to all those cheese fries he just scarfed down. ";
        if (self.vFibTurns == 4)
            telegram('Oh. This isn\'t good. Oh.\bWell, crap. No one ever told
                me what to do if you go bad.');
        if (self.vFibTurns == 2)
            telegram('Uh-oh. I think something\'s wrong with you. See if you
                can fix it.');
    }
    avgRate = {
        local i, len, n, rate;
        
        n = self.pumpTurns;
        len = length(self.pumpList);
        for (i = 1; i <= len; i++)
            n += self.pumpList[i];
        rate = n / 5;
        if (rate >= 13)
            global.flow = 25;
        else if (rate >= 11)
            global.flow = 33;
        else if (rate >= 10)
            global.flow = 50;
        else if (rate >= 7)
            global.flow = 75;
        else if (rate >= 4)
            global.flow = 100;
        else if (rate >= 3)
            global.flow = 200;
        else if (rate >= 2)
            global.flow = 400;
        else if (rate < 2)
            global.flow = -1;
    }
    pumpOnce = {
        local i, len;
        
        if (self.vFib) {
            self.pumpVFib;
            return;
        }
        if (global.turnsofar > 40) {
            "When you try to contract, you find yourself unable to. You flail
                around for a second, trying to get the blood to go where it's
                supposed to, but nothing happens.\bThis isn't good. ";
            self.vFib = true;
            return;
        }
        "You contract smoothly, sending blood flowing through the arteries and
            accepting new blood from the veins. ";
        if (self.pumpTurns <= 3) {
            self.fastTimes++;
            self.slowTimes = 0;
            self.brainDrunk = nil;
            if (self.fastTimes == 5) {
                "\bYou hear a faint popping sound from somewhere under your
                    epicardium. \(--TELEGRAM--\) begins the message from
                    the brain, but you are too far gone. ";
                die();
            }
            if (self.fastTimes == 4)
                telegram('I mean it! Joe\'s eyes are starting to bulge! That\'s
                    not a pretty sight, I\'d bet.');
            if (self.fastTimes == 3)
                telegram('Slow it down! You\'re going to kill us all!');
        }
        else if (self.pumpTurns > 10) {
            self.fastTimes = 0;
            self.slowTimes++;
            if (self.slowTimes == 6) {
                "\bWoozy. You feel so...very...woozy.... ";
                die();
            }
            if (self.slowTimes == 5)
                telegram('Ground control to Major Tom...hee hee....');
            if (self.slowTimes == 4) {
                telegram('Ooh...I feel so drunk...so...spacy....');
                self.brainDrunk = true;
            }
            if (self.slowTimes == 3)
                telegram('C\'mon! You\'ve got to speed it up, or we\'ll all
                    suffocate!');
        }
        else self.brainDrunk = nil;
        len = length(self.pumpList);
        for (i = 2; i <= len; i++)
            self.pumpList[i-1] = self.pumpList[i];
        self.pumpList[len] = self.pumpTurns;
        self.pumpTurns = 0;
    }
    pumpVFib = {
        if (self.pumpTurns < 10) {
            "You contract. Well, you try to contract. Instead, you feel
                yourself flailing around, your four chambers pulsing out of
                sync and getting blood nowhere. ";
            self.pumpTurns = 0;
            return;
        }
        "You pump with renewed vigor, sending blood throughout Joe's body.
            It seems your pumping troubles were temporary.\b
            Congratulations! You have finished the demo of \(Pumping!\) You
            have successfully overcome ventricular fibrillation, but have yet
            to face high blood pressure, an arterial clot, a stroke, or the
            ravages of too many pork rinds.\b";
        terminate();
        quit();
        abort;
    }
;

aboutVerb : sysverb
    verb = 'info' 'about'
    sdesc = "about"
    action(actor) = {
        "\bThe demo version of \(Pumping!\) was written by Riley Hawkins
        for TextFire, Inc.
        \b\(About the author:\)
\bAfter receiving a degree in library science from Texas Woman's
University, Riley Hawkins was directionless. Eventually, like so many
other Okies, Riley went west in search of opportunity, finally stopping in
Piedmont, CA. Unfortunately, Piedmont has a noticeable dearth of
libraries; fortunately, one of the numerous branches of the Oakland
Public Library system had a job opening.

\bA few months later, Riley met Shea Davis when Shea was doing research for
a game. In an odd series of coincidences, Deborah Keyes attended the same
PFLAG meeting Riley did, and Riley and Flip Winkler crossed paths at a
show at Slim's. Though still working at the library, Riley is now spending
more and more time writing interactive fiction.
        \b\(About TextFire:\)
        \bTextFire, Inc.\ is a partnership of authors formed for the
purpose of providing quality works of interactive fiction to a worldwide
community of players, hobbyists, and enthusiasts.  Help support our efforts
to keep interactive fiction alive! ";
        abort;
    }
;

hintVerb : sysverb
    verb = 'hint' 'hints' 'help'
    sdesc = "help"
    action(actor) = {
        "Hints are available with the full registered version of
\(Pumping!\)  Type REGISTER for additional information.";
        abort;
    }
;

registerVerb : sysverb
    verb = 'register' 'registration'
    sdesc = "register"
    action(actor) = {
        "On-line registration for TextFire products will be available at
http://www.textfire.com on or before June 30, 1998.  Thanks for your
patience!";
        abort;
    }
;

modify quitVerb
    verb = 'q'
;

modify creditsVerb
    action(actor) = { aboutVerb.action(actor); }
;
