! Standard single precision
!
! 15 14  7 6      0  15           0
! +-+-----+-------+  +------------+
! |S| Exp |  Frac |  |    Frac    |
! +-+-----+-------+  +------------+
!
! S Exp Frac
! x  0  0         +/- Zero
! x  0  non-0     Subnormal number              +/- 0.frac x 2^(-126)
! x FF  0         +/- Infinity
! x FF  non-0     NaN (signalling if Frac MSB is clear)
!  others         Normal number                 +/- 1.frac x 2^(exp-127)

!
! Single Extended precision
!
! 15      10      0  15 14        0  15          0
! +-+----+--------+  +-+----------+  +-----------+
! |S|zero|  Exp   |  |J|  Frac    |  |   Frac    |
! +-+----+--------+  +-+----------+  +-----------+
!
! S Exp J Frac
! x  0  0 0       +/- Zero
! x  0  0 non-0   Subnormal number              +/- 0.frac x 2^(-1023)
! x 7FF 0 0       +/- Infinity
! x 7FF 0 non-0   NaN (signalling if Frac MSB is clear)
! x  x  1 x       Normal number (exp<7FF)       +/- 1.frac x 2^(exp-1023)
!  others         illegal

!
! Packed decimal (BCD)
!
!   12  8  4  0    12  8  4  0    12  8  4  0
! +--+--+--+--+  +--+--+--+--+  +--+--+--+--+ 
! | S|e0|e1|d0|  |d1|d2|d3|d4|  |d5|d6|d7|d8| 
! +--+--+--+--+  +--+--+--+--+  +--+--+--+--+ 
!
! BCD packed decimal (capable of accurately storing a single-precision number)
! Top bit of S is sign of number, next bit is sign of exponent
!
!   ee       ddddddddd
!    0           0           +/- Zero
! [0-99]  [1-9.99999999]     Finite number       +/- d.dddddddd x 10^(+/- ee)
!   FF           0           +/- Infinity
!   FF        non-zero       NaN (signalling if MSB of d0 is clear)
!      others                Illegal

! Note about NaNs - internally generated NaNs contain a small number giving
! a reason for the NaN - this is stored as an integer aligned at the bottom of
! a single-precision number, or 8 bits above the bottom of extended-precision,
! so it survives format conversion. In single -> packed decimal conversion,
! the 22 arbitrary bits of NaN are converted to BCD as an integer, and placed
! in d2-d8. In packed decimal -> single conversion, d0 (apart from the quiet
! bit) and d1 are ignored, and d2-d8 is converted back to binary (modulo 2^22).
! (Signalling NaNs cannot be converted between formats - the Invalid Operation
! exception is raised, and the result is a quiet NaN).


! ----------------------------------------------------------------------------
!  Useful routine: unsigned comparison (for addresses in Z-machine)
!    Returns 1 if x>y, 0 if x=y, -1 if x<y
! ----------------------------------------------------------------------------

Zcharacter terminating 133;

Replace UnsignedCompare;

[ UnsignedCompare x y;
 ! Yup - hand-crafted assembler. We need the speed, and Inform won't
 ! compile ?rtrue
  @jl x 0 ?~l1;      ! if x < 0 && y>=0 return 1;
  @jl y 0 ?~rtrue;
 .l1;
  @jl x 0 ?l2;       ! if x>=0 && y<0 return -1;
  @jl y 0 ?l3;
 .l2;
  @jg x y ?rtrue;    ! if x>y return 1;
  @jl x y ?~rfalse;  ! if (~~x<y) return 0;
 .l3;
  @ret $ffff;        ! return -1;
];

![ UnsignedCompare x y;
!  if (x<0 && y>=0) return 1;
!  if (x>=0 && y<0) return -1;
!  if (x>y) return 1;
!  if (x<y) return -1;
!  return 0;
!];

[ _mul32 dest ah bh
         al bl
         r1 r2 r3 r4
         rl rh tmp c;
  !print "_mul32(", (hex) ah, ",", (hex) bh, ")=";
  ! Split a and b into two 8-bit halves
  @and ah $ff -> al;
  @log_shift ah $fff8 -> ah;
  @and bh $ff -> bl;
  @log_shift bh $fff8 -> bh;
  ! Four 8x8->16 multiplies
  r1 = ah * bh;
  r2 = al * bh;
  r3 = ah * bl;
  r4 = al * bl;

  ! Add r3 into (r1,r4)  
  @log_shift r3 8 -> tmp;
  rl = r4 + tmp;
  c = UnsignedCompare(rl, r4) < 0;
  @log_shift r3 $fff8 -> tmp;
  rh = r1 + tmp + c;
  
  ! Add r2 into (rh,rl)
  @log_shift r2 8 -> tmp;
  rl = rl + tmp;
  c = UnsignedCompare(rl, tmp) < 0;
  @log_shift r2 $fff8 -> tmp;
  rh = rh + tmp + c;
  
  !print (hex) rh, (hex) rl; new_line;
  
  dest-->0 = rh;
  dest-->1 = rl;
];

Constant FE_INVALID    = $01;
Constant FE_DIVBYZERO  = $02;
Constant FE_OVERFLOW   = $04;
Constant FE_UNDERFLOW  = $08;
Constant FE_INEXACT    = $10;
Constant FE_ALL_EXCEPT = $1F;

Constant FE_TRAP_INVALID    = $01;
Constant FE_TRAP_DIVBYZERO  = $02;
Constant FE_TRAP_OVERFLOW   = $04;
Constant FE_TRAP_UNDERFLOW  = $08;
Constant FE_TRAP_INEXACT    = $10;
Constant FE_TRAP_ALL        = $1F;

Constant INXE = $1000;
Constant UFLE = $0800;
Constant OFLE = $0400;
Constant DVZE = $0200;
Constant IVOE = $0100;
Constant INX = $10;
Constant UFL = $08;
Constant OFL = $04;
Constant DVZ = $02;
Constant IVO = $01;

Object fptrap
  with inx_handler [; print "[** Floating-point trap: Inexact **]^"; quit; ],
       ufl_handler [; print "[** Floating-point trap: Underflow **]^"; quit; ],
       ofl_handler [; print "[** Floating-point trap: Overflow **]^"; quit; ],
       ivo_handler [; print "[** Floating-point trap: Invalid operation **]^"; quit; ],
       dvz_handler [; print "[** Floating-point trap: Division by zero **]^"; quit; ];
       
Global fpstatus;

Constant FE_TONEAREST = 1;
Constant FE_UPWARD = 2;
Constant FE_DOWNWARD = 3;
Constant FE_TOWARDZERO = 4;

Global fprounding = FE_TONEAREST;

Constant NanReason_SigNan    0;
Constant NanReason_MagSubInf 4;
Constant NaNReason_InfTimes0 5;
Constant NaNReason_0TimesInf 6;
Constant NaNReason_0Div0     7;
Constant NaNReason_InfDivInf 8;

Array _fpscratch --> 6;

Global _precision;
Global _dest;

Constant FE_PRINT_G = 0;
Constant FE_PRINT_E = 1;
Constant FE_PRINT_F = 2;
Global fpprintmode = FE_PRINT_G;
Global fpprintwidth = 0;
Global fpprintprecision = 6;

Constant FCMP_U = $8;
Constant FCMP_L = $4;
Constant FCMP_E = $2;
Constant FCMP_G = $1;

[ feclearexcept excepts;
  excepts = excepts & FE_ALL_EXCEPT;
  fpstatus = fpstatus &~ excepts;
];

[ fetestexcept excepts;
  excepts = excepts & FE_ALL_EXCEPT;
  return fpstatus & excepts;
];

[ fegetround round;
  return fprounding;
];

[ fesetround round;
  fprounding = round;
];

[ feholdexcept envp;
  fpstatus = 0;
];

[ fetesttrap traps;
  traps = traps & FE_TRAP_ALL;
  @log_shift fpstatus $fff8 -> sp;
  @and sp traps -> sp;
  @ret_popped;
];

[ feenabletrap traps;
  traps = traps & FE_TRAP_ALL;
  @log_shift traps 8 -> sp;
  @or fpstatus sp -> fpstatus;
];   

[ fedisabletrap traps;
  traps = traps & FE_TRAP_ALL;
  !@log_shift traps 8 -> sp;
  !@not sp -> sp;  XXX argh - @not nonfunctional on some compilers
  !@and fpstatus sp -> fpstatus;
  @log_shift traps 8 -> traps;
  fpstatus = fpstatus &~ traps;
];   

[ fpenabletraps x;
  x = x & $1f;
  @log_shift x 8 -> x;
  fpstatus = fpstatus | x;
];

[ fpdisabletraps x;
  x = x & $1f;
  @log_shift x 8 -> x;
  fpstatus = fpstatus &~ x;
];

[ isnan x h;
  h = x-->0;
  if ((h & $7f80) == $7f80 && ((h & $7f) | x-->1) ~= 0)
    rtrue;
  else
    rfalse;
];

[ isnanx x;
  if ((x-->0 & $07ff) == $07ff && (x-->1 | x-->2) ~= 0)
    rtrue;
  else
    rfalse;
];

[ _isnan sex mhi mlo;
  if ((sex & $07ff) == $7ff && (mhi | mlo) ~= 0)
    rtrue;
  else
    rfalse;
];

[ issignalling x h;
  h = x-->0;
  if ((h & $7fc0) == $7f80 && ((h & $7f) | x-->1) ~= 0)
    rtrue;
  else
    rfalse;
];

[ issignallingx x h;
  h = x-->1;
  if ((x-->0 & $07ff) == $07ff && ((h & $7fff) | x-->2) ~= 0 && (h & $4000) == 0)
    rtrue;
  else
    rfalse;
];

[ _issignalling sex mhi mlo;
  if ((sex & $07ff) == $7ff && (mhi | mlo) ~= 0 && (mhi & $4000) == 0)
    rtrue;
  else
    rfalse;
];

[ isinf x;
  if ((x-->0 & $7fff) == $7f80 && x-->1 == 0)
    return 1;
  else
    return 0;
];

[ isinfx x;
  if ((x-->0 & $07ff) == $07ff && (x-->1|x-->2) == 0)
    return 1;
  else
    return 0;
];

[ fcmp_common OP1 OP2
              OP1emh OP2emh OP1sxm OP2sxm OP1mlo OP2mlo;
  OP1sxm = OP1-->0;
  OP1mlo = OP1-->1;
  OP2sxm = OP2-->0 + $8000;
  OP2mlo = OP2-->1;
  OP1emh = OP1sxm & $7FFF;
  OP2emh = OP2sxm & $7FFF;
  if (OP1emh>OP2emh) jump op1bigger;
  if (OP1emh<OP2emh) jump op2bigger;
  if (OP1mlo>OP2mlo) jump op1bigger;
  if (OP1mlo==OP2mlo) jump equalmag;
 .op2bigger;
  if (OP2sxm>=0) return FCMP_G; else return FCMP_L;
 .op1bigger;
  if (OP1sxm>=0) return FCMP_G; else return FCMP_L;
 .equalmag;
  if ((OP1sxm<0 && OP2sxm>=0) ||
      (OP1sxm>=0 && OP2sxm<0) ||
      (OP1emh|OP1mlo)==0)
    return FCMP_E;
  if (OP1sxm>=0) return FCMP_G; else return FCMP_L;
];

[ fcmpe OP1 OP2;
  if (isnan(OP1) || isnan(OP2))
  {
    ! IVO exception
    if (fpstatus & IVOE)
      fptrap.ivo_handler();
    else
      fpstatus = fpstatus | IVO;
    return FCMP_U;
  }
  return fcmp_common(OP1,OP2);
];

[ fcmp OP1 OP2
       OP1emh OP2emh OP1sxm OP2sxm OP1mlo OP2mlo;
  if (isnan(OP1) || isnan(OP2))
  {
    if (issignalling(OP1) || issignalling(OP2))
    {
      ! IVO exception
      if (fpstatus & IVOE)
        fptrap.ivo_handler();
      else
        fpstatus = fpstatus | IVO;
    }
    return FCMP_U;
  }
  return fcmp_common(OP1,OP2);
];

[ fcmpx_common OP1 OP2
               OP1exp OP2exp OP1sex OP2sex OP1mhi OP2mhi OP1mlo OP2mlo;
  OP1sex = OP1-->0;
  OP1mhi = OP1-->1;
  OP1mlo = OP1-->2;
  OP2sex = OP2-->0 + $8000;
  OP2mhi = OP2-->1;
  OP2mlo = OP2-->2;
  OP1exp = OP1sex & $7FFF;
  OP2exp = OP2sex & $7FFF;
  if (OP1exp>OP2exp) jump op1bigger;
  if (OP1exp<OP2exp) jump op2bigger;
  if (OP1mhi>OP2mhi) jump op1bigger;
  if (OP1mhi<OP2mhi) jump op2bigger;
  if (OP1mlo>OP2mlo) jump op1bigger;
  if (OP1mlo==OP2mlo) jump equalmag;
 .op2bigger;
  if (OP2sex>=0) return FCMP_G; else return FCMP_L;
 .op1bigger;
  if (OP1sex>=0) return FCMP_G; else return FCMP_L;
 .equalmag;
  if ((OP1sex<0 && OP2sex>=0) ||
      (OP1sex>=0 && OP2sex<0) ||
      (OP1exp|OP1mhi|OP1mlo)==0)
    return FCMP_E;
  if (OP1sex>=0) return FCMP_G; else return FCMP_L;
];

[ fcmpex OP1 OP2;
  if (isnanx(OP1) || isnanx(OP2))
  {
    ! IVO exception
    if (fpstatus & IVOE)
      fptrap.ivo_handler();
    else
      fpstatus = fpstatus | IVO;
    return FCMP_U;
  }
  return fcmpx_common(OP1,OP2);
];

[ fcmpx OP1 OP2
        OP1emh OP2emh OP1sxm OP2sxm OP1mlo OP2mlo;
  if (isnanx(OP1) || isnanx(OP2))
  {
    if (issignallingx(OP1) || issignallingx(OP2))
    {
      ! IVO exception
      if (fpstatus & IVOE)
        fptrap.ivo_handler();
      else
        fpstatus = fpstatus | IVO;
    }
    return FCMP_U;
  }
  return fcmpx_common(OP1,OP2);
];

[ fmax dest OP1 OP2
       cmp;
  cmp = fcmp(OP1,OP2);
  if (cmp & (FCMP_E|FCMP_G)) jump ret1;
  if (cmp & FCMP_L) jump ret2;
  if (isnan(OP1)) jump ret2;
 .ret1; @copy_table OP1 dest 4; return;
 .ret2; @copy_table OP2 dest 4; return;
];

[ fmaxx dest OP1 OP2
        cmp;
  cmp = fcmpx(OP1,OP2);
  if (cmp & (FCMP_E|FCMP_G)) jump ret1;
  if (cmp & FCMP_L) jump ret2;
  if (isnanx(OP1)) jump ret2;
 .ret1; @copy_table OP1 dest 6; return;
 .ret2; @copy_table OP2 dest 6; return;
];

[ fmin dest OP1 OP2
       cmp;
  cmp = fcmp(OP1,OP2);
  if (cmp & (FCMP_E|FCMP_L)) jump ret1;
  if (cmp & FCMP_G) jump ret2;
  if (isnan(OP1)) jump ret2;
 .ret1; @copy_table OP1 dest 4; return;
 .ret2; @copy_table OP2 dest 4; return;
];

[ fminx dest OP1 OP2
        cmp;
  cmp = fcmpx(OP1,OP2);
  if (cmp & (FCMP_E|FCMP_L)) jump ret1;
  if (cmp & FCMP_G) jump ret2;
  if (isnanx(OP1)) jump ret2;
 .ret1; @copy_table OP1 dest 6; return;
 .ret2; @copy_table OP2 dest 6; return;
];

[ fstoi src;
  fstox(_fpscratch, src);
  return fxtoi(_fpscratch);
];

[ fxtoi src
        sgn exp mhi mlo tmp res grs dir;
  print "fxtoi(", (fhexx) src, ")^";
  exp = src-->0;
  mhi = src-->1;
  mlo = src-->2;
  sgn = exp & $8000;
  exp = exp & $07ff;
  ! Want to slide the binary point to the bottom
  tmp = 1023 + 31 - exp;
  print "tmp=", tmp, "^";
  if (tmp <= 0) 
    jump outofrange;

  _precision = 1;
  res = _Denorm(mhi, mlo, tmp);
  mhi = res-->0;
  mlo = res-->1;
  grs = res-->2;
  _RoundNum(sgn, exp, mhi, mlo, grs);
  mhi = res-->0;
  mlo = res-->1;
  exp = res-->2;
  dir = res-->4;
  ! Now have a 32-bit number in mhi, mlo
  if (sgn < 0)
  {
    ! 2's complement it
    mhi = ~mhi;
    mlo = -mlo;
    if (mlo == 0) ++mhi;
  }
  if ((mlo >= 0 && mhi ~= 0) ||
      (mlo < 0 && mhi ~= -1))
    jump outofrange;
  
  if (dir)
  {
    if (fpstatus & INXE)
      fptrap.inx_handler(); !Need params
    else
      fpstatus = fpstatus | INX;
  }
  
  return mlo;
  
 .outofrange;
  if (sgn < 0)
    mlo = $8000;
  else
    mlo = $7fff;
  
  if (fpstatus & IVOE)
    return fptrap.ivo_handler(); !Need params
  else
    fpstatus = fpstatus | IVO;
    
  return mlo;
];

[ fstox dst src internal
        sign exp mhi mlo res;
  mhi = src-->0;
  mlo = src-->1;
  !print "fstox(", (hex) mhi, (hex) mlo, ")=";
  exp = (mhi & $7f80);
  @log_shift exp $fff9 -> exp;
  sign = mhi & $8000;
  mhi = mhi & $7f;
  
  @log_shift mhi 8 -> mhi;
  @log_shift mlo $fff8 -> sp;
  @or mhi sp -> mhi;
  @log_shift mlo 8 -> mlo;
  
  if (exp == 0)
  {
    if (mhi | mlo)
    {
      ! Subnormal
      exp = 1023-126;
      if (~~internal)
      {
        res = _Normalise(exp, mhi, mlo);
        exp = res-->0;
        mhi = res-->1;
        mlo = res-->2;
      }
    }
    else
    {
      ! Zero
    }
  }
  else
  {
    if (exp == $ff)
    {
      ! Infinite or NaN
      exp = $7ff;
      if (mhi | mlo)
      {
        if ((~~internal) && (mhi & $4000)==0)
        {
          ! Conversion of signalling NaN
          print "Invalid operation^";
          if (fpstatus & IVOE)
            fptrap.ivo_handler(); ! Need operands - what about return
          else
            fpstatus = fpstatus | IVO;
          mhi = $4000;
          mlo = NaNReason_SigNaN * 256;
        }
      }
    }
    else
    {
      ! Normal
      mhi = mhi | $8000;
      exp = exp + (1023 - 127);
    }
  }
  
  dst-->0 = sign | exp;
  dst-->1 = mhi;
  dst-->2 = mlo;
 ! print (hex) dst-->0, "|", (hex) dst-->1, (hex) dst-->2, "^";  
];

[ fxtos dst src
        sign exp sex mhi mlo res;
  sex = src-->0;
  mhi = src-->1;
  mlo = src-->2;
  !print "fstox(", (hex) mhi, (hex) mlo, ")=";
  exp = sex & $07ff;
  sign = sex & $8000;

  _dest = dst;
  _precision = 0;
  _RoundResult(sign, exp, mhi, mlo, 0);
];

[ fcpy dst src;
  @copy_table src dst 4;
];

[ fcpyx dst src;
  @copy_table src dst 6;
];

[ fneg dst src;
  dst-->1 = src-->1;
  dst-->0 = src-->0 + $8000;
];

[ fnegx dst src;
  dst-->2 = src-->2;
  dst-->1 = src-->1;
  dst-->0 = src-->0 + $8000;
];

[ fabs dst src;
  dst-->1 = src-->1;
  dst-->0 = src-->0 & $7fff;
];

[ fabsx dst src;
  dst-->2 = src-->2;
  dst-->1 = src-->1;
  dst-->0 = src-->0 & $7fff;
];

[ _Denorm h l s
          w b t1 grs;
  !print "Denormalising ", (hex) h, (hex) l, " by ", s, " bits^";
  @log_shift s $fffc -> w;    ! words to shift
  b = s & $f;                 ! bits to shift
  t1 = 16 - b;
  b = -b;
  @log_shift l t1 -> grs;     ! bottom b bits of l into grs
  @log_shift l b -> l;        ! shift l down b bits
  @log_shift h t1 -> t1;      ! bottom b bits of h into l
  l = l | t1;
  @log_shift h b -> h;	      ! shift h down b bits
  if (w == 1 || w >= 3)
  {
      @log_shift grs 2 -> t1;
      grs = grs | t1;
      @log_shift grs $fffe -> t1;
      grs = l | t1;
      l = h;
      h = 0;
  }
  if (w >= 2)
  {
      grs = grs | l;
      @log_shift grs 2 -> t1;
      grs = grs | t1;
      @log_shift grs $fffe -> t1;
      grs = h | t1;
      l = 0;
      h = 0;
  }
  
 ! print "Result is ", (hex) h, (hex) l, "/", (hex) grs; new_line;
  
  _fpscratch-->0=h;
  _fpscratch-->1=l;
  _fpscratch-->2=grs;
  
  return _fpscratch;
];

! Normalise (mhi,mlo), by shifting bits up such that the MSB of mhi is set.
! Returns adjusted (possibly negative) exponent. (mhi,mlo) may be zero or
! already normal (in which case no change is made).
[ _Normalise exp mhi mlo
               tmp;
 ! print "Normalising ", (hex) mhi, (hex) mlo, (char) '/', exp; new_line;
  if (mhi == 0)
  {
    if (mlo == 0) jump out;
    mhi = mlo;
    mlo = 0;
    exp = exp - 16;
  }
  if ((mhi & $ff00) == 0)
  {
    @log_shift mhi 8 -> mhi;
    tmp = 8;
  }
  if ((mhi & $f000) == 0)
  {
    @log_shift mhi 4 -> mhi;
    tmp = tmp + 4;
  }
  if ((mhi & $c000) == 0)
  {
    @log_shift mhi 2 -> mhi;
    tmp = tmp + 2;
  }
  if (mhi >= 0)
  {
    mhi = mhi + mhi;
    tmp ++;
  }
  @sub tmp 16 -> sp;
  @log_shift mlo sp -> sp;
  @or mhi sp -> mhi;
  @log_shift mlo tmp -> mlo;
  exp = exp - tmp;
 ! print "Result is ", (hex) mhi, (hex) mlo, (char) '/', exp; new_line;
 .out;
  _fpscratch-->0 = exp;
  _fpscratch-->1 = mhi;
  _fpscratch-->2 = mlo;
  return _fpscratch;
];

[ _RoundNum RNDsgn RNDexp RNDmhi RNDmlo RNDgrs RNDdir
            dir lsb;
  
  if (_precision == 0)
  {
    @log_shift RNDgrs 8 -> sp;
    @or RNDgrs sp -> RNDgrs;
    @log_shift RNDgrs $fff8 -> sp;
    @log_shift RNDmlo 8 -> sp;
    @or sp sp -> RNDgrs;
    RNDmlo = RNDmlo & $ff00;
    lsb = $0100;
  }
  else
    lsb = $0001;
    
  switch (fprounding)
  {
     FE_TONEAREST:
       !print "To nearest^";
       if (RNDgrs < 0) ! round (top) bit set
       {
         !print "Round bit set^";
         if (RNDgrs ~= $8000) ! sticky bits set
           dir = 1;
         else ! halfway case
         {
           !print "Halfway case^";
           if (RNDdir < 0 || (RNDdir == 0 && (RNDmlo & lsb)))
             dir = 1;
           else
             dir = -1;
         }
       }
       else if (RNDgrs > 0)
         dir = -1;
       
     FE_DOWNWARD:
       if (RNDgrs)
         dir = -RNDsgn;
         
     FE_UPWARD:
       if (RNDgrs)
         dir = RNDsgn;
         
     FE_TOWARDZERO:
       if (RNDgrs)
         dir = -1;
  }
  
  !if (dir > 0)
  !  print "Rounding up^";
  !else if (dir < 0)
  !  print "Rounding down^";
  !else
  !  print "Exact^";
    
  if (dir > 0)
  {
    RNDmlo = RNDmlo + lsb;
    if (RNDmlo == 0)
    {
      if (++RNDmhi == $0) ! Mantissa overflow
      {
        RNDmhi = $8000;
        RNDexp++;
      }
    }
  }
  
  ! Update rounding so far
  if (dir) RNDdir = dir;
  
  _fpscratch-->0 = RNDmhi;
  _fpscratch-->1 = RNDmlo;
  _fpscratch-->2 = RNDexp;
  _fpscratch-->3 = dir;       ! direction of this rounding
  _fpscratch-->4 = RNDdir;
  return _fpscratch;
];

[ _ReturnResult sex mhi mlo tmp tmp2;
  if (_precision == 1)
  {
    _dest-->0 = sex;
    _dest-->1 = mhi;
    _dest-->2 = mlo;
  }
  else
  {
    tmp = sex & $ff;
    @log_shift tmp 7 -> tmp;
    mhi = mhi & $7fff;
    @log_shift mhi $fff8 -> tmp2;
    _dest-->0 = (sex & $8000) | tmp | tmp2;
    @log_shift mhi 8 -> mhi;
    @log_shift mlo $fff8 -> mlo;
    _dest-->1 = mhi | mlo;
  }
];

! "Exact", normalised result provided, as extended number split into 5 parts.
! Round it to destination precision, then check for over/underflow.
! Denormalise if necessary, and store.
[ _RoundResult RNDsgn RNDexp RNDmhi RNDmlo RNDgrs RNDdir
               ExpMin ExpMax BiasAdjust
               res;
  
  !print "_RoundResult(",RNDsgn, (hex)RNDexp, " ";
  !print (hex) RNDmhi, (hex) RNDmlo, "|", (hex) RNDgrs, (char) ')'; new_line;
  res=_RoundNum(RNDsgn, RNDexp, RNDmhi, RNDmlo, RNDgrs, RNDdir);
  RNDmhi = res-->0;
  RNDmlo = res-->1;
  RNDexp = res-->2;
  RNDdir = res-->4;
  
  ! Rebias exponent to destination format
  if (_precision == 0)
  {
    RNDexp = RNDexp + 127 - 1023;
    ExpMin = $0001;
    ExpMax = $00fe;
    BiasAdjust = 192;
  }
  else
  {
    ExpMin = $0000;
    ExpMax = $07fe;
    BiasAdjust = 1536;
  }
  
  if (RNDexp < ExpMin || RNDexp > ExpMax)
  {
  !  print "Exponent out of range^";
    if (RNDexp < ExpMin)
    {
      ! Potential underflow
      if (RNDmhi | RNDmlo)
      {
   !     print "Tinyness^";
        if (fpstatus & UFLE)
        {
          ! Take underflow trap
    !      print "Underflow trap^";
          if (RNDexp + BiasAdjust < ExpMin)
          {
            ! Massive underflow
     !       print "Massive underflow^";
            if (_precision == 0)
              BiasAdjust = 127 - RNDexp;
            else
              BiasAdjust = 1023 - RNDexp;
          }
          _ReturnResult(RNDsgn | (RNDexp + BiasAdjust), RNDmhi, RNDmlo);
          fptrap.ufl_handler(_precision, _dest, -BiasAdjust);
          return;
        }
        else
        {
          res = _Denorm(RNDmhi, Rndmlo, ExpMin - RNDexp);
          RNDmhi = res-->0;
          RNDmlo = res-->1;
          RNDgrs = res-->2;
          RNDexp = 0;
          res = _RoundNum(RNDsgn, RNDexp, RNDmhi, RNDmlo, RNDgrs, RNDdir);
          RNDmhi = res-->0;
          RNDmlo = res-->1;
          RNDdir = res-->4;
          ! Check it didn't round back up to be normalised
          if (RNDmhi < 0) RNDexp = ExpMin;
          if (res-->3) ! Denormalisation loss
          {
            fpstatus = fpstatus | UFL;
      !      print "Underflow (denormalisation loss)^";
          }
        }
      }
      else
        RNDexp = 0;
    }
    else ! RNDexp > $FE
    {
     ! print "Overflow^";
      if (fpstatus & OFLE)
      {
      !  print "Overflow trap^";
        if (RNDexp - BiasAdjust > ExpMax)
        {
          ! Massive underflow
       !   print "Massive overflow^";
          if (_precision == 0)
            BiasAdjust = RNDexp - 127;
          else
            BiasAdjust = RNDexp - 1023;
        }
        _ReturnResult(RNDsgn | (RNDexp - BiasAdjust), RNDmhi, RNDmlo);
        fptrap.ofl_handler(_precision, _dest, BiasAdjust);
        return;
      }
      else
      {
        fpstatus = fpstatus | OFL;
        if (fprounding == FE_TONEAREST ||
            (fprounding == FE_UPWARD && RNDsgn >= 0) ||
            (fprounding == FE_DOWNWARD && RNDsgn < 0))
        {
          RNDexp = $7ff;
          RNDmhi = 0;
          RNDmlo = 0;
          RNDdir = 1;
        }
        else
        {
          RNDexp = ExpMax;
          RNDmhi = $ffff;
          RNDmlo = $ffff;
          RNDdir = -1;
        }
      }
    }
  }
  
  !print (hex) RNDexp, (hex) RNDmhi, (hex) RNDmlo; new_line;
  
  _ReturnResult(RNDsgn | RNDexp, RNDmhi, RNDmlo);

  if (RNDdir ~= 0)
  {
  !  print "Inexact^";
    if (fpstatus & INXE)
      fptrap.inx_handler(_precision, _dest);
    else
      fpstatus = fpstatus | INX;
  }    
  
];

[ fadd dest OP1 OP2;
  faddsub(0, dest, OP1, OP2);
];

[ faddx dest OP1 OP2;
  faddsubx(0, dest, OP1, OP2);
];

[ fsub dest OP1 OP2;
  faddsub($8000, dest, OP1, OP2);
];

[ fsubx dest OP1 OP2;
  faddsubx($8000, dest, OP1, OP2);
];

[ faddsub op dest OP1 OP2;
  
  if (isnan(OP1) || isnan(OP2))
  {
    if (issignalling(OP1) || issignalling(OP2))
    {
  !    print "Invalid operation^";
      dest-->0 = $7fc0;
      dest-->1 = NaNReason_SigNaN;
    }
    else if (isnan(OP1))
      @copy_table dest OP1 4;
    else
      @copy_table dest OP2 4;
    return;
  }
  
  fstox(_fpscratch+0, OP1, 1);
  fstox(_fpscratch+6, OP2, 1);
  
  _precision = 0;
  _dest = dest;
  _faddsub(      _fpscratch-->0, _fpscratch-->1, _fpscratch-->2,
              op+_fpscratch-->3, _fpscratch-->4, _fpscratch-->5);
];
  
[ faddsubx op dest OP1 OP2;
  
  if (isnanx(OP1) || isnanx(OP2))
  {
    if (issignallingx(OP1) || issignallingx(OP2))
    {
      print "Invalid operation^";
      dest-->0 = $07ff;
      dest-->1 = $4000;
      dest-->2 = NaNReason_SigNaN*256;
    }
    else if (isnanx(OP1))
      @copy_table dest OP1 6;
    else
      @copy_table dest OP2 6;
    return;
  }
  
  _precision = 1;
  _dest = dest;
  _faddsub(      OP1-->0, OP1-->1, OP1-->2,
              op+OP2-->0, OP2-->1, OP2-->2);
];
  
[ _faddsub OP1sex OP1mhi OP1mlo OP2sex OP2mhi OP2mlo
           RNDgrs RNDexp tmp tmp2 OP1exp OP2exp;
  
  OP1exp = OP1sex & $07ff;
  OP2exp = OP2sex & $07ff;
  
  if (OP1exp == $7ff || OP2exp == $7ff)
  {
    print "Summation of infinities^";
    if (OP2exp ~= $7ff)  ! infinity + finite
    { _ReturnResult(OP1sex, OP1mhi, OP1mlo); return; }

    if (OP1exp ~= $7ff)  ! finite + infinity
    { _ReturnResult(OP2sex, OP2mhi, OP2mlo); return; }
    
    ! two infinities 
    if ((OP1sex & $8000) == (OP2sex & $8000))
    { _ReturnResult(OP1sex, OP1mhi, OP1mlo); return; }
    else
    {
      print "Invalid operation^";
      fpstatus = fpstatus | IVO;
      _ReturnResult($07ff, $4000, NaNReason_MagSubInf*256);
      return;
    }
  }
  
  ! We let zeros and subnormals through. Algorithm will work
  ! fine, but we may end up with a subnormal result.
  
  
  ! Denormalise the smaller operand to the same exponent
  ! as the larger.
  if (OP1exp < OP2exp)
  {
    RNDexp = OP2exp;
    tmp = _Denorm(OP1mhi, OP1mlo, OP2exp - OP1exp);
    OP1mhi = tmp-->0;
    OP1mlo = tmp-->1;
    tmp = tmp-->2;
    tmp2 = 0;    
  }
  else if (OP1exp > OP2exp)
  {
    RNDexp = OP1exp;
    tmp = _Denorm(OP2mhi, OP2mlo, OP1exp - OP2exp);
    OP2mhi = tmp-->0;
    OP2mlo = tmp-->1;
    tmp2 = tmp-->2;
    tmp = 0;
  }
  else
  {
    RNDexp = OP1exp;
    tmp = tmp2 = 0;
  }
  
  ! Don't need original numbers any longer
  OP1sex = OP1sex & $8000;
  OP2sex = OP2sex & $8000;
  
  ! Now OP1sex/OP2sex = signs of OP1/OP2
  !     OP1mhi/OP1mlo = operand 1 mantissa
  !     RNDexp = prospective result exponent
  !     OP2mhi/OP2mlo = operand 2 mantissa
  !     tmp = operand 1 guard, round and sticky bits
  !     tmp2 = operand 2 guard, round and sticky bits
  if (OP1sex == OP2sex)
  {
    ! summation case
    !print "Summing^";
    !font off;
    !print "  ", (hex) OP1mhi, (hex) OP1mlo, (char) '/', (hex) tmp, " (", RNDexp, ")^";
    !print "+ ", (hex) OP2mhi, (hex) OP2mlo, (char) '/', (hex) tmp2, "^";
    !print "  -------------^";
    RNDgrs= tmp + tmp2; ! no carry possible - one of these is 0
    OP1mlo = OP1mlo + OP2mlo;
    tmp = UnsignedCompare(OP1mlo, OP2mlo) < 0; ! get carry
    tmp2 = OP1mhi + OP2mhi + tmp;
    tmp = (OP1mhi < 0 && OP2mhi < 0) ||
          (OP1mhi < 0 && tmp2 >= 0) ||
          (OP2mhi < 0 && tmp2 >= 0);
    OP1mhi = tmp2;
          
 !   print " ", tmp, (hex) OP1mhi, (hex) OP1mlo, (char) '/', (hex) RNDgrs,
 !                      " (", RNDexp, ")^";
    
    if (tmp)
    {
      @log_shift RNDgrs 1 -> tmp;
      RNDgrs = RNDgrs | tmp;
      @log_shift RNDgrs $ffff -> RNDgrs;
      if (OP1mlo & 1) RNDgrs = RNDgrs | $8000;
      @log_shift OP1mlo $ffff -> OP1mlo;
      if (OP1mhi & 1) OP1mlo = OP1mlo | $8000;
      @log_shift OP1mhi $ffff -> OP1mhi;
      OP1mhi = OP1mhi | $8000;
      RNDexp = RNDexp + 1;
    !  print "  ", (hex) OP1mhi, (hex) OP1mlo, (char) '/', (hex) RNDgrs,
    !                   " (", RNDexp, ")^";
    }
    !font on;
  }
  else
  {
    !print "Difference^";
    !font off;
    !print "  ", (hex) OP1mhi, (hex) OP1mlo, (char) '/', (hex) tmp, " (", RNDexp, ")^";
    !print "- ", (hex) OP2mhi, (hex) OP2mlo, (char) '/', (hex) tmp2, "^";
    !print "  -------------^";
    RNDgrs = tmp - tmp2;
    tmp = UnsignedCompare(tmp, tmp2) >= 0; ! Carry
    tmp2 = OP1mlo - OP2mlo + tmp - 1;
    tmp = (OP1mlo < 0 && OP2mlo >= 0) ||
          (OP1mlo < 0 && tmp2 >= 0) ||
          (OP2mlo >= 0 && tmp2 >= 0);
    OP1mlo = tmp2;
    tmp2 = OP1mhi - OP2mhi + tmp - 1;
    tmp = (OP1mhi < 0 && OP2mhi >= 0) ||
          (OP1mhi < 0 && tmp2 >= 0) ||
          (OP2mhi >= 0 && tmp2 >= 0);
    OP1mhi = tmp2;
  !  print " ", 1-tmp, (hex) OP1mhi, (hex) OP1mlo, (char) '/', (hex) RNDgrs,
  !                     " (", RNDexp, ")^";
    ! If it came out negative, reverse the sign and mantissa
    if (~~tmp)
    {
      OP1sex = OP1sex + $8000;
      RNDgrs = -RNDgrs; tmp = RNDgrs == 0;
      tmp2 = -OP1mlo + tmp - 1;
      tmp = (OP1mlo >= 0 && tmp2 >= 0);
      OP1mlo = tmp2;
      OP1mhi = -OP1mhi + tmp - 1;
   !   print "N ", (hex) OP1mhi, (hex) OP1mlo, (char) '/', (hex) RNDgrs,
   !                      " (", RNDexp, ")^";
    }
    if (OP1mhi >= 0)
    {
      ! Need to normalise. Try a single bit at first, bringing the guard
      ! bit back into the mantissa.
      OP1mhi = OP1mhi + OP1mhi;
      if (OP1mlo < 0) OP1mhi++;
      OP1mlo = OP1mlo + OP1mlo;
      if (RNDgrs < 0) OP1mlo++;
      RNDgrs = RNDgrs + RNDgrs;
      RNDexp--;
    !  print "  ", (hex) OP1mhi, (hex) OP1mlo, (char) '/', (hex) RNDgrs,
    !                     " (", RNDexp, ")^";
      ! If still not normalised, exponent difference must have been 0 or 1,
      ! so round and sticky bits are both zero. Will normalise below (in
      ! the same code that clears up for subnormal operands).
      if ((OP1mhi | OP1mlo) == 0)
      {
        ! Zero result - sign determined by rounding mode
        if (fprounding == FE_DOWNWARD) OP1sex = $8000; else OP1sex = 0;
        RNDexp = 0;
      }
    }
  }
  
  if (OP1mhi >= 0 && (OP1mhi | OP1mlo) ~= 0)
  {
    ! Subnormal result
    if (RNDgrs ~= 0) print "[** Internal error: faddsub - RNDgrs @126= 0 **]";
    tmp = _Normalise(RNDexp, OP1mhi, OP1mlo);
    RNDexp = tmp-->0;
    OP1mhi = tmp-->1;
    OP1mlo = tmp-->2;
   ! print "  ", (hex) OP1mhi, (hex) OP1mlo, (char) '/', (hex) RNDgrs,
   !                    " (", RNDexp, ")^";
  }
  !font on;

  _RoundResult(OP1sex, RNDexp, OP1mhi, OP1mlo, RNDgrs);
];

[ fmul dest OP1 OP2;
  
  if (isnan(OP1) || isnan(OP2))
  {
    if (issignalling(OP1) || issignalling(OP2))
    {
  !    print "Invalid operation^";
      dest-->0 = $07ff;
      dest-->1 = $4000;
      dest-->2 = NaNReason_SigNaN*256;
      if (fpstatus & IVOE)
        fptrap.ivo_handler(); ! Need args
      else
        fpstatus = fpstatus | IVO;
    }
    else if (isnan(OP1))
      @copy_table dest OP1 4;
    else
      @copy_table dest OP2 4;
    return;
  }
  
  fstox(_fpscratch+0, OP1, 1);
  fstox(_fpscratch+6, OP2, 1);
  
  _precision = 0;
  _dest = dest;
  _fmul(_fpscratch-->0, _fpscratch-->1, _fpscratch-->2,
        _fpscratch-->3, _fpscratch-->4, _fpscratch-->5);
];

[ fmulx dest OP1 OP2;
  
  if (isnanx(OP1) || isnanx(OP2))
  {
    if (issignallingx(OP1) || issignallingx(OP2))
    {
   !   print "Invalid operation^";
      dest-->0 = $07ff;
      dest-->1 = $4000;
      dest-->2 = NaNReason_SigNaN*256;
      if (fpstatus & IVOE)
        fptrap.ivo_handler(); ! Need args
      else
        fpstatus = fpstatus | IVO;
    }
    else if (isnanx(OP1))
      @copy_table dest OP1 6;
    else
      @copy_table dest OP2 6;
    return;
  }
  
  _precision = 1;
  _dest = dest;
  _fmul(OP1-->0, OP1-->1, OP1-->2,
        OP2-->0, OP2-->1, OP2-->2);
];

[ _fmul OP1sex OP1mhi OP1mlo OP2sex OP2mhi OP2mlo
        tmp tmp2 r1 r2 r3 r4 t1 t2 c;

  ! Extract exponents
  tmp = OP1sex & $07ff;
  tmp2 = OP2sex & $07ff;
  
  ! Work out sign
  OP1sex = OP1sex & $8000;
  OP2sex = OP2sex & $8000;
  if (OP1sex ~= OP2sex)
    OP1sex = $8000;
  else
    OP1sex = 0;
    
  if (tmp == $7ff || tmp2 == $7ff)
  {
    ! Multiplication by infinity
    if ((tmp2 == 0 && (OP2mhi|OP2mlo) == 0) ||
        (tmp == 0 && (OP1mhi|OP1mlo) == 0))
    {
    !  print "Infinity times zero^";
      if (fpstatus & IVOE)
        fptrap.ivo_handler(); ! XXX need params
      else
        fpstatus = fpstatus | IVO;
      if (tmp2 == 0)
        tmp = NaNReason_InfTimes0*256;
      else
        tmp = NaNReason_0TimesInf*256;
      _ReturnResult($07ff, $4000, tmp);
      return;
    }
    ! Return correctly signed infinity
    _ReturnResult(OP1sex | $07ff, $0000, $0000);
    return;
  }
  
  if (tmp == 0)
  {
    r1 = _Normalise(tmp, OP1mhi, OP1mlo);
    tmp = r1-->0;
    OP1mhi = r1-->1;
    OP1mlo = r1-->2;
  }
  
  if (tmp2 == 0)
  {
    r1 = _Normalise(tmp2, OP2mhi, OP2mlo);
    tmp2 = r1-->0;
    OP2mhi = r1-->1;
    OP2mlo = r1-->2;
  }
  
  if ((OP1mhi&OP2mhi) == 0)
    jump multzeros;
  
  OP2sex = tmp + tmp2 - 1022;
  
 ! print (hex) OP1mhi, (hex) OP1mlo, " x ", (hex) OP2mhi, (hex) OP2mlo;
 ! new_line;
  ! OP1mhi * OP2mhi -> (r1,r2)
  _mul32(_fpscratch, OP1mhi, OP2mhi);
  r1 = _fpscratch-->0;
  r2 = _fpscratch-->1;
  if (OP1mlo ~= 0 && OP2mlo ~= 0)
  {
    ! OP1mlo * OP2mlo -> (r3,r4)
    _mul32(_fpscratch, OP1mlo, OP2mlo);
    r3 = _fpscratch-->0;
    r4 = _fpscratch-->1;
  }
  if (OP2mlo ~= 0)
  {
    ! OP1mhi * OP2mlo -> (t1, t2)
    _mul32(_fpscratch, OP1mhi, OP2mlo);
    t1 = _fpscratch-->0;
    t2 = _fpscratch-->1;
    ! Add ( 0, t1, t2,  0)
    !  to (r1, r2, r3, r4)
    r3 = r3 + t2;
    c = UnsignedCompare(r3, t2) < 0;
    tmp = r2 + t1 + c;
    if ((r2 < 0 && t1 < 0) || (r2 < 0 && tmp >= 0) || (t1 < 0 && tmp >= 0))
      r1++;
    r2 = tmp;
  }
  if (OP1mlo ~= 0)
  {
    ! OP1mlo * OP2mhi -> (t1, t2)
    _mul32(_fpscratch, OP1mlo, OP2mhi);
    t1 = _fpscratch-->0;
    t2 = _fpscratch-->1;
    ! Add ( 0, t1, t2,  0)
    !  to (r1, r2, r3, r4)
    r3 = r3 + t2;
    c = UnsignedCompare(r3, t2) < 0;
    tmp = r2 + t1 + c;
    if ((r2 < 0 && t1 < 0) || (r2 < 0 && tmp >= 0) || (t1 < 0 && tmp >= 0))
      r1++;
    r2 = tmp;
  }
 ! font off;  
  !print "  ", (hex) r1, (hex) r2, (hex) r3, (hex) r4, " (", OP2sex, ")^";
  
  ! Make up guard, round and sticky bits:
  @log_shift r4 2 -> sp;
  @or r4 sp -> r4;
  @log_shift r4 $fffe -> sp;
  @or r3 sp -> r3;
  
  !print "  ", (hex) r1, (hex) r2, "|", (hex) r3, " (", OP2sex, ")^";
  
  if (r1 >= 0)
  {
    ! Renormalise, recovering the guard bit.
    r1 = r1 + r1;
    if (r2 < 0) ++r1;
    r2 = r2 + r2;
    if (r3 < 0) ++r2;
    r3 = r3 + r3;
    --OP2sex;
   ! print "  ", (hex) r1, (hex) r2, "|", (hex) r3, " (", OP2sex, ")^";
  }
 ! font on;
  
  _RoundResult(OP1sex, OP2sex, r1, r2, r3);
  return;
 .multzeros;
  _RoundResult(OP1sex);
];

[ fdiv dest OP1 OP2;
  
  if (isnan(OP1) || isnan(OP2))
  {
    if (issignalling(OP1) || issignalling(OP2))
    {
  !    print "Invalid operation^";
      dest-->0 = $07ff;
      dest-->1 = $4000;
      dest-->2 = NaNReason_SigNaN*256;
      if (fpstatus & IVOE)
        fptrap.ivo_handler(); ! Need args
      else
        fpstatus = fpstatus | IVO;
    }
    else if (isnan(OP1))
      @copy_table dest OP1 4;
    else
      @copy_table dest OP2 4;
    return;
  }
  
  fstox(_fpscratch+0, OP1, 1);
  fstox(_fpscratch+6, OP2, 1);
  
  _precision = 0;
  _dest = dest;
  _fdiv(_fpscratch-->0, _fpscratch-->1, _fpscratch-->2,
        _fpscratch-->3, _fpscratch-->4, _fpscratch-->5);
];

[ fdivx dest OP1 OP2;
  
  if (isnanx(OP1) || isnanx(OP2))
  {
    if (issignallingx(OP1) || issignallingx(OP2))
    {
      print "Invalid operation^";
      dest-->0 = $07ff;
      dest-->1 = $4000;
      dest-->2 = NaNReason_SigNaN*256;
      if (fpstatus & IVOE)
        fptrap.ivo_handler(); ! Need args
      else
        fpstatus = fpstatus | IVO;
    }
    else if (isnanx(OP1))
      @copy_table dest OP1 6;
    else
      @copy_table dest OP2 6;
    return;
  }
  
  _precision = 1;
  _dest = dest;
  _fdiv(OP1-->0, OP1-->1, OP1-->2,
        OP2-->0, OP2-->1, OP2-->2);
];

[ _fdiv OP1sex OP1mhi OP1mlo OP2sex OP2mhi OP2mlo
        c RNDexp Qmhi Qmmi Qmlo bits reqbits;

  ! Extract exponents
  Qmhi = OP1sex & $07ff;
  Qmlo = OP2sex & $07ff;
  
  ! Work out sign
  OP1sex = OP1sex & $8000;
  OP2sex = OP2sex & $8000;
  if (OP1sex ~= OP2sex)
    OP1sex = $8000;
  else
    OP1sex = 0;
    
  if (Qmhi == $7ff || Qmlo == $7ff)
  {
    if (Qmlo ~= $7ff)
    {
      ! Infinity / x
      ! Return correctly signed infinity
      _ReturnResult(OP1sex | $07ff, $0000, $0000);
    }
    else if (Qmhi ~= $7ff)
    {
      ! x / Infinity
      ! Return correctly signed zero
      _ReturnResult(OP1sex, $0000, $0000);
    }
    else
    {
      ! Infinity / Infinity
      if (fpstatus & IVOE)
        fptrap.ivo_handler(); ! XXX need params
      else
        fpstatus = fpstatus | IVO;
      _ReturnResult(OP1sex | $07ff, $4000, NaNReason_InfDivInf*256);
    }
    return;
  }
  
  if ((OP1mhi|OP1mlo)==0)
  {
    if ((OP2mhi|OP2mlo)==0)
    {
      ! Zero / Zero
      if (fpstatus & IVOE)
        fptrap.ivo_handler(); ! XXX need params
      else
        fpstatus = fpstatus | IVO;
      _ReturnResult(OP1sex | $07ff, $4000, NaNReason_0Div0*256);
    }
    else
      ! Zero / X
      _ReturnResult(OP1sex, $0000, $0000);
    return;
  }
  
  if ((OP2mhi|OP2mlo)==0)
  {
    ! X / 0
    if (fpstatus & DVZE)
      fptrap.dvz_handler(); ! XXX need params
    else
      fpstatus = fpstatus | DVZ;
    _ReturnResult(OP1sex | $07ff, $0000, $0000);
    return;
  }
  
  if (Qmhi == 0)
  {
    c = _Normalise(Qmhi, OP1mhi, OP1mlo);
    Qmhi = c-->0;
    OP1mhi = c-->1;
    OP1mlo = c-->2;
  }
  
  if (Qmlo == 0)
  {
    c = _Normalise(Qmlo, OP2mhi, OP2mlo);
    Qmlo = c-->0;
    OP2mhi = c-->1;
    OP2mlo = c-->2;
  }
  
  ! Prospective exponent
  RNDexp = Qmhi - Qmlo + 1023;
  
  ! A basic long division algorithm.
  ! (Qmhi,Qmmi,Qmlo) will be the quotient
  ! (c,OP1mhi,OP1mlo) is the dividend (c using 1 bit only)
  if (_precision == 0)
    reqbits = 24 + 2; ! + 2 for guard + round
  else
    reqbits = 32 + 2;
    
  c=0;
  Qmhi=0;
  Qmmi=0;
  Qmlo=0;
  
!  font off;
  for (bits = 0: bits < reqbits && (c|OP1mhi|OP1mlo): bits++)
  {
  !  print c, (hex) OP1mhi, (hex) OP1mlo, "   ", (hex) OP2mhi, (hex) OP2mlo;
    Qmhi = Qmhi + Qmhi; if (Qmmi < 0) ++Qmhi;
    Qmmi = Qmmi + Qmmi; if (Qmlo < 0) ++Qmmi;
    Qmlo = Qmlo + Qmlo;
    if (c || UnsignedCompare(OP2mhi, OP1mhi) < 0 ||
             (OP2mhi == OP1mhi && UnsignedCompare(OP2mlo, OP1mlo) <= 0))
    {
      Qmlo = Qmlo | 1;
      c = UnsignedCompare(OP1mlo, OP2mlo) >= 0;
      OP1mlo = OP1mlo - OP2mlo;
      OP1mhi = OP1mhi - OP2mhi + c - 1;
      c = 0;
    }
 !   print "   ", (hex) Qmhi, (hex) Qmmi, (hex) Qmlo; new_line;
    c = OP1mhi < 0;
    OP1mhi = OP1mhi + OP1mhi; if (OP1mlo<0) ++OP1mhi;
    OP1mlo = OP1mlo + OP1mlo;
  }
  bits = 48 - bits;
  while (bits >= 16)
  {
    Qmhi = Qmmi;
    Qmmi = Qmlo;
    Qmlo = 0;
    bits = bits-16;
  }
  
  reqbits = bits-16;
  @log_shift Qmhi bits -> sp;
  @log_shift Qmmi reqbits -> sp;
  @or sp sp -> Qmhi;
  @log_shift Qmmi bits -> sp;
  @log_shift Qmlo reqbits -> sp;
  @or sp sp -> Qmmi;
  @log_shift Qmlo bits -> Qmlo;
!  print "   ", (hex) Qmhi, (hex) Qmmi, (hex) Qmlo; new_line;
 ! font on;
  
  if (c|OP1mhi|OP1mlo)
    Qmlo = Qmlo | 1;
  
  if (Qmhi>=0)
  {
    Qmhi = Qmhi + Qmhi; if (Qmmi<0) ++Qmhi;
    Qmmi = Qmmi + Qmmi; if (Qmlo<0) ++Qmmi;
    Qmlo = Qmlo + Qmlo;
    --RNDexp;
  }
  _RoundResult(OP1sex, RNDexp, Qmhi, Qmmi, Qmlo);
];

[ hex x y;
  y = (x & $7F00) / $100;
  if (x & $8000) y = y + $80;
  x = x & $FF;
  hexdigit(y / $10);
  hexdigit(y);
  hexdigit(x / $10);
  hexdigit(x);
];

[ hexdigit x;
  x = x & $F;
  switch (x) {
    0 to 9: print x;
    10 to 15: print (char) 'A'-10+x;
  }
];

[ fraw x;
  print (hex) x-->0, (hex) x-->1;
];

[ frawx x;
  print (hex) x-->0, (char) '|', (hex) x-->1, (hex) x-->2;
];

[ fhex x;
  fstox(_fpscratch, x, 1);
  fhexx(_fpscratch);
];

[ fhexx x
       exp mhi mlo tmp;
  exp = x-->0;
  mhi = x-->1;
  mlo = x-->2;
  
  if (exp < 0)
  {
    print (char) '-';
    exp = exp - $8000;
  }
  exp = exp & $7ff;
  if (exp == $7ff)
  {
     if ((mhi | mlo) == 0)
         print "Infinity";
     else
     {
       print "NaN";
       if ((mhi & $4000) == 0) print (char) 'S';
       mhi = mhi & $3fff;
       ! Rearrange 8 extra bits of extra precision to come out at the top
       @log_shift mlo 8 -> sp;
       @log_shift sp $fffe -> sp;
       @log_shift mlo $fff8 -> sp;
       @log_shift mhi 8 -> sp;
       @or sp sp -> mlo;
       @log_shift mhi $fff8 -> sp;
       @or sp sp -> mhi;
       print "($";
       x = 0;
       do
       {
         @log_shift mhi $fff4 -> tmp;
         if (x || tmp)
         {
           x = 1;
           print (hexdigit) tmp;
         }
         @log_shift mhi 4 -> sp;
         @log_shift mlo $fff4 -> sp;
         @or sp sp -> mhi;
         @log_shift mlo 4 -> mlo;
         @log_shift mhi $fff4 -> tmp;
       } until ((mhi|mlo)==0);
       if (~~x)
         print (char) '0';
       print (char) ')';
     }
         
     return;
  }
  @log_shift mhi $fff4 -> tmp;
  print (char) '$', (hexdigit) tmp;
  mhi = mhi & $0fff;
  if (mhi | mlo)
  {
    print (char) '.';
    @log_shift mhi $fff8 -> tmp;
    print (hexdigit) tmp;
    mhi = mhi & $00ff;
    if (mhi | mlo)
    {     
      @log_shift mhi $fffc -> tmp;      
      print (hexdigit) tmp;
      mhi = mhi & $000f;
      if (mhi | mlo)
      {
        print (hexdigit) mhi;
        if (mlo)
        {
          @log_shift mlo $fff4 -> tmp;
          print (hexdigit) tmp;
          mlo = mlo & $0fff;
          if (mlo)
          {
            @log_shift mlo $fff8 -> tmp;
            print (hexdigit) tmp;
            mlo = mlo & $00ff;
            if (mlo)
            {
              @log_shift mlo $fffc -> tmp;
              print (hexdigit) tmp;
              mlo = mlo & $000f;
              if (mlo)
                print (hexdigit) mlo;
            }
          }
        }
      }
    }
    exp = exp - 1023 - 3;
  }
  else
    exp = 0;
  print (char) 'P', exp;
];

[ fp x
     sxm mhi mlo tmp exp i first last dp sig up wantexp;
  fstop(fpdigF0, x);
  sxm = fpdigF0-->0;
  mhi = fpdigF0-->1;
  mlo = fpdigF0-->2;
  
  exp = (sxm & $ff0);
  @log_shift exp $fffc -> exp;
  
  ! No attempt to handle field width (yet?)
  
  if (exp == $ff)
  {
    if (sxm<0) print (char) '-';
    if ((mhi|mlo|(sxm & $f))==0)
      print "Infinity";
    else
      print "NaN";
    return;
  }
  
  ! Turn exponent back into decimal
  @log_shift exp $fffc -> sp;
  @mul sp 10 -> sp;
  @and exp $f -> sp;
  @add sp sp -> exp;
  if (sxm & $4000) exp = -exp;
  
  ! Turn 9 digits into ZSCII
  for (i=8, tmp=mlo: i>=5: i--)
  {
    _fpscratch->i = '0' + (tmp & $f);
    @log_shift tmp $fffc -> tmp;
  }
  for (tmp=mhi: i>=1: i--)
  {
    _fpscratch->i = '0' + (tmp & $f);
    @log_shift tmp $fffc -> tmp;
  }
  _fpscratch->0 = '0' + (sxm & $f);
  
  ! Find how many significant digits we have after the decimal point
  for (sig=8: sig>0: sig--)
  {
    if (_fpscratch->sig ~= '0') break;
  }
  ++sig;

 .reposition;  
  ! Decide what the first and last digits we want are
  switch (fpprintmode)
  {
    FE_PRINT_E:
      first = 0;
      dp = 1;
      last = fpprintprecision;
      wantexp = true;
    FE_PRINT_F:
      if (exp >= 0) first = 0; else first = exp;
      dp = 1+exp;
      last = fpprintprecision + exp;
      wantexp = false;
    FE_PRINT_G:
      tmp = fpprintprecision;
      if (fpprintprecision<=0) tmp=1;
      if (exp < -4 || exp >= tmp)
      {
        first = 0;
        dp = 1;
        last = tmp-1;
        if (last > sig-1) last = sig-1;
        wantexp = true;
      }
      else
      {
        if (exp >= 0) first = 0; else first = exp;
        dp = 1+exp;
        last = tmp-1;
        if (last > sig-1 && last > dp-1)
        {
          if (sig > dp)
            last = sig-1;
          else
            last = dp-1;
        }
        wantexp = false;
      }
  }
  
  !print "first=", first, " last=", last, " sig=", sig, " dp=", dp; new_line;
  if (last < sig-1)
  {
    ! trailing (non-zero) digits beyond last one we're printing
    ! we need to round again
    i = _fpscratch->(last+1);
    sig = last+1;
    tmp = 0;
    switch (fprounding)
    {
      FE_TONEAREST:
        if (i > '5' ||
            (i == '5' && sig > (last+2)) ||
            (i == '5' && (_fpscratch->last & 1)))
          tmp = 1;
      FE_UPWARD:
        tmp = 1;
      FE_TOWARDZERO:
        if (sxm < 0) tmp = 1;
    }
    if (tmp)
    {
      ! Round up - add one, looping to do carries
      for (i=last: i>=0: i--)
      {
        if (++(_fpscratch->i) == '9'+1)
        {
          _fpscratch->i = '0';
          sig = i;
        }
        else
          break;
      }
      if (i<0)
      {
        ! Whoops - rounded right up
        _fpscratch->0 = '1';
        sig = 1;
        exp++;
      }
    }
    else
    {
      ! Round down - just check trailing zeros again
      for (i=last: i>=1: i--)
      {
        if (_fpscratch->i == '0')
          sig = i;
        else
          break;
      }
    }
    ! Think again about what we're printing
    jump reposition;
  }

  ! XXX should we print -0?
  if (sxm < 0) print (char) '-';
  
  for (i=first: i<=last: i++)
  {
    if (i==dp)
      print (char) '.';
    if (i>=0 && i<sig)
      print (char) _fpscratch->i;
    else
      print (char) '0';
  }
  
  if (wantexp)
  {
    print (char) 'E', exp;
  }
];

Array X_Ten --> $0402 $A000 $0000;

! Table look-up of powers of ten up to 10^45.
[ _GetPowerOfTen dest power
                 a b c n s;
  n = power;
  s = 0;
  
  ! Halve n until it is in the range of the table
  while (n > 13)
  {
    @log_shift n $ffff -> n;
    ++s;
  }
  
  ! Table of powers of ten - contains all exactly representable powers
  switch (n)
  {
     0: a = $03FF; b = $8000;
     1: a = $0402; b = $A000;
     2: a = $0405; b = $C800;
     3: a = $0408; b = $FA00;
     4: a = $040C; b = $9C40;
     5: a = $040F; b = $C350;
     6: a = $0412; b = $F424;
     7: a = $0416; b = $9896; c = $8000; 
     8: a = $0419; b = $BEBC; c = $2000; 
     9: a = $041C; b = $EE6B; c = $2800;
    10: a = $0420; b = $9502; c = $F900; 
    11: a = $0423; b = $BA43; c = $B740;
    12: a = $0426; b = $E8D4; c = $A510; 
    13: a = $042A; b = $9184; c = $E72A;
  }
  dest-->0 = a;
  dest-->1 = b;
  dest-->2 = c;
  while (s > 0)
  {
    ! Square result so far
    fmulx(dest, dest, dest);
    ! Check next bit of power
    --s;
    @sub 0 s -> sp;
    @log_shift power sp -> n;
    if (n & 1)
      fmulx(dest, dest, X_Ten);
  }
];

!#IfDef HAAA;
Array fpdigF0 --> 3;
Array fpdigF1 --> 3;

[ _fstop_naninf dst sex mhi mlo
                digits dhi dlo tmp tmp2;
  sex = (sex & $8000) | $0ff0;
  if (mhi & $4000)
  {
    sex = sex | $0008; ! Quiet bit for NaN
    mhi = mhi & $3fff;
  }
  !print "Rearranging InfNan: ", (hex)mhi, (hex)mlo; new_line;
  ! Infinity/NaN
  ! Should trap signalling NaNs - currently caught by fstox()
  !
  ! We do actually convert the bits of the NaN (or indeed infinity) into
  ! BCD. We are actually converting a single-precision NaN, and don't
  ! want the bottom 8 bits. So it's a conversion of 22 bits -> 7 digits.
  @log_shift mlo $fff8 -> sp;
  @log_shift mhi 8 -> sp;
  @or sp sp -> mlo;
  @log_shift mhi $fff8 -> mhi;
  !print "Rearranged InfNan: ", (hex)mhi, (hex)mlo; new_line;
  ! Oh gawd, don't ask. This is a binary->decimal
  ! conversion of (mhi,mlo) -> (dhi,dlo). Each step of
  ! the loop divides (mhi,mlo) by ten, by using an approximation
  ! 1/10 = 4/5 * 1/8 ~= $0.CCCCCCCC * 1/8
  ! m = $0.CCCCCCCC * i is approximated by j = i - (i>>2), k = j + (j>>4),
  ! l = k + (k>>8), m = l + (l>>16)
  ! This approvidation gives i DIV 10 <= (m>>3) <= i DIV 10 + 15, and
  ! we just check the remainder at the end.
  
  for (digits=0: digits<7: digits++)
  {
    tmp2 = mlo;
    !print "Digit ", digits; new_line;
    if (mhi ~= 0 || mlo < 0)
    {
    !  print "i=", (hex) mhi, (hex) mlo; new_line;
  
      @log_shift mlo $fffe -> sp;
      @log_shift mhi 14 -> sp;
      @or sp sp -> tmp;
      @log_shift mhi $fffe -> sp;
      @sub mhi sp -> mhi;
      if (UnsignedCompare(mlo, tmp) < 0) --mhi;
      mlo = mlo - tmp;
    !  print "j=", (hex) mhi, (hex) mlo; new_line;
      
      @log_shift mlo $fffc -> sp;
      @log_shift mhi 12 -> sp;
      @or sp sp -> tmp;
      mlo = mlo + tmp;
      @log_shift mhi $fffc -> sp;
      @add mhi sp -> mhi;
      if (UnsignedCompare(mlo, tmp) < 0) ++mhi;      
    !  print "k=", (hex) mhi, (hex) mlo; new_line;
      
      @log_shift mlo $fff8 -> sp;
      @log_shift mhi 8 -> sp;
      @or sp sp -> tmp;
      mlo = mlo + tmp;
      @log_shift mhi $fff8 -> sp;
      @add mhi sp -> mhi;
      if (UnsignedCompare(mlo, tmp) < 0) ++mhi;
    !  print "l=", (hex) mhi, (hex) mlo; new_line;
  
      mlo = mlo + mhi;
      if (UnsignedCompare(mlo, mhi) < 0) ++mhi;
     ! print "m=", (hex) mhi, (hex) mlo; new_line;
      
      @log_shift mlo $fffd -> mlo;
      @log_shift mhi 13 -> sp;
      @or mlo sp -> mlo;
      @log_shift mhi $fffd -> mhi;
    !  print "m>>3=", (hex) mhi, (hex) mlo; new_line;
      
      @log_shift mlo 2 -> sp;
      @add mlo sp -> tmp;
      @log_shift tmp 1 -> tmp;
      tmp = tmp2 - tmp;
    !  print "remainder=", tmp; new_line;
      if (tmp >= 10)
      {
        tmp = tmp - 10;
        if (++mlo==0) ++mhi;
      }
    }
    else
    {
      tmp = mlo % 10;
      mlo = mlo / 10;
    }
    
    @log_shift dlo $fffc -> dlo;
    @log_shift dhi 12 -> sp;
    @or dlo sp -> dlo;
    @log_shift dhi $fffc -> dhi;
    @log_shift tmp 8 -> sp;
    @or dhi sp -> dhi;
   ! print "squirreled=", (hex)dhi, (hex)dlo; new_line;
  }
  
 ! print"Converted decimal = ", (hex) dhi, (hex) dlo; new_line;
  dst-->0 = sex;
  dst-->1 = dhi;
  dst-->2 = dlo;
];

! This is hard
! Binary -> decimal conversion. We only provide single-precision conversions,
! as required by the standard, using extended precision to make it work.
!
! The destination format is BCD:  $SEEM $MMMM $MMMM  representing
! <+/->M.MMMMMMMM x 10^(<+/->EE), M and E being BCD, top bit of S being the
! sign of the number, next bit of S being the sign of the exponent.
[ fstop dst src
            sex mhi mlo exp arith tmp tmp2 tmp3 tmp4
            inexact digits grs c;
  fstox(fpdigF0, src);
  sex = fpdigF0-->0;
  mhi = fpdigF0-->1;
  mlo = fpdigF0-->2;
  exp = sex & $07ff;
  
 ! print "fstop(", (hex) sex, "|", (hex) mhi, (hex) mlo, ")^";
  
  if (exp == $7ff)
  {
    _fstop_naninf(dst, sex, mhi, mlo);
    return;
  }
  
  if ((mhi | mlo) == 0)
  {
     sex = sex & $8000;
     jump done;
  }
  
  ! Now have a normalised (originally single precision) number, in
  ! extended form. exp is in the range 1-23+(1023-127) to +254+(1023-127)
  ! = $36A to $47E. We now add one to the exponent (so the mantissa
  ! lies within [1/2 .. 1), and remove the bias.
  
  arith = exp - 1023 + 1;
  ! arith is now in the range [-148 .. +128]. We need to
  ! make it a decimal exponent. This needs a logarithm, but we'll start off
  ! with an approximation that can only be off by +1.
  !
  ! We know:
  !
  !   2^(arith-1) <= value < 2^arith
  !
  ! Taking base-10 logarithms:
  !
  !   (arith-1)*log(2) <= log(value) < arith*log(2)
  !
  ! Let log2lo and log2hi be slightly too low and high approximations to log(2).
  !
  !   if (arith > 0):  (arith-1)*log2lo <= log(value) < arith*log2hi
  !   if (arith <= 0): (arith-1)*log2hi <= log(value) < arith*log2lo
  !
  ! Let D = log2hi-log2lo:
  !
  !   if (arith > 0)
  !      arith*log2hi - arith*D - log2lo    <= log(value) < arith*log2hi
  !   if (arith <= 0)
  !      arith*log2lo - (-arith*D) - log2hi <= log(value) < arith*log2lo
  !
  ! Then, provided that log2lo and log2hi are such that (128*D+log2lo) <= 1
  ! and (148*D+log2hi) <= 1:
  !
  !   if (arith > 0)
  !      floor(arith*log2hi) - 1 <= floor(log(value)) <= floor(arith*log2hi)
  !   if (arith <= 0)
  !      floor(arith*log2lo) - 1 <= floor(log(value)) <= floor(arith*log2lo)
  !
  ! Which gives us the desired bounds.
  !
  ! The conditions are satisfied as long as D <= 2^(-8), but we want as
  ! much accuracy as we can get without overflowing a 16-bit multiplication.
  ! We can afford to set D to 2^(-9) giving us 8 bits of accuracy in log2,
  ! and 7 bits of accuracy in arith, leading to a 15-bit result.
  !
  ! So we choose log2lo = 154 * 2^(-9) = ~ 0.30078     (log(2) = ~0.30103)
  !              log2hi = 155 * 2^(-9) = ~ 0.30273
  if (arith > 0)
    tmp2 = 155; ! 2^9 * log2hi
  else
    tmp2 = 154; ! 2^9 * log2lo
  tmp2 = arith * tmp2;
  @art_shift tmp2 $fff7 -> arith;
 ! print "approximate exponent=", arith; new_line;
  
  ! Now arith-1 <= floor(log(value)) = base-10 exponent <= arith
  if (arith >= 0)
  {
    tmp = arith;
    if (arith == 0)
      jump expadjustdone;
  }
  else
    tmp = -arith;
  
  ! We now need to multiply the original value by 10^(-arith) to get
  ! the correct decimal mantissa.
  
  ! We'll use some FP - remember status, and disable exceptions
  tmp2 = fpstatus;
  tmp3 = fprounding;
  
  fpstatus = 0;
  fprounding = FE_TONEAREST;
  
  _GetPowerOfTen(fpdigF1, tmp);
  
  if (arith >= 0)
    fdivx(fpdigF0, fpdigF0, fpdigF1);
  else
    fmulx(fpdigF0, fpdigF0, fpdigF1);
  
  ! Check inexact (either in 10^tmp, or multiplication/division)
  inexact = fpstatus & INX;
  
  fpstatus = tmp2;
  fprounding = tmp3;
    
  !print "After exponent extraction: ", (frawx) fpdigF0; new_line;
  
  ! Get the value back
  sex = fpdigF0-->0;
  mhi = fpdigF0-->1;
  mlo = fpdigF0-->2;
 .expadjustdone;
  exp = sex & $7ff;
  sex = sex & $8000;
  
  digits = 9;
  
  ! Shift the mantissa so the binary point is between bits 12 and 11 of mhi
  ! The extra bits (not many) go into grs.
  exp = 1023 + 3 - exp;
  tmp = 16 - exp;
  tmp2 = -exp;
  @log_shift mlo tmp -> grs;
  @log_shift mlo tmp2 -> mlo;
  @log_shift mhi tmp -> sp;
  @or mlo sp -> mlo;
  @log_shift mhi tmp2 -> mhi;
  
 ! print "Shifted mantissa: ", (hex) mhi, (hex) mlo, (hex) grs; new_line;
  
  ! If the mantissa is <1, decrement the arith exponent, and proceed
  ! to "multiply by ten", otherwise extract the first digit.
  exp = 0;
  tmp2 = 0;
  if (mhi & $F000)
    jump extract_digit;
  --arith;
  
  ! Stage one - three words to go, accumulating into two words
  do
  {
    ! First multiply by 2
    mhi = mhi + mhi; if (mlo < 0) ++mhi;
    mlo = mlo + mlo; if (grs < 0) ++mlo;
    grs = grs + grs;
    ! Then by five - work out (mhi,mlo,grs)*4 + (mhi,mlo,grs)
    @log_shift grs 2 -> tmp3;
    @log_shift grs $fff2 -> sp;
    @log_shift mlo 2 -> sp;
    @or sp sp -> tmp4;
    @log_shift mlo $fff2 -> sp;
    @log_shift mhi 2 -> sp;
    @or sp sp -> tmp;
    grs = grs + tmp3;
    c = UnsignedCompare(grs, tmp3) < 0;
    tmp3 = mlo + tmp4 + c;
    c = (mlo < 0 && tmp4 < 0) ||
        (mlo < 0 && tmp3 >= 0) ||
        (tmp4 < 0 && tmp3 >= 0);
    mlo = tmp3;
    mhi = mhi + tmp + c;
    
   ! print "Times 10: ", (hex) mhi, (hex) mlo, (hex) grs; new_line;
    
   .extract_digit;
    ! The integer part of the number is the next digit. Move it up into
    ! exp, and decrement the digit count.
    @log_shift tmp2 4 -> sp;
    @log_shift exp $fff4 -> sp;
    @or sp sp -> tmp2;
    @log_shift exp 4 -> sp;
    @log_shift mhi $fff4 -> sp;
    @or sp sp -> exp;
    mhi = mhi & $0fff;
    --digits;
  } until (grs==0);
  
  tmp = 0;
  
  ! Second loop - two words to process in (mhi,mlo) - accumulating
  ! into (tmp,tmp2,exp).
  do
  {
    ! Multiply by 2 then 5, as before
    mhi = mhi + mhi; if (mlo < 0) ++mhi;
    mlo = mlo + mlo;
    @log_shift mlo $fff2 -> sp;
    @log_shift mlo 2 -> tmp3;
    mlo = mlo + tmp3;
    c = UnsignedCompare(mlo, tmp3) < 0;
    @log_shift mhi 2 -> sp;
    @or sp sp -> tmp3;
    mhi = mhi + tmp3 + c;
    
   ! print "Times 10: ", (hex) mhi, (hex) mlo; new_line;
    
    ! Extract the digit
    @log_shift tmp 4 -> sp;
    @log_shift tmp2 $fff4 -> sp;
    @or sp sp -> tmp;
    @log_shift tmp2 4 -> sp;
    @log_shift exp $fff4 -> sp;
    @or sp sp -> tmp2;
    @log_shift exp 4 -> sp;
    @log_shift mhi $fff4 -> sp;
    @or sp sp -> exp;
    mhi = mhi & $0fff;
    --digits;
  } until (digits==0);
  
  inexact = inexact | mhi | mlo;
  @log_shift mhi $fff5 -> c; ! Round bit
  mhi = (mhi & $07ff) | mlo; ! Sticky bits
  !if (inexact || c || mhi)
  !{
  !  if (inexact) print "Inexact ";
  !  if (c) print "Round ";
  !  if (mhi) print "Sticky ";
  !  new_line;
  !}
  
  mlo = 0; ! round up flag
  switch (fprounding)
  {
    FE_TONEAREST:
      if (c)
        if (mhi || (exp & 1))
          mlo = 1;     
    FE_UPWARD:
      if (sex >= 0 && (c | mhi))
        mlo = 1;
    FE_DOWNWARD:
      if (sex < 0 && (c | mhi))
        mlo = 1;
  }
        
  if (mlo)
  {
  !  print "BCD++: ", (hex) tmp, (hex) tmp2, (hex) exp;
    ++exp;
    if ((exp & $f) == 10)
    {
      ! Need to start do a BCD carry
      @log_shift exp $ffff -> c;
      tmp3 = c + $3333;
      tmp3 = (tmp3 &~ c) | (c &~ tmp3);
      tmp3 = tmp3 & $8888;
      @log_shift tmp3 $fffe -> sp;
      @mul sp 3 -> sp;
      @add exp sp -> exp;
      if (tmp3 & $8000)
      {
        tmp2 = tmp2 + 1;
        @log_shift tmp2 $ffff -> c;
        tmp3 = $3333 + c;
        tmp3 = (tmp3 &~ c) | (c &~ tmp3);
        tmp3 = tmp3 & $8888;
        @log_shift tmp3 $fffe -> sp;
        @mul sp 3 -> sp;
        @add tmp2 sp -> tmp2;
        if (tmp3 & $8000)
        {
          tmp = tmp + 1;
          @log_shift tmp $ffff -> c;
          tmp3 = $3333 + c;
          tmp3 = (tmp3 &~ c) | (c &~ tmp3);
          tmp3 = tmp3 & $8888;
          @log_shift tmp3 $fffe -> sp;
          @mul sp 3 -> sp;
          @add tmp sp -> tmp;
          if (tmp & $0010)
          {
            tmp = 1;
            ++arith;
          }
        }
      }
    }
  !  print " -> ", (hex) tmp, (hex) tmp2, (hex) exp;
  }
  
  sex = sex | tmp;
  mhi = tmp2;
  mlo = exp;
  
  if (arith < 0)
  {
    sex = sex | $4000;
    arith = -arith;
  }
  tmp = 0;
  if (arith >= 80) { arith = arith - 80; tmp = tmp + $80; }
  if (arith >= 40) { arith = arith - 40; tmp = tmp + $40; }
  if (arith >= 20) { arith = arith - 20; tmp = tmp + $20; }
  if (arith >= 10) { arith = arith - 10 + $10; }
  tmp = tmp + arith;
  @log_shift tmp 4 -> tmp;
  sex = sex | tmp;
  
  if (inexact)
  {
    if (fpstatus & INXE)
      fptrap.inx_handler();
    else
      fpstatus = fpstatus | INX;
  }
  
 ! print "^Final result: ", (hex) sex, (hex) mhi, (hex) mlo; new_line;
  
 .done;
  dst-->0 = sex;
  dst-->1 = mhi;
  dst-->2 = mlo;
];

! Accumulate n BCD digits from the top of src into (dst-->0,dst-->1)
[ _readdigits dest src n
              hi lo tmp c rnd;
  hi = dest-->0;
  lo = dest-->1;
  do
  {
    ! First multiply by 10
    hi = hi + hi; if (lo < 0) ++hi;
    lo = lo + lo;
    @log_shift lo $fff2 -> sp;
    @log_shift lo 2 -> tmp;
    lo = lo + tmp;
    c = UnsignedCompare(lo, tmp) < 0;
    @log_shift hi 2 -> sp;
    @or sp sp -> tmp;
    hi = hi + tmp + c;
    ! Then add in the new digit
    @log_shift src $fff4 -> tmp;
    lo = lo + tmp;
    if (UnsignedCompare(lo, tmp) < 0) ++hi;
    @log_shift src 4 -> src;
  }
  until (--n == 0);
  dest-->0 = hi;
  dest-->1 = lo;
];

[ _fptos_naninf dest sex mhi mlo;
  !print "_fptos_naninf(", (hex) sex, (hex) mhi, (hex) mlo; print ")^";
  if ((mhi | mlo | (sex & $f)) == 0)
  {
    ! Infinity
    sex = (sex & $8000) | $7f80;
    jump done;
  }
  if ((sex & $0008) == 0)
  {
    ! Signalling NaN
    if (fpstatus & IVOE)
      fptrap.ivo_handler(); ! blargh;
    else
    {
      fpstatus = fpstatus | FE_INVALID;
      dest-->0 = $7fc0;
      dest-->1 = NaNReason_SigNaN*256;
    }
    return;
  }
  sex = (sex & $8000) | $7fc0;
  ! Pull out the bottom 7 digits only - all we care about for NaNs
  _fpscratch-->0 = 0;
  _fpscratch-->1 = 0;
  @log_shift mhi 4 -> mhi;
  _readdigits(_fpscratch, mhi, 3);
  _readdigits(_fpscratch, mlo, 4);
  ! (mhi,mlo) = value for NaN. Could be 24 bits if unusual - knock back to
  ! 22, and then we're all ready
  mhi = _fpscratch-->0;
  mlo = _fpscratch-->1;
  !print "Read digits: ", (hex) mhi, (hex) mlo; new_line;
  mhi = mhi & $003f;
  sex = sex | mhi;
 .done;
  dest-->0 = sex;
  dest-->1 = mlo;
];

[ fptos dest src
        sex mhi mmi mlo tmp arith;
  
  sex = src-->0;
  mmi = src-->1;
  mlo = src-->2;
  @log_shift sex 4 -> tmp;
  _fpscratch-->0 = 0;
  _fpscratch-->1 = 0;
  _readdigits(_fpscratch, tmp, 2);
  !print "Exponent = ", _fpscratch-->1; new_line;
  arith = _fpscratch-->1;
  if (arith > 99)
  {
    _fptos_naninf(dest, sex, mmi, mlo);
    return;
  }
  _fpscratch-->0 = 0;
  _fpscratch-->1 = sex & $f;
  _readdigits(_fpscratch, mmi, 4);
  _readdigits(_fpscratch, mlo, 4);
  !print "Full mantissa = ", (hex) _fpscratch-->0, (hex) _fpscratch-->1; new_line;
  mhi = _fpscratch-->0;
  mlo = _fpscratch-->1;
  
  ! Short cut for zero
  if ((mhi|mlo)==0)
  {
    dest-->0 = sex & $8000;
    dest-->1 = 0;
    return;
  }
  
  if (sex & $4000)
    arith = -arith;

  ! Adjust because we've got a 9 digit integer, not 1.8 digit decimal
  arith = arith - 8;
  
  ! Want to convert (mhi,mlo) into an extended precision number
  ! Need to normalise. We know (mhi,mlo)< 10^9 < 2^30
  sex = sex & $8000;
  sex = sex + 1023 + 31;
  while (mhi >= 0)
  {
    mhi = mhi + mhi; if (mlo < 0) ++mhi;
    mlo = mlo + mlo;
    --sex;
  }
  
  !print (hex) sex, (char) '|', (hex) mhi, (hex) mlo; new_line;
  
  fpdigF0-->0 = sex;
  fpdigF0-->1 = mhi;
  fpdigF0-->2 = mlo;
  
  ! Now just need to multiply by 10^arith. |arith| <= 99, so overflow
  ! isn't possible (we're spared because we refuse to do binary<->decimal
  ! on extended precision).
  
  tmp = fpstatus;
  fpstatus = 0;
  
  if (arith > 0)
  {
    _GetPowerOfTen(fpdigF1, arith);
    fmulx(fpdigF0, fpdigF0, fpdigF1);
  }
  else if (arith < 0)
  {
    _GetPowerOfTen(fpdigF1, -arith);
    fdivx(fpdigF0, fpdigF0, fpdigF1);
  }
  
  ! print "Extended result = ", (frawx) fpdigF0; new_line;
  
  ! Round back to single
  _precision = 0;
  _dest = dest;
  _RoundResult(fpdigF0-->0 & $8000, fpdigF0-->0 & $07ff,
               fpdigF0-->1, fpdigF0-->2, 0, $8000);
  
  !print "Final result = ", (fraw) dest; new_line;
];

! Convert ZSCII string to single. len is length of string (will
! not read beyond this). Returns number of characters consumed.
[ strtof dest str len
         c hex sign had_dot num_ok dhi dmi dlo exp cnt exp2 negexp;
 ! print "strtof($", (hex) dest, ",$", (hex) str, ",", len, ")^";
  if (len>=0) c = str->(cnt++);
  
  while (len>0 && c==' ')
    if (--len>0) c = str->(cnt++);
    
  if (len>0 && (c=='+' || c=='-'))
  {
    if (c=='-') sign = $8000;
    if (--len>0) c = str->(cnt++);
  }
  if (len>0 && c=='$')
  {
    hex = 1;
    if (--len>0) c = str->(cnt++);
  }
  while (len > 0)
  {
    !print "Got '", (char) c, "'^";
    if (c=='.' && ~~had_dot)
      had_dot=true;
    else if ((c>='0' && c<='9') ||
             (hex && ((c>='a' && c<='f') || (c>='A' && c<='F'))))
    {
      num_ok=true;
      if (c<='9') c=c-'0';
      else if (c<='F') c=c-'A'+10;
      else c=c-'a'+10; 
      if (dhi==0)
      {
        @log_shift dmi $fff4 -> dhi;
        @log_shift dmi 4 -> sp;
        @log_shift dlo $fff4 -> sp;
        @or sp sp -> dmi;
        @log_shift dlo 4 -> sp;
        @or sp c -> dlo;
        if (had_dot) --exp;
      }
      else
      {
        if (hex && c ~= '0') dlo = dlo | 1;
        if (~~had_dot) ++exp;
      }
    }
    else
      break;
    if (--len>0) c = str->(cnt++);
  }
  if (hex) exp = exp * 4;
  if (len>0 && num_ok &&
      (c=='e' || c=='E' || (hex && (c=='p' || c=='P'))))
  {
    num_ok=false;
    if (--len>0)
    {
      c = str->(cnt++);
      if (c=='-' || c=='+')
      {
        if (c=='-') negexp = true;
        if (--len>0) c = str->(cnt++);
      }
      while (len>0 && c>='0' && c<='9')
      {
        num_ok=true;
        exp2 = exp2*10 + c-'0';
        if (--len>0) c = str->(cnt++);
      }
      if (negexp)
        exp = exp-exp2;
      else
        exp = exp+exp2;
    }
  }
  if (len>0) cnt--;
  if (~~num_ok)
  {
    dest-->0=$0000;
    dest-->1=$0000;
    return 0;
  }
  !print (hex) dhi, (hex) dmi, (hex) dlo, " E", exp; new_line;
  if ((dhi|dmi|dlo)==0)
  {
    dest-->0=sign; ! Do we want signed zero input? Why not?
    dest-->1=$0000;
  }
  else
  {
    if ((dhi|dmi)==0)
    {
      dmi = dlo;
      dlo = 0;
      if (hex) exp=exp-16; else exp=exp-4;
    }
  !  print (hex) dhi, (hex) dmi, (hex) dlo, " E", exp; new_line;
    
    if (hex)
    {
      exp=exp+23+16+127;
      while ((dhi & $0080)==0 && exp > 0)
      {
        dhi = dhi+dhi; if (dmi < 0) dhi++;
        dmi = dmi+dmi; if (dlo < 0) dmi++;
        dlo = dlo+dlo;
        exp--;
      }
      
     ! print (hex) dhi, (hex) dmi, (hex) dlo, " P", exp; new_line;
      if (dlo < 0)
      {
        c=0;
        switch (fprounding)
        {
          FE_TONEAREST:
            if (dlo ~= $8000 || (dmi & 1))
              c=1;
          FE_UPWARD:
            if (sign >= 0) c=1;
          FE_DOWNWARD:
            if (sign < 0) c=1;
        }
        if (c)
        {
          if (++dmi==0)
          {
            if (++dhi==$0100)
            {
              dhi=$0080;
              dmi=$0000;
              exp--;
            }
          }
        }
      }
     ! print (hex) dhi, (hex) dmi, " P", exp; new_line;
      
      if (exp < 1 || exp > 254)
      {
        ! raise overflow exception (or underflow)
        dest-->0 = sign | $7F80;
        dest-->1 = 0;
        return cnt;
      }
      else if ((dhi & $0080)==0)
      {
        exp = 0; ! Subnormal
      }
      else
        dhi = dhi &~ $0080;
      
      @log_shift exp 7 -> sp;
      @or dhi sp -> dhi;
      dhi = dhi | sign;
      dest-->0 = dhi;
      dest-->1 = dmi;
     ! print (fraw) dest; new_line;
    }
    else
    {
      while (dhi==0)
      {
        @log_shift dmi $fff4 -> dhi;
        @log_shift dmi 4 -> sp;
        @log_shift dlo $fff4 -> sp;
        @or sp sp -> dmi;
        @log_shift dlo 4 -> dlo;
        exp--;
      }
      exp = exp+8;
    
      if (exp < -99 || exp > 99)
      {
        ! raise overflow exception (or underflow)
        dest-->0 = sign | $7F80;
        dest-->1 = 0;
        return cnt;
      }
      else
      {
        if (exp < 0) { exp = -exp; dhi = dhi | $4000; }
        dhi = dhi | sign;
        @div exp 10 -> sp;
        @log_shift sp 8 -> sp;
        @or dhi sp -> dhi;
        @mod exp 10 -> sp;
        @log_shift sp 4 -> sp;
        @or dhi sp -> dhi;
        fpdigF0-->0=dhi;
        fpdigF0-->1=dmi;
        fpdigF0-->2=dlo;
      !  print (frawx) fpdigF0; new_line;
        fptos(dest, fpdigF0);
      }
    }
  }
  return cnt;
];

Array f1 --> $4000 $0000 $0000;
Array f2 --> $4043 $0000;
Array f3 --> 2;

Array Zero --> $0000 $0000;
Array MinusZero --> $8000 $0000;
Array OneThird --> $3EAA $AAAB;
Array OneFifth --> $3E4C $CCCD;
Array OneSixteenth --> $3D80 $0000;
Array TwentyFive --> $41C8 $0000;
Array TwentySeven --> $41D8 $0000;
Array Infinity --> $7f80 $0000;
Array NaN --> $7fc0 $0001;

Array One --> $3f80 $0000;
Array OneTwentySixth --> $3D1D $89D9;
Array OneTwentySeventh --> $3D17 $B426;
Array OneTwentySeventhX --> $03DF $97B4 $25ED;
Array NotTricky --> $3D10 $0000;
Array Tricky --> $3D10 $0001;
Array TwoToSixteen --> $4780 $0000;

Array OneTwentySeventhPlusUlp --> $3D17 $B427;

Array NextSub1 --> $B180 $0000;
Array NextSub2 --> $B180 $0001;

Array NextSub3 --> $2600 $0000;
Array NextSub4 --> $2600 $0001;

Array NextSub5 --> $9A80 $0000;
Array NextSub6 --> $9A80 $0001;

Array NextSub7 --> $0F00 $0000;
Array NextSub8 --> $0F00 $0001;

Array NextSub9  --> $0380 $0000;
Array NextSub10 --> $0380 $0001;

Array Tiny --> $0000 $0001;
Array SmallestNormal --> $0080 $0000;

Array Fred --> $3F7F $FFFF;
Array Ten --> $4120 $0000;

Array Pi --> $4049 $0FDB;

Array TenMil --> $4B18 $9680;
Array HundredMil --> $4CBE $BC20;
Array Bil --> $4E6E $6B28;
Array TenBil --> $5015 $02F9;
Array HundredBil --> $523A $43B7;
Array HundredBilP --> $523A $43B8;

Array ftestres --> 2;
Array fx --> 3;

Array dec --> 3;

[ ftest func fnam f1 f2;
  func(ftestres,f1,f2);
  style bold;
  print (fraw) f1, " ", (string) fnam, " ", (fraw) f2, " = ";
  print (fraw) ftestres; new_line;
  print (fhex) ftestres; new_line;
  style roman;
];

[ randombindecbin a b;
  @random 256 -> a;
  @random 256 -> b;
  ftestres-->0 = (a-1)*256+(b-1);
  @random 256 -> a;
  @random 256 -> b;
  ftestres-->1 = (a-1)*256+(b-1);
  testbindecbin(ftestres);
];

[ fulltest a b;
  ftestres-->0 = a;
  ftestres-->1 = b;
  for (::)
  {
    testbindecbin(ftestres);
    if (++b==0) { if (++a==0) return; print (hex) a; new_line; }
  }
];

Array st string "3.1415";

Array inputbuf -> 256;
[ inputtest len;
  inputbuf->0 = 200;
  read inputbuf 0;
  len = strtof(fx, inputbuf+2, inputbuf->1);
  print len, " characters read -> ", (fp) fx; new_line;
];

Array py_guess --> 3;
Array py_error --> 3;
Array py_temp --> 3;
Array py_deltax --> 3;
Array py_deltay --> 3;
Array py_r --> 3;
Array py_s --> 3;
Array py_2 --> $0400 $8000 $0000;
Array py_4 --> $0401 $8000 $0000;

[ pysum dest x y;
  fstox(py_deltax, x);
  fstox(py_deltay, y);
  fabsx(py_deltax, py_deltax);
  fabsx(py_deltay, py_deltay);
  fmaxx(py_guess, py_deltax, py_deltay);
  fminx(py_error, py_deltax, py_deltay);
  
  fdivx(py_temp, py_error, py_guess);
  fmulx(py_r, py_temp, py_temp);
  faddx(py_temp, py_r, py_4);
  fdivx(py_s, py_r, py_temp);
  fmulx(py_temp, py_2, py_s);
  fmulx(py_temp, py_temp, py_guess);
  faddx(py_guess, py_guess, py_temp);
  
  fmulx(py_error, py_error, py_s);
  fdivx(py_temp, py_error, py_guess);
  fmulx(py_r, py_temp, py_temp);
  faddx(py_temp, py_r, py_4);
  fdivx(py_s, py_r, py_temp);
  fmulx(py_temp, py_2, py_s);
  fmulx(py_temp, py_temp, py_guess);
  faddx(py_guess, py_guess, py_temp);
  
  fmulx(py_error, py_error, py_s);
  fdivx(py_temp, py_error, py_guess);
  fmulx(py_r, py_temp, py_temp);
  faddx(py_temp, py_r, py_4);
  fdivx(py_s, py_r, py_temp);
  fmulx(py_temp, py_2, py_s);
  fmulx(py_temp, py_temp, py_guess);
  faddx(py_guess, py_guess, py_temp);
  
  fxtos(dest, py_guess);
];

[ zsciitable s
             len i;
  len = s-->0;
  s = s + 2;
  for (i=0: i<len: i++)
    print (char) s->i;
];
