/*
// Abstract:
//	OBJECTS---HP 48SX Objects
//
//	The HP 48SX Objects module contains routines to format and
//	display objects contained in the HP 48SX.
//
// Author:
//	Derek S. Nickel
//
// Creation date:
//	25 October 1990
//
// History:
// V01-001	Derek S. Nickel		25-OCT-1990
//	Original.
//
*/

#include <ctype.h>
#include <io.h>
#include <stdlib.h>
#include <string.h>

#include "memory.h"
#include "modes.h"
#include "objects.h"
#include "pager.h"
#include "index.h"
#include "instruct.h"

static int display_object(bin5_t *origWP, bin5_t *ta);
static char *get_prolog_description(bin5_t prolog, char *buf);
static bin5_t follow_refernece(bin5_t adr);

/***********************************************************************
	Object Prologs and Other Addresses
***********************************************************************/

bin5_t
	prolog_system_binary = 0x02911,
	prolog_real_number = 0x02933,
	prolog_long_real = 0x02955,
	prolog_complex_number = 0x02977,
	prolog_long_complex = 0x0299D,
	prolog_character = 0x029BF,
	prolog_array = 0x029E8,
	prolog_linked_array = 0x02A0A,
	prolog_string = 0x02A2C,
	prolog_binary_integer = 0x02A4E,
	prolog_list = 0x02A74,
	prolog_directory = 0x02A96,
	prolog_algebraic = 0x02AB8,
	prolog_unit = 0x02ADA,
	prolog_tagged = 0x02AFC,
	prolog_graphic = 0x02B1E,
	prolog_library = 0x02B40,
	prolog_backup = 0x02B62,
	prolog_library_data = 0x02B88,
	prolog_program = 0x02D9D,
	prolog_code = 0x02DCC,
	prolog_global_name = 0x02E48,
	prolog_local_name = 0x02E6D,
	prolog_xlib_name = 0x02E92,
	end_marker = 0x0312B;

/***********************************************************************
	get_prolog_description
***********************************************************************/

static char *get_prolog_description(bin5_t prolog, char *buf)
{
	static char *prolog_names[] = {
		"External",
		"System Binary",
		"Real Number",
		"Long Real",
		"Complex Number",
		"Long Complex",
		"Character",
		"Array",
		"Linked Array",
		"String",
		"Binary Integer",
		"List",
		"Directory",
		"Algebraic",
		"Unit",
		"Tagged",
		"Graphic",
		"Library",
		"Backup",
		"Library Data",
		"Program",
		"Code",
		"Global Name",
		"Local Name",
		"XLIB Name" };

	int idx = 0;

	if (prolog == prolog_system_binary) {
		idx = 1;
	} else if (prolog == prolog_real_number) {
		idx = 2;
	} else if (prolog == prolog_long_real) {
		idx = 3;
	} else if (prolog == prolog_complex_number) {
		idx = 4;
	} else if (prolog == prolog_long_complex) {
		idx = 5;
	} else if (prolog == prolog_character) {
		idx = 6;
	} else if (prolog == prolog_array) {
		idx = 7;
	} else if (prolog == prolog_linked_array) {
		idx = 8;
	} else if (prolog == prolog_string) {
		idx = 9;
	} else if (prolog == prolog_binary_integer) {
		idx = 10;
	} else if (prolog == prolog_list) {
		idx = 11;
	} else if (prolog == prolog_directory) {
		idx = 12;
	} else if (prolog == prolog_algebraic) {
		idx = 13;
	} else if (prolog == prolog_unit) {
		idx = 14;
	} else if (prolog == prolog_tagged) {
		idx = 15;
	} else if (prolog == prolog_graphic) {
		idx = 16;
	} else if (prolog == prolog_library) {
		idx = 17;
	} else if (prolog == prolog_backup) {
		idx = 18;
	} else if (prolog == prolog_library_data) {
		idx = 19;
	} else if (prolog == prolog_program) {
		idx = 20;
	} else if (prolog == prolog_code) {
		idx = 21;
	} else if (prolog == prolog_global_name) {
		idx = 22;
	} else if (prolog == prolog_local_name) {
		idx = 23;
	} else if (prolog == prolog_xlib_name) {
		idx = 24;
	}

	return strcpy(buf,prolog_names[idx]);
}

/***********************************************************************
	hex_to_text
***********************************************************************/

static char *hex_to_text(char *buf, char *display)
{
	int i;
	char ch, nib0, nib1;
	char *p;
	char *q;

	p = buf;
	q = display;

	while (*p) {
		nib0 = *p++;
		nib1 = *p++;
		ch = hexval(nib0) + (hexval(nib1) << 4);

		if (isprint(ch))
			*q++ = ch;
		else
			*q++ = '.';
	}

	*q++ = '\0';

	return display;
}

/***********************************************************************
	format_real_number
***********************************************************************/

static void format_real_number(char *buf, char *display)
{
	char buf2[17];
	char *p;
	char *q;
	long digits;
	long i;
	long exp;

	strcpy(buf2,buf);
	strrev(buf2);

	p = &buf2[12];
	q = &buf2[1];

	while (p > q && *p == '0')
		p--;

	digits = p - q + 1;

	exp = atol(&buf2[13]);
	if (exp > 499)
		exp = exp - 1000;

	p = buf2;
	q = display;

	if (*p++ != '0') *q++ = '-';

	if (exp >= 0 && exp <= 11) {

		for (i = 0; i < digits-1; i++) {
			*q++ = *p++;
			if (i == exp) *q++ = '.';
		}

		*q++ = *p++;

		for (i = digits; i <= exp; i++) {
			*q++ = '0';
		}

		*q = '\0';

	} else if (exp >= -12 && exp <= -1) {

		*q++ = '.';

		for (i = 1; i < -exp; i++) {
			*q++ = '0';
		}

		for (i = 0; i < digits; i++) {
			*q++ = *p++;
		}

		*q = '\0';

	} else {
		*q++ = *p++;
		*q++ = '.';

		for (i = 1; i < digits; i++)
			*q++ = *p++;

		*q++ = 'E';
		*q = '\0';

		ltoa(exp,q,10);
	}
}

/***********************************************************************
	format_long_real
***********************************************************************/

static void format_long_real(char *buf, char *display)
{
	char buf2[22];
	char *p;
	char *q;
	long digits;
	long i;
	long exp;

	strcpy(buf2,buf);
	strrev(buf2);

	p = &buf2[15];
	q = &buf2[1];

	while (p > q && *p == '0')
		p--;

	digits = p - q + 1;

	exp = atol(&buf2[16]);
	if (exp > 49999)
		exp = exp - 100000;

	p = buf2;
	q = display;

	if (*p++ != '0') *q++ = '-';

	if (exp >= 0 && exp <= 14) {

		for (i = 0; i < digits-1; i++) {
			*q++ = *p++;
			if (i == exp) *q++ = '.';
		}

		*q++ = *p++;

		for (i = digits; i <= exp; i++) {
			*q++ = '0';
		}

		*q = '\0';

	} else if (exp >= -15 && exp <= -1) {

		*q++ = '.';

		for (i = 1; i < -exp; i++) {
			*q++ = '0';
		}

		for (i = 0; i < digits; i++) {
			*q++ = *p++;
		}

		*q = '\0';

	} else {
		*q++ = *p++;
		*q++ = '.';

		for (i = 1; i < digits; i++)
			*q++ = *p++;

		*q++ = 'E';
		*q = '\0';

		ltoa(exp,q,10);
	}
}

/***********************************************************************
	format_character
***********************************************************************/

static void format_character(char *buf,char *display)
{
	char buf2[3];
	char ch;

	strcpy(buf2,buf);
	strrev(buf2);

	ch = (hexval(buf2[0]) << 4) + hexval(buf2[1]);
	if (!isprint(ch)) ch = '.';

	sprintf(display,"'%c' 0x%s", ch, buf2);
}

/***********************************************************************
	get_ascic
***********************************************************************/

static void get_ascic(char *display)
{
	char buf2[3];
	char buf3[65];
	bin5_t size;

	GetNNibbles(buf2,2);
	strrev(buf2);
	size = 2*str2adr(buf2,0);

	if (size <= 64) {
		/*
		// Can be displayed on one line.
		*/

		GetNNibbles(buf3,size);
		hex_to_text(buf3,display);

	} else {
		/*
		// Truncate the name (and fetch remaining data).
		*/

		GetNNibbles(buf3,58);
		size -= 58;
		hex_to_text(buf3,display);
		strcat(display,"...");

		while (size >= 64) {
			GetNNibbles(buf3,64);
			size -= 64;
		}
		GetNNibbles(buf3,size);
	}
}

/***********************************************************************
	get_ascix
***********************************************************************/

static void get_ascix(char *display)
{
	char buf2[3];
	char buf3[65];
	bin5_t size;

	GetNNibbles(buf2,2);
	strrev(buf2);
	size = 2*str2adr(buf2,0);

	if (size == 0) {
		/*
		// Zero length name, no trailing size field.
		*/

		*display = '\0';

	} else if (size <= 64) {
		/*
		// Can be displayed on one line.
		*/

		GetNNibbles(buf3,size);
		hex_to_text(buf3,display);
		GetNNibbles(buf2,2);

	} else {
		/*
		// Truncate the name and fetch remaining data.
		*/

		GetNNibbles(buf3,58);
		size -= 58;
		hex_to_text(buf3,display);
		strcat(display,"...");

		while (size >= 64) {
			GetNNibbles(buf3,64);
			size -= 64;
		}
		GetNNibbles(buf3,size);
		GetNNibbles(buf2,2);
	}
}

/***********************************************************************
***********************************************************************/

static bin5_t follow_reference(bin5_t adr)
{
	bin5_t tmp;

	/*
	// Get the prolog.
	*/

	SetWorkPtr(adr);
	tmp = get_bin5();

	/*
	// If this is a System Binary...
	*/

	if (tmp == prolog_system_binary) {
		/*
		// ...display the reference...
		*/

		bin5_t data;
		char *acom;

		acom = get_address_comment(adr);
		if (*acom) {
			pager(0);
			printf("%05lX: ; *** %s ***\n", adr, acom);
		}

		data = get_bin5();

		pager(0);
		acom = get_address_comment(data);
		if (*acom)
			printf("%05lX: [...] ! <%lXh>\t; %s\n",
				adr, data, acom );
		else
			printf("%05lX: [...] ! <%lXh>\n", adr, data );

		/*
		// ...and follow reference.
		*/

		adr = data;
	}

	return adr;
}

/***********************************************************************
	display_hash_table
***********************************************************************/

void display_hash_table(bin5_t adr, bin5_t lib_no)
{
	bin5_t ht_size;
	bin5_t nt_size;
	bin5_t ht_end;
	bin5_t nt_end;
	bin5_t tmp;
	bin5_t obj_no;
	char buf3[3];
	char display[33];
	char *acom;
	int i;
	int names;

	/*
	// Follow reference, if any.
	*/

	adr = follow_reference(adr);
	SetWorkPtr(adr);

	/*
	// Display header comment.
	*/

	acom = get_address_comment(adr);
	if (*acom) {
		pager(0);
		printf("%05lX: ; *** %s ***\n", adr, acom);
	}
/*
	pager(0);
	printf("%05lX: ! *** Hash table for library %03lX (XLIB %ld) ***\n",
		adr, lib_no, lib_no);
*/

	tmp = get_bin5();
	pager(0);
	if (tmp == prolog_binary_integer)
		printf("%05lX: %05lX ! (Binary Integer)\n", adr, tmp);
	else
		printf("%05lX: %05lX ! ???\n", adr, tmp);

	adr = WorkPtr;
	ht_size = get_bin5();
	ht_end = adr + ht_size;

	/*
	// Name hashed by length.
	*/

	pager(0);
	printf("%05lX: %05lX ! %ld nibbles (next at %05lX)\n",
		adr, ht_size, ht_size, ht_end );

	for (i = 1; i <= 16; i++) {

		adr = WorkPtr;
		tmp = get_bin5();

		pager(0);
		if (tmp == 0) {
			printf("%05lX: %05lX !   %2d none\n",
				adr, tmp, i );
		} else {
			printf("%05lX: %05lX !   %2d at %05lX\n",
				adr, tmp, i, (adr + tmp) & 0xFFFFF);
		}
	}

	adr = WorkPtr;
	nt_size = get_bin5();
	nt_end = adr + nt_size;

	/*
	// Names sorted first by length and the alphabetically.
	*/

	pager(0);
	printf("%05lX: %05lX ! %ld nibbles (next at %05lX)\n",
		adr, nt_size, nt_size, nt_end );

	names = 0;
	adr = WorkPtr;

	while (adr < nt_end) {
		names++;
		get_ascic(display);
		GetNNibbles(buf3,3);
		strrev(buf3);
		obj_no = str2adr(buf3,0);

		pager(0);
		printf("%05lX: [...] !   %s (XLIB %ld %ld)\n",
			adr, display, lib_no, obj_no );

		adr = WorkPtr;
	}

	/*
	// XLIB number to name offsets (backward pointing).
	*/

	i = 0;
	adr = WorkPtr;

	while (adr < ht_end) {
		tmp = get_bin5();

		pager(0);
		printf("%05lX: %05lX !   name for XLIB %ld %d at %05lX\n",
			adr, tmp, lib_no, i, (adr - tmp) & 0xFFFFF);

		i++;
		adr = WorkPtr;
	}
}

/***********************************************************************
	display_message_table
***********************************************************************/

void display_message_table(bin5_t adr, bin5_t lib_no)
{
	bin5_t tmp;

	/*
	// Follow reference, if any.
	*/

	adr = follow_reference(adr);
	SetWorkPtr(adr);
/*
	pager(0);
	printf("%05lX: ! *** Message table for library %03lX (XLIB %ld) ***\n",
		adr, lib_no, lib_no);
*/
	tmp = get_bin5();
	display_object(&adr, &tmp);
}

/***********************************************************************
	display_link_table
***********************************************************************/

void display_link_table(bin5_t adr, bin5_t lib_no)
{
	bin5_t lt_size;
	bin5_t lt_end;
	bin5_t tmp;
	int i;
	char *acom;

	/*
	// Follow reference, if any.
	*/

	adr = follow_reference(adr);
	SetWorkPtr(adr);

	/*
	// Display header comment.
	*/

	acom = get_address_comment(adr);
	if (*acom) {
		pager(0);
		printf("%05lX: ; *** %s ***\n", adr, acom);
	}
/*
	pager(0);
	printf("%05lX: ! *** Link table for library %03lX (XLIB %ld) ***\n",
		adr, lib_no, lib_no);
*/
	tmp = get_bin5();
	pager(0);
	if (tmp == prolog_binary_integer)
		printf("%05lX: %05lX ! (Binary Integer)\n", adr, tmp);
	else
		printf("%05lX: %05lX ! ???\n", adr, tmp);

	adr = WorkPtr;
	lt_size = get_bin5();
	lt_end = adr + lt_size;

	pager(0);
	printf("%05lX: %05lX ! %ld nibbles (next at %05lX)\n",
		adr, lt_size, lt_size, lt_end );

	i = 0;
	adr = WorkPtr;

	while (adr < lt_end) {

		tmp = get_bin5();

		pager(0);
		printf("%05lX: %05lX !   XLIB %ld %d at %05lX\n",
			adr, tmp, lib_no, i, (adr + tmp) & 0xFFFFF);

		i++;
		adr = WorkPtr;
	}
}

/***********************************************************************
	display_conf_code
***********************************************************************/

static void display_conf_code(bin5_t adr)
{
	bin5_t ta;

	/*
	// Follow reference, if any.
	*/

	adr = follow_reference(adr);
	SetWorkPtr(adr);

	ta = get_bin5();
	display_object(&adr, &ta);
}

/***********************************************************************
	display_short_library
***********************************************************************/

void display_short_library(bin5_t adr)
{
	char buf2[6];
	bin5_t tmp;
	bin5_t lib_no;
	bin5_t hash_table;
	bin5_t message_table;
	bin5_t link_table;
	bin5_t conf_code;
	char *acom;

	/*
	// Follow reference, if any.
	*/

	adr = follow_reference(adr);
	SetWorkPtr(adr);

	/*
	// Display header comment.
	*/

	acom = get_address_comment(adr);
	if (*acom) {
		pager(0);
		printf("%05lX: ; *** %s ***\n", adr, acom);
	}

	/*
	// Display the short library header.
	*/

	adr = WorkPtr;
	GetNNibbles(buf2,3);
	strrev(buf2);
	lib_no = str2adr(buf2,0);

	pager(0);
	printf("%05lX: %03lX   ! Library %03lX (XLIB %ld)\n",
		adr, lib_no, lib_no, lib_no );

	adr = WorkPtr;
	tmp = get_bin5();

	pager(0);
	if (tmp == 0) {
		hash_table = 0;
		printf("%05lX: %05lX ! No hash table\n",
			adr, tmp );
	} else {
		hash_table = (tmp + adr) & 0xFFFFF;
		printf("%05lX: %05lX ! Hash table at %05lX\n",
			adr, tmp, hash_table );
	}

	adr = WorkPtr;
	tmp = get_bin5();

	pager(0);
	if (tmp == 0) {
		message_table = 0;
		printf("%05lX: %05lX ! No message table\n",
			adr, tmp );
	} else {
		message_table = (tmp + adr) & 0xFFFFF;
		printf("%05lX: %05lX ! Message table at %05lX\n",
			adr, tmp, message_table );
	}

	adr = WorkPtr;
	tmp = get_bin5();

	pager(0);
	if (tmp == 0) {
		link_table = 0;
		printf("%05lX: %05lX ! No link table\n",
			adr, tmp );
	} else {
		link_table = (tmp + adr) & 0xFFFFF;
		printf("%05lX: %05lX ! Link table at %05lX\n",
			adr, tmp, link_table );
	}

	adr = WorkPtr;
	tmp = get_bin5();

	pager(0);
	if (tmp == 0) {
		conf_code = 0;
		printf("%05lX: %05lX ! No configuration code\n",
			adr, tmp );
	} else {
		conf_code = (tmp + adr) & 0xFFFFF;
		printf("%05lX: %05lX ! Configuration code at %05lX\n",
			adr, tmp, conf_code );
	}
}

/***********************************************************************
	display_object
***********************************************************************/

static int display_object(bin5_t *origWP, bin5_t *ta)
{
	int asmMode = 0;
	char *acom;

	/*
	// Display header comment.
	*/

	acom = get_address_comment(*origWP);
	if (*acom) {
		pager(0);
		printf("%05lX: ; *** %s ***\n", *origWP, acom);
	}

	/*
	// What kind of instruction or object is this?
	*/

	if (*ta == WorkPtr) {
		/*
		// Machine code entry point.  Thread points to the very
		// next nibble.
		*/

		pager(0);
		printf("%05lX: %05lX ! Machine Code at %05lX\n",
			*origWP, *ta, *ta );

		asmMode = 1;

	} else if (*ta == prolog_system_binary) {
		/*
		// System Binary.
		*/

		char buf2[6];
		char display[9];
		bin5_t data;

		data = get_bin5();

		pager(0);
		acom = get_address_comment(data);
		if (*acom)
			printf("%05lX: [...] ! <%lXh>\t; %s\n",
				*origWP, data, acom );
		else
			printf("%05lX: [...] ! <%lXh>\n", *origWP, data );

	} else if (*ta == prolog_real_number) {
		/*
		// Real Number.
		*/

		char buf2[17];
		char display[20];

		GetNNibbles(buf2,16);
		format_real_number(buf2,display);

		pager(0);
		printf("%05lX: [...] ! %s\n", *origWP, display);

	} else if (*ta == prolog_long_real) {
		/*
		// Long Real.
		*/

		char buf2[22];
		char display[25];

		GetNNibbles(buf2,21);
		format_long_real(buf2,display);

		pager(0);
		printf("%05lX: [...] ! %s (Long Real)\n", *origWP, display);

	} else if (*ta == prolog_complex_number) {
		/*
		// Complex Number.
		*/

		char buf2[17];
		char re_display[20];
		char im_display[20];

		GetNNibbles(buf2,16);
		format_real_number(buf2,re_display);

		GetNNibbles(buf2,16);
		format_real_number(buf2,im_display);

		pager(0);
		printf("%05lX: [...] ! (%s,%s)\n",
			*origWP, re_display, im_display);

	} else if (*ta == prolog_long_complex) {
		/*
		// Long Complex.
		*/

		char buf2[22];
		char re_display[25];
		char im_display[25];

		GetNNibbles(buf2,21);
		format_long_real(buf2,re_display);

		GetNNibbles(buf2,21);
		format_long_real(buf2,im_display);

		pager(0);
		printf("%05lX: [...] ! (%s,%s) (Long Complex)\n",
			*origWP, re_display, im_display );

	} else if (*ta == prolog_character) {
		/*
		// Character.
		*/

		char buf2[3];
		char display[9];

		GetNNibbles(buf2,2);
		format_character(buf2,display);
		pager(0);
		printf("%05lX: [...] ! %s (Character)\n",
			*origWP, display);

	} else if (*ta == prolog_array) {
		/*
		// Array.
		*/

		char buf2[6];
		char buf3[33];
		bin5_t size;
		bin5_t item_prolog;
		bin5_t num_dims;
		bin5_t dim_n;
		bin5_t c_dim;
		bin5_t total_elements;
		bin5_t c_element;

		pager(0);
		printf("%05lX: %05lX ! Array\n", *origWP, *ta );

		*origWP = WorkPtr;
		size = get_bin5();
		pager(0);
		printf("%05lX: %05lX ! %ld nibbles\n",
			*origWP, size, size);

		*origWP = WorkPtr;
		item_prolog = get_bin5();
		pager(0);
		printf("%05lX: %05lX ! ...of %s\n",
			*origWP, item_prolog,
			get_prolog_description(item_prolog,buf3) );

		*origWP = WorkPtr;
		num_dims = get_bin5();
		pager(0);
		printf("%05lX: %05lX ! %ld dimensional\n",
			*origWP, num_dims, num_dims );

		total_elements = 1;

		for (dim_n = 1; dim_n <= num_dims; dim_n++) {
			*origWP = WorkPtr;
			c_dim = get_bin5();
			total_elements *= c_dim;
			pager(0);
			printf("%05lX: %05lX ! Dim-%ld = %ld\n",
				*origWP, c_dim, dim_n, c_dim );
		}

		pager(0);
		printf(" . . . . . . ! %ld total elements\n",
			total_elements );

		for (c_element = 1; c_element <= total_elements; c_element++) {
			*origWP = WorkPtr;
			asmMode = display_object(origWP, &item_prolog);
		}

	} else if (*ta == prolog_string) {
		/*
		// String.
		*/

		char buf3[65];
		char display[33];
		bin5_t size;
		int j, ch, dlen;

		size = (get_bin5()-5)/2;

		if (size <= 32) {
			/*
			// Can be printed on one line.
			*/

			GetNNibbles(buf3,2*size);
			hex_to_text(buf3, display);

			pager(0);
			printf("%05lX: [...] ! \"%s\"\n",
				*origWP, display);

		} else {
			/*
			// Needs two or more lines to print.
			*/

			GetNNibbles(buf3,64);
			hex_to_text(buf3, display);
			pager(0);
			printf("%05lX: [...] ! \"%s\n",
				*origWP, display);
			size -= 32;

			while (size > 32) {
				*origWP = WorkPtr;
				GetNNibbles(buf3,64);
				hex_to_text(buf3, display);
				pager(0);
				printf("%05lX: [...] ! %s\n",
					*origWP, display);
				size -= 32;
			}

 			*origWP = WorkPtr;
			GetNNibbles(buf3,2*size);
			hex_to_text(buf3, display);
			pager(0);
			printf("%05lX: [...] ! %s\"\n",
				*origWP,display);
		}

	} else if (*ta == prolog_binary_integer) {
		/*
		// Binary Integer.
		*/

		char buf3[33];
		char display[17];
		bin5_t size;

		size = get_bin5() - 5;

		if (size <= 32) {
			/*
			// Can be printed on one line.
			*/

			GetNNibbles(buf3,size);
			strrev(buf3);
			pager(0);
			printf("%05lX: [...] ! # %sh\n", *origWP, buf3);

		} else {
			/*
			// Needs more than one line to print.
			*/

			pager(0);
			printf("%05lX: [...] ! Binary Integer (%ld nibbles)\n",
				*origWP, size );

			while (size > 32) {
				*origWP = WorkPtr;
				GetNNibbles(buf3,32);
				hex_to_text(buf3,display);
				strrev(buf3);
				pager(0);
				printf(" %s %s %05lX\n",
					buf3, display, *origWP );
				size -= 32;
			}

			*origWP = WorkPtr;
			GetNNibbles(buf3,size);
			hex_to_text(buf3,display);
			strrev(buf3);
			pager(0);
			printf(" %32s %16s %05lX\n",
				buf3, display, *origWP );
		}

	} else if (*ta == prolog_list) {
		/*
		// List.
		*/

		pager(0);
		printf("%05lX: %05lX ! List\n", *origWP, *ta);
		pager(0);
		printf(" . . . . . . ! {\n");

		*origWP = WorkPtr;
		*ta = get_bin5();

		while (*ta != end_marker) {
			asmMode = display_object(origWP,ta);
			*origWP = WorkPtr;
			*ta = get_bin5();
		}

		pager(0);
		printf("%05lX: %05lX ! End Marker\n", *origWP, *ta);
		pager(0);
		printf(" . . . . . . ! }\n");

	} else if (*ta == prolog_directory) {
		/*
		// Directory.
		*/

		char buf2[6];
		char display[33];
		bin5_t attach;
		bin5_t offset;
		bin5_t tmp2;
		bin5_t end_of_dir;
		unsigned long idx = 0;

		pager(0);
		printf("%05lX: %05lX ! Directory\n", *origWP, *ta);

		*origWP = WorkPtr;
		GetNNibbles(buf2,3);
		strrev(buf2);
		attach = str2adr(buf2,0);
		pager(0);
		if (attach == 0x7FF)
			printf("%05lX: %03lX   ! attachments: none\n",
				*origWP, attach);
		else
			printf("%05lX: %03lX   ! attachments: %ld\n",
				*origWP, attach, attach);

		idx++;
		*origWP = WorkPtr;
		offset = get_bin5();
		tmp2 = (*origWP + offset) & 0xFFFFF;

		end_of_dir = WorkPtr;

		while (offset != 0) {

			pager(0);
			printf("%05lX: %05lX ! object %ld at %05lX\n",
				*origWP, offset, idx, tmp2);

			SetWorkPtr(tmp2);

			*origWP = WorkPtr;
			get_ascix(display);
			pager(0);
			printf("%05lX: [...] ! *** %s ***\n",
				*origWP, display );

			*origWP = WorkPtr;
			*ta = get_bin5();
			asmMode = display_object(origWP,ta);

			if (idx == 1) end_of_dir = WorkPtr;

			tmp2-= 5;
			SetWorkPtr(tmp2);

			idx++;
			*origWP = WorkPtr;
			offset = get_bin5();
			tmp2 = (*origWP - offset) & 0xFFFFF;
		}

		pager(0);
		printf("%05lX: %05lX ! End of Directory Marker\n",
			*origWP, offset );

		/*
		// Restore work pointer to the physical end of the
		// directory; as if we had scanned the directory in
		// memory address order.
		*/

		SetWorkPtr(end_of_dir);
		*origWP = WorkPtr;

	} else if (*ta == prolog_algebraic) {
		/*
		// Algebraic.
		*/

		pager(0);
		printf("%05lX: %05lX ! Algebraic\n", *origWP, *ta);

		*origWP = WorkPtr;
		*ta = get_bin5();

		while (*ta != end_marker) {
			asmMode = display_object(origWP,ta);
			*origWP = WorkPtr;
			*ta = get_bin5();
		}

		pager(0);
		printf("%05lX: %05lX ! End Marker\n", *origWP, *ta);

	} else if (*ta == prolog_unit) {
		/*
		// Unit.
		*/

		pager(0);
		printf("%05lX: %05lX ! Unit\n", *origWP, *ta);

		*origWP = WorkPtr;
		*ta = get_bin5();

		while (*ta != end_marker) {
			asmMode = display_object(origWP,ta);
			*origWP = WorkPtr;
			*ta = get_bin5();
		}

		pager(0);
		printf("%05lX: %05lX ! End Marker\n", *origWP, *ta);

	} else if (*ta == prolog_tagged) {
		/*
		// Tagged.
		*/

		char display[33];

		pager(0);
		printf("%05lX: %05lX ! Tagged\n", *origWP, *ta);

		*origWP = WorkPtr;
		get_ascic(display);
		pager(0);
		printf("%05lX: [...] ! :%s:\n", *origWP, display );

		*origWP = WorkPtr;
		*ta = get_bin5();
		asmMode = display_object(origWP,ta);

		pager(0);
		printf(" . . . . . . ! End Tagged\n", *origWP, *ta);

	} else if (*ta == prolog_graphic) {
		/*
		// Graphic.
		*/

		char buf2[65];
		bin5_t columns;
		bin5_t rows;
		bin5_t size;
		int row;
		int nibs_per_row;
		div_t x;

		pager(0);
		printf("%05lX: %05lX ! Graphic\n", *origWP, *ta);

		*origWP = WorkPtr;
		size = get_bin5();
		pager(0);
		printf("%05lX: %05lX ! %ld nibbles (next at %05lX)\n",
			*origWP, size, size, *origWP + size);

		*origWP = WorkPtr;
		rows = get_bin5();
		pager(0);
		printf("%05lX: %05lX ! %ld rows\n",
			*origWP, rows, rows);

		*origWP = WorkPtr;
		columns = get_bin5();
		pager(0);
		printf("%05lX: %05lX ! %ld columns\n",
			*origWP, columns, columns);

		size -= 15;

		x = div(columns,8);
		nibs_per_row = 2*x.quot;
		if (x.rem != 0) nibs_per_row += 2;

		for (row = 0; row < rows; row++) {
			*origWP = WorkPtr;
			GetNNibbles(buf2,nibs_per_row);
			pager(0);
			printf("%05lX: [...] ! %s\n", *origWP, buf2);
		}

		pager(0);
		printf(" . . . . . . ! End of Graphic\n");

	} else if (*ta == prolog_library) {
		/*
		// Library.
		*/

		char buf2[6];
		char display[33];
		bin5_t size;
		bin5_t end_of_library;
		bin5_t tmp;
		bin5_t lib_no;
		bin5_t hash_table;
		bin5_t message_table;
		bin5_t link_table;
		bin5_t conf_code;

		pager(0);
		printf("%05lX: %05lX ! Library\n", *origWP, *ta );

		*origWP = WorkPtr;
		size = get_bin5();
		end_of_library = *origWP + size;

		pager(0);
		printf("%05lX: %05lX ! %ld nibbles (next RPL at %05lX)\n",
			*origWP, size, size, end_of_library );

		*origWP = WorkPtr;
		get_ascix(display);

		pager(0);
		printf("%05lX: [...] ! Library name: \"%s\"\n",
			*origWP, display );

		*origWP = WorkPtr;
		GetNNibbles(buf2,3);
		strrev(buf2);
		lib_no = str2adr(buf2,0);

		pager(0);
		printf("%05lX: %03lX   ! Library %03lX (XLIB %ld)\n",
			*origWP, lib_no, lib_no, lib_no );

		*origWP = WorkPtr;
		tmp = get_bin5();

		pager(0);
		if (tmp == 0) {
			hash_table = 0;
			printf("%05lX: %05lX ! No hash table\n",
				*origWP, tmp );
		} else {
			hash_table = (tmp + *origWP) & 0xFFFFF;
			printf("%05lX: %05lX ! Hash table at %05lX\n",
				*origWP, tmp, hash_table );
		}

		*origWP = WorkPtr;
		tmp = get_bin5();

		pager(0);
		if (tmp == 0) {
			message_table = 0;
			printf("%05lX: %05lX ! No message table\n",
				*origWP, tmp );
		} else {
			message_table = (tmp + *origWP) & 0xFFFFF;
			printf("%05lX: %05lX ! Message table at %05lX\n",
				*origWP, tmp, message_table );
		}

		*origWP = WorkPtr;
		tmp = get_bin5();

		pager(0);
		if (tmp == 0) {
			link_table = 0;
			printf("%05lX: %05lX ! No link table\n",
				*origWP, tmp );
		} else {
			link_table = (tmp + *origWP) & 0xFFFFF;
			printf("%05lX: %05lX ! Link table at %05lX\n",
				*origWP, tmp, link_table );
		}

		*origWP = WorkPtr;
		tmp = get_bin5();

		pager(0);
		if (tmp == 0) {
			conf_code = 0;
			printf("%05lX: %05lX ! No configuration code\n",
				*origWP, tmp );
		} else {
			conf_code = (tmp + *origWP) & 0xFFFFF;
			printf("%05lX: %05lX ! Configuration code at %05lX\n",
				*origWP, tmp, conf_code );
		}
/*
		if (hash_table != 0)
			display_hash_table(hash_table, lib_no);

		if (message_table != 0)
			display_message_table(message_table, lib_no);

		if (link_table != 0)
			display_link_table(link_table, lib_no);

		if (conf_code != 0)
			display_conf_code(conf_code);
*/
	} else if (*ta == prolog_backup) {
		/*
		// Backup.
		*/

		bin5_t size;
		bin5_t end_of_backup;
		char display[33];

		pager(0);
		printf("%05lX: %05lX ! Backup\n", *origWP, *ta);

		*origWP = WorkPtr;
		size = get_bin5();
		end_of_backup = *origWP + size;

		pager(0);
		printf("%05lX: %05lX ! %ld nibbles (next RPL at %05lX)\n",
			*origWP, size, size, end_of_backup );

		*origWP = WorkPtr;
		get_ascix(display);
		pager(0);
		printf("%05lX: [...] ! name: %s\n",
			*origWP, display );

		*origWP = WorkPtr;

		while (*origWP < end_of_backup) {
			*ta = get_bin5();
			asmMode = display_object(origWP,ta);
			*origWP = WorkPtr;
		}

		pager(0);
		printf(" . . . . . . ! End of Backup\n");

	} else if (*ta == prolog_library_data) {
		/*
		// Library Data.
		*/

		bin5_t size;
		bin5_t end_of_libdat;
		bin5_t ident;
		bin5_t libno;
		bin5_t objno;
		int stop;
		int save_follow;

		pager(0);
		printf("%05lX: %05lX ! Library Data\n", *origWP, *ta );

		*origWP = WorkPtr;
		size = get_bin5();
		end_of_libdat = *origWP + size;

		pager(0);
		printf("%05lX: %05lX ! %ld nibbles (next RPL at %05lX)\n",
			*origWP, size, size, end_of_libdat );

		*origWP = WorkPtr;
		ident = get_bin5();

		libno = ident & 0xFFF;
		objno = 12>>ident;

		pager(0);
		printf("%05lX: %05lX ! ident: %03lX/%02lX (%ld/%ld)\n",
			*origWP, ident, libno, objno, libno, objno );

		while (WorkPtr < end_of_libdat) {
			*origWP = WorkPtr;
			*ta = get_bin5();
			asmMode = display_object(origWP,ta);
		}

		pager(0);
		printf(" . . . . . . ! End of Library Data\n");

	} else if (*ta == prolog_program) {
		/*
		// Program.
		*/

		pager(0);
		printf("%05lX: %05lX ! Program\n", *origWP, *ta);

		*origWP = WorkPtr;
		*ta = get_bin5();

		while (*ta != end_marker) {
			asmMode = display_object(origWP,ta);
			*origWP = WorkPtr;
			*ta = get_bin5();
		}

		pager(0);
		printf("%05lX: %05lX ! End Marker\n", *origWP, *ta);

	} else if (*ta == prolog_code) {
		/*
		// Code.
		*/

		bin5_t size;
		bin5_t end_of_code;
		int stop;
		int save_follow;

		pager(0);
		printf("%05lX: %05lX ! Code\n", *origWP, *ta );

		*origWP = WorkPtr;
		size = get_bin5();
		end_of_code = *origWP + size;

		pager(0);
		printf("%05lX: %05lX ! %ld nibbles (next RPL at %05lX)\n",
			*origWP, size, size, end_of_code );

		save_follow = modes.follow;
		modes.follow = 0;

		while (WorkPtr < end_of_code)
			unassem1instr(&stop);

		pager(0);
		printf(" . . . . . . ! End of Code\n");

		modes.follow = save_follow;

	} else if (*ta == prolog_global_name) {
		/*
		// Global Name.
		*/

		char display[33];

		get_ascic(display);

		pager(0);
		printf("%05lX: [...] ! '%s' (Global Name)\n",
			*origWP, display );

	} else if (*ta == prolog_local_name) {
		/*
		// Local Name.
		*/

		char display[33];

		get_ascic(display);

		pager(0);
		printf("%05lX: [...] ! '%s' (Local Name)\n",
			*origWP, display );

	} else if (*ta == prolog_xlib_name) {
		/*
		// XLIB Name.
		*/

		char buf2[4];
		bin5_t lib;
		bin5_t fn;

		GetNNibbles(buf2,3);
		strrev(buf2);
		lib = str2adr(buf2,0);

		GetNNibbles(buf2,3);
		strrev(buf2);
		fn = str2adr(buf2,0);

		pager(0);
		printf("%05lX: [...] ! XLIB %ld %ld\n",
			*origWP, lib, fn );

	} else {
		/*
		// Plain RPL.
		*/

		pager(0);
		acom = get_address_comment(*ta);
		if (*acom)
			printf("%05lX: %05lX ; %s\n", *origWP, *ta, acom);
		else
			printf("%05lX: %05lX\n", *origWP, *ta);

	}

	return asmMode;
}

/***********************************************************************
	display_current_object
***********************************************************************/

int display_current_object(int *stop)
{
	bin5_t origWP = WorkPtr;
	bin5_t ta;
	int asmMode;

	/*
	// Get the next threaded instruction or object, display it and
	// check for stop condition,
	*/

	ta = get_bin5();
	asmMode = display_object(&origWP, &ta);
	*stop = (ta == end_marker);

	return asmMode;
}
