/*
// Abstract:
//	COMMANDS---Parse Command Line
//
//	The Parse Command Line module parses a Voyager command and
//	dispatches to the appropriate routine.
//
// Author:
//	Derek S. Nickel
//
// Creation date:
//	27 October 1990
//
// History:
// V01-001	Derek S. Nickel		27-OCT-1990
//	Original.
//
*/

#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "commands.h"
#include "dump.h"
#include "find.h"
#include "list.h"
#include "memory.h"
#include "modes.h"
#include "objects.h"
#include "pager.h"
#include "ports.h"
#include "search.h"
#include "version.h"

extern void output_help(void);
extern void unany(char *q);

/***********************************************************************
	Keyword Data Structures
***********************************************************************/

typedef struct _keyword_t keyword_t;
typedef void (*cmd_proc_t)(void);

struct _keyword_t {
	char *keyword;
	cmd_proc_t procedure;
};

/***********************************************************************
	Local Routines
***********************************************************************/

static void ask(char *prompt, char *cmdline);
static int check_maxparm(void);
static void cmd_comment_address(void);
static void cmd_comment_code(void);
static void cmd_comment_list(void);
static void cmd_dump(void);
static void cmd_dump_alt(void);
static void cmd_dump_hex(void);
static void cmd_dump_text(void);
static void cmd_examine(void);
static void cmd_examine_hash_table(void);
static void cmd_examine_link_table(void);
static void cmd_exit(void);
static void cmd_examine_message_table(void);
static void cmd_examine_short_library(void);
static void cmd_find(void);
static void cmd_header(void);
static void cmd_help(void);
static void cmd_list(void);
static void cmd_load(void);
static void cmd_merge(void);
static void cmd_search(void);
static void cmd_set(void);
static void cmd_set_limit(void);
static void cmd_set_mode(void);
static void cmd_show(void);
static void cmd_show_limit(void);
static void cmd_show_memory(void);
static void cmd_show_modes(void);
static void cmd_show_ports(void);
static void cmd_show_version(void);
static void cmd_sort(void);
static void cmd_unassemble(void);
static void cmd_uncomment_address(void);
static void cmd_uncomment_code(void);
static void cmd_uncontinue(void);
static void cmd_unload(void);
static void cmd_unthread(void);

static int convert_address(char *token, bin5_t *adr);
static int get_address(bin5_t *adr);
static int get_address_opt(bin5_t *adr);
static int get_address_range(bin5_t *start, bin5_t *end);
static int get_dec_number(char *prompt, bin5_t *number);
static char *get_first_token(void);
static int get_hex_number(char *prompt, bin5_t *number);
static char *get_next_token(char *prompt);
static char *get_next_token_opt(void);
static char *get_rest_of_line(char *prompt);
static char *get_rest_of_line_opt(void);
static void lookup_command(char *input, const keyword_t *kw,
	cmd_proc_t *proc,int *matches );
static void lookup_command_exact(char *input, const keyword_t *kw,
	cmd_proc_t *proc,int *matches );

/***********************************************************************
	Command Tables.
***********************************************************************/

static const keyword_t verbs[] = {
	"DUMP",		cmd_dump,
	"EXAMINE",	cmd_examine,
	"EXIT",		cmd_exit,
	"FIND",		cmd_find,
	"HEADER",	cmd_header,
	"HELP",		cmd_help,
	"LIST",		cmd_list,
	"LOAD",		cmd_load,
	"MERGE",	cmd_merge,
	"SEARCH",	cmd_search,
	"SET",		cmd_set,
	"SHOW",		cmd_show,
	"SORT",		cmd_sort,
	"UNASSEMBLE",	cmd_unassemble,
	"UNCONTINUE",	cmd_uncontinue,
	"UNTHREAD",	cmd_unthread,
	"UNLOAD",	cmd_unload,
	NULL,		NULL
};

static const keyword_t verbs2[] = {
	"CA", 		cmd_comment_address,
	"CAD", 		cmd_uncomment_address,
	"CC", 		cmd_comment_code,
	"CCD", 		cmd_uncomment_code,
	"CL", 		cmd_comment_list,
	"DA", 		cmd_dump_alt,
	"DH", 		cmd_dump_hex,
	"DT", 		cmd_dump_text,
	"U", 		cmd_uncontinue,
	"UA", 		cmd_unassemble,
	"UT", 		cmd_unthread,
	NULL,		NULL
};

static const keyword_t examine_kw[] = {
	"HASH_TABLE",	cmd_examine_hash_table,
	"LINK_TABLE",	cmd_examine_link_table,
	"MESSAGE_TABLE", cmd_examine_message_table,
	"SHORT_LIBRARY", cmd_examine_short_library,
	NULL,		NULL
};

static const keyword_t set_kw[] = {
	"LIMIT",	cmd_set_limit,
	"MODE",		cmd_set_mode,
	NULL,		NULL
};

static const keyword_t show_kw[] = {
	"LIMIT",	cmd_show_limit,
/*	"MEMORY",	cmd_show_memory,	*/
	"MODES",	cmd_show_modes,
	"PORTS",	cmd_show_ports,
	"VERSION",	cmd_show_version,
	NULL,		NULL
};

/***********************************************************************
	Local Data.
***********************************************************************/

static char zero = '\0';
static char g_cmdline[128];

static int done = 0;

static bin5_t last_address = 0;

/***********************************************************************
	Messages.
***********************************************************************/

// 00038008
#define cli$_abverb \
"%%CLI-W-ABKEYW, ambiguous command verb - supply more characters\n"

// 00038010
#define cli$_abkeyw \
"%%CLI-W-ABKEYW, ambiguous qualifier or keyword - supply more characters\n"

// 00038040
#define cli$_insfprm \
"%%CLI-W-INSFPRM, missing command parameters - supply all required parameters\n"

// 00038050
#define cli$_ivchar \
"%%CLI-W-IVCHAR, invalid numeric value - check for invalid digits\n"

// 00038060
#define cli$_ivkeyw \
"%%CLI-W-IVKEYW, unrecognized keyword - check validity and spelling\n"

// 00038090
#define cli$_ivverb \
"%%CLI-W-IVVERB, unrecognized command verb - check validity and spelling\n"

// 00038098
#define cli$_maxparm \
"%%CLI-W-MAXPARM, too many parameters - reenter command with fewer parameters\n"

// 00038248
#define cli$_cmdseg \
" \\%s\\\n"

// no numeric value...
#define vgr__author \
"-VOYAGER-I-AUTHOR, programmed by Derek S. Nickel\n"

// no numeric value...
#define vgr__version \
"%%VOYAGER-I-VERSION, this is VOYAGER version %s\n"

/***********************************************************************
	parse_command
***********************************************************************/

int parse_command(void)
{
	int matches;
	int plen;
	char *p;
	cmd_proc_t cmd;

	/*
	// Get a command.
	*/

	ask("Voyager> ", g_cmdline);

	/*
	// Reset the pager.
	*/

	pager(-1);

	/*
	// Get the command verb.
	*/

	p = get_first_token();
	plen = strlen(p);

	/*
	// Dispatch to command.
	*/

	if (p == NULL) {
		/* No Command */

	} else {
		/*
		// Look up in the brief/exact verbs commands table.
		*/

		lookup_command_exact(p, verbs2, &cmd, &matches);

		/*
		// If not found, look up in the verbs command table.
		*/

		if (matches == 0)
			lookup_command(p, verbs, &cmd, &matches);

		/*
		// Process based on number of matches.
		*/

		if (matches < 1) {
			/* Unrecognized command verb */
			pager(0);
			printf(cli$_ivverb);
			pager(0);
			printf(cli$_cmdseg,p);

		} else if (matches > 1) {
			/* Ambiguous qualifier or keyword */
			pager(0);
			printf(cli$_abverb);
			pager(0);
			printf(cli$_cmdseg,p);

		} else {
			/* Execute the command */
			cmd();
		}
	}

	return done;
}

/***********************************************************************
	cmd_comment_address
***********************************************************************/

static void cmd_comment_address(void)
{
	char *rest;
	int port_no;
	bin5_t port_adr;
	bin5_t adr;

	/*
	// Get address and comment; add to database.
	*/

	if (get_address(&adr))
	  if (rest = get_rest_of_line("_Comment: "))
	    if (xlate_adr(adr, &port_no, &port_adr))
	      AddTextLine(ports[port_no].ca_file, port_adr, rest);
}

/***********************************************************************
	cmd_comment_code
***********************************************************************/

static void cmd_comment_code(void)
{
	char *rest;
	int port_no;
	bin5_t port_adr;
	bin5_t adr;

	/*
	// Get address and comment; add to database.
	*/

	if (get_address(&adr))
	  if (rest = get_rest_of_line("_Comment: "))
	    if (xlate_adr(adr, &port_no, &port_adr))
	      AddTextLine(ports[port_no].cc_file, port_adr, rest);
}

/***********************************************************************
	cmd_comment_list
***********************************************************************/

static void cmd_comment_list(void)
{
	char *rem;
	bin5_t adr;

	if (get_address(&adr)) if (check_maxparm()) {
		rem = get_address_comment(adr);
		if (*rem) {
			pager(0);
			printf("(addr) %05lX: %s\n", adr, rem);
		}

		rem = get_code_comment(adr);
		if (*rem) {
			pager(0);
			printf("(code) %05lX: %s\n", adr, rem);
		}
	}
}

/***********************************************************************
	cmd_dump
***********************************************************************/

static void cmd_dump(void)
{
	bin5_t adr;

	if (get_address_opt(&adr))
	  if (check_maxparm())
		dump_memory(adr,0);
}

/***********************************************************************
	cmd_dump_alt
***********************************************************************/

static void cmd_dump_alt(void)
{
	bin5_t adr;

	if (get_address_opt(&adr))
	  if (check_maxparm()) 
	    dump_memory(adr,3);
}

/***********************************************************************
	cmd_dump_hex
***********************************************************************/

static void cmd_dump_hex(void)
{
	bin5_t adr;

	if (get_address_opt(&adr))
	  if (check_maxparm()) 
	    dump_memory(adr,1);
}

/***********************************************************************
	cmd_dump_text
***********************************************************************/

static void cmd_dump_text(void)
{
	bin5_t adr;

	if (get_address_opt(&adr))
	  if (check_maxparm()) 
	    dump_memory(adr,2);
}

/***********************************************************************
	cmd_examine
***********************************************************************/

static void cmd_examine(void)
{
	int matches;
	char *p;
	cmd_proc_t cmd;

	/*
	// Get EXAMINE keyword.
	*/

	if (p = get_next_token("_Examine what: ")) {
		/*
		// Lookup the EXAMINE keyword.
		*/

		strupr(p);
		lookup_command(p, examine_kw, &cmd, &matches);
	
		/*
		// Process based on number of matches.
		*/

		if (matches < 1) {
			/* Unrecognized keyword */
			pager(0);
			printf(cli$_ivkeyw);
			pager(0);
			printf(cli$_cmdseg,p);

		} else if (matches > 1) {
			/* Ambiguous qualifier or keyword */
			pager(0);
			printf(cli$_abkeyw);
			pager(0);
			printf(cli$_cmdseg,p);

		} else {
			/* Execute the command */
			cmd();
		}
	}
}

/***********************************************************************
	cmd_examine_hash_table
***********************************************************************/

static void cmd_examine_hash_table(void)
{
	bin5_t adr;
	bin5_t lib;

	if (get_address(&adr))
	  if (get_hex_number("_Library number: ", &lib))
	    if (check_maxparm())
	      display_hash_table(adr,lib);
}

/***********************************************************************
	cmd_examine_link_table
***********************************************************************/

static void cmd_examine_link_table(void)
{
	bin5_t adr;
	bin5_t lib;

	if (get_address(&adr))
	  if (get_hex_number("_Library number: ", &lib))
	    if (check_maxparm())
	      display_link_table(adr,lib);
}

/***********************************************************************
	cmd_examine_message_table
***********************************************************************/

static void cmd_examine_message_table(void)
{
	bin5_t adr;
	bin5_t lib;

	if (get_address(&adr))
	  if (get_hex_number("_Library number: ", &lib))
	    if (check_maxparm())
	      display_message_table(adr,lib);
}

/***********************************************************************
	cmd_examine_short_library
***********************************************************************/

static void cmd_examine_short_library(void)
{
	bin5_t adr;

	if (get_address(&adr))
	  if (check_maxparm())
	    display_short_library(adr);
}

/***********************************************************************
	cmd_exit
***********************************************************************/

static void cmd_exit(void)
{
	if (check_maxparm())
	  done = 1;
}

/***********************************************************************
	cmd_find
***********************************************************************/

static void cmd_find(void)
{
	bin5_t start;
	bin5_t end;
	char *data;

	if (get_address_range(&start,&end))
	  if (data = get_rest_of_line("_Search for: "))
	    find_comment(start,end,data);
}

/***********************************************************************
	cmd_header
***********************************************************************/

static void cmd_header(void)
{
	bin5_t adr;

	if (get_address_opt(&adr))
	  if (check_maxparm()) {
		char buf2[4];
		bin5_t flag;
		bin5_t lib_no;
		bin5_t obj_no;

		SetWorkPtr(adr);

		GetNNibbles(buf2,1);
		flag = str2adr(buf2,0);

		GetNNibbles(buf2,3);
		strrev(buf2);
		lib_no = str2adr(buf2,0);

		GetNNibbles(buf2,3);
		strrev(buf2);
		obj_no = str2adr(buf2,0);

		pager(0);
		printf("%05lX: [...] ! header: XLIB %ld %ld (%lX)\n",
			adr, lib_no, obj_no, flag );
	  }
}

/***********************************************************************
	cmd_help
***********************************************************************/

static void cmd_help(void)
{
	if (check_maxparm())
	  output_help();
}

/***********************************************************************
	cmd_list
***********************************************************************/

static void cmd_list(void)
{
	bin5_t start;
	bin5_t end;

	if (get_address_range(&start,&end))
	  if (check_maxparm())
	    list_comment(start,end);
}

/***********************************************************************
	cmd_load
***********************************************************************/

static void cmd_load(void)
{
	bin5_t p;
	char *q;

	if (get_dec_number("_Port number: ", &p))
	  if (q = get_next_token("_Module: "))
	    if (check_maxparm())
	      open_port(p,q);
}

/***********************************************************************
	cmd_merge
***********************************************************************/

static void cmd_merge(void)
{
	bin5_t p;
	char *q;

	if (get_dec_number("_Port number: ", &p))
	  if (q = get_next_token("_New module: "))
	    if (check_maxparm())
	      merge_port(p,q);
}

/***********************************************************************
	cmd_search
***********************************************************************/

static void cmd_search(void)
{
	bin5_t start;
	bin5_t end;
	char *data;

	if (get_address_range(&start,&end))
	  if (data = get_next_token("_Search for: "))
	    if (check_maxparm())
	      search_memory(start,end,data);
}

/***********************************************************************
	cmd_set
***********************************************************************/

static void cmd_set(void)
{
	int matches;
	char *p;
	cmd_proc_t cmd;

	/*
	// Get SET keyword.
	*/

	if (p = get_next_token("_Set what: ")) {
		/*
		// Lookup the SET keyword.
		*/

		strupr(p);
		lookup_command(p, set_kw, &cmd, &matches);
	
		/*
		// Process based on number of matches.
		*/

		if (matches < 1) {
			/* Unrecognized keyword */
			pager(0);
			printf(cli$_ivkeyw);
			pager(0);
			printf(cli$_cmdseg,p);

		} else if (matches > 1) {
			/* Ambiguous qualifier or keyword */
			pager(0);
			printf(cli$_abkeyw);
			pager(0);
			printf(cli$_cmdseg,p);

		} else {
			/* Execute the command */
			cmd();
		}
	}
}

/***********************************************************************
	cmd_set_limit
***********************************************************************/

static void cmd_set_limit(void)
{
	bin5_t p;

	/*
	// Get SET LIMIT size.
	*/

	if (get_dec_number("_Limit: ", &p))
	  if (check_maxparm()) {
		if (p < 1) p = 1;
		modes.limit = p;
	  }
}

/***********************************************************************
	cmd_set_mode
***********************************************************************/

static void cmd_set_mode(void)
{
	char *p;
	int plen;

	/*
	// Get SET MODE keyword.
	*/

	if (p = get_next_token("_Mode: "))
	  if (check_maxparm()) {
		plen = strlen(p);
		strupr(p);

		if (!strncmp(p,"P",plen)) {
			/* Ambiguous qualifier or keyword */
			pager(0);
			printf(cli$_abkeyw);
			pager(0);
			printf(cli$_cmdseg,p);

		} else if (!strncmp(p,"NOP",plen)) {
			/* Ambiguous qualifier or keyword */
			pager(0);
			printf(cli$_abkeyw);
			pager(0);
			printf(cli$_cmdseg,p);

		} else if (!strncmp(p,"FOLLOW",plen)) {
			/* SET MODE FOLLOW */
			modes.follow = 1;

		} else if (!strncmp(p,"NOFOLLOW",plen)) {
			/* SET MODE NOFOLLOW */
			modes.follow = 0;

		} else if (!strncmp(p,"INFO",plen)) {
			/* SET MODE INFO */
			modes.info = 1;

		} else if (!strncmp(p,"NOINFO",plen)) {
			/* SET MODE NOINFO */
			modes.info = 0;

		} else if (!strncmp(p,"PAGER",plen)) {
			/* SET MODE PAGER */
			modes.pager = 1;

		} else if (!strncmp(p,"NOPAGER",plen)) {
			/* SET MODE NOPAGER */
			modes.pager = 0;

		} else if (!strncmp(p,"PROMPT",plen)) {
			/* SET MODE PROMPT */
			modes.prompt = 1;

		} else if (!strncmp(p,"NOPROMPT",plen)) {
			/* SET MODE NOPROMPT */
			modes.prompt = 0;

		} else if (!strncmp(p,"STOP",plen)) {
			/* SET MODE STOP */
			modes.stop = 1;

		} else if (!strncmp(p,"NOSTOP",plen)) {
			/* SET MODE NOSTOP */
			modes.stop = 0;

		} else {
			/* Unrecognized keyword */
			pager(0);
			printf(cli$_ivkeyw);
			pager(0);
			printf(cli$_cmdseg,p);
		}
	  }
}

/***********************************************************************
	cmd_show
***********************************************************************/

static void cmd_show(void)
{
	int matches;
	char *p;
	cmd_proc_t cmd;

	/*
	// Get SHOW keyword.
	*/

	if (p = get_next_token("_Show what: ")) {
		/*
		// Lookup the SHOW keyword.
		*/

		strupr(p);
		lookup_command(p, show_kw, &cmd, &matches);
	
		/*
		// Process based on number of matches.
		*/

		if (matches < 1) {
			/* Unrecognized keyword */
			pager(0);
			printf(cli$_ivkeyw);
			pager(0);
			printf(cli$_cmdseg,p);

		} else if (matches > 1) {
			/* Ambiguous qualifier or keyword */
			pager(0);
			printf(cli$_abkeyw);
			pager(0);
			printf(cli$_cmdseg,p);

		} else {
			/* Execute the command */
			cmd();
		}
	}
}

/***********************************************************************
	cmd_show_limit
***********************************************************************/

static void cmd_show_limit(void)
{
	if (!check_maxparm()) return;
	pager(0);
	printf("Limit = %d\n", modes.limit);
}

/***********************************************************************
	cmd_show_memory
***********************************************************************/

static void cmd_show_memory(void)
{
	if (!check_maxparm()) return;
	pager(0);
	printf("Memory available = %u bytes\n", _memavl());
}

/***********************************************************************
	cmd_show_modes
***********************************************************************/

static void cmd_show_modes(void)
{
	if (!check_maxparm()) return;
	pager(0); printf("Current modes:\n");
	pager(0); printf("  %sfollow\n", modes.follow ? "" : "no ");
	pager(0); printf("  %sinfo\n", modes.info ? "" : "no ");
	pager(0); printf("  %spager\n", modes.pager ? "" : "no ");
	pager(0); printf("  %sprompt\n", modes.prompt ? "" : "no ");
	pager(0); printf("  %sstop\n", modes.stop ? "" : "no ");
}

/***********************************************************************
	cmd_show_ports
***********************************************************************/

static void cmd_show_ports(void)
{
	int port_no;
	int port_cnt = 0;
	int cnt;

	if (!check_maxparm()) return;

	for (port_no = 0; port_no <= 2; port_no++) {
		if (ports[port_no].loaded) {
			port_cnt++;
			pager(0);
			printf("Port %d: %s\n", port_no,
				ports[port_no].module);

			pager(0);
			printf("        address range: %05lX:%05lX\n",
				ports[port_no].min_adr,
				ports[port_no].max_adr);

			cnt = ports[port_no].ca_file->count;
			pager(0);
			printf("        %d address comment%s\n",
				cnt, cnt == 1 ? "" : "s");

			cnt = ports[port_no].cc_file->count;
			pager(0);
			printf("        %d code comment%s\n",
				cnt, cnt == 1 ? "" : "s");
		}
	}

	if (port_cnt == 0) {
		pager(0);
		printf("No ports loaded\n");
	}
}

/***********************************************************************
	cmd_show_version
***********************************************************************/

static void cmd_show_version(void)
{
	if (!check_maxparm()) return;
	pager(0); printf(vgr__version, VERSION);
	pager(0); printf(vgr__author);
}

/***********************************************************************
	cmd_sort
***********************************************************************/

static void cmd_sort(void)
{
	bin5_t p;

	if (get_dec_number("_Port number: ", &p))
	  if (check_maxparm())
	    sort_port(p);
}

/***********************************************************************
	cmd_unassemble
***********************************************************************/

static void cmd_unassemble(void)
{
	modes.unmode = 1;
	cmd_uncontinue();
}

/***********************************************************************
	cmd_uncomment_address
***********************************************************************/

static void cmd_uncomment_address(void)
{
	int port_no;
	bin5_t port_adr;
	bin5_t adr;

	if (get_address(&adr))
	  if (check_maxparm())
	    if (xlate_adr(adr, &port_no, &port_adr))
	      DelTextLine(ports[port_no].ca_file, port_adr);
}

/***********************************************************************
	cmd_uncomment_code
***********************************************************************/

static void cmd_uncomment_code(void)
{
	int port_no;
	bin5_t port_adr;
	bin5_t adr;

	if (get_address(&adr))
	  if (check_maxparm())
	    if (xlate_adr(adr, &port_no, &port_adr))
	      DelTextLine(ports[port_no].cc_file, port_adr);
}

/***********************************************************************
	cmd_uncontinue
***********************************************************************/

static void cmd_uncontinue(void)
{
	bin5_t adr;

	if (get_address_opt(&adr))
	  if (check_maxparm()) {
	    SetWorkPtr(adr);
	    unany(&zero);
	  }
}

/***********************************************************************
	cmd_unload
***********************************************************************/

static void cmd_unload(void)
{
	bin5_t p;

	if (get_dec_number("_Port number: ", &p))
	  if (check_maxparm())
	    close_port(p);
}

/***********************************************************************
	cmd_unthread
***********************************************************************/

static void cmd_unthread(void)
{
	modes.unmode = 0;
	cmd_uncontinue();
}

/***********************************************************************
	ask
***********************************************************************/

static void ask(char *prompt, char *cmdline)
{
	/*
	// Prompt the user for input...
	*/

	*cmdline = '\0';

	do {
		if (modes.prompt) printf(prompt);
		gets(cmdline);
	} while (*cmdline == '\0');
}

/***********************************************************************
	lookup_command_exact
***********************************************************************/

static void lookup_command_exact(
	char *input,
	const keyword_t *kw,
	cmd_proc_t *proc,
	int *matches )
{
	*proc = NULL;
	*matches = 0;

	while (kw->keyword) {
		if (!strcmp(input, kw->keyword)) {
			if (*matches == 0) *proc = kw->procedure;
			(*matches)++;
		}
		kw++;
	}
}

/***********************************************************************
	lookup_command
***********************************************************************/

static void lookup_command(
	char *input,
	const keyword_t *kw,
	cmd_proc_t *proc,
	int *matches )
{
	int ilen;

	*proc = NULL;
	*matches = 0;
	ilen = strlen(input);

	while (kw->keyword) {
		if (!strncmp(input, kw->keyword, ilen)) {
			if (*matches == 0) *proc = kw->procedure;
			(*matches)++;
		}
		kw++;
	}
}

/***********************************************************************
	get_first_token
***********************************************************************/

static char *get_first_token(void)
{
	static char sep[] = " \t";
	char *token;

	/*
	// Get the first token.
	*/

	token = strtok(g_cmdline, sep);
	strupr(token);
	return token;
}

/***********************************************************************
	get_next_token
***********************************************************************/

static char *get_next_token(char *prompt)
{
	static char sep[] = " \t";
	char *token;

	/*
	// Get the next token.
	*/

	token = strtok(NULL, sep);

	/*
	// If no next token, ask the user and try again.
	*/

	if (token == NULL) {
		ask(prompt, g_cmdline);
		token = strtok(g_cmdline, sep);
	}

	strupr(token);

	/*
	// If still no next token...
	*/

	if (token == NULL) {
		/* Missing command parameters */
		pager(0);
		printf(cli$_insfprm);
	}

	return token;
}

/***********************************************************************
	get_next_token_opt
***********************************************************************/

static char *get_next_token_opt(void)
{
	static char sep[] = " \t";
	char *token;

	/*
	// Get the next token (optional).
	*/

	token = strtok(NULL, sep);
	strupr(token);
	return token;
}

/***********************************************************************
	get_rest_of_line
***********************************************************************/

static char *get_rest_of_line(char *prompt)
{
	static char sep[] = "";
	char *token;

	/*
	// Get the rest of the line.
	*/

	token = strtok(NULL, sep);

	/*
	// If nothing available, ask the user.
	*/

	if (token == NULL) {
		ask(prompt, g_cmdline);
		token = strtok(g_cmdline, sep);
	}

	/*
	// If still nothing...
	*/

	if (token == NULL) {
		/* Missing command parameters */
		pager(0);
		printf(cli$_insfprm);
	}

	return token;
}

/***********************************************************************
	get_rest_of_line_opt
***********************************************************************/

static char *get_rest_of_line_opt(void)
{
	static char sep[] = "";
	char *token;

	/*
	// Get the rest of the line (optional).
	*/

	token = strtok(NULL, sep);
	return token;
}

/***********************************************************************
	check_maxparm
***********************************************************************/

static int check_maxparm(void)
{
	static char sep[] = " \t";
	char *token;

	/*
	// Get the next token (there shouldn't be one).
	*/

	token = strtok(NULL, sep);
	strupr(token);

	/*
	// Well?
	*/

	if (token == NULL) {
		/* No more parameters - OK */
		return 1;
	} else {
		/* Too many parameters */
		pager(0);
		printf(cli$_maxparm);
		pager(0);
		printf(cli$_cmdseg,token);
		return 0;
	}
}

/***********************************************************************
	get_address
***********************************************************************/

static int get_address(bin5_t *adr)
{
	char *token;
	int status;

	if (token = get_next_token("_Address: ")) {
		status = convert_address(token,adr);
	} else {
		status = 0;
	}

	if (status) last_address = *adr;

	return status;
}

/***********************************************************************
	get_address_opt
***********************************************************************/

static int get_address_opt(bin5_t *adr)
{
	char *token;
	int status;

	if (token = get_next_token_opt()) {
		status = convert_address(token,adr);
	} else {
		status = 1;
		*adr = WorkPtr;
	}

	if (status) last_address = *adr;

	return status;
}

/***********************************************************************
	get_address_range
***********************************************************************/

static int get_address_range(bin5_t *start, bin5_t *end)
{
	char *token;
	char *p;
	int status = 1;

	if (token = get_next_token("_Address range: ")) {
		if (p = strchr(token,':')) {
			*p++ = '\0';

			if (*token) status = convert_address(token,start);
			else *start = 0x00000;
			
			if (*p) status = convert_address(p,end);
			else *end = 0xFFFFF;

		} else {
			status = convert_address(token,start);
			*end = *start;
		}
	} else {
		status = 0;
	}

	if (status) last_address = *start;

	return status;
}

/***********************************************************************
	get_dec_number
***********************************************************************/

static int get_dec_number(char *prompt, bin5_t *number)

{
	char *token;
	char *p;

	if (token = get_next_token(prompt)) {
		/*
		// Convert text to integer and check validity.
		*/

		*number = strtol(token, &p, 10);

		switch (*p) {
		    case '\0':
		    case ' ':
		    case '\t':
			break;

		    default:
			/* Invalid numeric value */
			pager(0);
			printf(cli$_ivchar);
			pager(0);
			printf(cli$_cmdseg,token);
			return 0;
		}
	} else {
		return 0;
	}

	return 1;
}

/***********************************************************************
	get_hex_number
***********************************************************************/

static int get_hex_number(char *prompt, bin5_t *number)
{
	char *token;
	char *p;

	if (token = get_next_token(prompt)) {
		/*
		// Convert text to integer and check validity.
		*/

		*number = str2adr(token, &p);

		switch (*p) {
		    case '\0':
		    case ' ':
		    case '\t':
			break;

		    default:
			/* Invalid numeric value */
			pager(0);
			printf(cli$_ivchar);
			pager(0);
			printf(cli$_cmdseg,token);
			return 0;
		}
	} else {
		return 0;
	}

	return 1;
}

/***********************************************************************
	convert_address
***********************************************************************/

static int convert_address(char *token, bin5_t *adr)
{
	char *p;
	int status = 1;

	if (!strcmp(token,".")) {
		/*
		// Current address.
		*/

		*adr = WorkPtr;

	} else if (!strcmp(token,"$")) {
		/*
		// Last address.
		*/

		*adr = last_address;

	} else {
		/*
		// Get specified address.
		*/

		*adr = str2adr(token, &p);

		switch (*p) {
		    case '\0':
		    case ' ':
		    case '\t':
			break;

		    default:
			/* Invalid numeric value */
			pager(0);
			printf(cli$_ivchar);
			pager(0);
			printf(cli$_cmdseg,token);
			status = 0;
		}
	}

	return status;
}
