/* sad.c -- Saturn Disassembler.

This file is part of SAD, the Saturn Disassembler package.

SAD is not distributed by the Free Software Foundation. Do not ask
them for a copy or how to obtain new releases. Instead, send e-mail to
the address below. SAD is merely covered by the GNU General Public
License.

Please send your comments, ideas, and bug reports to
Jan Brittenson <bson@ai.mit.edu>

*/


/* Copyright (C) 1990 Jan Brittenson.

SAD is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 1, or (at your option) any later
version.

SAD is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with SAD; see the file COPYING.  If not, write to the Free
Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. */


#include <stdio.h>
#include "macros.h"
#include "formats.h"
#include "sad.h"


/* Options */
int
  opt_pass1 = TRUE,		/* Run pass 1 */
  opt_formats = FALSE,		/* Run pass F */
  opt_debug = FALSE,		/* Output debug info */
  opt_locals = TRUE,		/* Local symbols */
  opt_code = TRUE,		/* Include code in disassembly */
  opt_startpt,			/* Staring address */
  opt_endpt,			/* End address */
  opt_symbols = TRUE,		/* Symbols in code, or symbols in comments */
  opt_symdef = TRUE,		/* Include definition of referenced symbols */
  opt_opcode = TRUE,		/* Include opcodes/addresses in listing */
  opt_comments = TRUE,		/* Generate auto comments */
  opt_commentcol = COMMENT_COL,	/* Comment column */
  opt_alonzo = FALSE,		/* Alonzo-style formatting */
  opt_xref = FALSE;		/* Include full XREF */


/* Various RPL related variables */
int
  rpl_new_ind_level = 0,	/* RPL indent level for next instruction */
  rpl_ind_level = 0;		/* Current RPL indent level */


/* Various, variables. Most of the context
 * can be found in misc.c.
 */
int
  pass,				/* Current pass (PASS1 or PASS2) */
  org_pc;			/* PC at start of instruction */

FILE
  *core_file;			/* File used to access .core */

static char
  l_code[1024],			/* List buf: code */
  *l_codep,			/* Pointer to above */
  l_instr[1024],		/* List buf: instr */
  l_comment[1024],		/* List buf: comment */
  *l_commentp,			/* Pointer to above */

  *flen[] = {
    "0", "1", "b", "x", "4", "a", "6", "7", "8", "9", "10",
    "11", "12", "13", "14", "w" },

  *args_0e[] = {
    "b, a", "c, b", "a, c", "c, d", "a, b", "b, c", "c, a", "d, c" },

  *std_fields[] = {
    "p", "wp", "xs", "x", "s", "m", "b", "w", "*", "*", "*", "*",
    "*", "*", "*", "a"},

  *std_rno[] = {"r0", "r1", "r2", "r3", "r4", "*", "*", "*",
		  "r0", "r1", "r2", "r3", "r4", "*", "*", "*"};

  
/* Comments are considered to be symbols,
 * and reside in the comment table. See misc.c.
 */
struct symbol
  **nextsym = NULL,		/* Next symbol to expect */
  **nextcom = NULL,		/* Next comment to expect */
  **curfmtref = NULL,		/* Current format ref */
  **nextfmt = NULL;		/* Next format directive */

struct format
  *curfmt = NULL;		/* Current format active */


/* Add to comment field */
void comment(ctrl, a1, a2, a3, a4)
  char *ctrl, *a1, *a2, *a3, *a4;
{
  if(!opt_comments)
    return;

  *l_commentp++ = ' ';

  sprintf(l_commentp, ctrl, a1, a2, a3, a4);
  l_commentp += strlen(l_commentp);
}


/* Open core file */
static void open_core(fname)
  char *fname;
{
  if(!(core_file = fopen(fname, "r")))
    {
      perror(fname);
      exit(1);
    }

  if(fseek(core_file, (long) opt_startpt, 0) < 0)
    {
      error("address %x not in corefile.\n", opt_startpt);
      exit(1);
    }
}


/* Set new PC */
static void set_pc(addr)
  int addr;
{
  if(fseek(core_file, (long) addr, 0) < 0)
    {
      error("address %x not in corefile.\n", addr);
      exit(1);
    }

  pc = addr;
}


/* Load comments. */
static void load_comments(comfname)
  char *comfname;
{
  FILE *comfile;
  char lbuf[132], *cp;
  struct symbol *comtemp, **comrefptr, *comptr;
  int seqaddr, seq;
  extern char *malloc();


  if(!(comfile = fopen(comfname, "r")))
    return;
      
  comroot = NULL;
  for(ncomments = seqaddr = seq = 0; !feof(comfile); )
    {
      fgets(lbuf, sizeof lbuf, comfile);
      if(feof(comfile))
	break;

      lbuf[strlen(lbuf)-1] = '\0';
      comtemp = (struct symbol *) malloc(sizeof(struct symbol));
      comtemp->link = comroot;
      comroot = comtemp;

      cp = lbuf;
      hexstrtoi(&cp, &comtemp->val);

      if(!*cp)
	{
	  comroot = comtemp->link;
	  free(comtemp);
	  continue;
	}

      comtemp->seq = ++seq;
      comtemp->type = (*cp++ == '=' ? T_COM_MAJOR : T_COM_MINOR);
      comtemp->id = strdup(cp);
      ncomments++;
    }

  fclose(comfile);


  /* Create reference array, end with NULL guard */
  comref = (struct symbol **) malloc(sizeof(struct symbol *) * (ncomments+1));

  for(comrefptr = comref, comptr = comroot; comptr; comptr = comptr->link)
    *comrefptr++ = comptr;

  *comrefptr++ = NULL;


  /* Sort the reference array in ascending value order */
  qsort((char *) comref, ncomments, sizeof(struct symbol *), comcmp);
}


/* Add node to xref list */
static void append_xref(sym, xrefval)
  struct symbol *sym;
  int xrefval;
{
  register struct xrefaddr
    *xrefnode = (struct xrefaddr *) malloc(sizeof(struct xrefaddr));

  if(!xrefnode)
    {
      perror("malloc");
      return;
    }

  xrefnode->val = xrefval;
  xrefnode->link= NULL;

  if(!sym->xrefhead)
    sym->xrefhead = sym->xreftail = xrefnode;
  else
    {
      sym->xreftail->link = xrefnode;
      sym->xreftail = xrefnode;
    }
}

  
/* Convert address to hex */
static char *addrtohex(addr)
  int addr;
{
  static char addrbuf[32];


  if(abs(addr) < 0xa)
    sprintf(addrbuf, "%d", addr);
  else
    sprintf(addrbuf, "#%x", addr);

  return(addrbuf);
}


/* Map address to symbol with or w/o offset, if feasible */
char *symbolic(addr, maxoffs)
  int addr, maxoffs;
{
  static char symbuf[132];
  int bot, top, cent;
  struct symbol *closest;


  if(!symref || !symroot || !nsymbols)
    {
      strcpy(symbuf, addrtohex(addr));
      return(symbuf);
    }

  /* Look up symbol */
  top = nsymbols-1;
  bot = 0;
  cent = nsymbols >> 1;

  for(; top > bot+1; cent = bot + ((top-bot) >> 1))
    if(addr > symref[cent]->val)
      bot = cent;
    else
      if(addr < symref[cent]->val)
	top = cent;
      else
	{
	  int islocal = (symref[cent]->type & T_SYM_LOCAL) != 0;

	  if(opt_symbols)
	    {
	      strcpy(symbuf,
		      (islocal ? local_id(symref[cent]) : symref[cent]->id));

	      if(!islocal)
		comment(addrtohex(addr));
	    }
	  else
	    {
	      comment("%s",
		      (islocal ? local_id(symref[cent]) : symref[cent]->id));

	      strcpy(symbuf, addrtohex(addr));
	    }
	  
	  if(opt_xref && pass == PASS2)
	    append_xref(symref[cent], org_pc);

	  symref[cent]->ref++;
	  return(symbuf);
	}


  /* Now, we use bot if it's close enough, otherwise top if it's close
   * enough. This since we always prefer + offsets to - offsets.
   */
  if(abs(addr - symref[bot]->val) < maxoffs && addr <= symref[top]->val)
    closest = symref[bot];
  else
    if(abs(symref[top]->val - addr) < maxoffs)
      closest = symref[top];
    else
      {
	if(opt_locals && pass == PASS1 &&
	   addr >= opt_startpt && addr <= opt_endpt)
	  {
	    add_local_symbol(addr, T_SYM_CODE);
	    strcpy(symbuf, local_id(symroot));
	  }
	else
	  strcpy(symbuf, addrtohex(addr));

	return(symbuf);
      }
  
  if(addr == closest->val)
    { 
      int islocal = (closest->type & T_SYM_LOCAL) != 0;

      if(opt_symbols)
	{
	  strcpy(symbuf, (islocal ? local_id(closest) : closest->id));

	  if(!islocal)
	    comment(addrtohex(addr));
	}
      else
	{
	  comment("%s", (islocal ? local_id(closest) : closest->id));

	  strcpy(symbuf, addrtohex(addr));
	}
      
      if(opt_xref && pass == PASS2)
	append_xref(closest, org_pc);

      closest->ref++;
      return(symbuf);
    }
  else
    /* Build expression in local buffer, and recurse for offset */
    {
      char lsymbuf[132];
      
      if(opt_locals && pass == PASS1 &&
	 addr >= opt_startpt && addr <= opt_endpt)
	{
	  add_local_symbol(addr, T_SYM_CODE);
	  strcpy(symbuf, local_id(symroot));
	}
      else
	{
	  closest->ref++;
	  sprintf(lsymbuf, "%s%c%s",
		  (closest->type & T_SYM_LOCAL ?
		   local_id(closest) : closest->id),
		  (addr > closest->val ? '+' : '-'),
		  symbolic(abs(closest->val - addr), NOREL));
	  
	  if(opt_symbols)
	    {
	      strcpy(symbuf, lsymbuf);
	      
	      if(!(closest->type & T_SYM_LOCAL))
		comment(addrtohex(addr));
	    }
	  else
	    {
	      comment(lsymbuf);
	      strcpy(symbuf, addrtohex(addr));
	    }
	}

      if(opt_xref)
	append_xref(closest, org_pc);
    }
  
  return(symbuf);
}


/* Fetch nibbles as unsigned integer */
get_unibbles(n)
  register n;
{
  register data = 0;
  char *cp, rbuf[64], *pt;

  if(!n)
    return;

  if(n == 1)
    rbuf[0] = fgetc(core_file);
  else
    fread(rbuf, 1, n, core_file);

  pt = rbuf+n;
  pc += n;
  cp = (l_codep += n);

  while(n--)
    {
      register nib = *--pt & 0xf;

      data <<= 4;
      *--cp = I_TO_HEX(nib);
      data |= nib;
    }

  return(data);
}


/* Fetch nibbles as integer and extend sign */
get_nibbles(n)
  register n;
{
  register data = 0;
  char *cp, rbuf[64], *pt;
  int nib;

  if(n <= 0)
    return;

  if(n == 1)
    rbuf[0] = fgetc(core_file);
  else
    fread(rbuf, 1, n, core_file);

  pt = rbuf+n;
  pc += n;
  cp = (l_codep += n);

  nib = *--pt & 0xf;
  *--cp = I_TO_HEX(nib);

  if(nib & 010)
    data = (-1 << 4);

  for(data |= nib; --n; )
    {
      nib = *--pt & 0xf;

      *--cp = I_TO_HEX(nib);
      data <<= 4;
      data |= nib;
    }

  return(data);
}


/* Decode 80x instruction */
static void decode_group_80()
{
  static char *i807[] =
    { "out.s\tc", "out.x\tc", "in.4\ta", "in.4\tc",
	"uncnfg", "config", "move.a\tid, c", "shutdn" };

  int op3, op4, op5, op6;


  op3 = get_unibbles(1);

  if(op3 < 8)
    {
      strcpy(l_instr, i807[op3]);
      return;
    }

  switch(op3)
    {
    case 9:  strcpy(l_instr, "add.a\tp+1, c"); break;
    case 0xa: strcpy(l_instr, "reset"); break;
    case 0xb: strcpy(l_instr, "buscc"); break;
    case 0xc: sprintf(l_instr, "move.1\tp, c, %d", get_unibbles(1)); break;
    case 0xd: sprintf(l_instr, "move.1\tc, %d, p", get_unibbles(1)); break;
    case 0xe: strcpy(l_instr, "sreq"); break;
    case 0xf: sprintf(l_instr, "swap.1\tp, c, %d", get_unibbles(1)); break;

    case 8:

      op4 = get_unibbles(1);

      switch(op4)
	{
	case 0: strcpy(l_instr, "inton"); break;
	case 3: strcpy(l_instr, "buscb"); break;
	case 4: sprintf(l_instr, "clrb\t%s, a",
			symbolic(get_unibbles(1), NOREL)); break;
	case 5: sprintf(l_instr, "setb\t%s, a",
			symbolic(get_unibbles(1), NOREL)); break;
	case 8: sprintf(l_instr, "clrb\t%s, c",
			symbolic(get_unibbles(1), NOREL)); break;
	case 9: sprintf(l_instr, "setb\t%s, c",
			symbolic(get_unibbles(1), NOREL)); break;

	case 0xc: strcpy(l_instr, "jump.a\t@a"); break;
	case 0xd: strcpy(l_instr, "buscd"); break;
	case 0xe: strcpy(l_instr, "jump.a\t@c"); break;
	case 0xf: strcpy(l_instr, "intoff"); break;
	case 1:

	  sprintf(l_instr, "rsi%s", (get_unibbles(1) ? "*" : ""));
	  break;

	case 2:

	  op5 = get_unibbles(1);
	  op6 = get_unibbles(op5+1);

	  sprintf(l_instr, "move.p%d\t%s, a", op5+1, symbolic(op6, DATAREL));

	  if(op5 > 7)
	    comment("*** truncated ***");

	  break;

	case 6:
	case 7:
	case 0xa:
	case 0xb:

	  {
	    char s[132];

	    op5 = get_unibbles(1);
	    op6 = get_nibbles(2);
	    strcpy(s, symbolic(op5, NOREL));
	    
	    if(op6)
	      sprintf(l_instr, "brb%c\t%s, %c, %s",
		      (op4 & 1 ? 's' : 'c'),
		      s,
		      (op4 >= 0xa ? 'c' : 'a'),
		      symbolic((org_pc+5+op6) & 0xfffff, JUMPREL));
	    else
	      sprintf(l_instr, "retb%c\t%s, %c",
		      (op4 & 1 ? 's' : 'c'),
		      s,
		      (op4 >= 0xa ? 'c' : 'a'));
	    
	    if(opt_formats && op6 && pass == PASSF)
	      add_auto_format(org_pc + 5 + op6, T_FMT_CODE);
	    
	    break;
	  }
	}
    }
}

	  
/* Decode instructions starting with 8-f */
static void decode_8_thru_f(op1)
  int op1;
{
  int op, op2, op3, op4, op5, op6;
  static char
    *args_i81[] = {"a", "b", "c", "d"},
    *fixed_81b[] =
      {"*", "*", "jump.a\ta", "jump.a\tc", "move.a\tpc, a",
	 "move.a\tpc, c", "swap.a\ta, pc", "swap.a\tc, pc" },
    *fields_82_83[] = 
      { "[]", "[xm]", "[sb]", "[xm,sb]", "[sr]", "[xm,sr]",
	  "[sb,sr]", "[xm,sb,sr]",
	  "[mp]", "[xm,mp]", "[sb,mp]", "[xm,sb,mp]", "[sr,mp]", "[xm,sr,mp]",
	  "[sb,sr,mp]", "[xm,sb,sr,mp]" },
    *srcdest[] = {"a, b", "b, c", "c, a", "d, c"},
    smap[] = "bcacabcdabcd",
    dmap[] = "abcdabcdbcac",
    *rr_instr[] = {"sln", "srn", "neg", "not" };


  op2 = get_unibbles(1);

  if(op1 == 8)
    switch(op2)
      {
      case 0: decode_group_80(); return;
      case 1:
	
	op3 = get_unibbles(1);
	
	if(op3 <= 3)
	  {
	    sprintf(l_instr, "rln.w\t%s", args_i81[op3]);
	    return;
	  }
	
	if(op3 <= 7)
	  {
	    sprintf(l_instr, "rrn.w\t%s", args_i81[op3 & 3]);
	    return;
	  }
	
	if(op3 >= 0xc)
	  {
	    sprintf(l_instr, "srb.w\t%s", args_i81[op3 & 3]);
	    return;
	  }
	
	op4 = get_unibbles(1);
	
	switch(op3)
	  {
	  case 8:
	    
	    op5 = get_unibbles(1);
	    op6 = get_unibbles(1);
	    
	    sprintf(l_instr, "%s.%s\t%d, %s",
		    (op5 < 4 ? "add" : "sub"),
		    std_fields[op4],
		    op6+1,
		    args_i81[op5 & 3]);
	    break;
	    
	  case 9:
	    
	    op5 = get_unibbles(1);
	    sprintf(l_instr, "srb.%s\t%s",
		    std_fields[op4],
		    args_i81[op5 & 3]);
	    break;
	    
	  case 0xa:
	    
	    op5 = get_unibbles(1);
	    op6 = get_unibbles(1);
	    
	    switch(op5)
	      {
	      case 0:
	      case 2:
		
		sprintf(l_instr, "%s.%s\t%c, %s",
			(!op5 ? "move" : "swap"),
			std_fields[op4],
			(op6 > 3 ? 'c' : 'a'),
			std_rno[op6]);
		break;
		
	      case 1:
		
		sprintf(l_instr, "move.%s\t%s, %c",
			std_fields[op4],
			std_rno[op6],
			(op6 > 3 ? 'c' : 'a'));
	      }
	    
	    break;
	    
	  case 0xb:
	    
	    strcpy(l_instr, fixed_81b[op4 & 7]);
	    break;
	  }
	return;
	
      case 2:
	
	sprintf(l_instr, "clrb\t%s", fields_82_83[get_unibbles(1) & 017]);
	return;
	
      case 3:
	
	op3 = get_unibbles(1);
	op4 = get_nibbles(2);
	
	sprintf(l_instr, (op4 ? "brbc\t%s, %s" : "retbc\t%s"),
		fields_82_83[op3],
		(op4 ? symbolic((org_pc + 3 + op4) & 0xfffff, JUMPREL) :
		 "*"));

	if(opt_formats && op4 && pass == PASSF)
	  add_auto_format(org_pc + 3 + op4);

	return;
	
      case 4:
      case 5:
	  
	sprintf(l_instr, "%s\t%s, st",
		(op2 == 4 ? "clrb" : "setb"),
		symbolic(get_unibbles(1), NOREL));
	return;
	
      case 6:
      case 7:
      case 8:
      case 9:
	
	{
	  static char *xrbn[] =
	    {"brbc\t%s, st, %s", "brbs\t%s, st, %s",
	       "brne.1\tp, %s, %s", "breq.1\tp, %s, %s"},
	    *xrbnr[] =
	      {"retbc\t%s, st", "retbs\t%s, st",
		 "retne.1\tp, %s", "reteq.1\tp, %s"};
	  char
	      s1[132];
	  
	  op3 = get_unibbles(1);
	  op4 = get_nibbles(2);
	  
	  if(op4)
	    strcpy(s1, symbolic((org_pc + 3 + op4) & 0xfffff, JUMPREL));

	  sprintf(l_instr, (op4 ? xrbn : xrbnr)[(op2 - 6) & 3],
		  symbolic(op3, NOREL),
		  s1);

	  if(opt_formats && op4 && pass == PASSF)
	    add_auto_format(org_pc + 3 + op4, T_FMT_CODE);
		  
	  return;
	}
	
      case 0xc:
	
	op3 = get_nibbles(4);
	
	sprintf(l_instr, "jump.4\t%s",
		symbolic((org_pc + 2 + op3) & 0xfffff, JUMPREL));

	if(opt_formats && pass == PASSF)
	  add_auto_format(org_pc + 2 + op3, T_FMT_CODE);

	return;
	
      case 0xd:
	
	op3 = get_unibbles(5);
	
	sprintf(l_instr, "jump.a\t%s", symbolic(op3, JUMPREL));

	if(opt_formats && pass == PASSF)
	  add_auto_format(op3, T_FMT_CODE);

	return;
	
      case 0xe:
	
	op3 = get_nibbles(4);
	
	sprintf(l_instr, "call.4\t%s",
		symbolic((org_pc + 6 + op3) & 0xfffff, JUMPREL));

	if(opt_formats && pass == PASSF)
	  add_auto_format(org_pc + 6 + op3, T_FMT_CODE);

	return;
	
      case 0xf:
	
	op3 = get_unibbles(5);
	
	sprintf(l_instr, "call.a\t%s", symbolic(op3, JUMPREL));

	if(opt_formats && pass == PASSF)
	  add_auto_format(op3, T_FMT_CODE);

	return;
      }


  /* Now we've got either 8[AB], 9x, or [A-F]x */
  
  if(op1 <= 9)
    {
      static char
	*fields[] = {"a", "a", "", "", "", "",
		       "p", "wp", "xs", "x", "s", "m", "b", "w",
		       "p", "wp", "xs", "x", "s", "m", "b", "w"},
	*cc[] = {"eq", "ne", "z", "nz", "gt", "lt", "ge", "le"};

      char *ctrl1, *ctrl2, s[132];
      int ccix;

      op = (op1 << 4) + op2;
      op3 = get_unibbles(1);
      op4 = get_nibbles(2);

      if(op4)
	{
	  ctrl1 = "br%s.%s\t%c, %s";
	  ctrl2 = "br%s.%s\t%s, %s";
	  strcpy(s, symbolic((org_pc+3+op4) & 0xfffff, JUMPREL));

	  if(opt_formats && pass == PASSF)
	    add_auto_format(org_pc + 3 + op4, T_FMT_CODE);
	}
      else
	{
	  ctrl1 = "ret%s.%s\t%c";
	  ctrl2 = "ret%s.%s\t%s";
	}

      ccix = (op3 >> 2) | (op == 0x8b || op >= 0x98 ? 4 : 0);

      if((op == 0x8a || (op < 0x98 && op >= 0x90)) && op3 >= 8)
	sprintf(l_instr, ctrl1,
		cc[ccix],
		fields[op - 0x8a],
		srcdest[op3 & 3][0],
		s);
      else
	sprintf(l_instr, ctrl2,
		cc[ccix],
		fields[op - 0x8a],
		srcdest[op3 & 3],
		s);

      return;
    }


  if(op1 < 0xc)
    op3 = get_unibbles(1);
  else
    op3 = op2;

  op = (op1 << 4) + op2;

  if(op <= 0xa7 || op1 == 0xc)
    {
      if(op3 >= 0xc)
	sprintf(l_instr, "dec.%s\t%c",
		std_fields[(op1 == 0xc ? 0xf : op2 & 7)],
		dmap[op3-0xc]);
      else
	sprintf(l_instr, "add.%s\t%c, %c",
		std_fields[(op1 == 0xc ? 0xf : op2 & 7)],
		smap[op3], dmap[op3]);

      return;
    }

  if(op <= 0xaf || op1 == 0xd)
    {
      int mapix = op3 - 4;

      if(op3 >= 8)
	mapix += 4;

      if(op3 < 4)
	sprintf(l_instr, "clr.%s\t%c",
		std_fields[(op1 == 0xd ? 0xf : op2 & 7)],
		dmap[op3]);
      else
	if(op3 < 0xc)
	  sprintf(l_instr, "move.%s\t%c, %c",
		  std_fields[(op1 == 0xd ? 0xf : op2 & 7)],
		  smap[mapix], dmap[mapix]);
	else
	  sprintf(l_instr, "swap.%s\t%c, %c",
		  std_fields[(op1 == 0xd ? 0xf : op2 & 7)],
		  smap[op3-0xc], dmap[op3-0xc]);

      return;
    }


  if(op < 0xb8 || op1 == 0xe)
    {
      if(op3 >= 3 && op3 < 8)
	sprintf(l_instr, "inc.%s\t%c",
		std_fields[(op1 == 0xe ? 0xf : op2 & 7)],
		dmap[op3 & 3]);
      else
	if(op3 >= 0xc)
	  sprintf(l_instr, "subn.%s\t%c, %c",
		  std_fields[(op1 == 0xe ? 0xf : op2 & 7)],
		  smap[op3 & 3], dmap[op3 & 3]);
	else
	  sprintf(l_instr, "sub.%s\t%c, %c",
		  std_fields[(op1 == 0xe ? 0xf : op2 & 7)],
		  smap[op3], dmap[op3]);

      return;
    }

  sprintf(l_instr, "%s.%s\t%c",
	  rr_instr[(op3 >> 2) & 3],
	  std_fields[(op1 == 0xf ? 0xf : op2 & 7)],
	  dmap[op3 & 3]);
}

      
/* Decode instructions starting with '1' */
static void decode_group_1()
{
  int op2, op3, op4;

  static char
    *i_f1[] = { "move.w", "move.w", "swap.w" },
    *args_f1[] =
      { "a, r0", "a, r1", "a, r2", "a, r3", "a, r4", "*", "*", "*",
	  "c, r0", "c, r1", "c, r2", "c, r3", "c, r4", "*", "*", "*" },

    *args_rf1[] =
      { "r0, a", "r1, a", "r2, a", "r3, a", "r4, a", "*", "*", "*",
	  "r0, c", "r1, c", "r2, c", "r3, c", "r4, c", "*", "*", "*" },
      
    *args_f3[] =
      { "a, d0", "a, d1", "a, d0", "a, d1",
	  "c, d0", "c, d1", "c, d0", "c, d1" },

    *args_f4[] =
      { "a, @d0", "a, @d1", "@d0, a", "@d1, a", "c, @d0", "c, @d1",
	  "@d0, c", "@d1, c" };


  op2 = get_unibbles(1);


  if(op2 < 8)
    op3 = get_unibbles(1);

  switch(op2)
    {
    case 0:
    case 1:
    case 2:

      sprintf(l_instr, "%s\t%s", i_f1[op2],
	      (op2 == 1 ? args_rf1 : args_f1)[op3]);
      break;

    case 3:

      sprintf(l_instr, "%s.%c\t%s",
	      (op3 & 2 ? "swap" : "move"),
	      (op3 >= 8 ? '4' : 'a'),
	      args_f3[op3 & 7]);
      break;

    case 4:

      sprintf(l_instr, "%s\t%s",
	      (op3 >= 8 ? "move.b" : "move.a"),
	      args_f4[op3 & 7]);
      break;

    case 5:

      op4 = get_unibbles(1);

      if(op3 >= 8)
	sprintf(l_instr, "move.%d\t%s", op4+1, args_f4[op3 & 7]);
      else
	sprintf(l_instr, "move.%s\t%s",
		std_fields[op4],
		args_f4[op3]);
      break;

    case 6: sprintf(l_instr, "add.a\t%d, d0", op3+1); break;
    case 7: sprintf(l_instr, "add.a\t%d, d1", op3+1); break;
    case 8: sprintf(l_instr, "sub.a\t%d, d0", get_unibbles(1)+1); break;
    case 9: 

      sprintf(l_instr, "move.2\t%s, d0",
	      symbolic(get_unibbles(2), DATAREL));
      break;

    case 0xa:

      sprintf(l_instr, "move.4\t%s, d0",
	      symbolic(get_unibbles(4), DATAREL));
      break;

    case 0xb:

      sprintf(l_instr, "move.5\t%s, d0",
	      symbolic(get_unibbles(5), DATAREL));
      break;

    case 0xc: sprintf(l_instr, "sub.a\t%d, d1", get_unibbles(1)+1); break;
    case 0xd: 

      sprintf(l_instr, "move.2\t%s, d1",
	      symbolic(get_unibbles(2), DATAREL));
      break;

    case 0xe:

      sprintf(l_instr, "move.4\t%s, d1",
	      symbolic(get_unibbles(4), DATAREL));
      break;

    case 0xf:

      sprintf(l_instr, "move.5\t%s, d1",
	      symbolic(get_unibbles(5), DATAREL));
      break;

    }
}


/* Decode one instruction */
static void decode_instr()
{
  int
    op0 = get_unibbles(1),
    op1, op2, op3;

  static char
    *fixed_0[] = {
      "retsetxm", "ret", "retsetc", "retclrc", "sethex", "setdec",
      "push.a\tc", "pop.a\tc", "clr.x\tst", "move.x\tst, c",
      "move.x\tc, st", "swap.x\tc, st", "inc.1\tp", "dec.1\tp",
      "***", "reti"};
  

  switch(op0)
    {
    case 0:

      op1 = get_unibbles(1);

      if(op1 != 0xe)
	{
	  strcpy(l_instr, fixed_0[op1]);
	  return;
	}

      op2 = get_unibbles(1);
      op3 = get_unibbles(1);
      
      sprintf(l_instr, "%s.%s\t%s", (op3 > 7 ? "or" : "and"),
	      std_fields[op2 & 017], args_0e[op3 & 7]);

      return;

    case 1:

      decode_group_1();
      return;

    case 2:

      op2 = get_unibbles(1);

      sprintf(l_instr, "move.1\t%d, p", op2);
      return;

    case 3:

      op2 = get_unibbles(1);
      op3 = get_unibbles(op2 + 1);

      sprintf(l_instr, "move.p%d\t%s, c", op2+1, symbolic(op3, DATAREL));

      if(op2 > 7)
	comment("*** truncated ***");

      return;

    case 4:

      op2 = get_nibbles(2);

      if(op2 == 0x20)
	strcpy(l_instr, "nop3");
      else
	if(op2)
	  {
	    sprintf(l_instr, "brcs\t%s",
		    symbolic((org_pc+1+op2) & 0xfffff, JUMPREL));

	    if(opt_formats && pass == PASSF)
	      add_auto_format(org_pc + 1 + op2, T_FMT_CODE);
	  }
	else
	  strcpy(l_instr, "retcs");

      return;

    case 5:

      if(op2 = get_nibbles(2))
	{
	  sprintf(l_instr, "brcc\t%s",
		  symbolic((org_pc+1+op2) & 0xfffff, JUMPREL));

	  if(opt_formats && pass == PASSF)
	    add_auto_format(org_pc + 1 + op2, T_FMT_CODE);
	}
      else
	strcpy(l_instr, "retcc");

      return;

    case 6:

      op2 = get_nibbles(3);

      if(op2 == 0x300)
	strcpy(l_instr, "nop4");
      else if(op2 == 0x400)
	{
	  get_unibbles(1);
	  strcpy(l_instr, "nop5");
	}
      else
	{
	  sprintf(l_instr, "jump.3\t%s",
		  symbolic((org_pc+1+op2) & 0xfffff, JUMPREL));

	  if(opt_formats && pass == PASSF)
	    add_auto_format(org_pc + 1 + op2, T_FMT_CODE);
	}

      return;

    case 7:

      op2 = get_nibbles(3);

      sprintf(l_instr, "call.3\t%s",
	      symbolic((org_pc+4+op2) & 0xfffff, JUMPREL));

      if(opt_formats && pass == PASSF)
	add_auto_format(org_pc + 4 + op2, T_FMT_CODE);
      
      return;
      
    default:
      
      decode_8_thru_f(op0);
      return;
    }
}


/* Return position where cursor would be, were we to
 * print this string following CR.
 */
static prpos(str)
  register char *str;
{
  register pos;
  
  for(pos = 0; *str;)
    switch(*str++)
      {
      case '\t':
	
	pos = (pos & ~7) + 8;
	break;
	
      case '\n':
	
	pos = 0;
	break;
	
      default:
	
	pos++;
      }
  
  return(pos);
}


/* Tabulate to specific column.
 * Note: column must be an even multiple of 8.
 */
void tabulate(from, to)
  register  from, to;
{
  while(from < to)
    {
      putchar('\t');
      from = (from & ~7) + 8;
    }
}


/* Tell from instruction buffer whether disassembly failed */
static failed(str)
  register char *str;
{
  char delim;
  
  if(!str || !*str)
    return(TRUE);
  
  for(; *str; str++)
    if(*str == '\\' && str[1])
      str++;
    else
      if(*str == '"' || *str == '\'')
	for(delim = *str++; *str && *str != delim; str++)
	  if(*str == '\\' && str[1])
	    str++;
	  else ;
      else
	if(*str == '*')
	  return(TRUE);
  
  return(FALSE);
}


/* Edit out (sigh) spaces following colons. */
static void edit_out_spaces_following_commas(str)
  register char *str;
{
  register char delim, *dst;
  
  if(!str || !*str)
    return;
  
  for(dst = str; *str; *dst++ = *str++)
    if(*str == '\\' && str[1])
      *dst++ = *str++;
    else
      if(*str == '"' || *str == '\'')
	for(delim = *dst++ = *str++; *str && *str != delim; *dst++ = *str++)
	  if(*str == '\\' && str[1])
	    *dst++ = *str++;
	  else ;
      else
	if(*str == ',')
	  {
	    *dst++ = *str++;
	    str = byspace(str);
	  }
	else
	  if(*str == ';')
	    {
	      while(*dst++ = *str++);
	      return;
	    }
  
  *dst = '\0';
}


/* Return spaces according to current indentation level */
static char *rpl_indent()
{
  static char spaces[256];
  
#ifndef SYSV
  register char *cp;
#endif
  
  int ind;
  
  if(rpl_ind_level < 0)
    rpl_ind_level = 0;

  ind = rpl_ind_level * 2;
  
#ifdef SYSV
  memset(spaces, ' ', ind);
  spaces[ind] = '\0';
#else
  for(cp = spaces; cp < spaces + ind; *cp++ = ' ');
  *cp = '\0';
#endif
  
  return(spaces);
}


/* Print disassembled line */
static void print_line()
{
  char
    *lcp = l_code,
    code[1024], lbuf[1024],
    pc_str[32], pc_str_blank[32],
    *indstr;
  int
    lpos, max_codelen;
  
  
  /* No printing unless pass 2 and code enabled */
  if(pass != PASS2 || !opt_code)
    {
      /* Attempted to pass a format point? If so, adjust the PC */
      if(nextfmt && (*nextfmt) && (*nextfmt)->val < pc && 
	 (*nextfmt)->val > org_pc)
	{
	  set_pc((*nextfmt)->val);
	  prevfmt = NULL;
	  fmtexpire_tag = 0xffffff;
/*	  rpl_new_ind_level--;
	  rpl_ind_level--; */
	}
      
      l_commentp = l_comment;
      return;
    }

  max_codelen = (opt_alonzo ? 12 : 7);
  indstr = rpl_indent();
  
  *l_codep = '\0';

  if(opt_alonzo)
    {
      sprintf(pc_str, "#%05x:", org_pc);
      sprintf(pc_str_blank, "#%05x:   ", org_pc);
      edit_out_spaces_following_commas(l_instr);
    }
  else
    {
      sprintf(pc_str, "%05x", org_pc);
      sprintf(pc_str_blank, "%05x   ", org_pc);
    }
  
  /* Attempted to pass a format point? If so,
   * we need to adjust the PC, by outputting a dummy data op.
   * and wind back the PC. Next time around, we'll get the real
   * thing.
   */
  if(nextfmt && (*nextfmt) && (*nextfmt)->val < pc && 
     (*nextfmt)->val > org_pc)
    {
      int nibbles = (*nextfmt)->val - org_pc, n;
      register char *cp, *bp;

      set_pc((*nextfmt)->val);
      prevfmt = NULL;
      fmtexpire_tag = 0xffffff;
/*      rpl_new_ind_level--;
      rpl_ind_level--; */

      /* Create data statement */
      if(nibbles <= 15)
	sprintf(l_instr, "data.%s\t", flen[nibbles]);
      else
	sprintf(l_instr, "data.%d\t", nibbles);

      bp = l_instr + strlen(l_instr);
      cp = l_code + nibbles;

      *cp-- = '\0';

      /* Skip leading zeroes */
      while(cp > l_code && *cp == '0')
	cp--;

      /* Make it hex, if necessary */
      if(!(cp == l_code && *cp <= '9'))
	*bp++ = '#';

      /* Duplicate */
      while(cp >= l_code)
	*bp++ = *cp--;

      *bp = '\0';

      /* Add comment */
      comment("Sync");
    }

  code[max_codelen] = '\0';
  strncpy(code, l_code, max_codelen);
  if(strlen(l_code) > max_codelen)
    lcp += max_codelen;
  else
    lcp += strlen(code);
  

  /* Major comment point? */
  if(nextcom && *nextcom && (*nextcom)->val == org_pc &&
     (*nextcom)->type == T_COM_MAJOR)
    {
      if(opt_opcode)
	fputs(pc_str, stdout);

      putchar('\n');
      
      while(*nextcom && (*nextcom)->val == org_pc &&
	    (*nextcom)->type == T_COM_MAJOR)
	{
	  if(opt_opcode)
	    fputs(pc_str_blank, stdout);

	  printf("%s; %s\n", indstr, (*nextcom)->id);

	  nextcom++;
	}

      if(opt_opcode)
	fputs(pc_str, stdout);

      putchar('\n');
    }

  /* Symbol point? */
  while(nextsym && *nextsym && (*nextsym)->val == org_pc)
    {
      int tag;

      if(opt_opcode)
	fputs(pc_str_blank, stdout);

      printf("%s:\n", 
	     ((*nextsym)->type & T_SYM_LOCAL ? 
	      local_id(*nextsym) : (*nextsym)->id));

      (*nextsym)->type |= T_SYM_ISDEF;

      /* Skip remaining local symbols */
      if((*nextsym)->type & T_SYM_LOCAL)
	for(tag = (*nextsym++)->val;
	    ((*nextsym)->type & T_SYM_LOCAL) && (*nextsym)->val == tag;
	    nextsym++);
      else
	nextsym++;
    }

  /* Symbol caught in-between? */
  while(nextsym && *nextsym && (*nextsym)->val > org_pc &&
	(*nextsym)->val < pc)
    {
      comment("%s=#%x", 
	      ((*nextsym)->type & T_SYM_LOCAL ?
	       local_id(*nextsym) : (*nextsym)->id),
	      (*nextsym)->val);
      nextsym++;
    }

  /* Comment caught in-between?
   * Then skip, regardless of type.
   */
  while(nextcom && *nextcom && (*nextcom)->val > org_pc &&
	(*nextcom)->val < pc)
    nextcom++;


  /* Now check if we actually managed to disassemble the instruction.
   * If l_instr is blank, or contains wildcard characters (*'s),
   * we'll substitute it with a suitable data instruction.
   */
  if(failed(l_instr))
    {
      int n = strlen(l_code);
      register char *cp, *bp;


      if(l_instr[0])
	comment("%s", l_instr);

      if(n <= 15)
	sprintf(l_instr, "data.%s\t", flen[n]);
      else
	sprintf(l_instr, "data.%d\t", n);

      bp = l_instr + strlen(l_instr);
      cp = l_code + n - 1;

      /* Skip leading zeroes */
      while(cp > l_code && *cp == '0')
	cp--;

      /* Make it hex, if necessary */
      if(!(cp == l_code && *cp <= '9'))
	*bp++ = '#';

      /* Duplicate */
      while(cp >= l_code)
	*bp++ = *cp--;

      *bp = '\0';
    }

  /* Print instruction */
  if(opt_opcode)
    sprintf(lbuf,
	    (opt_alonzo ? "%s #%-12s  %s%s" : "%s %-7s  %s%s"),
	    pc_str,
	    code,
	    indstr,
	    l_instr);
  else
    sprintf(lbuf, "\t%s%s",
	    indstr,
	    l_instr);
  
  fputs(lbuf, stdout);

  /* Add comments */
  if(l_comment[0] ||
     (nextcom && *nextcom && (*nextcom)->val == org_pc))
    {
      lpos = prpos(lbuf);
      
      if(lpos >= opt_commentcol)
	putchar(' ');
      else
	tabulate(lpos, opt_commentcol);
      
      putchar(';');

      if(nextcom && *nextcom && (*nextcom)->val == org_pc)
	{
	  printf(" %s", (*nextcom)->id, stdout);
	  nextcom++;
	}

      if(l_comment[0])
	printf(" ; %s", l_comment+1);
    }

  putchar('\n');
  
  
  /* List any remaining code and/or comments */
  while((opt_opcode && *lcp) ||
	(nextcom && *nextcom && (*nextcom)->val == org_pc))
    {
      int lpos = 0;

      l_instr[0] = '\0';

      if(opt_opcode && *lcp)
	{
	  strncpy(code, lcp, max_codelen);
	  if(strlen(lcp) > max_codelen)
	    lcp += max_codelen;
	  else
	    lcp += strlen(lcp);
	  
	  if(opt_alonzo)
	    printf("       #%s", code);
	  else
	    printf("      %s", code);

	  lpos = strlen(code) + 6;
	}

      if(nextcom && *nextcom && (*nextcom)->val == org_pc)
	{
	  tabulate(lpos, opt_commentcol);

	  printf("; %s", (*nextcom)->id);
	  nextcom++;
	}

      putchar('\n');
    }
}


/* Print symbol definitions */
print_sym_defs()
{
  register struct symbol *symp;


  putchar('\n');

  for(symp = symroot; symp; symp = symp->link)
    if(symp->ref && !(symp->type & T_SYM_ISDEF))
      switch(symp->type & T_SYM_TYPE)
	{
	case T_SYM_DATA:
	default:

	  if(opt_alonzo)
	    printf("%05x:   %s:\n", symp->val,
		   (symp->type & T_SYM_LOCAL ? local_id(symp) : symp->id));
	  else
	    printf("%s=%s\n",
		   (symp->type & T_SYM_LOCAL ? local_id(symp) :  symp->id),
		   addrtohex(symp->val));
	}

  putchar('\n');
}


/* Print XREF data */
void print_xref()
{
  register struct symbol *symp;
  register struct xrefaddr *xrefp;


  putchar('\n');
  putchar('\n');

  for(symp = symroot; symp; symp = symp->link)
    if(symp->ref)
      {
	int tabpos;
	char lbuf[132];

	printf("\n; ; %s:",
	       (symp->type & T_SYM_LOCAL ? local_id(symp) : symp->id));

	tabpos = 0;

	for(xrefp = symp->xrefhead; xrefp; xrefp = xrefp->link)
	  {
	    int w;

	    sprintf(lbuf, "\t%s", addrtohex(xrefp->val));

	    if(tabpos + 8 > 79)
	      {
		fputs("\n; ; ", stdout);
		tabpos = 0;
	      }

	    fputs(lbuf, stdout);
	    tabpos += 8;
	  }

	putchar('\n');
      }
}


/* Set up for new instruction */
static void new_instr()
{
  l_code[0] = l_instr[0] = l_comment[0] = 0;
  org_pc = pc;
  l_codep = l_code;
  l_commentp = l_comment;
}



/* Temporarily set format to code */
void format_code()
{
  static struct format f;
  
  if(curfmtref && *curfmtref)
    {
      f.nformats = 1;
      f.fmtspec[0].ftype = FT_CODE;
      f.fmtspec[0].width = 1;
      f.fmtspec[0].repeat = 1;
      curfmt = &f;
    }
}

/* Decode RPL code */
static void decode_rpl()
{
  int tag;
  char *s;
  struct symbol *mdef;

  rpl_new_ind_level = rpl_ind_level;

  new_instr();
  tag = get_unibbles(5);

  /* First see if special LINK */
  if(tag == pc)
    {
      strcpy(l_instr, "MCODE");
      format_code();
    }
  else
    {
      /* See if we have a suitable macro definition */
      if((mdef = macro(5, tag)) && (s = macro_expand(mdef)))
	strcpy(l_instr, s);
      else
	strcpy(l_instr, symbolic(tag, NOREL));
      
      /* Add format */
      if(opt_formats && pass == PASSF)
	add_auto_format(tag, T_FMT_RPL);
    }

  print_line();

  rpl_ind_level = rpl_new_ind_level;
}


/* Create data pseudo opcode for unsigned integers */
static void mk_udata(intpref, ctrl, nibbles)
  char *intpref, ctrl;
  int nibbles;
{
  char ctrlbuf[64], *s;
  int i;


  i = get_unibbles(nibbles);

  s = symbolic(i, NOREL);

  if(*s == '#')
    {
      if(nibbles > 16)
	sprintf(ctrlbuf, "data.%d\t%s%%%c", nibbles, intpref, ctrl);
      else
	sprintf(ctrlbuf, "data.%s\t%s%%%c", flen[nibbles], intpref,  ctrl);
      
      sprintf(l_instr, ctrlbuf, i);
    }
  else
    if(nibbles > 16)
      sprintf(l_instr, "data.%d\t%s", nibbles, s);
    else
      sprintf(l_instr, "data.%s\t%s", flen[nibbles], s);
}


/* Decode data according to format */
static void decode_data(fmt)
  struct format *fmt;
{
  struct format_ent *fent;

  if(!fmt)
    {
      decode_instr();
      print_line();
      return;
    }

  for(fent = fmt->fmtspec; fent < fmt->fmtspec + fmt->nformats; fent++)
    decode_data_ent(fent);
}

  
/* Decode according to data format entry */
decode_data_ent(fent)
  struct format_ent *fent;
{
  int rept;

  for(rept = fent->repeat; rept > 0; rept--)
    if(fent->ftype & FT_SUBD)
      decode_data(fent->fmtdescr);
    else
      {
	new_instr();

	switch(fent->ftype & FT_DATA_MASK)
	  {
	  case FT_HEX: mk_udata("#", 'x', fent->width); break;
	  case FT_DEC: mk_udata("", 'd', fent->width); break;
	  case FT_OCT: mk_udata("#o", 'o', fent->width); break;
	  case FT_BIN: mk_udata("", 'u', fent->width); break;

	  case FT_VHEX:

	    mk_udata("#", 'x', get_unibbles(fent->width)); 
	    break;

	  case FT_STR:

	    {
	      char sbuf[132], *cp;
	      int slen, c;

	      for(cp = sbuf, slen = fent->width; slen > 0; slen--)
		{
		  c = get_unibbles(2);

		  if(c < ' ' || c > 127)
		    {
		      sprintf(cp, "\\%03o", c);
		      cp += strlen(cp);
		    }
		  else
		    *cp++ = c;
		}

	      *cp = '\0';

#ifdef NEVER
	      sprintf(l_instr, "ascii.%d\t\"%s\"", fent->width, sbuf);
#endif
	      sprintf(l_instr, "ascii\t\"%s\"", sbuf);
	      break;
	    }

	  case FT_VSTR:		/* Vectored string */

	    {
	      char sbuf[132], *cp;
	      int slen, c, w;

	      w = get_unibbles(fent->width);

	      if(w < 0 || w > 64)
		{
		  comment("bad string length, %d", w);
		  w = 64;
		}

	      for(cp = sbuf, slen = w; slen > 0; slen--)
		{
		  c = get_unibbles(2);

		  if(c < ' ' || c > 127)
		    {
		      sprintf(cp, "\\%03o", c);
		      cp += strlen(cp);
		    }
		  else
		    *cp++ = c;
		}

	      *cp = '\0';

#ifdef NEVER
	      sprintf(l_instr, "ascii.%d\t\"%s\"", w, sbuf);
#endif
	      sprintf(l_instr, "ascii\t\"%s\"", sbuf);
	      break;
	    }

	  case FT_FLOAT:

	    {
	      hp_real r;
	      char realbuf[132];

	      r.x = get_unibbles(3);
	      r.mr= get_unibbles(3);
	      r.m = get_unibbles(8);
	      r.m1= get_unibbles(1);
	      r.s = get_unibbles(1);

	      sprintf(l_instr, "word\t%s", realtos(&r, realbuf));
	      break;
	    }

	  default:

	    mk_udata("#", 'x', 1);
	    break;
	  }
	
	print_line();
      }
}


/* Disassemble according to current format */
static void decode()
{
  /* Is the current format temporary, and if so, has it expired? */
  if(prevfmt && pc >= fmtexpire_tag)
    {
      curfmt= prevfmt;
      prevfmt = NULL;
      fmtexpire_tag = 0xffffff;
      rpl_ind_level--;
    }

  /* If code */
  if(!curfmt ||
     !curfmt->nformats ||
     (curfmt->fmtspec[0].ftype & FT_CODE))
    {
      decode_instr();
      print_line();
      return;
    }

  /* If RPL */
  if(curfmt->fmtspec[0].ftype & FT_RPL)
    {
      decode_rpl();
      return;
    }

  /* Otherwise data */
  decode_data(curfmt);
}

  
/* Print usage */
void usage(argc, argv)
  int argc;
  char **argv;
{
  fprintf(stderr, "usage: %s [-acsdgCxfz1] start end\n", *argv);
  exit(1);
}


/* Parse arguments */
static void args(argc, argv)
  int argc;
  char **argv;
{
  char *cp1, *cp2;
  int flagsp = 0;

  if(argc > 1 && argv[1][0] == '-')
    {
      flagsp = 1;

      for(cp1 = argv[1]+1; *cp1; cp1++)
	switch(*cp1)
	  {
	  case '1':		/* Skip pass one */

	    opt_pass1 = !opt_pass1;
	    break;

	  case 'Q':		/* Debug */

	    opt_debug = !opt_debug;
	    break;

	  case 'f':		/* Auto format generation */

	    opt_formats = !opt_formats;
	    break;

	  case 'C':		/* Code */

	    opt_code = !opt_code;
	    break;

	  case 'g':		/* Symbol generation */

	    opt_locals = !opt_locals;
	    break;

	  case 'a':		/* Assembler format - opcodes (off) */

	    opt_opcode = !opt_opcode;
	    break;

	  case 's':		/* (No) symbols in listings */

	    opt_symbols = !opt_symbols;
	    break;

	  case 'd':		/* (No) undef-but-ref  symbol definitions */

	    opt_symdef = !opt_symdef;
	    break;

	  case 'x':		/* Full XREF */

	    opt_xref = !opt_xref;
	    break;

	  case 'c':		/* Auto comments */

	    opt_comments = !opt_comments;
	    break;

	  case 'z':		/* Alonzo-styled formatting */

	    opt_alonzo = !opt_alonzo;
	    break;

	  default:

	    usage(argc, argv);
	  }
    }


  if(argc != 3+flagsp)
    usage(argc, argv);

  cp1 = argv[++flagsp]; 
  cp2 = argv[++flagsp];
  
  if(!hexstrtoi(&cp1, &opt_startpt) || !hexstrtoi(&cp2, &opt_endpt) ||
     *cp1 || *cp2)
    usage(argc, argv);

  if(opt_alonzo)
    {
      opt_commentcol += 8;
      opt_opcode = 1;
    }

  if(opt_debug)
    fprintf(stderr, "Debug mode enabled.\nThis is SAD %s %s\n\n",
	    sad_version, *argv);
}  


/* Run one disassembly pass */
static void disassemble(npass)
  int npass;
{
  struct format
    code_fmt, rpl_fmt;

  code_fmt.nformats = rpl_fmt.nformats = 1;

  code_fmt.fmtspec[0].ftype = FT_CODE;
  code_fmt.fmtspec[0].fmtdescr = NULL;
  code_fmt.fmtspec[0].width = 1;
  code_fmt.fmtspec[0].repeat = 1;

  rpl_fmt.fmtspec[0].ftype = FT_RPL;
  rpl_fmt.fmtspec[0].fmtdescr = NULL;
  rpl_fmt.fmtspec[0].width = 1;
  rpl_fmt.fmtspec[0].repeat = 1;

  pass = npass;

  lineno = 0;
  
  if(!opt_opcode && opt_startpt && pass == PASS2)
    printf("\t org\t%s\n", addrtohex(opt_startpt));
  
  nextsym = symref;
  nextcom = comref;
  nextfmt = fmtref;
  curfmt = NULL;

  set_pc(opt_startpt);

  rpl_ind_level = rpl_new_ind_level = 0;

  
  while(pc <= opt_endpt)
    {
      /* Skip to next symbol, comment, and format to look for */
      for(; nextsym && *nextsym && (*nextsym)->val < pc; nextsym++);
      for(; nextcom && *nextcom && (*nextcom)->val < pc; nextcom++);
      for(; nextfmt && *nextfmt && (*nextfmt)->val <= pc; nextfmt++)
	curfmt = (*nextfmt)->form, curfmtref = nextfmt;
      
      if(!curfmt)
	switch((*curfmtref)->type & T_FMT_TYPE)
	  {
	  case T_FMT_CODE: curfmt = &code_fmt; break;
	  case T_FMT_RPL:  curfmt = &rpl_fmt; break;
	  }

      new_instr();		/* Set up for new instruction */
      decode();			/* Decode */
    }
}


main(argc, argv)
  int argc;
  char **argv;
{
  init();			/* Initialize tables */
  
  args(argc, argv);		/* Decode arguments */

  open_core(SAD_CORE);		/* Load core from stdin */
  load_symbols(SAD_SYMBOLS, symcmp); /* Load symbol table */
  load_comments(SAD_COMMENTS);	/* Load comments table */
  load_formats(SAD_FORMATS);	/* Load formats table */
  load_macros(SAD_MACROS);	/* Load macro definitions */
  

  if(opt_pass1)
    disassemble(PASS1);		/* Run pass 1 */

  add_locals_to_symtab(symcmp);	/* Add local symbols to table */

  if(opt_symdef && opt_pass1)
    print_sym_defs();		/* Define referenced undefined symbols */

  /* Repeat pass F until no changes */
  if(opt_formats)
    do
      {
	nautoformats = 0;
	format_added = FALSE;
	
	disassemble(PASSF);
	
	if(nautoformats)
	  add_autos_to_fmttab();
      }
    while(format_added);

      
  disassemble(PASS2);		/* Print output and collect XREF info */

  if(opt_symdef && !opt_pass1)
    print_sym_defs();		/* Define referenced undefined symbols */

  if(opt_xref)
    print_xref();		/* Print full xref info */

  if(opt_formats)
    print_format_file(SAD_AUTOFMT); /* Print auto formats */
}
