/* misc.c -- SAD context and some common functions.

This file is part of SAD, the Saturn Disassembler package.

SAD is not distributed by the Free Software Foundation. Do not ask
them for a copy or how to obtain new releases. Instead, send e-mail to
the address below. SAD is merely covered by the GNU General Public
License.

Please send your comments, ideas, and bug reports to
Jan Brittenson <bson@ai.mit.edu>

*/


/* Copyright (C) 1990 Jan Brittenson.

SAD is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free
Software Foundation; either version 1, or (at your option) any later
version.

SAD is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with SAD; see the file COPYING.  If not, write to the Free
Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. */


#include <stdio.h>
#include <signal.h>
#include "sad.h"


/* Context */
int
  pc,				/* Current location */
  lineno = 0,			/* Current input file line number */
  ncomments = 0,		/* # of comments in .comments */
  nlocals = 0,			/* # of local symbols from pass 1 */
  nsymbols = 0;			/* # of symbols in .symbols */


/* Various mandatory options */
int
  opt_lines = 0;		/* Line numbers in errors and warnings */

char
  hextoi_map[0400],
  hexmap[] = "0123456789abcdef";


/* Comment table pointer and ref data.
 */
struct symbol
  *comroot = NULL,		/* Comment chain start */
  **comref = NULL;		/* Comment search/sort ref array */


/* Symbol table pointer and ref data.
 */
struct symbol
  *symroot = NULL,		/* Symbol chain start */
  **symref = NULL;		/* Symbol search/sort ref array */


/* Initialize */
void init()
{
  char *cp, i;

  for(cp = hextoi_map; cp < hextoi_map + sizeof hextoi_map; *cp++ = -1);
  for(i = 0, cp = hexmap; *cp; hextoi_map[*cp++] = i++);
  for(i = 0xa, cp = "ABCDEF"; *cp; hextoi_map[*cp++] = i++);

  symroot = comroot = NULL;
  comref = symref = NULL;
}

  
/* Issue an error */
void error(ctrl, a1, a2, a3, a4)
  char *ctrl, *a1, *a2, *a3, *a4;
{
  if(opt_lines)
    fprintf(stderr, "%d: ", lineno);

  fputs("ERROR - ", stderr);

  fprintf(stderr, ctrl, a1, a2, a3, a4);

  fputc('\n', stderr);
}


/* Issue a warning */
void warning(ctrl, a1, a2, a3, a4)
  char *ctrl, *a1, *a2, *a3, *a4;
{
  if(opt_lines)
    fprintf(stderr, "%d: ", lineno);

  fputs("WARNING - ", stderr);

  fprintf(stderr, ctrl, a1, a2, a3, a4);

  fputc('\n', stderr);
}


/* Convert hex string to integer,
 * leave pointer at first nonhex char.
 */
hexstrtoi(cpp, ip)
  register char **cpp;
  int *ip;
{
  register data;
  char *save;

  
  if(!cpp || !*cpp)
    return(0);

  save = *cpp;

  /* Skip optional initial hash sign */
  if(**cpp == '#')
    (*cpp)++;

  for(data = 0; IS_HEX(**cpp); (*cpp)++)
    {
      data <<= 4;
      data |= HEX_TO_I(**cpp);
    }

  if(IS_INNER_SYM(**cpp))
    {
      *cpp = save;
      return(FALSE);
    }
  
  *ip = data;
  return(TRUE);
}


/* Convert any (< 10) base string to integer,
 * leave pointer at first nonhex char.
 */
anystrtoi(cpp, ip, base)
  register char **cpp;
  int *ip, base;
{
  register data;
  char *save, basemax;

  
  if(!cpp || !*cpp)
    return(FALSE);

  save = *cpp;

  basemax = base + '0' - 1;
  for(data = 0; (**cpp >= '0' && **cpp <= basemax); (*cpp)++)
    {
      data *= base;
      data += **cpp-'0';
    }

  if(IS_INNER_SYM(**cpp))
    {
      *cpp = save;
      return(FALSE);
    }
  
  *ip = data;
  return(TRUE);
}


/* Convert integer string to integer,
 * leave pointer at first nonhex char.
 */
intstrtoi(cpp, ip)
  register char **cpp;
  int *ip;
{
  if(!cpp || !*cpp)
    return(0);

  if(**cpp == '#')
    if((*cpp)[1] == 'o')
      {
	(*cpp) += 2;
	return(anystrtoi(cpp, ip, 8));
      }
    else
      return(hexstrtoi(cpp, ip));

  return(anystrtoi(cpp, ip, 10));
}


/* Duplicate string */
char *strdup(s)
  char *s;
{
  register char *tmp;
  extern char *malloc(), *strcpy();

  if(!s)
    return("");

  if(!(tmp = malloc(strlen(s)+1)))
    {
      perror("malloc");
      exit(1);
    }

  return(strcpy(tmp, s));
}


/* Compare two symbols by value for ascending sort. */
symcmp(s1, s2)
  struct symbol **s1, **s2;
{
  if((*s1)->val > (*s2)->val)
    return(1);

  if((*s1)->val < (*s2)->val)
    return(-1);

  if((*s1)->seq > (*s2)->seq)
    return(1);

  if((*s1)->seq < (*s2)->seq)
    return(-1);

  return(0);
}


/* Compare two comments by value, type, and sequence for ascending sort. */
comcmp(c1, c2)
  struct symbol **c1, **c2;
{
  /* First on value */
  if((*c1)->val > (*c2)->val)
    return(1);

  if((*c1)->val < (*c2)->val)
    return(-1);

  /* Then on type */
  if(((*c1)->type & T_COM_TYPE) > ((*c2)->type & T_COM_TYPE))
    return(1);

  if(((*c1)->type & T_COM_TYPE) < ((*c2)->type & T_COM_TYPE))
    return(-1);

  /* Then on sequence */
  if((*c1)->seq > (*c2)->seq)
    return(1);

  if((*c1)->seq < (*c2)->seq)
    return(-1);

  return(0);
}


/* Print comment file on stdout */
void print_comfile(comfname)
  char *comfname;
{
  struct symbol *comp;
  FILE *comfile;

  comfile = stdout;

  if(comfname)
    disable_intr();		/* No interruptions */

  if(comfname && !(comfile = fopen(comfname, "w")))
    {
      perror(comfname);
      exit(1);
    }

  for(comp = comroot; comp; comp = comp->link)
    if(!(comp->type & T_SYM_ERASE))
      {
	fprintf(comfile, "%x", comp->val);

	switch(comp->type & T_SYM_TYPE)
	  {
	  case T_COM_MAJOR:
	    
	    fputc('=', comfile);
	    break;
	    
	  default:
	    
	    fputc(':', comfile);
	    break;
	  }

	fputs(comp->id, comfile);
	fputc('\n', comfile);
      }
}


/* Create symbol reference array */
void create_symref(compfunc)
  int (*compfunc)();  
{
  struct symbol **symrefptr, *symptr;

  if(symref)
    free(symref);

  /* Create reference array, end with NULL guard */
  symref = (struct symbol **) malloc(sizeof(struct symbol *) * (nsymbols+1));

  for(symrefptr = symref, symptr = symroot; symptr; symptr = symptr->link)
    *symrefptr++ = symptr;

  *symrefptr++ = NULL;

  /* Sort the reference array in ascending order by name */
  qsort((char *) symref, nsymbols, sizeof(struct symbol *), compfunc);
}


/* Load symbols */
void load_symbols(symfname, compfunc)
  char *symfname;
  int (*compfunc)();
{
  FILE *symfile;
  char lbuf[132], *cp;
  struct symbol *symtemp, **symrefptr, *symptr;
  extern char *malloc();


  if(!(symfile = fopen(symfname, "r")))
    return;
      
  for(nsymbols = 0; !feof(symfile); )
    {
      fgets(lbuf, sizeof lbuf, symfile);
      if(feof(symfile))
	break;

      lbuf[strlen(lbuf)-1] = '\0';
      symtemp = (struct symbol *) malloc(sizeof(struct symbol));
      symtemp->link = symroot;
      symroot = symtemp;

      cp = lbuf;

      hexstrtoi(&cp, &symtemp->val);
      
      switch(*cp++)
	{
	case '\0':

	  symroot = symtemp->link;
	  free(symtemp);
	  continue;

	case '=':

	  symtemp->type = T_SYM_DATA;
	  break;

	case ':':
	default:

	  symtemp->type = T_SYM_CODE;
	}

      symtemp->ref = 0;
      symtemp->id = strdup(cp);
      symtemp->xrefhead = symtemp->xreftail = NULL;

      nsymbols++;
    }

  fclose(symfile);

  create_symref(compfunc);
}


/* Return ID of local symbol.
 * Return value is a static buffer.
 */
char *local_id(symp)
  struct symbol *symp;
{
  static char sname[132];

  sprintf(sname, "L_%05x", symp->val);

  return(sname);
}


/* Add local symbol */
void add_local_symbol(val, type)
  int val, type;
{
  register struct symbol *symtemp;

  symtemp = (struct symbol *) malloc(sizeof(struct symbol));
  symtemp->link = symroot;
  symroot = symtemp;
  
  symtemp->val = val;
  symtemp->type = type | T_SYM_LOCAL;
  symtemp->ref = 0;
  symtemp->id = NULL;
  symtemp->xrefhead = symtemp->xreftail = NULL;
  
  nlocals++;
}


/* Add local symbols to table */
void add_locals_to_symtab(compfunc)
  int (*compfunc)();
{
  nsymbols += nlocals;

  create_symref(compfunc);
}


/* Skip whitespace */
char *byspace(str)
  register char *str;
{
  while(*str && *str <= '\040')
    str++;

  return(str);
}



/* True if symbol is quoted string.
 * Quoteflag determines whether the quotes are to be retained.
 * Maxlen is the maximum length of the string.
 */
stringp(strp, strbuf, maxlen, quoteflag)
  register char **strp, *strbuf;
  int quoteflag, maxlen;
{
  char *save;

  if(!strp || !*strp)
    return(FALSE);

  if(quoteflag)
    maxlen -= 2;

  save = *strp;
  if(**strp != '"')
    return(FALSE);

  if(quoteflag)
    *strbuf++ = '"';

  for((*strp)++; maxlen > 0 && **strp && **strp != '"';
      *strbuf++ = *(*strp)++, maxlen--);

  if(!**strp || maxlen <= 0)
    {
      *strp = save;
      return(FALSE);
    }

  (*strp)++;

  if(quoteflag)
    *strbuf++ = '"';

  *strbuf = '\0';
  return(TRUE);
}


/* Scan symbol id.
 * Returns true (non-zero) if valid symbol id, with
 * id in symname buffer.
 * Returns false (zero) if invalid symbol name, with
 * symname undefined.
 * Maxlen is maximum length of symbol name.
 */
symbolidp(strp, symname, maxlen)
  register char **strp, *symname;
  int maxlen;
{
  char *symbuf = symname;

  if(!strp || !*strp)
    return(FALSE);

  /* Check initial char to determine action */
  if(IS_INITIAL_SYM(**strp))
    {
      /* Scan one word */
      for(; maxlen > 0 && IS_INNER_SYM(**strp);
	  *symname++ = *(*strp)++, maxlen--);

      *symname = '\0';

      return(maxlen > 0);
    }

  /* String? */
  return(stringp(strp, symname, maxlen, TRUE));
}


/* Try to determine PC.
 */
determine_pc(fp)
  FILE *fp;
{
  long oldpos = ftell(fp);
  char lbuf[132], *lp;
  int ahead_pc;

  while(!feof(fp))
    {
      fgets(lbuf, sizeof lbuf, fp);
      if(feof(fp))
	break;

      lp = lbuf;
      if(IS_HEX(*lp) && hexstrtoi(&lp, &ahead_pc))
	return(ahead_pc);
    }

  fseek(fp, oldpos, 0);
  fgets(lbuf, sizeof lbuf, fp);
  fseek(fp, oldpos, 0);

  warning("Unable to determine location, assuming %05x:\n%s", pc, lbuf);

  return(pc);
}


/* Print symbol file on stdout */
void print_symfile(symfname)
  char *symfname;
{
  struct symbol *symp;
  FILE *symfile;


  symfile = stdout;

  if(symfile)
    disable_intr();		/* Disable various signals */

  if(symfname && !(symfile = fopen(symfname, "w")))
    {
      perror(symfname);
      exit(1);
    }
  
  for(symp = symroot; symp; symp = symp->link)
    if(!(symp->type & T_SYM_ERASE))
      {
	fprintf(symfile, "%x", symp->val);

	switch(symp->type & T_SYM_TYPE)
	  {
	  case T_SYM_DATA:
	    
	    fputc('=', symfile);
	    break;
	    
	  case T_SYM_CODE:
	  default:
	    
	    fputc(':', symfile);
	    break;
	  }

	fputs(symp->id, symfile);
	fputc('\n', symfile);
      }
}


/* Convert real to string */
char *realtos(rp, bp)
  hp_real *rp;
  char *bp;
{
  char *bp0 = bp;

  sprintf(bp, "%s%x.%08x%03x",
	  (rp->s >= 5 ? "-" : ""),
	  rp->m1, rp->m, rp->mr);

  bp += strlen(bp) - 1;
	      
  while(*bp == '0') bp--;
  if(*bp == '.') bp--;
  *++bp = '\0';
	      
  if(rp->x)
    sprintf(bp, "E%s%x",
	    (rp->x >= 0x500 ? "-" : ""),
	    (rp->x < 0x500 ? rp->x : 0x999 - rp->x + 1));

  return(bp0);
}


/* Convert long real to string */
char *longrealtos(lrp, bp)
  hp_longreal *lrp;
  char *bp;
{
  char *bp0 = bp;

  sprintf(bp, "%s%x.%08x%06x",
	  (lrp->s >= 5 ? "-" : ""),
	  lrp->m1, lrp->m, lrp->mr);

  bp += strlen(bp) - 1;
	      
  while(*bp == '0') bp--;
  if(*bp == '.') bp--;
  *++bp = '\0';
	      
  if(lrp->x)
    sprintf(bp, "E%s%x",
	    (lrp->x >= 0x50000 ? "-" : ""),
	    (lrp->x < 0x50000 ? lrp->x : 0x99999 - lrp->x + 1));

  return(bp0);
}


/* Disable signals interrupt and quit signals */
void disable_intr()
{
  signal(SIGHUP, SIG_IGN);
  signal(SIGINT, SIG_IGN);
  signal(SIGQUIT, SIG_IGN);
  signal(SIGTERM, SIG_IGN);
  signal(SIGPIPE, SIG_IGN);
  signal(SIGUSR1, SIG_IGN);
  signal(SIGUSR2, SIG_IGN);

#ifdef SIGSYS
  signal(SIGSYS, SIG_IGN);
#endif

#ifdef SIGSTOP
  signal(SIGSTOP, SIG_IGN);
#endif

#ifdef SIGTSTP
  signal(SIGTSTP, SIG_IGN);
#endif

#ifdef SIGWINCH
  signal(SIGWINCH, SIG_IGN);
#endif
}


/* Convert integer to bit mask */
char *itobmask(i)
  register unsigned i;
{
  register bitno;
  static char bitbuf[3*32+5];
  register char *bp = bitbuf;

  *bp++ = '[';

  for(bitno = 0; i; i >>= 1, bitno++)
    if(i & 1)
      {
	sprintf(bp, "%d", bitno);
	bp += strlen(bp);

	if(i > 1)
	  *bp++ = ',';
      }

  *bp++ = ']';
  *bp = '\0';

  return(bitbuf);
}
