/*
** Program fdb - family tree database generation and manipulation
**
** Copyright (C) 1994 Andy Burrows 
**
**            email: cadellin@corum.me.man.ac.uk (130.88.29.14)
**
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public Licence as published by the Free
** Software Foundation; either version 1, or any later version.
** 
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
** or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
** more details. It should be included in this distribution in a file 
** called COPYING
**
*/


/*
   this file contains the functions relating to hardcopy output
*/

/* #define DEBUG */

/* standard headers */

#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>

/* XView headers (needed by definitions.h) */

#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/canvas.h>
#include <xview/font.h>
#include <xview/xv_xrect.h>

/* fdb database definitions */

#include "definitions.h"
#include "prototypes.h"

/*
   plot a line between the specified points 
*/

void  plot_line(fp, x1, y1, x2, y2)

FILE  *fp;
float  x1, y1, x2, y2;
{
fprintf(fp,"%.1f %.1f moveto\n%.1f %.1f lineto\nstroke\n",x1,y1,x2,y2);
#ifdef DEBUG
printf("plot_line - (%f,%f) to (%f,%f)\n",x1,y1,x2,y2);
#endif
}

/*
   plot the specified string at the specified coordinates, centred
   in the x direction
*/

void  plot_centred(fp, x, y, string, font)

FILE  *fp;
float  x, y, font;
char  *string;
{
/* 
   I know that resetting the font each time is hideously expensive
   but it's a lot easier :-}
*/

fprintf(fp,"/Times-Roman findfont %.1f scalefont setfont\n", font);

fprintf(fp,"(%s)  stringwidth  pop  2  div  neg %.1f  add  %.1f  moveto\n(%s)  show\n",
                      string, x, y, string); 
#ifdef DEBUG
printf("placing string %s at (%f,%f)\n",string,x,y);
#endif
}

/*
   plot the current descent line to a PostScript file

   the name of the file will be name_lineN.ps where 
   name is the person whose line is being plotted and
   N is the smallest integer for which the file does not
   already exist
*/

void  plot_gen_tree()

{
float  height_scale, width_scale, scale, ratio;
float  larger_font_size, smaller_font_size;
float  x_offset, y_offset;
int    i, j, num_lines, num_items;
char   filename[255],working[255],plot_file[255],substring[255],initial[3];
char   test_filename[255];
FILE  *fp;
/* 
   first check that there is some data to plot i.e. at least one
   display struct is non-NULL (they are used sequentially, so
   if the first one is NULL they all are)
*/

if(generation[0] == (struct db_generation *) NULL)
    {
    sprintf(error_message,"No Descent Line Data Available!\n");
    show_error();
    return;
    }

/* 
   find a suitable name for the output file, the code assumes
   that if a file cannot be opened for reading, then it does
   not exist
*/

/* initialise the file name string */
strcpy(filename,"");

/* 
   read the person at foot of the descent line's forename string into the work space, 
   converting to upper case on the way 
*/

i = 0;
while((working[i] = toupper(entry[generation[0]->person]->forenames[i])) != '\0')
    i++;

#ifdef DEBUG
printf("forename string: %s\n",working);
#endif

/* read in substrings and extract the first letters */

j = 0;

while( ((num_items = sscanf((working + j),"%s",substring)) != EOF) &&
       (j < strlen(working)) )
    {
#ifdef DEBUG
    printf("working on substring: %s\n",substring);
#endif

    sprintf(initial,"%c",substring[0]);

#ifdef DEBUG
    printf("first letter: **%s**\n",initial);
#endif
    /* add the initial to the file name */

    strcat(filename,initial);

    /* add an underscore */

    strcat(filename,"_");
	
    /* increment the distance along the string counter */

    j += strlen(substring) + 1;
#ifdef DEBUG
    printf("j = %d\n",j);
#endif
    }

/* add the surname on the end, ensuring the first letter is capitalised */

sscanf(entry[generation[0]->person]->surname,"%s",working);
working[0] = toupper(working[0]);

/* append the surname to the file name */

strcat(filename,working);

/* 
   ensure that file does not already exist, add an integer to the
   file name to make it unique if necessary 
*/

sprintf(test_filename,"%s_line.ps",filename);

/* check if the file already exists */

j = 1;
while((fp = fopen(test_filename,"r")) != NULL)
    {
    sprintf(test_filename,"%s_line%d.ps",filename,j);
    j++;
    }

strcpy(plot_file, test_filename);

#ifdef DEBUG
printf("file name: %s\n",plot_file);
#endif

/* open the chosen file name for writing and write the header info */

if ((fp = fopen(plot_file, "w")) == NULL)
    {
    sprintf(error_message,"Unable to Open File %s for Plot Creation!",plot_file);
    show_error();
    return;
    }

/* write the header comment to the file */
fprintf(fp,"%%!PS - descent line plot generated using fdb V%s - cadellinsoft 94\n\n",
             FDB_VERSION);

/* decide whether to draw the plot portrait or landscape */

ratio = ((float) gen_canvas_height)/((float) gen_canvas_width);

#ifdef DEBUG
printf("ratio = %f\n",ratio);
#endif

/* 
   also calculate the scale factor from pixels to points to allow, amongst
   other things, the font sizes to be set
*/

if(ratio < 1.0)
    {
    /* rotate and translate the axes appropriately */
    fprintf(fp,"%%alter axes for landscape plot\n-90 rotate\n-%.1f 0.0 translate\n\n",A4_HEIGHT_PTS);
    /* set the y origin at the top of the page */
    fprintf(fp,"%%translate y axis to accomodate X11 coordinates\n");
    fprintf(fp,"0.0 %.1f translate\n\n",A4_WIDTH_PTS);
    height_scale = A4_WIDTH_PTS/((float) gen_canvas_height);
    width_scale  = A4_HEIGHT_PTS/((float) gen_canvas_width);
#ifdef DEBUG
    printf("selecting landscape\n");
#endif
    }
else
    {
    /* set the y origin at the top of the page */
    fprintf(fp,"%%translate y axis to accomodate X11 coordinates\n");
    fprintf(fp,"0.0 %.1f translate\n\n",A4_HEIGHT_PTS);
    height_scale = A4_HEIGHT_PTS/((float) gen_canvas_height);
    width_scale  = A4_WIDTH_PTS/((float) gen_canvas_width);
#ifdef DEBUG
    printf("selecting portrait\n");
#endif
    }

if(height_scale < width_scale)
    {
    if(ratio < 1.0)
        x_offset = (A4_HEIGHT_PTS - (gen_canvas_width * height_scale ))/2;
    else
        x_offset = (A4_WIDTH_PTS - (gen_canvas_width * height_scale ))/2;

    y_offset = 0;
    scale = height_scale;
    }
else
    {
    if(ratio < 1.0)
        y_offset = (A4_WIDTH_PTS - (gen_canvas_height * width_scale ))/2;
    else
        y_offset = (A4_HEIGHT_PTS - (gen_canvas_height * width_scale ))/2;

    x_offset = 0;
    scale = width_scale;
    }

scale = height_scale < width_scale ? height_scale : width_scale;

#ifdef DEBUG
printf("x_offset = %f, y_offset = %f, scale = %f\n",x_offset,y_offset,scale);
#endif

larger_font_size  = scale * ((float) LARGER_FONT);
smaller_font_size = scale * ((float) SMALLER_FONT);

/* add the cadellinsoft logo */

put_moggy(fp, 36, -36);

/* loop for all people in the tree */

for(i=0;i<number_of_generations;i++)
    {
    for(j=0;j<generation[i]->number_of_children;j++)
        {
        /* write the strings into the window, centred on the x hot spot */

        /* add the forenames if not empty */
        num_lines = 0;
        if(strcmp(generation[i]->forenames[j],""))
            {
            plot_centred(fp, x_offset + scale*generation[i]->x[j],
                             - y_offset - scale*generation[i]->y[j], 
                    generation[i]->forenames[j], larger_font_size);
            num_lines++;
            }

        /* add the surname if not empty */
        if(strcmp(generation[i]->surname[j],""))
            {
            plot_centred(fp, x_offset + scale*generation[i]->x[j], 
                    - y_offset - scale*(generation[i]->y[j] + 
                        num_lines * vertical_text_separation), 
                    generation[i]->surname[j], larger_font_size);
            num_lines++;
            }

        /* add the date of birth of known */
        if(strcmp(generation[i]->birth_date[j],"b. "))
            {
            plot_centred(fp, x_offset + scale*generation[i]->x[j], 
                    - y_offset - scale*(generation[i]->y[j] + 
                        num_lines * vertical_text_separation),
                    generation[i]->birth_date[j], smaller_font_size);
            num_lines++;
            }

        /* if this is the traced person, add their spouse if known */
        if((j == generation[i]->location) && (generation[i]->spouse != -1))
            {
            plot_centred(fp, x_offset + scale*generation[i]->x[j], 
                    - y_offset - scale*(generation[i]->y[j] + 
                        num_lines * vertical_text_separation),
                    generation[i]->spouse_name, smaller_font_size);
            }

        /* 
           draw the little stub line up to the cross line except for
           the last generation
        */

        if(i < number_of_generations - 1)
            plot_line(fp, x_offset + scale*generation[i]->x[j], 
                 - y_offset - scale*(generation[i]->y[j] - vertical_text_separation),
                 x_offset + scale*generation[i]->x[j],
                 - y_offset - scale*(generation[i]->y[j] - 2 * vertical_text_separation));
        }
    /* 
       draw the line linking the children and the line to the next generation 
       (except for the last generation) 
       Note: the length of the line to the next generation depends on the 
       number of non-empty entries in the traced person of the generation above
    */
    if(i < number_of_generations - 1)
        {
        plot_line(fp, x_offset + scale*generation[i]->x[0], 
                 - y_offset - scale*(generation[i]->y[0] - 2 * vertical_text_separation),
                 x_offset + scale*generation[i]->x[generation[i]->number_of_children - 1],
                 - y_offset - scale*(generation[i]->y[generation[i]->
                     number_of_children - 1] - 2 * vertical_text_separation));

        num_lines = 0;
        if(strcmp(generation[i+1]->forenames[generation[i+1]->location],""))
            num_lines++;
        if(strcmp(generation[i+1]->surname[generation[i+1]->location],""))
            num_lines++;
        if(strcmp(generation[i+1]->birth_date[generation[i+1]->location],"b. "))
            num_lines++;
        if(generation[i+1]->spouse != -1)
            num_lines++;

        plot_line(fp, x_offset + scale*generation[i]->x[generation[i]->location], 
                 - y_offset - scale*(generation[i]->y[generation[i]->location] - 
                             2 * vertical_text_separation),
                 x_offset + scale*generation[i]->x[generation[i]->location],
                 - y_offset - scale*(generation[i]->y[generation[i]->location] - 
                             vertical_generation_separation +
                             num_lines * vertical_text_separation));
        }
    }
/* write the closing info to the file and close the file */
fprintf(fp,"\nshowpage\n");
fclose(fp);

/* tell the user where the plot has been written to */

sprintf(error_message,"Descent Line Plot Written to File %s",plot_file);
show_error();

}

/*
   function to output the information for the specified database
   entry to the specified file in LaTeX
*/

void  latex_output(fp, index)

FILE   *fp;
int     index;
{
int    j, k, l, initialise;
char   string1[100], string2[100], string3[100];

/* title and name */

if(leave_spaces_in_output)
    {
    if(strcmp(entry[index]->title,""))
        strcpy(string1,entry[index]->title);
    else
        strcpy(string1,"\\underline{~~~~~}");
    if(strcmp(entry[index]->forenames,""))
        strcpy(string2,entry[index]->forenames);
    else
        strcpy(string2,"\\underline{~~~~~~~~~~~~~~~~~}");
    if(strcmp(entry[index]->surname,""))
        strcpy(string3,entry[index]->surname);
    else
        strcpy(string3,"\\underline{~~~~~~~~~~~~~~~~~~~~~}");
    }
else
    {
    strcpy(string1,entry[index]->title);
    strcpy(string2,entry[index]->forenames);
    strcpy(string3,entry[index]->surname);
    }

fprintf(fp,"\\Large\n");
fprintf(fp,"\\section{~%s %s {\\bf %s} ",string1, string2, string3);

/* if there is a maiden name, display it */

if(strcmp(entry[index]->maiden_name,""))
    fprintf(fp,"(nee %s)}",entry[index]->maiden_name);
else
    {
    if(leave_spaces_in_output)
        {
        /* check the person is female and married before leaving space for a maiden name */
        if((entry[index]->gender == FEMALE) && (entry[index]->number_of_spouses > 0))
            fprintf(fp,"(nee \\underline{~~~~~~~~~~~~~~~~~~~~~~} )}");
        else
            fprintf(fp,"}");
        }
    else
        fprintf(fp,"}");
    }

fprintf(fp,"\n\\large\n\n");

fprintf(fp,"\\vspace*{15mm}\n\n");

/* only enter the flushleft environment if there is something to say */

if(strcmp(entry[index]->birth_date,"") || strcmp(entry[index]->birth_place,"") ||
        strcmp(entry[index]->baptism_date,"") || strcmp(entry[index]->baptism_place,"") ||
        strcmp(entry[index]->death_date,"") || strcmp(entry[index]->death_place,"") ||
        strcmp(entry[index]->resting_place,"") || (leave_spaces_in_output))
    fprintf(fp,"\\begin{flushleft}\n");

/* birth date, place and source if available */

if(strcmp(entry[index]->birth_date,"") || 
                strcmp(entry[index]->birth_place,"") || (leave_spaces_in_output))
    {
    fprintf(fp,"Born ");

    if(strcmp(entry[index]->birth_date,""))
        fprintf(fp," %s ",entry[index]->birth_date);
    else
        if(leave_spaces_in_output)
            fprintf(fp," \\underline{~~~~~~~~~~~~~~~} ");

    if(strcmp(entry[index]->birth_place,""))
        fprintf(fp,"at %s", entry[index]->birth_place);
    else
        if(leave_spaces_in_output)
            fprintf(fp,"at \\underline{~~~~~~~~~~~~~~~} ");

    fprintf(fp,"  \\\\\n");

    if(strcmp(entry[index]->birth_source,""))
        fprintf(fp,"~~~~~source: %s             \\\\\n",
                        entry[index]->birth_source);
    else
        if(leave_spaces_in_output)
            fprintf(fp,"~~~~~source: \\underline{~~~~~~~~~~~~~~~~~~~~}\\\\\n");
    }

/* baptism date, place and source if available */

if(strcmp(entry[index]->baptism_date,"") || 
                strcmp(entry[index]->baptism_place,"") || (leave_spaces_in_output))
    {
    fprintf(fp,"Baptised ");
    if(strcmp(entry[index]->baptism_date,""))
        fprintf(fp," %s ",entry[index]->baptism_date);
    else
        if(leave_spaces_in_output)
            fprintf(fp," \\underline{~~~~~~~~~~~~~~~} ");
    if(strcmp(entry[index]->baptism_place,""))
        fprintf(fp,"at %s", entry[index]->baptism_place);
    else
        if(leave_spaces_in_output)
            fprintf(fp,"at \\underline{~~~~~~~~~~~~~~~} ");

    fprintf(fp,"  \\\\\n");

    if(strcmp(entry[index]->baptism_source,""))
        fprintf(fp,"~~~~~source: %s             \\\\\n",
                        entry[index]->baptism_source);
    else
        if(leave_spaces_in_output)
            fprintf(fp,"~~~~~source: \\underline{~~~~~~~~~~~~~~~~~~~~}\\\\\n");
    }

/* death date, place and source, and resting place if available (and the person is deceased) */

if((strcmp(entry[index]->death_date,"") || strcmp(entry[index]->death_place,"") || 
        strcmp(entry[index]->resting_place,"") || (leave_spaces_in_output)) && 
        (entry[index]->status == DECEASED))
    {
    fprintf(fp,"Died ");

    if(strcmp(entry[index]->death_date,""))
        fprintf(fp," %s ",entry[index]->death_date);
    else
        if(leave_spaces_in_output)
            fprintf(fp," \\underline{~~~~~~~~~~~~~~~} ");

    if(strcmp(entry[index]->death_place,""))
        fprintf(fp,"at %s", entry[index]->death_place);
    else
        if(leave_spaces_in_output)
            fprintf(fp,"at \\underline{~~~~~~~~~~~~~~~} ");

    fprintf(fp,"  \\\\\n");

    if(strcmp(entry[index]->death_source,""))
        fprintf(fp,"~~~~~source: %s             \\\\\n",
                        entry[index]->death_source);

    if(strcmp(entry[index]->resting_place,""))
        fprintf(fp,"Resting Place: %s             \\\\\n",
                        entry[index]->resting_place);
    else
        if(leave_spaces_in_output)
            fprintf(fp,"Resting Place: \\underline{~~~~~~~~~~~~~~~~~~~~}\\\\\n");
    }

/* only leave the flushleft environment if there was something to say */

if(strcmp(entry[index]->birth_date,"") || strcmp(entry[index]->birth_place,"") ||
        strcmp(entry[index]->baptism_date,"") || strcmp(entry[index]->baptism_place,"") ||
        strcmp(entry[index]->death_date,"") || strcmp(entry[index]->death_place,"") ||
        strcmp(entry[index]->resting_place,"") || (leave_spaces_in_output))
    fprintf(fp,"\\end{flushleft}\n");

/* mothers name */

if(entry[index]->mother != -1)
    {
    if(leave_spaces_in_output)
        {
        if(strcmp(entry[entry[index]->mother]->title,""))
            strcpy(string1,entry[entry[index]->mother]->title);
        else
            strcpy(string1,"\\underline{~~~~~}");

        if(strcmp(entry[entry[index]->mother]->forenames,""))
            strcpy(string2,entry[entry[index]->mother]->forenames);
        else
            strcpy(string2,"\\underline{~~~~~~~~~~~~~~~~~}");

        if(strcmp(entry[entry[index]->mother]->surname,""))
            strcpy(string3,entry[entry[index]->mother]->surname);
        else
            strcpy(string3,"\\underline{~~~~~~~~~~~~~~~~~~~~~}");
        }
    else
        {
        strcpy(string1,entry[entry[index]->mother]->title);
        strcpy(string2,entry[entry[index]->mother]->forenames);
        strcpy(string3,entry[entry[index]->mother]->surname);
        }

    fprintf(fp,"Mother: %s %s %s\n\n",string1, string2, string3);
    }
else
    {
    /* if there is no mother but leave spaces is enabled */
    if(leave_spaces_in_output)
        {
        strcpy(string1,"\\underline{~~~~~}");
        strcpy(string2,"\\underline{~~~~~~~~~~~~~~~~~}");
        strcpy(string3,"\\underline{~~~~~~~~~~~~~~~~~~~~~}");
        fprintf(fp,"Mother: %s %s %s\n\n",string1, string2, string3);
        }
    } 

/* fathers name */

if(entry[index]->father != -1)
    {
    if(leave_spaces_in_output)
        {
        if(strcmp(entry[entry[index]->father]->title,""))
            strcpy(string1,entry[entry[index]->father]->title);
        else
            strcpy(string1,"\\underline{~~~~~}");

        if(strcmp(entry[entry[index]->father]->forenames,""))
            strcpy(string2,entry[entry[index]->father]->forenames);
        else
            strcpy(string2,"\\underline{~~~~~~~~~~~~~~~~~}");

        if(strcmp(entry[entry[index]->father]->surname,""))
            strcpy(string3,entry[entry[index]->father]->surname);
        else
            strcpy(string3,"\\underline{~~~~~~~~~~~~~~~~~~~~~}");
        }
    else
        {
        strcpy(string1,entry[entry[index]->father]->title);
        strcpy(string2,entry[entry[index]->father]->forenames);
        strcpy(string3,entry[entry[index]->father]->surname);
        }

    fprintf(fp,"Father: %s %s %s\n\n",string1, string2, string3);
    }
else
    {
    /* if there is no father but leave spaces is enabled */
    if(leave_spaces_in_output)
        {
        strcpy(string1,"\\underline{~~~~~}");
        strcpy(string2,"\\underline{~~~~~~~~~~~~~~~~~}");
        strcpy(string3,"\\underline{~~~~~~~~~~~~~~~~~~~~~}");
        fprintf(fp,"Father: %s %s %s\n\n",string1, string2, string3);
        }
    } 

fprintf(fp,"\\vspace*{5mm}\n\n");

/* spouse information */

if(entry[index]->number_of_spouses > 0)
    fprintf(fp,"\\begin{flushleft}\n");

for(j=0;j<entry[index]->number_of_spouses;j++)
    {
    if(leave_spaces_in_output)
        {
        if(strcmp(entry[entry[index]->spouse[j]]->forenames,""))
            strcpy(string1,entry[entry[index]->spouse[j]]->forenames);
        else
            strcpy(string1,"\\underline{~~~~~~~~~~~~~~~~~}");

        if(strcmp(entry[entry[index]->spouse[j]]->surname,""))
            strcpy(string2,entry[entry[index]->spouse[j]]->surname);
        else
            strcpy(string2,"\\underline{~~~~~~~~~~~~~~~~~~~~~}");
        }
    else
        {
        strcpy(string1,entry[entry[index]->spouse[j]]->forenames);
        strcpy(string2,entry[entry[index]->spouse[j]]->surname);
        }

    fprintf(fp,"Married %s %s ", string1, string2);

    /* add the maiden name if there is one */

    if(strcmp(entry[entry[index]->spouse[j]]->maiden_name,""))
        {
        fprintf(fp," (nee %s)",entry[entry[index]->spouse[j]]->maiden_name);
        }
    else
        {
        /* 
           if no maiden name info but spouse is female and leave spaces is enabled, 
           add the spaces 
        */
        if((leave_spaces_in_output) && (entry[entry[index]->spouse[j]]->gender == FEMALE))
            {
            fprintf(fp," (nee \\underline{~~~~~~~~~~~~~~~~~}) ",
                                entry[entry[index]->spouse[j]]->maiden_name);
            }
        }

    /*
       the marriage date and place are going to have to wait until I incorporate
       an array of marriage dates into the struct, one for each spouse
       for now if there is only one spouse and there is marriage date and/or place
       information, use it
    */

    if(entry[index]->number_of_spouses == 1)
        {
        if(strcmp(entry[index]->marriage_date,""))
            fprintf(fp," %s ",entry[index]->marriage_date);
        else
            if(leave_spaces_in_output)
                fprintf(fp," \\underline{~~~~~~~~~~~~~~~~~~~~~~~} ");

        if(strcmp(entry[index]->marriage_place,""))
            fprintf(fp,"at %s", entry[index]->marriage_place);
        else
            if(leave_spaces_in_output)
                fprintf(fp,"at \\underline{~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~} ");
        }
    fprintf(fp,"\\\\\n");

    initialise = 1;
    for(k=0;k<entry[index]->number_of_children;k++)
        {
        for(l=0;l<entry[entry[index]->spouse[j]]->number_of_children;l++)
            {
            if(entry[index]->child[k] == 
                       entry[entry[index]->spouse[j]]->child[l])
                {
                if(initialise)
                    fprintf(fp,"~~~~~~~~~~Children: \\\\\n");
                if(leave_spaces_in_output)
                    {
                    if(strcmp(entry[entry[index]->child[k]]->forenames,""))
                        strcpy(string1,entry[entry[index]->child[k]]->forenames);
                    else
                        strcpy(string1,"\\underline{~~~~~~~~~~~~~~~~~}");

                    if(strcmp(entry[entry[index]->child[k]]->surname,""))
                        strcpy(string2,entry[entry[index]->child[k]]->surname);
                    else
                        strcpy(string2,"\\underline{~~~~~~~~~~~~~~~~~~~~~}");
                    }
                else
                    {
                    strcpy(string1,entry[entry[index]->child[k]]->forenames);
                    strcpy(string2,entry[entry[index]->child[k]]->surname);
                    }

                fprintf(fp,"~~~~~~~~~~~~~~~%s %s \\\\\n", string1, string2);
                initialise = 0;
                }
            }
        }
    }
if(entry[index]->number_of_spouses > 0)
    {
    fprintf(fp,"\\end{flushleft}\n\n");
    fprintf(fp,"\\vspace*{5mm}\n\n");
    }

/* person's occupation */

if(strcmp(entry[index]->occupation,"") || (leave_spaces_in_output))
    {
    if(strcmp(entry[index]->occupation,""))
        fprintf(fp,"Occupation(s): %s\n\n",entry[index]->occupation);
    else
        fprintf(fp,"Occupation(s): \\underline{~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~}\n\n");
        
    fprintf(fp,"\\vspace*{5mm}\n\n");
    }

/* miscellaneous notes */

if(strcmp(entry[index]->notes,"") || (leave_spaces_in_output))
    {
    if(strcmp(entry[index]->notes,""))
        fprintf(fp,"Notes: %s\n",entry[index]->notes);
    else
        fprintf(fp,"Notes: \\underline{~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~}\n\n");
    }

return;
}

/*
   function to output the information for the specified database
   entry to the specified file in PostScript
*/


void  postscript_output(fp, index)

FILE   *fp;
int     index;
{
float  x_coord, y_coord, left_margin, vert_line_separation, page_width;
float  indent_1, indent_2, indent_3, top_of_page, bottom_margin;
int    i, j, k, l, initialise1, initialise2, attached;
char   sub_string[300], main_string[300], string1[100], string2[100], string3[100];

/* initialise x and y-coords and set vertical line separation */

x_coord = left_margin = 72.0;        /* left margin in points */
top_of_page = A4_HEIGHT_PTS - 108.0;  /* initial y location for text */
y_coord = top_of_page;
vert_line_separation = 20.0;     /* the vertical line separation in points  */
                                 /* hard coded at present in a lot of cases below */
page_width = 490.0;              /* the horizontal space for text in points */
bottom_margin = 72.0;            /* height from bottom of page below which
                                    a new page is called */
indent_1 = 36.0;                 /* first indentation distance in points */
indent_2 = 72.0;                 /* second indentation distance in points */
indent_3 = 108.0;                /* third indentation distance in points */

/* move to the start */
fprintf(fp,"/yline %.1f def\n", y_coord);
fprintf(fp,"%.1f yline moveto\n", x_coord);

/* set the font */
fprintf(fp,"/Times-Roman findfont 12 scalefont setfont\n");

/* title and forenames */

if(leave_spaces_in_output)
    {
    if(strcmp(entry[index]->title,""))
        strcpy(string1,entry[index]->title);
    else
        strcpy(string1,"_____");
    if(strcmp(entry[index]->forenames,""))
        strcpy(string2,entry[index]->forenames);
    else
        strcpy(string2,"____________________");
    }
else
    {
    strcpy(string1,entry[index]->title);
    strcpy(string2,entry[index]->forenames);
    }

fprintf(fp,"(%s %s ) show\n", string1, string2);

/* surname (maybe with maiden name) */

if(leave_spaces_in_output)
    {
    if(strcmp(entry[index]->surname,""))
        strcpy(string1,entry[index]->surname);
    else
        strcpy(string1,"____________________");
    }
else
    {
    strcpy(string1,entry[index]->surname);
    }

/* if there is a maiden name, add it on */

if(strcmp(entry[index]->maiden_name,""))
    {
    sprintf(string2," (nee %s)",entry[index]->maiden_name);
    strcat(string1,string2);
    }
else
    {
    if(leave_spaces_in_output)
        {
        /* check the person is female and married before leaving space for a maiden name */
        if((entry[index]->gender == FEMALE) && (entry[index]->number_of_spouses > 0))
            strcat(string1," (nee ______________________ )");
        }
    }

/* set the font larger for the person's surname */
fprintf(fp,"/Times-Roman findfont 15 scalefont setfont\n");

fprintf(fp,"(%s) show\n", string1);

/* reset the font */
fprintf(fp,"/Times-Roman findfont 12 scalefont setfont\n");

/* move down 2 lines */
fprintf(fp,"/yline yline %.1f sub def\n",2.0 * vert_line_separation);

/* birth date, place and source if available */

if(strcmp(entry[index]->birth_date,"") || strcmp(entry[index]->birth_place,"") ||
                                         (leave_spaces_in_output))
    {
    sprintf(main_string,"Born ");
    if(strcmp(entry[index]->birth_date,""))
        {
        sprintf(sub_string," %s ",entry[index]->birth_date);
        strcat(main_string,sub_string);
        }
    else
        {
        if(leave_spaces_in_output)
            strcat(main_string," _______________");
        }
    if(strcmp(entry[index]->birth_place,""))
        {
        sprintf(sub_string,"at %s",entry[index]->birth_place);
        strcat(main_string,sub_string);
        }
    else
        {
        if(leave_spaces_in_output)
            strcat(main_string," at ____________________");
        }
    fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} if %.1f \
yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width,bottom_margin,top_of_page,x_coord);

    if(strcmp(entry[index]->birth_source,""))
        {
        sprintf(main_string,"source: %s", entry[index]->birth_source);
        fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} if %.1f \
yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width - indent_1,bottom_margin,
            top_of_page,x_coord + indent_1);
        }
    else
        {
        if(leave_spaces_in_output)
            {
            strcpy(main_string,"source: ____________________");
            fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} if %.1f \
yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width - indent_1,bottom_margin,
                top_of_page,x_coord + indent_1);
            }
        }
    }

/* baptism date, place and source if available */

if(strcmp(entry[index]->baptism_date,"") || strcmp(entry[index]->baptism_place,"") ||
                                         (leave_spaces_in_output))
    {
    sprintf(main_string,"Baptised ");
    if(strcmp(entry[index]->baptism_date,""))
        {
        sprintf(sub_string," %s ",entry[index]->baptism_date);
        strcat(main_string,sub_string);
        }
    else
        {
        if(leave_spaces_in_output)
            strcat(main_string," _______________");
        }
    if(strcmp(entry[index]->baptism_place,""))
        {
        sprintf(sub_string,"at %s",entry[index]->baptism_place);
        strcat(main_string,sub_string);
        }
    else
        {
        if(leave_spaces_in_output)
            strcat(main_string," at ____________________");
        }
    fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} if %.1f \
yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width,bottom_margin,top_of_page,x_coord);

    if(strcmp(entry[index]->baptism_source,""))
        {
        sprintf(main_string,"source: %s", entry[index]->baptism_source);
        fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} if %.1f \
yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width - indent_1,bottom_margin,
            top_of_page,x_coord + indent_1);
        }
    else
        {
        if(leave_spaces_in_output)
            {
            strcpy(main_string,"source: ____________________");
            fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} if %.1f \
yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width - indent_1,bottom_margin,
                top_of_page,x_coord + indent_1);
            }
        }
    }

/* 
   death date, place and source, and resting place if available (or necessary i.e.
   don't need to display if person is still alive
*/

if((strcmp(entry[index]->death_date,"") || strcmp(entry[index]->death_place,"") || 
             (leave_spaces_in_output)) && (entry[index]->status == DECEASED))
    {
    strcpy(main_string,"Died ");
    if(strcmp(entry[index]->death_date,""))
        {
        sprintf(sub_string," %s ",entry[index]->death_date);
        strcat(main_string,sub_string);
        }
    else
        {
        if(leave_spaces_in_output)
            strcat(main_string," _______________");
        }
    if(strcmp(entry[index]->death_place,""))
        {
        sprintf(sub_string,"at %s",entry[index]->death_place);
        strcat(main_string,sub_string);
        }
    else
        {
        if(leave_spaces_in_output)
            strcat(main_string," at ____________________");
        }
    fprintf(fp,"(%s)\n %.1f {%.1f yline moveto show /yline yline 18 sub def}\n\
                BreakIntoLines\n\n",main_string,page_width,x_coord);

    if(strcmp(entry[index]->death_source,""))
        {
        sprintf(main_string,"source: %s", entry[index]->death_source);
        fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} if %.1f \
yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width - indent_1,bottom_margin,
            top_of_page,x_coord + indent_1);
        }
    else
        {
        if(leave_spaces_in_output)
            {
            strcpy(main_string,"source: ____________________");
            fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} if %.1f \
yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width - indent_1,bottom_margin,
                top_of_page,x_coord + indent_1);
            }
        }

    if(strcmp(entry[index]->resting_place,""))
        {
        sprintf(main_string,"Resting Place: %s", entry[index]->resting_place);
        fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} if %.1f \
yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width,bottom_margin,top_of_page,x_coord);
        }
    else
        {
        if(leave_spaces_in_output)
            {
            strcpy(main_string,"Resting Place: ____________________");
            fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} if %.1f \
yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width,bottom_margin,top_of_page,x_coord);
            }
        }
    }

/* mothers name */

if(entry[index]->mother != -1)
    {
    if(leave_spaces_in_output)
        {
        if(strcmp(entry[entry[index]->mother]->title,""))
            strcpy(string1,entry[entry[index]->mother]->title);
        else
            strcpy(string1,"_____");

        if(strcmp(entry[entry[index]->mother]->forenames,""))
            strcpy(string2,entry[entry[index]->mother]->forenames);
        else
            strcpy(string2,"_________________");

        if(strcmp(entry[entry[index]->mother]->surname,""))
            strcpy(string3,entry[entry[index]->mother]->surname);
        else
            strcpy(string3,"_________________");
        }
    else
        {
        strcpy(string1,entry[entry[index]->mother]->title);
        strcpy(string2,entry[entry[index]->mother]->forenames);
        strcpy(string3,entry[entry[index]->mother]->surname);
        }

    /* move down a line */
    fprintf(fp,"/yline yline %.1f sub def\n",vert_line_separation);

    /* position the current point at the left margin */
    fprintf(fp,"%.1f yline moveto\n",x_coord);

    fprintf(fp,"(Mother: %s %s %s) show\n",string1, string2, string3);
    }
else
    {
    /* if there is no mother but leave spaces is enabled */
    if(leave_spaces_in_output)
        {
        /* move down a line */
        fprintf(fp,"/yline yline %.1f sub def\n",vert_line_separation);

        /* position the current point at the left margin */
        fprintf(fp,"%.1f yline moveto\n",x_coord);

        strcpy(string1,"_____");
        strcpy(string2,"_________________");
        strcpy(string3,"_________________");

        fprintf(fp,"(Mother: %s %s %s) show\n",string1, string2, string3);
        }
    } 

/* fathers name */

if(entry[index]->father != -1)
    {
    if(leave_spaces_in_output)
        {
        if(strcmp(entry[entry[index]->father]->title,""))
            strcpy(string1,entry[entry[index]->father]->title);
        else
            strcpy(string1,"_____");

        if(strcmp(entry[entry[index]->father]->forenames,""))
            strcpy(string2,entry[entry[index]->father]->forenames);
        else
            strcpy(string2,"_________________");

        if(strcmp(entry[entry[index]->father]->surname,""))
            strcpy(string3,entry[entry[index]->father]->surname);
        else
            strcpy(string3,"_________________");
        }
    else
        {
        strcpy(string1,entry[entry[index]->father]->title);
        strcpy(string2,entry[entry[index]->father]->forenames);
        strcpy(string3,entry[entry[index]->father]->surname);
        }

    /* move down a line */
    fprintf(fp,"/yline yline %.1f sub def\n",vert_line_separation);

    /* position the current point at the left margin */
    fprintf(fp,"%.1f yline moveto\n",x_coord);

    fprintf(fp,"(Father: %s %s %s) show\n",string1, string2, string3);
    }
else
    {
    /* if there is no father but leave spaces is enabled */
    if(leave_spaces_in_output)
        {
        /* move down a line */
        fprintf(fp,"/yline yline %.1f sub def\n",vert_line_separation);

        /* position the current point at the left margin */
        fprintf(fp,"%.1f yline moveto\n",x_coord);

        strcpy(string1,"_____");
        strcpy(string2,"_________________");
        strcpy(string3,"_________________");

        fprintf(fp,"(Father: %s %s %s) show\n",string1, string2, string3);
        }
    } 

/* move down 2 lines if there are any spouses or any children */
 
if((entry[index]->number_of_spouses > 0) || (entry[index]->number_of_children > 0))
    {
    /* move down 2 lines */
    fprintf(fp,"/yline yline %.1f sub def\n",2.0 * vert_line_separation);
    }

/* spouse (and related child) information */

initialise1 = 1;
for(j=0;j<entry[index]->number_of_spouses;j++)
    {
    if(leave_spaces_in_output)
        {
        if(strcmp(entry[entry[index]->spouse[j]]->forenames,""))
            strcpy(string1,entry[entry[index]->spouse[j]]->forenames);
        else
            strcpy(string1,"_________________");

        if(strcmp(entry[entry[index]->spouse[j]]->surname,""))
            strcpy(string2,entry[entry[index]->spouse[j]]->surname);
        else
            strcpy(string2,"_________________");
        }
    else
        {
        strcpy(string1,entry[entry[index]->spouse[j]]->forenames);
        strcpy(string2,entry[entry[index]->spouse[j]]->surname);
        }
    sprintf(main_string,"Married %s %s",string1, string2);

    /* add the maiden name if there is one */

    if(strcmp(entry[entry[index]->spouse[j]]->maiden_name,""))
        {
        sprintf(sub_string," (nee %s)",entry[entry[index]->spouse[j]]->maiden_name);
        strcat(main_string,sub_string);
        }
    else
        {
        /* 
           if no maiden name info but spouse is female and leave spaces is enabled, 
           add the spaces 
        */
        if((leave_spaces_in_output) && (entry[entry[index]->spouse[j]]->gender == FEMALE))
            {
            sprintf(sub_string," (nee _________________)",
                                entry[entry[index]->spouse[j]]->maiden_name);
            strcat(main_string,sub_string);
            }
        }

    /*
       the marriage date, place and source are going to have to wait until I incorporate
       an array of marriage dates into the struct, one for each spouse
       for now if there is only one spouse and there is marriage date, place and/or source
       information, use it
    */

    if(entry[index]->number_of_spouses == 1)
        {
        if(strcmp(entry[index]->marriage_date,""))
            sprintf(string1," %s ",entry[index]->marriage_date);
        else
            if(leave_spaces_in_output)
                strcpy(string1," _________________ ");
            else
                strcpy(string1,"");

        strcat(main_string, string1);

        if(strcmp(entry[index]->marriage_place,""))
            sprintf(string1," at %s ",entry[index]->marriage_place);
        else
            if(leave_spaces_in_output)
                strcpy(string1," at _________________ ");
            else
                strcpy(string1,"");

        strcat(main_string, string1);

        if(strcmp(entry[index]->marriage_source,""))
            sprintf(string1,"(source: %s)", entry[index]->marriage_source);
        else
            if(leave_spaces_in_output)
                strcpy(string1,"(source: ____________________)");
            else
                strcpy(string1,"");

        strcat(main_string, string1);

        }

    fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} if %.1f \
yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width,bottom_margin,top_of_page,x_coord);

    initialise1 = 1;
    for(k=0;k<entry[index]->number_of_children;k++)
        {
        for(l=0;l<entry[entry[index]->spouse[j]]->number_of_children;l++)
            {
            if(entry[index]->child[k] == 
                       entry[entry[index]->spouse[j]]->child[l])
                {
                if(initialise1)
                    {
                    sprintf(main_string,"Children");
                    fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} \
if %.1f yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width - indent_1,bottom_margin,
            top_of_page,x_coord + indent_1);
                    }
                if(leave_spaces_in_output)
                    {
                    if(strcmp(entry[entry[index]->child[k]]->forenames,""))
                        strcpy(string1,entry[entry[index]->child[k]]->forenames);
                    else
                        strcpy(string1,"_________________");

                    if(strcmp(entry[entry[index]->child[k]]->surname,""))
                        strcpy(string2,entry[entry[index]->child[k]]->surname);
                    else
                        strcpy(string2,"_________________");
                    }
                else
                    {
                    strcpy(string1,entry[entry[index]->child[k]]->forenames);
                    strcpy(string2,entry[entry[index]->child[k]]->surname);
                    }

                sprintf(main_string,"%s %s",string1, string2);

                if(strcmp(entry[entry[index]->child[k]]->birth_date,""))
                    {
                    sprintf(sub_string,", born %s",
                               entry[entry[index]->child[k]]->birth_date);
                    strcat(main_string,sub_string);
                    }
                else
                    {
                    if(leave_spaces_in_output)
                        {
                        sprintf(sub_string,", born _______________");
                        strcat(main_string,sub_string);
                        }
                    }
                if(strcmp(entry[entry[index]->child[k]]->birth_place,""))
                    {
                    sprintf(sub_string," at %s",
                               entry[entry[index]->child[k]]->birth_place);
                    strcat(main_string,sub_string);
                    }
                else
                    {
                    if(leave_spaces_in_output)
                        {
                        sprintf(sub_string," at ____________________");
                        strcat(main_string,sub_string);
                        }
                    }

                fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} \
if %.1f yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width - indent_2,bottom_margin,
                    top_of_page,x_coord + indent_2);
                initialise1 = 0;
                }
            }
        }
    }

/* handle any children who do not have a related spouse */

initialise2 = 1;
for(i=0;i<entry[index]->number_of_children;i++)
    {
    attached = 0;
    for(j=0;j<entry[index]->number_of_spouses;j++)
        {
        if(entry[index]->gender == MALE)
            {
            if(entry[entry[index]->child[i]]->mother == entry[index]->spouse[j])
                attached = 1;
            }
        else
            {
            if(entry[entry[index]->child[i]]->father == entry[index]->spouse[j])
                attached = 1;
            }
        }
    if(!attached)
        {
        if(initialise2)
            {
            /* 
               if there are no children so far then heading is "Children", otherwise
               it is "Other Children"
            */
            if(!initialise1)
                sprintf(main_string,"Other Children");
            else
                sprintf(main_string,"Children");
            fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} \
if %.1f yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width,bottom_margin,top_of_page,x_coord);
            initialise2 = 0;
            }
            if(leave_spaces_in_output)
                {
                if(strcmp(entry[entry[index]->child[i]]->forenames,""))
                    strcpy(string1,entry[entry[index]->child[i]]->forenames);
                else
                    strcpy(string1,"_________________");

                if(strcmp(entry[entry[index]->child[i]]->surname,""))
                    strcpy(string2,entry[entry[index]->child[i]]->surname);
                else
                    strcpy(string2,"_________________");
                }
            else
                {
                strcpy(string1,entry[entry[index]->child[i]]->forenames);
                strcpy(string2,entry[entry[index]->child[i]]->surname);
                }
        sprintf(main_string,"%s %s",string1, string2);

        if(strcmp(entry[entry[index]->child[i]]->birth_date,""))
            {
            sprintf(sub_string,", born %s",
                       entry[entry[index]->child[i]]->birth_date);
            strcat(main_string,sub_string);
            }
            else
                {
                if(leave_spaces_in_output)
                    {
                    sprintf(sub_string,", born _______________");
                    strcat(main_string,sub_string);
                    }
                }
        if(strcmp(entry[entry[index]->child[i]]->birth_place,""))
            {
            sprintf(sub_string," at %s",
                       entry[entry[index]->child[i]]->birth_place);
            strcat(main_string,sub_string);
            }
            else
                {
                if(leave_spaces_in_output)
                    {
                    sprintf(sub_string," at ____________________");
                    strcat(main_string,sub_string);
                    }
                }
        fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} \
if %.1f yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width - indent_1,bottom_margin,
                top_of_page,x_coord + indent_1);
        }
    }

if(strcmp(entry[index]->occupation,"") || strcmp(entry[index]->notes,"") || 
                                                  (leave_spaces_in_output))
    {
    /* move down 2 lines */
    fprintf(fp,"/yline yline %.1f sub def\n",2.0 * vert_line_separation);
    }

/* person's occupation */

if(strcmp(entry[index]->occupation,"") || (leave_spaces_in_output))
    {
    if(strcmp(entry[index]->occupation,""))
        sprintf(main_string,"Occupation(s): %s",entry[index]->occupation);
    else
        strcpy(main_string,"Occupation(s): ___________________________________");
        
    fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} \
if %.1f yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width,bottom_margin,top_of_page,x_coord);
 
    /* move down a line */
    fprintf(fp,"/yline yline %.1f sub def\n",vert_line_separation);
    }

/* miscellaneous notes */

if(strcmp(entry[index]->notes,"") || (leave_spaces_in_output))
    {
    if(strcmp(entry[index]->notes,""))
        sprintf(main_string,"Notes: %s",entry[index]->notes);
    else
        strcpy(main_string,"Notes: ___________________________________");
        

    fprintf(fp,"(%s)\n %.1f {yline %.1f lt {showpage /yline %.1f def} \
if %.1f yline moveto show /yline yline 18 sub def}\n\
BreakIntoLines\n\n",main_string,page_width,bottom_margin,top_of_page,x_coord);
 
    /* move down a line */
    fprintf(fp,"/yline yline %.1f sub def\n",vert_line_separation);
    }

return;
}

/*
   function to write the initial information to a PostScript text
   output file
*/

void  postscript_init(fp)

FILE *fp;
{

fprintf(fp,"%%!PS - generated by fdb V%s\n\n\n", FDB_VERSION);

fprintf(fp,"%% BreakIntoLines deals with long text lines gracefully\n");
fprintf(fp,"/wordbreak ( ) def\n");
fprintf(fp,"/BreakIntoLines\n");
fprintf(fp,"    {\n");
fprintf(fp,"    /proc       exch  def\n");
fprintf(fp,"    /linewidth  exch  def\n");
fprintf(fp,"    /textstring exch  def\n");
fprintf(fp,"    /breakwidth wordbreak stringwidth pop def\n");
fprintf(fp,"    /curwidth 0 def\n");
fprintf(fp,"    /lastwordbreak 0 def\n");
fprintf(fp,"    /startchar 0 def\n");
fprintf(fp,"    /restoftext textstring def\n");
fprintf(fp,"        {restoftext wordbreak search\n");
fprintf(fp,"            {/nextword exch def pop\n");
fprintf(fp,"            /restoftext exch def\n");
fprintf(fp,"            /wordwidth nextword stringwidth pop def\n");
fprintf(fp,"            curwidth wordwidth add linewidth gt\n");
fprintf(fp,"                {textstring startchar\n");
fprintf(fp,"                 lastwordbreak startchar sub\n");
fprintf(fp,"                 getinterval proc\n");
fprintf(fp,"                 /startchar lastwordbreak def\n");
fprintf(fp,"                 /curwidth wordwidth breakwidth add def}\n");
fprintf(fp,"                {/curwidth curwidth wordwidth add breakwidth add def\n");
fprintf(fp,"                }ifelse\n");
fprintf(fp,"            /lastwordbreak lastwordbreak nextword length add 1 add def\n");
fprintf(fp,"            }\n");
fprintf(fp,"            { pop exit }\n");
fprintf(fp,"            ifelse\n");
fprintf(fp,"        } loop\n");
fprintf(fp,"        /lastchar textstring length def\n");
fprintf(fp,"        textstring startchar lastchar startchar sub\n");
fprintf(fp,"        getinterval proc \n");
fprintf(fp,"    }def\n");
fprintf(fp,"\n");
}

/*
   the recursive descendants tree PostScript plotter
*/

void  plot_node(person, fp, x_offset, y_offset, scale, 
                            larger_font_size, smaller_font_size)

struct db_descendant  *person;
FILE                  *fp;
float                  x_offset, y_offset, scale, larger_font_size, smaller_font_size;
{
int  person_index, i, j;

#ifdef DEBUG
printf("in plot_node - x_offset = %f, y_offset = %f, scale = %f\n",x_offset,y_offset,scale);
#endif

/* read the index of the descendant */

person_index = person->index;

/* 
   display the information of the descendant
*/

/* add the forenames if not empty, otherwise put a line in */

if(strcmp(person->forenames,""))
    {
    plot_centred(fp, x_offset + scale * person->location.x, 
                             - y_offset - scale * person->location.y, 
                             person->forenames, larger_font_size);
    }
else
    {
    plot_line(fp, x_offset + scale * person->location.x, 
                  - y_offset - scale * (person->location.y + 
                      ((int) floor(-0.8 * ((double) vertical_text_separation)))),
                  x_offset + scale * person->location.x,
                  - y_offset - scale * (person->location.y + 
                      ((int) floor(0.2 * ((double) vertical_text_separation)))));
    }


/* add the surname if not empty, otherwise put a line in */
if(strcmp(person->surname,""))
    {
    plot_centred(fp, x_offset + scale * person->location.x, 
            - y_offset - scale * (person->location.y + 1 * vertical_text_separation), 
            person->surname, larger_font_size);
    }
else
    {
    plot_line(fp,  
              x_offset + scale * person->location.x, 
              - y_offset - scale * (person->location.y + 
                  ((int) floor(0.2 * ((double) vertical_text_separation)))),
              x_offset + scale * person->location.x,
              - y_offset - scale * (person->location.y + 
                  ((int) floor(1.2 * ((double) vertical_text_separation)))));

    }


/* add the date of birth of known, otherwise put a line in */
if(strcmp(person->birth_date,""))
    {
    plot_centred(fp, x_offset + scale * person->location.x, 
            - y_offset - scale * (person->location.y + 2 * vertical_text_separation), 
            person->birth_date, smaller_font_size);
    }
else
    {
#ifdef DEBUG
    printf("no birthdate for for %s %s - drawing line from (%f,%f) to (%f,%f)\n",
        person->forenames,person->surname,x_offset + scale * (float) person->location.x,
              - y_offset - scale * (float) (person->location.y + 
                  ((int) floor(1.2 * ((double) vertical_text_separation)))),
              x_offset + scale * (float) person->location.x,
              - y_offset - scale * (float) (person->location.y + 
                  ((int) floor(2.2 * ((double) vertical_text_separation)))));
#endif
    plot_line(fp, x_offset + scale * (float) person->location.x, 
                  - y_offset - scale * (float) (person->location.y + 
                      ((int) floor(1.2 * ((double) vertical_text_separation)))),
                  x_offset + scale * (float) person->location.x,
                  - y_offset - scale * (float) (person->location.y + 
                      ((int) floor(2.2 * ((double) vertical_text_separation)))));
    }

/* 
   loop through the spouses, displaying the relevant info 
*/ 

/* draw the little stub down from the person to the spouse across line */

if(person->number_of_spouses > 0)
    {
#ifdef DEBUG
    printf("drawing person to spouse-cross-line stub\n");
#endif
    plot_line(fp, x_offset + scale * person->location.x, 
                  - y_offset - scale * (float) (person->location.y + 
                      ((int) floor(2.2 * ((double) vertical_text_separation)))),
                  x_offset + scale * person->location.x,
                  - y_offset - scale * (float) (person->location.y + 
                      ((int) floor(2.5 * ((double) vertical_text_separation)))));
    }

if(person->number_of_spouses > 0)
    {
    /* 
       actually there should always be at least one spouse if there are any
       children, even if it is only a single dummy entry
    */
    /* draw a line above the spouses from first to last */

#ifdef DEBUG
    printf("drawing spouses (0 to %d) cross line\n", person->number_of_spouses - 1);
#endif

    plot_line(fp, x_offset + scale * person->spouse_location[0].x, 
                  - y_offset - scale * (person->spouse_location[0].y - 
                      ((int) floor((double) 1.5 * vertical_text_separation))),
                  x_offset + scale * person->spouse_location[person->number_of_spouses - 1].x,
                  - y_offset - scale * (person->spouse_location[person->number_of_spouses - 1].y - 
                      ((int) floor((double) 1.5 * vertical_text_separation))));

    for(i=0;i<person->number_of_spouses;i++)
        {
        /* 
           if there is actually a spouse (not a dummy) display the name 
           otherwise draw a line through
        */

        if(strcmp(person->spouse_name[i],""))
            {
            plot_centred(fp, x_offset + scale * person->spouse_location[i].x, 
                    - y_offset - scale * person->spouse_location[i].y,
                    person->spouse_name[i], smaller_font_size);
            }
        else
            {
#ifdef DEBUG
            printf("dummy spouse, drawing line\n");
#endif
            plot_line(fp, x_offset + scale * person->spouse_location[i].x, 
                          - y_offset - scale * (person->spouse_location[i].y + 
                              1 * vertical_text_separation),
                          x_offset + scale * person->spouse_location[i].x, 
                          - y_offset - scale * (person->spouse_location[i].y - 
                              1 * vertical_text_separation));
            }

        /* draw the stub under the spouse */

#ifdef DEBUG
        printf("drawing spouse under-stub\n");
#endif
        plot_line(fp, x_offset + scale * person->spouse_location[i].x, 
                      - y_offset - scale * (person->spouse_location[i].y + 
                          ((int) floor(0.25 *((double) vertical_text_separation)))),
                      x_offset + scale * person->spouse_location[i].x, 
                      - y_offset - scale * (person->spouse_location[i].y + 
                          ((int) floor(1.0 *((double) vertical_text_separation)))));
          
            /* draw the stub over the spouse */

#ifdef DEBUG
        printf("drawing spouse over-stub\n");
#endif
        plot_line(fp, x_offset + scale * person->spouse_location[i].x, 
                      - y_offset - scale * (person->spouse_location[i].y - 
                          ((int) floor(1.0 *((double) vertical_text_separation)))),
                      x_offset + scale * person->spouse_location[i].x, 
                      - y_offset - scale * (person->spouse_location[i].y -
                          ((int) floor(1.5 *((double) vertical_text_separation)))));
          
        /* draw the children`s cross bar (if there are any) */

        if(person->num_children_of_spouse[i] > 0)
            {
#ifdef DEBUG
            printf("drawing child cross bar\n");
#endif
            plot_line(fp, x_offset + scale * person->children[i][0]->location.x, 
                          - y_offset - scale * (person->spouse_location[i].y + 
                              vertical_text_separation),
                          x_offset + scale * person->children[i][person->
                              num_children_of_spouse[i] - 1]->location.x,
                          - y_offset - scale * (person->spouse_location[i].y + 
                              vertical_text_separation));
            }
          
        /* loop through the children, plotting them and so on and on and on... */

        for(j=0;j<person->num_children_of_spouse[i];j++)
            {
            /* draw the stub to the cross bar above the child */

            plot_line(fp, x_offset + scale * person->children[i][j]->location.x, 
                          - y_offset - scale * (person->children[i][j]->location.y - 
                              ((int) floor(1.0 *((double) vertical_text_separation)))),
                          x_offset + scale * person->children[i][j]->location.x, 
                          - y_offset - scale * (person->children[i][j]->location.y -
                              ((int) floor(3.0 *((double) vertical_text_separation)))));
          
            plot_node(person->children[i][j], fp, x_offset, y_offset, scale, 
                                  larger_font_size, smaller_font_size);
            }
        }
    }
else
    {
    /*
       if there are no known spouses then simply plot the children
    */

    for(j=0;j<entry[person_index]->number_of_children;j++)
        plot_node(person->children[0][j], fp, x_offset, y_offset, scale, 
                              larger_font_size, smaller_font_size);
    }
}

/*
   plot the current descendants tree to a PostScript file

   the name of the file will be name_treeN.ps where name
   is the person at the top of the tree and N is
   the smallest integer for which the file does not
   already exist
*/

void  plot_des_tree()

{
float  height_scale, width_scale, scale, ratio;
float  larger_font_size, smaller_font_size;
float  x_offset, y_offset;
int    i, j, num_lines, num_items;
char   filename[255],working[255],plot_file[255],substring[255],initial[3];
char   test_filename[255];
FILE  *fp;
/* 
   first check that there is some data to plot i.e. at least one
   display struct is non-NULL (they are used sequentially, so
   if the first one is NULL they all are)
*/

if(descendant[0] == (struct db_descendant *) NULL)
    {
    sprintf(error_message,"No Descendants Tree Data Available!\n");
    show_error();
    return;
    }

/* 
   find a suitable name for the output file, the code assumes
   that if a file cannot be opened for reading, then it does
   not exist
*/

/* initialise the file name string */
strcpy(filename,"");

/* 
   read the person at foot of the descent line's forename string into the work space, 
   converting to upper case on the way 
*/

i = 0;
while((working[i] = toupper(entry[descendant[0]->index]->forenames[i])) != '\0')
    i++;

#ifdef DEBUG
printf("forename string: %s\n",working);
#endif

/* read in substrings and extract the first letters */

j = 0;

while( ((num_items = sscanf((working + j),"%s",substring)) != EOF) &&
       (j < strlen(working)) )
    {
#ifdef DEBUG
    printf("working on substring: %s\n",substring);
#endif

    sprintf(initial,"%c",substring[0]);

#ifdef DEBUG
    printf("first letter: **%s**\n",initial);
#endif
    /* add the initial to the file name */

    strcat(filename,initial);

    /* add an underscore */

    strcat(filename,"_");
	
    /* increment the distance along the string counter */

    j += strlen(substring) + 1;
#ifdef DEBUG
    printf("j = %d\n",j);
#endif
    }

/* add the surname on the end, ensuring the first letter is capitalised */

sscanf(entry[descendant[0]->index]->surname,"%s",working);
working[0] = toupper(working[0]);

/* append the surname to the file name */

strcat(filename,working);

/* 
   ensure that file does not already exist, add an integer to the
   file name to make it unique if necessary 
*/

sprintf(test_filename,"%s_tree.ps",filename);

/* check if the file already exists */

j = 1;
while((fp = fopen(test_filename,"r")) != NULL)
    {
    sprintf(test_filename,"%s_tree%d.ps",filename,j);
    j++;
    }

strcpy(plot_file, test_filename);

#ifdef DEBUG
printf("file name: %s\n",plot_file);
#endif


/* open the chosen file name for writing and write the header info */

if ((fp = fopen(plot_file, "w")) == NULL)
    {
    sprintf(error_message,"Unable to Open File %s for Plot Creation!",plot_file);
    show_error();
    return;
    }

/* write the header comment to the file */
fprintf(fp,"%%!PS - descendants tree plot generated using fdb V%s - cadellinsoft 94\n\n",
             FDB_VERSION);

/* decide whether to draw the plot portrait or landscape */

ratio = ((float) des_canvas_height)/((float) des_canvas_width);

#ifdef DEBUG
printf("ratio = %f\n",ratio);
#endif

/* 
   also calculate the scale factor from pixels to points to allow, amongst
   other things, the font sizes to be set
*/

if(ratio < 1.0)
    {
    /* rotate and translate the axes appropriately */
    fprintf(fp,"%%alter axes for landscape plot\n-90 rotate\n-%.1f 0.0 translate\n\n",A4_HEIGHT_PTS);
    /* set the y origin at the top of the page */
    fprintf(fp,"%%translate y axis to accomodate X11 coordinates\n");
    fprintf(fp,"0.0 %.1f translate\n\n",A4_WIDTH_PTS);
    height_scale = A4_WIDTH_PTS/((float) des_canvas_height);
    width_scale  = A4_HEIGHT_PTS/((float) des_canvas_width);
#ifdef DEBUG
    printf("selecting landscape\n");
#endif
    }
else
    {
    /* set the y origin at the top of the page */
    fprintf(fp,"%%translate y axis to accomodate X11 coordinates\n");
    fprintf(fp,"0.0 %.1f translate\n\n",A4_HEIGHT_PTS);
    height_scale = A4_HEIGHT_PTS/((float) des_canvas_height);
    width_scale  = A4_WIDTH_PTS/((float) des_canvas_width);
#ifdef DEBUG
    printf("selecting portrait\n");
#endif
    }

if(height_scale < width_scale)
    {
    if(ratio < 1.0)
        x_offset = (A4_HEIGHT_PTS - (des_canvas_width * height_scale ))/2;
    else
        x_offset = (A4_WIDTH_PTS - (des_canvas_width * height_scale ))/2;

    y_offset = 0;
    scale = height_scale;
    }
else
    {
    if(ratio < 1.0)
        y_offset = (A4_WIDTH_PTS - (des_canvas_height * width_scale ))/2;
    else
        y_offset = (A4_HEIGHT_PTS - (des_canvas_height * width_scale ))/2;

    x_offset = 0;
    scale = width_scale;
    }

scale = height_scale < width_scale ? height_scale : width_scale;

#ifdef DEBUG
printf("in plot_descendants - x_offset = %f, y_offset = %f, scale = %f\n",x_offset,y_offset,scale);
#endif

larger_font_size  = scale * ((float) LARGER_FONT);
smaller_font_size = scale * ((float) SMALLER_FONT);

/* add the cadellinsoft logo */

put_moggy(fp, 36, -36);

/* call the recursive draw routine */

plot_node(descendant[0], fp, x_offset, y_offset, scale, 
                             larger_font_size, smaller_font_size);

/* write the closing info to the file and close the file */
fprintf(fp,"\nshowpage\n");
fclose(fp);

/* tell the user where the plot has been written to */

sprintf(error_message,"Descendants Tree Line Plot Written to File %s",plot_file);
show_error();

}

/* draws the cadellinsoft logo in the specified file at the specified coordinates */

void  put_moggy(fp, x, y)

FILE  *fp;
int    x, y;
{
/* temporarily move the origin to the desired location */
fprintf(fp,"%d %d translate\n",x,y);

fprintf(fp,"36 36 true [2 0 0 -2 0 0]\n");

fprintf(fp,"{< 00 06 40 80 00 00 02 61 80 00 00 09\n  \
   7f 80 00 00 07 ff 80 00 00 08 ff c0\n  \
   00 00 00 ff fe 00 00 01 ff e1 00 00\n  \
   01 ff f0 00 00 07 ff ce 00 00 0d ff\n  \
   c1 00 00 1a 3f c0 00 00 14 3f e0 00\n  \
   00 04 7f e0 00 00 00 7f e0 00 00 00\n  \
   7f e0 00 00 00 ff e0 00 1c 01 ff c0\n  \
   00 38 03 ff e0 00 30 07 ff e0 00 30\n  \
   07 ff f0 00 70 0f ff f0 00 70 0f ff\n  \
   f0 00 70 0f ff f0 00 70 0f ff f0 00\n  \
   38 0f ff f0 00 3c 07 ff f0 00 1e 07\n  \
   ff e0 00 0f 07 ff e0 00 0f c7 ff e0\n  \
   00 07 e3 ff e0 00 03 f3 ff f0 00 01\n  \
   ff ff f0 00 00 ff ff f8 00 00 3f ff\n  \
   f8 00 00 0f ff f8 00 00 00 00 00 00>}\n\n");

fprintf(fp,"imagemask\n");

/* put the origin back */
fprintf(fp,"%d %d translate\n",-x,-y);
}
