/*=============================================================
 * valgdcom.c -- Validate GEDCOM file
 * Copyright(c) 1993-94 by T.T. Wetmore IV; all rights reserved
 *   2.3.4 - 24 Jun 93     2.3.5 - 02 Sep 93
 *   3.0.0 - 06 Jul 94     3.0.2 - 18 Feb 95
 *===========================================================*/

#include "standard.h"
#include "table.h"
#include "gedcom.h"
#include "gedcheck.h"
#include "translat.h"

static TABLE convtab;
static INT rec_type;
static BOOLEAN named = FALSE;
static INT person = -1;
static INT family = -1;
static INT event  = -1;
static INT source = -1;
static INT other  = -1;
static INT struct_len = 0;
static INT struct_max = 0;
static INT num_errors;
static INT num_warns;
static INT noline;
static INT defline;
static BOOLEAN logopen;
static FILE *flog;

ELMNT *index_data = NULL;

#define SS (STRING)

STRING misixr = SS "Line %d: The person defined here has no key.";
STRING misfxr = SS "Line %d: The family defined here has no key.";
STRING misexr = SS "Line %d: The event defined here has no key.";
STRING missxr = SS "Line %d: The source defined here has no key.";
STRING misrxr = SS "Line %d: The record defined here has no key.";
STRING mulper = SS "Lines %d and %d: Person %s is multiply defined.";
STRING mulfam = SS "Lines %d and %d: Family %s is multiply defined.";
STRING mulsrc = SS "Lines %d and %d: Source %s is multiply defined.";
STRING mulevn = SS "Lines %d and %d: Event %s is multiply defined.";
STRING muloth = SS "Lines %d and %d: Record %s is multiply defined.";
STRING matper = SS "Line %d: Person %s has an incorrect key.";
STRING matfam = SS "Line %d: Family %s has an incorrect key.";
STRING matsrc = SS "Line %d: Source %s has an incorrect key.";
STRING matevn = SS "Line %d: Event %s has an incorrect key.";
STRING matoth = SS "Line %d: Record %s has an incorrect key.";
STRING misval = SS "Line %d: This %s line is missing a value field.";
STRING undper = SS "Person %s is referred to but not defined.";
STRING undfam = SS "Family %s is referred to but not defined.";
STRING undrec = SS "Record %s is referred to but not defined.";
STRING undsrc = SS "Source %s is referred to but not defined.";
STRING undevn = SS "Event %s is referred to but not defined.";
STRING badlev = SS "Line %d: This line has a level number that is too large.";
STRING noname = SS "Line %d: Person defined here has no name.";
STRING mulfth = SS "Line %d: Person %s has multiple father links.";
STRING mulmth = SS "Line %d: Person %s has multiple mother links.";
STRING mulhsb = SS "Line %d: Family %s has multiple husband links.";
STRING mulwif = SS "Line %d: Person %s has multiple wife links.";
#if 0
STRING noxref = SS "Line %d: This record has no cross reference value.";
#endif

static INT add_even_defn();
static INT add_fam_defn();
static INT add_indi_defn();
static INT add_othr_defn();
static INT add_sour_defn();
static INT add_to_structures();
static check_even_links();
static check_fam_links();
static check_indi_links();
static check_othr_links();
static check_references();
static check_sour_links();
static handle_fam_lev1();
static handle_indi_lev1();
static handle_value();
static ELMNT xref_to_elmnt();

/*===================================================
 * validate_gedcom -- Validate GEDCOM records in file
 *=================================================*/
BOOLEAN validate_gedcom (fp)
FILE *fp;
{
	INT lev, rc, curlev = 0;
	INT nindi, nfam, nsour, neven, nothr;
	extern INT lineno;
	ELMNT el;
	TRANTABLE tt = tran_tables[MGDIN];
	STRING xref, tag, val, msg;
	char scratch[200];

	nindi = nfam = nsour = neven = nothr = 0;
	num_errors = num_warns = 0;
	logopen = FALSE;
	noline = 0;
	defline = 0;
	curlev = 0;
	convtab = create_table();

	wfield(1, 1, "     0 Persons");
	wfield(2, 1, "     0 Families");
	wfield(3, 1, "     0 Events");
	wfield(4, 1, "     0 Sources");
	wfield(5, 1, "     0 Others");
	wfield(6, 1, "     0 Errors");
	wfield(7, 1, "     0 Warnings");

	rc = file_to_line(fp, tt, &lev, &xref, &tag, &val, &msg);
	xref = xref ? rmvat(xref) : NULL;
	rec_type = OTHR_REC;
	noline++;
	while (rc != DONE)  {
		if (lev > curlev + 1 || rc == ERROR) {
			if (rc == ERROR)
				handle_err(msg);
			else
				handle_err(badlev, lineno);
			handle_value(val, lineno);
			curlev = lev;
			rc = file_to_line(fp, tt, &lev, &xref, &tag, &val,
			    &msg);
			xref = xref ? rmvat(xref) : NULL;
			noline++;
			continue;
		}
		if (lev > 1) {
			handle_value(val, lineno);
			curlev = lev;
			rc = file_to_line(fp, tt, &lev, &xref, &tag, &val,
			    &msg);
			xref = xref ? rmvat(xref) : NULL;
			noline++;
			continue;
		}
		if (lev == 0) {
			char str[10];
			if (rec_type == INDI_REC && !named)
				handle_err(noname, defline);
			defline = lineno;
			if (eqstr("HEAD", tag) || eqstr("TRLR", tag))
				rec_type = IGNR_REC;
			else if (eqstr("INDI", tag)) {
				sprintf(str, "%6d", ++nindi);
				wfield(1, 1, str);
				rec_type = INDI_REC;
				named = FALSE;
				person = add_indi_defn(xref, lineno, &el);
			} else if (eqstr("FAM", tag)) {
				sprintf(str, "%6d", ++nfam);
				wfield(2, 1, str);
				rec_type = FAM_REC;
				family = add_fam_defn(xref, lineno);
			} else if (eqstr("EVEN", tag)) {
				sprintf(str, "%6d", ++neven);
				wfield(3, 1, str);
				rec_type = EVEN_REC;
				event = add_even_defn(xref, lineno);
			} else if (eqstr("SOUR", tag)) {
				sprintf(str, "%6d", ++nsour);
				wfield(4, 1, str);
				rec_type = SOUR_REC;
				source = add_sour_defn(xref, lineno);
			} else {
				sprintf(str, "%6d", ++nothr);
				wfield(5, 1, str);
				rec_type = OTHR_REC;
				other = add_othr_defn(xref, lineno);
			}
		} else {
			if (rec_type == INDI_REC)
				handle_indi_lev1(tag, val, lineno);
			else if (rec_type == FAM_REC)
				handle_fam_lev1(tag, val, lineno);
			else
				handle_value(val, lineno);
		}
		curlev = lev;
		rc = file_to_line(fp, tt, &lev, &xref, &tag, &val, &msg);
		xref = xref ? rmvat(xref) : NULL;
		noline++;
	}
	check_references();
	if (logopen) {
		fclose(flog);
		logopen = FALSE;
	}
	return num_errors == 0;
}
/*=======================================
 * add_indi_defn -- Add person definition
 *=====================================*/
static INT add_indi_defn (xref, line, pel)
STRING xref;	/* ref value */
INT line;	/* line num */
ELMNT *pel;
{
	ELMNT el;
	INT dex;

	*pel = NULL;
	if (!xref || *xref == 0) {
		handle_err(misixr, line);
		return -1;
	}
	if ((dex = xref_to_index(xref)) == -1) {
		*pel = el = (ELMNT) stdalloc(sizeof(*el));
		Type(el) = INDI_REC;
		Key(el) = xref = strsave(xref);
		Line(el) = 0;
		New(el) = NULL;
		Sex(el) = SEX_UNKNOWN;
		Male(el) = 0;
		Fmle(el) = 0;
		dex = add_to_structures(xref, el);
	} else
		*pel = el = index_data[dex];
	if (KNOWNTYPE(el) && Type(el) != INDI_REC) {
		handle_err(matper, line);
		return -1;
	}
	if (Type(el) == INDI_REC) {
		if (Line(el) && line) {
			handle_err(mulper, Line(el), line, xref);
			return -1;
		}
		if (line) Line(el) = line;
		return dex;
	}
	Type(el) = INDI_REC;
	Line(el) = line;
	Sex(el) = SEX_UNKNOWN;
	Male(el) = 0;
	Fmle(el) = 0;
	return dex;
}
/*======================================
 * add_fam_defn -- Add family definition
 *====================================*/
static INT add_fam_defn (xref, line)
STRING xref;	/* cross ref value */
INT line;	/* line num */
{
	ELMNT el;
	INT dex;
	if (!xref || *xref == 0) {
		handle_err(misfxr, line);
		return NULL;
	}
	if ((dex = xref_to_index(xref)) == -1) {
		el = (ELMNT) stdalloc(sizeof(*el));
		Type(el) = FAM_REC;
		Key(el) = xref = strsave(xref);
		Line(el) = 0;
		New(el) = NULL;
		Male(el) = 0;
		Fmle(el) = 0;
		dex = add_to_structures(xref, el);
	} else
		el = index_data[dex];
	if (KNOWNTYPE(el) && Type(el) != FAM_REC) {
		handle_err(matfam, line, xref);
		return -1;
	}
	if (Type(el) == FAM_REC) {
		if (Line(el) && line) {
			handle_err(mulfam, Line(el), line, xref);
			return -1;
		}
		if (line) Line(el) = line;
		return dex;
	}
	Type(el) = FAM_REC;
	Line(el) = line;
	Male(el) = 0;
	Fmle(el) = 0;
	return dex;
}
/*=======================================
 * add_sour_defn -- Add source definition
 *=====================================*/
static INT add_sour_defn (xref, line)
STRING xref;	/* cross ref value */
INT line;	/* line num */
{
	ELMNT el;
	INT dex;
	if (!xref || *xref == 0) {
		handle_err(missxr, line);
		return -1;
	}
	if ((dex = xref_to_index(xref)) == -1) {
		el = (ELMNT) stdalloc(sizeof(*el));
		Type(el) = SOUR_REC;
		Key(el) = xref = strsave(xref);
		Line(el) = 0;
		New(el) = NULL;
		dex = add_to_structures(xref, el);
	} else
		el = index_data[dex];
	if (KNOWNTYPE(el) && Type(el) != SOUR_REC) {
		handle_err(matsrc, line);
		return -1;
	}
	if (Type(el) == SOUR_REC) {
		if (Line(el) && line) {
			handle_err(mulsrc, Line(el), line, xref);
			return -1;
		}
		if (line) Line(el) = line;
		return dex;
	}
	Type(el) = SOUR_REC;
	Line(el) = line;
	return dex;
}
/*======================================
 * add_even_defn -- Add event definition
 *====================================*/
static INT add_even_defn (xref, line)
STRING xref;	/* cross ref value */
INT line;	/* line num */
{
	ELMNT el;
	INT dex;
	if (!xref || *xref == 0) {
		handle_err(misexr, line);
		return -1;
	}
	if ((dex = xref_to_index(xref)) == -1) {
		el = (ELMNT) stdalloc(sizeof(*el));
		Type(el) = EVEN_REC;
		Key(el) = xref = strsave(xref);
		Line(el) = 0;
		New(el) = NULL;
		dex = add_to_structures(xref, el);
	} else
		el = index_data[dex];
	if (KNOWNTYPE(el) && Type(el) != EVEN_REC) {
		handle_err(matevn, line, xref);
		return -1;
	}
	if (Type(el) == EVEN_REC) {
		if (Line(el) && line) {
			handle_err(mulevn, Line(el), line, xref);
			return -1;
		}
		if (line) Line(el) = line;
		return dex;
	}
	Type(el) = EVEN_REC;
	Line(el) = line;
	return dex;
}
/*==================================================
 * add_othr_defn -- Add other record type definition
 *================================================*/
static INT add_othr_defn (xref, line)
STRING xref;	/* cross ref value */
INT line;	/* line num */
{
	ELMNT el;
	INT dex;
	if (!xref || *xref == 0) {
		handle_err(misrxr, line);
		return -1;
	}
	if ((dex = xref_to_index(xref)) == -1) {
		el = (ELMNT) stdalloc(sizeof(*el));
		Type(el) = OTHR_REC;
		Key(el) = xref = strsave(xref);
		Line(el) = 0;
		New(el) = NULL;
		dex = add_to_structures(xref, el);
	} else
		el = index_data[dex];
	if (KNOWNTYPE(el) && Type(el) != OTHR_REC) {
		handle_err(matoth, line, xref);
		return -1;
	}
	if (Type(el) == OTHR_REC) {
		if (Line(el) && line) {
			handle_err(muloth, Line(el), line, xref);
			return -1;
		}
		if (line) Line(el) = line;
		return dex;
	}
	Type(el) = OTHR_REC;
	Line(el) = line;
	return dex;
}
/*===========================================================
 * handle_indi_lev1 -- Handle level 1 lines in person records
 *=========================================================*/
static handle_indi_lev1 (tag, val, line)
STRING tag, val;
INT line;
{
	INT fam, dex;
	ELMNT indi, pers;
	ASSERT(person != -1);
	indi = index_data[person];
	if (eqstr(tag, "FAMC")) {
		if (!pointer_value(val)) {
			handle_err(misval, line, "FAMC");
			return;
		}
		(void) add_fam_defn(rmvat(val), 0);
	} else if (eqstr(tag, "FAMS")) {
		if (!pointer_value(val)) {
			handle_err(misval, line, "FAMS");
			return;
		}
		(void) add_fam_defn(rmvat(val), 0);
	} else if (eqstr(tag, "FATH")) {
		if (!pointer_value(val)) {
			handle_warn(misval, line, "FATH");
			return;
		}
		Male(indi) += 1;
		if ((dex = add_indi_defn(rmvat(val), 0, &pers)) != -1)
			Sex(pers) |= BE_MALE;
	} else if (eqstr(tag, "MOTH")) {
		if (!pointer_value(val)) {
			handle_warn(misval, line, "MOTH");
			return;
		}
		Fmle(indi) += 1;
		if ((dex = add_indi_defn(rmvat(val), 0, &pers)) != -1)
			Sex(pers) |= BE_FEMALE;
	} else if (eqstr(tag, "SEX")) {
		if (*val == 'M')
			Sex(indi) |= IS_MALE;
		else if (*val == 'F')
			Sex(indi) |= IS_FEMALE;
	} else if (eqstr(tag, "NAME")) {
		named = TRUE;
		if (!val || *val == 0 || !valid_name(val)) {
			handle_err("Line %d: Bad NAME syntax.\n", line);
			return;
		}
	} else
		handle_value(val, line);
}
/*==========================================================
 * handle_fam_lev1 -- Handle level 1 lines in family records
 *========================================================*/
static handle_fam_lev1 (tag, val, line)
STRING tag, val;
INT line;
{
	INT dex;
	ELMNT fam, pers;
/*wprintf("handle_fam_lev1: %s, %s, %d\n",tag,val,line);/*DEBUG*/
/*wprintf("handle_fam_lev1: family == %d\n", family);/*DEBUG*/
	fam = (family != -1) ? index_data[family] : NULL;
	if (eqstr(tag, "HUSB")) {
		if (!pointer_value(val)) {
			handle_err(misval, line, "HUSB");
			return;
		}
		if (fam) Male(fam) += 1;
		if ((dex = add_indi_defn(rmvat(val), 0, &pers)) != -1)
			Sex(pers) |= BE_MALE;
	} else if (eqstr(tag, "WIFE")) {
		if (!pointer_value(val)) {
			handle_err(misval, line, "WIFE");
			return;
		}
		if (fam) Fmle(fam) += 1;
		if ((dex = add_indi_defn(rmvat(val), 0, &pers)) != -1)
			Sex(pers) |= BE_FEMALE;
	} else if (eqstr(tag, "CHIL")) {
		if (!pointer_value(val)) {
			handle_err(misval, line, "CHIL");
			return;
		}
		(void) add_indi_defn(rmvat(val), 0, &pers);
	} else
		handle_value(val, line);
}
/*=================================================
 * check_references -- Check for undefined problems
 *===============================================*/
static check_references ()
{
	INT i;
	for (i = 0; i < struct_len; i++) {
		ELMNT el = index_data[i];
		switch (Type(el)) {
		case INDI_REC: check_indi_links(el); break;
		case FAM_REC:  check_fam_links(el);  break;
		case EVEN_REC: check_even_links(el); break;
		case SOUR_REC: check_sour_links(el); break;
		case OTHR_REC: check_othr_links(el); break;
		default: handle_err(undrec, Key(el));
		}
	}
}
/*=======================================
 * check_indi_links -- Check person links
 *=====================================*/
static check_indi_links (per)
ELMNT per;
{
	INT fam, par;
	INT sam, opp;
	ELMNT el;
	STRING msg;
	if (Line(per) == 0) handle_err(undper, Key(per));
	if (Male(per) > 1)  handle_warn(mulfth, Line(per), Key(per));
	if (Fmle(per) > 1)  handle_warn(mulmth, Line(per), Key(per));
}
/*======================================
 * check_fam_links -- Check family links
 *====================================*/
static check_fam_links (fam)
ELMNT fam;
{
	if (Line(fam) == 0) handle_err(undfam, Key(fam));
	if (Male(fam) > 1)  handle_warn(mulhsb, Line(fam), Key(fam));
	if (Fmle(fam) > 1)  handle_warn(mulwif, Line(fam), Key(fam));
}
/*======================================
 * check_even_links -- Check event links
 *====================================*/
static check_even_links (evn)
ELMNT evn;
{
	if (Line(evn) == 0) handle_err(undevn, Key(evn));
}
/*=======================================
 * check_sour_links -- Check source links
 *=====================================*/
static check_sour_links (src)
ELMNT src;
{
	if (Line(src) == 0) handle_err(undsrc, Key(src));
}
/*======================================
 * check_othr_links -- Check other links
 *====================================*/
static check_othr_links (otr)
ELMNT otr;
{
	if (Line(otr) == 0) handle_err(undrec, Key(otr));
}
/*=======================================
 * handle_value -- Handle arbitrary value
 *=====================================*/
static handle_value (val, line)
STRING val;
INT line;
{
	ELMNT el;
	INT dex;
	STRING xref;

	if (rec_type == IGNR_REC) return;
	if (!pointer_value(val)) return;
	xref = rmvat(val);
	if (xref_to_index(xref) != -1) return;
	el = (ELMNT) stdalloc(sizeof(*el));
	Type(el) = UNKN_REC;
	Key(el) = xref = strsave(xref);
	New(el) = NULL;
	Line(el) = line;
	add_to_structures(xref, el);
}
/*=========================================
 * pointer_value -- See if value is pointer
 *=======================================*/
BOOLEAN pointer_value (val)
STRING val;
{
	if (!val || *val != '@' || strlen(val) < 3) return FALSE;
	return val[strlen(val)-1] == '@';
}
/*==================================
 * handle_err -- Handle GEDCOM error
 *================================*/
handle_err (fmt, arg1, arg2, arg3, arg4)
STRING fmt;
WORD arg1, arg2, arg3, arg4;
{
	char str[100];
	if (!logopen) {
		ASSERT(flog = fopen("err.log", "w"));
		logopen = TRUE;
	}
	fprintf(flog, "Error: ");
	fprintf(flog, fmt, arg1, arg2, arg3, arg4);
	fprintf(flog, "\n");
	sprintf(str, "%6d Errors (see log file `err.log')", ++num_errors);
	wfield(6, 1, str);
}
/*=====================================
 * handle_warn -- Handle GEDCOM warning
 *===================================*/
handle_warn (fmt, arg1, arg2, arg3, arg4)
STRING fmt;
WORD arg1, arg2, arg3, arg4;
{
	char str[100];
	if (!logopen) {
		ASSERT(flog = fopen("err.log", "w"));
		logopen = TRUE;
	}
	fprintf(flog, "Warning: ");
	fprintf(flog, fmt, arg1, arg2, arg3, arg4);
	fprintf(flog, "\n");
	sprintf(str, "%6d Warnings (see log file `err.log')", ++num_warns);
	wfield(7, 1, str);
}
/*=========================================
 * xref_to_index - Convert pointer to index
 *=======================================*/
INT xref_to_index (xref)
STRING xref;
{
	BOOLEAN there;
	INT dex = valueofbool(convtab, xref, &there);
	return there ? dex : -1;
}
/*=========================================================
 * add_to_structures -- Add new elements to data structures
 *=======================================================*/
static INT add_to_structures(xref, el)
STRING xref;
ELMNT el;
{
	INT i, n;

	if ((n = struct_len) >= struct_max)  {
		ELMNT *newi = (ELMNT *) stdalloc(sizeof(ELMNT)*
		    (n + 10000));
		for (i = 0; i < n; i++)
			newi[i] = index_data[i];
		if (n) stdfree(index_data);
		index_data = newi;
		struct_max += 10000;
	}
	i = struct_len;
	index_data[i] = el;
	insert_table(convtab, xref, struct_len++);
	return struct_len - 1;
}
